# coding: utf-8
import json
import logging
import os

from dataiku.base.folder_context import build_model_cache_folder_context
from dataikuapi.dss.future import DSSFuture
from dataiku.core.intercom import backend_json_call
from dataiku.core.intercom import is_lambda_server
from dataiku.core.intercom import lambda_api_post_call
from dataiku.core.intercom import lambda_or_backend_get_call
from dataiku import api_client

logger = logging.getLogger(__name__)


TAR_BUFFER_SIZE = 10 * 16 * 1024


# The model provider API is semi public it is not used by customers but it however used by
# Other features within DSS:
#   - NER plugins: https://github.com/dataiku/dss-plugin-nlp-named-entity-recognition/blob/master/resource/compute_dynamic_select.py#L6
#   - Doctor for TabICL model: https://github.com/dataiku/dip/pull/40440/files#:~:text=model_provider%20%3D%20get_model_provider()
# If you modify `get_or_download_model()` do not break the existing calls
class ModelProvider(object):
    def __init__(self):
        self._models = {}
        self._dss_client = api_client() if not is_lambda_server() else None

    def get_or_download_model(self, model_key, connection_name=None):
        """
        Returns a path to the requested model
        """
        model_root = self._get_model_root_path(model_key, connection_name)

        model_metadata_dic = self.get_model_metadata(model_key, connection_name)
        logger.info("model to be used:" + os.linesep + json.dumps(model_metadata_dic, indent=4))

        model_path = os.path.join(model_root, 'model')
        return model_path

    def get_model_metadata(self, model_key, connection_name=None):
        """
        Returns a dic containing the requested model metadata
        """
        return lambda_or_backend_get_call("ml/hf/model/metadata", data=None,
                                          params={"modelKey": model_key, "connectionName": connection_name})

    def _get_model_root_path(self, model_key, connection_name):
        """
        Returns the root path to the requested model
        """
        if model_key in self._models:
            return self._models[model_key]

        logger.info("Downloading model to DSS instance if needed...")

        if is_lambda_server():
            # We cannot use futures in the lambda server so the call is synchronous
            lambda_api_post_call("ml/hf/download-model", data={
                "modelKey": model_key,
                "connectionName": connection_name
            })
        else:
            future_response = backend_json_call("ml/hf/download-model", data={
                "modelKey": model_key,
                "connectionName": connection_name
            })
            future = DSSFuture(self._dss_client, future_response.get("jobId", None), future_response)
            future.wait_for_result()

        logger.info("Model available on the DSS instance")

        local_model_info = lambda_or_backend_get_call("ml/hf/model/local-path", data=None,
                                                      params={"modelKey": model_key})
        folder_context = build_model_cache_folder_context(local_model_info["modelPath"])
        with folder_context.get_folder_path_to_read() as local_model_folder_path:
            self._models[model_key] = local_model_folder_path
            return local_model_folder_path

_model_provider = None


def get_model_provider():
    global _model_provider
    if _model_provider is None:
        _model_provider = ModelProvider()
    return _model_provider


# Public method for customers to download a model to the DSS model cache programmatically
def download_model_to_cache(model_name, connection_name=None):
    if is_lambda_server():
        # We cannot use futures in the lambda server so the call is synchronous
        lambda_api_post_call("ml/hf/download-enabled-model", data={
            "modelKey": model_name,
            "connectionName": connection_name
        })
    else:
        dss_client = api_client()
        future_response = backend_json_call("ml/hf/download-enabled-model", data={
            "modelKey": model_name,
            "connectionName": connection_name
        })
        future = DSSFuture(dss_client, future_response.get("jobId", None), future_response)
        future.wait_for_result()

    logger.info("Model available on the DSS instance")

# Public method for customers to get a model from the DSS model cache programmatically
def get_model_from_cache(model_name):
    local_model_info = lambda_or_backend_get_call("ml/hf/model/local-path", data=None,
                                                      params={"modelKey": model_name})
    folder_context = build_model_cache_folder_context(local_model_info["modelPath"])
    with folder_context.get_folder_path_to_read() as local_model_folder_path:
        return os.path.join(local_model_folder_path, "model")
