/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs.services;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.net.UnknownHostException;
import java.time.Duration;
import java.util.List;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.fs.ClosedIOException;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.AbfsStatistic;
import org.apache.hadoop.fs.azurebfs.constants.HttpOperationType;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsDriverException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidAbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.services.AbfsAHCHttpOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsApacheHttpClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsCounters;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpHeader;
import org.apache.hadoop.fs.azurebfs.services.AbfsHttpOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsIoUtils;
import org.apache.hadoop.fs.azurebfs.services.AbfsJdkHttpOperation;
import org.apache.hadoop.fs.azurebfs.services.AbfsRestOperationType;
import org.apache.hadoop.fs.azurebfs.services.AbfsRetryPolicy;
import org.apache.hadoop.fs.azurebfs.services.AbfsThrottlingIntercept;
import org.apache.hadoop.fs.azurebfs.services.RetryReason;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.fs.statistics.DurationTrackerFactory;
import org.apache.hadoop.fs.statistics.impl.IOStatisticsBinding;
import org.apache.http.impl.execchain.RequestAbortedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AbfsRestOperation {
    private final AbfsRestOperationType operationType;
    private final AbfsClient client;
    private final AbfsThrottlingIntercept intercept;
    private final String method;
    private final URL url;
    private final List<AbfsHttpHeader> requestHeaders;
    private final boolean hasRequestBody;
    private final String sasToken;
    private static final Logger LOG = LoggerFactory.getLogger(AbfsClient.class);
    private byte[] buffer;
    private int bufferOffset;
    private int bufferLength;
    private int retryCount = 0;
    private AbfsHttpOperation result;
    private AbfsCounters abfsCounters;
    private String failureReason;
    private AbfsRetryPolicy retryPolicy;
    private final AbfsConfiguration abfsConfiguration;
    private TracingContext lastUsedTracingContext;
    private int apacheHttpClientIoExceptions = 0;

    public boolean hasResult() {
        return this.result != null;
    }

    public AbfsHttpOperation getResult() {
        return this.result;
    }

    public void hardSetResult(int httpStatus) {
        this.result = AbfsHttpOperation.getAbfsHttpOperationWithFixedResult(this.url, this.method, httpStatus);
    }

    public URL getUrl() {
        return this.url;
    }

    public List<AbfsHttpHeader> getRequestHeaders() {
        return this.requestHeaders;
    }

    public boolean isARetriedRequest() {
        return this.retryCount > 0;
    }

    String getSasToken() {
        return this.sasToken;
    }

    AbfsRestOperation(AbfsRestOperationType operationType, AbfsClient client, String method, URL url, List<AbfsHttpHeader> requestHeaders, AbfsConfiguration abfsConfiguration) {
        this(operationType, client, method, url, requestHeaders, null, abfsConfiguration);
    }

    AbfsRestOperation(AbfsRestOperationType operationType, AbfsClient client, String method, URL url, List<AbfsHttpHeader> requestHeaders, String sasToken, AbfsConfiguration abfsConfiguration) {
        this.operationType = operationType;
        this.client = client;
        this.method = method;
        this.url = url;
        this.requestHeaders = requestHeaders;
        this.hasRequestBody = "PUT".equals(method) || "POST".equals(method) || "PATCH".equals(method);
        this.sasToken = sasToken;
        this.abfsCounters = client.getAbfsCounters();
        this.intercept = client.getIntercept();
        this.abfsConfiguration = abfsConfiguration;
        this.retryPolicy = client.getExponentialRetryPolicy();
    }

    AbfsRestOperation(AbfsRestOperationType operationType, AbfsClient client, String method, URL url, List<AbfsHttpHeader> requestHeaders, byte[] buffer, int bufferOffset, int bufferLength, String sasToken, AbfsConfiguration abfsConfiguration) {
        this(operationType, client, method, url, requestHeaders, sasToken, abfsConfiguration);
        this.buffer = buffer;
        this.bufferOffset = bufferOffset;
        this.bufferLength = bufferLength;
        this.abfsCounters = client.getAbfsCounters();
    }

    public void execute(TracingContext tracingContext) throws AzureBlobFileSystemException {
        this.lastUsedTracingContext = this.createNewTracingContext(tracingContext);
        try {
            IOStatisticsBinding.trackDurationOfInvocation((DurationTrackerFactory)this.abfsCounters, (String)AbfsStatistic.getStatNameFromHttpCall(this.method), () -> this.completeExecute(this.lastUsedTracingContext));
        }
        catch (AzureBlobFileSystemException aze) {
            throw aze;
        }
        catch (IOException e) {
            throw new UncheckedIOException("Error while tracking Duration of an AbfsRestOperation call", e);
        }
    }

    void completeExecute(TracingContext tracingContext) throws AzureBlobFileSystemException {
        String latencyHeader = this.getClientLatency();
        if (latencyHeader != null && !latencyHeader.isEmpty()) {
            AbfsHttpHeader httpHeader = new AbfsHttpHeader("x-ms-abfs-client-latency", latencyHeader);
            this.requestHeaders.add(httpHeader);
        }
        this.retryCount = 0;
        this.retryPolicy = this.client.getExponentialRetryPolicy();
        LOG.debug("First execution of REST operation - {}", (Object)this.operationType);
        while (!this.executeHttpOperation(this.retryCount, tracingContext)) {
            try {
                ++this.retryCount;
                tracingContext.setRetryCount(this.retryCount);
                long retryInterval = this.retryPolicy.getRetryInterval(this.retryCount);
                LOG.debug("Rest operation {} failed with failureReason: {}. Retrying with retryCount = {}, retryPolicy: {} and sleepInterval: {}", new Object[]{this.operationType, this.failureReason, this.retryCount, this.retryPolicy.getAbbreviation(), retryInterval});
                Thread.sleep(retryInterval);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
        }
        int status = this.result.getStatusCode();
        if (status < 100) {
            throw new InvalidAbfsRestOperationException(null, this.retryCount);
        }
        if (status >= 400) {
            throw new AbfsRestOperationException(this.result.getStatusCode(), this.result.getStorageErrorCode(), this.result.getStorageErrorMessage(), null, this.result);
        }
        LOG.trace("{} REST operation complete", (Object)this.operationType);
    }

    @VisibleForTesting
    String getClientLatency() {
        return this.client.getAbfsPerfTracker().getClientLatency();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean executeHttpOperation(int retryCount, TracingContext tracingContext) throws AzureBlobFileSystemException {
        AbfsHttpOperation httpOperation;
        boolean wasKnownExceptionThrown = false;
        try {
            httpOperation = this.createHttpOperation();
            this.incrementCounter(AbfsStatistic.CONNECTIONS_MADE, 1L);
            tracingContext.constructHeader(httpOperation, this.failureReason, this.retryPolicy.getAbbreviation());
            this.signRequest(httpOperation, this.hasRequestBody ? this.bufferLength : 0);
        }
        catch (IOException e) {
            LOG.debug("Auth failure: {}, {}", (Object)this.method, (Object)this.url);
            throw new AbfsRestOperationException(-1, null, "Auth failure: " + e.getMessage(), e);
        }
        try {
            AbfsIoUtils.dumpHeadersToDebugLog("Request Headers", httpOperation.getRequestProperties());
            this.intercept.sendingRequest(this.operationType, this.abfsCounters);
            if (this.hasRequestBody) {
                httpOperation.sendPayload(this.buffer, this.bufferOffset, this.bufferLength);
                this.incrementCounter(AbfsStatistic.SEND_REQUESTS, 1L);
                this.incrementCounter(AbfsStatistic.BYTES_SENT, this.bufferLength);
            }
            httpOperation.processResponse(this.buffer, this.bufferOffset, this.bufferLength);
            this.incrementCounter(AbfsStatistic.GET_RESPONSES, 1L);
            if (httpOperation.getStatusCode() >= 200 && httpOperation.getStatusCode() <= 206) {
                this.incrementCounter(AbfsStatistic.BYTES_RECEIVED, httpOperation.getBytesReceived());
            } else if (httpOperation.getStatusCode() == 503) {
                this.incrementCounter(AbfsStatistic.SERVER_UNAVAILABLE, 1L);
            }
            LOG.debug("HttpRequest: {}: {}", (Object)this.operationType, (Object)httpOperation);
            int status = httpOperation.getStatusCode();
            this.failureReason = RetryReason.getAbbreviation(null, status, httpOperation.getStorageErrorMessage());
            this.retryPolicy = this.client.getRetryPolicy(this.failureReason);
            if (this.retryPolicy.shouldRetry(retryCount, httpOperation.getStatusCode())) {
                boolean bl = false;
                return bl;
            }
            this.result = httpOperation;
        }
        catch (UnknownHostException ex) {
            wasKnownExceptionThrown = true;
            String hostname = null;
            hostname = httpOperation.getHost();
            this.failureReason = RetryReason.getAbbreviation(ex, null, null);
            this.retryPolicy = this.client.getRetryPolicy(this.failureReason);
            LOG.warn("Unknown host name: {}. Retrying to resolve the host name...", (Object)hostname);
            if (httpOperation instanceof AbfsAHCHttpOperation) {
                this.registerApacheHttpClientIoException();
            }
            if (!this.retryPolicy.shouldRetry(retryCount, -1)) {
                throw new InvalidAbfsRestOperationException(ex, retryCount);
            }
            boolean statusCode = false;
            return statusCode;
        }
        catch (IOException ex) {
            wasKnownExceptionThrown = true;
            if (LOG.isDebugEnabled()) {
                LOG.debug("HttpRequestFailure: {}, {}", (Object)httpOperation, (Object)ex);
            }
            this.failureReason = RetryReason.getAbbreviation(ex, -1, "");
            this.retryPolicy = this.client.getRetryPolicy(this.failureReason);
            if (httpOperation instanceof AbfsAHCHttpOperation) {
                this.registerApacheHttpClientIoException();
                if (ex instanceof RequestAbortedException && ex.getCause() instanceof ClosedIOException) {
                    throw new AbfsDriverException((IOException)ex.getCause());
                }
            }
            if (!this.retryPolicy.shouldRetry(retryCount, -1)) {
                throw new InvalidAbfsRestOperationException(ex, retryCount);
            }
            boolean bl = false;
            return bl;
        }
        finally {
            int statusCode = httpOperation.getStatusCode();
            if (this.shouldUpdateCSTMetrics(statusCode) && !wasKnownExceptionThrown) {
                this.intercept.updateMetrics(this.operationType, httpOperation);
            }
        }
        return true;
    }

    private void registerApacheHttpClientIoException() {
        ++this.apacheHttpClientIoExceptions;
        if (this.apacheHttpClientIoExceptions >= this.abfsConfiguration.getMaxApacheHttpClientIoExceptionsRetries()) {
            AbfsApacheHttpClient.registerFallback();
        }
    }

    @VisibleForTesting
    public void signRequest(AbfsHttpOperation httpOperation, int bytesToSign) throws IOException {
        switch (this.client.getAuthType()) {
            case Custom: 
            case OAuth: {
                LOG.debug("Authenticating request with OAuth2 access token");
                httpOperation.setRequestProperty("Authorization", this.client.getAccessToken());
                break;
            }
            case SAS: {
                httpOperation.setMaskForSAS();
                break;
            }
            default: {
                LOG.debug("Signing request with shared key");
                this.client.getSharedKeyCredentials().signRequest(httpOperation, bytesToSign);
            }
        }
    }

    @VisibleForTesting
    AbfsHttpOperation createHttpOperation() throws IOException {
        HttpOperationType httpOperationType = this.abfsConfiguration.getPreferredHttpOperationType();
        if (httpOperationType == HttpOperationType.APACHE_HTTP_CLIENT && this.isApacheClientUsable()) {
            return this.createAbfsAHCHttpOperation();
        }
        return this.createAbfsHttpOperation();
    }

    private boolean isApacheClientUsable() {
        return AbfsApacheHttpClient.usable();
    }

    @VisibleForTesting
    AbfsJdkHttpOperation createAbfsHttpOperation() throws IOException {
        return new AbfsJdkHttpOperation(this.url, this.method, this.requestHeaders, Duration.ofMillis(this.client.getAbfsConfiguration().getHttpConnectionTimeout()), Duration.ofMillis(this.client.getAbfsConfiguration().getHttpReadTimeout()));
    }

    @VisibleForTesting
    AbfsAHCHttpOperation createAbfsAHCHttpOperation() throws IOException {
        return new AbfsAHCHttpOperation(this.url, this.method, this.requestHeaders, Duration.ofMillis(this.client.getAbfsConfiguration().getHttpConnectionTimeout()), Duration.ofMillis(this.client.getAbfsConfiguration().getHttpReadTimeout()), this.client.getAbfsApacheHttpClient());
    }

    private void incrementCounter(AbfsStatistic statistic, long value) {
        if (this.abfsCounters != null) {
            this.abfsCounters.incrementCounter(statistic, value);
        }
    }

    private boolean shouldUpdateCSTMetrics(int statusCode) {
        return statusCode < 300 || "ING".equals(this.failureReason) || "EGR".equals(this.failureReason) || "OPR".equals(this.failureReason);
    }

    @VisibleForTesting
    public TracingContext createNewTracingContext(TracingContext tracingContext) {
        return new TracingContext(tracingContext);
    }

    @VisibleForTesting
    public final TracingContext getLastTracingContext() {
        return this.lastUsedTracingContext;
    }
}

