/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.sanitycheck.detectors;

import com.dataiku.dip.connections.AbstractCloudStorageConnection;
import com.dataiku.dip.connections.AzureConnection;
import com.dataiku.dip.connections.ConnectionsDAO;
import com.dataiku.dip.connections.DSSConnection;
import com.dataiku.dip.connections.DatabricksConnection;
import com.dataiku.dip.connections.DatabricksVolumeConnection;
import com.dataiku.dip.connections.EC2Connection;
import com.dataiku.dip.connections.GCSConnection;
import com.dataiku.dip.connections.HDFSConnection;
import com.dataiku.dip.connections.SnowflakeConnection;
import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.dao.GeneralSettingsDAO;
import com.dataiku.dip.datasets.fs.AzureBlobDatasetHandler;
import com.dataiku.dip.datasets.fs.S3DatasetHandler;
import com.dataiku.dip.hadoop.HadoopLoader;
import com.dataiku.dip.sanitycheck.SanityCheckDetectorBase;
import com.dataiku.dip.sanitycheck.SanityCheckInfoMessages;
import com.dataiku.dip.security.impersonation.IImpersonationResolverService;
import com.dataiku.dip.server.services.TransactionService;
import com.dataiku.dip.transactions.ifaces.Transaction;
import com.dataiku.dip.utils.DKULogger;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
public class ConnectionConfigurationDetector
implements SanityCheckDetectorBase {
    @Autowired
    private TransactionService transactionService;
    @Autowired
    private ConnectionsDAO connectionsDAO;
    @Autowired
    private GeneralSettingsDAO generalSettingsDAO;
    @Autowired
    private IImpersonationResolverService impersonationService;
    private static final DKULogger logger = DKULogger.getLogger((String)"dip.sanitycheck.connectionconfigdetector");

    public List<InfoMessage.MessageCode> getCodes() {
        return Arrays.asList(Codes.values());
    }

    public SanityCheckInfoMessages runAnalysis(Set<String> exclusionList) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (exclusionList.containsAll(Arrays.stream(Codes.values()).map(InfoMessage.MessageCode::getCode).collect(Collectors.toList()))) {
            return messages;
        }
        try {
            GeneralSettingsDAO.GeneralSettings generalSettings;
            Map<String, DSSConnection> allConnections;
            try (Transaction t = this.transactionService.beginRead();){
                allConnections = this.connectionsDAO.listUnsafe();
                generalSettings = this.generalSettingsDAO.getUnsafe();
            }
            allConnections.values().forEach(conn -> {
                String type;
                switch (type = conn.getType()) {
                    case "GCS": {
                        messages.mergeFrom((InfoMessage.InfoMessages)this.checkGCSConnection(exclusionList, (GCSConnection)conn));
                        messages.mergeFrom(this.checkConnectionReadDetailsPermission(exclusionList, (DSSConnection)conn, generalSettings));
                        break;
                    }
                    case "Azure": {
                        messages.mergeFrom((InfoMessage.InfoMessages)this.checkAzureConnection(exclusionList, (AzureConnection)conn));
                        messages.mergeFrom(this.checkConnectionReadDetailsPermission(exclusionList, (DSSConnection)conn, generalSettings));
                        break;
                    }
                    case "EC2": {
                        messages.mergeFrom((InfoMessage.InfoMessages)this.checkEC2Connection(exclusionList, (EC2Connection)conn));
                        messages.mergeFrom(this.checkConnectionReadDetailsPermission(exclusionList, (DSSConnection)conn, generalSettings));
                        break;
                    }
                    case "Snowflake": {
                        messages.mergeFrom(this.checkSnowflakeConnection(exclusionList, (SnowflakeConnection)conn, allConnections));
                        messages.mergeFrom(this.checkConnectionReadDetailsPermission(exclusionList, (DSSConnection)conn, generalSettings));
                        break;
                    }
                    case "HDFS": {
                        messages.mergeFrom(this.checkHDFSConnection(exclusionList, (HDFSConnection)conn));
                        messages.mergeFrom(this.checkConnectionReadDetailsPermission(exclusionList, (DSSConnection)conn, generalSettings));
                        break;
                    }
                    case "Databricks": {
                        messages.mergeFrom(this.checkDatabricksConnection(exclusionList, (DatabricksConnection)conn, allConnections));
                        break;
                    }
                    case "iceberg": {
                        messages.mergeFrom(this.checkConnectionReadDetailsPermission(exclusionList, (DSSConnection)conn, generalSettings));
                        break;
                    }
                }
            });
        }
        catch (IOException e) {
            logger.warn((Object)"Sanity check failed", (Throwable)e);
            messages.addMessage(this.createFatalMessage(e));
        }
        return messages;
    }

    private InfoMessage.InfoMessages checkConnectionReadDetailsPermission(Set<String> exclusionList, DSSConnection conn, GeneralSettingsDAO.GeneralSettings generalSettings) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_SPARK_NO_GROUP_WITH_DETAILS_READ_ACCESS.getCode()) && generalSettings.sparkSettings.sparkEnabled && (conn.detailsReadability.readableBy == DSSConnection.DetailsReadableBy.NONE || conn.detailsReadability.readableBy == DSSConnection.DetailsReadableBy.ALLOWED && conn.detailsReadability.allowedGroups.size() == 0)) {
            messages.withWarningV((InfoMessage.MessageCode)Codes.WARN_CONNECTION_SPARK_NO_GROUP_WITH_DETAILS_READ_ACCESS, "Spark is enabled but connection '%s' has no groups allowed to read its details. Spark interaction may be slow.", new Object[]{conn.name});
        }
        return messages;
    }

    private InfoMessage.InfoMessages checkHDFSConnection(Set<String> exclusionList, HDFSConnection hdfs) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_HDFS_ACL_SUBDIRECTORY.getCode()) && this.impersonationService.isEnabled() && hdfs.params.aclSynchronizationMode == HDFSConnection.ACLSynchronizationMode.SUBDIRECTORY) {
            messages.withTrustedHTMLV(InfoMessage.Severity.WARNING, (InfoMessage.MessageCode)Codes.WARN_CONNECTION_HDFS_ACL_SUBDIRECTORY, "ACL synchronization mode is set to 'subdirectory' in connection '%s'. It is recommended to use Sentry or Ranger instead. See <a target='_blank' href='https://doc.dataiku.com/dss/latest/hadoop/hive.html#sentry-dss-user-isolation-enabled'>the documentation</a> for more details.", new String[]{hdfs.name});
        }
        return messages;
    }

    private InfoMessage.InfoMessages checkSnowflakeConnection(Set<String> exclusionList, SnowflakeConnection conn, Map<String, DSSConnection> allConnections) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_SNOWFLAKE_NO_AUTOFASTWRITE.getCode()) && !conn.params.useAutoFastPath && allConnections.values().stream().anyMatch(c2 -> c2 instanceof AbstractCloudStorageConnection)) {
            messages.withWarningV((InfoMessage.MessageCode)Codes.WARN_CONNECTION_SNOWFLAKE_NO_AUTOFASTWRITE, "Snowflake connection '%s' could leverage automatic fast-write given cloud storage connections are available.", new Object[]{conn.name});
        }
        return messages;
    }

    private InfoMessage.InfoMessages checkDatabricksConnection(Set<String> exclusionList, DatabricksConnection conn, Map<String, DSSConnection> allConnections) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_DATABRICKS_NO_AUTOFASTWRITE.getCode()) && !conn.params.useAutoFastPath) {
            int usableConnectionCount = 0;
            for (DSSConnection c2 : allConnections.values()) {
                if (c2 instanceof AzureConnection || c2 instanceof EC2Connection) {
                    ++usableConnectionCount;
                    continue;
                }
                if (c2 instanceof DatabricksVolumeConnection) {
                    ++usableConnectionCount;
                    continue;
                }
                if (!(c2 instanceof DatabricksConnection)) continue;
                DatabricksConnection d = (DatabricksConnection)c2;
                if (!d.allowWrite || !d.allowManagedFolders || !StringUtils.isNotBlank((String)d.params.chvolume)) continue;
                ++usableConnectionCount;
            }
            if (usableConnectionCount > 0) {
                messages.withWarningV((InfoMessage.MessageCode)Codes.WARN_CONNECTION_DATABRICKS_NO_AUTOFASTWRITE, "Databricks connection '%s' could leverage automatic fast-write given that cloud storage connections are available.", new Object[]{conn.name});
            }
        }
        return messages;
    }

    private SanityCheckInfoMessages checkGCSConnection(Set<String> exclusionList, GCSConnection gcs) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_NO_HADOOP_INTERFACE.getCode()) && HadoopLoader.hadoopEnabled() && (gcs.params.hdfsInterface == null || gcs.params.hdfsInterface == GCSConnection.HDFSInterface.NONE)) {
            messages.withWarningV((InfoMessage.MessageCode)Codes.WARN_CONNECTION_NO_HADOOP_INTERFACE, "No Hadoop interface set in %s connection '%s'.", new Object[]{gcs.getType(), gcs.name});
        }
        return messages;
    }

    private SanityCheckInfoMessages checkAzureConnection(Set<String> exclusionList, AzureConnection azure) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_NO_HADOOP_INTERFACE.getCode()) && HadoopLoader.hadoopEnabled() && (azure.params.hdfsInterface == null || azure.params.hdfsInterface == AzureBlobDatasetHandler.HDFSInterface.NONE)) {
            messages.withWarningV((InfoMessage.MessageCode)Codes.WARN_CONNECTION_NO_HADOOP_INTERFACE, "No Hadoop interface set in %s connection '%s'.", new Object[]{azure.getType(), azure.name});
        }
        return messages;
    }

    private SanityCheckInfoMessages checkEC2Connection(Set<String> exclusionList, EC2Connection ec2) {
        SanityCheckInfoMessages messages = new SanityCheckInfoMessages();
        if (!exclusionList.contains(Codes.WARN_CONNECTION_NO_HADOOP_INTERFACE.getCode()) && HadoopLoader.hadoopEnabled() && (ec2.params.hdfsInterface == null || ec2.params.hdfsInterface == S3DatasetHandler.HDFSInterface.NONE)) {
            messages.withWarningV((InfoMessage.MessageCode)Codes.WARN_CONNECTION_NO_HADOOP_INTERFACE, "No Hadoop interface set in %s connection '%s'.", new Object[]{ec2.getType(), ec2.name});
        }
        return messages;
    }

    public static enum Codes implements InfoMessage.MessageCode
    {
        WARN_CONNECTION_NO_HADOOP_INTERFACE("Cloud Storage - No Hadoop interface set", InfoMessage.FixabilityCategory.ADMIN_SETTINGS_CONNECTIONS),
        WARN_CONNECTION_DATABRICKS_NO_AUTOFASTWRITE("Databricks - Automatic fast-write disabled", InfoMessage.FixabilityCategory.ADMIN_SETTINGS_CONNECTIONS),
        WARN_CONNECTION_SNOWFLAKE_NO_AUTOFASTWRITE("Snowflake - Automatic fast-write disabled", InfoMessage.FixabilityCategory.ADMIN_SETTINGS_CONNECTIONS),
        WARN_CONNECTION_SPARK_NO_GROUP_WITH_DETAILS_READ_ACCESS("Spark - No groups allowed to read connection details", InfoMessage.FixabilityCategory.ADMIN_SETTINGS_CONNECTIONS),
        WARN_CONNECTION_HDFS_ACL_SUBDIRECTORY("HDFS - ACL synchronization mode", InfoMessage.FixabilityCategory.ADMIN_SETTINGS_CONNECTIONS);

        private final String title;
        private final InfoMessage.FixabilityCategory fixability;

        private Codes(String title, InfoMessage.FixabilityCategory fixability) {
            this.title = title;
            this.fixability = fixability;
        }

        public String getCode() {
            return this.name();
        }

        public String getCodeTitle() {
            return this.title;
        }

        public InfoMessage.FixabilityCategory getFixability() {
            return this.fixability;
        }
    }
}

