(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('DatasetTableInsightHandler', {
        name: 'Dataset',
        nameForTileParams: 'Dataset table',
        desc: 'Partial or whole datatable',
        i18nNameKey: 'INSIGHTS.DATASET_TABLE.NAME',
        i18nTileParamsNameKey: 'INSIGHTS.DATASET_TABLE.TILE_PARAMS_NAME',
        i18nDescKey: 'INSIGHTS.DATASET_TABLE.DESC',
        icon: 'icon-table',
        color: 'dataset',

        sourceType: 'DATASET',
        getSourceId: function(insight) {
            return insight.params.datasetSmartName;
        },
        hasEditTab: true,
        defaultTileParams: {
            viewKind: 'EXPLORE',
            showName: true,
            showDescription: true,
            showCustomFields: true,
            showMeaning: false,
            showProgressBar: false
        },
        defaultTileSize: [],
        defaultTileDimensions: [18, 9]
    });

    app.controller('DatasetTableViewCommon', function($scope, $stateParams, DashboardUtils) {
        $scope.resolvedDataset = resolveDatasetFullName($scope.insight.params.datasetSmartName, $stateParams.projectKey);
        $scope.loading = false;

        $scope.setLoading = (loading) => {
            $scope.loading = loading;
        };
    });

    app.directive('datasetTableInsightTile', function($controller, DashboardUtils, TileLoadingState) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                filters: '=',
                filtersParams: '<'
            },
            link: function($scope) {
                $controller('DatasetTableViewCommon', { $scope: $scope });

                $scope.ngShowLoaded = true;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.refreshNoSpinner = true;
                    $scope.setDashboardTileLoaded = DashboardUtils.setLoaded.bind([$scope, resolve]);
                    $scope.setDashboardTileError = DashboardUtils.setError.bind([$scope, reject]);
                    $scope.loading = true;
                };

                $scope.reload = function(resolve, reject) {
                    $scope.load(resolve, reject);
                    if ($scope.applySlideFilters) {
                        $scope.applySlideFilters();
                    }
                };

                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.reload;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.directive('datasetTableInsightView', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_view.html',
            scope: {
                insight: '=',
                tileParams: '='
            },
            link: function($scope) {
                $controller('DatasetTableViewCommon', { $scope: $scope });
            }
        };
    });

    app.directive('datasetTableInsightEdit', function($controller) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_edit.html',
            scope: true,
            link: function($scope) {
                $controller('DatasetTableViewCommon', { $scope: $scope });
            },
            controller: function($scope, translate) {
                $scope.translate = translate; // Add 'translate' to the scope as this directive might be called from Angular as a downgraded component
                $scope.sanitize = sanitize;
            }
        };
    });

    app.directive('shakerExploreInsight', function($filter, $timeout, $q, Assert, DataikuAPI, WT1, SmartId, DashboardFilters, $controller, DashboardPageUtils, Logger, translate) {
        return {
            scope: true,
            controller: function($scope, $stateParams, DatasetChartsUtils) {
                $controller('ShakerChartsCommonController', { $scope: $scope });

                const resolvedDataset = SmartId.resolve($scope.insight.params.datasetSmartName);
                $scope.inInsight = true;

                /* ********************* Callbacks for shakerExploreBase ******************* */

                $scope.shakerHooks.saveForAuto = function() {
                    return $q((resolve, reject) => {
                        try {
                            $scope.insight.params.shakerScript = $scope.getShakerData();
                            if ($scope.hook && $scope.hook.onSave) {
                                $scope.hook.onSave($scope.insight.params);
                            }

                            resolve();
                        } catch (error) {
                            reject(error);
                        }
                    });
                };

                $scope.shakerHooks.setColumnMeaning = function(column, newMeaning) {
                };

                $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType) {
                    return null;
                };
                $scope.shakerHooks.setColumnStorageType = function(column, newType, actionId) {
                };

                $scope.shakerHooks.updateColumnDetails = function(column) {
                };

                $scope.setSpinnerPosition = function() { };

                /*
                 * Keep the existing signature but ignore filterRequest because
                 * we override it with the filters from the filter tile
                 */
                // eslint-disable-next-line  no-unused-vars
                $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                    $scope.setLoading(true);
                    // shakerAndDashboardExplorationFilters is not null in case we have applied a globalFilter (Dashboard filters) on it
                    const tableChunkFilterRequest = $scope.insight.$shakerAndDashboardExplorationFilters == null ? filterRequest : { 'elements': $scope.buildFilterRequest($scope.insight.$shakerAndDashboardExplorationFilters) };
                    return DataikuAPI.shakers.getTableChunk(
                        $stateParams.projectKey,
                        $scope.inputDatasetProjectKey,
                        $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(),
                        $scope.requestedSampleId,
                        firstRow,
                        nbRows,
                        firstCol,
                        nbCols,
                        tableChunkFilterRequest
                    ).noSpinner()
                        .error((data, status, headers, config, statusText, xhrStatus) => {
                            const error = { data, status, headers, config, statusText, xhrStatus };
                            $scope.setLoading(false);
                            handleLoadingFailure(error);
                        }).then((data) => {
                            $scope.setLoading(false);
                            return data;
                        });
                };

                const baseExploreGetRefreshTablePromise = $scope.shakerHooks.getRefreshTablePromise;
                $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                    const tableChunkFilterRequest = $scope.insight.$shakerAndDashboardExplorationFilters == null ? filterRequest : { 'elements': $scope.buildFilterRequest($scope.insight.$shakerAndDashboardExplorationFilters) };
                    return baseExploreGetRefreshTablePromise(filtersOnly, tableChunkFilterRequest);
                };

                /* ********************* Main ******************* */

                // Set base context and call baseInit
                Assert.inScope($scope, 'shakerHooks');

                let unregisterLoadingStateWatcher;
                let unregisterVisibilityWatcher;
                let unregisterFiltersWatcher;

                $scope.$on('$destroy', function() {
                    if (unregisterLoadingStateWatcher != null) {
                        unregisterLoadingStateWatcher();
                    }
                    if (unregisterVisibilityWatcher != null) {
                        unregisterVisibilityWatcher();
                    }
                    if (unregisterFiltersWatcher != null) {
                        unregisterFiltersWatcher();
                    }
                });

                $scope.table = null;
                $scope.scriptId = '__pristine__';
                $scope.shakerWithSteps = false;
                $scope.shakerWritable = false;
                $scope.shakerReadOnlyActions = true;
                $scope.inputDatasetProjectKey = resolvedDataset.projectKey;
                $scope.inputDatasetName = resolvedDataset.id;
                $scope.inputDatasetSmartName = $scope.insight.params.datasetSmartName;

                WT1.event('shaker-explore-open');

                $scope.shaker = $scope.insight.params.shakerScript;
                $scope.shaker.origin = 'DATASET_EXPLORE';
                if ($scope.origInsight) {
                    $scope.origInsight.params.shakerScript.origin = 'DATASET_EXPLORE';
                }

                $scope.shakerState.writeAccess = true;
                if ($scope.insight.params.shakerState) {
                    Object.entries($scope.insight.params.shakerState).forEach(([key, value]) => $scope.shakerState[key] = value);
                }

                if ($scope.tile) {
                    $scope.shaker.$headerOptions = $scope.tile.tileParams;
                } else {
                    $scope.shaker.$headerOptions = {
                        showName: true,
                        showMeaning: true,
                        showDescription: true,
                        showCustomFields: true,
                        showProgressBar: true
                    };
                }

                $scope.fixupShaker();
                if ($scope.origInsight) {
                    $scope.fixupShaker($scope.origInsight.params.shakerScript);
                }
                $scope.requestedSampleId = null;

                $scope.insight.$tileFilteringInformation = {};
                const initialFilters = angular.copy($scope.shaker.explorationFilters);

                $scope.getDataSpec = () => {
                    return {
                        datasetProjectKey: $scope.inputDatasetProjectKey,
                        datasetName: $scope.inputDatasetName,
                        copyScriptFromExplore: true,
                        copySelectionFromScript: true,
                        sampleSettings: $scope.shaker.explorationSampling,
                        engineType: 'LINO'
                    };
                };

                $scope.fetchColumnsSummary = function(projectKey) {
                    if (!projectKey) {
                        projectKey = $scope.inputDatasetProjectKey;
                    }

                    return DashboardPageUtils.getColumnSummary(projectKey, $scope.getDataSpec())
                        .then(data => $scope.makeUsableColumns(data));
                };

                function watchLoadingState() {
                    if (unregisterLoadingStateWatcher) {
                        return;
                    }
                    unregisterLoadingStateWatcher = $scope.$watch('loaded', (nv) => {
                        if (nv === true) {
                            unregisterLoadingStateWatcher(); // remove watch
                            unregisterLoadingStateWatcher = null;
                            $scope.applySlideFilters();
                        }
                    });
                }

                function watchVisibilityChange() {
                    if (unregisterVisibilityWatcher) {
                        return;
                    }
                    unregisterVisibilityWatcher = $scope.$watch('hook.visibleStates[tile.$tileId]', function(nv) {
                        if (nv === true) {
                            unregisterVisibilityWatcher(); // remove watch
                            unregisterVisibilityWatcher = null;
                            $scope.applySlideFilters();
                        }
                    });
                }

                const handleLoadingFailure = (error) => {
                    if (!_.isObject(error) || !error.data || !error.data.detailedMessage) {
                        Logger.error(`Failed to load dataset  insight ${$scope.insight.id ? $scope.insight.id : ''}`, error);
                        error = {
                            data:{
                                detailedMessage: translate('DASHBOARD.FILTERABLE_INSIGHTS.ERRORS.LOADING_FAILED', 'error occurred while loading insight')
                            }
                        };
                        // In dataset tile context, handle possible client side error to avoid being stuck in loading state
                        if ($scope.setDashboardTileError) {
                            $scope.setDashboardTileError(error.data);
                        }
                        return;
                    }

                    const { data, status, headers, config, statusText, xhrStatus } = error;
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                    if ($scope.setDashboardTileError) {
                        $scope.setDashboardTileError(data, status, headers, config, statusText);
                    }
                    if ($scope.onRefreshFutureFailed) {
                        $scope.onRefreshFutureFailed(data, status, headers);
                    }
                };


                $scope.applySlideFilters = () => {
                    if (!$scope.loading && !$scope.loaded) {
                        watchLoadingState();
                        return;
                    }
                    if (!DashboardPageUtils.isTileVisibleOrAdjacent($scope.tile.$tileId, $scope.hook)) {
                        watchVisibilityChange();
                        return;
                    }
                    if ($scope.usableColumns == null) {
                        $scope.fetchColumnsSummary($stateParams.projectKey).then(() => {
                            return $scope.applySlideFilters();
                        }).catch(error => {
                            handleLoadingFailure(error);
                        });
                        return;
                    }
                    if (!$scope.deRegisterTableWatch) {
                        registerTableWatch();
                    }
                    $scope.globalFiltersData.usableColumns = $scope.usableColumns;

                    DashboardFilters.hasCaseInsensitiveColumnNames($scope.resolvedDataset.projectKey, $scope.resolvedDataset.datasetName, $stateParams.projectKey).then(hasCaseInsensitiveColumnNames => {
                        return DashboardFilters.applyFilters($scope.filters, $scope.filtersParams, $scope.globalFiltersData, null, hasCaseInsensitiveColumnNames);
                    }).catch((error) => {
                        handleLoadingFailure(error);
                    });
                };

                unregisterFiltersWatcher = $scope.$watch('filters', () => {
                    $scope.applySlideFilters();
                });

                function executeFiltersRequest(filters) {
                    const dashboardExplorationFilters = DashboardFilters.getDashboardExplorationFilters(filters);
                    $scope.insight.$shakerAndDashboardExplorationFilters = [...initialFilters, ...dashboardExplorationFilters];
                    return $scope.refreshTable(false);
                }

                $scope.globalFiltersData = {
                    insightId: $scope.insight.id,
                    applyFilters: executeFiltersRequest,
                    tileFilteringInformation: angular.copy($scope.insight.$tileFilteringInformation),
                    sourceDataset: $scope.insight.params.datasetSmartName,
                    sampleSettings: DatasetChartsUtils.makeSelectionFromScript($scope.insight.params.shakerScript),
                    pageId: $stateParams.pageId
                };

                // globalFiltersData.tileFilteringInformation is mutated by applyFilters, and we need to update insight.$tileFilteringInformation reference to trigger Angular's digest cycle.
                $scope.$watch('globalFiltersData.tileFilteringInformation', tileFilteringInformation => {
                    $scope.insight.$tileFilteringInformation = angular.copy(tileFilteringInformation);
                }, true);

                const registerTableWatch = () => {
                    $scope.deRegisterTableWatch = $scope.$watch('table', (table) => {
                        if (table) {
                            DashboardFilters.onApplyFiltersSuccess($scope.globalFiltersData, $scope.filtersParams, table);
                            if ($scope.tile) {
                                $scope.tile.$allFilteredOut = table.totalKeptRows === 0;
                            }
                        }
                    }, true);
                };

                $scope.$watch('shakerState.activeView', (nv, ov) => {
                    if (nv === 'table' && ov && nv !== ov) {
                        $scope.refreshTable(false);
                    }
                });

                $scope.$watch('table', (newValue) => {
                    if (!newValue) {
                        return;
                    }
                    setSamplingDetails(newValue);
                });

                $scope.getSamplingSummaryMessage = (clickActionMessage) => {
                    let samplingMessage = `<i class="dku-icon-warning-fill-16 tooltip-icon"></i> ${translate('DASHBOARD.TILES.HEADER.TOOLTIP.DATA_SAMPLING_NOT_REPRESENTATIVE', 'Data sampling can be misrepresentative')}</br>`;
                    if (clickActionMessage) {
                        samplingMessage += `${clickActionMessage}</br>`;
                    }
                    samplingMessage += `${$scope.getSampleDesc()}`;
                    return samplingMessage;
                };

                const setSamplingDetails = (table) => {
                    if (!$scope.tile) {
                        return;
                    }
                    const sampleMetadata = table.sampleMetadata;
                    if (sampleMetadata && !sampleMetadata.sampleIsWholeDataset) {
                        $scope.tile.$sampleMetadata = sampleMetadata;
                        $scope.tile.$samplingSummaryMessage = $scope.getSamplingSummaryMessage();
                        $scope.tile.$clickableSamplingSummaryMessage = $scope.getSamplingSummaryMessage(translate('DASHBOARD.TILES.HEADER.TOOLTIP.CLICK_TO_EDIT', 'Click to edit insight'));
                    }
                };
            }
        };
    });

    app.directive('datasetTableInsightTileParams', function(SmartId, DataikuAPI) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_tile_params.html',
            scope: {
                tile: '=',
                insight: '='
            },
            link: function($scope, element, attrs) {
                const resolvedDataset = SmartId.resolve($scope.insight.params.datasetSmartName);
                DataikuAPI.datasets.getFullInfo($scope.insight.projectKey, resolvedDataset.projectKey, resolvedDataset.id).noSpinner().success(function(data) {
                    $scope.datasetFullInfo = data;
                }).error(setErrorInScope.bind($scope));

                if (!['SEARCH', 'EXPLORE'].includes($scope.tile.tileParams.viewKind)) {
                    $scope.tile.tileParams.viewKind = 'EXPLORE';
                }

                $scope.onViewKindChange = function() {
                    if ($scope.tile.tileParams.viewKind === 'SEARCH') {
                        $scope.tile.clickAction = 'DO_NOTHING';
                    }
                };
            }
        };
    });

    app.directive('datasetTableInsightCreateForm', function(DataikuAPI, translate) {
        return {
            templateUrl: '/templates/dashboards/insights/dataset_table/dataset_table_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {

                $scope.hook.beforeSave = function(resolve, reject) {
                    DataikuAPI.explores.get($scope.insight.projectKey, $scope.insight.params.datasetSmartName)
                        .success(function(data) {
                            $scope.insight.params.shakerScript = data.script;
                            resolve();
                        })
                        .error(function(data, status, headers, config, statusText) {
                            reject(arguments);
                        });
                };

                $scope.hook.defaultName = translate('INSIGHTS.DATASET_TABLE.TILE_PARAMS_NAME', 'Dataset table');
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = translate(
                        'INSIGHTS.DATASET_TABLE.NAME_WITH_LABEL',
                        nv.label + ' table',
                        {
                            label: nv.label
                        }
                    );
                });
            }
        };
    });
})();
