import os.path

from dataiku.core import default_project_key
from dataiku.core import intercom

def get_workload_local_folder_path():
    """
    Returns a persistent folder for the current workload (Webapp, Code Agent or Agent Tool)

    This can only be called if the workload is running locally in DSS.

    Even if the workload is restarted, the same folder will be returned.

    If this call succeeds, you are guaranteed to be able to write into the folder,
    and the folder is guaranteed to exist

    Any synchronization for writing into the folder is user's responsibility.
    """

    api_resp = intercom.backend_json_call("workload-folder/get", data={
        "projectKey":  default_project_key()
    })
    return _check_and_return_path(api_resp)


def get_webapp_workload_local_folder_path(webapp_id):
    """
    Returns the path for the persistent workload folder of a webapp in the current project

    If this call succeeds, you are guaranteed to be able to write into the folder,
    and the folder is guaranteed to exist

    Any synchronization for writing into the folder is user's responsibility.
    """

    api_resp = intercom.backend_json_call("workload-folder/get-for-webapp", data={
        "projectKey":  default_project_key(), "webappId" : webapp_id
    })
    return _check_and_return_path(api_resp)


def get_agent_workload_local_folder_path(agent_id, version_id):
    """
    Returns the path for the persistent workload folder of an agent in the current project

    If this call succeeds, you are guaranteed to be able to write into the folder,
    and the folder is guaranteed to exist

    Any synchronization for writing into the folder is user's responsibility.
    """

    api_resp = intercom.backend_json_call("workload-folder/get-for-agent", data={
        "projectKey":  default_project_key(), "agentId" : agent_id, "versionId": version_id
    })
    return _check_and_return_path(api_resp)


def get_agent_tool_workload_local_folder_path(agent_tool_id):
    """
    Returns the path for the persistent workload folder of an agent tool in the current project

    If this call succeeds, you are guaranteed to be able to write into the folder,
    and the folder is guaranteed to exist

    Any synchronization for writing into the folder is user's responsibility.
    """

    api_resp = intercom.backend_json_call("workload-folder/get-for-agent-tool", data={
        "projectKey":  default_project_key(), "agentToolId" : agent_tool_id
    })
    return _check_and_return_path(api_resp)

def get_project_scope_workload_local_folder_path(prefix):
    """
    Returns the path for the persistent workload folder at a project scope with a given prefix

    If this call succeeds, you are guaranteed to be able to write into the folder,
    and the folder is guaranteed to exist

    Any synchronization for writing into the folder is user's responsibility.
    """

    api_resp = intercom.backend_json_call("workload-folder/get-project-scope", data={
        "projectKey":  default_project_key(), "prefix" : prefix
    })
    return _check_and_return_path(api_resp)



def _check_and_return_path(api_resp):
    folder_path = api_resp["path"]

    if not os.path.exists(folder_path):
        raise Exception("Workload folder path unexpectedly does not exist. Did you call this from a containerized execution context?")

    return folder_path