class Placeholder(object):
    def __init__(self, tagname, placeholder_type, is_closing, paragraph, start, end):
        self.tagname = tagname
        self.placeholder_type = placeholder_type
        self.is_closing = is_closing
        self.paragraph = paragraph
        self.start = start
        self.end = end
        self.is_conditional = self.placeholder_type == 'IF'

    def extract_name(self):
        """
        Extract the name of the placeholder.
        For a basic placeholder, this is just the tagname.
        For conditional placeholder, we extract the part before the equal/not equal sign.
        For a foreach placeholder, we extract "foreach $variable"
        :return: the name of the placeholder
        """
        if self.is_conditional and not self.is_closing:
            if "==" in self.tagname:
                return self.tagname.split('==')[0].strip()
            else:
                return self.tagname.split('!=')[0].strip()
        elif self.placeholder_type == 'FOREACH' and not self.is_closing:
            return self.tagname.split(' in ')[0].strip()
        else:
            return self.tagname

    def requires_nesting(self):
        return self.placeholder_type == 'FOREACH' or self.placeholder_type == 'IF'
    
    def __repr__(self):
        closing_msg = 'closing' if self.is_closing else 'opening'
        return "<" + self.placeholder_type + " - " + str(self.extract_name()) + " - " + closing_msg + ">"

    def to_java_repr(self):
        """
        Build the object that will be send to the java server to describe this placeholder
        """
        if self.placeholder_type == 'IF':
            op = "EQUAL" if "==" in self.tagname else "NOT_EQUAL"
            return {
                "type": self.placeholder_type,
                "op": op,
                "lefthand": self.extract_name(),
                "righthand": self.tagname.split('==')[1].strip() if op == "EQUAL" else self.tagname.split('!=')[1].strip(),
                "children": []
            }
        elif self.placeholder_type == 'FOREACH':
            return {
                "type": self.placeholder_type,
                "variable": self.tagname.split(' in ')[0].strip(),
                "iterable": self.tagname.split(' in ')[1].strip(),
                "children": []
            }
        else:
            return {
                "type": self.placeholder_type,
                "name": self.extract_name()
            }


class HeaderPlaceholder(Placeholder):
    """
    Placeholders with their location in the header
    """
    def __init__(self, tagname, placeholder_type, is_closing, paragraph, start, end, section_number):
        Placeholder.__init__(self, tagname, placeholder_type, is_closing, paragraph, start, end)
        self.section_number = section_number

class FooterPlaceholder(Placeholder):
    """
    Placeholders with their location in the footer
    """
    def __init__(self, tagname, placeholder_type, is_closing, paragraph, start, end, section_number):
        Placeholder.__init__(self, tagname, placeholder_type, is_closing, paragraph, start, end)
        self.section_number = section_number


class TablePlaceholder(object):
    """Placeholders that are contains inside a single Cell in a table."""

    def __init__(self, table, row_idx, col_idx, placeholders):
        self.table = table
        self.row_idx = row_idx
        self.col_idx = col_idx
        self.placeholders = placeholders

    def __repr__(self):
        return "<TablePlaceholder in [" + str(self.row_idx) + ", " + str(self.col_idx) + "] => " + str(self.placeholders) + " >"

    def get_placeholder_names(self):
        return [placeholder.extract_name() for placeholder in self.placeholders]

    def to_java_repr(self):
        """
        Build the object that will be send to the java server to describe this placeholder
        """
        return {
            "type": 'TABLE',
            "children": [inner.to_java_repr() for inner in self.placeholders]
        }

class BlockPlaceholder(object):
    """
    Block of two placeholders:
    {my.placeholder}
    format information
    {/my.placeholder}
    """

    def __init__(self, start, end):
        self.start = start
        self.end = end

    def extract_name(self):
        return self.start.extract_name()

    def __repr__(self):
        return "<BlockPlaceholder " + str(self.start) + ">"

    def to_java_repr(self):
        """
        Build the object that will be send to the java server to describe this placeholder
        """
        return self.start.to_java_repr()

class NestedPlaceholder(BlockPlaceholder):
    def __init__(self, start, end, children):
        BlockPlaceholder.__init__(self, start, end)
        self.children = children
    
    def __repr__(self):
        return "<NestedPlaceholder " + str(self.start) + " - " + str(self.children) + ">"

    def to_java_repr(self):
        """
        Build the object that will be send to the java server to describe this placeholder
        """
        start = self.start.to_java_repr()
        start["children"] = [child.to_java_repr() for child in self.children]
        return start
