from copy import deepcopy
from typing import Any

"""
[sc-282651]
LLMs trained with the Harmony format (like gpt-oss) struggle with tool JSON
schemas having nested definitions like "$refs" fields.

In this case, inlining the references ($ref) from the definitions ($defs) seems
to work experimentally -- although it may break in the future.

Inspired from https://github.com/langchain-ai/langchain/blob/3356d0555725c3e0bbb9408c2b3f554cad2a6ee2/libs/core/langchain_core/utils/json_schema.py
"""


def inline_schema_references(schema: dict[str, Any]):
    schema = _inline_object(schema, root_schema=schema)
    # clean up inlined top-level definitions
    for field in ["definitions", "$defs"]:
        if field in schema:
            schema.pop(field, None)

    return schema


def _inline_object(
    obj: Any,
    root_schema: dict[str, Any],
    already_inlined_refs: set[str] | None = None
):
    if already_inlined_refs is None:
        already_inlined_refs = set()

    # case: dict
    if isinstance(obj, dict):
        if "$ref" not in obj:
            return {
                key: _inline_object(value, root_schema, already_inlined_refs)
                for key, value in obj.items()
            }

        ref_path = obj["$ref"]
        if ref_path in already_inlined_refs:
            # [!] circular reference detected
            # copy the other properties, omitting this $ref
            return {
                key: _inline_object(value, root_schema, already_inlined_refs)
                for key, value in obj.items()
                if key != "$ref"
            }

        # resolve the type given its path
        already_inlined_refs.add(ref_path)
        resolved_type = _resolve_against_schema(ref_path, root_schema)
        inlined_type = _inline_object(resolved_type, root_schema, already_inlined_refs)
        already_inlined_refs.remove(ref_path)

        if len(obj) <= 1:
            # no extra properties to carry over
            return inlined_type

        # inline the object extra properties and merge them with the resolved type
        merged_type = {}
        if isinstance(inlined_type, dict):
            merged_type.update(inlined_type)

        for key, value in obj.items():
            if key == "$ref":
                continue

            merged_type[key] = _inline_object(value, root_schema, already_inlined_refs)

        return merged_type

    # case: list
    if isinstance(obj, list):
        return [
            _inline_object(item, root_schema, already_inlined_refs)
            for item in obj
        ]

    # case: simple
    return obj


def _resolve_against_schema(path: str, schema: dict[str, Any]) -> list | dict:
    """Resolves the reference given its path against the provided schema.
    Returns a deep copy of the resolved object."""
    components = path.split("/")
    if components[0] != "#":
        raise ValueError(f"Invalid reference '{path}'")

    obj: list | dict = schema
    for comp in components[1:]:
        if comp in obj:
            if isinstance(obj, dict):
                obj = obj[comp]
                continue

        elif comp.isdigit():
            index = int(comp)
            if isinstance(obj, list) and 0 <= index < len(obj):
                obj = obj[index]  # list
                continue

            elif isinstance(obj, dict) and index in obj:
                obj = obj[index]  # dict
                continue

        raise KeyError(f"Impossible reference '{path}'")

    return deepcopy(obj)
