import logging

from dataiku.llm.python.types import ChatMessage
from .. import NextBlock, BlockHandler
from ..utils import interpolate_cel

logger = logging.getLogger("dku.agents.blocks_graph")

class EmitOutputBlockHandler(BlockHandler):
    def __init__(self, turn, sequence_context, block_config):
        super().__init__(turn, sequence_context, block_config)

    def process_stream(self, trace):
        logger.info("Emit output block starting with config %s" % self.block_config)

        template_type = self.block_config.get("templateType") or self.block_config.get("templateMode") or "CEL_EXPANSION"
        template = self.block_config.get("template")
        if template is None:
            template = self.block_config.get("expression", "")

        if template_type in ("CEL_EXPANSION", "CEL"):
            engine = self.standard_cel_engine()
            output_value = interpolate_cel(engine, template)
        elif template_type == "JINJA":
            from jinja2 import Template
            template_context = self.jinja_template_context()
            output_value = Template(template).render(**template_context)
        else:
            raise ValueError("Unsupported templateType for EMIT_OUTPUT: %s" % template_type)

        self.sequence_context.last_text_output = str(output_value)
        yield {
            "chunk": {
                "text": str(output_value) + ("\n" if self.agent.config.get("newLineAfterBlockOutput", True) else "")
            }
        }

        if self.block_config.get("addToMessages") and str(output_value):
            assistant_message: ChatMessage = {
                "role": "assistant",
                "content": str(output_value)
            }
            self.sequence_context.generated_messages.append(assistant_message)
            logger.info("Response added to generated messages")

        yield NextBlock(id=self.block_config.get("nextBlock"))
