import base64
import logging
import dataiku
import json

from dataikuapi.dss.document_extractor import ManagedFolderDocumentRef
from .. import NextBlock, BlockHandler
from ..utils import interpolate_cel

logger = logging.getLogger("dku.agents.blocks_graph")


class GenerateArtifactBlockHandler(BlockHandler):
    def __init__(self, turn, sequence_context, block_config):
        super().__init__(turn, sequence_context, block_config)

    def process_stream(self, trace):
        logger.info("Generate artifact block starting with config %s", self.block_config)

        template_type = self.block_config.get("templateType")
        output_format = self.block_config.get("outputFormat")

        if not template_type:
            raise Exception("templateType must be specified in GenerateArtifact block")
        if not output_format:
            raise Exception("Output format must be specified in GenerateArtifact block")

        artifact_part = {}

        if template_type == "CEL_EXPANSION" or template_type == "JINJA":
            rendered_output = self._render_text_output()

            if output_format == "TEXT":
                artifact_part["type"] = "SIMPLE_DOCUMENT"
                artifact_part["textSnippet"] = rendered_output
            elif output_format == "MARKDOWN":
                artifact_part["type"] = "SIMPLE_DOCUMENT"
                artifact_part["markdownSnippet"] = rendered_output
            # elif output_format == "PDF":
            #     # TODO: Replace this placeholder with a markdown-to-pdf conversion library.
            #     placeholder_pdf_bytes = b"BLABLA"
            #     placeholder_pdf_b64 = base64.b64encode(placeholder_pdf_bytes).decode("ascii")
            #     artifact_part["type"] = "DATA_INLINE"
            #     artifact_part["mimeType"] = "application/pdf"
            #     artifact_part["dataBase64"] = placeholder_pdf_b64
            else:
                raise ValueError(f"Unsupported output format {output_format} for {template_type} template")

        else:
            output_filename = self.block_config.get("outputFilename")
            if not output_filename:
                if output_format == "PDF":
                    output_filename = "generated_document.pdf"
                elif output_format == "DOCX":
                    output_filename = "generated_document.docx"

            rendered_bytes, output_document_ref = self._render_docx_template(output_filename)

            if output_document_ref:
                artifact_part["type"] = "FILE_BASED_DOCUMENT"
                artifact_part["fileRef"] = {
                    "folderId": output_document_ref.managed_folder_id,
                    "path": output_document_ref.file_path,
                }
            else:
                if output_format == "PDF":
                    artifact_part["type"] = "DATA_INLINE"
                    artifact_part["mimeType"] = "application/pdf"
                    artifact_part["dataBase64"] = base64.b64encode(rendered_bytes).decode("ascii")
                    artifact_part["filename"] = output_filename
                elif output_format == "DOCX":
                    artifact_part["type"] = "DATA_INLINE"
                    artifact_part["mimeType"] = "application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                    artifact_part["dataBase64"] = base64.b64encode(rendered_bytes).decode("ascii")
                    artifact_part["filename"] = output_filename
                else:
                    raise ValueError(f"Unsupported output format {output_format} for DOCX template")

        artifact = {
            "parts": [artifact_part]
        }

        yield {"chunk": {"artifacts": [artifact]}}
        yield NextBlock(id=self.block_config.get("nextBlock"))

    def _render_docx_template(self, output_filename):
        api_client = dataiku.api_client()
        project = api_client.get_default_project()

        managed_folder_ref = self.block_config.get("managedFolderRef")
        template_filename = self.block_config.get("templateFilename") or "template.docx"
        if not managed_folder_ref:
            raise ValueError("Missing managedFolderRef for DOCX template")
        if not template_filename:
            raise ValueError("Missing templateFilename for DOCX template")
        template_document_ref = ManagedFolderDocumentRef(template_filename, managed_folder_ref)

        output_format = self.block_config.get("outputFormat")
        output_managed_folder_ref = self.block_config.get("outputManagedFolderRef")
        if output_managed_folder_ref:
            output_document_ref = ManagedFolderDocumentRef(output_filename, output_managed_folder_ref)
        else:
            output_document_ref = None

        template_context = {
            "context": self.turn.current_merged_context,
            "state": self.turn.state(),
            "scratchpad": self.sequence_context.scratchpad,
            "last_output": self.sequence_context.last_text_output,
        }

        response = project.render_document_template(
            "DOCX_JINJA",
            data=template_context,
            template_ref=template_document_ref,
            output_format=output_format,
            output_ref=output_document_ref,
        )

        if not response.ok:
            if response.error.get("code") == "ERR_CODEENV_MISSING_INTERNAL_ENV":
                raise Exception("The internal document templating code environment must be created to use Docx templating")
            raise ValueError(f"Document templating failed with error: %s" % (json.dumps(response.error)))

        if output_document_ref:
            return None, output_document_ref
        else:
            return response.stream().read(), None

    def _render_text_output(self):
        template_type = self.block_config.get("templateType")
        template_str = self.block_config.get("template")

        if not template_str:
            raise ValueError("template must be specified in GenerateArtifact block")

        if template_type == "JINJA":
            from jinja2 import Template
            template_context = {
                "context": self.turn.current_merged_context,
                "state": self.turn.state(),
                "scratchpad": self.sequence_context.scratchpad,
                "last_output": self.sequence_context.last_text_output,
            }
            return Template(template_str).render(**template_context)

        elif template_type in ("CEL", "CEL_EXPANSION"):
            engine = self.standard_cel_engine()
            ret = interpolate_cel(engine, template_str)
            return ret
