import os, json, shutil
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid_gid

from block_utils import LibLocationPathReplacer, build_generate_python_codenv_script

class VsCodeCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, "vscode-entrypoint.sh")
        return entrypoint_path

    def _get_port(self):
        return self.config.get("port", 8080)

    def build_spec(self, spec, env):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        open_in_path = self.config.get("openInPath", "/home/dataiku/workspace")
        settings_path = self.config.get("settingsPath", "/home/dataiku/.local/share/code-server/settings")
        machine_settings_path = self.config.get("machineSettingsPath", "/home/dataiku/.local/share/code-server/settings/Machine")
        extensions_path = self.config.get("extensionsPath", "/home/dataiku/.local/share/code-server/extensions")

        # when you bump these versions, do not forget to adapt the desc of correspongind advanced params
        codeserver_version = "4.106.3"
        copilot_ext_version = "v0.33.5"
        python_ext_version = "2025.4.0"
        jupyter_ext_version = "2025.7.0"
        additional_extensions = []
        if self.config.get("showAdvancedParams", False):
            codeserver_version = self.config.get("codeServerVersion") or codeserver_version
            python_ext_version = self.config.get("pythonExtensionVersion") or python_ext_version
            jupyter_ext_version = self.config.get("jupyterExtensionVersion") or jupyter_ext_version
            additional_extensions = self.config.get("additionalExtensions") or []

            # we want to be sure to have a value for copilot version as the latest will not be compatible with the latest code-server
            # code-server is almost always behind vscode, and copilot is tightly coupled to the vscode engine version.
            # engine requirement can be checked at https://github.com/microsoft/vscode-copilot-chat/blob/v0.35.2/package.json#L26
            copilot_ext_version = self.config.get("githubCopilotExtensionVersion") or copilot_ext_version

        ignoreTLS = 'export NODE_TLS_REJECT_UNAUTHORIZED=1' if self.config.get("ignoreTLS", False) else ""

        # replace the lib locations in settings_path and open_in_path
        replacer = LibLocationPathReplacer(spec)
        open_in_path = replacer.replace_variable_by_path(open_in_path)
        settings_path = replacer.replace_variable_by_path(settings_path)
        machine_settings_path = replacer.replace_variable_by_path(machine_settings_path)
        extensions_path = replacer.replace_variable_by_path(extensions_path)

        # prepare the code env for the "default" kernel
        py_dss_env_name = 'py39-dss'
        py_dss_env_label = 'Pandas 2.2 (Python 3.9)'
        kernel_packages = '"pandas>=2.2.2,<2.3" "numpy>=2,<2.1" "python-dateutil>=2.8.2,<3" "urllib3<2" "requests<3" "decorator==5.1.1" "ipykernel==6.23.3" "ipython>=8.12,<8.13" "ipython_genutils==0.2.0" "jupyter_client==6.1.12" "jupyter_core==4.12.0" "pexpect==4.8.0" "pickleshare==0.7.5" "ptyprocess==0.7.0" "pyzmq==23.2.1" "simplegeneric==0.8.1" "tornado>=6.3,<6.4" "traitlets==5.9.0"'
        py_dss_pyenv_path = "/opt/dataiku/" + py_dss_env_name
        generate_py_dss_codenv = build_generate_python_codenv_script("KERNEL", kernel_packages, py_dss_pyenv_path, "python3.9", env.get("globalCodeEnvsExtraSettings"))

        code_assistant_installation = ''
        if self.config.get('installCodeAssistant', True):
            shutil.copy(os.environ['DKUINSTALLDIR'] + '/resources/code-studios/aicodeassistant.vsix', env["buildDir"])
            code_assistant_installation = """

# Code Assistant extension

USER root
WORKDIR /opt/dataiku

COPY aicodeassistant.vsix /opt/dataiku/
RUN chown dataiku:dataiku /opt/dataiku/aicodeassistant.vsix && chmod +x /opt/dataiku/aicodeassistant.vsix

# USER dataiku
USER {uid_gid}
WORKDIR /home/dataiku

RUN code-server --extensions-dir /home/dataiku/.local/share/code-server/local-extensions --install-extension /opt/dataiku/aicodeassistant.vsix
""".format(uid_gid=get_dataiku_user_uid_gid())

        github_copilot_installation=""
        if self.config.get("installGithubCopilot", True):
            github_copilot_installation = """
USER root
# Install node & npm
RUN dnf module install nodejs:22/common -y \
    && npm install -g vsce \
    && dnf clean all

WORKDIR /opt/dataiku
RUN git clone https://github.com/microsoft/vscode-copilot-chat.git \
        && (cd vscode-copilot-chat \
        && git checkout {github_copilot_version} \
        && npm install \
        && npm run build \
        && vsce package --out github-copilot.vsix \
        && cp github-copilot.vsix /opt/dataiku) \
        && rm -rf vscode-copilot-chat \
        && chown dataiku:dataiku /opt/dataiku/github-copilot.vsix

# USER dataiku
USER {uid_gid}
WORKDIR /home/dataiku

RUN code-server --extensions-dir /home/dataiku/.local/share/code-server/local-extensions --install-extension /opt/dataiku/github-copilot.vsix
""".format(uid_gid=get_dataiku_user_uid_gid(),
           github_copilot_version=copilot_ext_version)

        extensions = []
        if python_ext_version is not None and python_ext_version != 'none':
            extensions.append('ms-python.python@{python_ext_version}'.format(python_ext_version=python_ext_version))
        if jupyter_ext_version is not None and jupyter_ext_version != 'none':
            extensions.append('ms-toolsai.jupyter@{jupyter_ext_version}'.format(jupyter_ext_version=jupyter_ext_version))
        extensions = extensions + additional_extensions
        if len(extensions) > 0:
            extensions_args = ' '.join(['--install-extension {extension}'.format(extension=extension) for extension in extensions])
            extensions_installation = 'RUN code-server --extensions-dir /home/dataiku/.local/share/code-server/local-extensions {extensions_args}'.format(extensions_args=extensions_args)
        else:
            extensions_installation = ''

        # add the entrypoint script in the buildir
        entrypoint_script = """
#! /bin/bash

mkdir -p {extensions_path}
# stash the json away so that code-server regenerates the extension list
mv {extensions_path}/extensions.json /home/dataiku/old.extensions.json
# symlink to default extensions needs to be remade each and every time.
for i in $(ls -d /home/dataiku/.local/share/code-server/local-extensions/*/ | while read name; do basename $name; done)
do
    echo "link extension $i"
    ln -s /home/dataiku/.local/share/code-server/local-extensions/$i {extensions_path}/$i
done
# same for the venvs that live in /opt
echo "link codenvs"
ln -s /opt/dataiku/python-code-envs /home/dataiku/template-python-code-envs
ln -s /opt/dataiku /home/dataiku/dataiku-python-code-envs
# and the link to Machine settings, since symlinks are not synced
echo "link machine settings"
ln -s {machine_settings_path} {settings_path}/Machine

# Initial config set-up. Should only run once, on the Code Studio first start. Afterwards, we'll get it from the config sync.
# these settings are user-specific, we expect users to have their own, ideally stored in user-versioned
if [ ! -d "{settings_path}/User" ]; then
    echo "init user settings"
    mkdir -p {settings_path}/User
    echo '{{"query":{{"folder":"{open_in_path}"}},"lastVisited":{{"url":"{open_in_path}","workspace":false}}}}' > {settings_path}/coder.json
    # default env and settings
    cat << 'EOF' > {settings_path}/User/settings.json
{{
 "workbench.colorTheme":"Visual Studio Dark",
 "git.ignoreLegacyWarning":true,
 "telemetry.enableTelemetry": false,
 "update.mode": "none",
 "extensions.autoCheckUpdates": false,
 "extensions.autoUpdate": false
}}
EOF
fi

# Initial config set-up. Should only run once, on the Code Studio first start. Afterwards, we'll get it from the config sync.
# these settings are machine-specific, ideally stored in code_studio-versioned
if [ ! -d "{machine_settings_path}" ]; then
    echo "init machine settings"
    mkdir -p {machine_settings_path}
    # Not sure how to handle the migration
    rm -r {settings_path}/Machine; ln -s {machine_settings_path} {settings_path}/Machine
    echo '{{"query":{{"folder":"{open_in_path}"}},"lastVisited":{{"url":"{open_in_path}","workspace":false}}}}' > {machine_settings_path}/coder.json
    # default env and settings
    cat << 'EOF' > {machine_settings_path}/settings.json
{{
 "workbench.secondarySideBar.defaultVisibility": "hidden",
 "chat.disableAIFeatures": true,
 "terminal.integrated.defaultProfile.linux": "bash",
 "python.experiments.enabled": false,
 "python.venvFolders": ["template-python-code-envs", "dataiku-python-code-envs"],
 "python.defaultInterpreterPath": "/home/dataiku/dataiku-python-code-envs/py39-dss/bin/python",
 "launch": {{
    "version": "0.2.0",
    "configurations": [
        {{
            "name": "Python 3.9 + dku APIs",
            "type": "python",
            "python": "/opt/dataiku/py39-dss/bin/python",
            "request": "launch",
            "program": "${{file}}",
            "console": "integratedTerminal"
        }},
        {{
            "name": "Default Python + dku APIs",
            "type": "python",
            "python": "/opt/dataiku/pyenv/bin/python",
            "request": "launch",
            "program": "${{file}}",
            "console": "integratedTerminal"
        }},
        {{
            "name": "Python: Current Interpreter",
            "type": "python",
            "request": "launch",
            "program": "${{file}}",
            "console": "integratedTerminal"
        }}
EOF
    # user envs
    if [ -d /home/dataiku/template-python-code-envs ]; then
        for i in $(ls /home/dataiku/template-python-code-envs); do
            cat << EOF >> {machine_settings_path}/settings.json
        ,
        {{
            "name": "$i",
            "type": "python",
            "python": "/home/dataiku/template-python-code-envs/$i/bin/python",
            "request": "launch",
            "program": "\${{file}}",
            "console": "integratedTerminal"
        }}
EOF
        done
    fi
    # close everything
    cat << EOF >> {machine_settings_path}/settings.json
        ]
    }}
}}
EOF
fi

if [ $DKU_CODE_STUDIO_IS_PUBLIC_PORT_{port} = "1" ]; then
    BIND_ADDR=0.0.0.0
else
    BIND_ADDR=127.0.0.1
fi

# required to overwrite some env vars that are copied from the build env and prevents some VS-CODE features to work properly
unset ELECTRON_NO_ATTACH_CONSOLE
unset ELECTRON_RUN_AS_NODE
unset XDG_RUNTIME_DIR
export SHELL=/bin/bash
unset JUPYTER_DATA_DIR
unset IPYTHONDIR
unset JUPYTER_RUNTIME_DIR
unset JUPYTER_CONFIG_DIR
{ignoreTLS}
code-server --user-data-dir {settings_path} --extensions-dir {extensions_path} --auth none --bind-addr $BIND_ADDR:{port} --trusted-origins "*" 2>&1 | tee ${{TMPDIR:-/tmp}}/code-server.log
""".format(settings_path=settings_path,
           machine_settings_path=machine_settings_path,
           extensions_path=extensions_path,
           open_in_path=open_in_path,
           port=port,
           ignoreTLS=ignoreTLS)
        with open(os.path.join(env["buildDir"], "vscode-entrypoint.sh"), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))

        # add the openfile script in the buildir
        openfile_script = """
#! /bin/bash

unset TMPDIR
openfile() {
    countExec=1
    until [ $countExec -gt 15 ]
        do
            # sleeping here as the socket might not be ready yet and haven't found a proper way to find the socket and make sure it's ready
            sleep 1.5
            code-server --user-data-dir ##SETTINGS_PATH## -r "$1/$2" 2>&1 | grep error
            if [ $? != 0 ]
                then
                    echo "Request n°$countExec to open file $1/$2 in VSCode succeeded!"
                    break
            fi
            echo "Request n°$countExec to open file $1/$2 in VSCode failed!"
            ((countExec=countExec+1))
        done
}
openfile "$1" "$2"
"""
        openfile_script = openfile_script.replace("##SETTINGS_PATH##", settings_path)
        with open(os.path.join(env["buildDir"], "vscode-openfile.sh"), "wb") as f:
            f.write(openfile_script.encode("utf8"))

        # the dockerfile addition

        # code server release: https://github.com/coder/code-server/releases/
        # ms python plugin: https://github.com/microsoft/vscode-python/releases/

        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### VS CODE BLOCK #####

USER root
WORKDIR /opt/dataiku

RUN curl -fOL https://github.com/coder/code-server/releases/download/v{codeserver_version}/code-server-{codeserver_version}-linux-amd64.tar.gz \
    && tar -xzf code-server-{codeserver_version}-linux-amd64.tar.gz \
    && rm code-server-{codeserver_version}-linux-amd64.tar.gz \
    && mv code-server-{codeserver_version}-linux-amd64 code-server \
    && chown -R dataiku:dataiku /opt/dataiku/code-server \
    && ln -s /opt/dataiku/code-server/bin/code-server /usr/local/bin/code-server

COPY vscode-entrypoint.sh {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}
COPY vscode-openfile.sh /opt/dataiku/vscode-openfile.sh
RUN chown dataiku:dataiku /opt/dataiku/vscode-openfile.sh && chmod +x /opt/dataiku/vscode-openfile.sh

# Default env Pandas 2.2 / Python 3.9 (just to demo that we can debug)
{generate_py_dss_codenv}
RUN source {py_dss_pyenv_path}/bin/activate && python3 -m ipykernel install --prefix pyenv-jupyter/ --name '{py_dss_env_name}' --display-name "{py_dss_env_label}"

# If R is present, install languageserver, in order for a user to be able to use the R plugin
RUN if [ -f /usr/bin/R -a ! -d /opt/dataiku/R/R.lib/languageserver ]; then /usr/bin/R --silent --slave -e "install.packages('languageserver', repos='https://cran.r-project.org/')"; fi

# fix permissions on folders VSCode might have created in the home dir
RUN find /home/dataiku/ -name .cache | while read line; do chown -R dataiku: $line; done
RUN find /home/dataiku/ -name .ipython | while read line; do chown -R dataiku: $line; done

#USER dataiku
USER {uid_gid}
WORKDIR /home/dataiku

# install to temporary folder, will be linked to user's dir on first launch, see vscode-entrypoint.sh
{extensions_installation}
{code_assistant_installation}
{github_copilot_installation}
""".format(entrypoint_path=entrypoint_path,
           code_assistant_installation=code_assistant_installation,
           github_copilot_installation=github_copilot_installation,
           codeserver_version=codeserver_version,
           extensions_installation=extensions_installation,
           generate_py_dss_codenv=generate_py_dss_codenv,
           py_dss_pyenv_path=py_dss_pyenv_path,
           py_dss_env_name=py_dss_env_name,
           py_dss_env_label=py_dss_env_label,
           uid_gid=get_dataiku_user_uid_gid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) +"/_static/src/browser/media/favicon.ico"
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {"label": "VS Code", "proxiedUrlSuffix": "/", "exposeHtml": True, "port": port, "readinessProbeUrl":readiness_probe_url}
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]

        if self.config.get("settingsPath", "") != "":
            # add exclusion rules to the sync, otherwise you bring back a ton of crap in DSS
            settings_path = self.config.get("settingsPath", "/home/dataiku/.local/share/code-server")

            # replace the lib locations in settings_path by their code name
            replacer = LibLocationPathReplacer(spec)
            settings_path = replacer.replace_path_by_variable(settings_path)

            # then remove the known sync locations to get the relative path
            settings_path = replacer.clear_variables(settings_path)

            spec['excludedFromSync'].append("vscode.lock")
            spec['excludedFromSync'].append("{settings_path}/extensions/".format(settings_path=settings_path))
            spec['excludedFromSync'].append("{settings_path}/local-extensions/".format(settings_path=settings_path))
            spec['excludedFromSync'].append("{settings_path}/logs/".format(settings_path=settings_path))
            spec['excludedFromSync'].append("{settings_path}/User/caches/".format(settings_path=settings_path))
            spec['excludedFromSync'].append("{settings_path}/CachedExtensions/".format(settings_path=settings_path))
            spec['excludedFromSync'].append("{settings_path}/CachedExtensionVSIXs/".format(settings_path=settings_path))
        return spec

    def build_creation(self, spec, env):
        return spec
