(function() {
'use strict';

const app = angular.module('dataiku.fm.base', ['dataiku.services', 'dataiku.filters', "dataiku.fm.dialogs"]);

function getParameters(location) {
    if (typeof location === 'undefined') {
        location = window.location;
    }
    let hashParams = new (function Params() {})();

    const urlParams = new URLSearchParams(window.location.search);
    let iterator = urlParams.entries();
    let result = iterator.next();

    while( !result.done ) {
        hashParams[result.value[0]] = result.value[1];
        result = iterator.next();
    }

    let hashArray = location.hash.substring(1).split('&');
    for (let i in hashArray) {
        let keyValPair = hashArray[i].split('=');
        hashParams[keyValPair[0]] = keyValPair[1];
    }
    return hashParams;
}

app.controller('RootRouteDispatchController', function($scope, $state, $location, TopNav) {
    window.location = '/instances/';
});

app.controller('FMBaseController', function($cacheFactory, $filter, $http, $injector, $state, $location, $modal, $q, $rootScope,
       $route, $scope, $controller, $stateParams, $templateCache, $timeout, $exceptionHandler, $httpParamSerializer, TopNav,
       Assert, Dialogs, ActivityIndicator, Throttle, FMAPI, ErrorReporting,
       CreateModalFromTemplate, localStorageService, ContextualMenu,
       LoggerProvider, Notification, WebSocketService, TrackingService, WT1, UserImageUrl, translate, FeatureFlagsService) {

    TopNav.setLocation(TopNav.HOME);

    $scope.FeatureFlagsService = FeatureFlagsService;

    const Logger = LoggerProvider.getLogger('FMBaseController');
    Logger.info("Starting Fleet Manager load");
    window.APIErrorLogger = LoggerProvider.getLogger("api.errors");

    $rootScope.translate = translate;
    $rootScope.wl = {
        productShortName: "Fleet Manager",
        productLongName: "Dataiku Fleet Manager"
    };
    // default version if we can not find one when getting configuration
    $rootScope.dssMajorVersion = "11";
    $rootScope.versionDocRoot = `https://doc.dataiku.com/dss/${$rootScope.dssMajorVersion}/`;
    $rootScope.apiDocRoot = `https://doc.dataiku.com/dss/api/${$rootScope.dssMajorVersion}/`;

    $scope.$on("$stateChangeStart", function(event, toState, toParams, fromState) {
        Logger.debug('State: '+((fromState && fromState.name)?fromState.name:'Unknown') + ' -> '+ ((toState && toState.name)?toState.name:'Unknown'), toParams);
    });

    // Check for unsaved changes in the page before leaving:
    window.addEventListener("beforeunload", function (event) {
        try {
            if (typeof window.dssHasDirtyThings == "function" && window.dssHasDirtyThings()) {
                var msg = 'Unsaved changes will be lost';
                event.returnValue = msg; //this string will not be displayed anyway
                return msg;
            }
        } catch (e){
            Logger.error("Failed to compute dirtiness. Let it go.", e);
        }
    });

    $scope.reflow = {};
    $scope.$on('reflow',function() {
        $scope.reflow = {};
    });

    Notification.registerEvent('websocket-status-changed',function(evt,data) {
        $scope.wsFail = false;
        $("body").removeClass("ws-disconnected");
        if(data.code === WebSocketService.ERROR_CODE.CONNECTION_FAILED) {
            $scope.wsFail = true;
        } else if(data.code === WebSocketService.ERROR_CODE.CONNECTION_LOST) {
            $("body").addClass("ws-disconnected");
        }
    });

    $scope.closeContextualMenus = function(){
        ContextualMenu.prototype.closeAny();
    };

    $scope.reconnectWebSocket = function() {
        WebSocketService.connect();
    };

    /* Put some stuff in the global scopes */
    $rootScope.$stateParams = $stateParams;
    $scope.$state = $state;
    $scope.sanitize = sanitize;
    $scope.JSON = JSON;
    $scope.$route = $route;
    $scope.Object = Object;
    $scope.pendingRequests = $http.pendingRequests;
    $rootScope.spinnerPosition = undefined;
    $scope.isTouchDevice = isTouchDevice();

    $timeout(function() {
        $('.selectpicker').selectpicker();
        $('[data-toggle=dropdown]').dropdown();
     },10);

    /* Some global state management */

    $scope.$on('$stateChangeSuccess', function (e, toState) {
        WT1.setSessionParam("currentState", toState.name);
        WT1.event("state-changed");
        $rootScope.$broadcast("dismissModals");
        $rootScope.$broadcast("dismissPopovers");
    });

    /* *************** Global login / config management ***************** */

    $scope.onConfigurationLoaded = function() {
        Assert.inScope($rootScope, 'appConfig');

        WT1.configure();
        if ($rootScope.appConfig.loggedIn) {
            ErrorReporting.configure();
            TrackingService.configurePingTracking();
        }
        if (window.devInstance) {
            Mousetrap.bind("@ r r", function(){
                $templateCache.removeAll();
                $cacheFactory.get("$http").removeAll();
                $state.go($state.current, $stateParams, {reload:true, inherit: false, notify: true});
            });
            Mousetrap.bind("@ c c", function(){
                $templateCache.removeAll();
                $cacheFactory.get("$http").removeAll();
            });
        }
    };

    FMAPI.core.getConfiguration().success(function(data) {
        $rootScope.appConfig = data;
        $scope.appConfig = data;
        window.dkuAppConfig = data;

        if ($scope.appConfig.tenancy != 'MONOTENANT') {
            let params = getParameters(window.location)
            if (params["tenantId"]) {
                localStorageService.set("tenantId", params["tenantId"]);    
            }
        } else {
            localStorageService.set("tenantId", 'main');    
        }

        var ac = data;
        WT1.event("fm-open", {
            loggedIn : ac.loggedIn,
            installId: ac.installId,
            version: ac.version,
            hasNodeName: !!ac.nodeName,
            hasExternalURL: !!ac.dssExternalURL,
        });

        var isReleaseVersionRegex = new RegExp(/[1-9]\d?\.\d\d?\.\d\d?(-\w+)?/);
        if (ac && ac.version && ac.version.product_version && !isReleaseVersionRegex.test(ac.version.product_version)) {
            if (!window.localStorage.forceRollbar) {
                // Disable WT1 reporting for dev kits
                window.devInstance = true;
            }

            if (ac.version.conf_version) {
                let major = ac.version.conf_version.substring(0, 2);
                if ((major > 0) && (major < 99)) { // let's keep some room for major version :-)
                    $rootScope.dssMajorVersion = major.toString();
                    $rootScope.versionDocRoot = "https://doc.dataiku.com/dss/" + $rootScope.dssMajorVersion;
                    $rootScope.apiDocRoot = "https://doc.dataiku.com/dss/api/" + $rootScope.dssMajorVersion;
                }
            }
        }

        if (ac && ac.version && ac.version.conf_version && isReleaseVersionRegex.test(ac.version.product_version)) {
            // check we have a correct version number in first place
            let parsedVersion = ac.version.product_version.split(".");
            let major = parseInt(parsedVersion[0], 10);
            if ((major > 0) && (major < 99)) { // let's keep some room for major version :-)
                $rootScope.dssMajorVersion = major.toString();
                $rootScope.versionDocRoot = "https://doc.dataiku.com/dss/" + $rootScope.dssMajorVersion;
                $rootScope.apiDocRoot = "https://doc.dataiku.com/dss/api/" + $rootScope.dssMajorVersion;
            }
        }

        if (!$scope.appConfig.loggedIn) {
            /* Don't redirect to login access to the login page */
            if ($location.path().indexOf("/login/openid-redirect-uri/") === 0) {
                let params = getParameters(window.location)

                if (params["error"]) {
                    Logger.info("An error happened during the authentication.");
                    $state.transitionTo("sso.error", {error : "Identity provider error: " + params["error"], errorDescription: params["errorDescription"]});
                    return;
                }

                var code = params["code"];
                var state = params["state"];

                let getStateFromLocalStorage = localStorageService.get("openid-state");
                localStorageService.remove("openid-state");
                if (getStateFromLocalStorage != state) {
                    Logger.info("An error happened during the authentication. The State is not matching the initial one");
                    $state.transitionTo("sso.error", {error : "FM error ", errorDescription: "Initial state value not matching request state param. This could be an attack attempt, please contact your administrator."});
                    return;
                }
                let tenantId = localStorageService.get("tenantId");
                FMAPI.auth.exchangeAuthorizationCode(code, state, tenantId).success(function(){
                    const redirectTo = localStorageService.get("postSSOLoginRedirect");
                    const search = localStorageService.get("postSSOLoginSearch") || "";
                    if (redirectTo) {
                        Logger.info("There is a post-SSO login redirect, following it", redirectTo + "?" + search);
                        localStorageService.remove("postSSOLoginRedirect");
                        localStorageService.remove("postSSOLoginSearch");
                        const url = new URL(window.location.href);
                        url.search = search;
                        url.pathname = redirectTo;  // Only follow redirects to a local path, not to another site
                        window.location = url.href;
                        return;
                    }
                }).error(function(error) {
                    Logger.info("An error happened during the authentication.");
                    var errorCode =  error.code ||  error.errorType
                    var errorDescription =  error.errorMessage ||  error.message
                    $state.transitionTo("sso.error", {error : "FM error " + errorCode , errorDescription: errorDescription});
                });

                return;
            } else if ($location.path().indexOf("/login/") === 0 || $location.path().indexOf("/login") === 0 || $location.path().indexOf("/logged-out") === 0) {
                return;
            } else if ($scope.appConfig.ssoLoginEnabled) {
                let params = getParameters(window.location)
                let tenantId = localStorageService.get("tenantId");;
                if (params["error"]) {
                    Logger.info("An error happened during the authentication.");
                    // Redirect to login
                    $state.transitionTo("sso.error", {error : params["error"], errorDescription: params["errorDescription"]});
                    return;
                }
                if ($scope.appConfig.ssoProtocol == "SAML") {
                    if ($location.path()) {
                        Logger.info("Setting a post-SSO redirect to", $location.path());
                        localStorageService.set("postSSOLoginRedirect", $location.path());
                        localStorageService.set("postSSOLoginSearch", $httpParamSerializer($location.search()));
                    }
                    FMAPI.auth.getSAMLRedirectURL(tenantId).success(function(data){
                        window.location = data.url;
                    }).error(function(error) {
                        Logger.info("An error happened during the authentication.");
                        $state.transitionTo("sso.error", {error : "FM error ", errorDescription: error.message ? error.message : "Please contact your administrator." });
                    });;;
                } else if ($scope.appConfig.ssoProtocol == "OPENID") {
                     if ($location.path()) {
                         Logger.info("Setting a post-SSO redirect to", $location.path());
                         localStorageService.set("postSSOLoginRedirect", $location.path());
                         localStorageService.set("postSSOLoginSearch", $httpParamSerializer($location.search()));
                     }
                     FMAPI.auth.getOpenIDRedirectURL().success(function(data){
                         let state = data.state;
                         localStorageService.remove("openid-state");
                         localStorageService.set("openid-state", state);

                         window.location = data.url;
                     }).error(function(error) {
                         Logger.info("An error happened during the authentication.");
                         $state.transitionTo("sso.error", {error : "FM error ", errorDescription: error.message? error.message : error});
                        });
                }
            } else {
                Logger.info("You are not logged in, redirecting you ...");
                // Redirect to login
                $state.transitionTo("login", {redirectTo : $location.path()});
            }
        } else {
            const redirectTo = localStorageService.get("postSSOLoginRedirect");
            const search = localStorageService.get("postSSOLoginSearch") || "";
            if (redirectTo) {
                Logger.info("There is a post-SSO login redirect, following it", redirectTo + "?" + search);
                localStorageService.remove("postSSOLoginRedirect");
                localStorageService.remove("postSSOLoginSearch");
                const url = new URL(window.location.href);
                url.search = search;
                url.pathname = redirectTo;  // Only follow redirects to a local path, not to another site
                window.location = url.href;
                return;
            }
        }
        $scope.onConfigurationLoaded();
    }).error(setErrorInScope.bind($scope));

    $scope.isDSSAdmin = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn && $scope.appConfig.admin;
    };
    $rootScope.isDSSAdmin = $scope.isDSSAdmin;

    $scope.logout = function(){
        FMAPI.auth.logout().success(function() {
            // Violent redirect to avoid keeping a cached appConfig
            window.location = "/logged-out";
        });
    };

    /* ********************* Keyboard shortcuts handling ******************* */
    $scope.keyboardsModal = { shown : false };
    $scope.showKeyboardShortcuts = function() {
        if (!$scope.keyboardsModal.shown) {
        	$scope.closeContextualMenus();
            $scope.keyboardsModal.shown = true;
            CreateModalFromTemplate("/templates/shortcuts.html", $scope, null, function(newScope) {
                newScope.$on("$destroy", function(){ $scope.keyboardsModal.shown = false});
            });
        }
    };

    Mousetrap.bind("?", function() {
        $scope.showKeyboardShortcuts();
        $scope.$apply();
    });

    Mousetrap.bind(": q", function() {
        window.location = "about:blank"
    });

    window.showNativeNotification = function(txt, tag, onclick, user) {
        if ("document.hasFocus", document.hasFocus()) return; // Only display notification when the user is not on the page
        if (window.Notification.permission === "default") {
            // User did not choose the notifications type yet, ask (but don't wait for the answer to display in-window notification)
            window.Notification.requestPermission(function (permission) {
                WT1.event("allow-browser-notification", {permission : permission});
            });
        } else if (window.Notification.permission === "granted") {
            // User native browser Notifications
            var options = {
                icon: UserImageUrl(user || $rootScope.appConfig.login, 200),
                dir: "ltr",
                tag: tag,
                renotify: false,
                silent: true
            };
            var notification = new window.Notification(txt, options);

            notification.onclick = (function(onclick) {return function () {
                window.focus();
                if (onclick) onclick();
                this.close();
            };})(onclick);

            setTimeout((function(n){return function(){ n.close()}; }(notification)), 5000);// native notifications have no custom timeout
        }
    };

    $scope.win = function() {
         Notification.publishToFrontend("achievement-unlocked", {achievementId : 'LOL'});
    };

    /* ********************* Various notification stuff ******************* */

    $scope.showAbout = function() {
        $scope.closeContextualMenus();
        CreateModalFromTemplate("/app/topbar/about-fm.html", $scope, null, function(modalScope){
            modalScope.currentYear = new Date().getFullYear();
        });
    };

    $scope.showFeedbackModal = function() {
        $scope.closeContextualMenus();
        CreateModalFromTemplate("/templates/widgets/topbar_drawers/feedback-modal.html", $scope, 'FeedbackController');
    };

    $rootScope.showHelp = function(){
        function showSupportWidget() {
            if (window.dkuAppConfig && window.dkuAppConfig.offlineFrontend) {
                ActivityIndicator.error("Offline mode - Support widget not available");
            } else {
                ActivityIndicator.error("Support widget not available.");
            }
        }
        CreateModalFromTemplate("/templates/widgets/topbar_drawers/get-help-modal.html", $scope, null, function(newScope) {
            newScope.openSupport = function(){
                newScope.dismiss();
                showSupportWidget();
            };
            newScope.openIntercom = function(){
                newScope.dismiss();
                $scope.forceShowIntercom()
            };
        });
    };

    $scope.isOnHome = () => {
        return $state.$current == "instances.list";
    }

    $scope.$on("$stateChangeSuccess", function() {
        if ($state.$current.pageTitle) {
            TopNav.setPageTitle($state.$current.pageTitle($stateParams));
        }
    });

    /* Shortcut : put the service in the scope so we can use it directly in templates */
    $scope.WT1Event = function(type, params) {
        WT1.event(type, params);
    };

    $scope.setSpinnerPosition = function(position){
    	$rootScope.spinnerPosition = position;
    };
});

app.controller('RegisterController', function($scope, $state, Assert, FMAPI) {
    Assert.inScope($scope, 'appConfig');

    $scope.register = {
        state: 'enter-license',
        step: '1'
    };

    $scope.existingKey = {
    };

    $scope.logMeIn = function() {
        window.location = '/';
    };

    $scope.switchStep = function(state, step) {
        $scope.register.state = state;
        $scope.register.step = step;
    };

    $scope.$watch("register.mode", function() {
        $scope.fatalAPIError = null;
    });

    $scope.setLicense = function() {
        FMAPI.registration.installLicense($scope.existingKey.license).success(function(data) {
            $scope.register.loginInfo = data;
            $scope.register.state = "thanks-license";
        }).error(setErrorInScope.bind($scope));
    };
});

app.service("LicenseParser", function(){
    var svc = {}
    svc.parseLicense = function(licenseText) {
        if (!licenseText) {
            return {empty: true};
        }
        try {
            const license = JSON.parse(licenseText);
            if (!license || !license.content.licenseId) {
                return {empty: false, error: true};
            }
            let expirationDate = undefined;
            let today = new Date().getTime();
            let isExpired = undefined;
            if (license.content.expiresOn) {
                const year = parseInt(license.content.expiresOn.substring(0, 4), 10);
                const month = parseInt(license.content.expiresOn.substring(4, 6), 10);
                const day = parseInt(license.content.expiresOn.substring(6, 8), 10);
                expirationDate = new Date(year, month - 1, day).getTime();
                isExpired = expirationDate < today;
            }
            return {empty: false, valid: true, expirationDate: expirationDate, isExpired: isExpired};
        } catch (e) {
            return {empty: false, error: true};
        }
    };
    return svc;
});

}());