(function() {
'use strict';

const app = angular.module('dataiku.directives.forms', ['dataiku.directives.forms']);

app.directive('checkUnique', function() {
    return {
        require: 'ngModel',
        scope: {
            exclude: '=',
            caseSensitive: '='
        },
        link: function(scope, elem, attrs, ngModel) {

            function format(v) {
                const input = v || '';
                return scope.caseSensitive ? input : input.toLowerCase();
            }

            function apply_validation(value) {
                ngModel.$setValidity('unique', true);
                if (scope.exclude && value) {
                    const valid = !scope.exclude.find(x => format(x) == format(value));
                    ngModel.$setValidity('unique', valid);
                }
                return value;
            }

            //For DOM -> model validation
            ngModel.$parsers.unshift(apply_validation);

            //For model -> DOM validation
            ngModel.$formatters.unshift(function(value) {
                apply_validation(value);
                return value;
            });
        }
    };
});


app.directive("objectTypePicker", function(TAGGABLE_TYPES) {
   return {
       templateUrl: '/templates/widgets/object-type-picker.html',
       restrict: 'A',
       scope: {
            objectTypePicker: '=',     // @param (optional) forwarded to dkuBsSelect
            objectType: '=',           // @model the selected taggable type
            exclude: '<?',             // @param (optional) array of types to exclude (from taggable-types)
            include: '<?',             // @param (optional) array of types to include (ignored if exclude if non-null) - may contain some non-taggable types. Currently supported: APP
            ngDisabled: '=?',
            allOption: '=?'
       },
       link: function($scope) {
            $scope.pickableTypes = TAGGABLE_TYPES;
            $scope.$watchGroup(['exclude', 'include'], ([exclude, include]) => {
                if (exclude) {
                    $scope.pickableTypes = TAGGABLE_TYPES.filter(type => !exclude.includes(type));
                } else if (include) {
                    $scope.pickableTypes = $scope.include;
                } else {
                    $scope.pickableTypes = TAGGABLE_TYPES;
                }
            });
       }
   }
});


app.directive("projectKeyPicker", function(DataikuAPI) {
    return {
        templateUrl: '/templates/widgets/project-key-picker.html',
        restrict: 'AE',
        scope: {
            projectKeyPicker: '=',  // @param (optional) forwarded to dkuBsSelect
            projectKey: '=',        // @model the selected projectKey
            project: '=?',          // @model bound to the selected project
            filter: '=',            // @param (optional) filter project function
        },
        link: function($scope) {
            function findProject() {
                $scope.project = $scope.projects.find(prj => prj.projectKey == $scope.projectKey);
            }
            DataikuAPI.projects.list()
                .success(function (projects) {
                    $scope.projects = $scope.filter ? projects.filter($scope.filter) : projects;
                    findProject();
                });//TODO @errorHandling
            $scope.$watch("projectKey", function() {
                if ($scope.projects) {
                    findProject();
                }
            });
        }
    }
});


app.directive('computablePicker', function(DataikuAPI, $stateParams) {
    return {
        template: '<div dataset-selector="computable" available-datasets="availableComputables"></div>',
        restrict: 'A',
        scope: {
            computable: '=computablePicker',
            type: '@'
        },
        link: function($scope, element) {
            DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {type: $scope.type}).success(function(data) {
                $scope.availableComputables = data;
            }).error(setErrorInScope.bind($scope.$parent));
        }
    };
});


app.directive('objectPicker', function(DataikuAPI, $stateParams, $rootScope, ActiveProjectKey, Logger, translate, $filter) {
    return {
        templateUrl : '/templates/widgets/object-picker.html',
        restrict: 'A',
        scope: {
            objectSmartId: '=objectPicker',                  // @model bound to the smartId of the selected object
            object: '=?',                                    // @model bound to the selected object
            type: '@',                                       // @param AccessibleObject type (taggable or APP)
            unusable: '=?',                                  // @param smartIds of unusable objects (as array, or map: id -> unusable (boolean))
            emptyMessage: '@?',
            selectClass: '@?',
            errorScope: '=?',
            inputAvailableObjects: '<?',                     // @param (array<AvailableObject>, optional) if provided, the list of options & you are responsible for updating it as required (notably when type & projectKey change). If not provided, the component will do the API queries to fetch available objects based on projectKey, type and permission mode. If you are using many instances if this component, you should probably use this input & share the lists of available objects in order to avoid triggering many identical API calls (see AccessibleObjectsCacheService).
            permissionMode: '@?',                            // @param the desired ReaderAuthorization.Mode (defaults to READ)
            hideForeign: '=?',                               // @param (boolean) to hide foreign objects,
            projectKey: '=?',
            disabled: '<?',
            multiSelection: '<?',
            toCustomIcon: '<?',
            toCustomColor: '<?',
            compareWith: '<?'
        },
        link: function($scope) {
            $scope.object = null;
            $scope.defaultPlaceholder = createDefaultPlaceholder($scope.type);
            const projectKey = $scope.projectKey || ActiveProjectKey.get();

            $scope.onChanges = (args) => {
                if (args && Array.isArray(args) && args.length) {
                    $scope.objectSmartId = args.map(arg => arg.smartId);
                    return;
                } else if (args && args.smartId) {
                    $scope.objectSmartId = args.smartId;
                    return;
                }
                $scope.objectSmartId = undefined;
            }

            function findObjects() {
                if ($scope.objectSmartId && Array.isArray($scope.objectSmartId)) {
                    $scope.object = $scope.availableObjects.filter(d => $scope.objectSmartId.includes(d.smartId));
                    return;
                } else if ($scope.objectSmartId) {
                    $scope.object = $scope.availableObjects.filter(d => d.smartId == $scope.objectSmartId)[0];
                    return;
                }
                $scope.object = null;
            }

            function refreshAvailableObjects(availableObjects) {
                $scope.availableObjects = availableObjects;
                updateUsability();
                findObjects();
            }

            function refreshObjects() {
                if (angular.isDefined($scope.inputAvailableObjects)) {
                    refreshAvailableObjects($scope.inputAvailableObjects);
                    return;
                }

                if (!projectKey && !['PROJECT', 'WORKSPACE', 'DATA_COLLECTION', 'APP'].includes($scope.type)) {
                    Logger.info('No project key specified, not listing accessible objects');
                    return;
                }

                DataikuAPI.taggableObjects.listAccessibleObjects(projectKey, $scope.type, $scope.permissionMode).success(function(data) {
                    refreshAvailableObjects(data)
                }).error(function(data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope.errorScope || $scope.$parent)(data, status, headers, config, statusText);
                });
            }

            function createDefaultPlaceholder(type) {
                const translatedType = translate('TAGGABLE_TYPE.'+type,type);
                return translate('SCENARIO.INTEGRATIONS.INTEGRATION_PARAMS.ATTACHMENTS.OBJECT_PICKER.NO_OBJECT_SELECTED','No {{type}} selected', {type: $filter('niceConst')(translatedType, ' ', true) });
            }

            $scope.$watch("type", function(nv, ov) {
                $scope.defaultPlaceholder = createDefaultPlaceholder($scope.type)

                if ($scope.type === "SAVED_MODEL") {
                    $scope.toCustomIcon = (item) => {
                        const object = item.object;
                        const taskType = (object.miniTask || {}).taskType;
                        const backendType = (object.miniTask || {}).backendType;
                        const predictionType = (object.miniTask || {}).predictionType;
                        const savedModelType = object.savedModelType;
                        const proxyModelProtocol = (object.proxyModelConfiguration || {}).protocol;
                        return $filter("savedModelSubtypeToIcon")(taskType, backendType, predictionType, savedModelType, proxyModelProtocol, 24);
                    }

                    $scope.toCustomColor = (item) => {
                        return $filter("savedModelTypeToClassColor")(item.object.savedModelType, true);
                    }
                }

                if (!nv || (nv == ov && $scope.availableObjects)) return;

                if (ov && nv != ov) {
                    $scope.object = null;
                    $scope.objectSmartId = null;
                }

                refreshObjects();
            });

            $scope.$watch("permissionMode", function(nv, ov) {
                if (!nv || !ov || nv == ov) return;
                refreshObjects();
            });

            $scope.$watch('objectSmartId', function() {
                if ($scope.availableObjects) {
                    findObjects()
                }
            }, true);

            $scope.$watch('object', function() {
                if ($scope.availableObjects && Array.isArray($scope.object)) {
                    $scope.objectSmartId = $scope.object.map(o => o.smartId);
                } else if ($scope.availableObjects && $scope.object && $scope.object.smartId) {
                    $scope.objectSmartId = $scope.object.smartId;
                }
            });

            $scope.$watch('inputAvailableObjects', function() {
                refreshObjects();
            });

            $scope.$watch(function(scope) {
                if (angular.isObject(scope.unusable)) {
                    var simplified = {};
                    for (var key in scope.unusable) {
                        simplified[key] = !!scope.unusable[key];
                    }
                    return simplified;
                } else {
                    return scope.unusable;
                }
            }, updateUsability, true);

            function updateUsability(nv, ov) {
                if (!$scope.availableObjects) return;

                if (!$scope.unusable) {
                    $scope.availableObjects = $scope.availableObjects.map(function (item) {
                        return {
                            ...item,
                            usable: true,
                        }
                    });
                } else if (angular.isArray($scope.unusable)) {
                    $scope.availableObjects = $scope.availableObjects.map(function (item) {
                        return {
                            ...item,
                            usable: ($scope.unusable || []).indexOf(item.smartId) == -1,
                        }
                    });
                } else if (angular.isObject($scope.unusable)) {
                    $scope.availableObjects = $scope.availableObjects.map(function (item) {
                        return {
                            ...item,
                            usable: !$scope.unusable[item.smartId],
                        }
                    });
                }
            }
        }
    };
});

app.directive('dashboardTilePicker', function(DataikuAPI, ActiveProjectKey, Logger) {
    return {
        templateUrl : '/templates/widgets/dashboard-tile-picker.html',
        restrict: 'A',
        scope: {
            objectSmartId: '=dashboardTilePicker',      // @model bound to the smartId of the selected object
            object: '=?',                               // @model bound to the selected object
            sourceObject: '=?',                         // @model bound to the dataset source of selected tile object
            objectChanged: '&',                                    
            emptyMessage: '@?',
            availableTiles: '=?',
        },
        link: function($scope) {
            $scope.availableTiles && $scope.availableTiles.forEach((item) => {
                item.usable = true;
            });

            const projectKey = $scope.projectKey || ActiveProjectKey.get();

            function findObjectSource() {
                if (!$scope.availableObjects) return;
                $scope.sourceObject = $scope.availableObjects.find(function (d) {
                    return d.smartId == $scope.object.datasetSmartName;
                });
            }

            function fetchSources() {
                if (!projectKey && $scope.type != 'PROJECT') {
                    Logger.info('No project key specified, not listing accessible objects');
                    return;
                }
                DataikuAPI.taggableObjects.listAccessibleObjects(projectKey, "DATASET", $scope.permissionMode).success(function (data) {
                    $scope.availableObjects = data;
                }).error(function (data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope.errorScope || $scope.$parent)(data, status, headers, config, statusText);
                });
            }

            fetchSources();

            $scope.$watch('objectSmartId', (nv) => {
                if (!nv) { return }
                for (let insight of $scope.availableTiles) {
                    if (insight.id === nv) {
                        $scope.object = insight;
                        findObjectSource();
                        break;
                    }
                }
            });
        }
    };
});

app.service('PopoverPosition', function() {

    this.computePosition = function($scope, element, attrs, popover) {
        var mainZone = $(".mainzone", element);
        popover.css("width", attrs.popoverWidth ? attrs.popoverWidth : Math.max(500, mainZone.width() + 20));
        if ($scope.popoverPlacement == 'right') {
            let ccsLeft = mainZone.offset().left - popover.width() + mainZone.width() +4;
            popover.css("left", ccsLeft > 0 ? ccsLeft: 1);
        } else {
            popover.css("left", mainZone.offset().left);
        }

        let cssTop = mainZone.offset().top + mainZone.height() + ($scope.marginTop ? parseInt($scope.marginTop) : 8);
        if ($scope.popoverPlacement == 'auto') {
            if (mainZone.offset().top + mainZone.height() + 280 > $(window).height()) {
                cssTop = mainZone.offset().top - 310;
            }
        }
        if (cssTop + mainZone.height() + popover.height() > $(window).height()) {
            // + 12 for (pad 10 + border 1 and li pad 1)
            cssTop = $(window).height() - popover.height() -  mainZone.height() + 13;
        }
        popover.css("top", cssTop);
    };

});

app.directive('datasetSelector', function(ListFilter, $compile, PopoverPosition, translate) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableDatasets : '=',
            datasetSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?',
            hideForeign: '=?',
            emptyMessage: '@?',
            multi: '@',
            hideGroupTitles: '@?',
            onDatasetSelectorChange: '&?',
            discriminationProperties: '@?',
            useNewIcons: '<?', // this argument is used to migrate gradually to the new icons, todo: it should be removed when all the icons are migrated
        },
        templateUrl : '/templates/dataset-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            $scope.translate = translate;

            var popover = null;

            if ($scope.multi) {
                $scope.noLiveUpdate = true;
                $scope.datasetSelector = $scope.datasetSelector || [];
            } 

            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
                $scope.$apply();
            });

            /* List management */
            function update() {
                $scope.displayedDatasets = [];
                if (!$scope.availableDatasets) return;

                const headerObjects = $scope.availableDatasets.filter(dataset => dataset.header);
                const otherObjects = $scope.availableDatasets.filter(dataset => !dataset.header);
                $scope.filtered = otherObjects;
                
                $scope.filtered = ListFilter.filter(otherObjects, $scope.filter.query, $scope.discriminationProperties, !!$scope.discriminationProperties);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        if ($scope.hideForeign) {
                            continue;
                        }
                        group = translate("FLOW.CREATE_RECIPE.PROJECT", "Project: ") +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, datasets : [], sort:sort}
                    }
                    groups[group].datasets.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].datasets.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.displayedGroups.unshift({
                    datasets: headerObjects,
                    header: true
                });

                $scope.currentlySelected = null;
                for (let i in $scope.availableDatasets) {
                    // datasets generally have a smartName/smartId; other objects may only have an id
                    if (($scope.availableDatasets[i].smartName || $scope.availableDatasets[i].smartId || $scope.availableDatasets[i].id) == $scope.datasetSelector) {
                        $scope.currentlySelected = $scope.availableDatasets[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", update, true);
            $scope.$watchCollection("availableDatasets", update);
            update();

            /* Model management */

            $scope.select = function(details) {
                const itemId = details.smartName || details.smartId || details.id;
                if (!$scope.multi) {
                    $scope.datasetSelector = itemId;
                    hide();
                } else {
                    $scope.datasetSelector.push(itemId);
                }
            };

            $scope.itemClicked = function(details) {
                const itemId = details.smartName || details.smartId || details.id;

                if (!$scope.multi) {
                    if (itemId === $scope.datasetSelector) {
                        $scope.datasetSelector = null;
                    } else {
                        $scope.datasetSelector = itemId;
                    }
                    hide();
                } else {
                    const detailsIndex = $scope.datasetSelector.indexOf(itemId);
                    if (detailsIndex >= 0) {
                        $scope.datasetSelector.splice(detailsIndex, 1);
                    } else {
                        $scope.datasetSelector.push(itemId);
                    }
                }
            };

            $scope.isItemSelected = function(item) {
                const itemId = item.smartName || item.smartId || item.id;

                if (!$scope.multi) {
                    return $scope.datasetSelector === itemId;
                } else {
                    return $scope.datasetSelector.indexOf(itemId) >= 0;
                }
            }

            $scope.$watch("datasetSelector", function(newValue, oldValue) {
                update();
                if (angular.isFunction($scope.onDatasetSelectorChange)) {
                    $scope.onDatasetSelectorChange();
                }
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();
            
            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };

            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");

                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });

                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    // Up and down in list
                    if(event.keyCode==38 || event.keyCode==40) {
                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.datasets.length ; j++) {
                                    var ds = group.datasets[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }
                             $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.datasetSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } 
                    // Enter
                    else if (event.keyCode === 13) {
                        if ($scope.currentlySelected) {
                            $scope.itemClicked($scope.currentlySelected);
                            $scope.$apply();
                        }
                    } else {
                        $scope.currentlySelected = null;
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


//TODO @dssObjects factorize
app.directive('savedModelSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableSavedModels : '=',
            savedModelSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/model-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedSavedModels = [];
                if (!$scope.availableSavedModels) return;

                $scope.filtered = $scope.availableSavedModels;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableSavedModels, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, savedModels : [], sort:sort}
                    }
                    groups[group].savedModels.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].savedModels.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableSavedModels) {
                    if (($scope.availableSavedModels[i].smartName || $scope.availableSavedModels[i].smartId) == $scope.savedModelSelector) {
                        $scope.currentlySelected = $scope.availableSavedModels[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableSavedModels", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.savedModelSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.savedModelSelector) {
                    $scope.savedModelSelector = null;
                } else {
                    $scope.savedModelSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("savedModelSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.savedModels.length ; j++) {
                                    var ds = group.savedModels[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.savedModelSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


//TODO @dssObjects factorize
app.directive('modelEvaluationStoreSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableModelEvaluationStores : '=',
            modelEvaluationStoreSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/evaluation-store-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedModelEvaluationStores = [];
                if (!$scope.availableModelEvaluationStores) return;

                $scope.filtered = $scope.availableModelEvaluationStores;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableModelEvaluationStores, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, modelEvaluationStores : [], sort:sort}
                    }
                    groups[group].modelEvaluationStores.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (let g in groups) {
                    groups[g].modelEvaluationStores.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableModelEvaluationStores) {
                    if (($scope.availableModelEvaluationStores[i].smartName || $scope.availableModelEvaluationStores[i].smartId) == $scope.modelEvaluationStoreSelector) {
                        $scope.currentlySelected = $scope.availableModelEvaluationStores[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableModelEvaluationStores", function() {
                update();
            }, true);
            update();

            /* Store management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.modelEvaluationStoreSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.modelEvaluationStoreSelector) {
                    $scope.modelEvaluationStoreSelector = null;
                } else {
                    $scope.modelEvaluationStoreSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("modelEvaluationStoreSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.modelEvaluationStores.length ; j++) {
                                    var ds = group.modelEvaluationStores[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.modelEvaluationStoreSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});

//TODO @dssObjects factorize
app.directive('apiEndpointSelector', function($timeout, ListFilter, $compile, PopoverPosition, WT1) {
    var ret = {
        restrict : 'A',
        scope : {
            availableApiEndpoints : '=',
            apiEndpointSelector : '=',
            entryEndpointId: '=',
            popoverPlacement: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?',
            wt1SelectEventName: '<?'
        },
        templateUrl : '/templates/api-endpoint-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;

            /* List management */
            function update() {
                $scope.displayedApiEndpoints = [];
                if (!$scope.availableApiEndpoints) return;

                $scope.filtered = $scope.availableApiEndpoints;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableApiEndpoints, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    group =  $scope.filtered[i].id[0].toUpperCase();
                    sort = "AAAAAAAA" + group;
                    if (! groups[group]) {
                         groups[group] = {title : group, apiEndpoints : [], sort:sort}
                    }
                    groups[group].apiEndpoints.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (let g in groups) {
                    groups[g].apiEndpoints.sort(function(a,b) { return a.id.localeCompare(b.id)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                if ($scope.apiEndpointSelector) {
                    for (let i in $scope.availableApiEndpoints) {
                        if (($scope.availableApiEndpoints[i].id) === $scope.apiEndpointSelector.id) {
                            $scope.currentlySelected = $scope.availableApiEndpoints[i];
                        }
                    }
                }
            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableApiEndpoints", function() {
                update();
            }, true);
            update();

            /* Store management */

            $scope.select = function(details) {
                $scope.apiEndpointSelector = details;
                hide();
            };

            $scope.itemClicked = function(details) {
                if ($scope.wt1SelectEventName) {
                    WT1.event($scope.wt1SelectEventName);
                }
                $scope.apiEndpointSelector = details;
                hide();
            };

            $scope.$watch("apiEndpointSelector", function(newValue, oldValue) {
                update();
            });

            $scope.typeToIcon = function(type) {
                switch (type) {
                    case "STD_PREDICTION":
                        return "dku-icon-modelize-24";
                    case "STD_FORECAST":
                        return "dku-icon-time-series-24";
                    case "STD_CAUSAL_PREDICTION":
                        return "dku-icon-lever-24";
                    case "R_FUNCTION":
                        return "dku-icon-r-24";
                    case "PY_FUNCTION":
                        return "dku-icon-python-24";
                    case "DATASETS_LOOKUP":
                        return "dku-icon-dataset-24";
                    case "SQL_QUERY":
                        return "dku-icon-sql-24";
                    case "STD_CLUSTERING":
                        return "dku-icon-clustering-24";
                    case "CUSTOM_PREDICTION":
                        return "dku-icon-custom-prediction-python-24";
                    case "CUSTOM_R_PREDICTION":
                        return "dku-icon-custom-prediction-r-24";
                    default:
                        return "dku-icon-gear-24";
                }
            }

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.apiEndpoints.length ; j++) {
                                    var ds = group.apiEndpoints[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.apiEndpointSelector = $scope.currentlySelected;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});

//TODO @dssObjects factorize
app.directive('folderSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableFolders : '=',
            folderSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/folder-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedFolders = [];
                if (!$scope.availableFolders) return;

                $scope.filtered = $scope.availableFolders;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableFolders, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, folders : [], sort:sort}
                    }
                    groups[group].folders.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].folders.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableFolders) {
                    if (($scope.availableFolders[i].smartName || $scope.availableFolders[i].smartId) == $scope.folderSelector) {
                        $scope.currentlySelected = $scope.availableFolders[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableFolders", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.folderSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.folderSelector) {
                    $scope.folderSelector = null;
                } else {
                    $scope.folderSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("folderSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.folders.length ; j++) {
                                    var ds = group.folders[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.folderSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
}
    return ret;
});


//TODO @dssObjects factorize
app.directive('labelingTaskSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableLabelingTasks : '=',
            labelingTaskSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/labeling-task-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedLabelingTasks = [];
                if (!$scope.availableLabelingTasks) return;

                $scope.filtered = $scope.availableLabelingTasks;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableLabelingTasks, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, labelingTasks : [], sort:sort}
                    }
                    groups[group].labelingTasks.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (let g in groups) {
                    groups[g].labelingTasks.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableLabelingTasks) {
                    if (($scope.availableLabelingTasks[i].smartName || $scope.availableLabelingTasks[i].smartId) == $scope.labelingTaskSelector) {
                        $scope.currentlySelected = $scope.availableLabelingTasks[i];
                    }
                }

            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableLabelingTasks", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.labelingTaskSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.labelingTaskSelector) {
                    $scope.labelingTaskSelector = null;
                } else {
                    $scope.labelingTaskSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("labelingTaskSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.labelingTasks.length ; j++) {
                                    var ds = group.labelingTasks[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.labelingTaskSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
}
    return ret;
});


//TODO @dssObjects factorize
app.directive('streamingEndpointSelector', function($timeout, ListFilter, $compile, StateUtils, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableStreamingEndpoints : '=',
            streamingEndpointSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/streaming-endpoint-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {
        
            $scope.StateUtils = StateUtils; 
            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedStreamingEndpoints = [];
                if (!$scope.availableStreamingEndpoints) return;

                $scope.filtered = $scope.availableStreamingEndpoints;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableStreamingEndpoints, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, streamingEndpoints : [], sort:sort}
                    }
                    groups[group].streamingEndpoints.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].streamingEndpoints.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableStreamingEndpoints) {
                    if (($scope.availableStreamingEndpoints[i].smartName || $scope.availableStreamingEndpoints[i].smartId) == $scope.streamingEndpointSelector) {
                        $scope.currentlySelected = $scope.availableStreamingEndpoints[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableStreamingEndpoints", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.streamingEndpointSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.streamingEndpointSelector) {
                    $scope.streamingEndpointSelector = null;
                } else {
                    $scope.streamingEndpointSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("streamingEndpointSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.streamingEndpoints.length ; j++) {
                                    var ds = group.streamingEndpoints[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.streamingEndpointSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


//TODO @dssObjects factorize
app.directive('knowledgeBankSelector', function($timeout, ListFilter, $compile, StateUtils, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableKnowledgeBanks : '=',
            knowledgeBankSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/knowledge-bank-selector.html',
    };

    ret.compile = function(element) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            $scope.StateUtils = StateUtils;
            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedKnowledgeBanks = [];
                if (!$scope.availableKnowledgeBanks) return;

                $scope.filtered = $scope.availableKnowledgeBanks;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableKnowledgeBanks, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, knowledgeBanks : [], sort:sort}
                    }
                    groups[group].knowledgeBanks.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].knowledgeBanks.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableKnowledgeBanks) {
                    if (($scope.availableKnowledgeBanks[i].smartName || $scope.availableKnowledgeBanks[i].smartId) == $scope.knowledgeBankSelector) {
                        $scope.currentlySelected = $scope.availableKnowledgeBanks[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableKnowledgeBanks", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.knowledgeBankSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.knowledgeBankSelector) {
                    $scope.knowledgeBankSelector = null;
                } else {
                    $scope.knowledgeBankSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("knowledgeBankSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                }
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.knowledgeBanks.length ; j++) {
                                    var ds = group.knowledgeBanks[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.knowledgeBankSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


})();
