(function(){
'use strict';

const app = angular.module('dataiku.modelevaluationstores', []);



/* ************************************ List / Right column  *************************** */

app.controller("ModelEvaluationStorePageRightColumnActions", function($controller, $scope, $state, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey,
    ActivityIndicator, CreateModalFromTemplate, ModelEvaluationStoreRenameService) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    const mesId = $stateParams.mesId || $scope.mesId;
    DataikuAPI.modelevaluationstores.getFullInfo(ActiveProjectKey.get(), mesId).success((data) => {
        $scope.mes = data.evaluationStore;
        $scope.mes.description = data.evaluationStore.shortDesc;
        $scope.mes.nodeType = 'LOCAL_MODELEVALUATIONSTORE';
        $scope.mes.realName = data.evaluationStore.name;
        $scope.mes.name = data.evaluationStore.id;
        $scope.mes.interest = data.interest;

        $scope.selection = {
            selectedObject : $scope.mes,
            confirmedItem : $scope.mes,
        };
    }).error(setErrorInScope.bind($scope));

    $scope.renameMES = function() {
        const modelEvaluationStore = $scope.modelEvaluationStore;
        ModelEvaluationStoreRenameService.renameModelEvaluationStore({
            scope: $scope,
            state: $state,
            projectKey: modelEvaluationStore.projectKey,
            modelEvaluationStoreId: modelEvaluationStore.id,
            modelEvaluationStoreName: modelEvaluationStore.name,
            onSave: () => { ActivityIndicator.success("Saved"); }
        });
    };

    $scope.isOnStoreObjectPage = function() {
        return $state.includes('projects.project.modelevaluationstores.modelevaluationstore');
    }
});


app.directive('modelEvaluationStoreRightColumnSummary', function($controller, $state, $stateParams, ModelEvaluationStoreCustomFieldsService, $rootScope, FlowGraphSelection,
    DataikuAPI, SmartId, CreateModalFromTemplate, QuickView, ActiveProjectKey, ActivityIndicator, Logger, FlowBuildService, AnyLoc, ModelEvaluationStoreRenameService) {

    return {
        templateUrl :'/templates/modelevaluationstores/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.refreshData = function() {
                var projectKey = scope.selection.selectedObject.projectKey;
                var name = scope.selection.selectedObject.name;
                scope.canAccessObject = false;

                DataikuAPI.modelevaluationstores.getFullInfo(ActiveProjectKey.get(), SmartId.create(name, projectKey)).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return; // too late!
                    }
                    scope.modelEvaluationStoreData = data;
                    scope.modelEvaluationStore = data.evaluationStore;
                    scope.modelEvaluationStore.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;
                    scope.isLocalModelEvaluationStore = projectKey == ActiveProjectKey.get();
                    scope.objectAuthorizations = data.objectAuthorizations;
                    scope.canAccessObject = true;
                }).catch(setErrorInScope.bind(scope));
            };

            const deregistrationFn = scope.$watch("$root.projectSummary.canWriteProjectContent", (nv) => {
                if (nv) {
                    deregistrationFn(); // Ensure that you'll never create the inner watch multiple times
                    scope.$on("objectSummaryEdited", function() {
                        DataikuAPI.modelevaluationstores.save(scope.modelEvaluationStore, {summaryOnly: true})
                            .success(function() {
                                ActivityIndicator.success("Saved");
                            })
                            .error(setErrorInScope.bind(scope));

                    });
                }
            });

            scope.$watch("selection.selectedObject",function(nv) {
                scope.modelEvaluationStoreData = {evaluationStore: nv, timeline: {}, interest: {}}; // display temporary (incomplete) data
                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.modelEvaluationStore = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.zoomToOtherZoneNode = function(zoneId) {
                const otherNodeId = scope.selection.selectedObject.id.replace(/zone__.+?__modelevaluationstore/, "zone__" + zoneId + "__modelevaluationstore");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId }))
                }
                else {
                    scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[otherNodeId], null);
                }
            }

            scope.editCustomFields = function() {
                if (!scope.selection.selectedObject) {
                    return;
                }
                DataikuAPI.modelevaluationstores.getSummary(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name).success(function(data) {
                    let modelEvaluationStore = data.object;
                    let modalScope = angular.extend(scope, {objectType: 'MODEL_EVALUATION_STORE', objectName: modelEvaluationStore.name, objectCustomFields: modelEvaluationStore.customFields});
                    CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                        ModelEvaluationStoreCustomFieldsService.saveCustomFields(modelEvaluationStore, customFields);
                    });
                }).error(setErrorInScope.bind(scope));
            };

            scope.buildModelEvaluationStore = function() {
                const modalOptions = {
                    upstreamBuildable: scope.modelEvaluationStoreData.upstreamBuildable,
                    downstreamBuildable: scope.modelEvaluationStoreData.downstreamBuildable,
                };
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "MODEL_EVALUATION_STORE", AnyLoc.makeLoc(scope.modelEvaluationStore.projectKey, scope.modelEvaluationStore.id), modalOptions);
            };

            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);

            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "MODEL_EVALUATION_STORE", ActiveProjectKey.get(), scope.selection.selectedObject.name).success(function(data) {

                    scope.selection.selectedObject.interest = data;
                    scope.modelEvaluationStoreData.interest = data;

                }).error(setErrorInScope.bind(scope));
            }

            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            scope.$on("$destroy", interestsListener);

            scope.renameMES = function() {
                const modelEvaluationStore = scope.modelEvaluationStore;
                ModelEvaluationStoreRenameService.renameModelEvaluationStore({
                    scope: scope,
                    state: $state,
                    projectKey: modelEvaluationStore.projectKey,
                    modelEvaluationStoreId: modelEvaluationStore.id,
                    modelEvaluationStoreName: modelEvaluationStore.name
                });
            };
        }
    }
});

app.service("ModelEvaluationStoreCustomFieldsService", function ($rootScope, TopNav, DataikuAPI, ActivityIndicator, WT1) {
    let svc = {};

    svc.saveCustomFields = function(modelEvaluationStore, newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'MODEL_EVALUATION_STORE'});
        let oldCustomFields = angular.copy(modelEvaluationStore.customFields);
        modelEvaluationStore.customFields = newCustomFields;
        return DataikuAPI.modelevaluationstores.save(modelEvaluationStore, {summaryOnly: true})
            .success(function(data) {
                ActivityIndicator.success("Saved");
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), modelEvaluationStore.customFields);
                $rootScope.$broadcast('reloadGraph');
            })
            .error(function(a, b, c) {
                modelEvaluationStore.customFields = oldCustomFields;
                setErrorInScope.bind($rootScope)(a, b, c);
            });
    };

    return svc;
});


app.controller("ModelEvaluationStoreListController", function($scope, $controller, $stateParams, DataikuAPI, CreateModalFromTemplate, $state, TopNav, WT1) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'realName', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["realName", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.modelevaluationstores.listHeads($stateParams.projectKey).success(function(data) {
            // dirty things to handle the discrepancy between the types of selected objects
            // which can have info displayed in the right panel
            data.forEach(mes => {
                mes.realName = mes.name;
                mes.name = mes.id;
            });
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
            WT1.event("model-evaluation-store-menu-opening", {nbOfMes: data.length});
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_MODEL_EVALUATION_STORES, TopNav.LEFT_MODEL_EVALUATION_STORES, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.modelevaluationstores.modelevaluationstore.evaluations", {projectKey : $stateParams.projectKey, mesId : data.id});
    }

    $scope.newModelEvaluationStore = function() {
        CreateModalFromTemplate("/templates/modelevaluationstores/new-model-evaluation-store-modal.html", $scope);
    }

});

app.controller("ModelEvaluationStoreController", function($scope, $rootScope, Assert, DataikuAPI, CreateModalFromTemplate, $state, $stateParams,
            ActiveProjectKey, ActivityIndicator, StateUtils, TopNav, $q, SmartId, MLModelsUIRouterStates) {
    $scope.versionsContext = {}
    $scope.mesContext = {};
    $scope.uiState = {};
    $scope.clearVersionsContext = function(){
        // eslint-disable-next-line no-undef
        clear($scope.versionsContext);
    };

    $scope.$on("$destroy", $scope.clearVersionsContext);

    $scope.comparisonForbiddenReason = "No model evaluation"
    $scope.compare = function() {
        $scope.$broadcast('triggerCreateModelComparison'); // Will be caught by the perf-comparator or ModelEvaluationStoreBaseEvaluationController, that'll actually do the comparison
    };
    $scope.$on("refreshComparisonForbiddenReason", function(event, comparisonForbiddenReason) { // sent by the perf comparator or ModelEvaluationStoreBaseEvaluationController
        $scope.comparisonForbiddenReason = comparisonForbiddenReason;
    });

    $scope.savedSettings = {}
    const mesId = $stateParams.mesId || $scope.mesId;
    DataikuAPI.modelevaluationstores.getFullInfo(ActiveProjectKey.get(), mesId).success(function(data){
        $scope.modelEvaluationStoreFullInfo = data;
        $scope.modelEvaluationStore = data.evaluationStore;
        if (!$scope.modelEvaluationStore.displayParams.sortColumn) {
            $scope.modelEvaluationStore.displayParams.sortColumn = "labels.evaluation:date";
            $scope.modelEvaluationStore.displayParams.sortDescending = true;
        }
        if (!$scope.modelEvaluationStore.displayParams.xLabel) {
            $scope.modelEvaluationStore.displayParams.xLabel = DEFAULT_X_LABEL;
            $scope.modelEvaluationStore.displayParams.yLabels = [...DEFAULT_Y_LABELS];
            if ($scope.modelEvaluationStoreFullInfo.evaluations?.length > 0 &&  $scope.modelEvaluationStoreFullInfo.evaluations[0].predictionType=="TIMESERIES_FORECAST"){
                $scope.modelEvaluationStore.displayParams.yLabels.push("evaluation:nb-evaluation-timesteps")
            }
        }
        $scope.saveSettings(angular.copy($scope.modelEvaluationStore)); // We want to track differences between modelEvaluationStore and savedSettings, so the angular.copy()
        TopNav.setItem(TopNav.ITEM_MODEL_EVALUATION_STORE, mesId, {name: $scope.modelEvaluationStore.name});
    }).error(setErrorInScope.bind($scope));

    $scope.save = function() {
        const settings = angular.copy($scope.modelEvaluationStore);
        DataikuAPI.modelevaluationstores.save($scope.modelEvaluationStore, {summaryOnly: true})
            .success(function(data) {
                    $scope.saveSettings(settings);
                    ActivityIndicator.success("Saved");
            })
            .error(setErrorInScope.bind($scope));
    };

    $scope.dirtySettings = function() {
        return !angular.equals($scope.savedSettings, $scope.modelEvaluationStore);
    }

    $scope.saveSettings = function(settings) {
        $scope.savedSettings = settings;
    }

    $scope.goToEvaluatedModel = function() {
        if (!$scope.mesContext.evaluationFullInfo.evaluation) return;
        const modelRef = $scope.mesContext.evaluationFullInfo.evaluation.modelRef;
        StateUtils.go.savedModel(modelRef.smId, modelRef.projectKey);
    };

    $scope.goToEvaluatedModelVersion = function() {
        if (!$scope.mesContext.evaluationFullInfo.evaluation) return;
        const modelRef = $scope.mesContext.evaluationFullInfo.evaluation.modelRef;
        StateUtils.go.savedModelVersion('PREDICTION', modelRef.smId, modelRef.fullId, modelRef.projectKey);
    }

    $scope.goToEvaluatedDataset = function() {
        if (!$scope.mesContext.evaluationFullInfo.evaluation) return;
        StateUtils.go.dssObject('DATASET', $scope.mesContext.evaluationFullInfo.evaluation.dataParams.ref);
    };
    $scope.createAndPinInsight = function(mesFullInfo, settingsPane) {
        const insight = {
            projectKey: ActiveProjectKey.get(),
            type: 'model-evaluation_report',
            params: {mesSmartId: SmartId.create(mesFullInfo.evaluationStore.id, ActiveProjectKey.get())},
            name: "Full report of last model evaluation from " + mesFullInfo.evaluationStore.name
        };
        let params;
        if (settingsPane) {
            params = MLModelsUIRouterStates.savedModelPaneToDashboardTile(settingsPane, $stateParams);
        }
        CreateModalFromTemplate("/templates/dashboards/insights/create-and-pin-insight-modal.html", $rootScope, "CreateAndPinInsightModalController", function (newScope) {
            newScope.init(insight, params);
        });
    }
});


/* ************************************ Settings *************************** */

app.controller("ModelEvaluationStoreSettingsController", function($scope, DataikuAPI, $q, TopNav, ActivityIndicator){
    TopNav.setLocation(TopNav.TOP_MODEL_EVALUATION_STORES, TopNav.LEFT_MODEL_EVALUATION_STORES, TopNav.TABS_MODEL_EVALUATION_STORE, "settings");

    $scope.save = function() {
        const settings = angular.copy($scope.modelEvaluationStore);
        $q.all([
            $scope.saveMetricsNow(),
            DataikuAPI.modelevaluationstores.save($scope.modelEvaluationStore, {summaryOnly: true})
        ]).then(function(data) {
                $scope.saveSettings(settings);
                ActivityIndicator.success("Saved");
        });
    };
});


app.controller("ModelEvaluationsMetricsHandlingCommon", function($scope, PMLFilteringService, CustomMetricIDService, MetricsUtils, LLMEvaluationMetrics) {
    $scope.FilteringService = PMLFilteringService;

    $scope.sortValue = function(sortColumn) {
        const metricsPattern = /^metrics\./;
        const labelsPattern = /^labels\./;
        if(metricsPattern.exec(sortColumn)) {
            return me => me.metrics[$scope.FilteringService.metricMap[sortColumn.replace(metricsPattern, '')]];
        }
        if(labelsPattern.exec(sortColumn)) {
            return me => me.labels.get(sortColumn.replace(labelsPattern, ''));
        }
        return me => me[sortColumn];
    };

    $scope.getAllCustomMetricNames = function() {
        const customMetricNameSet = new Set();
        $scope.uiState.refs.map(o => {
            if(o.metrics && o.metrics.customMetricsResults) {
                o.metrics.customMetricsResults.forEach(customMetricResult => {
                    customMetricNameSet.add(customMetricResult.metric.name?customMetricResult.metric.name:customMetricResult.metric.metricCode);
                });
            }
        });
        return customMetricNameSet;
    }

    $scope.refreshMetrics = function(predictionType, customMetricNames) {
        $scope.possibleMetrics = [];
        let toDropdownElems = function(a) {
            return a.map(m => [m, PMLFilteringService.getEvaluationMetricName(m)]);
        };
        if ('BINARY_CLASSIFICATION' === predictionType) {
            $scope.possibleMetrics = $scope.possibleMetrics.concat(toDropdownElems([
                'ACCURACY',
                'PRECISION',
                'RECALL',
                'F1',
                'COST_MATRIX',
                'ROC_AUC',
                'CUMULATIVE_LIFT',
                'AVERAGE_PRECISION',
                'LOG_LOSS',
                'CALIBRATION_LOSS',
                'DATA_DRIFT',
                'DATA_DRIFT_PVALUE',
                'MIN_KS',
                'MIN_CHISQUARE',
                'MAX_PSI',
                'PREDICTION_DRIFT_PSI',
                'PREDICTION_DRIFT_CHISQUARE'
            ]));
        }
        if ('MULTICLASS' === predictionType) {
            $scope.possibleMetrics = $scope.possibleMetrics.concat(toDropdownElems([
                'ACCURACY',
                'PRECISION',
                'RECALL',
                'F1',
                'ROC_AUC',
                'AVERAGE_PRECISION',
                'LOG_LOSS',
                'CALIBRATION_LOSS',
                'DATA_DRIFT',
                'DATA_DRIFT_PVALUE',
                'MIN_KS',
                'MIN_CHISQUARE',
                'MAX_PSI',
                'PREDICTION_DRIFT_PSI',
                'PREDICTION_DRIFT_CHISQUARE'
            ]));
        }
        if ('REGRESSION' === predictionType) {
            $scope.possibleMetrics = $scope.possibleMetrics.concat(toDropdownElems(['EVS', 'MAPE', 'MAE', 'MSE', 'RMSE', 'RMSLE', 'R2', 'PEARSON', 'DATA_DRIFT', 'DATA_DRIFT_PVALUE', 'MIN_KS', 'MIN_CHISQUARE', 'MAX_PSI', 'PREDICTION_DRIFT_PSI', 'PREDICTION_DRIFT_KS']));
        }
        if ('TIMESERIES_FORECAST' === predictionType) {
            $scope.possibleMetrics = $scope.possibleMetrics.concat(toDropdownElems([
                'MASE',
                'MAPE',
                'SMAPE',
                'MAE',
                'MEAN_ABSOLUTE_QUANTILE_LOSS',
                'MEAN_WEIGHTED_QUANTILE_LOSS',
                'MSE',
                'RMSE',
                'MSIS',
                'ND',
                'WORST_MASE',
                'WORST_MAPE',
                'WORST_SMAPE',
                'WORST_MAE',
                'WORST_MSE',
                'WORST_MSIS'
            ]));
        }
        if ('CAUSAL_BINARY_CLASSIFICATION' === predictionType || 'CAUSAL_REGRESSION' === predictionType) {
            $scope.possibleMetrics = $scope.possibleMetrics.concat(toDropdownElems(['AUUC', 'QINI', 'NET_UPLIFT']));
        }
        if ($scope.isLLM) {
            $scope.possibleMetrics = toDropdownElems(Object.keys(LLMEvaluationMetrics.nameByMetricCode));
        } else if (!predictionType) {
            $scope.possibleMetrics = [];
        }

        const ksSet = new Set();
        const psiSet = new Set();
        const chiSquareSet = new Set();
        const euclidianDistanceSet = new Set();
        const cosineSimilaritySet = new Set();
        const classifierGiniSet = new Set();

        if ($scope.evaluationHeaders){
            $scope.evaluationHeaders.forEach(addUnivariateMetrics);
            $scope.evaluationHeaders.forEach(addEmbeddingMetrics);
        }
        else if ($scope.evaluationDetails){ // Depends on which screen we are (interactive or MES)
            addUnivariateMetrics($scope.evaluationDetails);
            addEmbeddingMetrics($scope.evaluationDetails);
        }

        function addUnivariateMetrics(evaluation){
            if (evaluation.metrics.ksPerFeature){
                Object.keys(evaluation.metrics.ksPerFeature).forEach(key => ksSet.add(key + "_KS"));
            }
            if (evaluation.metrics.psiPerFeature){
                Object.keys(evaluation.metrics.psiPerFeature).forEach(key => psiSet.add(key + "_PSI"));
            }
            if (evaluation.metrics.chiSquarePerFeature){
                Object.keys(evaluation.metrics.chiSquarePerFeature).forEach(key => chiSquareSet.add(key + "_Chi-square"));
            }
        }

        function addEmbeddingMetrics(evaluation) {
            if (evaluation.metrics.euclidianDistancePerFeature){
                Object.keys(evaluation.metrics.euclidianDistancePerFeature).forEach(key => euclidianDistanceSet.add(key + "_ED"));
            }
            if (evaluation.metrics.cosineSimilarityPerFeature){
                Object.keys(evaluation.metrics.cosineSimilarityPerFeature).forEach(key => cosineSimilaritySet.add(key + "_CS"));
            }
            if (evaluation.metrics.classifierGiniPerFeature){
                Object.keys(evaluation.metrics.classifierGiniPerFeature).forEach(key => classifierGiniSet.add(key + "_Classifier_gini"));
            }
        }

        $scope.possibleMetrics = $scope.possibleMetrics.concat(toDropdownElems(Array.from(ksSet).concat(Array.from(psiSet), Array.from(chiSquareSet), Array.from(euclidianDistanceSet), Array.from(cosineSimilaritySet), Array.from(classifierGiniSet))));


        customMetricNames.forEach(metricName => {
            const metricId = CustomMetricIDService.getCustomMetricId(metricName);
            $scope.possibleMetrics.push([metricId, metricName]);
            $scope.possibleMetrics.push(["WORST_" + metricId , "Worst " + metricName])
        });

        if ($scope.uiState.currentMetric && $scope.possibleMetrics.filter(_ => _[0] == $scope.uiState.currentMetric).length == 0) {
            // old selected metric isn't possible anymore
            $scope.uiState.currentMetric = null;
        }
        if ($scope.uiState.currentMetric == null) {
            if ('BINARY_CLASSIFICATION' === predictionType) {
                $scope.uiState.currentMetric = 'ROC_AUC';
            }
            if ('MULTICLASS' === predictionType) {
                $scope.uiState.currentMetric = 'ROC_AUC';
            }
            if ('REGRESSION' === predictionType) {
                $scope.uiState.currentMetric = 'R2';
            }
        }

        var topObject = $scope.modelEvaluationStore || $scope.modelComparison;
        if (!topObject.displayParams.displayedMetrics || !topObject.displayParams.displayedMetrics.length || topObject.displayParams.predictionType != predictionType) {
            topObject.displayParams.predictionType = predictionType;
            topObject.displayParams.displayedMetrics = $scope.possibleMetrics.map(pm => pm[0])
                .filter(x => x)
                .filter(x => !((x.endsWith("Chi-square") && x !== "MIN_CHISQUARE" && x !== "PREDICTION_DRIFT_CHISQUARE")
                                || (x.endsWith("PSI") && x !== "MAX_PSI" && x !== "PREDICTION_DRIFT_PSI")
                                || (x.endsWith("KS") && x !== "MIN_KS" && x !== "PREDICTION_DRIFT_KS")
                                || (x.startsWith("ROUGE_2") || x.startsWith("ROUGE_L"))
                                || (('TIMESERIES_FORECAST' === predictionType) && x.startsWith("WORST_"))
                                ));
        }
        const possibleMetricNames = $scope.possibleMetrics.map(pm => pm[0]);
        if (topObject.displayParams.allAvailableMetrics) {
            const newMetrics = _.difference(possibleMetricNames, topObject.displayParams.allAvailableMetrics);
            if (newMetrics.length) {
                // we want the user to always see metrics she adds to the (S,LLM,_)ER
                topObject.displayParams.displayedMetrics.unshift(...newMetrics)
                topObject.displayParams.displayedMetrics = _.uniq(topObject.displayParams.displayedMetrics);
                topObject.displayParams.allAvailableMetrics = possibleMetricNames;
            }
        } else {
            topObject.displayParams.allAvailableMetrics = possibleMetricNames;
        }
        $scope.refreshCurrentMetricNames();
    }

    $scope.refreshCurrentMetricNames = function() {
        var topObject = $scope.modelEvaluationStore || $scope.modelComparison;
        if (topObject && topObject.displayParams.displayedMetrics && $scope.possibleMetrics) {
            $scope.uiState.currentFormattedNames = topObject.displayParams.displayedMetrics
                .filter(dm => dm).map(cur => {
                    const foundItem = $scope.possibleMetrics.find(x => x[0] === cur);
                    if (foundItem) {
                        return {
                            key: cur,
                            label: foundItem[1]
                        };
                    }
                });
        } else {
            $scope.uiState.currentFormattedNames = [];
        }
        $scope.refreshMetricsValues();
    }

    $scope.refreshMetricsValues = function() {
        var topObject = $scope.modelEvaluationStore || $scope.modelComparison;
        let refs;
        if ($scope.uiState.refs && $scope.uiState.refs.length) {
            refs = $scope.uiState.refs;
        } else if ($scope.ctrl && $scope.ctrl.refs && $scope.ctrl.refs.length) {
            refs = $scope.ctrl.refs;
        }
        if (refs) {
            for (let item of refs) {
                item.$formattedMetrics = {};
                if (topObject.displayParams.displayedMetrics) {
                    for (let metric of topObject.displayParams.displayedMetrics) {
                        item.$formattedMetrics[metric] = MetricsUtils.getMetricValue(item.metrics, metric);
                    }
                }
            }
        }
    }

});

app.component("meLikesTagsSelector", {
    bindings: {
        titlePrefix: '<',
        selectedLabels: '=',
        availableLabels: '<',
        selectedTextFormat: '<',
        width: '<',
        single: '<?'
    },
    template: `
    <select dku-bs-select="{'style': 'dku-select-button', titlePrefix: $ctrl.titlePrefix, width: $ctrl.width, selectedTextFormat: $ctrl.selectedTextFormat}"
        multiple="multiple"
        ng-model="$ctrl.selectedLabels"
        ng-options="l as $ctrl.labelSuffix(l) group by $ctrl.labelPrefix(l) for l in $ctrl.availableLabels"
        ng-if="!$ctrl.single">
    </select>
    <select dku-bs-select="{'style': 'dku-select-button', titlePrefix: $ctrl.titlePrefix, width: $ctrl.width, selectedTextFormat: $ctrl.selectedTextFormat}"
        ng-model="$ctrl.selectedLabels"
        ng-options="l as $ctrl.labelSuffix(l) group by $ctrl.labelPrefix(l) for l in $ctrl.availableLabels"
        class="show-tick"
        ng-if="$ctrl.single">
    </select>
    `,
    controller: function() {
        const $ctrl = this;
        $ctrl.labelPrefix = function(labelKey) {
            return getDomainLabel(labelKey);
        }

        $ctrl.labelSuffix = function(labelKey)  {
            return getDomainSubLabel(labelKey);
        }

        function refresh() {
        }

        $ctrl.$onChanges = function() {
            refresh();
        }
    }
});

/* ************************************ Evaluations list and perf drift *************************** */

app.controller("ModelEvaluationStoreListCommon", function($scope, DataikuAPI, $stateParams, TopNav, PMLFilteringService, PMLSettings, ActiveProjectKey, Fn, $filter, ModelEvaluationUtils, $controller, WT1) {
    $controller("ModelEvaluationsMetricsHandlingCommon", {$scope, PMLFilteringService, PMLSettings});

    TopNav.setLocation(TopNav.TOP_MODEL_EVALUATION_STORES, TopNav.LEFT_MODEL_EVALUATION_STORES, TopNav.TABS_MODEL_EVALUATION_STORE, "evaluations");

    $scope.uiState = {
        refs: [],
        titleLabels: []
    };

    $scope.refreshStatus = function() {
        const mesId = $stateParams.mesId || $scope.mesId;
        DataikuAPI.modelevaluationstores.listEvaluations(ActiveProjectKey.get(), mesId).success(function(data) {
            const isOnOpening = $scope.evaluationHeaders === undefined;
            $scope.infoMessages = data.infoMessages;
            $scope.evaluationHeaders = data.evaluationHeaders;
            TopNav.setItem(TopNav.ITEM_MODEL_EVALUATION_STORE, mesId, {name: data.modelEvaluationStore.name});
            $scope.isLLM = $scope.evaluationHeaders && $scope.evaluationHeaders.length && $scope.evaluationHeaders[0].evaluation.type === 'llm';
            $scope.predictionType = $scope.evaluationHeaders && $scope.evaluationHeaders.length?$scope.evaluationHeaders[0].evaluation.predictionType:null;
            $scope.metricParamsList = $scope.evaluationHeaders && $scope.evaluationHeaders.map(evaluationHeader => evaluationHeader.evaluation.metricParams);
            $scope.computeRefs();
            WT1.event('model-evaluation-store-refresh-status', {nbOfMe: data.evaluationHeaders.length, isLLM: $scope.isLLM, predictionType: $scope.predictionType, onOpening: isOnOpening});
        }).error(setErrorInScope.bind($scope));
    };

    $scope.$watch("modelEvaluationStore", (nv) => { if (nv) $scope.refreshStatus(); });

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: ''
        },
        filterParams: {
            userQueryTargets: ["evaluationId", "trainDataSetName", "modelName", "evaluationDatasetName", "value"],
            propertyRules: {}
        },
        orderQuery: "-value",
        orderReversed: false
    }, $scope.selection || {});

    $scope.deleteSelectedEvaluations = function() {
        const mesId = $stateParams.mesId || $scope.mesId;
        DataikuAPI.modelevaluationstores.deleteEvaluations(ActiveProjectKey.get(), mesId,
                $scope.selection.selectedObjects.filter(function(o){return !o.active}).map(Fn.prop('ref')).map(Fn.prop('evaluationId')))
            .success($scope.refreshStatus)
            .error(setErrorInScope.bind($scope));
    };

    $scope.computeRefs = function() {
        if ($scope.evaluationHeaders) {
            $scope.uiState.refs = $scope.evaluationHeaders.map(x => {
                let ret = ModelEvaluationUtils.makeRefDisplayItemFromEvaluation(x.evaluation, [$scope.modelEvaluationStore]);
                ret.metrics = x.metrics;
                ret.mlDiagnostics = x.mlDiagnostics;
                return ret;
            });

            $scope.refreshMetrics($scope.predictionType, $scope.getAllCustomMetricNames());
        } else {
            $scope.uiState.refs = [];
        }
        $scope.refreshMetricsValues();
    }

    DataikuAPI.modelevaluationstores.listWithAccessible($stateParams.projectKey).success(function(data){
        $scope.storeList = data;
    });

    $scope.computeTitleLabels = function() {
        $scope.uiState.titleLabels = ($scope.uiState.shownLabels && $scope.uiState.shownLabels.length)?generateTitleLabelsFromRefs($scope.uiState.shownLabels):[];
        $scope.uiState.domainLabels = $scope.uiState.titleLabels.filter(x => x.domain);
    }

    $scope.$watch("uiState.shownLabels", $scope.computeTitleLabels);

    $scope.computeAllLabels = function() {
        $scope.uiState.possibleLabels = _.sortBy(_.uniq(_.flatten($scope.uiState.refs.map(r => Array.from(r.labels.keys())))));
        if ($scope.uiState.possibleLabels && $scope.uiState.possibleLabels.length && !$scope.uiState.shownLabels) {
            $scope.uiState.shownLabels = [DEFAULT_X_LABEL].concat(DEFAULT_Y_LABELS);
        }
        if ($scope.uiState.shownLabels && $scope.uiState.shownLabels.length) {
            $scope.uiState.shownLabels = $scope.uiState.shownLabels.filter(l => $scope.uiState.possibleLabels.includes(l));
        }
    }

    $scope.$watch('uiState.refs', $scope.computeAllLabels);

    $scope.$watch("modelEvaluationStore.displayParams", () => {
        if ($scope.$root && $scope.$root.projectSummary && $scope.$root.projectSummary.canWriteProjectContent
            && $scope.savedSettings && $scope.modelEvaluationStore && !angular.equals($scope.savedSettings.displayParams, $scope.modelEvaluationStore.displayParams)) {
            $scope.save();
        }
    }, true)
});

app.controller("ModelEvaluationStoreEvaluationsController", function($scope, $filter, $stateParams, TopNav, $controller, ModelEvaluationUtils, PMLFilteringService, DataikuAPI, ActiveProjectKey, Dialogs) {
    $controller("ModelEvaluationStoreListCommon", {$scope});
    TopNav.setLocation(TopNav.TOP_MODEL_EVALUATION_STORES, TopNav.LEFT_MODEL_EVALUATION_STORES, TopNav.TABS_MODEL_EVALUATION_STORE, "evaluations");

    $scope.onReverseDisplay = function(items) {
        if (!items || !items.length) {
            return;
        }
        if ($scope.modelEvaluationStore.displayParams.disabledMERunIds.includes(items[0].ref.evaluationId)) {
            for(const item of items) {
                $scope.modelEvaluationStore.displayParams.disabledMERunIds =
                    $scope.modelEvaluationStore.displayParams.disabledMERunIds.filter(d => d !== item.ref.evaluationId);
            }
        } else {
            for(const item of items) {
                $scope.modelEvaluationStore.displayParams.disabledMERunIds.push(item.ref.evaluationId);
            }
        }
        $scope.computeRefs();
    }

    $scope.onDeleteClicked = function(items) {
        Dialogs.confirm($scope, 'Confirm deletion', 'Are you sure you want to delete ' + items.length + ' model evaluation' + (items.length > 1 ? 's' : '') + ' ?')
            .then(() => $scope.onDelete(items))
            .catch(()=> {});
    }

    $scope.onDelete = function(items) {
        const mesId = $stateParams.mesId || $scope.mesId;
        DataikuAPI.modelevaluationstores.deleteEvaluations(ActiveProjectKey.get(), mesId,
        items.map(i => i.ref.evaluationId))
            .success($scope.refreshStatus)
            .error(setErrorInScope.bind($scope));
    }
});

app.service("ModelEvaluationUtils", function($stateParams, CreateExportModal, DataikuAPI, FutureProgressModal, $state, Dialogs) {
    function addOrderToTitle(title, order) {
        if (!title) {
            return null;
        }
        if (!order) {
            return title;
        }
        if (!isNaN(order)) {
            let intOrder = parseInt(order);
            if (intOrder > 1451606400) // 2016/1/1 00:00:00
                return title + " " + moment(intOrder).format("YYYY-MM-DD HH:mm:ss");
        }
        return title + " " + order;
    }

    function processDatasetOrder(datasetParams) {
        return processOrder(datasetParams.generationDate);
    }

    function processOrder(order) {
        if (!order) {
            return null;
        }
        if (!isNaN(order)) {
            let intOrder = parseInt(order);
            if (intOrder > 1451606400) // 2016/1/1 00:00:00
                return moment(intOrder).format("YYYY-MM-DD HH:mm:ss");
            return intOrder;
        }
        return order;
    }

    var makeRefDisplayItemFromEvaluation = function (evaluation, storeList) {
        if (!evaluation) {
            return null;
        }
        if (!storeList) {
            storeList = [];
        }
        let modelUniqueId = null;
        const trainDataSetOrder = evaluation.trainDataParams ? processDatasetOrder(evaluation.trainDataParams) : null;
        let trainDataSetName = (evaluation.trainDataParams && evaluation.trainDataParams.datasetName)?evaluation.trainDataParams.datasetName: "Unknown train dataset";
        trainDataSetName = addOrderToTitle(trainDataSetName, trainDataSetOrder);

        const modelOrder = (evaluation.modelParams && evaluation.modelParams.trainEndTime)?processOrder(evaluation.modelParams.trainEndTime):null;
        let modelName = (evaluation.modelParams && evaluation.modelParams.versionName)?evaluation.modelParams.versionName:"Unknown model";
        if ("SAVED_MODEL" === evaluation.modelType) {
            modelUniqueId = "S-" + evaluation.ref.projectKey + "-" + evaluation.modelParams.ref + "-" + evaluation.modelParams.versionId;
        }

        // cheating a bit : setting evaluation dataset generation date to evaluation creation date - which is right as of now
        evaluation.dataParams.generationDate = evaluation.created;
        const evaluationDataSetOrder = evaluation.dataParams?processDatasetOrder(evaluation.dataParams):null;
        let evaluationDatasetName = (evaluation.dataParams && evaluation.dataParams.ref)?evaluation.dataParams.ref:"Unknown test dataset";
        evaluationDatasetName = addOrderToTitle(evaluationDatasetName, evaluationDataSetOrder);

        const store = storeList.find(s => (s.id === evaluation.ref.id) && (s.projectKey === evaluation.ref.projectKey));
        const isDisabled = store?store.displayParams.disabledMERunIds.includes(evaluation.ref.evaluationId):false;
        const storeName = store?store.name:null;
        let subtitle = undefined;
        if (evaluation.userMeta.labels) {
            const label = evaluation.userMeta.labels.find(l => l.key === "model:name");
            if (label)  {
                subtitle = label.value;
            }
        }

        return {
            trainDataSetName: trainDataSetName,
            trainDataSetOrder: trainDataSetOrder,
            evaluationDatasetName: evaluationDatasetName,
            evaluationDataSetOrder: evaluationDataSetOrder,
            modelName: modelName,
            modelUniqueId: modelUniqueId,
            modelOrder: modelOrder,
            predictionType: evaluation.predictionType,
            llmTaskType: evaluation.llmTaskType,
            evaluationId: evaluation.ref.evaluationId,
            ref: evaluation.ref,
            refStr: evaluation.userMeta.name,
            refSubtitle: subtitle,
            modelLikeType: "MODEL_EVALUATION",
            labels: (evaluation.userMeta.labels?evaluation.userMeta.labels:[]).reduce((map, obj) => {
                map.set(obj.key, obj.value);
                return map;
            }, new Map()),
            isDisabled,
            ...(storeName && { storeName }),
            isPartitioned: evaluation.modelParams ? evaluation.modelParams.isPartitioned : null,
            isEnsembled: evaluation.modelParams ? evaluation.modelParams.isEnsembled : null,
            modelType: evaluation.modelType,
        };
    };

    var makeRefDisplayItemFromFMInfo = function(modelInfo, savedModelList, analysesWithHeads) {
        if (!savedModelList) {
            savedModelList = [];
        }
        const trainDataSetOrder = modelInfo.trainDataParams?processDatasetOrder(modelInfo.trainDataParams):null;
        let trainDataSetName = (modelInfo.trainDataParams && modelInfo.trainDataParams.datasetName)?modelInfo.trainDataParams.datasetName: "Unknown train dataset";
        trainDataSetName = addOrderToTitle(trainDataSetName, trainDataSetOrder);

        const modelOrder = (modelInfo.modelParams && modelInfo.modelParams.trainEndTime)?processOrder(modelInfo.modelParams.trainEndTime):null;

        const labels = modelInfo.userMeta.labels?modelInfo.userMeta.labels.reduce((map, obj) => {
            map.set(obj.key, obj.value);
            return map;
        }, new Map()):new Map();
        let refStr;
        let mlTaskName = null;
        let smName = null;
        let sm = null;
        switch (modelInfo.ref.type) {
            case "ANALYSIS": {
                let analysisName = "unknown";
                mlTaskName = "unknown";
                const analysis = analysesWithHeads.find(anal => anal.id == modelInfo.ref.taskLoc.analysisId);
                if (analysis) {
                    analysisName = analysis.name;
                    const mlTask = analysis.mlTasks.find(mlTask => mlTask.mlTaskId == modelInfo.ref.taskLoc.mlTaskId);
                    if (mlTask) {
                        mlTaskName = mlTask.name;
                    }
                }
                refStr = `${modelInfo.modelParams.name}, session ${modelInfo.ref.sessionId.replace(/s/,'')} of '${mlTaskName}' of '${analysisName}'`;
                break;
            }
            case "SAVED": {
                sm = savedModelList.find(s => (s.id === modelInfo.ref.smId) && (s.projectKey === modelInfo.ref.projectKey));
                smName = sm?sm.name:"unknown";
                const smStr = (sm ? sm.name : modelInfo.ref.smId) + ($stateParams.projectKey == modelInfo.ref.projectKey ? '' : ` in ${modelInfo.ref.projectKey}`);
                refStr = `version ${modelInfo.ref.smVersionId} of ${smStr}`
                break;
            }
            default:
                throw new Error("Unexpected type: " + modelInfo.ref.type);
        }
        return {
            trainDataSetName: trainDataSetName,
            trainDataSetOrder: trainDataSetOrder,
            evaluationDatasetName: null,
            evaluationDataSetOrder: -1, // ###TODO change this
            modelName: modelInfo.modelName,
            modelOrder: modelOrder,
            modelUniqueId: modelInfo.ref.fullId,
            modelTaskType: modelInfo.modelTaskType,
            ref: modelInfo.ref,
            refStr: refStr,
            savedModelType: sm ? sm.savedModelType : undefined,
            taskType: (sm && sm.miniTask) ? sm.miniTask.taskType : undefined,
            backendType: modelInfo.details ? modelInfo.details.backendType : undefined,
            proxyModelProtocol: (sm && sm.proxyModelConfiguration) ? sm.proxyModelConfiguration.protocol : undefined,
            modelLikeType: "DOCTOR_MODEL",
            labels: labels,
            ...(smName && { smName }),
            ...(mlTaskName && { mlTaskName }),
            modelType: modelInfo.modelType
        };
    };

    var makeRefDisplayItemFromComparable = function(comparable, storeList, savedModelList, analysesWithHeads) {
        if (!comparable) {
            return null;
        }
        if (comparable instanceof Array) {
            return comparable.map(cur => makeRefDisplayItemFromComparable(cur, storeList, savedModelList, analysesWithHeads));
        }
        if ("DOCTOR_MODEL" === comparable.modelLikeType) {
            return makeRefDisplayItemFromFMInfo(comparable, savedModelList, analysesWithHeads);
        } else if("MODEL_EVALUATION" === comparable.modelLikeType) {
            return makeRefDisplayItemFromEvaluation(comparable, storeList);
        } else {
            throw new Error(`Unknown model-like type ${comparable.modelLikeType}`) ;
        }
    }

    var computeReferenceLabels = function(references) {
        if (!references || !references.length) {
            return [];
        }
        return references.map(r => {
            if ("DOCTOR_MODEL" === r.modelLikeType) {
                r.inputDriftLabel = `Dataset ${r.trainDataSetName} used to train model ${r.modelName}`;
                r.perfDriftLabel = `Model ${r.modelName} trained on ${r.trainDataSetName}`;
            } else if ("MODEL_EVALUATION" === r.modelLikeType) {
                r.inputDriftLabel  = `Dataset ${r.evaluationDatasetName} used to evaluate Model ${r.modelName} trained on ${r.trainDataSetName}`;
                r.perfDriftLabel = `Model ${r.modelName} trained on ${r.trainDataSetName} evaluated on ${r.evaluationDatasetName}`;
            } else {
                throw new Error(`Unknown model-like type ${r.modelLikeType}`) ;
            }
            r.inputDriftLabel += ` (${r.refStr})`;
            r.perfDriftLabel += ` (${r.refStr})`;
            return r;
        });
    }

    const hasNoAssociatedModelText = function(evaluationDetails, fullModelId) {
        if (!fullModelId) {
            return "Not available for evaluations without a model";
        }
        if (evaluationDetails) {
            if (evaluationDetails.evaluation.modelType === 'EXTERNAL') {
                return "Not available for model evaluations without a backing DSS model";
            }
            if (evaluationDetails.backingModelVersionDeleted) {
                return "Not available as the evaluated DSS model version is no longer available";
            }
        }
        return null;
    }

    const defaultDriftParams = {
        nbBins: 20,
        confidenceLevel: 0.95,
        psiThreshold: 0.2,
        columns: {}
    }

    const CLASSIC_EVALUATION_DATASET_TYPE = "CLASSIC";
    const API_NODE_LOGS_EVALUATION_DATASET_TYPE = "API_NODE_LOGS";
    const CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE = "CLOUD_API_NODE_LOGS";
    const SAGEMAKER_EVALUATION_DATASET_TYPE = "SAGEMAKER_LOGS";

    const evaluationDatasetTypesFormatted = {
        "API_NODE_LOGS": "API node logs",
        "CLOUD_API_NODE_LOGS": "API node logs", // Same name than api node logs ; we use internal the cloud prefix because there is a different treatment behind the scenes
        "SAGEMAKER_LOGS": "SageMaker data capture logs",
    };

    const API_NODE_FEATURE_PREFIX = "clientEvent.features.";
    const API_NODE_PREDICTION_COLUMN = "clientEvent.result.prediction";
    const CLOUD_API_NODE_FEATURE_PREFIX = "message.features.";
    const CLOUD_API_NODE_PREDICTION_COLUMN = "message.result.prediction";
    const SAGEMAKER_FEATURE_PREFIX = "captureData.";
    const SAGEMAKER_PREDICTION_COLUMN = "captureData.endpointOutput.data";
    const CLASSIC_PREDICTION_COLUMN = "prediction";
    const TIMESERIES_FORECAST_COLUMN = "forecast";

    const INPUT_DATA_DRIFT_TYPE = "Input data drift";
    const PREDICTION_DRIFT_TYPE = "Prediction drift";
    const PERFORMANCE_DRIFT_TYPE = "Performance drift";

    const tabNameToDriftType = {
        "tabular-input_data_drift": INPUT_DATA_DRIFT_TYPE,
        "tabular-prediction_drift": PREDICTION_DRIFT_TYPE,
        "tabular-performance_drift": PERFORMANCE_DRIFT_TYPE,
    }

    const REFERENCE_SELECTOR = "Reference selector";
    const COMPUTE_BUTTON = "Compute button";
    const COMPUTE_FROM_URL = "Auto compute from url parameter";
    const CHANGE_COLUMN_PARAMS = "Change column params";

    const isLogsDatasetType = datasetType => [API_NODE_LOGS_EVALUATION_DATASET_TYPE, CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE, SAGEMAKER_EVALUATION_DATASET_TYPE].includes(datasetType);

    const wt1Properties = {
        INPUT_DATA_DRIFT_TYPE,
        PREDICTION_DRIFT_TYPE,
        PERFORMANCE_DRIFT_TYPE,
        REFERENCE_SELECTOR,
        COMPUTE_BUTTON,
        COMPUTE_FROM_URL,
        CHANGE_COLUMN_PARAMS: CHANGE_COLUMN_PARAMS
    }

    function synchronizeLimitSamplingUiStateAndDesc($scope) { // This function avoids a migration on desc.limitSampling that has been introduced later and is true by default;
        const isLimitSamplingUndefined = $scope.desc.limitSampling === undefined;
        $scope.uiState.limitSampling = isLimitSamplingUndefined ? true : $scope.desc.limitSampling;
        $scope.$watch('uiState.limitSampling', (nv, ov) => {
            if (nv !== ov) {
                if (nv === true && isLimitSamplingUndefined) {
                    $scope.desc.limitSampling = undefined;
                } else {
                    $scope.desc.limitSampling = nv;
                }
            }
        });
    }

    function showProgressDialog(scope, futureResponse, destinationDatasetName, dataTypeName, warningMessage) {
        FutureProgressModal.show(scope, futureResponse, "Exporting the " + dataTypeName).then(function(result) {
            if (result) {
                const datasetPage = "projects.project.datasets.dataset.explore";
                const datasetPageParams = {projectKey: $stateParams.projectKey, datasetName: destinationDatasetName};
                const datasetLink = `<a href="` + $state.href(datasetPage, datasetPageParams) + `">` + sanitize(destinationDatasetName) + "</a>"
                let message;
                if (warningMessage) {
                    message = `<p class="alert alert-warning">Dataset ` + datasetLink + ` has been created.<br />` + sanitize(warningMessage) + `</p>`;
                } else {
                    message = `<p class="alert alert-success">Dataset ` + datasetLink + ` has been created successfully.</p>`;
                }
                Dialogs.confirm(scope,
                    "Exported the " + dataTypeName, message,
                    {
                        btnCancel: "Close",
                        btnConfirm: "View dataset",
                        positive: true,
                    }).then(function() {
                        $state.go(datasetPage, datasetPageParams);
                    }, function() {
                        // Dialog closed
                    });
            }
        });
    }

    function exportEvaluationSample(scope, fme) {
        const datasetDialog = {
            title : "Export the evaluation sample"
        };
        const features = {  // only allow export to dataset
            hideDestinationTabs: true,
            hideAdvancedParameters: true
        };
        CreateExportModal(scope, datasetDialog, features, { destinationType: 'DATASET' }).then(function(params) {
            DataikuAPI.modelevaluations.exportEvaluationSample(fme, params).then(function(response) {
                showProgressDialog(scope, response.data.futureResponse, params.destinationDatasetName, "evaluation sample");
            }).catch(function(response) {
                Dialogs.error(scope, "An error occurred while exporting the evaluation sample", response.data.detailedMessageHTML);
            });
        });
    }


    return {
        makeRefDisplayItemFromEvaluation,
        makeRefDisplayItemFromFMInfo,
        makeRefDisplayItemFromComparable,
        hasNoAssociatedModelText,
        computeReferenceLabels,
        defaultDriftParams,
        CLASSIC_EVALUATION_DATASET_TYPE,
        API_NODE_LOGS_EVALUATION_DATASET_TYPE,
        CLOUD_API_NODE_LOGS_EVALUATION_DATASET_TYPE,
        SAGEMAKER_EVALUATION_DATASET_TYPE,
        evaluationDatasetTypesFormatted,
        API_NODE_FEATURE_PREFIX,
        API_NODE_PREDICTION_COLUMN,
        CLOUD_API_NODE_FEATURE_PREFIX,
        CLOUD_API_NODE_PREDICTION_COLUMN,
        SAGEMAKER_FEATURE_PREFIX,
        SAGEMAKER_PREDICTION_COLUMN,
        CLASSIC_PREDICTION_COLUMN,
        TIMESERIES_FORECAST_COLUMN,
        tabNameToDriftType,
        wt1Properties,
        synchronizeLimitSamplingUiStateAndDesc,
        isLogsDatasetType,
        exportEvaluationSample
    }
});

const DEFAULT_X_LABEL = "evaluation:date";
const DEFAULT_Y_LABELS = ["model:name", "evaluationDataset:dataset-name", "referenceDataset:dataset-name"];
const DEFAULT_SELECTED_LABELS = [DEFAULT_X_LABEL].concat(DEFAULT_Y_LABELS); // in 'driftReferencesSelector'


function generateTitleLabelsFromRefs(refLabels) {
    var titleLabels = [];
    let curDomainTitle = null;
    for (const curLabel of refLabels) {
        const labelParts = curLabel.split(":");
        if (1 == labelParts.length) {
            curDomainTitle = null;
            titleLabels.push({
                domain: "-",
                subLabel: curLabel,
                fullLabel: curLabel
            });
            continue;
        }
        titleLabels.push({
            parent:labelParts[0],
            domain: labelParts[0] !== curDomainTitle?labelParts[0]:null,
            subLabel: labelParts.slice(1).join(":"),
            fullLabel: curLabel
        });
        curDomainTitle = labelParts[0];
    }

    for (let i = titleLabels.length-1 ; i >=0 ; i--) {
        if (titleLabels[i].domain) {
            titleLabels[i].span = titleLabels.filter(x => x.parent == titleLabels[i].parent).length
        }
    }
    return titleLabels;
}



function getDomainLabel(labelKey) {
    const labelParts = labelKey.split(":");
    if (1 >= labelParts.length) {
        return "(no domain)";
    }
    return labelParts[0];
}

function getDomainSubLabel(labelKey) {
    const labelParts = labelKey.split(":");
    if (1 >= labelParts.length) {
        return labelKey;
    }
    return labelParts.slice(1).join(":");
}

app.controller('DriftReferencesSelectorModalController', function($scope, CollectionFiltering) {


});

app.component('driftReferencesSelector',{
    bindings: {
        ref: '=', // Generated from ModelLikeEvaluationInfo (Java)
        cur: '<', // Generated from ModelEvaluationUtils.makeRefDisplayItemFromEvaluation() (JS)
        compatibleReferences: '<',
        action: '<',
        fnLabel: '<',
        refLabels: '=',
        driftParams: '<', // Used to show the configuration popover within this directive
        wt1Params: '<', // Used for wt1
        canCompute: '<' // Whether to show the "Compute" button or not
    },
    templateUrl: '/templates/modelevaluationstores/drift-references-selector.html',
    controller: function ctrlModelLikesInfo($scope, ClipboardUtils, openDkuPopin, $q, CreateModalFromTemplate, TreeViewSelectorUtils, PMLSettings, ModelEvaluationUtils) {
        $scope.$ctrl = this;

        $scope.wt1Properties = ModelEvaluationUtils.wt1Properties;

        // State of the drift reference selector widget
        $scope.uiState = {
            titleLabels: [],
            selectedLabels: [],
            allLabels: []
        };

        // State of the drift reference selector modal
        $scope.driftModalState = {
            // Tree view content
            treeRoot: null,

            // User filter
            searchQuery: '',

            // List of ME-like after filtering (null = uninitialized)
            filteredCompatibleReferences: null,

            // Selected rows in the tree view (can only be 0 or 1, null = uninitialized)
            selectedLeafs: null,

            // Expanded rows in the tree view
            expandedNodes: [],

            // List of all selected labels (~ columns of the tree view)
            selectedLabels: angular.copy($scope.uiState.selectedLabels),

            // Filtering on compatible references only
            showOnlyCompatibleReferences: true,
        }

        $scope.getSelectedComparable = ()=> {
            if($scope.driftModalState.selectedLeafs) {
                if($scope.driftModalState.selectedLeafs.length > 0) {
                    return $scope.driftModalState.selectedLeafs[0].payload;
                }
            } else {
                if($scope.$ctrl.ref) {
                    return $scope.$ctrl.compatibleReferences.find(c => c.mli.fullId == $scope.$ctrl.ref.mli.fullId)
                }
            }
            return null;
        }

        $scope.getExpandedComparables = ()=> {
            return $scope.driftModalState.expandedNodes.flatMap(n => $scope.toComparables(n))
        }

        $scope.toComparables = (node) => {
            if(node.children.some(c => c.type == 'leaf')) {
                return node.children.flatMap(c => c.payload);
            }
        }

        $scope.queryMatch = function(ref, query) {
            const lcQuery = query.toLowerCase()
            return (ref.analysisName && ref.analysisName.toLowerCase().includes(lcQuery))
                || (ref.modelName && ref.modelName.toLowerCase().includes(lcQuery))
                || (ref.savedModelName && ref.savedModelName.toLowerCase().includes(lcQuery))
                || (ref.evaluationName && ref.evaluationName.toLowerCase().includes(lcQuery))
                || (ref.displayName && ref.displayName.toLowerCase().includes(lcQuery))
                || (ref.storeName && ref.storeName.toLowerCase().includes(lcQuery))
                || (ref.labels && ref.labels['evaluation:date'] && ref.labels['evaluation:date'].toLowerCase().includes(lcQuery))
                || (ref.labels && ref.labels['evaluationDataset:dataset-name'] && ref.labels['evaluationDataset:dataset-name'].toLowerCase().includes(lcQuery))
        }


        $scope.currentLabel = function() {
            return $scope.$ctrl.cur.refStr;
        }

        $scope.referenceLabel = function() {
            if (!$scope.$ctrl.ref) {
                return null;
            }
            let ret;
            if ($scope.$ctrl.ref.type.includes('MODEL_EVALUATION')) { // tabular or llm ME
                ret = 'Model Evaluation';
            } else if ($scope.$ctrl.ref.type == 'SAVED_MODEL_VERSION') {
                    ret = 'Train-time evaluation';
            } else if ($scope.$ctrl.ref.type == 'ANALYSIS_TRAINED_MODEL') {
                ret = 'Model from Analysis';
            }
            if($scope.$ctrl.cur.modelUniqueId && $scope.$ctrl.ref.modelUniqueId) {
                if ($scope.$ctrl.cur.modelUniqueId == $scope.$ctrl.ref.modelUniqueId) {
                ret += ' (Same model)';
                } else {
                    ret += ' (Different model)';
                }

            } else {
                if ($scope.$ctrl.cur.ref.evaluationId == $scope.$ctrl.ref.mli.evaluationId) {
                    ret += ' (Same evaluation';
                    if ($scope.$ctrl.ref.type.includes('MODEL_EVALUATION')  // tabular or llm ME
                    && $scope.$ctrl.ref.modelType == 'EXTERNAL'
                    && $scope.$ctrl.ref.hasDriftReference) {
                        ret += ', reference dataset)';
                    } else {
                        ret+= ")"
                    }
                } else {
                    ret += ' (Different evaluation)';
                }
            }
            return ret;
        }

        $scope.changeReference = function() {
            $scope.driftModalState.selectedLabels = angular.copy($scope.uiState.selectedLabels);
            const deferred = $q.defer();
            CreateModalFromTemplate("/templates/modelevaluationstores/drift-references-selector-modal.html", $scope, "DriftReferencesSelectorModalController", function(newScope) {
                newScope.acceptDeferred = deferred;

                // Recontruct the tree view
                function refreshTree() {
                    if(!$scope.driftModalState.filteredCompatibleReferences) {
                        return;
                    }
                    const comparablesWithGroups = $scope.driftModalState.filteredCompatibleReferences
                        .filter(ref => !$scope.driftModalState.showOnlyCompatibleReferences || ref.isCompatibleReference)
                        .map(comparableModelItem => {
                            let groups = [];
                            if(comparableModelItem.type == 'ANALYSIS_TRAINED_MODEL') {
                                groups = ['Lab models', comparableModelItem.analysisName, comparableModelItem.mlTaskName, comparableModelItem.mli.sessionId];
                            }
                            if(comparableModelItem.type.includes('MODEL_EVALUATION')) {  // tabular or llm ME
                                groups = ['Model evaluation stores', comparableModelItem.storeName];
                            }
                            if(comparableModelItem.type == 'SAVED_MODEL_VERSION') {
                                groups = ['Saved model versions', comparableModelItem.savedModelName];
                            }
                            return {
                                payload: comparableModelItem,
                                groups: groups
                            };
                        });

                    const selectedComparable = $scope.getSelectedComparable();
                    const expandedComparables = $scope.getExpandedComparables();
                    const sortedComparablesWithGroups = _.orderBy(comparablesWithGroups, cwg => cwg.payload.createdOn, 'desc');
                    [newScope.driftModalState.treeRoot, newScope.driftModalState.selectedLeafs, newScope.driftModalState.expandedNodes] = TreeViewSelectorUtils.treeify(sortedComparablesWithGroups, (comparable) => {
                        const node = {
                            type: 'leaf',
                            data: [],
                            payload: comparable,
                            // Keep the currently selected item selected
                            preselected: selectedComparable == comparable,

                            // Keep the currently selected item always expanded
                            preexpanded: selectedComparable == comparable || expandedComparables.includes(comparable)
                        };
                        node.data.push(comparable.displayName);
                        for (const label of newScope.driftModalState.selectedLabels) {
                            node.data.push(comparable.labels[label]);
                        }
                        if (comparable.isPartitioned){
                            node.disabled = true;
                            node.message = "Item is partitioned and can not be selected as drift reference"
                        }
                        if (!["BINARY_CLASSIFICATION", "MULTICLASS", "REGRESSION", "TIMESERIES_FORECAST"].includes(comparable.modelTaskType)){
                            node.disabled = true;
                            const formatedModelTaskType = PMLSettings.task.predictionTypes.find(e => e.type === comparable.modelTaskType);
                            node.message = `Item has prediction type "${formatedModelTaskType ? formatedModelTaskType.fullName : comparable.modelTaskType}" thus can not be selected as drift reference`;
                        }
                        return node;
                    });
                    newScope.driftModalState.displayedColumns = ['', ...newScope.driftModalState.selectedLabels];
                };

                refreshTree();

                newScope.$watch('driftModalState.selectedLabels', ()=> refreshTree(), true);
                newScope.$watch('driftModalState.showOnlyCompatibleReferences', ()=> refreshTree(), true);
                newScope.$watch('driftModalState.searchQuery', searchQuery => {
                    const selectedComparable = newScope.getSelectedComparable();
                    newScope.driftModalState.filteredCompatibleReferences = $scope.$ctrl.compatibleReferences ?
                        $scope.$ctrl.compatibleReferences
                            .filter(ref => searchQuery == "" || newScope.queryMatch(ref, searchQuery)) : []
                    if(selectedComparable && !newScope.driftModalState.filteredCompatibleReferences.includes(selectedComparable)) {
                        // Force the selected ME-like to always stay
                        newScope.driftModalState.filteredCompatibleReferences.push(selectedComparable);
                    }
                    refreshTree();
                });

                newScope.$on("$destroy",function() {
                    if(newScope.acceptDeferred) {
                        newScope.acceptDeferred.reject();
                    }
                    newScope.acceptDeferred = null;
                });
                newScope.sortValue = sortColumn => me => me.labels.get(sortColumn);

                newScope.save = function() {
                    if(newScope.driftModalState.selectedLeafs && newScope.driftModalState.selectedLeafs.length > 0) {
                        newScope.acceptDeferred.resolve(newScope.driftModalState.selectedLeafs[0].payload);
                    }
                    newScope.dismiss();
                }
            });
            deferred.promise.then((newRef)=> {

                $scope.$applyAsync(() => $scope.$ctrl.action({from: ModelEvaluationUtils.wt1Properties.REFERENCE_SELECTOR, driftType: $scope.$ctrl.wt1Params['driftType']}));
                $scope.$ctrl.ref = newRef;
            }, () => {});
        };

        $scope.computeAllLabels = function() {
            $scope.uiState.allLabels = [];
            if ($scope.$ctrl.compatibleReferences) {
                $scope.uiState.allLabels = [...new Set($scope.$ctrl.compatibleReferences.flatMap(me => [...Object.keys(me.labels)]))];
            }
            $scope.uiState.allLabels.sort((v1,v2) => {
                const domainV1 = getDomainLabel(v1);
                const domainV2 = getDomainLabel(v2);
                const cmp = domainV1.localeCompare(domainV2);
                if (domainV1 === domainV2) {
                    const subLabelV1 = getDomainSubLabel(v1);
                    const subLabelV2 = getDomainSubLabel(v2);
                    return subLabelV1.localeCompare(subLabelV2);
                }
                return cmp;
            })

            if($scope.uiState.selectedLabels.length == 0) {
                // No label is selected (likely, initial state) => set defaults if they exist
                $scope.uiState.selectedLabels = DEFAULT_SELECTED_LABELS.filter(label=> $scope.uiState.allLabels.includes(label));
            }
        }

        $scope.computeTitleLabels = function() {
            $scope.uiState.titleLabels = generateTitleLabelsFromRefs($scope.uiState.selectedLabels);
            $scope.uiState.domainLabels = $scope.uiState.titleLabels.filter(x => x.domain);
        }

        $scope.$watch("$ctrl.compatibleReferences", $scope.computeAllLabels);
        $scope.$watch("uiState.selectedLabels", $scope.computeTitleLabels);

        $scope.copyDriftParamsToClipboard = function() {
            ClipboardUtils.copyToClipboard(JSON.stringify($scope.$ctrl.driftParams, null, 2));
        };

        $scope.getLabelStyle = function(label){
            if ($scope.$ctrl.ref) {
                const areRefAndCurrentSame = $scope.$ctrl.cur.labels.get(label) == $scope.$ctrl.ref.labels[label];
                if(!areRefAndCurrentSame){
                    return " background-color : #F6FBFF; font-weight : bold; ";
                }
            }
            return "font-weight : normal;";
        }

        let dismissComputeParamsPopin = null;
        $scope.toggleComputationParamsPopin = function($event) {
            if (!dismissComputeParamsPopin) {
                function isElsewhere(elt, e) {
                    return $(e.target).parents(".dropdown-menu").length == 0;
                }
                const dkuPopinOptions = {
                    template: `
                        <ul class="dropdown-menu" style="padding: 15px;" listen-keydown="{'enter': 'save()', 'esc': 'dismiss()' }">
                            <form class="dkuform-horizontal">
                                <div class="control-group">
                                    <label for="" class="control-label">Confidence level</label>
                                    <div class="controls">
                                        <input type="number" required ng-model="$ctrl.driftParams.confidenceLevel" min="0.5" max="0.999"
                                            step="0.001" />
                                        <div class="help-inline">
                                            Used to compute confidence interval and determine significance level of statistical tests
                                        </div>
                                    </div>
                                </div>

                                <div class="control-group">
                                    <label for="" class="control-label">PSI threshold</label>
                                    <div class="controls">
                                        <input type="number" required ng-model="$ctrl.driftParams.psiThreshold" min="0" max="1"
                                            step="0.001" />
                                        <div class="help-inline">Using a fixed random seed allows for reproducible result</div>
                                    </div>
                                </div>

                                <button type="button" class="pull-right btn btn--secondary" ng-click="copyDriftParamsToClipboard()">
                                    <i class="icon-copy interactive-scoring__edit-icon"></i>
                                    Copy settings to clipboard
                                </button>
                            </form>
                        </ul>
                    `,
                    isElsewhere,
                    popinPosition: 'SMART',
                    onDismiss: () => {
                        dismissComputeParamsPopin = null;
                    }
                };
                dismissComputeParamsPopin = openDkuPopin($scope, $event, dkuPopinOptions);
            } else {
                dismissComputeParamsPopin();
                dismissComputeParamsPopin = null;
            }
        };
    }
});

app.component('perfComparator', {
    bindings: {
        refs: '<',
        predictionType: '<',
        labels: '<',
        storeList: '<',
        refreshStatus: '<',
        excludedLabels: '<',
        withinEvaluationStore: '<',
        displayParams: '=',
        displayDiagnostics: '<',
        possibleMetrics: '<',
        metricParamsList: '<',
        manualGraphType: '<',
        colors: '<',
        disableSelection: '<',
        editSelection: '<',
        pinnedMetricsAffectOtherTables: '<',
        onDelete: '&?',
        onAdd: '&?',
        onAddLabel: '<',
        clearChampion: '<?',
        onSetChampion: '&?',
        onReverseItemDisability: '&?',
        whiteBackground: '<',
        trackFrom: '<?',
        sortColumn: '=',
        sortDescending: '=',
        displayDriftMetrics: '<',
        isLlm: '<',
    },
    templateUrl: '/templates/modelevaluationstores/perfcomparator-component.html',
    controller:
        function ctrlPerfDrift($scope, $element, $filter, PMLFilteringService, $state, $stateParams, ChartIconUtils, StateUtils,
                ModelComparisonHelpers, CreateModalFromComponent, createOrAppendMELikeToModelComparisonModalDirective,
                MLDiagnosticsService, NumberFormatter, CustomMetricIDService, MetricsUtils, CreateModalFromTemplate) {
            $scope.$state = $state;
            $scope.$stateParams = $stateParams;
            $scope.$element = $element;
            $scope.metricMap = PMLFilteringService.metricMap;
            let ctrl = this;
            $scope.ctrl = ctrl;

            $scope.diagnosticsService = MLDiagnosticsService;
            ctrl.championSymbol = ModelComparisonHelpers.getChampionSymbol()

            $scope.uiState = {
                query:  null,
                titleLabels: [],
                domainLabels: [],
                focusedLabels: [],
                selectedCount: 0,
                displayedMetrics: [],
                evaluationsColormap: new Map(),
                linesColormap: null
            };

            $scope.columnFoldingState = new Map();

            $scope.foldOrUnfold = function(columnLabel) {
                if($scope.columnFoldingState[columnLabel] == undefined || $scope.columnFoldingState[columnLabel] == false){
                    $scope.columnFoldingState[columnLabel] = true;
                } else {
                    $scope.columnFoldingState[columnLabel] = false;
                }
            }

            $scope.getCellVerticalMergeSize = function() {
                return ctrl.refs.length + 2;
            }

            $scope.hasMetrics = function(metricType) {
                if (metricType === 'all') {
                    return $scope.hasMetrics('PERFORMANCE') || $scope.hasMetrics('DATA_DRIFT') || $scope.hasMetrics('PREDICTION_DRIFT');
                }
                return $scope.selection.filteredObjects.some(x =>
                    x.metrics
                    && Object.keys(x.metrics).length
                    && Object.keys(x.metrics).some(metric => {
                        if (metric === "customMetricsResults"  && x.metrics[metric].length > 0) {
                            return metricType === "PERFORMANCE";
                        }
                        if (!$scope.isMetricSelected(metric)) {
                            return false;
                        }
                        if (metricType === "PERFORMANCE"){
                            return !PMLFilteringService.isDriftMetric(metric);
                        } else if (metricType === "DATA_DRIFT") {
                            return PMLFilteringService.isDataDriftMetric(metric);
                        } else if (metricType === "PREDICTION_DRIFT") {
                            return PMLFilteringService.isPredictionDriftMetric(metric);
                        } else {
                            return false;
                        }
                }));
            }

            $scope.isMetricSelected = function(metric) {
                return $scope.uiState.currentOrderedFormattedNames.some(metricSelected => {
                    if (PMLFilteringService.univariateMetrics.includes(metric) && PMLFilteringService.isUnivariateMetric(metricSelected.key)){
                        return true;
                    }
                    if (PMLFilteringService.embeddingMetrics.includes(metric) && PMLFilteringService.isEmbeddingMetric(metricSelected.key)){
                        return true;
                    }
                    return PMLFilteringService.metricMap[metricSelected.key] === metric;
                });
            }

            $scope.isColumnFolded = function(columnLabel) {
                return $scope.columnFoldingState[columnLabel] == true;
            }

            $scope.getColumnClass = function(columnLabel) {
                if($scope.isColumnFolded(columnLabel)){
                    return 'model-comparator-table-folded-th';
                }
            }

            $scope.getTextClass = function(columnLabel) {
                if($scope.isColumnFolded(columnLabel)){
                    return 'vertical-text';
                }
                return 'horizontal-text'
            }

            $scope.isHighlighted = function(item) {
                return ($scope.uiState.hoverId == item.$id) && !item.isDisabled
            }

            $scope.getModelEvaluationColSpan = function() {
                if($scope.isColumnFolded('model evaluation')){
                    return 1;
                }
                return $scope.enabledColumnsCount;
            }

            $scope.getColSpan = function(columnDefinition) {
                if($scope.isColumnFolded(columnDefinition.domain)){
                    return 1;
                }
                return columnDefinition.span;
            }

            $scope.getSubLabels = function(domain) {
                if (domain === "-"){
                    return $scope.uiState.titleLabels.filter(x => x.domain == domain);
                } else {
                    return $scope.uiState.titleLabels.filter(x => x.parent == domain)
                }
            }

            $scope.getSubLabelDisplayValue = function(item, subLabel) {
                let displayLabel = item.labels.get(subLabel.fullLabel);

                if (!displayLabel) {
                    return "-";
                }

                if (subLabel.fullLabel === "model:algorithm") {
                    displayLabel = $filter('niceConst')(displayLabel);
                }
                if (subLabel.fullLabel === "model:learning-method") {
                    // Rendering META_LEARNER as "Meta-learner" instead of "Meta learner" for consistency with templates/ml/prediction-model/algorithm.html
                    displayLabel = displayLabel === "META_LEARNER" ? $filter('niceConst')(displayLabel, "-") : $filter('niceConst')(displayLabel);
                }

                if (displayLabel.length > 500) {
                    displayLabel = displayLabel.substr(0, 500) + ' (and more)';
                }
                return displayLabel;
            }

            $scope.getAfterSortingStyle = function(column) {
                if (column != ctrl.sortColumn){
                    return "no-sort";
                }
                else if(ctrl.sortDescending) {
                    return "sort-descending"
                }
                return "sort-ascending"
            }

            $scope.getMinWidth = function(domainDetails) {
                if($scope.isColumnFolded(domainDetails.domain)){
                    return;
                }
                if(domainDetails.domain == 'evaluationDataset'){
                    return {'min-width': '180px'};
                }
                const minWidth = $scope.getSubLabels(domainDetails.domain).length * 160;
                return {'min-width': minWidth+'px'};
            }


            $scope.getFontTextSize = function(columnLabel) {
                if($scope.isColumnFolded(columnLabel) &&
                    ctrl.refs.length == 1 &&
                    columnLabel == 'evaluationDataset'){
                    return {'font-size': '14px'};
                }
            }

            $scope.hoveredRow = undefined;

            $scope.isRowOnHover = function(id) {
                return $scope.hoveredRow == id;
            }

            $scope.mouseOverRow = function(id) {
                $scope.hoveredRow = id;
            }

            $scope.mouseLeavesRow = function(id) {
                if($scope.isRowOnHover(id)){
                    $scope.hoveredRow = undefined;
                }
            }

            $scope.saveNewDisplayedMetrics = (newMetrics) => ctrl.displayParams.displayedMetrics = newMetrics;

            $scope.getNbSelectedMetrics = () => ctrl.displayParams.displayedMetrics.length + "/" + $scope.uiState.possibleMetrics.length;

            $scope.openSelectionMetricsModal = function () {
                CreateModalFromTemplate("/templates/modelevaluationstores/select-displayed-metrics-modal.html", $scope, null, function(modalScope) {
                    modalScope.isLLM = $scope.ctrl.isLlm;

                    modalScope.MetricsUtils = MetricsUtils;
                    modalScope.PMLFilteringService = PMLFilteringService;

                    // Left side
                    modalScope.getPossibleMetrics = () => modalScope.filterMetricsOnQueryAndType(modalScope.possibleMetrics, modalScope.uiState.activeTab, modalScope.filter.queryLeft);

                    modalScope.add = function(metric) {
                        modalScope.possibleMetrics.splice(modalScope.possibleMetrics.indexOf(metric), 1);
                        modalScope.displayedMetrics.push(metric);
                    }

                    modalScope.addAll = function() {
                        modalScope.displayedMetrics = modalScope.displayedMetrics.concat(modalScope.getPossibleMetrics());
                        modalScope.getPossibleMetrics().forEach(metric => modalScope.possibleMetrics.splice(modalScope.possibleMetrics.indexOf(metric), 1));
                    }

                    // Right side
                    modalScope.getDisplayedMetrics = (metricType) => modalScope.filterMetricsOnQueryAndType(modalScope.displayedMetrics, metricType, modalScope.filter.queryRight);

                    modalScope.remove = function(metric) {
                        modalScope.displayedMetrics.splice(modalScope.displayedMetrics.indexOf(metric), 1);
                        modalScope.possibleMetrics.push(metric);
                    }

                    modalScope.removeAll = function() {
                        modalScope.possibleMetrics = modalScope.possibleMetrics.concat(modalScope.getDisplayedMetrics());
                        modalScope.getDisplayedMetrics().forEach(metric => modalScope.displayedMetrics.splice(modalScope.displayedMetrics.indexOf(metric), 1));
                    }

                    // Helpers
                    modalScope.getDataDriftMetrics = (metrics) => metrics.filter(x => PMLFilteringService.isDataDriftMetric(x));
                    modalScope.getPredictionDriftMetrics = (metrics) => metrics.filter(x => PMLFilteringService.isPredictionDriftMetric(x));
                    modalScope.getPerformanceMetrics = (metrics) => metrics.filter(x => !PMLFilteringService.isDriftMetric(x));

                    modalScope.dataDriftMetricsSorter = function(x, y) {
                        //Since we can have potentially hundreds of univariate drift metrics, it is better UX friendly to
                        //always display at the top of the choices the 1/ global data drift 2/ min/max of univariates
                        const getMetricRank = (x) => {
                            if (["DATA_DRIFT", "DATA_DRIFT_PVALUE"].includes(x)) {
                                return 0;
                            } else if (["MIN_KS", "MIN_CHISQUARE", "MAX_PSI"].includes(x)){
                                return 1;
                            } else {
                                return 2;
                            }
                        }
                        const xMetricRank = getMetricRank(x);
                        const yMetricRank = getMetricRank(y);

                        if (xMetricRank === yMetricRank){
                            return x < y ? -1 : 1;
                        } else {
                            return xMetricRank < yMetricRank ? -1 : 1;
                        }
                    }

                    modalScope.filterMetricsOnQueryAndType = function(metrics, metricType, query){
                        switch(metricType) {
                            case 'data drift':
                                return $filter('filter')(modalScope.getDataDriftMetrics(metrics).sort(modalScope.dataDriftMetricsSorter), query);
                            case 'prediction drift':
                                return $filter('filter')(modalScope.getPredictionDriftMetrics(metrics), query);
                            case 'performance':
                                return $filter('filter')(modalScope.getPerformanceMetrics(metrics), query);
                            default:
                                return $filter('filter')(metrics, query);
                        }
                    }

                    // On modal opening
                    modalScope.allPossibleMetrics = $scope.uiState.possibleMetrics.map(x => x[0]);
                    modalScope.possibleMetrics = modalScope.allPossibleMetrics
                        .filter(x => !ctrl.displayParams.displayedMetrics.includes(x));
                    modalScope.displayedMetrics = angular.copy(ctrl.displayParams.displayedMetrics);

                    if (modalScope.getDataDriftMetrics(modalScope.possibleMetrics).length === 0) {
                        modalScope.uiState.activeTab =  "performance"; // For SER without ref
                    } else {
                        modalScope.uiState.activeTab = "data drift";
                    }
                    modalScope.filter = {queryLeft:'', queryRight:''};

                    // Actions
                    modalScope.save = function() {
                        $scope.saveNewDisplayedMetrics(modalScope.displayedMetrics);
                        modalScope.resolveModal()
                    }
                });
            }

            ctrl.$onChanges = function() {
                $scope.labels= ctrl.labels;
                $scope.updatePalette();
                $scope.refreshLabels();
                $scope.refreshTableData();
                $scope.refreshCurrentMetricNames();
                $scope.computeSelectedCount();
            }

            $scope.goToDataDriftTab = function(item) {
                $state.go('projects.project.modelevaluationstores.modelevaluationstore.evaluation.report.tabular-input_data_drift', {
                    projectKey: item.ref.projectKey,
                    mesId: item.ref.id,
                    evaluationId: item.evaluationId
                });
            }

            $scope.computeSelectedCount = function() {
                $scope.uiState.selectedCount = ctrl.refs.filter(r => !r.isDisabled).length;
            }

            const dkuColors = window.dkuColorPalettes.discrete[0].colors.filter((x,idx) => idx%2 === 0);
            $scope.updatePalette = function() {
                if (ctrl.colors) {
                    $scope.colors = ctrl.colors;
                } else {
                    $scope.colors = dkuColors;
                };
            }

            $scope.sortValue = function(sortColumn) {
                const metricsPattern = /^metrics\./;
                const labelsPattern = /^labels\./;
                const customMetricsPattern = /^metrics\.!!/;
                if (customMetricsPattern.exec(sortColumn)) {
                    return me => {
                        for (const customMetric of me.metrics.customMetricsResults) {
                            if (customMetric["metric"]["name"] === sortColumn.replace(customMetricsPattern, '')) {
                                return customMetric["value"]
                            }
                        }
                    }
                }
                if (metricsPattern.exec(sortColumn)) {
                    return me => me.metrics[$scope.metricMap[sortColumn.replace(metricsPattern, '')]];
                }
                if (labelsPattern.exec(sortColumn)) {
                    return me => me.labels.get(sortColumn.replace(labelsPattern, ''));
                }
                return me => me[sortColumn];
            };

            $scope.graphData = [];

            $scope.selection = $.extend({
                filterQuery: {
                    userQuery: ''
                },
                filterParams: {
                    userQueryTargets: ["refStr", "refSubtitle"],
                    propertyRules: {}
                },
                orderQuery: "-value",
                orderReversed: false
            }, $scope.selection || {});

            $scope.computeChartIcone = function(type, variant, webAppType) {
                return ChartIconUtils.computeChartIcon(type, variant, $scope.isInAnalysis, webAppType);
            }

            $scope.setGraphTypes = function() {
                if (!ctrl.displayParams || ctrl.manualGraphType && ctrl.displayParams.graphStyle || !$scope.getDisplayedObjects().length) {
                    return;
                }
                ctrl.displayParams.graphStyle =
                    ($scope.getDisplayedObjects() &&
                     $scope.getDisplayedObjects().filter(x => Object.keys(x.metrics).length > 0).length > 2)?'LINE':'BAR';
            }

            $scope.generateTitleLabels = function() {
                var labels = [];
                if (ctrl.displayParams.graphStyle === 'LINE') {
                    if (ctrl.displayParams.xLabel) {
                        labels.push(ctrl.displayParams.xLabel);
                    }
                    if (ctrl.displayParams.yLabels) {
                        labels = labels.concat(ctrl.displayParams.yLabels);
                    }
                    if (ctrl.displayParams.alsoDisplayedLabels) {
                        labels = labels.concat(ctrl.displayParams.alsoDisplayedLabels);
                    }
                } else {
                    if (ctrl.displayParams.barLabels) {
                        labels = labels.concat(ctrl.displayParams.barLabels);
                    }
                }
                labels.sort();
                $scope.uiState.titleLabels = generateTitleLabelsFromRefs(labels);
                $scope.uiState.domainLabels = $scope.uiState.titleLabels.filter(x => x.domain);
            }

            $scope.refreshLabels = function() {
                let labelKeys = _.sortBy(_.uniq(_.flatten(ctrl.refs.map(r => Array.from(r.labels.keys())))));
                if ($scope.ctrl.excludedLabels) {
                    labelKeys = labelKeys.filter(l => !$scope.ctrl.excludedLabels.includes(l));
                }
                $scope.possibleXLabels = labelKeys;
                if (ctrl.displayParams.xLabel && !$scope.possibleXLabels.includes(ctrl.displayParams.xLabel)) {
                    ctrl.displayParams.xLabel = undefined;
                }
                if (!ctrl.displayParams.xLabel && !$scope.ctrl.excludedLabels) {
                    ctrl.displayParams.xLabel = (labelKeys && labelKeys.length)?(labelKeys.includes(DEFAULT_X_LABEL)?DEFAULT_X_LABEL:labelKeys[0]):undefined;
                }
                $scope.possibleYLabels = $scope.possibleXLabels.slice().filter(x => x !== ctrl.displayParams.xLabel);
                $scope.possibleOtherLabels = $scope.possibleYLabels.slice().filter(x => ctrl.displayParams.yLabels?!ctrl.displayParams.yLabels.includes(x):true);
            }


            $scope.$watch("ctrl.displayParams.xLabel", function() {
                $scope.setGraphTypes();
                $scope.possibleYLabels = $scope.possibleXLabels.slice().filter(x => x !== ctrl.displayParams.xLabel);
                if (!ctrl.displayParams.yLabels || !ctrl.displayParams.yLabels.length) {
                    ctrl.displayParams.yLabels = [...DEFAULT_Y_LABELS];
                    if (ctrl.predictionType == "TIMESERIES_FORECAST") {
                        ctrl.displayParams.yLabels = [...DEFAULT_Y_LABELS, "evaluation:nb-evaluation-timesteps"]
                    }
                }
                if (ctrl.displayParams.yLabels && ctrl.displayParams.yLabels.length) {
                    ctrl.displayParams.yLabels = ctrl.displayParams.yLabels.filter(x => (x !== ctrl.displayParams.xLabel) && $scope.possibleYLabels.includes(x));
                }
                if (ctrl.displayParams.alsoDisplayedLabels && ctrl.displayParams.alsoDisplayedLabels.length) {
                    ctrl.displayParams.alsoDisplayedLabels = ctrl.displayParams.alsoDisplayedLabels.filter(x => x !== ctrl.displayParams.xLabel);
                }
                if (!ctrl.displayParams.barLabels) {
                    ctrl.displayParams.barLabels = [DEFAULT_X_LABEL, ...DEFAULT_Y_LABELS];
                    if (["CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION"].includes(ctrl.predictionType)) {
                        ctrl.displayParams.barLabels.push("model:meta-learner");
                    }
                    if (ctrl.predictionType == "TIMESERIES_FORECAST") {
                        ctrl.displayParams.barLabels.push("evaluation:nb-evaluation-timesteps")
                    }

                }
                if (ctrl.displayParams.barLabels && ctrl.displayParams.barLabels.length) {
                    ctrl.displayParams.barLabels = ctrl.displayParams.barLabels.filter(l => $scope.possibleXLabels.includes(l));
                }
            });

            $scope.$watch("ctrl.displayParams.yLabels", function() {
                $scope.possibleOtherLabels = $scope.possibleYLabels.slice().filter(x => !ctrl.displayParams.yLabels.includes(x));
                if (ctrl.displayParams.alsoDisplayedLabels && ctrl.displayParams.alsoDisplayedLabels.length) {
                    ctrl.displayParams.alsoDisplayedLabels = ctrl.displayParams.alsoDisplayedLabels.filter(x => !ctrl.displayParams.yLabels.includes(x));
                }
            });

            $scope.$watch("ctrl.displayParams.graphStyle", function(nv, ov) {
                if (nv === ov) return;
                if ('BAR' == nv) {
                    $scope.uiState._yLabels = ctrl.displayParams.yLabels;
                    ctrl.displayParams.yLabels = [];
                } else {
                    if (!ctrl.displayParams.yLabels || !ctrl.displayParams.yLabels.length) {
                        ctrl.displayParams.yLabels = $scope.uiState._yLabels || [];
                    }
                }
            });

            $scope.refreshTableData = function() {
                if (ctrl.refs && ctrl.labels) {
                    $scope.tableData = ctrl.refs.map((v,i) => {
                        v.$id = i+1;
                        v.$selected = !ctrl.disableSelection;
                        v.$hasDiagnostics = MLDiagnosticsService.hasDiagnostics(v);
                        return v;
                    });
                } else if (ctrl.refs && ctrl.refs.length > 0) {
                    $scope.tableData = ctrl.refs.map((v,i) => {
                        v.$id = i+1;
                        v.$selected = !ctrl.disableSelection;
                        v.$hasDiagnostics = MLDiagnosticsService.hasDiagnostics(v);
                        return v;
                    });
                } else {
                    $scope.tableData = null;
                }
                $scope.uiState.hasAnyDiagnostics = ctrl.refs.some(ref => ref.$hasDiagnostics) && ctrl.displayDiagnostics;

                $scope.enabledColumns = {
                    selection: !ctrl.disableSelection || ctrl.onDelete !== undefined,
                    display: true,
                    champion: ctrl.onSetChampion !== undefined,
                    line: ctrl.displayParams.graphStyle === 'LINE',
                    bar: ctrl.manualGraphType|| ctrl.displayParams.graphStyle === 'BAR',
                    name: true,
                    diagnostics: $scope.uiState.hasAnyDiagnostics,
                    modelOrigin: ctrl.manualGraphType !== undefined,
                };
                $scope.enabledColumnsCount = Object.values($scope.enabledColumns).reduce((a, nv) => a + nv, 0);
            }

            const ISO8861_RE = /\d{4}-[01]\d(-[0-3]\d(( |T)[0-2]\d(:[0-5]\d(:[0-5]\d(\.\d+([+-][0-2]\d:[0-5]\d|Z)?)?)?)?)?)?/;
            const X_VALUE_CACHE_ATTR = "_xValue";
            const DISPLAY_LABEL_CACHE_ATTR = "_displayLabel";
            const DISPLAY_LABELS_CACHE_ATTR = "_displayLabels";

            $scope.getDisplayedObjects = function() {
                if (ctrl.refs && ctrl.labels) {
                    return ctrl.refs;
                } else {
                    return !ctrl.disableSelection?$scope.selection.filteredSelectedObjects:$scope.selection.allObjects;
                }
            }

            $scope.chartMinimumAmplitude = 0.2;

            $scope.getLineSeriesYMinMaxValues = (series) => {
                const allSeriesYValues = series
                .flatMap(serie => serie // Merging all the series in one array
                    .data.map(d => d[1]) // We want only the Y values out of X,Y,Z
                    )

                return $scope.getMinMax(allSeriesYValues)
            }

            $scope.getBarSeriesYMinMaxValues = (series) => {
                if(series.length == 0){
                    return [0, 1];
                }
                const allSeriesYValues = series
                .flatMap(serie => serie.data)

                return $scope.getMinMax(allSeriesYValues)
            }

            $scope.getMinMax = (array) => {
                const min = _.min(array);
                const max = _.max(array);
                return [min, max]
            }

            $scope.getChartYMinMaxValues = (minSerieValue, maxSerieValue) => {
                const amplitude = maxSerieValue - minSerieValue;

                let graphMin = undefined;
                let graphMax = undefined;

                if(amplitude < $scope.chartMinimumAmplitude) {
                    if(minSerieValue > 0) {
                        graphMin = Math.max(0, minSerieValue - ($scope.chartMinimumAmplitude - amplitude)/2);
                    } else {
                        graphMin = minSerieValue - ($scope.chartMinimumAmplitude - amplitude)/2;
                    }
                    graphMax = graphMin + $scope.chartMinimumAmplitude;
                }
                return [graphMin, graphMax];
            }

            $scope.getYAxisFormatter = function(value, seriesYMinMax, chartMinimumAmplitude = 0) {
                const seriesYAmplitude = seriesYMinMax[1] - seriesYMinMax[0];
                if (!value) {
                    return "";
                }
                const formatter = NumberFormatter.getForRange(0, Math.max(seriesYAmplitude, chartMinimumAmplitude), 10);
                return formatter(value);
            }

            function getWarningMessageOnMetricComparison(currentMetric) {
                let warningMessage;

                if (currentMetric === 'NET_UPLIFT') {
                    const firstNetUpliftPoint = ctrl.metricParamsList[0].netUpliftPoint;
                    const sameNetUpliftPointsForAll = ctrl.metricParamsList.every(metricParams => metricParams.netUpliftPoint === firstNetUpliftPoint);
                    if (!sameNetUpliftPointsForAll) {
                        warningMessage = "The models have different net uplift points";
                    }
                }

                if (currentMetric === 'CUMULATIVE_LIFT') {
                    const firstLiftPoint = ctrl.metricParamsList[0].liftPoint;
                    const sameLiftPointsForAll = ctrl.metricParamsList.every(metricParams => metricParams.liftPoint === firstLiftPoint);
                    if (!sameLiftPointsForAll) {
                        warningMessage = "The models have different lift points";
                    }
                }

                if (currentMetric === 'COST_MATRIX') {
                    const firstCmgMatrix = ctrl.metricParamsList[0].costMatrixWeights;
                    const sameCmgMatrixForAll = ctrl.metricParamsList.every(function(metricParams) {
                        const currentCmgMatrix = metricParams.costMatrixWeights;
                        return currentCmgMatrix.tpGain === firstCmgMatrix.tpGain && currentCmgMatrix.tnGain === firstCmgMatrix.tnGain
                            && currentCmgMatrix.fpGain === firstCmgMatrix.fpGain && currentCmgMatrix.fnGain === firstCmgMatrix.fnGain;
                    });
                    if (!sameCmgMatrixForAll) {
                        warningMessage = "The models have different cost matrix gain weights";
                    }
                }
                return warningMessage;
            }

            $scope.computeChartOptionsLines = function() {
                $scope.xMinMax = [];
                $scope.barDisplayedObjects = null;
                return ctrl.displayParams.displayedMetrics.map((currentMetric) => {
                    const isCustom = CustomMetricIDService.checkMetricIsCustom(currentMetric);
                    const isWorst = CustomMetricIDService.checkMetricIsWorst(currentMetric);
                    const xAxisLabel = ctrl.displayParams.xLabel || "";
                    const yLabels = (ctrl.displayParams.yLabels && ctrl.displayParams.yLabels.length)?ctrl.displayParams.yLabels: [];
                    const valuesWithTag = $scope.getDisplayedObjects().filter(item => {
                        const hasLabel = item.labels.has(xAxisLabel)
                        return hasLabel && !item.isDisabled && Object.keys(item.metrics).length > 0;
                    }).map(item => {
                        let copy = _.cloneDeep(item);
                        copy[X_VALUE_CACHE_ATTR] = copy.labels.get(xAxisLabel);
                        return copy;
                    });
                    let isNumeric = valuesWithTag.reduce((acc,cur) => acc && !isNaN(cur[X_VALUE_CACHE_ATTR]), true);
                    let isTemporal = valuesWithTag.reduce((acc,cur) => acc && ISO8861_RE.test(cur[X_VALUE_CACHE_ATTR]), true);
                    const convertedValuesWithTag = valuesWithTag.map(v => {
                        if (!isTemporal && !isNumeric) return v;
                        let copy = _.cloneDeep(v);
                        if (isTemporal) {
                            copy[X_VALUE_CACHE_ATTR] = moment(copy[X_VALUE_CACHE_ATTR]).valueOf();
                        } else {
                            copy[X_VALUE_CACHE_ATTR] = parseFloat(copy[X_VALUE_CACHE_ATTR]);
                        }
                        return copy;
                    });
                    const xValues = convertedValuesWithTag.map(v => v[X_VALUE_CACHE_ATTR]);
                    if (isTemporal || isNumeric) {
                        let minValue = Math.min(...xValues);
                        let maxValue = Math.max(...xValues);
                        if (isTemporal) {
                            const diffValue = maxValue - minValue;
                            let format;
                            if (diffValue < 86400000) {
                                // les that 24h
                                format = "HH:mm:ss";
                            } else if (diffValue < 31536000000) {
                                // less than a year of 365 days
                                format = "MM-DD";
                            } else {
                                format = "YYYY-MM-DD";
                            }
                            $scope.xMinMax.push([
                                moment(minValue).format(format),
                                moment(maxValue).format(format)
                            ]);
                        } else {
                            $scope.xMinMax.push([minValue.toFixed(2), maxValue.toFixed(2)]);
                        }
                    }

                    const sortedValuesWithTag = _.orderBy(convertedValuesWithTag, item => item[X_VALUE_CACHE_ATTR], 'asc');

                    sortedValuesWithTag.forEach(item => {
                        let labelValues = [];
                        let curLabels = new Map();
                        for (let curYLabel of yLabels) {
                            const found = item.labels.get(curYLabel);
                            if (found && "" != found) {
                                labelValues.push(found);
                                curLabels.set(curYLabel, found);
                            }
                        }
                        let ret = labelValues.join("-");
                        if ("" === ret) {
                            ret = "(no labels)"
                        }
                        item[DISPLAY_LABEL_CACHE_ATTR] = ret;
                        item[DISPLAY_LABELS_CACHE_ATTR] = curLabels;
                    });
                    // generate unique label list
                    let labels = _.uniq(sortedValuesWithTag.map(item => item[DISPLAY_LABEL_CACHE_ATTR]));

                    // assign colors to the labels
                    let labelColors = new Map();
                    labels.forEach((label,index) => {
                        labelColors.set(label, dkuColors[index%dkuColors.length]);
                    });

                    // let's generate line segments
                    let currentLabel = null;
                    let series = [];
                    let currentSerie = null;
                    let mapSegments = new Map();
                    let mapSegmentsLabels = new Map();

                    let lineColors = new Map();
                    for (const currentItem of sortedValuesWithTag) {
                        if (currentLabel != currentItem[DISPLAY_LABEL_CACHE_ATTR]) {
                            currentLabel = currentItem[DISPLAY_LABEL_CACHE_ATTR];
                            currentSerie = mapSegments.get(currentLabel);
                            if (!currentSerie) {
                                currentSerie = {
                                    data: [],
                                    name: currentLabel,
                                    type: 'line',
                                    symbol: 'roundRect',
                                    symbolSize: 5,
                                    color: labelColors.get(currentLabel)
                                };
                                series.push(currentSerie);
                                mapSegments.set(currentLabel, currentSerie);
                                mapSegmentsLabels.set(currentLabel, currentItem[DISPLAY_LABELS_CACHE_ATTR]);
                            }
                        }

                        let currentItemValue;

                        if (isCustom && typeof currentItem.metrics.customMetricsResults !== 'undefined') {
                            const metricName = CustomMetricIDService.getCustomMetricBaseName(currentMetric);
                            const matchedCustomMetrics = currentItem.metrics.customMetricsResults.filter(customMetricResult => (customMetricResult.metric.name?customMetricResult.metric.name:customMetricResult.metric.metricCode) === metricName);
                            if (matchedCustomMetrics.length > 0) {
                                if (!isWorst) {
                                    currentItemValue = [currentItem[X_VALUE_CACHE_ATTR], matchedCustomMetrics[0].value, currentItem["$id"]];
                                }
                                else {
                                    currentItemValue = [currentItem[X_VALUE_CACHE_ATTR], matchedCustomMetrics[0].worstValue, currentItem["$id"]];
                                }
                            } else {
                                currentItemValue = [currentItem[X_VALUE_CACHE_ATTR], null, currentItem["$id"]];
                            }
                        } else if (PMLFilteringService.isUnivariateMetric(currentMetric)) {
                            currentItemValue = [currentItem[X_VALUE_CACHE_ATTR], PMLFilteringService.univariateMetric(currentMetric, currentItem.metrics), currentItem["$id"]];
                        } else if (PMLFilteringService.isEmbeddingMetric(currentMetric)) {
                            currentItemValue = [currentItem[X_VALUE_CACHE_ATTR], PMLFilteringService.embeddingMetric(currentMetric, currentItem.metrics), currentItem["$id"]];
                        } else {
                            currentItemValue = [currentItem[X_VALUE_CACHE_ATTR],currentItem.metrics[$scope.metricMap[currentMetric]],currentItem["$id"]];
                        }
                        if (currentItem.isChampion) {
                            currentSerie.data.push(ModelComparisonHelpers.getChampionDataElem(currentItemValue));  // NOSONAR
                        } else {
                            currentSerie.data.push(currentItemValue); // NOSONAR
                        }
                        lineColors.set(currentItem.$id, currentSerie.color);

                    }

                    // Find y amplitude (yMax - yMin) and force minimum amplitude of 0.2
                    const seriesYMinMax = $scope.getLineSeriesYMinMaxValues(series);

                    const minSerieValue = seriesYMinMax[0];
                    const maxSerieValue = seriesYMinMax[1]

                    const chartYMinMax = $scope.getChartYMinMaxValues(minSerieValue, maxSerieValue);

                    const chartYMin = chartYMinMax[0];
                    const chartYMax = chartYMinMax[1];

                    const evaluationColors = new Map();
                    $scope.getDisplayedObjects().forEach((v,i) => {
                        evaluationColors.set(v.$id, $scope.colors[i]);
                    });
                    $scope.evaluationsColormap(evaluationColors);
                    $scope.linesColormap(lineColors);
                    $scope.mapSegmentsLabels = mapSegmentsLabels;

                    const warningMessage = getWarningMessageOnMetricComparison(currentMetric);

                    return {
                        animation: false,
                        tooltip: {
                            trigger: 'item',
                            axisPointer: { type: 'none' },
                            formatter: (params) => {
                                $scope.uiState.hoverId = (params.data && params.data.length > 2)?params.data[2]:undefined;
                                $scope.$applyAsync();
                                const serieLabels = $scope.mapSegmentsLabels.get(params.seriesName);
                                const paramsData = ((params.data instanceof Array)?params.data:params.data.value);
                                const X = isTemporal?moment(paramsData[0]).format("YYYY-MM-DDTHH:mm:ss.SSSZ"):paramsData[0];
                                if (!serieLabels || !serieLabels.size) {
                                    return `${sanitize(params.seriesName)} - ${sanitize(X)}: ${sanitize(paramsData[1].toFixed(2))}`;
                                } else {
                                    let ret = "<table style='background: none;'><tbody>";
                                    for (const labelEntry of serieLabels) {
                                        const labelName = sanitize(labelEntry[0]);
                                        const labelValue = sanitize(labelEntry[1]);
                                        ret += `<tr><td>${labelName}</td><td>${labelValue}</td></tr>`;
                                    }
                                    ret += `<tr><td>X</td><td>${sanitize(X)}</td></tr>`;
                                    const baseValue = paramsData[1];
                                    let convertedValue = baseValue.toFixed(4);
                                    if (convertedValue === Number(0).toFixed(4)) {
                                        convertedValue = baseValue.toExponential(4);
                                    }
                                    const value = sanitize(convertedValue);
                                    ret += `<tr><td>Value</td><td>${value}</td></tr>`;
                                    ret += '</tbody></table>';
                                    return ret;
                                }
                            },
                            position: (point) => {
                                return [point[0] + 5, point[1] - 40];
                            }
                        },
                        textStyle: { fontFamily: 'SourceSansPro' },
                        xAxis: [{
                            type: isTemporal?'time':(isNumeric?'value':'category'),
                            axisLine: { show: true },
                            axisTick: { show: true },
                            axisLabel: ctrl.displayParams.xLabel || "",
                            scale: true
                        }],
                        yAxis: [{
                            type: 'value',
                            min: chartYMin,
                            max: chartYMax,
                            axisTick: { show: true },
                            axisLine: { show: true },
                            axisLabel: {
                                show: true,
                                width: 40,
                                formatter: function(value) {
                                    return sanitize($scope.getYAxisFormatter(value, seriesYMinMax, $scope.chartMinimumAmplitude));
                                }
                            },
                            scale: true,
                        }],
                        series,
                        grid: {
                            top: 10,
                            bottom: 20,
                            left: 45,
                            right: 3
                        },
                        metric: ctrl.possibleMetrics.find(x => x[0] === currentMetric),
                        warningMessage
                    };
                });
            }

            $scope.onChartInit = (echartsObject, chartOptions) => {
                const previousEchartsObject = chartOptions.echartsObject;
                chartOptions.echartsObject = echartsObject;
                if (echartsObject) {
                    // stop hover highlighting in table when no longer hoverving point
                    echartsObject.on('click', (event) => {
                        let selectedId;
                        if ($scope.barDisplayedObjects) {
                            selectedId = $scope.barDisplayedObjects[event.seriesIndex].$id;
                        } else if (event.data.length == 3) {
                            selectedId = event.data[2];
                        }
                        const elem  = document.getElementById(`${selectedId}`);
                        $scope.$element.parent().scrollTop(elem.offsetTop + window.innerHeight / 2);
                    });
                    if (previousEchartsObject) {
                        // In some edge cases, typically when adding new models and pinning a new optimization metric
                        // the echarts object of the pinned graph can be instantiated after the update of the options
                        // In this case, we update its chartsOptions entry, to be sure it will be updated
                        // with the right data
                        const metric =  chartOptions.metric[0];
                        const curIndex = $scope.latestChartsOptions.findIndex(lco => lco.metric[0] == metric);
                        $scope.chartsOptions[curIndex] = $scope.latestChartsOptions[curIndex];
                    }
                }
            }

            $scope.computeChartOptionsBars = function() {
                const allObjects = $scope.getDisplayedObjects().filter(x => Object.keys(x.metrics).length > 0);
                $scope.barDisplayedObjects = allObjects.filter(o => {
                    return !o.isDisabled;
                });
                const displayedColors = $scope.colors.filter((_,idx) => {
                    if (allObjects.length <= idx) {
                        return true;
                    } else {
                        return !allObjects[idx].isDisabled;
                    }
                })

                const newBarOptions = (ctrl.displayParams.displayedMetrics || []).map((currentMetric) => {
                    const series = $scope.barDisplayedObjects.map(
                        (v, idx) => {
                            let labelElem = {};

                            let data;
                            const isCustom = CustomMetricIDService.checkMetricIsCustom(currentMetric) && typeof v.metrics.customMetricsResults !== 'undefined';
                            const isWorst = CustomMetricIDService.checkMetricIsWorst(currentMetric) && typeof v.metrics.customMetricsResults !== 'undefined';
                            let metricName;
                            if (isCustom) {
                                metricName = CustomMetricIDService.getCustomMetricBaseName(currentMetric);
                                const foundValue = v.metrics.customMetricsResults.filter(customMetricResult => (customMetricResult.metric.name?customMetricResult.metric.name:customMetricResult.metric.metricCode) === metricName);
                                if (!isWorst) {
                                    data = foundValue[0] ? foundValue[0].value : null;
                                }
                                else {
                                    data = foundValue[0] ? foundValue[0].worstValue : null;
                                }
                            } else if (PMLFilteringService.isUnivariateMetric(currentMetric)) {
                                data = PMLFilteringService.univariateMetric(currentMetric, v.metrics);
                            } else if (PMLFilteringService.isEmbeddingMetric(currentMetric)) {
                                data = PMLFilteringService.embeddingMetric(currentMetric, v.metrics);
                            } else {
                                data = v.metrics[$scope.metricMap[currentMetric]];
                            }
                            if (v.isChampion) {
                                const iconWidth = (($scope.isPinned(currentMetric)?400:200) - 50) / (2*$scope.barDisplayedObjects.length);
                                labelElem = {
                                    label: {
                                        ...ModelComparisonHelpers.getChampionBarElem(),
                                        fontSize: iconWidth
                                    }
                                };

                                let minRatio = 0;
                                let onlyZeroValues = false;
                                if (typeof data !== 'undefined') {
                                    onlyZeroValues = data === 0;
                                    const ratios = $scope.barDisplayedObjects.filter(elem => !elem.isChampion).map((challenger) => {
                                        let challengerData;
                                        if (isCustom && typeof metricName !== 'undefined' && challenger.metrics.customMetricsResults) {
                                            const challengerMetric = challenger.metrics.customMetricsResults.find((elem => elem.metric.name === metricName));
                                            if (challengerMetric) {
                                                challengerData = challengerMetric.value;
                                            }
                                        }
                                        else {
                                            challengerData = challenger.metrics[$scope.metricMap[currentMetric]];
                                        }
                                        if (typeof challengerData !== 'undefined') {
                                            if (challengerData !== 0) {
                                                onlyZeroValues = false;
                                            }
                                            return (challengerData > 0) ? data / challengerData : 1;
                                        }
                                    }).filter(elem => elem !== undefined);
                                    minRatio = ratios.length ? Math.min(...ratios) : data === 0 ? 0 : 1;
                                }

                                if (minRatio > 0.20 && !onlyZeroValues) {
                                    const approxHeight = $scope.isPinned(currentMetric) ? minRatio * 100 + 10 : minRatio * 50;
                                    labelElem.label.fontSize = Math.min(approxHeight, iconWidth);
                                }
                                else {
                                    labelElem.label.position = 'top';
                                }
                            }

                            return {
                                name: $scope.labels?$scope.labels[idx]:v.graphStr||v.refStr,
                                type: 'bar',
                                data: [data],
                                ...labelElem,
                                color: displayedColors[idx]
                            };
                    });

                    const seriesYMinMax = $scope.getBarSeriesYMinMaxValues(series);
                    const warningMessage = getWarningMessageOnMetricComparison(currentMetric);

                    return {
                        animation: false,
                        tooltip: {
                            trigger: 'item',
                            confine: true,
                            axisPointer: { type: 'none' },
                            formatter: (params) => {
                                $scope.uiState.hoverId = $scope.barDisplayedObjects[params.seriesIndex].$id;
                                $scope.$applyAsync();
                                let value = params.value.toFixed(4);
                                if (value === Number(0).toFixed(4)) {
                                    value = params.value.toExponential(4);
                                }
                                return `${sanitize(params.seriesName)}: ${sanitize(value)}`;
                            }
                        },
                        textStyle: { fontFamily: 'SourceSansPro' },
                        xAxis: {
                            type: 'category',
                            axisLabel: {
                                show: false
                            },
                            axisLine: { show: true },
                            axisTick: { show: true }
                        },
                        yAxis: {
                            type: 'value',
                            axisTick: { show: true },
                            axisLine: { show: true },
                            axisLabel: {
                                show: true,
                                width: 40,
                                formatter: function(value) {
                                    return sanitize($scope.getYAxisFormatter(value, seriesYMinMax, $scope.chartMinimumAmplitude));
                                }
                            },
                        },
                        series,
                        grid: {
                            top: 10,
                            bottom: 20,
                            left: 45,
                            right: 0
                        },
                        metric: ctrl.possibleMetrics.find(x => x[0] === currentMetric),
                        warningMessage
                   };
                });
                const evaluationColors = new Map();
                allObjects.forEach((v,i) => {
                    evaluationColors.set(v.$id, $scope.colors[i]);
                });
                $scope.linesColormap(null);
                $scope.evaluationsColormap(evaluationColors);
                return newBarOptions;
            }

            $scope.refreshEchartsData = function() {
                let newOptions = [];
                if (ctrl.refs && ctrl.labels) {
                    newOptions = $scope.computeChartOptionsBars();
                } else if ($scope.getDisplayedObjects() && $scope.getDisplayedObjects().length) {
                    if (!$scope.ctrl.displayParams) return;
                    if ('LINE' === $scope.ctrl.displayParams.graphStyle) {
                        newOptions = $scope.computeChartOptionsLines();
                    } else {
                        // 'BAR'
                        newOptions = $scope.computeChartOptionsBars();
                    }
                } else {
                    $scope.chartsOptions = [];
                }
                if ($scope.chartsOptions && $scope.chartsOptions.length && $scope.chartsOptions.length == newOptions.length) {
                    $scope.latestChartsOptions = newOptions;
                    $scope.chartsOptions.forEach((co,idx) => {
                        if (co.echartsObject) {
                            const echartsObject = co.echartsObject;
                            echartsObject.setOption(newOptions[idx], {
                                notMerge: true
                            });
                            co.warningMessage = newOptions[idx].warningMessage;
                        } else {
                            $scope.chartsOptions[idx] = newOptions[idx];
                        }
                    });
                } else {
                    $scope.chartsOptions = newOptions;
                    $scope.latestChartsOptions = undefined;
                }
            }

            $scope.refreshCurrentMetricNames = function() {
                if (ctrl.displayParams.displayedMetrics && ctrl.possibleMetrics && ctrl.possibleMetrics.length) {
                    $scope.uiState.currentFormattedNames = ctrl.displayParams.displayedMetrics
                        .filter(dm => dm).map(cur => {
                        const isCustom = CustomMetricIDService.checkMetricIsCustom(cur);
                        const isWorst = CustomMetricIDService.checkMetricIsWorst(cur);
                        return {
                            isCustom,
                            isWorst,
                            key: cur,
                            label: !isCustom ? ctrl.possibleMetrics.find(x => x[0] === cur)[1] : CustomMetricIDService.getCustomMetricName(cur),
                            isPinned: $scope.isPinned(cur)
                        };
                    });
                    // Sort with pinned metrics first
                    $scope.uiState.currentOrderedFormattedNames = [...$scope.uiState.currentFormattedNames].sort(
                        (a, b) => (a.isPinned === b.isPinned) ? 0 : (a.isPinned ? -1 : 1));
                } else {
                    $scope.uiState.currentFormattedNames = [];
                    $scope.uiState.currentOrderedFormattedNames = [];
                }
                $scope.refreshMetricsValues();
            }

            $scope.refreshMetricsValues = function() {
                let refs = $scope.ctrl.refs;
                if (refs) {
                    for (let item of refs) {
                        item.$formattedMetrics = {};
                        for (let metric of ctrl.possibleMetrics) {
                            item.$formattedMetrics[metric[0]] = MetricsUtils.getMetricValue(item.metrics, metric[0]);
                        }
                    }
                }
            }

            $scope.updatePossibleMetrics = function() {
                if (!ctrl.possibleMetrics || !ctrl.possibleMetrics.length) {
                    $scope.uiState.possibleMetrics = [];
                    return;
                }
                const displayedObjects = $scope.getDisplayedObjects();
                if (!displayedObjects || !displayedObjects.length)  {
                    $scope.uiState.possibleMetrics = [];
                    return;
                }

                $scope.uiState.possibleMetrics = ctrl.possibleMetrics.filter(
                    dm => ctrl.refs.map(o => o.$formattedMetrics[dm[0]]).some(v => !!v && v !== "-")
                );

                if (ctrl.displayParams.displayedMetrics) {
                    const possibleMetricCodes = $scope.uiState.possibleMetrics.map(pm => pm[0]);
                    const newDisplayedMetrics = ctrl.displayParams.displayedMetrics.filter(dm => possibleMetricCodes.includes(dm));
                    if(!angular.equals(ctrl.displayParams.displayedMetrics, newDisplayedMetrics)) {
                        ctrl.displayParams.displayedMetrics = newDisplayedMetrics;
                    }
                }
            }
            $scope.$watch("ctrl.displayParams.graphStyle", () => {
                $scope.refreshTableData();
                $scope.chartsOptions = [];
                $scope.refreshEchartsData();
                $scope.generateTitleLabels();
            });

            $scope.$watch("ctrl.displayParams.displayedMetrics", () => {
                $scope.refreshTableData();
                $scope.refreshEchartsData();
                $scope.refreshCurrentMetricNames();
            });

            $scope.$watch("ctrl.possibleMetrics", $scope.updatePossibleMetrics);

            $scope.$watch("getDisplayedObjects()", () => {
                $scope.refreshMetricsValues();
                $scope.updatePossibleMetrics();
                $scope.setGraphTypes();
                $scope.refreshEchartsData();
            });

            $scope.$watch("ctrl.displayParams.xLabel", () => {
                $scope.refreshEchartsData();
                $scope.generateTitleLabels();
            });

            $scope.$watch("ctrl.displayParams.yLabels", () => {
                $scope.refreshEchartsData();
                $scope.generateTitleLabels();
            });

            $scope.$watch("ctrl.displayParams.barLabels", $scope.generateTitleLabels);
            $scope.$watch("ctrl.displayParams.alsoDisplayedLabels", $scope.generateTitleLabels);


            $scope.uiState.hoverId = null;

            $scope.evaluationsColormap = function(obj) {
                $scope.uiState.evaluationsColormap = obj;
            }

            $scope.linesColormap = function(obj) {
                $scope.uiState.linesColormap = obj;
            }

            $scope.styleEvaluationColor = function(item) {
                let color = "#FFFFFF";
                if ($scope.uiState.evaluationsColormap) {
                    color = $scope.uiState.evaluationsColormap.get(item.$id);
                }
                return { "background-color": color };
            }

            $scope.styleLineColor = function(item) {
                let color = "#FFFFFF";
                if ($scope.uiState.linesColormap) {
                    color = $scope.uiState.linesColormap.get(item.$id);
                }
                return color;
            }

            $scope.getStoreName = function(storeId) {
                if (!storeId) {
                    return null;
                }
                if (ctrl.storeList && ctrl.storeList.length) {
                    let store = ctrl.storeList.find(s => s.id === storeId);
                    if (store) {
                        return store.name;
                    }
                }
                return storeId;
            }

            $scope.addFocusedGraphLabel = function(label) {
                if (ctrl.displayParams.pinnedMetrics.includes(label)) return;
                // kludge, as echartsInstance.setOption does not take in acccount the
                // change of size of the champion symbol => we force full redraw
                $scope.chartsOptions = [];
                ctrl.displayParams.pinnedMetrics.push(label);
                $scope.refreshCurrentMetricNames();
                $scope.refreshEchartsData();
            }

            $scope.removeFocusedGraphLabel = function(label) {
                // kludge, as echartsInstance.setOption does not take in acccount the
                // change of size of the champion symbol => we force full redraw
                $scope.chartsOptions = [];
                ctrl.displayParams.pinnedMetrics = ctrl.displayParams.pinnedMetrics.filter(e => e !== label);
                $scope.refreshCurrentMetricNames();
                $scope.refreshEchartsData();
            }

            $scope.isPinned = function(key) {
                return ctrl.displayParams.pinnedMetrics.includes(key);
            }

            $scope.shouldDisplayMetric = function(metric){
                return !(!ctrl.displayDriftMetrics && PMLFilteringService.isDriftMetric(metric));
            }

            $scope.firstMetricOfDomain = function(key) {
                // true for first pinned, and first not pinned metric
                if ($scope.uiState.currentOrderedFormattedNames[0].key == key) return true;
                let firstNotPinned = $scope.uiState.currentOrderedFormattedNames.find(k => !k.isPinned)
                if (firstNotPinned && firstNotPinned.key == key) return true;
                return false;
            }

            $scope.goToItem = function(item) {
                return StateUtils.href.fullModelLikeId(item.ref.fullId, 'PREDICTION', ctrl.isLlm);
            }

            $scope.showItems = function(items) {
                for (const item of items) {
                    if (item.isDisabled) {
                        ctrl.onReverseItemDisability({toProcess: [item]});
                    }
                    item.isDisabled = false;
                }
                $scope.refreshEchartsData();
            }

            $scope.hideItems = function(items) {
                for (const item of items) {
                    if (!item.isDisabled) {
                        ctrl.onReverseItemDisability({toProcess: [item]});
                    }
                    item.isDisabled = true;
                }
                $scope.refreshEchartsData();
            }

            $scope.reverseItemDisability = function(item) {
                item.isDisabled = !item.isDisabled;
                $scope.refreshEchartsData();
                if (ctrl.onReverseItemDisability) {
                    ctrl.onReverseItemDisability({toProcess: [item]});
                }
            }

            $scope.clearChampion = function(item) {
                //cheating a bit to make it for reactive
                $scope.getDisplayedObjects().forEach(cm => {
                    cm.isChampion = false;
                });
                $scope.refreshEchartsData();
                setTimeout(() => ctrl.clearChampion(), 0);
            }

            $scope.onSetChampion = function(item) {
                $scope.getDisplayedObjects().forEach(cm => {
                    cm.isChampion = item.ref.fullId == cm.ref.fullId;
                });
                ctrl.onSetChampion({toSet: item});
                $scope.refreshEchartsData();
            }

            $scope.comparisonForbiddenReason = function() {
                // todo @deephub: block deephub models from being compared after deephub evaluation recipe is fixed
                if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length < 1) {
                    return "At least one evaluation must be selected";
                }
                if($scope.selection.selectedObjects.some(model => !model.predictionType && !model.llmTaskType)) {
                    return "Evaluations can not be compared";
                }
                if(!$scope.selection.selectedObjects.every(model => model.predictionType == $scope.selection.selectedObjects[0].predictionType)) {
                    return "Compared evaluations must have the same prediction type";
                }
                if($scope.selection.selectedObjects.some(model => model.isPartitioned)) {
                    return "Partitioned models cannot be compared";
                }
                if($scope.selection.selectedObjects.some(model => model.isEnsembled)) {
                    return "Ensembled models cannot be compared";
                }
            }

            $scope.$watch("selection.selectedObjects", () => $scope.$emit('refreshComparisonForbiddenReason', $scope.comparisonForbiddenReason()));

            $scope.compareSelectedEvaluations = function() {
                const nbModels = $scope.selection.selectedObjects.length;
                CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
                    fullIds: $scope.selection.selectedObjects.map(me => me.ref.fullId),
                    modelTaskType: ctrl.isLlm ? "LLM" : $scope.selection.selectedObjects[0].predictionType,
                    allowImportOfRelatedVersions: ctrl.withinEvaluationStore,
                    suggestedMCName: `Compare ${nbModels} evaluations`,
                    projectKey: $stateParams.projectKey,
                    trackFrom: ctrl.trackFrom || 'perf-comparator',
                    areAllModelsExternal: $scope.selection.selectedObjects.every(me => me.modelType === 'EXTERNAL')
                });
            }
            $scope.$on('triggerCreateModelComparison', function(event) {
                $scope.compareSelectedEvaluations();
            });

            $scope.getOriginStyle = function(item) {
                return ModelComparisonHelpers.getOriginStyle(item);
            }
            $scope.getOriginTooltip = function(item) {
                return ModelComparisonHelpers.getOriginTooltip(item);
            }

            $scope.toggleItem = function(item) {
                item.isSelected = !item.isSelected;
                $scope.uiState.refs = [];
            }
        }
});

app.filter('metricsFilter', function(PMLFilteringService) {
    return function(metrics, metricType) {
        let filtered = [];
        if (metricType === "PERFORMANCE") {
            angular.forEach(metrics, function(item) {
                if (!PMLFilteringService.isDriftMetric(item.key) && !PMLFilteringService.isUnivariateMetric(item.key)) {
                    filtered.push(item);
                }
            });
        }
        if (metricType === "DATA_DRIFT") {
            angular.forEach(metrics, function(item) {
                if (PMLFilteringService.isDataDriftMetric(item.key)) {
                    filtered.push(item);
                }
            });
        }
        if (metricType === "PREDICTION_DRIFT") {
            angular.forEach(metrics, function(item) {
                if (PMLFilteringService.isPredictionDriftMetric(item.key)) {
                    filtered.push(item);
                }
            });
        }
        return filtered;
    }
});

app.controller("ModelEvaluationStoreLLMEvaluationController", function($scope, DataikuAPI, $stateParams, TopNav,
    ActiveProjectKey, FullModelLikeIdUtils, MLDiagnosticsService,  MetricsUtils, Debounce, ActivityIndicator, CustomMetricIDService, LLMEvaluationMetrics,
    CachedAPICalls){

    $scope.refresh = function(data) {
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        $scope.mesContext.evaluationFullInfo = data.evaluationDetails;

        $scope.evaluationDetails = data.evaluationDetails;
        $scope.modelEvaluation = data.evaluationDetails.evaluation;
        $scope.evaluationDiagnostics = MLDiagnosticsService.groupByType($scope.evaluationDetails.mlDiagnostics);
        $scope.evaluationDiagnosticsCount = MLDiagnosticsService.countDiagnostics($scope.evaluationDetails);
        $scope.isLLM = true;
        $scope.embeddingLLMFriendlyName = $scope.modelEvaluation.embeddingLLMFriendlyName ? $scope.modelEvaluation.embeddingLLMFriendlyName : $scope.modelEvaluation.embeddingLLMId;
        $scope.completionLLMFriendlyName = $scope.modelEvaluation.completionLLMFriendlyName ? $scope.modelEvaluation.completionLLMFriendlyName : $scope.modelEvaluation.completionLLMId;

        CachedAPICalls.mlCommonDiagnosticsDefinition.then(mlCommonDiagnosticsDefinition => {
            $scope.diagnosticsDefinition = mlCommonDiagnosticsDefinition;
        });

        $scope.refreshMetrics($scope.modelEvaluation.predictionType, LLMEvaluationMetrics.getCustomMetricNamesFromEvaluationDetails($scope.evaluationDetails));

        TopNav.setItem(TopNav.ITEM_MODEL_EVALUATION_STORE, mesId, {name: data.modelEvaluationStore.name, evaluationId: evaluationId, runName: $scope.evaluationDetails.evaluation.userMeta.name});

        if (!$scope.readOnly) {
            const simpleApiResult = function(msg, r) {
                    r.success(function(){ ActivityIndicator.success(msg) })
                        .error(setErrorInScope.bind($scope));
            };

            const saveEvaluationUserMeta = function() {
                let fme = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);
                simpleApiResult("Saved", DataikuAPI.modelevaluations.saveEvaluationUserMeta(
                    fme, $scope.evaluationDetails.evaluation.userMeta));
            }
            const debouncedSaveEvaluationUserMeta = Debounce().withDelay(400, 1000).wrap(saveEvaluationUserMeta);
            $scope.$watch("evaluationDetails.evaluation.userMeta", function(nv, ov){
                if (nv && ov && nv != ov) {
                    debouncedSaveEvaluationUserMeta();
                }
            }, true)
        }

        $scope.isPartitionedModel = function() { return false; };

        $scope.fullModelEvaluationId = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);

        $scope.$emit('refreshComparisonForbiddenReason', $scope.comparisonForbiddenReason());
    }

    // Overriden in order to take into account custom metrics
    $scope.refreshCurrentMetricNames = function() {
        var topObject = $scope.modelEvaluationStore || $scope.modelComparison;
        if (topObject && topObject.displayParams.displayedMetrics && $scope.possibleMetrics) {
            $scope.uiState.currentFormattedNames = topObject.displayParams.displayedMetrics
                .filter(dm => dm).map(cur => {
                    const isCustom = CustomMetricIDService.checkMetricIsCustom(cur);
                    return {
                        isCustom,
                        key: cur,
                        label: !isCustom ? $scope.possibleMetrics.find(x => x[0] === cur)[1] : CustomMetricIDService.getCustomMetricName(cur)
                    };
                });
        } else {
            $scope.uiState.currentFormattedNames = [];
        }
        $scope.refreshMetricsValues();
    }

    // Override, as we don't really have a ref
    $scope.refreshMetricsValues = function() {
        let metrics = ($scope.evaluationDetails && $scope.evaluationDetails.metrics) ? $scope.evaluationDetails.metrics : null;

        // MES uses $formattedMetrics
        $scope.uiState.$formattedMetrics = {};
        // Summary uses $metricsToDisplay
        $scope.uiState.$metricsToDisplay = [];

        for (let kv of $scope.uiState.currentFormattedNames) {
            const metricCode = kv.key;
            const metricName = kv.label;
            const formattedValue = MetricsUtils.getMetricValue(metrics, metricCode, 3);
            $scope.uiState.$formattedMetrics[metricCode] = formattedValue;

            let customMetric = undefined;
            if (metrics.customMetricsResults) {
                customMetric = metrics.customMetricsResults.filter(m => m.metric.name === metricName)[0];
            }

            const metricforDescription = customMetric ? customMetric.metric : LLMEvaluationMetrics.getMetricByCode(metricCode);
            const description = metricforDescription ? metricforDescription.shortDescription : undefined;

            $scope.uiState.$metricsToDisplay.push({
                metricName: metricName,
                formattedValue: formattedValue,
                description: description,
                customMetric: customMetric ? customMetric.metric : undefined
            })
        }

        $scope.uiState.$metricsToDisplay = $scope.uiState.$metricsToDisplay.sort((a, b) => a.metricName.localeCompare(b.metricName))
    }
})

app.controller("LLMEvaluationReportSummaryController", function ($scope, $controller, LLMEvaluationMetrics, SamplingData) {

    $controller("_MLReportSummaryController", {$scope:$scope});
    $controller("EvaluationLabelUtils", { $scope: $scope });

    $scope.$watch("evaluationDetails", () => {
        if($scope.evaluationDetails && $scope.evaluationDetails.evaluation) {
            $scope.refreshMetrics($scope.evaluationDetails.evaluation.predictionType, LLMEvaluationMetrics.getCustomMetricNamesFromEvaluationDetails($scope.evaluationDetails));
            if ($scope.evaluationDetails.evaluation.selection) {
                $scope.sampleConfig = SamplingData.getSamplingMethodForDocumentation($scope.evaluationDetails.evaluation.selection);
            }
        }
        // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
        $scope.puppeteerHook_elementContentLoaded = true;
    });

});

app.controller("_LLMRowByRowAnalysisController", function($scope, $rootScope, $stateParams, $state, $controller, $q, Assert, DataikuAPI, MonoFuture, Logger, ActiveProjectKey, FullModelLikeIdUtils, LocalStorage, DashboardUtils, FutureWatcher){
    $scope.shakerWithSteps = false;
    $scope.isRowByRowInitialCompareColumnsScreenDisplayed = false;
    $scope.isInDashboard = DashboardUtils.isInDashboard();
    if ($scope.isInDashboard){
        $scope.tableId = Math.random().toString(36).slice(2);
    }
    const mesId = $stateParams.mesId || $scope.mesId;
    const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
    const fme = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);

    function addMESPrefix(key) {
        let prefix = "dss.modelevaluation";
        if ($scope.isInDashboard) {
            prefix += ".dashboard"; // dashboards are read-only, so, different configuration than everywhere else
        }
        return prefix + `.${key}.${mesId}`;
    }

    /* ********************* Callbacks for shakerExploreBase ******************* */

    // Nothing to save
    $scope.shakerHooks.saveForAuto = function() {
        return Promise.resolve({});
    }

    // We somehow miss that part of DataikuController, that is needed for shaker-in-dashboard
    $scope.setSpinnerPosition = $scope.setSpinnerPosition || function(position) {
        $rootScope.spinnerPosition = position;
    };

    var monoFuturizedRefresh = MonoFuture($scope).wrap(DataikuAPI.modelevaluations.sampleRefreshTable);

    /*
    Since we are plugged into the shaker table, we send a request with a list of selected columns. We therefore don't receive the full row. Since we need it for the bottom part of the
    comparison screen, we made the backend call return both the tableWithSelectedColumns and the fullTable. We hack the monoFuturizedRefresh and extract the fullTable and passing to the shaker
    only the tableWithSelectedColumns
        */
    $scope.fullChunks = [];
    $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
        function transform(future) {
            if (future.status === 200 && future.data.result) {
                $scope.fullTable = future.data.result.fullTable;
                $scope.tableWithSelectedColumns = future.data.result.tableWithSelectedColumns;
                future.data.result = {...future.data.result.tableWithSelectedColumns};
            }
            return future;
        }
        const initialPromise = monoFuturizedRefresh(fme, $scope.shaker, filterRequest);
        const newPromise = initialPromise.then(transform, null, transform);
        FutureWatcher.enrichPromise({promise : newPromise});
        return newPromise;
    }

    $scope.shakerHooks.shakerForQuery = function(){
        return angular.copy($scope.shaker);
    }

    $scope.shakerHooks.updateColumnWidth = function(name, width) {
        $scope.shaker.columnWidthsByName[name] = width;
        $scope.refreshTable(false);
    };

    $scope.shakerHooks.fetchDetailedAnalysis = function() { };

    function displayInitialCompareColumnsScreen() {
        let savedColNames = LocalStorage.get(addMESPrefix('columns'));
        let selectedColIds = undefined;
        const colNames = $scope.fullTable.headers.map(h => h.name);
        // Since we use the fullTable, index of selectedColIds remains the same
        if (savedColNames) {
            selectedColIds = LocalStorage.get(addMESPrefix('selectedColIds'));
        } else {
            const evaluation = $scope.evaluationDetails.evaluation;

            let inputColumnName = evaluation.inputColumnName;
            let outputColumnName = evaluation.outputColumnName;
            if (evaluation.inputFormat === 'PROMPT_RECIPE') {
                inputColumnName = "dkuReconstructedInput";
                outputColumnName = "dkuParsedOutput";
            }
            let contextColumnName = evaluation.contextColumnName;
            if (['PROMPT_RECIPE', 'DATAIKU_ANSWERS'].includes(evaluation.inputFormat)) {
                contextColumnName = "dkuParsedContexts";
            }
            const inputColumnIndex = $scope.fullTable.headers.findIndex(header => header.name == inputColumnName);
            const contextColumnIndex = $scope.fullTable.headers.findIndex(header => header.name == contextColumnName);
            const groundTruthColumnIndex = $scope.fullTable.headers.findIndex(header => header.name == evaluation.groundTruthColumnName);
            const outputColumnIndex = $scope.fullTable.headers.findIndex(header => header.name == outputColumnName);
            selectedColIds = [inputColumnIndex, outputColumnIndex, groundTruthColumnIndex, contextColumnIndex];
        }
        $scope.compareColumnValuesView.isCompareViewVisible = true;
        $scope.compareColumnValuesView.initialSelectedColIds = [];
        $scope.compareColumnValuesView.lastSelectedColId = null;
        for (const columnToCompare of selectedColIds) {
            if (columnToCompare >= 0) {
                $scope.compareColumnValuesView.initialSelectedColIds.push(columnToCompare);
            }
        }
        $scope.isRowByRowInitialCompareColumnsScreenDisplayed = true;
        LocalStorage.set(addMESPrefix('columns'), colNames);
    }

    $scope.shakerHooks.afterTableRefresh = function() {
        const content = $scope.table.initialChunk.content;
        if (!content || !content.length) {
            return;
        }
        const cols = $scope.table.initialCols;
        let row = [];
        for (let i = 0; i < cols; i++) {
            row.push(content[i]);
        }
        $scope.shakerState.selectedRow = 0;
        $scope.compareColumnValuesView.lastSelectedRow = row;
        if (!$scope.isRowByRowInitialCompareColumnsScreenDisplayed) {
            displayInitialCompareColumnsScreen();
        }
        LocalStorage.set(addMESPrefix('shaker'), $scope.shaker);
        setTimeout(() => {
            $rootScope.$broadcast("compareColumnValuesChangeRow", 'previous', $scope.tableId);
            // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
            $scope.puppeteerHook_elementContentLoaded = true;
        });
    }

    $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
        return DataikuAPI.modelevaluations.sampleGetTableChunk(fme, $scope.shaker,
            firstRow, nbRows, firstCol, nbCols, filterRequest)
    }
    $scope.shakerReadOnlyActions = true;
    $scope.shakerWritable = false;
    $scope.shakerCellClickable = true;
    $scope.isCompareCellAvailable = true;

    function getRowFromTable (table, rowIdx) {
            const content = table.initialChunk.content;
            if (!content || !content.length) {
                return;
            }
            const nbCols = table.initialCols;
            if (!nbCols) {
                return;
            }
            let fullRow = [];
            for (let col = 0; col < nbCols; col++){
                const globalIdx = rowIdx * nbCols + col;
                fullRow.push({
                  content: content[globalIdx],
                  colId:col,
                  origRowIdx:rowIdx,
                  rowId:rowIdx,
                  colorBin: null,
                  status: "VU"
                });
            }
            return fullRow
        }

    $scope.$on('triggerCompareColumnValues', function(event, rowPromise, colId) {
        rowPromise.then(row => {
            $scope.compareColumnValuesView.lastSelectedRow = getRowFromTable($scope.fullTable, row[0].rowId);
            $scope.$apply();
        });
        const colName = $scope.table.headers[colId].name;
        const colIdInFullTable = $scope.fullTable.headers.findIndex(header => header.name == colName);
        $scope.compareColumnValuesView.lastSelectedColId = colIdInFullTable;
    });

    $scope.$on('shakerCellClick', function(event, rowPromise, colId, fatId) {
        if (fatId && $scope.tableId !== fatId){
            return; // This event is no destined to us
        }
        rowPromise.then(row => {
            $scope.compareColumnValuesView.lastSelectedRow = getRowFromTable($scope.fullTable, row[0].rowId);
            $scope.$apply();
        });
    });

    $scope.$on('compareColumnValuesColumnChange', function(event, selectedColIds, compareColumnId) {
        if (compareColumnId && $scope.tableId !== compareColumnId){
            return; // This event is no destined to us
        }
        LocalStorage.set(addMESPrefix('selectedColIds'), selectedColIds);
    });

    $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
        // withFullSampleStatistics, fullSamplePartitionId are not relevant in this context
        DataikuAPI.modelevaluations.sampleDetailedColumnAnalysis(fme, $scope.shakerHooks.shakerForQuery(), columnName, alphanumMaxResults).success(function(data){
                    setAnalysis(data);
        }).error(function(a, b, c) {
            if (handleError) {
                handleError(a, b, c);
            }
            setErrorInScope.bind($scope)(a, b, c);
        });
    };

    // Load shaker
    $scope.$watch('evaluationDetails', () => {
        $scope.compareColumnValuesView = {
            isCompareViewVisible: false,
            lastSelectedRow: null,
            lastSelectedColId: null,
            initialSelectedColIds: [0],
            maxColumns: 50
        }
        $scope.searchableDataset = false;
        $scope.baseInit();
        $scope.shaker = LocalStorage.get(addMESPrefix('shaker')) || {
            steps: [],
            globalSearchQuery: "",
            "$headerOptions": {
                showName: true,
                showMeaning: false,
                showDescription: false,
                showCustomFields: false,
                showProgressBar: false,
                disableHeaderMenu: $scope.isInDashboard,
            }
        };
        $scope.shakerState.filtersExplicitlyAllowed = true;
        $scope.originalShaker = angular.copy($scope.shaker);
        $scope.fixupShaker();
        $scope.refreshTable(false);
    });
});


app.directive("llmRowByRowAnalysis", function() {
    return {
        scope: true,
        controller: function($scope, $stateParams, $controller, DataikuAPI, ActiveProjectKey, PMLFilteringService, TopNav, WT1) {
            WT1.event("llmevaluation-row-by-row-analysis-open");
            $controller("_LLMRowByRowAnalysisController", {$scope:$scope});
        }
    }
});

app.controller("ModelEvaluationStoreBaseEvaluationController", function($scope, DataikuAPI, $stateParams, TopNav, $controller,
    PMLFilteringService, PMLSettings, ActiveProjectKey, CreateModalFromComponent, createOrAppendMELikeToModelComparisonModalDirective,
    ModelEvaluationUtils, FullModelLikeIdUtils) {
    $controller("ModelEvaluationsMetricsHandlingCommon", {$scope, PMLFilteringService, PMLSettings});
    $scope.noMlReportTourHere = true; // the tabs needed for the tour are not present
    const mesId = $stateParams.mesId || $scope.mesId;
    const evaluationId = $stateParams.evaluationId || $scope.evaluationId;

    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_MODEL_EVALUATION_STORES, TopNav.LEFT_MODEL_EVALUATION_STORES, "MODEL_EVALUATION_STORE-EVALUATION", "report");
        TopNav.setItem(TopNav.ITEM_MODEL_EVALUATION_STORE, mesId, {evaluationId: evaluationId});
    }

    $scope.refreshStatus = function() {
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        DataikuAPI.modelevaluationstores.getEvaluationDetails(ActiveProjectKey.get(), mesId, evaluationId).success(function(data) {
            $scope.isLLM = data.evaluationDetails.evaluation && data.evaluationDetails.evaluation.type === 'llm';
            if ($scope.isLLM) {
                $controller("ModelEvaluationStoreLLMEvaluationController", {$scope, PMLFilteringService, PMLSettings});
                $scope.refresh(data);
            } else {
                $controller("ModelEvaluationStoreTabularEvaluationController", {$scope, PMLFilteringService, PMLSettings});
                $scope.refresh(data);
            }
        }).error(setErrorInScope.bind($scope));

        if ($scope.onLoadSuccess) $scope.onLoadSuccess();
    };
    $scope.$watch("modelEvaluationStore", (nv) => { if (nv) $scope.refreshStatus(); });


    $scope.comparisonForbiddenReason = function() {
        // todo @deephub: block deephub models from being compared after deephub evaluation recipe is fixed
        if(!$scope.modelEvaluation.predictionType && !$scope.modelEvaluation.llmTaskType) {
            return "Evaluation can not be compared";
        }

        if($scope.modelEvaluation.modelParams && $scope.modelEvaluation.modelParams.isPartitioned) {
            return "Partitioned models cannot be compared";
        }
        if($scope.modelEvaluation.modelParams && $scope.modelEvaluation.modelParams.isEnsembled) {
            return "Ensembled models cannot be compared";
        }
    }

    $scope.compareSelectedEvaluations = function() {
        let evaluationName = $scope.modelEvaluation.userMeta.name;
        if (!evaluationName) {
            evaluationName = '1 evaluation';
        }
        CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
            fullIds: [$scope.modelEvaluation.ref.fullId],
            modelTaskType: $scope.isLLM ? "LLM" : $scope.modelEvaluation.predictionType,
            allowImportOfRelatedVersions: true,
            suggestedMCName: `Compare ${evaluationName}`,
            projectKey: $stateParams.projectKey,
            trackFrom: 'model-evaluation-page',
            areAllModelsExternal: $scope.modelEvaluation.modelType === 'EXTERNAL'
        });
    }
    $scope.$on('triggerCreateModelComparison', function(event) {
        $scope.compareSelectedEvaluations();
    });

    $scope.exportEvaluationSample = function() {
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        ModelEvaluationUtils.exportEvaluationSample($scope, FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId));
    }
});

app.controller("ModelEvaluationStoreTabularEvaluationController", function($scope, DataikuAPI, $stateParams, TopNav, $controller, $location,
    ActiveProjectKey, ModelEvaluationUtils, FullModelLikeIdUtils, MLDiagnosticsService, BinaryClassificationModelsService, WT1, FutureProgressModal, CachedAPICalls) {

    if (!$scope.readOnly) {
        DataikuAPI.analysis.listHeads($stateParams.projectKey, true).success(function(data) {
            $scope.analysesWithHeads = data;
        });
    }

    function getCustomMetricNamesFromModelDetails(modelDetails) {
        let names = [];

        if (modelDetails.modeling && modelDetails.modeling.metrics && modelDetails.modeling.metrics.customMetrics) {
            names = modelDetails.modeling.metrics.customMetrics.map(function(metric) { return metric.name });
        }

        return names;
    }

    $scope.isTimeseriesPrediction = function() {
        return $scope.isForecast();
    }

    $scope.refresh = function(data){
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        try {
            data.evaluationDetails = adjustEvaluationDetailsForOverriddenThreshold(data.evaluationDetails);
        } catch (error) {
            // proceed with the original data
        }

        $scope.mesContext.evaluationFullInfo = data.evaluationDetails;

        $scope.versionsContext.activeMetric = data.evaluationDetails.evaluation.metricParams.evaluationMetric;
        $scope.evaluationDetails = data.evaluationDetails;
        $scope.modelEvaluation = data.evaluationDetails.evaluation;
        $scope.evaluationDiagnostics = MLDiagnosticsService.groupByType($scope.evaluationDetails.mlDiagnostics);
        $scope.evaluationDiagnosticsCount = MLDiagnosticsService.countDiagnostics($scope.evaluationDetails);
        $scope.modelData = data.evaluationDetails.details;
        $scope.uiState.driftParams = {
            ...ModelEvaluationUtils.defaultDriftParams,
            ...$scope.modelEvaluation.hasDriftReference
                && ($scope.modelData.dataEvaluationMetrics && $scope.modelData.dataEvaluationMetrics.driftResult && $scope.modelData.dataEvaluationMetrics.driftResult.perColumnSettings)
                &&
            {
                columns: $scope.modelData.dataEvaluationMetrics.driftResult.perColumnSettings
                .reduce((map, column) => ({
                    ...map,
                    [column.name]: {
                    enabled: column.actualHandling !== "IGNORED",
                    handling: column.actualHandling
                    }
                }), {}),
                confidenceLevel: $scope.modelData.dataEvaluationMetrics.driftModelAccuracy && $scope.modelData.dataEvaluationMetrics.driftModelAccuracy.confidenceLevel ?
                $scope.modelData.dataEvaluationMetrics.driftModelAccuracy.confidenceLevel : 0.95,
            }
        };

        if ($scope.modelData.coreParams) {
            CachedAPICalls.pmlDiagnosticsDefinition.then(pmlDiagnosticsDefinition => {
                $scope.diagnosticsDefinition = pmlDiagnosticsDefinition($scope.modelData.coreParams.backendType, $scope.modelData.coreParams.prediction_type);
            });
        }
        $scope.refreshMetrics($scope.modelEvaluation.predictionType, getCustomMetricNamesFromModelDetails($scope.modelData));
        if (!$scope.noSetLoc) {
            TopNav.setItem(TopNav.ITEM_MODEL_EVALUATION_STORE, mesId, {name: data.modelEvaluationStore.name, evaluationId: evaluationId, runName: $scope.evaluationDetails.evaluation.userMeta.name});
        }
        $scope.modelData.modelEvaluation = data.evaluationDetails.evaluation;
        $controller("_PredictionModelReportController",{$scope:$scope});

        $scope.isPartitionedModel = function() { return false; };

        $scope.fullModelEvaluationId = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);
        $scope.currentMEReference = ModelEvaluationUtils.makeRefDisplayItemFromEvaluation($scope.modelEvaluation, $scope.storeList, $scope.analysesWithHeads);

        DataikuAPI.modelevaluations.listCompatibleReferencesForDrift(ActiveProjectKey.get(), $scope.fullModelEvaluationId, false).success(function(data) {
            $scope.compatibleReferences = data.compatibleReferences;
            if(!$scope.uiState.driftState.selectedReference) {
                if ($location.search().driftReference){
                    $scope.uiState.driftState.selectedReference = $scope.compatibleReferences.find(comparableModelItem => $location.search().driftReference && comparableModelItem.mli.fullId == $location.search().driftReference);
                    if($scope.uiState.driftState.selectedReference.mli.evaluationId !== $scope.modelEvaluation.ref.evaluationId){
                        // we directly got a different reference from the url. We should re-compute, as the infos were not persisted
                        $scope.applyPersistedDrift();
                        $scope.updatePerDriftRefs();
                        $scope.computeDrift({driftType: ModelEvaluationUtils.tabNameToDriftType[$scope.uiState.settingsPane], from: ModelEvaluationUtils.wt1Properties.COMPUTE_FROM_URL})
                    }
                } else if($scope.modelEvaluation && $scope.modelEvaluation.modelType == 'EXTERNAL' && $scope.modelEvaluation.hasDriftReference){
                    $scope.uiState.driftState.selectedReference = $scope.compatibleReferences.find(comparable => comparable.mli.evaluationId == $scope.modelEvaluation.ref.evaluationId);
                } else if ($scope.modelEvaluation && $scope.modelEvaluation.modelType == 'EXTERNAL' && !$scope.modelEvaluation.hasDriftReference){
                    $scope.uiState.driftState.selectedReference = null;
                } else {
                    $scope.uiState.driftState.selectedReference = $scope.compatibleReferences.find(comparableModelItem => data.defaultReference && comparableModelItem.mli.fullId == data.defaultReference.fullId);
                }
            }
        }).error(setErrorInScope.bind($scope));
        if ($scope.modelData.classes) {
            $scope.uiState.driftState.currentClass = $scope.modelData.classes[0];
        }

        $scope.$emit('refreshComparisonForbiddenReason', $scope.comparisonForbiddenReason());
    }

    $scope.$watch('uiState.driftState.selectedReference', function(nv, ov) {
        if($scope.shouldApplyPersistedDrift(ov, nv)) {
            $scope.applyPersistedDrift();
        }
        if ($scope.isForecast() && nv !== null) {
            $scope.generatePerformanceDriftTab();
        }
        $scope.updatePerDriftRefs();
    });

    $scope.shouldApplyPersistedDrift = function(ov, nv) {
        if (!($scope.modelData && $scope.modelData.dataEvaluationMetrics &&  $scope.modelData.dataEvaluationMetrics.driftResult && nv !== null && !angular.equals(nv, ov))){
            return false;
        }

        if (nv === undefined) {
            return true;
        }

        if ($scope.modelEvaluation && $scope.modelEvaluation.modelType == 'EXTERNAL' && $scope.modelEvaluation.hasDriftReference){
            return $scope.modelEvaluation.ref.evaluationId === nv.mli.evaluationId;
        } else {
            return $scope.fullModelId === nv.mli.fullId;
        }
    }

    $scope.applyPersistedDrift = function() {
        $scope.uiState.driftParams = {
            ...ModelEvaluationUtils.defaultDriftParams,
            columns: $scope.modelData.dataEvaluationMetrics.driftResult.perColumnSettings
                .reduce((map, column) => ({
                    ...map, [column.name]: {
                        'enabled': column.actualHandling !== "IGNORED",
                        'handling': column.actualHandling
                    }
                }), {}),
            confidenceLevel: $scope.modelData.dataEvaluationMetrics.driftModelAccuracy && $scope.modelData.dataEvaluationMetrics.driftModelAccuracy.confidenceLevel ?
                $scope.modelData.dataEvaluationMetrics.driftModelAccuracy.confidenceLevel : 0.95,
        };
        $scope.uiState.driftState = {
            selectedReference: $scope.uiState.driftState.selectedReference,
            driftResult: $scope.modelData.dataEvaluationMetrics.driftResult,
            driftParamsOfResult: angular.copy($scope.uiState.driftParams),
            driftVersusImportanceChart: null, // For now it is not saved during initial run of the ME
            driftVersusImportanceChartComputed: false,
        };

        $scope.uiState.driftVersusImportanceChart = null;

        $scope.generatePredictionDriftTab();
        $scope.generatePerformanceDriftTab();
    }

    $scope.updatePerDriftRefs = function() {
        if ($scope.uiState.driftState.perfDriftRefs) {
            const [ cur, ref ] = $scope.uiState.driftState.perfDriftRefs;
            $scope.metricParamsList = [ cur.metricParams, ref.metricParams ];
        }
    }

    function getPredictedClassCount(modelOrEvaluationData, predData, threshold) {
        // Fallback to predData.predictedClassCount for backward compatibility (and non-binary classification, and non-probabilistic models),
        // but prefer predictedClassCountPerCut if we can make any use of it, ie. if we have per cut data and an active threshold.
        if (!modelOrEvaluationData || !modelOrEvaluationData.details) {
            return predData.predictedClassCount;
        }
        let referenceDetails = modelOrEvaluationData.details;

        // Do we have per cut data?
        if (!referenceDetails.perf || !referenceDetails.perf.perCutData) {
             return predData.predictedClassCount;
        }
        const cuts = referenceDetails.perf.perCutData.cut;
        if (!cuts || !cuts.length) {
            return predData.predictedClassCount;
        }

        // Do we have a threshold?
        if (threshold == undefined) {
            return predData.predictedClassCount;
        }

        // Do we have predicted class count per cut?
        if (!predData.predictedClassCountPerCut || !predData.predictedClassCountPerCut.length) {
            return predData.predictedClassCount;
        }

        // We have everything, let's find the actual predicted class count for the reference threshold.
        let cutIndex = BinaryClassificationModelsService.findCut(cuts, threshold);
        if (cutIndex >= predData.predictedClassCountPerCut.length) {
            return predData.predictedClassCount;
        }
        return predData.predictedClassCountPerCut[cutIndex];
    }

    $scope.generatePredictionDriftTab = function() {
        DataikuAPI.modelcomparisons.getModelsDetails([$scope.uiState.driftState.selectedReference.mli.fullId]).success(function(dataArray) {
            const referenceData = dataArray[0];
            let referencePredictionInfo;
            $scope.uiState.predictionDriftCurrentThreshold = undefined;
            $scope.uiState.predictionDriftReferenceThreshold = undefined;
            if ($scope.uiState.driftState.driftResult.currentThreshold !== undefined) {
                // If we have currentThreshold directly in driftResult, this is the preferred option (should be the case with MEs in DSS >= 13.2.2).
                $scope.uiState.predictionDriftCurrentThreshold = $scope.uiState.driftState.driftResult.currentThreshold;
            }
            else {
                // If we don't have currentThreshold in driftResult, fallback on activeClassifierThreshold (same behaviour as DSS < 13.2.2).
                $scope.uiState.predictionDriftCurrentThreshold = $scope.evaluationDetails.details.userMeta.activeClassifierThreshold;
            }
            if ($scope.uiState.driftState.driftResult.referenceThreshold !== undefined) {
                // If we have referenceThreshold in driftResult, use that. Otherwise, predictionDriftReferenceThreshold will stay undefined, which will be close
                // to what DSS < 13.2.2 was doing.
                $scope.uiState.predictionDriftReferenceThreshold = $scope.uiState.driftState.driftResult.referenceThreshold;
            }
            if ($scope.uiState.driftState.selectedReference.mli.evaluationId === $scope.evaluationDetails.evaluation.ref.evaluationId && $scope.modelEvaluation.modelType === 'EXTERNAL') {
                referencePredictionInfo = referenceData.details.referencePredictionInfo;
            } else {
                referencePredictionInfo = referenceData.details.predictionInfo;
            }
            if ("REGRESSION" === $scope.evaluationDetails.evaluation.predictionType) {
                let xs = [];
                let ys = [];
                let labels = [];
                const refPredDataAvailable = referencePredictionInfo && referencePredictionInfo.pdf;
                const curPredDataAvailable = $scope.evaluationDetails.details.predictionInfo && $scope.evaluationDetails.details.predictionInfo.pdf;

                $scope.uiState.driftState.unavailableReference = !refPredDataAvailable;
                $scope.uiState.driftState.unavailableCurrent = !curPredDataAvailable;

                if (refPredDataAvailable && curPredDataAvailable) {
                    xs.push(referencePredictionInfo.x);
                    ys.push(referencePredictionInfo.pdf)
                    labels.push('Reference');

                    xs.push($scope.evaluationDetails.details.predictionInfo.x);
                    ys.push($scope.evaluationDetails.details.predictionInfo.pdf)
                    labels.push('Current');


                    $scope.uiState.driftState.pdfs = {
                        xs: xs,
                        ys: ys,
                        colors: $scope.colors.slice(0, 2),
                        labels: labels
                    };
                }
            } else {
                const refPredDataAvailable = referencePredictionInfo;
                const curPredDataAvailable = $scope.evaluationDetails.details.predictionInfo;

                $scope.uiState.driftState.unavailableReference = !refPredDataAvailable;
                $scope.uiState.driftState.unavailableCurrent = !curPredDataAvailable;

                if(refPredDataAvailable && curPredDataAvailable){
                    const refClasses = listClassesFromPreprocessingOrPredData(referenceData.details.preprocessing, refPredDataAvailable);
                    const curClasses = listClassesFromPreprocessingOrPredData($scope.evaluationDetails.details.preprocessing, curPredDataAvailable);
                    $scope.uiState.driftState.classes = _.sortedUniq(_.sortBy(refClasses.concat(curClasses)));
                    $scope.uiState.driftState.currentClass = $scope.uiState.driftState.classes[0];
                    let refPredDataClassCount = getPredictedClassCount(referenceData, refPredDataAvailable, $scope.uiState.predictionDriftReferenceThreshold);
                    let curPredDataClassCount = getPredictedClassCount($scope.evaluationDetails, curPredDataAvailable, $scope.uiState.predictionDriftCurrentThreshold);
                    if(refPredDataClassCount && curPredDataClassCount){
                        $scope.uiState.driftState.refPredValueCount = getPredictionValuesFromPredictionData(refPredDataClassCount);
                        $scope.uiState.driftState.curPredValueCount = getPredictionValuesFromPredictionData(curPredDataClassCount);
                        $scope.computePredictionHistogram();
                    }
                    if(refPredDataAvailable.probabilityDensities && curPredDataAvailable.probabilityDensities){
                        $scope.uiState.driftState.refProbabilityDensities = referencePredictionInfo.probabilityDensities;
                        $scope.computePddForClass($scope.uiState.driftState.classes[0]);
                    }
                }


            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.generatePerformanceDriftTab = function() {
        $scope.uiState.driftState.perfDriftAvailable = false;
        if ($scope.uiState.driftState.selectedReference?.mli.evaluationId === $scope.evaluationDetails.evaluation.ref.evaluationId) { // For SER with reference dataset, we do not compute yet the performance for the reference dataset.
            return;
        }
        DataikuAPI.modelcomparisons.getModelsDetails([$scope.uiState.driftState.selectedReference.mli.fullId]).success(function(dataArray) {
            const data = dataArray[0];
            let ref = ModelEvaluationUtils.makeRefDisplayItemFromComparable(data, $scope.storeList, $scope.modelList, $scope.analysesWithHeads);
            ref.metrics = data.metrics;
            ref.metricParams = data.metricParams;
            let cur = ModelEvaluationUtils.makeRefDisplayItemFromEvaluation($scope.evaluationDetails.evaluation, $scope.storeList);
            cur.metrics = $scope.evaluationDetails.metrics;
            cur.metricParams = $scope.evaluationDetails.evaluation.metricParams;
            $scope.metricParamsList = [ cur.metricParams, ref.metricParams ];
            $scope.uiState.driftState.perfDriftRefs = [cur, ref];
            $scope.uiState.driftState.perfDriftAvailable = true;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.computeDrift = function(wt1Params) {
        const colsAsArray = Object.values($scope.uiState.driftParams.columns);
        if (colsAsArray.length && colsAsArray.every(c => c.enabled && c.handling === "TEXT")) {
            // Interactive data drift does not support text drift. But if we only have text columns, they will get ignored,
            // we won't have any column left to compute the drift against, and it will crash. So, we just don't do anything.
            return;
        }
        $scope.uiState.driftState = {
            selectedReference: $scope.uiState.driftState.selectedReference
        };
        $scope.computeDriftAndGenerateTabs(wt1Params);
    }

    function adjustEvaluationDetailsForOverriddenThreshold(evaluationDetails) {
        // Fix for the bug described in https://app.shortcut.com/dataiku/story/141374. Mimics the threshold-modification
        // logic applied in reg_standalone_evaluation_recipe.py. A cleaner solution would be to modify the code in
        // StandaloneEvaluationRecipeRunner.java, but this would not fix the issue for existing recipes (barring a
        // migration).
        const predictionType = evaluationDetails.evaluation.predictionType;
        const isProbaAware = evaluationDetails.evaluation.evaluateRecipeParams.isProbaAware;
        const dontComputePerformance = evaluationDetails.evaluation.evaluateRecipeParams.dontComputePerformance;
        let thresholdAutoOptimized = evaluationDetails.evaluation.thresholdAutoOptimized;

        const thresholdOverrideApplied = predictionType === "BINARY_CLASSIFICATION" && isProbaAware && dontComputePerformance && thresholdAutoOptimized;
        if (!thresholdOverrideApplied) {
            return evaluationDetails;
        }

        const updatedEvaluationDetails = angular.copy(evaluationDetails);
        let activeClassifierThreshold;
        if (isProbaAware) {
            thresholdAutoOptimized = dontComputePerformance ? false : thresholdAutoOptimized;

            let threshold = evaluationDetails.evaluation.activeClassifierThreshold;
            threshold = threshold == null ? 0.5 : threshold;
            activeClassifierThreshold = thresholdAutoOptimized ? threshold : 0.5;
        } else {
            thresholdAutoOptimized = false;
            activeClassifierThreshold = 0.5;
        }
        if (thresholdAutoOptimized == null || activeClassifierThreshold == null) {  // Also handles undefined
            return evaluationDetails;
        }
        updatedEvaluationDetails.evaluation.thresholdAutoOptimized = thresholdAutoOptimized;
        // Threshold appears in two locations in the evaluationDetails object
        updatedEvaluationDetails.evaluation.activeClassifierThreshold = activeClassifierThreshold;
        updatedEvaluationDetails.details.userMeta.activeClassifierThreshold = activeClassifierThreshold;
        return updatedEvaluationDetails;
    }

    function listClassesFromPreprocessingOrPredData(preprocessing, predData) {
        const listFromClasses = preprocessing.target_remapping.map(mapping => mapping.sourceValue);
        if (listFromClasses.length > 0){
            return listFromClasses;
        } else if (predData.predictedClassCount){
            return Object.keys(predData.predictedClassCount);
        } else {
            return [];
        }
    }

    function getPredictionValuesFromPredictionData(predDataClassCount) {
        const classes = $scope.uiState.driftState.classes;
        var ret = {};
        const preds = predDataClassCount;
        const allPreds = Object.values(preds).reduce((s, p) => s + p, 0);

        classes.map(c => ret[c] = { records: preds[c] ? preds[c] : 0, pct: preds[c] ? ((100.*preds[c])/allPreds).toFixed(2) : (0.).toFixed(2) });
        return ret;
    }

    $scope.computeDriftAndGenerateTabs = function(wt1Params) {
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        const referenceId = $scope.uiState.driftState.selectedReference.mli.fullId;
        const currentId = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);
        if ($scope.isForecast()) {
            $scope.generatePerformanceDriftTab();
        } else {
            const promise = DataikuAPI.modelevaluations.computeDrift(ActiveProjectKey.get(), referenceId, currentId, $scope.uiState.driftParams);
            const copiedParams = angular.copy($scope.uiState.driftParams);
            return promise.then(({data}) => {
                return FutureProgressModal.show($scope, data, "Computing Drift");
            })
                .then((driftResults) => {
                    $scope.uiState.driftState.driftResult = driftResults;
                    $scope.uiState.driftState.driftParamsOfResult = copiedParams;
                    generateDriftVersusImportanceChart();
                    $scope.generatePredictionDriftTab();
                    $scope.generatePerformanceDriftTab();
                    $location.search('driftReference', $scope.uiState.driftState.selectedReference.mli.fullId);
                    WT1.event("compute-drift", wt1Params);
                })
                .catch(setErrorInScope.bind($scope));
        }
    }

    function generateDriftVersusImportanceChart() {
        $scope.uiState.driftState.driftVersusImportanceChartComputed = true;

        if(!$scope.uiState.driftState.driftResult.driftModelResult || !$scope.uiState.driftState.driftResult.driftModelResult.driftVersusImportance.columnImportanceScores) {
            $scope.uiState.driftVersusImportanceChart = null;
            return;
        }

        $scope.uiState.driftVersusImportanceChart = {
            animation: false,
            tooltip: {
                trigger: 'item',
                axisPointer: { type: 'cross', label: { formatter: ({value})=> Math.round(100 * value) + '%'  } }
            },
            grid: { left: 40, top: 20, right: 20, bottom: 30, containLabel: true },
            xAxis: {
                type: 'value',
                min: 0,
                name: "Drift model feature importance (%)",
                nameLocation: "middle",
                nameGap: 30,
                axisLabel: { formatter: value => Math.round(100 * value) + '%' }
            },
            yAxis: {
                type: 'value',
                min: 0,
                name: "Original model feature importance (%)",
                nameLocation: "middle",
                nameGap: 40,
                axisLabel: { formatter: value => Math.round(100 * value) + '%' }
            },
            series: {
                type: 'scatter',
                symbolSize: 10,
                data: _.zip(
                    $scope.uiState.driftState.driftResult.driftModelResult.driftVersusImportance.columnDriftScores,
                    $scope.uiState.driftState.driftResult.driftModelResult.driftVersusImportance.columnImportanceScores,
                    $scope.uiState.driftState.driftResult.driftModelResult.driftVersusImportance.columns,
                ),
                tooltip: {
                    formatter: ({value}) => '<b>Column: '+sanitize(value[2])+'</b><br>'
                        + 'Drift model feature importance: '+ Math.round(100 * value[0]) + '%<br>'
                        + 'Original model feature importance: '+ Math.round(100 * value[1]) + '%<br>'
                },
                itemStyle: {
                    color: value => window.dkuColorPalettes.discrete[0].colors[value.dataIndex % window.dkuColorPalettes.discrete[0].colors.length]
                }
            }
        };
    }

    $scope.computePredictionHistogram = function() {
        const source = [['Class', 'Current', 'Reference']];
        for (const currentClass of $scope.uiState.driftState.classes) {
            source.push([
                currentClass,
                $scope.uiState.driftState.curPredValueCount[currentClass].pct,
                $scope.uiState.driftState.refPredValueCount[currentClass].pct
            ]);
        }
        const colors = window.dkuColorPalettes.discrete[0].colors.filter((x,idx) => idx%2 === 0)
        $scope.uiState.driftState.predHistogramOptions = {
            tooltip: {},
            dataset: {
                source
            },
            xAxis: {type: 'category', name: 'Predicted class', nameLocation: 'middle', nameGap: 25},
            yAxis: {name: '% of predicted classes', nameLocation: 'middle', nameGap: 25},
            series: [
                {type: 'bar', color: colors[0]},
                {type: 'bar', color: colors[1]}
            ]
        }
    }

    $scope.computePddForClass = function(className) {
        $scope.uiState.driftState.pdd = null;
        if ($scope.uiState.driftState.refProbabilityDensities && (className != null) && Object.keys($scope.uiState.driftState.refProbabilityDensities).length
            && $scope.evaluationDetails.details.predictionInfo.probabilityDensities
            && Object.keys($scope.evaluationDetails.details.predictionInfo.probabilityDensities).length) {
            $scope.uiState.driftState.pdd = {};
            let dd_values = $scope.evaluationDetails.details.predictionInfo.probabilityDensities[className].density;
            $scope.uiState.driftState.pdd.x = dd_values.map(function(_, i, a) { return i / a.length; });
            let dd2_values = $scope.uiState.driftState.refProbabilityDensities[className].density;
            $scope.uiState.driftState.pdd.ys = [dd_values, dd2_values];
            $scope.uiState.driftState.pdd.labels = ['class ' + className + ' current', 'class ' + className + ' reference'];
            $scope.uiState.driftState.pdd.colors = $scope.colors.slice(0, 2).concat("#9467bd");
        } else {
            $scope.uiState.driftState.pdd = null;
        }
    }

    $scope.uiState.driftState = {
        selectedReference: null
    };

    $scope.uiState.driftParams = angular.copy(ModelEvaluationUtils.defaultDriftParams);

    $scope.refreshCurrentMEReference = function() {
        $scope.currentMEReference = ModelEvaluationUtils.makeRefDisplayItemFromEvaluation($scope.modelEvaluation, $scope.storeList);
    }

    DataikuAPI.modelevaluationstores.listWithAccessible($stateParams.projectKey).success(function(data){
        $scope.storeList = data;
    });

    DataikuAPI.savedmodels.listWithAccessible($stateParams.projectKey).success(function(data){
        $scope.modelList = data;
    });

    $scope.$watch('modelEvaluation', $scope.refreshCurrentMEReference);
    $scope.$watch('storeList', $scope.refreshCurrentMEReference);

    $scope.$watch('storeList', $scope.refreshCompatibleReferences);
    $scope.$watch('modelList', $scope.refreshCompatibleReferences);

    $scope.$on("$destroy", function() {
        $scope.mesContext.evaluationFullInfo = null;
    });

});


/************************** creation modal ***************************/

app.controller("NewModelEvaluationStoreController", function($scope, $state, DataikuAPI, WT1, $stateParams) {
    WT1.event("new-model-evaluation-store-modal-open");

    $scope.newMES = {
        name : null,
        settings : {
            zone: $scope.getRelevantZoneId($stateParams.zoneId)
        }
    };

    DataikuAPI.datasets.getModelEvaluationStoreOptionsNoContext($stateParams.projectKey).success(function(data) {
        $scope.managedDatasetOptions = data;
        $scope.partitioningOptions = [
            {"id" : "NP", "label" : "Not partitioned"},
        ].concat(data.projectPartitionings)

        $scope.newMES.settings.partitioningOptionId = "NP";
    }).error(setErrorInScope.bind($scope));

    $scope.create = function(){
        resetErrorInScope($scope);
        WT1.event("new-model-evaluation-store-modal-create");
        DataikuAPI.datasets.newModelEvaluationStore($stateParams.projectKey, $scope.newMES.name, $scope.newMES.settings).success(function(data) {
            $scope.dismiss();
            $state.go("projects.project.modelevaluationstores.modelevaluationstore.evaluations", {mesId: data.id})
        }).error(setErrorInScope.bind($scope));
    }
});

/************************ metrics and checks ******************************/
app.controller("ModelEvaluationStoreStatusPageController", function($scope, $stateParams, TopNav, $filter, WT1, ActivityIndicator){
    TopNav.setLocation(TopNav.TOP_MODEL_EVALUATION_STORES, TopNav.LEFT_MODEL_EVALUATION_STORES, TopNav.TABS_MODEL_EVALUATION_STORE, "status");

    $scope.addAllMetricsDatasetInFlow = function(view, partition, filter) {
        WT1.event("metrics-add-dataset-in-flow", {all:true});
        $scope.metricsCallbacks.createMetricsDataset(view, partition, filter).success(function() {
            ActivityIndicator.success("Metrics dataset created");
            var i = $scope.allComputedMetrics.notExistingViews.indexOf(view);
            if (i >= 0) {
                $scope.allComputedMetrics.notExistingViews.splice(i, 1);
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.clearAll  = function() {
        WT1.event("metrics-clear");
        $scope.metricsCallbacks.clearMetrics().success(function() {
            ActivityIndicator.success("Metrics cleared");
            $scope.$broadcast('metrics-refresh-displayed-data');
        }).error(setErrorInScope.bind($scope));
    };

});

app.controller("ModelEvaluationStoreMetricsViewController", function($scope, Debounce, FutureProgressModal, MetricsUtils, $filter, Fn) {
    $scope.views = {
        selected : 'versionsTable'
    };

    $scope.displayedMetrics = {metrics : [], $loaded : false};
    // function is not there when the page is loaded the first time, but is there when tabs change
    if ( $scope.refreshAllComputedMetrics ) $scope.refreshAllComputedMetrics();

    $scope.$watch("allComputedMetrics", function(nv){
        $scope.allPartitions = [];

        var set = {}
        if (nv) {
            nv.metrics.forEach(function(metric) {
                metric.partitionsWithValue.forEach(function(p) {
                    set[p] = 1
                })
            });
        }
        $scope.allPartitions = Object.keys(set);

        filterMetricsPartitions();
        $scope.refreshDisplayedMetrics();
    }, true);

    $scope.uiState = {listMode : 'banner', partitionQuery : ''};
    $scope.displayedMetricByPartitionData = [];

    $scope.orderByFunc = function(metricIdx) {
        if (metricIdx === '__partition__') return Fn.SELF;

        return function(partitionId) {
            return MetricsUtils.getFormattedValueForPartition($scope.displayedMetrics.metrics[metricIdx], partitionId, $scope.displayedMetricByPartitionData);
        }
    };

    $scope.getDisplayedPartitionsData = function(displayedMetric) {
        var metricId = displayedMetric.metric.id;
        var found = null;
        $scope.displayedMetricByPartitionData.forEach(function(displayedMetricPartition) {
            if ( displayedMetricPartition.metricId == metricId ) {
                found = displayedMetricPartition;
            }
        });
        return found;
    };

    // TODO code duplication
    $scope.refreshDisplayedMetrics = function() {
        if ( $scope.metrics == null || $scope.allComputedMetrics == null || $scope.metrics.displayedState == null) return;

        if ( !$scope.displayedMetrics.$loaded && $scope.allComputedMetrics.metrics.length > 0 ) {
            // select back the metrics as the persisted state says
            $scope.displayedMetrics.metrics = $scope.allComputedMetrics.metrics.filter(function(metric) {return metric.displayedAsMetric;});
            // re-order according to $scope.metrics.displayedState.metrics
            $scope.displayedMetrics.metrics.forEach(function(displayedMetric) {
                var i = $scope.metrics.displayedState.metrics.indexOf(displayedMetric.metric.id);
                if ( i < 0 ) {
                    i = $scope.metrics.displayedState.metrics.length;
                }
                displayedMetric.$indexInDisplayedState = i;
            });
            $scope.displayedMetrics.metrics.sort(function(a, b) {return a.$indexInDisplayedState - b.$indexInDisplayedState;});
            $scope.displayedMetrics.$loaded = true;
            refreshDisplayedPartitionData();
        }
    };

    var refreshDisplayedList = function() {
        if ( $scope.displayedMetrics == null || $scope.metrics?.displayedState == null) return;
        $scope.metrics.displayedState.metrics = $scope.displayedMetrics.metrics.map(function(metric) {return metric.metric.id;});
        // don't forget to tweak the allComputedMetrics for when we switch tabs and reload the displayedMetrics list
        $scope.allComputedMetrics.metrics.forEach(function(metric) {metric.displayedAsMetric = $scope.displayedMetrics.metrics.indexOf(metric) >= 0;});
    };

       $scope.$watch('displayedMetrics', function() {
                refreshDisplayedList();
                refreshDisplayedPartitionData();
            }, true);

    var refreshDisplayedPartitionData = function() {
        if ( $scope.displayedMetrics == null || !$scope.displayedMetrics.$loaded || $scope.metrics == null || $scope.metrics.displayedState == null ) {
            return;
        }
        // fetch the data
        $scope.metricsCallbacks.getPreparedMetricPartitions($scope.metrics.displayedState).success(function(data) {
            $scope.displayedMetricByPartitionData = data.metrics;
            //refreshPartitionsRange();
        }).error(setErrorInScope.bind($scope));
    };
    refreshDisplayedPartitionData();

    $scope.$on('metrics-refresh-displayed-data', $scope.refreshAllComputedMetrics);

    var filterMetricsPartitions = function() {
        if (!$scope.allPartitions) $scope.filteredPartitions = [];
        $scope.filteredPartitions = $filter('filter')($scope.allPartitions, $scope.uiState.partitionQuery);
    };

});

})();
