(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LiftChart', LiftChart);

    // (!) This service previously was in static/dataiku/js/simple_report/misc/lift.js
    function LiftChart(ChartManager, ChartDataWrapperFactory, ColorUtils, ChartDataUtils, SVGUtils, ChartAxesUtils, ChartYAxisPosition) {
        return function($container, chartDef, chartHandler, axesDef, data) {

            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef),
                facetLabels = chartData.getAxisLabels('facet') || [null], // We'll through the next loop only once if the chart is not facetted
                xExtent = ChartDataUtils.getMeasureExtent(chartData, 0),
                yExtent = ChartDataUtils.getMeasureExtent(chartData, 1),
                color = ColorUtils.toRgba(chartDef.colorOptions.singleColor, chartDef.colorOptions.transparency);


            xExtent[0] = 0;
            yExtent[0] = 0;

            const lineData = ['ORIGIN'].concat(chartData.getAxisLabels('group'));

            const drawFrame = function(frameIdx, chartBase) {
                chartData.fixAxis('animation', frameIdx);
                facetLabels.forEach(function(facetLabel, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    LiftChartDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', f), chartBase, f);
                });
            };

            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                {
                    x: { type: 'MEASURE', extent: xExtent, measure: chartDef.xMeasure, customExtent: chartDef.xAxisFormatting.customExtent },
                    [yAxisID]: { id: yAxisID, type: 'MEASURE', extent: yExtent, measure: chartDef.yMeasure, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT }
                });

            function LiftChartDrawer(g, chartDef, chartHandler, chartData, chartBase, f) {

                const xCoord = function(d, i) {
                    const v = (d === 'ORIGIN') ? 0 : chartData.aggr(0).get({ group: i - 1 });
                    return chartBase.xAxis.scale()(v);
                };

                const yCoord = function(d, i) {
                    const v = (d === 'ORIGIN') ? 0 : chartData.aggr(1).get({ group: i - 1 });
                    return chartBase.yAxes[0].scale()(v);
                };

                const line = d3.svg.line()
                    .interpolate('monotone')
                    .x(xCoord)
                    .y(yCoord);

                const lineWrapper = g.selectAll('g.wrapper').data([null]);
                const newWrapper = lineWrapper.enter().append('g').attr('class', 'wrapper');

                newWrapper
                    .append('path')
                    .attr('class', 'visible')
                    .attr('stroke', color)
                    .attr('stroke-width', '1.5')
                    .attr('fill', 'none');

                // Thicker invisible line to catch mouseover
                newWrapper
                    .append('path')
                    .attr('stroke-width', '20')
                    .attr('stroke', 'transparent')
                    .attr('fill', 'none');

                const medianLine = lineWrapper.selectAll('line.median').data([null]);
                medianLine.enter().append('line').attr('class', 'median')
                    .attr('x1', xCoord('ORIGIN', 0))
                    .attr('y1', yCoord('ORIGIN', 0))
                    .style('stroke', '#333');


                medianLine.transition()
                    .attr('x2', xCoord(null, lineData.length - 1))
                    .attr('y2', yCoord(null, lineData.length - 1));

                const points = lineWrapper.selectAll('circle.point').data(lineData);

                points.enter().append('circle')
                    .attr('class', 'point')
                    .attr('r', 5)
                    .attr('fill', color)
                    .attr('opacity', 0)
                    .each(function(d, i) {
                        if (i === 0) {
                            return 0;
                        } else {
                            chartBase.tooltips.addTooltipHandlers(this, { group: i - 1, facet: f }, color);
                            chartBase.contextualMenu.addContextualMenuHandler(this, { group: i - 1, facet: f });
                            return 1;
                        }
                    });

                points.transition()
                    .attr('cx', xCoord)
                    .attr('cy', yCoord);

                lineWrapper.selectAll('path').datum(lineData).transition()
                    .attr('d', line);

                lineWrapper.on('mouseover.path', function(d) {
                    lineWrapper.select('path.visible').attr('stroke-width', 3);
                    points.transition(500).attr('opacity', function(d, i) {
                        return i === 0 ? 0 : 1;
                    });
                }).on('mouseout.path', function(d) {
                    lineWrapper.select('path.visible').attr('stroke-width', 1.5);
                    points.transition(250).attr('opacity', 0);
                });

                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, lineWrapper);
            }

            chartHandler.legendsWrapper.deleteLegends();

        };
    }
})();
