(function() {
    'use strict';

    const app = angular.module("dataiku.fm.licensing", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs"]);

    app.controller("LicensingWelcomeController", function($scope, $rootScope, $state, $stateParams, $filter, FMAPI, TaggingService,
                                                          FutureProgressModal, CreateModalFromTemplate, Dialogs, Debounce, WT1,
                                                          ListFilter) {
        let users = [];
        $scope.pagination = new ListFilter.Pagination([], 20);
        $scope.filteredUsers = [];
        $scope.usersPage = [];
        $scope.usersCount = 0;
        $scope.uiState = {
            users: {
                sort: {
                    column: '',
                    order: '',
                    css: {}
                }
            },
            filterLabels: {
                profiles: 'All profiles',
                instances: 'All instances'
            }
        };
        $scope.filters = {
            query: '',
            profiles: [],
            instances: []
        };
        const matrixRights = [
            ["admin", "Global admin"],
            ["mayManageUDM", "Manage user-defined meanings"],
            ["mayCreateProjects", "Create projects"],
            ["mayCreateWorkspaces", "Create workspaces"],
            ["mayShareToWorkspaces", "Publish to workspaces"],
            ["mayCreateCodeEnvs", "Create code envs"],
            ["mayManageCodeEnvs", "Manage all code envs"],
            ["mayCreateClusters", "Create clusters"],
            ["mayManageClusters", "Manage all clusters"],
            ["mayCreateCodeStudioTemplates", "Create Code Studio templates"],
            ["mayManageCodeStudioTemplates", "Manage all Code Studio templates"],
            ["mayDevelopPlugins", "Develop plugins"],
            ["mayEditLibFolders", "Edit lib folders"],
            ["mayCreateAuthenticatedConnections", "Create user connections"],
            ["mayWriteUnsafeCode", "Write unisolated code"],
            ["mayWriteSafeCode", "Write isolated code"],
            ["mayCreatePublishedAPIServices", "Create published API services"],
            ["mayCreatePublishedProjects", "Create published projects"],
            ["mayWriteInRootProjectFolder", "May write in root project folder"],
            ["mayCreateActiveWebContent", "May create active web content"],
            ["mayManageFeatureStore", "Manage Feature Store"],
            ["mayCreateDataCollections", "Create Data Collections"],
            ["mayPublishToDataCollections", "Publish to Data Collections"],
        ];

        const usersSortFns = {
            'login-asc': (a, b) => a.login.localeCompare(b.login),
            'login-desc': (a, b) => b.login.localeCompare(a.login),
            'name-asc': (a, b) => a.displayName.localeCompare(b.displayName),
            'name-desc': (a, b) => b.displayName.localeCompare(a.displayName),
            'activity-asc': (a, b) => b.lastActivityEpoch - a.lastActivityEpoch,
            'activity-desc': (a, b) => a.lastActivityEpoch - b.lastActivityEpoch,
        };

        function sortUsers(){
            const sortFn = usersSortFns[$scope.uiState.users.sort.column + '-' + $scope.uiState.users.sort.order];
            if (sortFn) $scope.filteredUsers.sort(sortFn);
            $scope.pagination.updateAndGetSlice($scope.filteredUsers);
        }

        $scope.setUserSort = function(col, order){
            const usort = $scope.uiState.users.sort;
            if (col === usort.column) {
                if (order) {
                    usort.order = order;
                } else if (usort.order === 'asc') {
                    usort.order = 'desc';
                } else {
                    usort.order = 'asc';
                }
            } else {
                usort.column = col;
                usort.order = order || 'asc';
            }
            usort.css = {[col]: 'sort-' + usort.order +'ending'};
            sortUsers();
        };
        $scope.setUserSort('activity', 'desc');


        function filterUsers(){
            const profiles = $scope.filters.profiles.filter(p => p.selected).map(p => p.label);
            const instances = $scope.filters.instances.filter(p => p.selected).map(p => p.label);
            if (profiles.length > 0 && instances.length > 0) {
                const qry = $scope.filters.query.replace(/\W+/g, ' ').trim();
                const qrex = new RegExp(qry || '.', 'i');
                $scope.filteredUsers = users.filter(user => 
                    user.instances.some(i => 
                        profiles.indexOf(i.profile) !== -1 &&
                        instances.indexOf(i.instanceLabel) !== -1 &&
                        qrex.test(user.login + ' ' + user.displayName)
                    )
                );
            } else {
                $scope.filteredUsers = [];
            }
            $scope.pagination.goToPage(1);
            sortUsers();
        };

        function flattenUsers(){
            //this function generate from $scope.globalLicensingStatus :
            // - a flatten users lists `users` from all instances (each users with it's instances list)
            // - a profiles list: `$scope.filters.profiles` displayed and selectable in the UI
            // - an instances list: `$scope.filters.instances` displayed and selectable in the UI
            //these 3 arrays are used mainly by filterUsers()

            //input structure ($scope.globalLicensingStatus):
            //  globalStatus:
            //    licensedProfiles: {profileKey: profileDetails}
            //    profileLimits: {profileKey: profileLimits}
            //  perInstance: [{
            //    instanceId,
            //    instanceLabel,            
            //    status: {
            //      licensedProfiles: {profileKey: profileDetails}, 
            //      profileLimits: {profileKey: profileLimits}
            //    },
            //    authorizationMatrix: {
            //      perUser: {
            //        authorizationCode1: [booluser0, booluser1, ...],
            //        authorizationCode2: [booluser0, booluser1, ...],
            //        authorizationCode3: [booluser0, booluser1, ...],
            //        ...,
            //        users: [
            //          {displayName, login, profile, lastSessionActivity}
            //        ]
            //      }
            //    }
            //  }, ...]

            //output structure (users)
            // [{
            //   login,
            //   displayName,
            //   lastActivityEpoch,
            //   instances: [{
            //     instanceId,
            //     instanceLabel,
            //     profile,
            //     rights: {authorizationCode1: true, ...}
            //     lastActivity
            //     fallbackFromProfile
            //   }
            // }, ...]

            const usersMap = {}, profiles = {}, instances = [];
            //set global profile limits into global licensed profiles
            const globalStatus = $scope.globalLicensingStatus.globalStatus;
            Object.keys(globalStatus.licensedProfiles).forEach(k => {
                globalStatus.licensedProfiles[k].limits = globalStatus.profileLimits[k];
            });
            //merge instances users into a single flattened list
            $scope.globalLicensingStatus.perInstance.forEach(instance => {
                //set instance profile limits into instance licensed profiles
                Object.keys(instance.status.licensedProfiles).forEach(k => {
                    instance.status.licensedProfiles[k].limits = instance.status.profileLimits[k];
                });

                if (instance.authorizationMatrix) {
                    const mtx = instance.authorizationMatrix.perUser;
                    mtx.users.forEach((user, idx) => {
                        //compute user rights for the current instance
                        const rights = matrixRights.reduce((rights, [rightKey]) => {
                            if (mtx[rightKey][idx]) {
                                rights[rightKey] = true;
                            }
                            return rights;
                        }, {});
                        //compute user last activity on the current instance
                        const lastActivityEpoch = user.lastSessionActivity ? new Date(user.lastSessionActivity).getTime() : 0;
                        const lastActivity = lastActivityEpoch 
                            ? $filter('friendlyTimeDelta')(lastActivityEpoch)
                            : '';
                        const userInstance = {
                            displayName: user.displayName,
                            instanceId: instance.instanceId,
                            instanceLabel: instance.instanceLabel,
                            profile: user.profile,
                            rights: rights,
                            lastActivity: lastActivity
                        };
                        //if the profile is not in the license, we use the fallback
                        if (!(user.profile in globalStatus.licensedProfiles)) {
                            userInstance.profile = globalStatus.fallbackProfile;
                            userInstance.fallbackFromProfile = user.profile;
                        }
                        //
                        let isNewUser = false;
                        if (user.login in usersMap) {
                            //user already known, we just add this instance
                            //and adjust its global lastActivity
                            usersMap[user.login].instances.push(userInstance);
                            if (lastActivityEpoch > usersMap[user.login].lastActivityEpoch) {
                                usersMap[user.login].lastActivityEpoch = lastActivityEpoch;
                            }
                        } else {
                            //new user
                            isNewUser = true;
                            usersMap[user.login] = {
                                login: user.login,
                                instances: [userInstance],
                                lastActivityEpoch: lastActivityEpoch
                            };
                        }
                        //setup profiles filter
                        if (userInstance.profile in profiles) {
                            if (isNewUser) profiles[userInstance.profile].count++;
                        } else {
                            profiles[userInstance.profile] = {
                                count: 1,
                                selected: true,
                                label: userInstance.profile
                            };
                        }
                    });
                    //setup instance filter
                    instances.push({
                        selected: true,
                        label: instance.instanceLabel,
                        id: instance.instanceId
                    });
                }
            });
            users = Object.values(usersMap).map(user => {
                //concat unique display names by user
                user.displayName = user.instances.reduce((names, i) => {
                    if (!names.find(n => n.toLocaleLowerCase() === i.displayName.toLocaleLowerCase())) {
                        names.push(i.displayName);
                    }
                    return names;
                }, []).join(', ');
                return user;
            });
            $scope.usersCount = users.length;
            $scope.filters.profiles = Object.values(profiles);
            $scope.filters.instances = instances;
            $scope.resetFilters();
        }

        $scope.$watch("filters", Debounce().withScope($scope).withDelay(50,200).wrap(filterUsers), true);

        function setFilterLabel(txt, filter) {
            const options = $scope.filters[txt];
            let selected = options.filter(p => p.selected);
            $scope.uiState.filterLabels[txt] =
                selected.length === 0 ? 'No ' + txt.replace(/s$/, '') :
                selected.length === options.length ? 'All ' + txt :
                selected.length === 1 
                    ? (filter ? $filter(filter)(selected[0].label) : selected[0].label)
                    : selected.length + ' ' + txt;
        }

        $scope.toggleFilter = function(option, txt, filter){
            option.selected = !option.selected;
            setFilterLabel(txt, filter);
        };

        $scope.isSelected = function(option, txt, labelKey){
            const options = $scope.filters[txt];
            const o = options.find(o => o.label === option[labelKey]);
            return o ? o.selected : false;
        };

        $scope.selectedInstances = function(instances){
            return instances.filter(instance => $scope.isSelected(instance, 'instances', 'instanceLabel'));
        }

        $scope.setFilters = function(profile, instance){
            $scope.filters.profiles = $scope.filters.profiles.map(p => {
                p.selected = p.label === profile;
                return p;
            });
            setFilterLabel('profiles', 'niceProfileName');
            $scope.filters.instances = $scope.filters.instances.map(i => {
                i.selected = i.label === instance;
                return i;
            });
            setFilterLabel('instances');
        };
        
        $scope.resetFilters = function(){
            $scope.filters.profiles.forEach(p => p.selected = true);
            setFilterLabel('profiles', 'niceProfileName');
            $scope.filters.instances.forEach(i => i.selected = true);
            setFilterLabel('instances');
            $scope.filters.query = '';
            filterUsers();
        };

        $scope.refreshSummary = function() {
            FMAPI.tenant.getGlobalLicensingSummary().success(function(data){
                $scope.globalLicensingStatus = data;
            }).error(setErrorInScope.bind($scope));;
        }

        $scope.startFetchLicensingStatus = function() {
            WT1.event("fm-fetch-licensing-status", {});
    
            FMAPI.tenant.getGlobalLicensingStatus().success(function(response){
                FutureProgressModal.show($scope, response, "Fetching licensing usage").then((result) => {
                    $scope.globalLicensingStatus = result;
                    flattenUsers();
                });
            }).error(setErrorInScope.bind($scope));
        }

        $scope.createSublicense = function() {
            $state.go("licensing.sublicenseCreate");
        }

        $scope.deleteSublicense = function(sublicenseId) {
            if ($scope.isDeletableSublicense(sublicenseId)) {
                WT1.event("fm-sub-licensing-delete", {});
                FMAPI.tenant.sublicenses.delete(sublicenseId).success(function(){
                    $scope.refreshSummary();
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.isDeletableSublicense = function(sublicenseId) {
            for (const instance of $scope.globalLicensingStatus.perInstance) {
                if (instance.usesSublicense && instance.sublicenseId == sublicenseId) {
                    return false;
                }
            }
            return true;
        };

        function getInstanceLabel(id){
            const instance = $scope.globalLicensingStatus.perInstance.find(i => i.instanceId === id);
            return instance ? instance.instanceLabel : id;
        }

        $scope.showUserDetails = function(user, instance){
            CreateModalFromTemplate("/app/licensing/dialogs/user-modal.html", $scope, null, function(newScope) {
                newScope.user = user;
                newScope.instance = instance;
                newScope.matrixRights = matrixRights;
                newScope.disableUser = () => newScope.resolveModal();
            }).then(() => {
                Dialogs.confirm(
                    $scope, 
                    'Disable user account', 
                    `<div>Are you sure you want disable <strong>${user.displayName}</strong>'s account on instances:</div>
                     <ul>${user.instances.map(i => `<li>${i.instanceLabel}</li>`).join('')}</ul>`
                ).then(() => {
                    FMAPI.instances.disableUser(user.login, user.instances.map(i => i.instanceId)).success(function(response){
                        FutureProgressModal.show($scope, response, "Disabling user account").then((result) => {
                            const instanceIdsErrors = Object.keys(result.perInstanceResults).filter(id => !result.perInstanceResults[id].success);
                            if (instanceIdsErrors.length) {
                                let msg = `<div>this operation has failed on ${instanceIdsErrors.length > 1 ? 'these instances' : 'this instance'}:</div>`;
                                msg += '<ul>';
                                msg += instanceIdsErrors.map(id => 
                                    `<li><strong>${getInstanceLabel(id)}</strong>: ${result.perInstanceResults[id].error.detailedMessage}</li>`
                                ).join('');
                                msg += '</ul>';
                                Dialogs.error($scope, `Disable ${user.displayName}'s account`, msg);
                            }
                            $scope.startFetchLicensingStatus();
                        }).error(setErrorInScope.bind($scope));
                    }).error(setErrorInScope.bind($scope));
                });
            });
        };

        $scope.refreshSummary();
    });

    app.component("sublicenseEditor", {
        templateUrl: 'app/licensing/sublicense-editor.html',
        bindings: {
            model: '=ngModel',
            sublicense: '<subLicense',
            globalLicensingStatus: '<'
        },
        controller: ["$scope", function($scope) {
            const $ctrl = this;
            $ctrl.stats = {};
            $ctrl.profiles = [];
            
            function computeStats(){
                $ctrl.stats = {};
                if ($ctrl.globalLicensingStatus && $ctrl.globalLicensingStatus.globalStatus) {
                    for (const profile in $ctrl.globalLicensingStatus.globalStatus.licensedProfiles) {
                        const allocated = 
                            $ctrl.globalLicensingStatus.sublicenses
                                .filter(sl => !$ctrl.sublicense || $ctrl.sublicense.id !== sl.id)
                                .map(sl => profile in sl.sublicense.profileLimits ? sl.sublicense.profileLimits[profile] : 0)
                                .reduce((a, b) => a + b, 0);
                        const limit = parseInt($ctrl.globalLicensingStatus.globalStatus.licensedProfiles[profile].licensedLimit, 10);
                        $ctrl.stats[profile] = {
                            limit: limit,
                            available: limit === -1 ? Number.MAX_SAFE_INTEGER : limit - allocated
                        };
                    }
                }
                $ctrl.profiles = Object.keys($ctrl.stats);
            }
            $scope.$watch('$ctrl.sublicense', computeStats);
            $scope.$watch('$ctrl.globalLicensingStatus', computeStats);

            $scope.getUnusedProfiles = (actualProfile) =>
                $ctrl.profiles.filter(p => p === actualProfile || !$ctrl.model.profileLimits.find(up => up.profile === p));

            $scope.getLicenseLimit = (profileName) =>
                $ctrl.stats[profileName] ? $ctrl.stats[profileName].limit : -1;

            $scope.getLicenseAvailable = (profileName) =>
                $ctrl.stats[profileName] ? $ctrl.stats[profileName].available : Number.MAX_SAFE_INTEGER;
        }]
    });

    //license with limits array to limits object
    function _computeSubLicenseObject(sl) {
        const sublicense = {
            label: sl.label,
            sublicense: {
                profileLimits: {}
            }
        };
        if (sl.id) sublicense.id = sl.id;
        for (const profileLimit of sl.profileLimits) {
            if (profileLimit.profile && profileLimit.sublicenseLimit)
                sublicense.sublicense.profileLimits[profileLimit.profile] = profileLimit.sublicenseLimit;
        }
        return sublicense;
    }

    //check label & all limits are properly defined
    function _isSubLicenseValid(sl) {
        return sl 
            && sl.label 
            && sl.label.trim().length > 0 
            && sl.profileLimits.length > 0
            && !sl.profileLimits.find(pl => !pl.profile || !pl.sublicenseLimit);
    }

    app.controller("SublicenseCreationController", function($scope, $state, FMAPI, WT1) {
        $scope.newProtoSL = {
            sublicense: {},
            profileLimits : []
        };

        $scope.create = function() {
            WT1.event("fm-sublicense-create", {});
    
            // Do the actual job
            const sublicense = _computeSubLicenseObject($scope.newProtoSL);
            FMAPI.tenant.sublicenses.create(sublicense).success(function(response) {
                $scope.newProtoSL = null;
                $state.go("licensing.sublicense.summary", {sublicenseId: response.id});
            }).error(setErrorInScope.bind($scope));
        };

        $scope.refresh = function() {
            FMAPI.tenant.getGlobalLicensingSummary().success(function(data) {
                $scope.globalLicensingStatus = data;
                $scope.profiles = Object.keys($scope.globalLicensingStatus.globalStatus.licensedProfiles);
            }).error(setErrorInScope.bind($scope));
        };
        
        $scope.isValid = () => _isSubLicenseValid($scope.newProtoSL);

        checkChangesBeforeLeaving($scope, () => $scope.newProtoSL && $scope.isValid());

        // Initialize
        $scope.refresh();
    });

    app.controller("SublicenseBaseController", function($scope, $state, $stateParams, FMAPI, ActivityIndicator, $q) {
        $scope.sublicenseFormState = { valid: false, dirty: false };

        $scope.refresh = function() {
            $q.all([
                FMAPI.tenant.sublicenses.get($stateParams.sublicenseId),
                FMAPI.tenant.getGlobalLicensingSummary()
            ]).then(([sublicense, globalLicensingStatus]) => {
                $scope.sublicense = sublicense.data;
                $scope.globalLicensingStatus = globalLicensingStatus.data;
                // eslint-disable-next-line no-undef
                $scope.editedSL = dkuDeepCopy(sublicense.data, filterDollarKey);
                $scope.editedSL.profileLimits = [];
                for (const profile in $scope.editedSL.sublicense.profileLimits) {
                    $scope.editedSL.profileLimits.push({
                        "profile": profile,
                        "sublicenseLimit": parseInt($scope.editedSL.sublicense.profileLimits[profile])
                    });
                }
                $scope.originSL = angular.copy($scope.editedSL);
            }, setErrorInScope.bind($scope));
        };

        $scope.save = function() {
            const sublicense = _computeSubLicenseObject($scope.editedSL);
            FMAPI.tenant.sublicenses.save(sublicense).success(function() {
                ActivityIndicator.success("Saved");
                $scope.sublicenseFormState.dirty = false;
                $scope.refresh();
                $state.go("licensing.sublicense.summary");
            }).error(setErrorInScope.bind($scope));
        };

        $scope.$watch("editedSL", (nv, ov) => {
            $scope.sublicenseFormState.dirty = ov ? !angular.equals($scope.editedSL, $scope.originSL) : false;
            $scope.sublicenseFormState.valid = _isSubLicenseValid($scope.editedSL);
        }, true);

        checkChangesBeforeLeaving($scope, () => $state.current.name === 'licensing.sublicense.settings' && $scope.sublicenseFormState.dirty);

        $scope.refresh();
    });
}());
