(function () {
    "use strict";

    const app = angular.module('dataiku.shaker');


    app.directive('datasetChartsBase', function ($rootScope, Assert, ChartTypeChangeHandler, Logger, CreateModalFromTemplate,
        DatasetUtils, WT1, TopNav, DataikuAPI, $timeout, ActivityIndicator, $state, $stateParams, $q, DatasetChartsUtils, ChartSetErrorInScope, ChartCustomMeasures, ChartsContext, DatasetErrorCta, ATSurveyService, DSSVisualizationThemeUtils, DefaultDSSVisualizationTheme, ChartNavigationService) {
        return {
            priority: 2,
            scope: true,
            controller: function ($scope, $stateParams) {
                ChartSetErrorInScope.defineInScope($scope);
                $scope.onLoad = function (projectKey, datasetName, contextProjectKey, datasetSmartName) {
                    //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)
                    $scope.errorCTA = {};

                    $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);

                    $scope.$watch("datasetFullInfo", () => $scope.updateUiState($scope.errorCTA.error), true);
                    $scope.$watch("errorCTA", () => $scope.updateUiState($scope.errorCTA.error), true);

                    /* ********************* Execute Callbacks for chartsCommon ******************* */

                    $scope.getDataSpec = function () {
                        var currentChart = $scope.charts[$scope.currentChart.index];
                        Assert.trueish(currentChart, 'no currentChart');

                        var dataSpec = {
                            datasetProjectKey: projectKey,
                            datasetName: datasetName,
                            script: angular.copy($scope.shaker),
                            copySelectionFromScript: currentChart.copySelectionFromScript,
                            sampleSettings: currentChart.refreshableSelection,
                            engineType: currentChart.engineType
                        };
                        dataSpec.script.origin = "DATASET_EXPLORE";
                        return dataSpec;
                    }

                    $scope.getExecutePromise = function (request, saveShaker = true, noSpinner = false, requiredSampleId = undefined, dataSpec = $scope.getDataSpec()) {
                        var currentChart = $scope.charts[$scope.currentChart.index];
                        Assert.trueish(currentChart.summary, "Current chart summary is not ready");
                        if (requiredSampleId === undefined) {
                            requiredSampleId = currentChart.summary.requiredSampleId;
                        }
                        (saveShaker !== false) && $scope.saveShaker();
                        if (request) {
                            let promise = DataikuAPI.shakers.charts.getPivotResponse(
                                contextProjectKey ? contextProjectKey : projectKey,
                                dataSpec,
                                request,
                                requiredSampleId
                            );

                            if (noSpinner === true) {
                                promise = promise.noSpinner();
                            }

                            return promise;
                        }
                    };

                    $rootScope.$on("chartSamplingChanged", function (event, opts) {
                        if (angular.equals($scope.charts[$scope.currentChart.index], opts.chart)) {
                            $scope.clearCachedSummaries();
                            $scope.fetchColumnsSummaryForCurrentChart().then(function () {
                                Logger.info("Sample reloaded, executing chart");
                                $scope.$broadcast("forceExecuteChart");
                            });
                        }
                    });

                    $scope.getDefaultNewChart = function () {
                        var newChart = null;
                        const defaultTheme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
                        if ($scope.charts.length > 0) {
                            // Copy to retrieve the same sample, copySample and engine settings
                            newChart = angular.copy($scope.charts[$scope.charts.length - 1]);
                            newChart.theme = defaultTheme;
                            newChart.def = ChartTypeChangeHandler.defaultNewChart(newChart.theme);
                        } else {
                            newChart = {
                                def: ChartTypeChangeHandler.defaultNewChart(defaultTheme),
                                copySelectionFromScript: true,
                                engineType: "LINO",
                                theme: defaultTheme
                            }
                        }
                        return newChart;
                    }

                    function exploreIsDirty(ignoreThumbnailChanges) {
                        try {
                            var savedExplore2 = angular.copy(savedExplore);
                            var explore = angular.copy($scope.explore);

                            if (ignoreThumbnailChanges) {
                                if (explore) {
                                    explore.charts.forEach(function (chart) {
                                        chart.def.thumbnailData = null;
                                    });
                                }
                                if (savedExplore2) {
                                    savedExplore2.charts.forEach(function (chart) {
                                        chart.def.thumbnailData = null;
                                    });
                                }
                            }
                            return !angular.equals(explore, savedExplore2);
                        } catch (e) {
                            Logger.error(e);
                            return true;
                        }
                    }

                    $scope.saveShaker = function (isThumbnail = false) {
                        Logger.info("Saving shaker");
                        var ignoreThumbnailChanges = !$scope.isProjectAnalystRW();
                        if (!exploreIsDirty(ignoreThumbnailChanges)) {
                            Logger.info("No changes: don't save explore");
                            return;
                        }

                        if ($scope.isProjectAnalystRW()) {
                            DataikuAPI.explores.save(contextProjectKey, datasetSmartName, $scope.explore)
                            .success(function (data) {
                                if (!isThumbnail && !ActivityIndicator.isDisplayed()) {
                                    ActivityIndicator.success("Charts saved");
                                }
                            }).error(setErrorInScope.bind($scope)).noSpinner();
                        } else {
                            ActivityIndicator.warning("You don't have write access - not saving");
                        }
                    };

                    $scope.saveChart = $scope.saveShaker;

                    /* ********************* Load callback ******************* */

                    var cachedColumnSummaries = {};

                    $scope.clearCachedSummaries = function () {
                        $scope.charts.forEach(function (x) {
                            x.summary = null;
                        });
                        cachedColumnSummaries = {};
                    }

                    $scope.fetchColumnsSummaryForCurrentChart = function (forceRefresh) {
                        var currentChart = $scope.charts[$scope.currentChart.index];
                        var dataSpec = $scope.getDataSpec();
                        var cacheKey = JSON.stringify(dataSpec).dkuHashCode();

                        var promise = null;
                        if (cachedColumnSummaries[cacheKey] != null && !forceRefresh) {
                            Logger.info("Already cached for", dataSpec);
                            promise = $q.when(cachedColumnSummaries[cacheKey]);
                        } else {
                            Logger.info("No cache for", dataSpec);
                            promise = DataikuAPI.shakers.charts.getColumnsSummary(contextProjectKey ? contextProjectKey : projectKey, dataSpec)
                                .error($scope.chartSetErrorInScope)
                                .then(function (response) {
                                    cachedColumnSummaries[cacheKey] = response.data;
                                    return response.data;
                                })
                        }

                        return promise.then(
                            function (data) {
                                currentChart.summary = data;
                                $scope.makeUsableColumns(data);
                                if ($scope.errorCTA) {
                                    $scope.errorCTA.error = null;
                                }
                            },
                            function (attr) {
                                if ($scope.errorCTA) {
                                    $scope.errorCTA.error = getErrorDetails(attr.data, attr.status, attr.headers, attr.statusText);
                                }
                            }
                        );
                    };

                    $scope.overrideFormattingWithTheme = function (theme) {
                        const currentChart = $scope.charts[$scope.currentChart.index];
                        const currentChartCopy = angular.copy(currentChart);
                        const formerTheme = currentChart.theme;
                        currentChart.theme = theme;

                        DSSVisualizationThemeUtils.applyToChart({ chart: currentChart.def, theme, formerTheme });
                        DSSVisualizationThemeUtils.showThemeAppliedSnackbar(currentChart, currentChartCopy);
                    };

                    $scope.$on('$destroy', () => DSSVisualizationThemeUtils.hideThemeAppliedSnackbar());

                    $scope.createAndPinInsight = function () {
                        const insights = $scope.charts.map(chart => {
                            let insight = {
                                type: 'chart',
                                projectKey: contextProjectKey,
                                name: chart.def.name + ' on ' + datasetSmartName,
                                params: {
                                    datasetSmartName: datasetSmartName,
                                    engineType: chart.engineType,
                                    refreshableSelection: chart.refreshableSelection,
                                    copySelectionFromScript: chart.copySelectionFromScript,
                                    def: chart.def,
                                    summary: chart.summary,
                                    theme: chart.theme || DefaultDSSVisualizationTheme,
                                    customMeasures: $scope.explore.customMeasures,
                                    reusableDimensions: $scope.explore.reusableDimensions
                                }
                            };
                            if (insight.params.refreshableSelection == null) {
                                insight.params.refreshableSelection = DatasetChartsUtils.makeSelectionFromScript($scope.shaker);
                            }

                            return insight;
                        });

                        CreateModalFromTemplate("/templates/dashboards/insights/create-and-pin-insights-modal.html", $scope, "CreateAndPinInsightsModalController", function (newScope) {
                            let selectedCharts = angular.copy($scope.charts);
                            selectedCharts.forEach(_ => _.selected = false);
                            selectedCharts[$scope.currentChart.index].selected = true;

                            newScope.insightData = {
                                items: selectedCharts,
                                type: 'chart'
                            }

                            newScope.init(insights);
                        });
                    };

                    /* ********************* Main ******************* */

                    var savedExplore;
                    var main = function () {
                        WT1.event("dataset-charts-open");
                        // Call ATSurveyService to trigger survey if needed
                        ATSurveyService.updateCounter('ChartsOpen');
                        TopNav.setLocation(TopNav.TOP_FLOW, 'datasets', TopNav.TABS_DATASET, "visualize");

                        ChartNavigationService.bindCurrentChartWithUrl();

                        $scope.$on('$destroy', () => {
                            ChartNavigationService.unbindCurrentChartWithUrl()
                        });

                        $scope.$watchGroup(
                            ChartNavigationService.getCurrentChartWatchers($scope),
                            function () {
                                ChartNavigationService.updateCurrentChart($scope.charts, $scope.currentChart.index)
                            }
                        );

                        DataikuAPI.explores.get(contextProjectKey, datasetSmartName).success(function (data) {
                            $scope.explore = data;
                            $scope.shaker = data.script;
                            $scope.charts = data.charts;
                            $scope.currentChart.index = ChartNavigationService.getChartIndexFromId($scope.charts.map(chart => chart.def), $stateParams.chartId);
                            savedExplore = angular.copy($scope.savedExplore);

                            DataikuAPI.datasets.get(projectKey, datasetName, $stateParams.projectKey).success(function (data) {
                                $scope.dataset = data;
                            }).error(setErrorInScope.bind($scope));

                            if ($scope.charts.length == 0) {
                                $scope.addChart({ datasetName });
                            }

                            $scope.addCustomMeasuresToScopeAndCache($scope.explore.customMeasures);
                            $scope.addBinnedDimensionToScopeAndCache($scope.explore.reusableDimensions);


                            Logger.info("Explore loaded, get summary");

                            $scope.$watch("charts[currentChart.index]", function (nv) {
                                Logger.info("Chart changed, fetching summary and executing");
                                if (nv) {
                                    $scope.fetchColumnsSummaryForCurrentChart()
                                        .then(function () {
                                            // Fixes a race condition that used to happen sometimes when explores.get returned before the
                                            // event listeners in chart_logic.js were properly set up, causing the forceExecuteChart to be missed
                                            // and nothing to be drawn.
                                            $scope.forceExecuteChartOrWait();
                                        })
                                }
                            });
                            if ($scope.errorCTA) {
                                $scope.errorCTA.error = null;
                            }
                        }).error(function (data, status, headers, config, statusText, xhrStatus) {
                            setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                            if ($scope.errorCTA) {
                                $scope.errorCTA.error = getErrorDetails(data, status, headers, statusText);
                            }
                        });
                    };

                    main();
                };
            }
        }
    });

    app.directive('datasetCharts', function () {
        return {
            scope: true,
            controller: function ($scope, $stateParams) {
                $scope.onLoad($stateParams.projectKey, $stateParams.datasetName, $stateParams.projectKey, $stateParams.datasetName);
            }
        }
    });

    app.directive('foreignDatasetCharts', function (Logger, DatasetUtils) {
        return {
            scope: true,
            controller: function ($scope, $stateParams) {
                var loc = DatasetUtils.getLocFromFull($stateParams.datasetFullName);
                $scope.onLoad(loc.projectKey, loc.name, $stateParams.projectKey, $stateParams.datasetFullName);
            }
        }
    });

    app.directive("datasetChartSamplingEditor", function (DatasetUtils, ChartFeatures, $rootScope, WT1, translate) {
        return {
            scope: {
                dataset: '=',
                chart: '=',
                script: '=',
                canCopySelectionFromScript: '=',
                useUpperButtons: '<?',
                initialChart: '=?', // reference data to check dirty state
            },
            templateUrl: "/templates/simple_report/dataset-chart-sampling-editor.html",
            controller: function ($scope, $controller) {
                $controller("_ChartOnDatasetSamplingEditorBase", { $scope: $scope });

                function makeEnginesStatus(script, chartSpec) {
                    var engines = [
                        ["LINO", $rootScope.wl.productShortName, true, ""]
                    ]
                    var sqlEngine = [
                        "SQL",
                        translate("CHARTS.SAMPLING.IN_DATABASE", "In-database"),
                        false,
                        ""
                    ];
                    if (!DatasetUtils.canUseSQL($scope.dataset)) {
                        sqlEngine[3] = translate("CHARTS.SAMPLING.DATASET_IS_NOT_SQL", "Dataset is not SQL");
                    } else if (script != null && script.steps.length) {
                        sqlEngine[3] = translate("CHARTS.SAMPLING.SCRIPT_CONTAINS_STEPS", "Script contains steps");
                    } else if (!ChartFeatures.canUseSQL(chartSpec)) {
                        sqlEngine[3] = translate("CHARTS.SAMPLING.CHART_INCOMPATIBLE_WITH_IN_DATABASE", "This chart is not compatible with in-database");
                    } else {
                        sqlEngine[2] = true;
                    }
                    engines.push(sqlEngine);
                    if ($rootScope.appConfig.interactiveSparkEngine != null) {
                        var sparksqlEngine = ["SPARKSQL", "SparkSQL", false, ""];
                        if (!DatasetUtils.canUseSparkSQL($scope.dataset)) {
                            sqlEngine[3] = translate("CHARTS.SAMPLING.DATASET_IS_SQL", "Dataset is SQL, use in-database engine");
                        } else if (script != null && script.steps.length) {
                            sparksqlEngine[3] = translate("CHARTS.SAMPLING.SCRIPT_CONTAINS_STEPS", "Script contains steps");
                        } else if (!ChartFeatures.canUseSQL(chartSpec)) {
                            sparksqlEngine[3] = translate("CHARTS.SAMPLING.CHART_INCOMPATIBLE_WITH_SPARK", "This chart is not compatible with SparkSQL");
                        } else {
                            sparksqlEngine[2] = true;
                        }
                        engines.push(sparksqlEngine);
                    }
                    return engines;
                }

                const useApplyCancelButtons = $scope.useUpperButtons && $scope.initialChart;

                $scope.objectName = translate("CHARTS.SAMPLING.OBJECT_NAME", "charts");

                $scope.$watch("chart", function () {
                    $scope.availableEngines = makeEnginesStatus($scope.script, $scope.chart.def);
                });

                /* Auto-revert to compatible settings */
                $scope.$watch("chart.engineType", function (nv, ov) {
                    if (!nv || !ov) return;

                    if ((nv == "SQL" || nv == "SPARKSQL") && !$scope.chart.refreshableSelection) {
                        $scope.chart.refreshableSelection = {
                            selection: {
                                samplingMethod: "FULL",
                                partitionSelectionMethod: "ALL"
                            }
                        }
                    }
                });

                $scope.save = function () {
                    if ($scope.chart.refreshableSelection != null) {
                        if (!$scope.chart.copySelectionFromScript) {
                            WT1.event("chart-sample-setting-update", {
                                chartId: `${$scope.dataset.projectKey.dkuHashCode()}.${$scope.dataset.name.dkuHashCode()}.${$scope.chart.def.name.dkuHashCode()}`,
                                analysis: false,
                                engineType: $scope.chart.engineType,
                                reuseFromExplore: false,
                                samplingMethod: $scope.chart.refreshableSelection.selection.samplingMethod,
                                recordsNumber: $scope.chart.refreshableSelection.selection.maxRecords,
                                targetRatio: $scope.chart.refreshableSelection.selection.targetRatio,
                                filtersNumber:
                                    $scope.chart.refreshableSelection.selection.filter &&
                                        $scope.chart.refreshableSelection.selection.filter.enabled &&
                                        $scope.chart.refreshableSelection.selection.filter.uiData &&
                                        $scope.chart.refreshableSelection.selection.filter.uiData.conditions
                                        ? $scope.chart.refreshableSelection.selection.filter.uiData.conditions.length
                                        : 0
                            });
                        } else {
                            WT1.event('chart-sample-setting-update', {
                                chartId: `${$scope.dataset.projectKey.dkuHashCode()}.${$scope.dataset.name.dkuHashCode()}.${$scope.chart.def.name.dkuHashCode()}`,
                                analysis: false,
                                engineType: $scope.chart.engineType,
                                reuseFromExplore: true
                            });
                        }
                        $scope.chart.refreshableSelection._refreshTrigger = new Date().getTime();
                    }

                    $rootScope.$emit("chartSamplingChanged", { chart: $scope.chart });

                    if (useApplyCancelButtons) {
                        $scope.initialChart = angular.copy($scope.chart); // new reference
                        $scope.samplingForm.$setPristine();
                    }
                };

                $scope.saveNoRefresh = function () {
                    $rootScope.$emit("chartSamplingChanged", { chart: $scope.chart });
                };

                if (useApplyCancelButtons) {

                    $scope.isDirty = function () {
                        return !angular.equals($scope.chart, $scope.initialChart);
                    };

                    $scope.cancel = function () {
                        $scope.chart = angular.copy($scope.initialChart);
                        $scope.samplingForm.$setPristine();
                    };
                }

            }
        }
    });

})();
