(function() {
'use strict';

var app = angular.module('dataiku.datasets');

app.controller("BaseSQLDatasetController", function($scope, $stateParams, $controller, DataikuAPI, withConsistency, SqlConnectionNamespaceService,
        DatasetTypesService) {

    $controller("BaseRowDatasetController", {$scope: $scope, withConsistency: withConsistency});

    $scope.partitionsList = {};

    $scope.isCatalogAware = function() {
        return $scope.dataset.type && ['Databricks', 'BigQuery', 'Snowflake', 'SQLServer', 'JDBC', 'Trino', 'FabricWarehouse'].includes($scope.dataset.type);
    };

    $scope.isSchemaAware = function() {
        return $scope.dataset.type && $scope.dataset.type != 'MySQL' && $scope.dataset.type != 'hiveserver2';
    };

    $scope.hooks = {};

    $scope.setConnections = function(data) {
        $scope.connections = data;
        if (!$scope.dataset.params.connection && data.length) {
            $scope.dataset.params.connection = data[0];
            $scope.test(false, false);
        }
    };

    $scope.hooks.getConnectionNames = function() {
        return DataikuAPI.connections.getNames($scope.dataset.type).success(function (data) {
            $scope.setConnections(data);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.$watch('dataset.type', function() {
        $scope.hooks.getConnectionNames();
    });

    function setAssumedTimezone(params) {
        $scope.dataset.params.assumedTzForUnknownTz = params.assumedTzForUnknownTz;
        $scope.dataset.params.assumedDbTzForUnknownTz = params.assumedDbTzForUnknownTz;
    }

    let initialPhase = true;
    $scope.$watch('dataset.params.connection', function(connection) {
        if (!initialPhase && connection) {
            // User is changing the connection, let's match the default timezone of the new connection
            DataikuAPI.connections.getDefaultTimezone(connection).success(function(defaultTimezone) {
                setAssumedTimezone({
                    assumedTzForUnknownTz: defaultTimezone.defaultAssumedTzForUnknownTz,
                    assumedDbTzForUnknownTz: defaultTimezone.defaultAssumedDbTzForUnknownTz
                });
            }).error(setErrorInScope.bind($scope));
        } else {
            // Initial phase, we just want to keep the default value
            initialPhase = false;
        }

        SqlConnectionNamespaceService.setTooltips($scope, $scope.dataset.type);
        SqlConnectionNamespaceService.resetState($scope, $scope);
    });

    $scope.supportsNativePartitioning = function () {
        return $scope.dataset !== null && $scope.dataset.type == 'Vertica';
    };
    $scope.listPartitions = function () {
        $scope.partitionsList.list = null;
        $scope.partitionsList.error = null;
        DataikuAPI.datasets.externalSQL.listPartitions($stateParams.projectKey, $scope.dataset).success(function (data) {
            $scope.partitionsList.list = data;
            if (data.length > 0 && $scope.dataset.params.previewPartition == null) {
                $scope.dataset.params.previewPartition = data[0];
            }
        }).error(function (data, status, error) {
            var err = getErrorDetails(data, status, error);
            $scope.partitionsList.errorMsg = getErrorDetails(data, status, error).detailedMessage;
            $scope.partitionsList.error = getErrorDetails(data, status, error);
        });
    };

    $scope.uiState = { codeSamplesSelectorVisible: false };

    /* Need to manually refresh code mirror on show.
     * as it will compute various dimensions and set them.
     *
     * Here we do refresh on any change of mode and partitioned,
     * to handle the two possible path to make the hidden codemirror
     * box to appear.
     */
    var refreshCodeMirrors = function() {
        $('.CodeMirror').each(function(i, el){
            if (el.CodeMirror != undefined) {
                setTimeout(function() {el.CodeMirror.refresh();}, 0);
            }
        });
    };
    $scope.isPartitioned = function() {
        return $scope.dataset && $scope.dataset.partitioning && $scope.dataset.partitioning.dimensions && $scope.dataset.partitioning.dimensions.length > 0;
    };
    // we resolved the "forbidPartitionsWriteToNonPartitionedTable" setting, if it is set to "INHERIT" then the setting at the connection level needs to be checked
    $scope.isPartitioningConsistencyEnabled = function() {
        return (
            $scope.dataset.params.forbidPartitionsWriteToNonPartitionedTable === "ENABLED" ||
            ($scope.dataset.params.forbidPartitionsWriteToNonPartitionedTable === "INHERIT" &&
                $scope.datasetFullInfo &&
                $scope.datasetFullInfo.connectionForbidPartitionsWriteToNonPartitionedTable)
        );
    };
    $scope.$watch("dataset.params.partitioned + dataset.params.mode", function() {
        if ($scope.isPartitioned()) {
            refreshCodeMirrors();
        }
    });

    $scope.setPartitioned = function(activate) {
        $scope.dataset.partitioning = $scope.dataset.partitioning || {};
        if (activate) {
            $scope.dataset.partitioning.dimensions = $scope.dataset.partitioning.dimensions || [];
            $scope.addDimension();
        } else {
            $scope.dataset.partitioning.dimensions = [];
        }
    };

    $scope.createBigQueryDatePartitioningPeriods = function(partitioningColumn) {
        if (partitioningColumn && DatasetTypesService.isTemporalType(partitioningColumn.type)) {
            const result = [ [ "YEAR", "By year" ], [ "MONTH", "By month" ], [ "DAY", "By day" ] ];
            if (!DatasetTypesService.isDateOnly(partitioningColumn.type)) {
                result.push([ "HOUR", "By hour" ]);
            }
            return result;
        } else {
            return [];
        }
    };

    $scope.$watch("dataset.params.bigQueryPartitioningField", function(newValue) {
        const partitioningColumn = $scope.dataset.schema.columns.find(c => c.name === newValue);
        $scope.bigQueryDatePartitioningPeriods = $scope.createBigQueryDatePartitioningPeriods(partitioningColumn);
    });

    $scope.shouldDisplayBigQueryPartitionMismatchMessage = function() {
        if (!$scope.dataset.managed || $scope.dataset.type !== 'BigQuery' || !$scope.isPartitioned()) {
            return false;
        }
        // BigQuery table partitioning is not set
        if (!$scope.dataset.params.useBigQueryPartitioning) {
            return true;
        }
        // dataset partitioning column and BigQuery table partitioning column do not match
        if ($scope.dataset.params.bigQueryPartitioningField !== $scope.dataset.partitioning.dimensions[0].name) {
            return true;
        }
        // dataset is partitioned by discrete dimension but BigQuery table partitioning is not configured by discrete
        if ($scope.dataset.params.bigQueryPartitioningType === 'RANGE' && $scope.dataset.partitioning.dimensions[0].type !== 'value') {
            return true;
        }
        // dataset is partitioned by date but BigQuery table partitioning is not configured by date
        if ($scope.dataset.params.bigQueryPartitioningType === 'DATE' && $scope.dataset.partitioning.dimensions[0].type !== 'time') {
            return true;
        }
        // both dataset and BigQuery table are partitioned by date but the dataset dimension period and the BigQuery table partitioning period do not match
        if (
            $scope.dataset.params.bigQueryPartitioningType === 'DATE' &&
            $scope.dataset.partitioning.dimensions[0].params?.period !== $scope.dataset.params.bigQueryPartitioningPeriod
        ) {
            return true;
        }
        return false;
    }

    $scope.removeDimension = function(index) {
        $scope.dataset.partitioning.dimensions.splice(index, 1);
    };

    $scope.addDimension = function() {
        $scope.dataset.partitioning.dimensions.push({
            name : 'dimension_' + $scope.dataset.partitioning.dimensions.length,
            type : 'value',
            params : {'period': $scope.timeDimensionPeriods[0] }
        });
    };

    $scope.fetchCatalogs = function(connectionName, origin, connectionType) {
        SqlConnectionNamespaceService.listSqlCatalogs(connectionName, $scope, origin, connectionType);
    };

    $scope.fetchSchemas = function(connectionName, origin, connectionType) {
        SqlConnectionNamespaceService.listSqlSchemas(connectionName, $scope, $scope.dataset.params.catalog, origin, connectionType);
    };

    $scope.$on('$destroy', destroy);

    function destroy () {
        SqlConnectionNamespaceService.abortListSqlSchemas($scope);
        SqlConnectionNamespaceService.abortListSqlCatalogs($scope);
    }
});


app.controller("ExternalSQLDatasetController", function($scope, $stateParams, $controller, LoggerProvider, 
    DataikuAPI, DatasetUtils, Dialogs, ActivityIndicator, $state) {
    $controller("BaseSQLDatasetController", {$scope: $scope, withConsistency: true});

    var Logger = LoggerProvider.getLogger('datasets.sql');
    $scope.expandedDatasetParams = {}

    $scope.overwriteSchemaFromTable = function () {
        $scope.dataset.schema = {
            'userModified': false,
            'columns': $scope.testResult.querySchema.columns
        };
    };

    $scope.trackByCatalogSchemaAndTableFn = function(table) {
        return table ? table.catalog + '.' + table.schema + '.' + table.table : '';
    }

    $scope.resetTableSelection = function () {
        if ($scope.testResult) {
            $scope.testResult.tablesList = null;
            $scope.dataset.params.table = null;
            $scope.dataset.params.schema = null;
            $scope.dataset.params.catalog = null;
        }
    };

    $scope.onLoadComplete = function () {
        if (angular.isUndefined($scope.dataset.params.mode)) {
            $scope.dataset.params.mode = 'table';
        }
        if (angular.isUndefined($scope.dataset.params.partitioningType)) {
            $scope.dataset.params.partitioningType = 'custom';
        }
        if (angular.isUndefined($scope.dataset.params.normalizeDoubles)) {
            $scope.dataset.params.normalizeDoubles = true;
        }
        if (angular.isUndefined($scope.dataset.params.dateonlyReadMode)) {
            $scope.dataset.params.dateonlyReadMode = $scope.appConfig.defaultDateonlyReadMode;
        }
        if (angular.isUndefined($scope.dataset.params.datetimenotzReadMode)) {
            $scope.dataset.params.datetimenotzReadMode = 'AS_IS';
        }
        if (angular.isUndefined($scope.dataset.params.variablesExpansionLoopConfig)) {
            $scope.dataset.params.variablesExpansionLoopConfig = {mode: "CREATE_VARIABLE_FOR_EACH_COLUMN"}
        }

        $scope.test(false, false);
    };

    // TODO : check if this function is still in use
    $scope.gotoSearchAndImport = function () {
        $scope.uiState.bypassDirtinessCheck = true;
        DataikuAPI.connections.countIndexedAndUnindexed().success(function (data) {
            if (data.indexedConnections > 0) {
                $state.go('projects.project.datacatalog.datasources', {selectedTab: 'external-tables'});
            } else {
                $state.go("projects.project.datacatalog.database-explorer");
            }

        }).error(setErrorInScope.bind($scope));
    };

    $scope.gotoDatasetsAndIndexedTables = function() {
        $scope.uiState.bypassDirtinessCheck = true;
        $state.go('projects.project.datacatalog.datasources', { selectedTab: 'all' });
    }

    $scope.getAllTables = function(){
        Dialogs.confirmUnsafeHTML($scope, "Really list all tables?",
            "<p>This will list <strong>all tables in all schemas</strong> of your database.</p>"+
            "<p> On large entreprise databases, this is likely "+
            "to be extremely long and to cause your browser to become unresponsive.</p>"+
            "<p> To import existing tables, "+
            "it is recommended to use<br /> <a ng-click=\"dismiss(); gotoDatasetsAndIndexedTables()\">" +
            "<strong>Data catalog &gt; Datasets and indexed tables</strong></a>, " +
            "which offers ability to filter, search and mass import</p>").then($scope.test.bind(this, true, false))
    }

    // When doing Test Connection on BigQuery, ensure the dimensions are compatible with the BQ partition dimension
    // Add the bqDimension if necessary, delete any others relating to the same column
    function fixDimensionsForBQIfNecessary(dssDimensions, bqDimension) {
        function dimensionsEqual(dim1, dim2) {
            // Check equality ignoring integerRangeInfo
            if (dim1.name !== dim2.name || dim1.type !== dim2.type) {
                return false;
            }
            if (dim1.type === "time") {
                return dim1.params?.period === dim2.params?.period;
            }
            return true;
        }

        const dimensionsOfSameName = [];
        const dimensionsOfSameNameIndexes = [];
        for (let i = 0; i < dssDimensions.length; ++i) {
            if (dssDimensions[i].name === bqDimension.name) {
                dimensionsOfSameName.push(dssDimensions[i]);
                dimensionsOfSameNameIndexes.push(i);
            }
        }

        // If there is one matching dimension, it is the first and is equal to the BQ one, all good, no action needed
        // Otherwise we have to fix things
        const hasMatchingFirstCorrectDimension = dimensionsOfSameName.length === 1
                && dimensionsOfSameNameIndexes[0] === 0
                && dimensionsEqual(dimensionsOfSameName[0], bqDimension);

        if (!hasMatchingFirstCorrectDimension) {
            if (dimensionsOfSameName.length === 0) {
                dssDimensions.unshift(bqDimension);
                ActivityIndicator.info("A partition dimension has been added to match that of the BigQuery table", 5000);
            } else {
                // Delete all dimensions of that name
                for (let i = dimensionsOfSameNameIndexes.length - 1; i >= 0; --i) {
                    dssDimensions.splice(dimensionsOfSameNameIndexes[i], 1);
                }
                // Add correct BQ dimension at the start
                dssDimensions.unshift(bqDimension);
                ActivityIndicator.info("Some partition dimensions have been adjusted or removed to match the BigQuery table", 5000);
            }
        }
    }


    $scope.test = function (listTables, testTableOrQuery) {
        var previousTableList = $scope.testResult == null ? null : $scope.testResult.tablesList;
        const getTableMetadata = !listTables && !previousTableList && $scope.isNewDataset &&
            $scope.dataset && $scope.dataset.params && $scope.dataset.params.table != null;

        $scope.testResult = null;
        $scope.testing = true;
        if (angular.isUndefined($scope.dataset.params.connection)) {
            $scope.testable = false;
            $scope.testing = false;
            return;
        } else {
            $scope.testable = true;
        }
        // Don't test obviously wrong stuff
        //if ($scope.dataset.params.partitioned == "true" && $scope.dataset.params.mode == "table")
        DataikuAPI.datasets.externalSQL.test($stateParams.projectKey, $scope.dataset, 15,
            listTables, testTableOrQuery, getTableMetadata).success(function (data) {
            Logger.info('Got test result');
            $scope.testing = false;
            $scope.testResult = data;
            if ($scope.testResult.queryOK && $scope.testResult.querySchema) {
                $scope.consistency = { empty : false, result : $scope.testResult.schemaDetection };
                $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
                $scope.dataset.schema = $scope.testResult.schemaDetection.newSchema;
                // For BigQuery we ensure the partition dimension in the external table is present as the first DSS partition dimension
                let ensureExternalPartitionDimension = false;
                if ($scope.dataset.type === 'BigQuery') {
                    $scope.dataset.params.externalPartitionField =
                        $scope.testResult.tablePartitionInfo &&
                        $scope.testResult.tablePartitionInfo.dimension &&
                        $scope.testResult.tablePartitionInfo.dimension.name;
                    ensureExternalPartitionDimension = ($scope.dataset.params.externalPartitionField != null);
                }
                if (ensureExternalPartitionDimension) {
                    $scope.dataset.partitioning = $scope.dataset.partitioning || {};
                    $scope.dataset.partitioning.dimensions = $scope.dataset.partitioning.dimensions || [];
                    fixDimensionsForBQIfNecessary($scope.dataset.partitioning.dimensions, $scope.testResult.tablePartitionInfo.dimension);
                }
            }
            if ($scope.testResult.preview) {
                $scope.table = $scope.testResult.preview;
            }
            if (listTables) {
                if ($scope.testResult.schemaAware) {
                    if ($scope.testResult.catalogAware) {
                        angular.forEach($scope.testResult.tablesList, function(item) {
                            const qualifiedSchema = item.catalog ? (item.catalog + "." + item.schema) : item.schema;
                            item.qualified = qualifiedSchema + "." + item.table;
                            item.label = item.table + " (" + qualifiedSchema + ")";
                        });
                    } else {
                        angular.forEach($scope.testResult.tablesList, function(item) {
                            item.qualified = item.schema + "." + item.table;
                            item.label = item.table + " (" + item.schema + ")";
                        });
                    }
                } else {
                    angular.forEach($scope.testResult.tablesList, function(item) {
                        item.qualified = item.table;
                        item.label = item.table;
                    })
                }
                // In case the currently selected table does not exists in the list, resets the selected item
                if ($scope.expandedDatasetParams && $scope.expandedDatasetParams.tableAndSchema) {
                    if (!$scope.testResult.tablesList.find(item =>
                        (item.table === $scope.expandedDatasetParams.tableAndSchema.table)
                        && (!$scope.testResult.schemaAware || item.schema === $scope.expandedDatasetParams.tableAndSchema.schema)
                        && (!$scope.testResult.catalogAware || item.catalog === $scope.expandedDatasetParams.tableAndSchema.catalog))) {
                        $scope.expandedDatasetParams.tableAndSchema = undefined;
                    }
                }
            } else if (previousTableList) {
                $scope.testResult.tablesList = previousTableList;
            }
            if (!$scope.dataset.name && !$scope.uiState.new_dataset_name_manually_edited) {
                $scope.uiState.new_dataset_name = $scope.testResult.suggestedName;
            }
            if (getTableMetadata && $scope.testResult.table) {
                $scope.dataset.description = $scope.testResult.table.remarks;
            }
        }).error(function (a, b, c) {
            $scope.testing = false;
            setErrorInScope.bind($scope)(a,b,c);
        });
    };

    $scope.$watch("expandedDatasetParams.tableAndSchema", function(nv, ov) {
        if (nv) {
            $scope.dataset.params.table = nv.table;
            $scope.dataset.params.schema = nv.schema;
            $scope.dataset.params.catalog = nv.catalog;
            if (nv.remarks && $scope.isNewDataset) {
                $scope.dataset.description = nv.remarks;
            }
        }
    });

    $scope.$watch("dataset.params", function(nv, ov) {
        if (nv) {
            $scope.expandedDatasetParams.tableAndSchema = {
                table : $scope.dataset.params.table,
                schema : $scope.dataset.params.schema,
                catalog : $scope.dataset.params.catalog
            }
        }
    }, true);

    /* Fixup everything we can ... */
    $scope.$watch('dataset.params', function (nv, ov) {
        if (nv && ov && nv.mode != ov.mode && $scope.testResult) {
            $scope.testResult.testedConnectionOnly = true;
        }
        // We DO NOT run a test each time params change because a test is a COSTLY
        // thing. We NEVER run it on simple model change, and in the specific case
        // of SQL, we even NEVER run it without explicit user action.
    }, true);
});

app.controller("ExternalHiveDatasetController", function($scope, $stateParams, $controller, LoggerProvider, DataikuAPI, DatasetUtils) {
    $controller("ExternalSQLDatasetController", {$scope: $scope});

    $scope.hooks.getConnectionNames = function() {
        return DataikuAPI.connections.getHiveNames($stateParams.projectKey).success(function (data) {
            $scope.hiveConnections = data.map(function(db) {return {name:"@virtual(hive-jdbc):"+db, label:db};});
            $scope.setConnections($scope.hiveConnections.map(function(c) {return c.name;}));
        }).error(setErrorInScope.bind($scope));
    };


});


app.controller("ManagedSQLDatasetController", function($scope, $stateParams, $controller, $q, LoggerProvider, DataikuAPI, Dialogs, CreateModalFromTemplate, ActivityIndicator, DatasetTypesService) {
    $controller("BaseSQLDatasetController", {$scope: $scope, withConsistency: true});

    var Logger = LoggerProvider.getLogger('datasets.sql');

    $scope.possibleTableCreationModes = [
        ["auto", "Automatically generate"],
        ["custom", "Manually define"]
    ]

    $scope.possibleSynapseDistribution = [
        ["ROUND_ROBIN", "Round robin"],
        ["HASH", "Hash"],
        ["REPLICATE", "Replicate"]
    ]

    $scope.copyCodeSnippet = function(snippet) {
        var stringToPutIntoClippboard = snippet.code;
        //ugly but necessary
        var textArea = document.createElement("textarea");
        textArea.style.position = 'absolute';
        textArea.style.top = '-1000px';
        textArea.style.left = '-1000px';
        textArea.value = stringToPutIntoClippboard;
        document.body.appendChild(textArea);
        textArea.select();
        try {
            var successful = document.execCommand('copy');
            if (successful) {
                ActivityIndicator.success("Sample copied into cliboard");
            } else {
                ActivityIndicator.error("Your browser does not support automatic copying into clibboard");
            }
        } catch (err) {
            ActivityIndicator.error("Your browser does not support automatic copying into clibboard");
        }
        document.body.removeChild(textArea);
    };

    $scope.revertToAutogeneratedStatement = function(){
        $scope.test(true).then(function() {
            $scope.dataset.params.customCreateStatement = $scope.testResult.autogeneratedCreateStatement;
        });
    };

    $scope.refreshToAutogeneratedStatement = function(){
        $scope.test(true).then(function() { });
    };

    $scope.onLoadComplete = function () {
        if (angular.isUndefined($scope.dataset.params.mode)) {
            $scope.dataset.params.mode = 'table';
        }
        if (angular.isUndefined($scope.dataset.params.partitioningType)) {
            $scope.dataset.params.partitioningType = 'custom';
        }
        if (angular.isUndefined($scope.dataset.params.normalizeDoubles)) {
            $scope.dataset.params.normalizeDoubles = true;
        }
        if (!$scope.dataset.params.tableCreationMode) {
            $scope.dataset.params.tableCreationMode = "auto";
        }

        if ($scope.dataset.type === "Synapse" && angular.isUndefined($scope.dataset.params.tableDistributionMode)) {
            $scope.dataset.params.tableDistributionMode = 'ROUND_ROBIN';
        }

        // Fire initial test
        // On managed, we more or less guarantee that we only target a table, so it can't be too heavy
        // EXCEPT IN CASE OF PARTITIONING
        $scope.test(false, !$scope.dataset.params.partitioned);
    };

    $scope.dropTable = function () {
        Dialogs.confirm($scope,'Drop table','Are you sure you want to drop the SQL table ?').then(function(){
            DataikuAPI.datasets.managedSQL.dropTable($stateParams.projectKey, $scope.dataset).success(function (data) {
                if(data.length > 0) {
                    // TODO @flow
                    // Some error happened!
                    CreateModalFromTemplate("/templates/datasets/delete-dataset-results.html", $scope, null, function(newScope) {
                        newScope.results = data;
                    });
                } else {
                    $scope.test(false, true);
                    $scope.checkConsistency();
                }
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.createTable = function () {
        DataikuAPI.datasets.managedSQL.createTable($stateParams.projectKey, $scope.dataset).success(function (data) {
            $scope.test(false, true);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.overwriteSchemaFromTable = function () {
        if (!$scope.dataset.schema) {
            $scope.dataset.schema = {};
        }
        $scope.dataset.schema.columns = $scope.testResult.currentTableSchema.columns;
        $scope.test(false);
    };

    $scope.filterEligibleBigQueryPartitionColumns = function(columns) {
        return columns.filter(c => (c.type === 'int' || c.type === 'tinyint' || c.type === 'bigint' || c.type === 'smallint' || c.type === 'date'|| c.type === 'dateonly'|| c.type === 'datetimenotz'));
    };

    $scope.filterEligibleBigQueryClusteringColumns = function(columns, clusteringIndex) {
        // Remove the columns that have a type that is not compatible with clustering
        let result = columns.filter(c => (c.type === 'string' || c.type === 'date' || c.type === 'dateonly'
            || c.type === 'datetimenotz' || c.type === 'boolean'
            || c.type === 'bigint' || c.type === 'int' || c.type === 'smallint' || c.type === 'tinyint'));

        // Remove the column used to partition the table
        if ($scope.dataset.params.useBigQueryPartitioning && $scope.dataset.params.bigQueryPartitioningField) {
            result = result.filter(c => (c.name !== $scope.dataset.params.bigQueryPartitioningField));
        }

        // Remove the columns used to cluster the table above.
        for (let i = 0; i < clusteringIndex; i++) {
            if ($scope.dataset.params.bigQueryClusteringColumns && $scope.dataset.params.bigQueryClusteringColumns.length > i) {
                result = result.filter(c => c.name !== $scope.dataset.params.bigQueryClusteringColumns[i].name);
            }
        }
        return result;
    };

    $scope.addBigQueryClusteringColumn = function() {
        if (!$scope.dataset.params.bigQueryClusteringColumns) {
            $scope.dataset.params.bigQueryClusteringColumns = [];
        }
        $scope.dataset.params.bigQueryClusteringColumns.push({name:''});
    };

    $scope.removeBigQueryClusteringColumn = function(index) {
        $scope.dataset.params.bigQueryClusteringColumns.splice(index, 1);
    };

    $scope.forbidBigQueryWritePartitionsOptions = [
        { n: 'Enabled', v: 'ENABLED' },
        { n: 'Disabled', v: 'DISABLED' },
        { n: 'Inherit connection settings', v: 'INHERIT' }
    ];

    const isColumnTemporal = (column) => {
        return column && ['date', 'dateonly', 'datetimenotz'].includes(column.type);
    }

    /**
     * Automatically adjust a BigQuery dataset partitioning if it's modified and partitioned with a single partitioning column of type time-based
     */
    $scope.$watch("dataset.partitioning", function(newPart, oldPart) {
        const isModified = newPart !== oldPart;
        const isSingleDimension = $scope.dataset.partitioning?.dimensions?.length === 1;
        if ($scope.dataset.type !== 'BigQuery' || !isModified || !isSingleDimension) {
            return;
        }
        const dimension = $scope.dataset.partitioning.dimensions[0];
        const schemaColumn = $scope.dataset.schema.columns.find(col => col.name === dimension.name);
        const validPartitioningPeriods = $scope.createBigQueryDatePartitioningPeriods(schemaColumn).map((x) => x[0]);
        if (dimension.type === "time" && validPartitioningPeriods.includes(dimension.params?.period) && isColumnTemporal(schemaColumn)) {
            const bigQueryPartitioningEnabled = $scope.dataset.params?.useBigQueryPartitioning && $scope.dataset.params?.bigQueryPartitioningField;
            if (bigQueryPartitioningEnabled) {
                if (
                    $scope.dataset.params.bigQueryPartitioningField === dimension.name &&
                    $scope.dataset.params.bigQueryPartitioningType === "DATE" &&
                    $scope.dataset.params.bigQueryPartitioningPeriod !== dimension.params.period
                ) {
                    $scope.dataset.params.bigQueryPartitioningPeriod = dimension.params.period;
                    ActivityIndicator.info("Native BigQuery partitioning period has been updated (see the Advanced tab)", 5000);
                }
            } else {
                Object.assign($scope.dataset.params, {
                    useBigQueryPartitioning: true,
                    bigQueryPartitioningField: dimension.name,
                    bigQueryPartitioningType: "DATE",
                    bigQueryPartitioningPeriod: dimension.params.period
                });
                ActivityIndicator.info("Native BigQuery partitioning has been enabled (see the Advanced tab)", 5000);
            }
        }
    }, true);

    $scope.$watchGroup(
        ["dataset.params.useBigQueryPartitioning", "dataset.params.bigQueryPartitioningField", "dataset.params.bigQueryPartitioningPeriod"],
        function([newPart, newField, newPeriod], [oldPart, oldField, oldPeriod]) {
            if (newField) {
                const partitioningColumn = $scope.dataset.schema.columns.find(c => c.name === newField);
                $scope.dataset.params.bigQueryPartitioningType = (!partitioningColumn || DatasetTypesService.isTemporalType(partitioningColumn.type))
                    ? "DATE"
                    : "RANGE";

                // If the new value was selected as a partitioning column, remove it from clustering columns if it was used as a clustering column
                const clusteringColumn = (
                    $scope.dataset.params.bigQueryClusteringColumns &&
                    $scope.dataset.params.bigQueryClusteringColumns.find(c => c.name === newField)
                );
                if (clusteringColumn) {
                    const index = $scope.dataset.params.bigQueryClusteringColumns.indexOf(clusteringColumn);
                    $scope.dataset.params.bigQueryClusteringColumns.splice(index, 1);
                }
            }
            const anyChanged = newPart !== oldPart || newField !== oldField || newPeriod !== oldPeriod;
            if (anyChanged) {
                // Test again when BigQuery native partitioning is changed so warnings are correctly raised
                $scope.test(false);
            }
        }
    );

    $scope.$watch("dataset.params.bigQueryClusteringColumns", function(nv, ov) {
        if (nv) {
            // Remove any duplicate column that may have been introduced by user.
            // It may happen if user has chosen [col1, col2, col3] as clustering columns and then select col2 in first dropdown.
            const columns = $scope.dataset.params.bigQueryClusteringColumns;
            $scope.dataset.params.bigQueryClusteringColumns = columns.filter((item, index) => {
                if (!item || !item.name) {
                    return true; // Keep empty placeholders
                }
                const firstMatchingColumn = columns.find(c => c.name === item.name);
                return columns.indexOf(firstMatchingColumn) === index;
            });
        }
    }, true);

    $scope.test = function (connectionOnly) {
        var deferred = $q.defer();
        $scope.testResult = null;
        $scope.testing = true;
        DataikuAPI.datasets.managedSQL.test($stateParams.projectKey, $scope.dataset, 15, connectionOnly).success(function (data) {
            Logger.info('Got test result');
            $scope.testing = false;
            $scope.testResult = data;
            
            if (!$scope.testResult) {
                 $scope.testResultCase = null;
            } else if (!$scope.testResult.connectionOK) {
                $scope.testResultCase = 'CONNECTION_FAILED';
            } else if ($scope.testResult.testedConnectionOnly) {
                $scope.testResultCase = 'CONNECTION_OK_ONLY';
            } else if (!$scope.testResult.tableExists) {
                $scope.testResultCase = 'CONNECTION_TABLE_DOES_NOT_EXIST';
            } else if (!$scope.testResult.preview) {
                $scope.testResultCase = 'CONNECTION_TABLE_PREVIEW_FAILED';
            } else if (!$scope.testResult.datasetPartitioningMatchesTable && $scope.isPartitioningConsistencyEnabled()) {
                $scope.testResultCase = 'CONNECTION_PARTITIONING_MISMATCH_FORBIDDEN';
            } else if (!$scope.testResult.nativePartitioningMatchesTable) {
                $scope.testResultCase = 'CONNECTION_PARTITIONING_MISMATCH_WARNING';
            } else if (!$scope.testResult.schemaMatchesTable) {
                $scope.testResultCase = 'CONNECTION_SCHEMA_MISMATCH';
            } else {
                // ALL previous error cases were false, so 'OK'
                $scope.testResultCase = 'CONNECTION_OK';
            }
        

            if ($scope.testResult.preview) {
                $scope.table = $scope.testResult.preview;
            }
            deferred.resolve();

        }).error(function (a,b,c) {
            $scope.testing = false;
            setErrorInScope.bind($scope)(a,b,c);
            deferred.reject();
        });
        return deferred.promise;
    };

    $scope.$watch("dataset.params.tableCreationMode", function(nv, ov) {
        if (nv == "custom" && ov == "auto" && !$scope.dataset.params.customCreateStatement) {
             $scope.dataset.params.customCreateStatement = $scope.testResult.autogeneratedCreateStatement;
        }
    });

    /* For autocompletion in primary / distribute / sort / ... keys */
    $scope.$watch("dataset.schema", function(nv, ov){
        if (!nv) return;
        $scope.schemaColumnNames = nv.columns.map(x => x.name);
    }, true);

    /* Fixup everything we can ... */
    $scope.$watch('dataset.params', function (nv, ov) {
    }, true);
});


app.component("sqlSparkOptions", {
    templateUrl: "/templates/datasets/fragments/sql-spark-options.html",
    controller: function($scope, DataikuAPI, $stateParams) {
        $scope.$ctrl = this;
        
        $scope.uiState = {accessOptions:[]};
        
        function setAccessOptions() {
            if (!$scope.$ctrl.dataset || !$scope.$ctrl.dataset.type) return;
            const type = $scope.$ctrl.dataset.type;
            if (type == 'Vertica') {
                // the vertica dialect for spark can't write null values, greatly limiting its usefulness
                $scope.uiState.accessOptions = [['NONE', 'None'], ['READ', 'For reads']];
            } else if (type == 'Redshift') {
                // writing via jdbc is just too slow
                $scope.uiState.accessOptions = [['NONE', 'None'], ['READ', 'For reads']];
            } else {
                $scope.uiState.accessOptions = [['NONE', 'None'], ['READ', 'For reads'], ['WRITE', 'For writes'], ['READ_WRITE', 'For reads and writes']];
            }
        };
        
        function checkSparkIntegrationPossible() {
            if (!$scope.$ctrl.dataset || !$scope.$ctrl.dataset.params || !$scope.$ctrl.dataset.params.connection || $scope.$ctrl.dataset.params.sparkJdbcAccess == 'NONE') return; // don't waste time
            DataikuAPI.datasets.SQL.checkSparkIntegration($stateParams.projectKey, $scope.$ctrl.dataset).success(function (data) {
                    $scope.uiState.sparkJdbcStatus = data;
                }).error(setErrorInScope.bind($scope));
        };
        
        $scope.hasMin = function() {
            return $scope.uiState.sparkJdbcStatus && $scope.$ctrl.dataset.params.partitionColumn in $scope.uiState.sparkJdbcStatus.columnMinMetrics;
        };
        $scope.getMin = function() {
            return $scope.uiState.sparkJdbcStatus.columnMinMetrics[$scope.$ctrl.dataset.params.partitionColumn];
        };
        $scope.hasMax = function() {
            return $scope.uiState.sparkJdbcStatus && $scope.$ctrl.dataset.params.partitionColumn in $scope.uiState.sparkJdbcStatus.columnMaxMetrics;
        };
        $scope.getMax = function() {
            return $scope.uiState.sparkJdbcStatus.columnMaxMetrics[$scope.$ctrl.dataset.params.partitionColumn];
        };
        
        // don't put a watch on the dataset's params to recheck integration, that would 
        // make tons of calls (like 1 per keystroke...) for something that's quite anecdotic
        $scope.recheck = function() {
            checkSparkIntegrationPossible();
        };
        $scope.$watch('$ctrl.dataset.params.sparkJdbcAccess', checkSparkIntegrationPossible); // that one is the exception :)
        checkSparkIntegrationPossible();
        $scope.$watch('$ctrl.dataset.type', setAccessOptions);
    },
    bindings: {
        dataset: "="
    }
});

}());
