(function() {
    'use strict';

    const THUMBNAIL_WIDTH = 120;
    const THUMBNAIL_HEIGHT = 80;

    const app = angular.module('dataiku.directives.insights', ['dataiku.filters', 'dataiku.charts']);

    /**
     * Service responsible for displaying charts based on a pivot response.
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_views.js
     */
    app.directive('pivotChartResult', function($timeout, $q, $state, CHART_TYPES, DKUPivotCharts, Logger, ChartFeatures, ChartDimension, ChartZoomControlAdapter, CanvasUtils, EChartsManager, Fn, ChartStoreFactory, ChartFormattingPane, ChartFormattingPaneSections, ChartActivityIndicator, ChartDefinitionChangeHandler, DSSVisualizationThemeUtils) {

        return {
            templateUrl: '/static/dataiku/js/simple_report/directives/pivot-chart-result/pivot-chart-result.directive.html',
            scope: true,
            link: function(scope, element) {

                //these functions need to be put in the scope to be accessible in dashboards
                scope.canHaveZoomControls = ChartFeatures.canHaveZoomControls;
                scope.hasScatterZoomControlActivated = ChartFeatures.hasScatterZoomControlActivated;
                scope.isEChart = ChartFeatures.isEChart;
                scope.canDisplayLegend = ChartFeatures.canDisplayLegend;

                scope.chartActivityIndicator = ChartActivityIndicator.buildDefaultActivityIndicator();

                scope.openSection = function(section) {
                    const readOnly = $state.current.name === 'projects.project.dashboards.insights.insight.view';
                    if (!readOnly && !(section === ChartFormattingPaneSections.MISC && scope.isInPredicted)) {
                        ChartFormattingPane.open(section, true, true);
                    }
                };

                // Chart lazy loading causes issues with dashboard export, so we disable it when in an export flow.
                scope.disableLazyLoading = getCookie('dku_unattended') === 'true';

                const loadChart = function(axesDef, echartDef, d3Chart, chartActivityIndicator, hideLegend = false) {
                    if (ChartFeatures.isEChart(scope.chart.def)) {
                        element.find('.mainzone').remove();

                        const data = scope.response.result.pivotResponse;

                        if (scope.chart.theme && !DSSVisualizationThemeUtils.isThemeFontLoaded(scope.chart.theme)) {
                            // Force loading of fonts for gauge so we have the font loaded and can measure text width on first paint
                            DSSVisualizationThemeUtils.loadThemeFont(scope.chart.theme)
                                .finally(() => EChartsManager.initEcharts(scope, element, data, axesDef, echartDef, chartActivityIndicator, hideLegend, scope.chart.theme));
                        } else {
                            EChartsManager.initEcharts(scope, element, data, axesDef, echartDef, chartActivityIndicator, hideLegend, scope.chart.theme);
                        }
                    } else {
                        EChartsManager.disposeEcharts(scope);
                        d3Chart(element.find('.pivot-charts').css('display', ''), scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                    }
                };

                const redrawChart = function() {
                    scope.uiDisplayState = scope.uiDisplayState || {};
                    scope.uiDisplayState.displayBrush = false;
                    scope.uiDisplayState.brushData = {};

                    // Make sure the new state of the chart is not built from obsolete settings.
                    ChartDefinitionChangeHandler.clearAllPrivateMembers(scope.chart.def);

                    scope.chart.def.hasEchart = ChartFeatures.hasEChartsDefinition(scope.chart.def.type);
                    scope.chart.def.hasD3 = ChartFeatures.hasD3Definition(scope.chart.def.type);
                    if (ChartFeatures.isEChart(scope.chart.def)) {
                        scope.onChartInit = EChartsManager.onInit(scope, element);
                        scope.onMetaChartInit = EChartsManager.onMetaInit();
                    } else {
                        scope.onChartInit = null;
                        scope.onMetaChartInit = null;
                        EChartsManager.disposeEcharts(scope);
                    }

                    if (!scope.echart && !scope.chart.def.type === CHART_TYPES.ADMINISTRATIVE_MAP) {
                        element.find('.legend-zone').remove();
                    }

                    if (scope.chart.def.$zoomControlInstanceId) {
                        ChartZoomControlAdapter.clear(scope.chart.def.$zoomControlInstanceId);
                        scope.chart.def.$zoomControlInstanceId = null;
                    }

                    const { store, id } = ChartStoreFactory.getOrCreate(scope.chart.def.$chartStoreId);
                    scope.chart.def.$chartStoreId = id;
                    const rootElement = getRootElement(scope.chart.def);
                    const dims = ChartDimension.getChartDimensions(scope.chart.def);
                    const axesDef = ChartDimension.getAxesDef(scope.chart.def, dims, store);

                    // no rootElement, means we probably have a echarts and font is handle in their part.
                    if (rootElement && scope.chart && scope.chart.theme && scope.chart.theme.generalFormatting && scope.chart.theme.generalFormatting.fontFamily) {
                        element.css('--visualization-font-family', scope.chart.theme.generalFormatting.fontFamily);
                    }
                    switch (scope.chart.def.type) {
                        case CHART_TYPES.GROUPED_COLUMNS: {
                            DKUPivotCharts.GroupedColumnsChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.MULTI_COLUMNS_LINES: {
                            DKUPivotCharts.MultiplotChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.STACKED_COLUMNS: {
                            DKUPivotCharts.StackedColumnsChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.LINES: {
                            DKUPivotCharts.LinesChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse, scope.getExecutePromise, scope.uiDisplayState, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.STACKED_BARS: {
                            DKUPivotCharts.StackedBarsChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.STACKED_AREA: {
                            DKUPivotCharts.StackedAreaChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.BINNED_XY: {
                            DKUPivotCharts.BinnedXYChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.GROUPED_XY: {
                            DKUPivotCharts.GroupedXYChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.PIE: {
                            loadChart(axesDef, DKUPivotCharts.PieEChartDef, DKUPivotCharts.PieChart, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.LIFT: {
                            DKUPivotCharts.LiftChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.SCATTER: {
                            DKUPivotCharts.ScatterPlotChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse, scope.uiDisplayState);
                            break;
                        }
                        case CHART_TYPES.SCATTER_MULTIPLE_PAIRS: {
                            DKUPivotCharts.ScatterPlotMultiplePairsChart.create(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse, scope.uiDisplayState);
                            break;
                        }
                        case CHART_TYPES.PIVOT_TABLE: {
                            DKUPivotCharts.PivotTableChart(rootElement, scope.chart.def, scope, axesDef, scope.response.result.pivotResponse);
                            break;
                        }
                        case CHART_TYPES.BOXPLOTS: {
                            rootElement.show();
                            DKUPivotCharts.BoxplotsChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope, axesDef);
                            break;
                        }
                        case CHART_TYPES.ADMINISTRATIVE_MAP: {
                            DKUPivotCharts.AdministrativeMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.GRID_MAP: {
                            DKUPivotCharts.GridMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.SCATTER_MAP: {
                            DKUPivotCharts.ScatterMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.DENSITY_HEAT_MAP: {
                            DKUPivotCharts.DensityHeatMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.GEOMETRY_MAP: {
                            DKUPivotCharts.GeometryMapChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.DENSITY_2D: {
                            rootElement.show();
                            DKUPivotCharts.Density2DChart(rootElement.get(0), scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.KPI: {
                            rootElement.show();
                            DKUPivotCharts.KpiChart(rootElement.get(0), scope, scope.chart.def, scope.getChartTheme());
                            break;
                        }
                        case CHART_TYPES.RADAR: {
                            loadChart(axesDef, DKUPivotCharts.RadarEChartDef, Fn.NOOP, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.SANKEY: {
                            loadChart(axesDef, DKUPivotCharts.SankeyEChartDef, Fn.NOOP, scope.chartActivityIndicator, true);
                            break;
                        }
                        case CHART_TYPES.GAUGE: {
                            loadChart(axesDef, DKUPivotCharts.GaugeEChartDef, Fn.NOOP, scope.chartActivityIndicator);
                            break;
                        }
                        case CHART_TYPES.WEBAPP: {
                            rootElement.show();
                            DKUPivotCharts.WebappChart(rootElement, scope.chart.def, scope.response.result.pivotResponse, scope);
                            break;
                        }
                        case CHART_TYPES.TREEMAP: {
                            loadChart(axesDef, DKUPivotCharts.TreemapEChartDef, Fn.NOOP, scope.chartActivityIndicator, true);
                            break;
                        }
                        default:
                            throw new Error('Unknown chart type: ' + scope.chart.def.type);
                    }
                };

                function getRootElement(def) {
                    switch (def.type) {
                        case CHART_TYPES.GROUPED_COLUMNS:
                        case CHART_TYPES.MULTI_COLUMNS_LINES:
                        case CHART_TYPES.STACKED_COLUMNS:
                        case CHART_TYPES.LINES:
                        case CHART_TYPES.STACKED_BARS:
                        case CHART_TYPES.STACKED_AREA:
                        case CHART_TYPES.BINNED_XY:
                        case CHART_TYPES.GROUPED_XY:
                        case CHART_TYPES.LIFT:
                        case CHART_TYPES.SCATTER:
                        case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        case CHART_TYPES.ADMINISTRATIVE_MAP:
                        case CHART_TYPES.GRID_MAP:
                        case CHART_TYPES.SCATTER_MAP:
                        case CHART_TYPES.DENSITY_HEAT_MAP:
                        case CHART_TYPES.GEOMETRY_MAP:
                            return element.find('.pivot-charts').css('display', '');
                        case CHART_TYPES.PIE:
                        case CHART_TYPES.RADAR:
                        case CHART_TYPES.SANKEY:
                        case CHART_TYPES.GAUGE:
                        case CHART_TYPES.TREEMAP:
                            return element;
                        case CHART_TYPES.PIVOT_TABLE:
                            return element.find('.pivot-table-container').css('display', '');
                        case CHART_TYPES.BOXPLOTS:
                            return element.find('.boxplots-container');
                        case CHART_TYPES.DENSITY_2D:
                            return element.find('.direct-svg');
                        case CHART_TYPES.KPI:
                            return element.find('.kpi-container');
                        case CHART_TYPES.WEBAPP:
                            return element.find('.webapp-charts-container');
                        default:
                            throw new Error('Unknown chart type: ' + scope.chart.def.type);
                    }
                }

                function subscribeThumbnailToRedraw() {
                    if (scope.abortThumbnail) {
                        scope.abortThumbnail();
                    }
                    const previousLoadedCallback = scope.loadedCallback;
                    /*
                     * create an AbortController so we can cancel the promise and
                     * avoid unnecessary thumbnail computation
                     */
                    const controller = new AbortController();
                    scope.abortThumbnail = () => {
                        controller.abort();
                        scope.loadedCallback = previousLoadedCallback;
                    };
                    scope.loadedCallback = () => {
                        window.requestAnimationFrame(() => scope.updateThumbnail(controller.signal));
                        scope.loadedCallback = previousLoadedCallback;
                        if (typeof (previousLoadedCallback) === 'function') {
                            previousLoadedCallback();
                        }
                    };
                }

                function redraw(options = {}) {
                    if (!scope.response || !scope.response.hasResult || !scope.isInitialDrawReady) {
                        return;
                    }

                    element.children().children().each(function() {
                        if (this.tagName !== 'ACTIVITY-INDICATOR') {
                            $(this).hide();
                        }
                    });

                    if (scope.validity && !scope.validity.valid) {
                        scope.validity.valid = true;
                    }

                    // for debug
                    element.attr('chart-type', scope.chart.def.type);

                    if (scope.timing) {
                        scope.timing.drawStart = new Date().getTime();
                    }

                    try {
                        Logger.info('Start draw chart', scope.chart.def.type);
                        if (options.updateThumbnail) {
                            subscribeThumbnailToRedraw();
                        }
                        redrawChart();
                    } catch (err) {
                        if (err instanceof ChartIAE) {
                            Logger.warn('CHART IAE', err);
                            if (scope.validity) {
                                scope.validity.valid = false;
                                scope.validity.type = 'DRAW_ERROR';
                                scope.validity.message = err.message;
                            }
                        } else {
                            throw err;
                        }
                    }
                }

                function getChartElementThumbnail() {
                    return $q((resolve) => {
                        let setup;
                        let canvas;

                        if (!scope.isEChart(scope.chart.def)) {
                            const canvasParent = document.querySelector('foreignObject');
                            let margins;

                            switch (scope.chart.def.type) {
                                case CHART_TYPES.BOXPLOTS:
                                    return scope.exportBoxPlots(true).then(canvas => resolve({ canvas: CanvasUtils.resize(canvas, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT) }));
                                case CHART_TYPES.SCATTER:
                                case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                                    margins = { x: canvasParent?.getAttribute('x'), y: canvasParent?.getAttribute('y') };
                                    return resolve({
                                        canvas: CanvasUtils.resize(
                                            document.querySelector('.chart-svg canvas'),
                                            THUMBNAIL_WIDTH,
                                            THUMBNAIL_HEIGHT,
                                            false,
                                            document.querySelectorAll('.chart-svg .reference-line'),
                                            margins
                                        )
                                    });
                                case CHART_TYPES.GEOMETRY_MAP:
                                case CHART_TYPES.SCATTER_MAP:
                                case CHART_TYPES.ADMINISTRATIVE_MAP:
                                case CHART_TYPES.GRID_MAP:
                                    return resolve({ canvas: CanvasUtils.resize(document.querySelector('canvas.leaflet-zoom-animated'), THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT) });
                                case CHART_TYPES.DENSITY_2D:
                                    return resolve({ svg: document.querySelector('svg.direct-svg') });
                                case CHART_TYPES.LIFT:
                                case CHART_TYPES.LINES:
                                case CHART_TYPES.MULTI_COLUMNS_LINES:
                                    // Thicker strokes
                                    setup = clonedSvg =>
                                        clonedSvg.querySelectorAll('path.visible').forEach(line =>
                                            line.setAttribute('stroke-width', line.getAttribute('stroke-width') * 3));
                                // falls through
                                default:
                                    canvas = document.querySelector('canvas');

                                    if (canvas) {
                                        return resolve({ canvas: CanvasUtils.resize(canvas, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT) });
                                    } else {
                                        return resolve({ svg: document.querySelector('svg.chart-svg'), setup });
                                    }
                            }
                        } else if (scope.echart && scope.echart.echartDefInstance){
                            const thumbnailOptions = scope.echart.echartDefInstance.getThumbnailOptions(scope.echart.options);
                            const canvasURL = EChartsManager.getThumbnailCanvasUrl(thumbnailOptions, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT);
                            return resolve({ canvasURL });
                        }
                    });
                }

                const isTooSmall = () => {
                    const chart = document.querySelector('.chart-wrapper') || document.querySelector('.pivot-chart');
                    if (chart) {
                        const ratio = chart.clientHeight / chart.clientWidth;
                        return chart.clientWidth < 150 || chart.clientHeight < 150 || ratio < 0.2 || ratio > 1.8;
                    }
                    return false;
                };

                /*
                 * If we are using webgl in order for toDataURL to return something
                 * we need to be in the drawing function and have nothing async
                 */
                scope.getThumbnailForWebgl = origCanvas => {
                    if (ChartFeatures.canHaveThumbnail(scope.chart.def) && !scope.noThumbnail) {
                        if (isTooSmall()) {
                            return;
                        }

                        return CanvasUtils.resize(origCanvas, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT).toDataURL();
                    }
                };

                scope.updateThumbnailWithData = dataURL => {
                    if (dataURL) {
                        scope.chart.def.thumbnailData = dataURL;
                    } else {
                        delete scope.chart.def.thumbnailData;
                    }
                };

                scope.updateThumbnail = (signal) => $q(resolve => {
                    if (ChartFeatures.canHaveThumbnail(scope.chart.def) && !scope.noThumbnail) {
                        Logger.info('Computing thumbnail');

                        const isAborted = () => signal && signal.aborted;

                        if (isAborted() || isTooSmall()) {
                            return resolve();
                        }

                        getChartElementThumbnail()
                            .then(({ canvas, svg, setup, canvasURL }) => {
                                if (svg && !isAborted()) {
                                    return CanvasUtils.svgToCanvas(svg, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT, { filters: ['text', '.axis', '.hlines', '.vlines', '.legend', '.background'], setup });
                                // ECharts charts are all using canvas. But so is our pure canvas scatter, that does not require canvasURL but canvas so it must go to the final return
                                } else if (canvasURL && (scope.chart.def.displayWithECharts || scope.chart.def.displayWithEChartsByDefault)) {
                                    scope.updateThumbnailWithData(canvasURL);
                                    return;
                                }
                                return canvas;
                            })
                            .then(canvas => {
                                if (canvas && !isAborted()) {
                                    scope.chart.def.thumbnailData = canvas.toDataURL();
                                    Logger.info('Thumbnail Done');
                                }
                                resolve();
                            });
                    } else {
                        delete scope.chart.def.thumbnailData;
                        resolve();
                    }
                });

                scope.$on('resize', () => redraw());
                scope.$on('redraw', (e, opts) => redraw(opts));

                scope.$on('export-chart', function() {
                    scope.export();
                });

                scope.$on('$destroy', function() {
                    if (scope.chart.def.$zoomControlInstanceId) {
                        ChartZoomControlAdapter.clear(scope.chart.def.$zoomControlInstanceId);
                        scope.chart.def.$zoomControlInstanceId = null;
                    }
                });

                scope.export = function() {
                    if (scope.chart.def.type === CHART_TYPES.BOXPLOTS) {
                        scope.exportBoxPlots().then(function(canvas) {
                            CanvasUtils.downloadCanvas(canvas, scope.chart.def.name + '.png');
                        });
                        return;
                    }

                    let width;
                    let height;
                    let $svg;

                    if (scope.chart.def.type === CHART_TYPES.DENSITY_2D) {
                        $svg = element.find('svg.direct-svg');
                    } else {
                        $svg = element.find('svg.chart-svg');
                    }

                    if ($svg.length) {
                        width = $svg.width();
                        height = $svg.height();
                    } else {
                        const echart = element.find('div.main-echarts-zone');
                        width = echart.width();
                        height = echart.height();
                    }
                    scope.exportData(width, height).then(function(canvas) {
                        CanvasUtils.downloadCanvas(canvas, scope.chart.def.name + '.png');
                    });
                };

                /**
                 * @returns A style element containing all the CSS rules relative to charts
                 */
                function getChartStyleRules(forCanvg) {
                    const svgNS = 'http://www.w3.org/2000/svg';
                    const style = document.createElementNS(svgNS, 'style');
                    for (let i = 0; i < document.styleSheets.length; i++) {
                        const str = document.styleSheets[i].href;
                        if (str != null && str.substr(str.length - 10) === 'charts.css') {
                            const rules = document.styleSheets[i].cssRules;
                            for (let j = 0; j < rules.length; j++) {
                                style.textContent += (rules[j].cssText);
                                style.textContent += '\n';
                            }
                            break;
                        }
                    }
                    if (forCanvg) {
                        // Yes it's ugly
                        style.textContent = `<![CDATA[ .totallyFakeClassBecauseCanvgParserIsBuggy  {}\n${style.textContent}{]]>`;
                        // "{" is here to workaround CanVG parser brokenness
                    }
                    return style;
                }

                /**
                 * Add the passed title to the passed canvas
                 * @param canvas: canvas that we want to add a title to
                 * @param title: title that will be added to the canvas
                 * @params scale: the scaling coefficient that we want to apply to the title
                 */
                function addTitleToCanvas(canvas, title, titleHeight, scale) {
                    const ctx = canvas.getContext('2d');
                    ctx.textAlign = 'center';
                    ctx.textBaseline = 'middle';
                    ctx.font = 'normal normal 100 ' + 18 * scale + 'px sans-serif';
                    ctx.fillStyle = '#777';
                    ctx.fillText(title, canvas.width / 2, titleHeight * scale / 2);
                }

                /**
                 * Compute a multiplier coefficient enabling to scale passed dimensions to reach an image containing the same amount of pixels as in a 720p image.
                 * @param w: width of original image that we'd like to scale to HD
                 * @param h: height of original image that we'd like to scale to HD
                 * @returns c so that (w * c) * (h * c) = 921600, the number of pixels contained in a 720p image
                 */
                function getCoeffToHD(w, h) {
                    const nbPixelsHD = 921600; // nb pixels contained in a 720p image
                    const multiplier = Math.sqrt(nbPixelsHD / (w * h)); // so that (w * multiplier) * (h * multiplier) = nbPixelsHD
                    return multiplier;
                }


                scope.exportData = function(w, h, simplified, svgEl, noTitle) {
                    /**
                     * @returns a canvas that fits the passed dimensions
                     */
                    function generateCanvas(w, h) {
                        const canvas = document.createElement('canvas');
                        canvas.setAttribute('width', w);
                        canvas.setAttribute('height', h);
                        return canvas;
                    }

                    /**
                     * @returns the svg that contains the chart.
                     */
                    function getChartSVG() {
                        let svg;
                        if (angular.isDefined(svgEl)) {
                            svg = svgEl.get(0);
                        } else if (scope.chart.def.type === CHART_TYPES.DENSITY_2D) {
                            svg = element.find('svg.direct-svg').get(0);
                        } else {
                            svg = element.find('svg.chart-svg').get(0);
                        }
                        return svg;
                    }

                    /**
                     * Adapted from https://code.google.com/p/canvg/issues/detail?id=143
                     * @param svg: the SVG to get cloned
                     * @returns a clone of the passed SVG
                     */
                    function cloneSVG(svg) {
                        const clonedSVG = svg.cloneNode(true);
                        const $clonedSVG = $(clonedSVG);
                        const $svg = $(svg);
                        $clonedSVG.width($svg.width());
                        $clonedSVG.height($svg.height());
                        const customFontFamily = getComputedStyle(svg).getPropertyValue('--visualization-font-family');
                        if (customFontFamily) {
                            $clonedSVG.css('font-family', `${customFontFamily}, 'SourceSansPro'`);
                        }
                        return clonedSVG;
                    }

                    function fillOldCanvasInNewCanvas(oldCanvas, newCanvas, horizontalOffset, verticalOffset) {

                        const context = newCanvas.getContext('2d');

                        const oldWidth = oldCanvas.width;
                        const oldHeight = oldCanvas.height;
                        const proportion = oldWidth / oldHeight;

                        const availableWidth = newCanvas.width - horizontalOffset;
                        const availableHeight = newCanvas.height - verticalOffset;

                        // scale image using proportionally smallest measurement
                        const shouldScaleWidth = availableWidth / oldWidth > availableHeight / oldHeight;
                        const newWidth = shouldScaleWidth ? availableHeight * proportion : availableWidth;
                        const newHeight = shouldScaleWidth ? availableHeight : availableWidth / proportion;

                        // apply the old canvas to the new one
                        context.drawImage(oldCanvas, 0, 0, oldWidth, oldHeight, horizontalOffset, verticalOffset, newWidth, newHeight);

                        // return the new canvas
                        return newCanvas;
                    }

                    /**
                     * Looks for a canvas hosted in a foreignObject element in the passed svg, scale it, and add it to the passed canvas
                     * @params svg: the svg that might contain a canvas in a foreignObject
                     * @param canvas: the canvas that we want to add the scatter canvas to
                     * @params scale: the scaling coefficient that we want to apply to the scatter canvas
                     */
                    function addInnerCanvasToCanvas(svg, canvas, scale, horizontalOffset, verticalOffset) {
                        const $svg = $(svg);
                        const $foreignObject = $svg.find('foreignObject'),
                            x = parseFloat($foreignObject.attr('x')),
                            y = parseFloat($foreignObject.attr('y')),
                            width = parseFloat($foreignObject.attr('width')),
                            height = parseFloat($foreignObject.attr('height'));
                        const origCanvas = $foreignObject.find('canvas').get(0);
                        canvas.getContext('2d').drawImage(origCanvas, (x + horizontalOffset) * scale, (y + verticalOffset) * scale, width * scale, height * scale);
                    }

                    /**
                     * @param chartElement: canvas or svg which contains the original chart
                     * @param canvas: the canvas that we want to add the legend to
                     * @params scale: the scaling coefficient that we want to apply to the DOM's legend
                     * @returns A promise that will resolve when the legend is added to the canvas
                     */
                    function addLegendToCanvas(chartElement, canvas, scale, verticalOffset) {
                        const d = $q.defer();
                        const $legendDiv = element.find('.legend-zone');
                        if ($legendDiv.size() === 0) {
                            d.resolve();
                        } else {
                            const legendOffset = $legendDiv.offset();
                            const wrapperOffset = element.offset();
                            const legendX = legendOffset.left - wrapperOffset.left;
                            const legendY = legendOffset.top - wrapperOffset.top;
                            const legendHeight = $legendDiv[0].clientHeight;
                            let chartHorizontalOffset = legendX * scale;
                            let chartVerticalOffset = legendY * scale;

                            switch(scope.chart.def.legendPlacement) {
                                case 'OUTER_RIGHT':
                                    chartHorizontalOffset = chartElement.clientWidth * scale;
                                    //  To avoid legend and title overlap, create a offset, as the title is centered, we need to divide the height by 2.
                                    chartVerticalOffset = verticalOffset * scale / 2;
                                    break;
                                case 'OUTER_BOTTOM':
                                    //  Consider chart and title height
                                    chartVerticalOffset = (chartElement.clientHeight + verticalOffset) * scale;
                                    break;
                                case 'OUTER_LEFT':
                                    chartVerticalOffset = verticalOffset * scale / 2;
                                    break;
                                case 'OUTER_TOP':
                                    //  We remove legendHeight to get the title height, also, as the title is centered, we need to divide the total height by 2 to center the legend too.
                                    chartVerticalOffset = (verticalOffset + legendHeight) * scale / 2;
                                    break;
                                case 'INNER_BOTTOM_LEFT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                                case 'INNER_BOTTOM_RIGHT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                                case 'INNER_TOP_LEFT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                                case 'INNER_TOP_RIGHT':
                                    chartVerticalOffset = ((legendY + (verticalOffset * scale / 2)) * scale);
                                    break;
                            }

                            CanvasUtils.htmlToCanvas($legendDiv, scale).then(function(legendCanvas) {
                                canvas.getContext('2d').drawImage(legendCanvas, chartHorizontalOffset, chartVerticalOffset, legendCanvas.width, legendCanvas.height);
                                d.resolve();
                            });
                        }
                        return d.promise;
                    }

                    // -- BEGINNING OF FUNCTION --

                    const deferred = $q.defer();
                    const chartTitle = simplified ? false : scope.chart.def.name;
                    const titleHeight = 52;

                    const horizontalOffset = 0;
                    const verticalOffset = chartTitle ? titleHeight : 0;
                    let verticalOffsetWithLegends = verticalOffset;
                    let horizontalOffsetWithLegends = horizontalOffset;

                    let legendWidth = 0;
                    let legendHeight = 0;

                    const hasLegend = !simplified && scope.chart.def.legendPlacement !== 'SIDEBAR' && ChartFeatures.canDisplayLegend(scope.chart.def.type);

                    if (hasLegend) {
                        const $legendDiv = element.find('.legend-zone');
                        switch(scope.chart.def.legendPlacement) {
                            case 'OUTER_LEFT':
                                legendWidth = $legendDiv[0].clientWidth;
                                horizontalOffsetWithLegends += legendWidth;
                                break;
                            case 'OUTER_RIGHT':
                                //  Otherwise, it's set apart to be added to the canvas' width only
                                legendWidth = $legendDiv[0].clientWidth;
                                break;
                            case 'OUTER_TOP':
                                legendHeight = $legendDiv[0].clientHeight;
                                verticalOffsetWithLegends += legendHeight;
                                break;
                            case 'OUTER_BOTTOM':
                                //  Otherwise, it's set apart to be added to the canvas' height only
                                legendHeight = $legendDiv[0].clientHeight;
                                break;
                        }
                    }

                    const isScatter = scope.chart.def.type === CHART_TYPES.SCATTER || scope.chart.def.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
                    let referenceLines;
                    const dimensions = { w: w + horizontalOffset + legendWidth + 5, h: h + verticalOffset + legendHeight + 5 };
                    // Creating a HD canvas that will "receive" the svg element
                    const scale = getCoeffToHD(dimensions.w, dimensions.h);
                    let canvas = generateCanvas(dimensions.w * scale, dimensions.h * scale);

                    if (!simplified) {
                        CanvasUtils.fill(canvas, 'white');
                    }

                    // Getting a clone SVG to inject in the canvas
                    const svg = getChartSVG();
                    const oldCanvas = document.getElementsByTagName('canvas')[0];
                    const clonedSVG = svg ? cloneSVG(svg) : undefined;

                    if (!svg) {
                        // Likely echart canvas
                        canvas = fillOldCanvasInNewCanvas(oldCanvas, canvas, horizontalOffsetWithLegends * scale, verticalOffsetWithLegends * scale);
                    } else {
                        clonedSVG.insertBefore(getChartStyleRules(true), clonedSVG.firstChild); //adding css rules

                        // For scatter, we remove the references lines because they should be draw on top on the points
                        if (isScatter) {
                            referenceLines = d3.select(clonedSVG).selectAll('.reference-line');
                            referenceLines.remove();
                        }
                        clonedSVG.setAttribute('transform', 'scale(' + scale + ')'); // scaling the svg samely as we scaled the canvas
                        if (simplified) {
                            d3.select(clonedSVG).selectAll('text').remove();
                            d3.select(clonedSVG).selectAll('.axis').remove();
                            d3.select(clonedSVG).selectAll('.hlines').remove();
                            d3.select(clonedSVG).selectAll('.vlines').remove();
                            d3.select(clonedSVG).selectAll('.legend').remove();
                        }
                        // Filling the canvas element that we created with the svg
                        const svgText = new XMLSerializer().serializeToString(clonedSVG);
                        canvg(canvas, svgText, { offsetY: verticalOffsetWithLegends, offsetX: horizontalOffsetWithLegends, ignoreDimensions: true, ignoreClear: true, renderCallback: function() {
                            $timeout(canvas.svg.stop);
                        } });
                    }


                    // In the case of scatter chart, the all chart content is already a canvas hosted in a foreignObject. Yet canvg doesn't handle foreignObjects, we'll manually copy the scatter canvas in the canvg canvas
                    if (isScatter && svg) {
                        addInnerCanvasToCanvas(svg, canvas, scale, horizontalOffsetWithLegends, verticalOffsetWithLegends);
                        // Put back only the reference lines, trash everything else.
                        if (clonedSVG && referenceLines && referenceLines.size() > 0) {
                            d3.select(clonedSVG).selectAll('g.chart').remove();
                            d3.select(clonedSVG).selectAll('foreignObject').remove();
                            referenceLines.forEach(referenceLine => clonedSVG.appendChild(referenceLine[0]));
                            const svgText = new XMLSerializer().serializeToString(clonedSVG);
                            canvg(canvas, svgText, { offsetY: verticalOffsetWithLegends, offsetX: horizontalOffsetWithLegends, ignoreDimensions: true, ignoreClear: true, renderCallback: function() {
                                $timeout(canvas.svg.stop);
                            } });
                        }
                    }

                    // Adding chart's title
                    if (chartTitle && !noTitle) {
                        addTitleToCanvas(canvas, chartTitle, titleHeight, scale);
                    }

                    // Adding chart's legend
                    if (hasLegend) {
                        addLegendToCanvas(svg || oldCanvas, canvas, scale, verticalOffset).then(() => deferred.resolve(canvas));
                    } else {
                        deferred.resolve(canvas);
                    }

                    return deferred.promise;
                };

                scope.exportBoxPlots = (thumbnail) => $q(resolve => {
                    const container = element.find('.boxplots-container');
                    const svg1 = container.find('svg.noflex');
                    const svg2 = container.find('div.flex.oa > svg');
                    const title = scope.chart.def.name;
                    const titleHeight = 52;
                    let verticalOffset = title ? titleHeight : 0;
                    const options = {
                        setup: (cloneSvg) => {
                            if (thumbnail) {
                                // Thicker strokes
                                cloneSvg.style.strokeWidth = 6;
                            } else {
                                cloneSvg.insertBefore(getChartStyleRules(), cloneSvg.firstChild);
                            }
                        },
                        filters: thumbnail ? ['.axis', '.hline'] : null
                    };

                    const allCanvas = [CanvasUtils.svgToCanvas(svg1.get(0), svg1.width(), svg1.height(), options)];
                    if (scope.chart.def.boxplotBreakdownDim.length) {
                        allCanvas.push(CanvasUtils.svgToCanvas(svg2.get(0), svg2.width(), svg2.height(), options));
                    }
                    $q.all(allCanvas).then(subcanvas => {
                        const canvas = document.createElement('canvas');
                        let horizontalOffset = 0;
                        let legendWidth = 0;
                        let legendHeight = 0;
                        const legendOffset = { top: 0, left: 0 };
                        const hasLegend = scope.chart.def.legendPlacement !== 'SIDEBAR' && ChartFeatures.canDisplayLegend(scope.chart.def.type) && !thumbnail;
                        function getCanvasProps(canvasToCheck, prop) {
                            if (canvasToCheck == null) {
                                return 0;
                            }
                            return canvasToCheck[prop];
                        }
                        if (hasLegend) {
                            const $legendDiv = container.find('.legend-zone');
                            switch(scope.chart.def.legendPlacement) {
                                case 'OUTER_LEFT':
                                    //  We add the legend width to the offset directly
                                    horizontalOffset += $legendDiv[0].clientWidth;
                                    legendOffset.top = verticalOffset;
                                    break;
                                case 'OUTER_RIGHT':
                                    //  Otherwise, it's set apart to be added to the canvas' width only
                                    legendWidth = $legendDiv[0].clientWidth;
                                    legendOffset.left = subcanvas[0].width + getCanvasProps(subcanvas[1], 'width');
                                    legendOffset.top = verticalOffset;
                                    break;
                                case 'OUTER_TOP':
                                    legendOffset.top = verticalOffset;
                                    //  We add the legend height to the offset directly
                                    verticalOffset += $legendDiv[0].clientHeight;
                                    break;
                                case 'OUTER_BOTTOM':
                                    //  Otherwise, it's set apart to be added to the canvas' height only
                                    legendHeight = $legendDiv[0].clientHeight;
                                    legendOffset.top = subcanvas[0].height + verticalOffset;
                                    break;
                                case 'INNER_BOTTOM_LEFT':
                                    legendOffset.top = subcanvas[0].height - $legendDiv[0].clientHeight;
                                    legendOffset.left = 30;
                                    break;
                                case 'INNER_BOTTOM_RIGHT':
                                    legendOffset.top = subcanvas[0].height - $legendDiv[0].clientHeight;;
                                    legendOffset.left = subcanvas[0].width + getCanvasProps(subcanvas[1], 'width') - $legendDiv[0].clientWidth;
                                    break;
                                case 'INNER_TOP_RIGHT':
                                    legendOffset.top = verticalOffset;
                                    legendOffset.left = subcanvas[0].width + getCanvasProps(subcanvas[1], 'width') - $legendDiv[0].clientWidth;
                                    break;
                                case 'INNER_TOP_LEFT':
                                    legendOffset.top = verticalOffset;
                                    legendOffset.left = 30;
                                    break;
                            }
                        }
                        const dimensions = { w: subcanvas[0].width + getCanvasProps(subcanvas[1], 'width') + horizontalOffset + legendWidth, h: subcanvas[0].height + verticalOffset + legendHeight };
                        const scale = getCoeffToHD(dimensions.w, dimensions.h);
                        canvas.setAttribute('width', dimensions.w);
                        canvas.setAttribute('height', dimensions.h);
                        if (!thumbnail) {
                            CanvasUtils.fill(canvas, 'white');
                        }
                        const ctx = canvas.getContext('2d');
                        ctx.drawImage(subcanvas[0], horizontalOffset, verticalOffset);
                        // 14 is a magic value to glue the first and second canvas to avoid cut in h lines
                        if (subcanvas.length > 1) {
                            ctx.drawImage(subcanvas[1], subcanvas[0].width + horizontalOffset - 14, verticalOffset + scale);
                        }

                        if (title && !thumbnail) {
                            addTitleToCanvas(canvas, title, titleHeight, 1);
                        }

                        if (hasLegend) {
                            CanvasUtils.htmlToCanvas(container.find('.legend-zone'), 1).then(function(legendCanvas) {
                                canvas.getContext('2d').drawImage(legendCanvas, legendOffset.left, legendOffset.top, legendCanvas.width, legendCanvas.height);
                                resolve(canvas);
                            });
                        } else {
                            resolve(canvas);
                        }
                    });
                });

                scope.$watch('response', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    if (!scope.response.hasResult) {
                        return;
                    }
                    $timeout(() => {
                        scope.isInitialDrawReady = true;
                        redraw({ updateThumbnail: true });
                    });
                });
            }
        };
    });
})();
