(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('AnimatedChartsUtils', AnimatedChartsUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/common/animation.js
     */
    function AnimatedChartsUtils($interval, ChartFormatting) {
        const unwatchers = {},
            intervals = {};

        return {
            /**
             * Setup chartHandler.animation (used by the animation widget) for the given chart
             * @param {$scope} chartHandler
             * @param {ChartTensorDataWrapper} chartData
             * @param {ChartDef} chartDef
             * @param {function} drawFrame: drawing callback
             */
            initAnimation: function(chartHandler, chartData, chartDef, drawFrame, ignoreLabels = new Set()) {
                if (unwatchers[chartHandler.$id]) {
                    unwatchers[chartHandler.$id]();
                    delete unwatchers[chartHandler.$id];
                }

                if (intervals[chartHandler.$id]) {
                    $interval.cancel(intervals[chartHandler.$id]);
                    delete intervals[chartHandler.$id];
                }

                const animation = chartHandler.animation;

                animation.labelify = function(label) {
                    return ChartFormatting.getForOrdinalAxis(label);
                };
                animation.labels = chartData.getAxisLabels('animation', ignoreLabels);

                animation.playing = false;

                animation.drawFrame = function(frameIdx) {
                    /*
                     * Put back currentFrame to zero when necessary. For instance, after unselecting
                     * "Group extra values in a 'Others' category".
                     */
                    animation.currentFrame = animation.currentFrame >= animation.labels.length ? 0 : frameIdx;
                };
                animation.chartData = chartData;

                animation.hasNext = function() {
                    return animation.currentFrame < animation.labels.length - 1;
                };

                animation.play = function() {
                    if (animation.playing) {
                        return;
                    }

                    if (animation.currentFrame === animation.labels.length - 1) {
                        animation.currentFrame = 0;
                    }
                    animation.playing = true;
                    intervals[chartHandler.$id] = $interval(function() {
                        animation.drawFrame((animation.currentFrame + 1) % animation.labels.length);
                        if (!chartDef.animationRepeat && !animation.hasNext()) {
                            animation.pause();
                        }
                    }, (chartDef.animationFrameDuration || 3000));
                };

                animation.dimension = chartDef.animationDimension[0];

                animation.pause = function() {
                    animation.playing = false;
                    $interval.cancel(intervals[chartHandler.$id]);
                };

                unwatchers[chartHandler.$id] = chartHandler.$watch('animation.currentFrame', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    drawFrame(animation.labels[nv].$tensorIndex);
                });

                chartHandler.$watch('chart.def.animationFrameDuration', function(nv) {
                    if (!nv) {
                        return;
                    }
                    if (animation.playing) {
                        animation.pause();
                        animation.play();
                    }
                });
            },

            unregisterAnimation: function(chartHandler) {
                if (unwatchers[chartHandler.$id]) {
                    unwatchers[chartHandler.$id]();
                    delete unwatchers[chartHandler.$id];
                }

                if (intervals[chartHandler.$id]) {
                    $interval.cancel(intervals[chartHandler.$id]);
                    delete intervals[chartHandler.$id];
                }
            },

            getAnimationCoord(animation){
                return animation.labels ? animation.labels[animation.currentFrame].$tensorIndex : animation.currentFrame;
            },

            getAnimationContext: function(animationDimension, animation) {
                return {
                    animation: animationDimension && animation.currentFrame || 0,
                    numberOfFrames: animationDimension && animation.labels.length || 1
                };
            }
        };
    }

})();
