(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * Generic helpers for svg manipulation
     */
    app.factory('SVGUtils', function(D3ChartAxes, ColorUtils, ValuesInChartOverlappingStrategy, MATCH_CHART_COLOR, ChartFeatures, CHART_COLOR_AUTO) {

        /**
         * Quadtree-based label collision detection handler for a chart.
         *
         * This system helps manage and avoid label collisions by leveraging a quadtree structure,
         * which efficiently handles spatial queries. Labels are represented as bounding boxes,
         * and the quadtree facilitates checking overlaps by dividing the chart area into smaller regions.
         *
         * - `addBoundingBoxesToQuadTree`: Adds new bounding boxes (representing labels) to the quadtree.
         *
         * - `checkOverlaps`: Traverses the quadtree to check if the new label's bounding box overlaps
         *    with any already placed labels.
         *
         */
        class LabelCollisionDetectionHandler {
            constructor(chartBase) {
                this.maxLabelWidth = 0;
                this.maxLabelHeight = 0;
                this.bboxes = [];
                this.quadtree = d3.geom.quadtree()
                    .x(d => d.x)
                    .y(d => d.y)
                    .extent([
                        [-chartBase.margins.top, -chartBase.margins.left],
                        [chartBase.vizWidth + chartBase.margins.right, chartBase.vizHeight + chartBase.margins.bottom]
                    ])([]);
            }

            _addBoundingBoxToQuadTree(bbox) {
                if (!bbox.width || !bbox.height) {
                    // don't add invalid bounding boxes
                    return;
                }

                try {
                    this.quadtree.add(bbox);
                } catch (error) {
                    // in some cases when bbox is outside the quadtree extent, it throws an error
                    return;
                }
                this.bboxes.push(bbox);
                this.maxLabelWidth = Math.max(this.maxLabelWidth, bbox.width);
                this.maxLabelHeight = Math.max(this.maxLabelHeight, bbox.height);
            }

            _isOverlapping(rect1, rect2) {
                return !(rect1.x + rect1.width < rect2.x ||
                    rect2.x + rect2.width < rect1.x ||
                    rect1.y + rect1.height < rect2.y ||
                    rect2.y + rect2.height < rect1.y);
            }

            _checkOverlaps(newRect) {
                let overlaps = false;
                this.quadtree.visit((node, x0, y0, x1, y1) => {
                    if (overlaps) {
                        return true;
                    }
                    if (!node.length) {
                        do {
                            const d = node.point;
                            if (d && this._isOverlapping(d, newRect)) {
                                overlaps = true;
                                return true;
                            }
                            // eslint-disable-next-line no-cond-assign
                        } while (node = node.next);
                    }
                    return x0 > newRect.x + newRect.width || x1 + this.maxLabelWidth < newRect.x ||
                            y0 > newRect.y + newRect.height || y1 + this.maxLabelHeight < newRect.y;
                });
                return overlaps;
            }

            checkOverlaps(rectangles) {
                return rectangles.some(rect => this._checkOverlaps(rect));
            }

            addBoundingBoxesToQuadTree(bboxes) {
                bboxes.forEach(bbox => this._addBoundingBoxToQuadTree(bbox));
            }

            // just used for debugging purpose, to color all bounding boxes contained by the quadtree for collision detection
            colorBoundingBoxes(svgNode) {
                this.bboxes.forEach(bbox => {
                    svgNode.append('rect')
                        .attr('fill', 'blue')
                        .attr('opacity', 0.2)
                        .attr('x', bbox.x)
                        .attr('y', bbox.y)
                        .attr('width', bbox.width)
                        .attr('height', bbox.height);
                });
            }
        };

        const computeMatchChartColor = (color, chartColor, opacity) => {
            if (color !== MATCH_CHART_COLOR) {
                return color;
            }
            const hexChartColor = ColorUtils.isValidHexColor(chartColor) ? chartColor : ColorUtils.RGBAToHex(ColorUtils.strToRGBA(chartColor));
            return _.isNil(opacity) ? chartColor : ColorUtils.getHexWithAlpha(hexChartColor, opacity);
        };

        function getTextNodeContentFromSubText(i) {
            const parent = d3.select(this.parentNode);
            let parentContent = parent.datum();
            let parentColumnIdx = i;
            if (Array.isArray(parentContent) && i !== undefined) {
                // compute the column index from the index in the texts list
                parentColumnIdx = 0;
                let nbTexts = 0;
                for (let index = 0; index < parentContent.length && nbTexts <= i; index++) {
                    const columnData = parentContent[index];
                    nbTexts += columnData.textsElements.length;
                    if (nbTexts > i) {
                        parentColumnIdx = index;
                    }
                }
                parentContent = parentContent[parentColumnIdx];
            }

            return { parentContent, parentColumnIdx };
        }

        const utils = {
            /**
             * (!) This funcion previously was in static/dataiku/js/simple_report/chart_view_commons.js
             *
             * Prevents chart to overlap its x and y axes by adding a clipPath
             *
             * @param {ChartBase}   chartBase
             * @param {Object}      g           SVG element to append clipPath to
             * @param {Object}      wrappers    SVG element to apply clipPath to
             */
            clipPaths: function(chartBase, g, wrappers) {
                const defs = g.append('defs');
                const CLIP_PATH_ID = 'chart-clip-path-' + generateUniqueId();

                /*
                 * Find y position of clip path based on axis positions (whether x is on top or bottom)
                 * if x axis is on bottom : don't crop margin-top, to allow displayValues
                 */
                const isAllNegativeRange = !(D3ChartAxes.getCurrentAxisExtent(chartBase.yAxes[0]).find(val => val > 0));
                const yPosition = isAllNegativeRange ? 0 : -chartBase.margins.top;

                // Add a bit of margin to handle smoothing mode.
                defs.append('clipPath')
                    .attr('id', CLIP_PATH_ID)
                    .append('rect')
                    .attr('y', yPosition)
                    .attr('width', chartBase.vizWidth)
                    .attr('height', chartBase.vizHeight + chartBase.margins.top);

                wrappers.attr('clip-path', 'url(#' + CLIP_PATH_ID + ')');
                wrappers.style('-webkit-clip-path', 'url(#' + CLIP_PATH_ID + ')');

            },

            drawLabels(drawContext, chartPrefix) {
                // first draw final labels
                const labelGroups = this.drawLabelsRaw(drawContext, chartPrefix, false);

                // then draw alternate labels, without overlapping detection with other measures
                this.drawLabelsRaw(drawContext, chartPrefix, true);

                // both "drawLabelsRaw" calls return the same selection, as alternate labels are drawn in the same label group as the main ones
                return labelGroups;
            },

            drawLabelsRaw(drawContext, chartPrefix, onHoverMode = false) {

                if (!drawContext.overlappingStrategy) {
                    return;
                }

                let labelClass = drawContext.isTotals ? 'total' : 'value';
                if (onHoverMode) {
                    labelClass = `${labelClass}-alt`;
                }
                const labelGroupsClassNames = [`${chartPrefix}-labels`, 'chart-value-labels'];

                const labelGroups = drawContext.node.selectAll(`g.${labelGroupsClassNames.join('.')}`).data(drawContext.data);

                labelGroups.enter().append('g')
                    .attr('class', labelGroupsClassNames.join(' '));
                if (drawContext.transform) {
                    labelGroups.attr('transform', drawContext.transform);
                    if (drawContext.transformations) {
                        // add the current translation to the data
                        labelGroups.each(function(d, i) {
                            d.translate = { x: drawContext.transformations.x(d, i), y: drawContext.transformations.y(d, i) };
                        });
                    }
                }
                labelGroups.exit().remove();

                const rectTexts = labelGroups.selectAll(`text.${labelClass}`).data(function(d) {
                    return Array.isArray(d) ? [...d] : [d];
                });

                rectTexts.enter().append('text')
                    .attr('class', labelClass)
                    .attr('text-anchor', 'middle')
                    // we set dominant-baseline to middle because other values are not supported by export as png
                    .attr('dominant-baseline', 'middle')
                    .style('pointer-events', 'none');
                rectTexts.exit().remove();

                if (onHoverMode) {
                    // "hovering" labels hidden by default
                    rectTexts.attr('opacity', 0);
                }

                const subTexts = rectTexts.selectAll(`tspan.${labelClass}`).data(function(d) {
                    return d.textsElements;
                });
                subTexts.enter().append('tspan')
                    .attr('class', labelClass);

                subTexts.exit().remove();

                subTexts
                    .text(function(d) {
                        const text = drawContext.getLabelText(d);
                        return text;
                    })
                    .attr('fill', function(d) {
                        const formattingOptions = d.valuesInChartDisplayOptions.textFormatting;

                        const chartColor = drawContext.colorScale(d.color + d.measure);
                        const backgroundColor = drawContext.hasBackground ? chartColor : '#fff';

                        let fontColor = computeMatchChartColor(formattingOptions.fontColor, chartColor, drawContext.opacity);
                        if (fontColor === CHART_COLOR_AUTO) {
                            const foregroundColor = formattingOptions.hasBackground ? computeMatchChartColor(formattingOptions.backgroundColor, chartColor, drawContext.opacity) : null;
                            const blendColor = ColorUtils.getBlendedColor(foregroundColor, backgroundColor);
                            fontColor = ColorUtils.getFontContrastColor(blendColor, drawContext.theme && drawContext.theme.generalFormatting.fontColor);
                        }

                        return fontColor;
                    })
                    .style('font-size', function(d) {
                        const formattingOptions = d.valuesInChartDisplayOptions.textFormatting;
                        return `${formattingOptions.fontSize}px`;
                    })
                    .each(function(d) {
                        const bbox = this.getBoundingClientRect();
                        // store children bounding boxes for easier access from the parent
                        d.width = bbox.width;
                        d.height = bbox.height;
                    })
                    .attr('dy', function(d) {
                        if (!this.previousSibling) {
                            // first text element, no offset
                            return 0;
                        }
                        const previousNodeDatum = d3.select(this.previousSibling).datum();
                        // to place the current text elem on a new line, we add the right y offset, which takes into account the previous text element font size and the current element height (as we use the 'middle' dominant-baseline)
                        const previousNodeFontSize = previousNodeDatum.valuesInChartDisplayOptions.textFormatting.fontSize;
                        const currentNodefontSize = d.valuesInChartDisplayOptions.textFormatting.fontSize;
                        return previousNodeFontSize / 2 + d.height - currentNodefontSize / 2;
                    });

                rectTexts
                    .filter(function(d) {
                        return !d.isInvalid;
                    })
                    .each(function(d, i, n) {
                        d.height = d.textsElements.reduce((acc, b) => acc + b.height, 0);
                        d.width = d.textsElements.reduce((acc, b) => Math.max(acc, b.width), 0);
                        if (drawContext.getExtraData) {
                            const extraData = drawContext.getExtraData.bind(this)(d, i, n, onHoverMode);
                            if (extraData) {
                                Object.assign(d, extraData);
                            }
                        }
                        const isOverlap = onHoverMode ? d.isOverlapAlt : d.isOverlap;
                        if ((d.valuesInChartDisplayOptions && !d.valuesInChartDisplayOptions.displayValues)
                                || (drawContext.overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO && isOverlap) || d.isInvalid) {
                            d3.select(this).attr('visibility', 'hidden');
                            d.width = 0;
                            d.height = 0;
                            // propagate to sub texts nodes
                            d.textsElements.forEach((t) => {
                                t.width = 0;
                                t.height = 0;
                            });
                        } else {
                            d3.select(this).attr('visibility', null);
                        }
                    })
                    .transition()
                    .duration(200)
                    .ease('easeOutQuad')
                    .attr('x', function(d, i) {
                        const x = drawContext.getLabelXPosition(d, i);
                        d.textX = x;
                        return x;
                    })
                    .attr('y', function(d, i) {
                        // this computes the top coordinate of the text
                        let y = drawContext.getLabelYPosition(d, i);
                        // but as we use the dominant-baseline "middle", we have to adjust the y position passed to the svg element
                        const firstSubText = d.textsElements[0];
                        const firstSubTextHeight = firstSubText.height || 0;
                        const firstSubTextFontSize = firstSubText.valuesInChartDisplayOptions.textFormatting.fontSize;
                        const offset = (firstSubTextHeight / 2 + (firstSubTextHeight - firstSubTextFontSize) / 2);
                        y += offset;
                        d.textY = y;
                        return y;
                    });

                // now set sub texts to the same X position, to have them correctly aligned
                subTexts
                    .attr('x', function() {
                        const parentX = d3.select(this.parentNode).datum().textX;
                        return parentX;
                    });

                if (drawContext.isTotals) {
                    subTexts.attr('font-weight', 500);
                }

                this.drawLabelsBackground(labelGroups, subTexts, drawContext.getBackgroundXPosition, drawContext.getBackgroundYPosition,
                    drawContext.backgroundXMargin, drawContext.colorScale, labelClass, drawContext.opacity, onHoverMode);

                return labelGroups;
            },

            drawLabelsBackground(labelGroups, subTexts, getBackgroundXPosition, getBackgroundYPosition, xMargin, colorScale, labelClass, opacity, onHoverMode) {

                const rectBackgrounds = labelGroups.selectAll(`rect.background-${labelClass}`).data(function(d) {
                    return Array.isArray(d) ? _.flatten(d.map(d => (d.textsElements))) : d.textsElements;
                });

                rectBackgrounds.enter().insert('rect', `text.${labelClass}`)
                    .attr('class', function(d) {
                        const backgroundClasses = [`background-${labelClass}`, `aggregation-${d.aggregationIndex}`];
                        return backgroundClasses.join(' ');
                    })
                    .attr('fill', function(d) {
                        const formattingOptions = d.valuesInChartDisplayOptions.textFormatting;
                        const chartColor = colorScale(d.color + d.measure);
                        return computeMatchChartColor(formattingOptions.backgroundColor, chartColor, opacity);
                    })
                    .style('pointer-events', 'none');
                rectBackgrounds.exit().remove();
                if (onHoverMode) {
                    // "hovering" labels hidden by default
                    rectBackgrounds.attr('opacity', 0);
                }
                rectBackgrounds
                    .attr('stroke-width', 0)
                    .each(function(d) {
                        const formattingOptions = d.valuesInChartDisplayOptions.textFormatting;
                        if (!formattingOptions.hasBackground) {
                            d3.select(this).attr('visibility', 'hidden');
                            d.hideBackground = true;
                        } else {
                            d3.select(this).attr('visibility', null);
                            d.hideBackground = false;
                        }
                    })
                    .transition()
                    .duration(200)
                    .ease('easeOutQuad')
                    .attr('x', function(d, i) {
                        const { parentContent, parentColumnIdx } = getTextNodeContentFromSubText.bind(this)(i);
                        // the sub texts have the same x position as parent text elem
                        return getBackgroundXPosition({ ...parentContent, ...d }, parentColumnIdx);
                    })
                    .attr('y', function(d, i) {
                        const { parentContent, parentColumnIdx } = getTextNodeContentFromSubText.bind(this)(i);
                        return getBackgroundYPosition({ ...parentContent, aggregationIndex: d.aggregationIndex }, parentColumnIdx);
                    })
                    .attr('width', function(d) {
                        if (!d.width || d.hideBackground) {
                            return 0;
                        }
                        return d.width + xMargin * 2;
                    })
                    .attr('height', function(d) {
                        if (!d.height || d.hideBackground) {
                            return 0;
                        }
                        return d.height;
                    });

            },

            initLabelCollisionDetection(chartBase) {
                return new LabelCollisionDetectionHandler(chartBase);
            },

            getLabelSubTexts(chartDef, measureIdx, valuesInChartDisplayOptions, isTotal) {
                const result = [];

                if (!valuesInChartDisplayOptions
                    || !valuesInChartDisplayOptions.addDetails
                    || !valuesInChartDisplayOptions.additionalMeasures
                    || !valuesInChartDisplayOptions.additionalMeasures.length
                    // for now we don't override values for 100% stacked charts
                    || chartDef.variant === 'stacked_100') {
                    // standard case, no custom content
                    result.push({
                        aggregationIndex: measureIdx,
                        valuesInChartDisplayOptions: valuesInChartDisplayOptions || chartDef.valuesInChartDisplayOptions
                    });
                } else {
                    const additionalMeasures = valuesInChartDisplayOptions.additionalMeasures;

                    if (isTotal) {
                        const offsetIdx = chartDef.genericMeasures.length + chartDef.tooltipMeasures.length;
                        additionalMeasures.forEach((additionalMeasure, idx) => {
                            const finalIdx = offsetIdx + idx;
                            result.push({
                                aggregationIndex: finalIdx,
                                valuesInChartDisplayOptions: additionalMeasure.valuesInChartDisplayOptions || valuesInChartDisplayOptions || chartDef.valuesInChartDisplayOptions
                            });
                        });
                    } else {
                        // generic measure with custom content
                        const hasTotalsAdditionalMeasures = ChartFeatures.canDisplayTotalValues(chartDef) && !!chartDef.stackedColumnsOptions
                            && !!chartDef.stackedColumnsOptions.totalsInChartDisplayOptions
                            && !!chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures;
                        const offsetIdx = chartDef.genericMeasures.length
                         + chartDef.tooltipMeasures.length
                         + (hasTotalsAdditionalMeasures ? chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures.length : 0)
                         + chartDef.genericMeasures.slice(0, measureIdx).reduce((acc, measure) => (acc + measure.valuesInChartDisplayOptions && measure.valuesInChartDisplayOptions.additionalMeasures) ? measure.valuesInChartDisplayOptions.additionalMeasures.length : 0, 0);
                        additionalMeasures.forEach((additionalMeasure, idx) => {
                            const finalIdx = offsetIdx + idx;
                            result.push({
                                aggregationIndex: finalIdx,
                                valuesInChartDisplayOptions: additionalMeasure.valuesInChartDisplayOptions || valuesInChartDisplayOptions || chartDef.valuesInChartDisplayOptions
                            });
                        });
                    }
                }

                return result;
            },

            getRectanglesFromPosition(position, textElement, defaultFontSize) {
                let currentYPosition = position.y;
                const res = [];
                textElement.textsElements.forEach((t) => {
                    // fallback value
                    let fontSize = defaultFontSize;
                    // if available take font size on the sub text element (additional measure level)
                    if (t.valuesInChartDisplayOptions && t.valuesInChartDisplayOptions.textFormatting) {
                        fontSize = t.valuesInChartDisplayOptions.textFormatting.fontSize;
                    } else if (textElement.valuesInChartDisplayOptions && textElement.valuesInChartDisplayOptions.textFormatting) {
                        // otherwise on the text parent (so at the 'root' measure level)
                        fontSize = textElement.valuesInChartDisplayOptions.textFormatting.fontSize;
                    }
                    // position the top of the detection rectangle at the top of the text itself, not the text element
                    const subTextElementTop = currentYPosition;
                    currentYPosition += (t.height - fontSize) / 2;
                    res.push({
                        x: position.x - t.width / 2,
                        y: currentYPosition,
                        width: t.width,
                        // for each sub text, we use the font size instead of the bboxHeight, as it adds some extra vertical margin. We try to detect collision only on the text itself
                        height: fontSize
                    });
                    // for the next iteration, place the cursor at the top of the next text element
                    currentYPosition = subTextElementTop + t.height;
                });

                return res;
            },

            getSubTextYPosition(d, parentTextTopYPosition) {
                let y = parentTextTopYPosition;
                // that's the top of the text container, we have to compute the right offset for each line
                for (let index = 0; index < d.textsElements.length && d.textsElements[index].aggregationIndex !== d.aggregationIndex; index++) {
                    const textElem = d.textsElements[index];
                    y += textElem.height;
                }
                return y;
            }
        };

        return utils;
    });

})();
