import logging

from mcp.types import TextContent

from dataiku.llm.agent_tools import BaseAgentTool
from dataiku.llm.agent_tools.mcp.client import FastMCPClient


logger = logging.getLogger(__name__)


class GenericStdioMCPClient(BaseAgentTool):
    def __init__(self):
        super().__init__()
        logger.info("MCP Client __init__")
        self.client = None

    def set_config(self, config, plugin_config):
        logger.info("MCP Client set_config: command %s, args %s", config.get("command", ""), config["args"])
        self.client = FastMCPClient(config)

    def get_descriptor(self, _):
        logger.info("Get descriptor")

        descriptor = {
            "multiple": True,
            "description": "", # no global description, each subtool has one
            "subtools" : []
        }
        result = self.client.list_tools()
        logger.info("Got subtools: %s" % result)

        for tool in result:
            # Shortcut MCP explodes here because one of the tool has a property with multiple types
            # {'type': ['number', 'null'], 'description': 'The epic id of the epic the story belongs to, or null to unset'}
            # this sounds valid according to https://json-schema.org/draft/2020-12/draft-bhutton-json-schema-01#name-assertions-and-instance-pri
            # but is hard to implement on the Java side
            for name, property in tool.inputSchema["properties"].items():
                if property.get("type") is None:
                    logger.warning("No type set for property '%s' of tool '%s'", name, tool.name)
                elif isinstance(property["type"], list):
                    logger.warning("Unsupported array type '%s' for property '%s' of tool '%s'", property["type"], name, tool.name)
                    property["type"] = property["type"][0]
            descriptor["subtools"].append({
                "name": tool.name,
                "description": tool.description.strip() if tool.description else "",
                "inputSchema": tool.inputSchema
            })

        return descriptor

    def invoke(self, input, trace):
        logger.info("MCP invoking tool: %s" % input)

        if not "subtoolName" in input:
            raise Exception("missing subtool name, please set 'subtoolName' in the payload")

        with trace.subspan("PYTHON_AGENT_MCP_SUBTOOL_CALL") as subtool_subspan:
            subtool_subspan.attributes["subtool_name"] = input["subtoolName"]
            subtool_subspan.attributes["subtool_args"] = input["input"]

            result = self.client.call_tool(input["subtoolName"], input["input"])
            logger.info("Got tool invocation result: %s" % result)

        if len(result.content) == 1:
            content = result.content[0]
            if isinstance(content, TextContent):
                tool_output = {
                    "text": content.text
                }
            else:
                raise RuntimeError(f"Cannot handle this type of content {type(content).__name__}")
        else:
            tool_output = []
            for content in result.content:
                if isinstance(content, TextContent):
                    tool_output.append({
                        "text": content.text
                    })
                else:
                    logger.warning("ignoring unsupported response content %s", type(content).__name__)

        return {
            "output": tool_output
        }
