import subprocess
import sys
import os
import textwrap


def log_nvidia_smi(display_topo_info, logger, additional_message=None):
    if additional_message is not None:
        logger.info(additional_message)
    command = "nvidia-smi"
    if display_topo_info:
        command += " topo -m"
    try:
        with subprocess.Popen(command, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE) as process:
            try:
                stdout, stderr = process.communicate(timeout=10)
            except subprocess.TimeoutExpired:
                logger.warning("Timeout expired for command " + command)
                process.kill()
                stdout, stderr = process.communicate()
            stdout = stdout.decode('utf-8').strip()
            stderr = stderr.decode('utf-8').strip()
            if len(stdout) > 0:
                logger.info(stdout)
            if len(stderr) > 0:
                logger.warning(stderr)
    except Exception as e:
        logger.warning("Failed to run 'nvidia-smi' command", e)


def log_shm_size(logger):
    try:
        if sys.platform != 'linux':
            # We don't care because LLM inference is Linux-only
            logger.info("Not running on Linux, cannot get shared memory size")
            return

        statvfs = os.statvfs('/dev/shm')
        shm_size = statvfs.f_frsize * statvfs.f_blocks
        logger.info("Shared memory size: %s bytes"%shm_size)

        # K8S sets shm size to 64MB by default, it's easy to forget to increase it
        if shm_size < 1e9:
            logger.warning(textwrap.dedent("""
                Shared memory size seems small. This might cause issues for multi-GPU LLM inference with vLLM.
                Make sure setting "Increase shared memory size" is enabled in the container execution configuration.
            """))
    except Exception as e:
        logger.exception("Failed to get shared memory size", e)
