import base64
import logging
import re

from multiprocessing.pool import ThreadPool
from typing import List, Union

from dataiku.base.utils import safe_unicode_str, RaiseWithTraceback
from dataiku.doctor.deep_learning.preprocessing import DummyFileReader

logger = logging.getLogger(__name__)

MAX_CONCURRENT_IMAGES_DOWNLOADS = 8

class ImageLoader:
    def __init__(self, fail_if_invalid_path: bool, file_reader: DummyFileReader) -> None:
        self._file_reader = file_reader
        self._fail_if_invalid_path = fail_if_invalid_path
        self._base64_regexp = re.compile("^([A-Za-z0-9+/]{4})*([A-Za-z0-9+/]{4}|[A-Za-z0-9+/]{3}=|[A-Za-z0-9+/]{2}==)$")
        self._base64_min_authorized_length = 200

    def load_images(self, img_paths: List[str]) -> List[Union[str, None]]:
        """
        Load the given image paths within a managed folder to base 64 and returns them
        """
        with ThreadPool(MAX_CONCURRENT_IMAGES_DOWNLOADS) as pool:
            return pool.map(self._load_b64_image, img_paths)

    def _load_b64_image(self, img_path: str) -> Union[str, None]:
        """
        Try to open the image from its relative path in the managed folder & convert it to the base 64 format.
        Raises an exception if an error is encountered or returns None if bad images path are passed.
        """
        # Usually happens when the image is passed in the base 64 format directly from the front-end such as the what-if page
        if self._is_base64_file(img_path):
            return img_path

        try:
            with RaiseWithTraceback(u"Invalid image path '{}'. Could not process the file".format(safe_unicode_str(img_path))):
                with self._file_reader.read(img_path) as img_file:
                    return base64.b64encode(img_file.read()).decode("utf-8")
        except Exception as e:
            if self._fail_if_invalid_path:
                raise e
            logger.warning("Image path invalid: '{}', imputing it".format(safe_unicode_str(img_path)))
            return None

    def _is_base64_file(self, file_id: str):
        return len(file_id) > self._base64_min_authorized_length and self._base64_regexp.match(file_id)
