(function () {
'use strict';

var app = angular.module('dataiku.admin', []);

app.controller("AdminGeneralSettingsController", function (
    $scope,
    $state,
    $stateParams,
    $timeout,
    $q,
    translate,
    DataikuAPI,
    WT1,
    ActivityIndicator,
    TopNav,
    CodeMirrorSettingService,
    TaggingService,
    FutureProgressModal,
    Dialogs,
    $rootScope,
    $filter,
    GlobalProjectActions,
    $anchorScroll,
    DeployerUtils,
    FutureWatcher,
    CreateModalFromTemplate,
    Assert,
    localStorageService,
    ProjectFolderService,
    PathUtils,
    PromiseService,
    $window,
    FeatureFlagsService,
    $http
) {
    if ($state.is('admin.general')) {
        $state.go('admin.general.themes');
    }

    // to be used by sub-controllers
    $scope.setErrorInGeneralSettingsControllerScope = setErrorInScope.bind($scope);

    $scope.hasUrlSuffix = DeployerUtils.hasUrlSuffix;

    $scope.httpMethods = ['GET', 'POST', 'OPTIONS', 'PUT', 'DELETE', 'HEAD'];

    $scope.virtualWebAppBackendSettingsModes = [{id:"USE_DEFAULT", label:"Run as local processes"}, {id:"EXPLICIT", label:"Run in container"}];

    $scope.globalVariables = {}
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.welcomeEmailTest = {
        testing: false,
        success: false,
        error: null,
    };
    $scope.welcomeEmailTest.recipient = $scope.welcomeEmailTest.recipient || $rootScope.appConfig.user.email || "";

    $scope.testWelcomeEmail = function() {
        $scope.welcomeEmailTest.testing = true;
        $scope.welcomeEmailTest.success = false;
        $scope.welcomeEmailTest.error = null;
        DataikuAPI.admin.testWelcomeEmail(
            $scope.welcomeEmailTest.recipient,
            $scope.generalSettings.welcomeEmailSettings,
            $scope.generalSettings.notifications,
            $scope.generalSettings.studioMailAddress,
            $scope.channels,
        )
            .success(function(result) {
                $scope.welcomeEmailTest.success = true;
            })
            .error(function(error) {
                $scope.welcomeEmailTest.error = error;
            })
            .finally(function() {
                $scope.welcomeEmailTest.testing = false;
            });
    };


    // ** Default folder rules stuff

    // all users virtual group - for catch-all / fall back folder rule
    const ALL_USERS_GROUP_DISPLAY = 'All Users';
    const ALL_USERS_GROUP_INTERNAL = '$$ALL_USERS$$';

    $scope.getProjectFolderName = (item) => item.name;

    $scope.browseFolder = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination)
            .catch(setErrorInScope.bind($scope)));
    };

    // Cache a promise for root folder as we use it for every new folder rule
    const ROOT_FOLDER_PROMISE = ProjectFolderService.getFolderWithPath('ROOT');

    function setFolderOnRuleFromPromise(rule, promise) {
        promise.then((folder) => {
            if (rule.folderId === "ROOT") {
                folder.pathElts = '/';
            }
            rule.$folder = folder;
        }).catch(setErrorInScope.bind($scope));
    }

    // Adds the $folder object which gives the folder-path-input enough info to start browsing
    // and gives us the path info to show in the form (for 'ROOT' it is just '/')
    // If the folder was deleted we display "(missing)" but allow the user to browse from root to replace it
    function setFolderOnRule(rule) {
        setFolderOnRuleFromPromise(rule, rule.folderId === "ROOT" ? ROOT_FOLDER_PROMISE : ProjectFolderService.getFolderFallbackRoot(rule.folderId, "(missing)"));
    }

    //insert new rules just above All Users group
    $scope.addFolderRulePosition = function() {
        let allGroupIndex = $scope.generalSettings.defaultFolderSettings.rules.findIndex( (rule) => rule.group === ALL_USERS_GROUP_INTERNAL);
        return allGroupIndex === -1 ? $scope.generalSettings.defaultFolderSettings.rules.length : allGroupIndex;
    }

    $scope.folderRuleTemplate = function() {
        const rule = { group : ""};
        rule.folderId = "ROOT";
        setFolderOnRule(rule);
        return rule;
    }

    let groupsPreparedForDefaultFolderUI = [];
    // Exposing groups in a callback for reliable use via transcopes
    $scope.getGroupsPreparedForDefaultFolderUI = () => groupsPreparedForDefaultFolderUI;

    function addMissingGroupsToGroupOptions(rule) {
        if (groupsPreparedForDefaultFolderUI.every(g => g.id != rule.group)) {
            const displayName = rule.group + " (missing)";
            const missingGroup = { display : displayName, id : rule.group};
            groupsPreparedForDefaultFolderUI.unshift(missingGroup);
        }
    }

    // When we retrieve the folder settings from the backend, add things to make the UI work and handle edge cases
    $scope.processFolderSettings = function(defaultFolderSettings) {
        $scope.allGroupsPromise.then((rawGroups) => {
            groupsPreparedForDefaultFolderUI = rawGroups.map((g) => ({ display : g, id : g}));
            groupsPreparedForDefaultFolderUI.push({display : ALL_USERS_GROUP_DISPLAY, id : ALL_USERS_GROUP_INTERNAL});

            if (defaultFolderSettings && defaultFolderSettings.rules) {
                defaultFolderSettings.rules.forEach(rule => {
                    addMissingGroupsToGroupOptions(rule);
                    // TODO replace this with a call to a new backend operation which does getFolderSummariesFallbackRoot on an array
                    setFolderOnRule(rule);
                });
            }
         });
    }

    // ** end of folder rules stuff



    $scope.alationRegister = {
    }
    $scope.urlWithProtocolAndHost = $window.urlWithProtocolAndHost();

    $scope.editorOptions = CodeMirrorSettingService.get('text/plain', {
        onLoad: function(cm) {
            $scope.codeMirror = cm;
        }
    });
    $scope.htmlEditorOptions = CodeMirrorSettingService.get('text/html', {
        onLoad: function(cm) {
            $scope.codeMirror = cm;
        }
    });

    DataikuAPI.security.listUsers().success(function(data) {
        $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
        $scope.allUsersLogin = data.map(user => '@' + user.login);
    }).error(setErrorInScope.bind($scope));

    $scope.allGroupsPromise = DataikuAPI.security.listGroups(false).error(setErrorInScope.bind($scope))
        .then(function({data}) {
            $scope.allGroups = data.sort((a, b) => a.localeCompare(b));
            return $scope.allGroups;
        });

    DataikuAPI.admin.connections.list().success(function(data) {
        $scope.connectionsByName = data;
        $scope.connections = Object.values(data);
    }).error(setErrorInScope.bind($scope));

    $scope.isAzureConnectionOAuth = function (connectionName) {
        return connectionName in $scope.connectionsByName
        && $scope.connectionsByName[connectionName].type == "Azure"
        && $scope.connectionsByName[connectionName].params.authType == "OAUTH2_APP";
    }

    var savedGeneralSettings, savedGlobalVariables, savedChannels;
    $scope.dirtySettings = function () {
        return !angular.equals($scope.generalSettings, savedGeneralSettings) ||
            !angular.equals($scope.globalVariables.asJSON, savedGlobalVariables) ||
            !angular.equals($scope.channels, savedChannels);
    };

    function allowedTransitionsFn(data) {
        return (data.toState && data.fromState && data.toState.name.startsWith('admin.general') && data.fromState.name.startsWith('admin.general'));
    }
    checkChangesBeforeLeaving($scope, $scope.dirtySettings, null, allowedTransitionsFn);

    $scope.hasBoundaryWhiteSpace = function (stringValue) {
        const leadingOrTrailingWhitespace = /^\s+|\s+$/;
        return leadingOrTrailingWhitespace.test(stringValue);
    };

    $scope.anyHaveBoundaryWhiteSpace = function (stringValues) {
        for(let value of stringValues) {
            if ($scope.hasBoundaryWhiteSpace(value)) {
                return true;
            }
        }
       return false;
    };

    // Hackish : to check for the existence of a nodes directory, we test to see if the deployer is registered in the nodes directory.
    // only use this in a context where you already know that the deployer is active.
    $scope.deployerAndNodesDirectoryEnabled = $scope.appConfig.nodesDirectoryManagedDeployerServer || $scope.appConfig.nodesDirectoryManagedDeployerClient;

    DataikuAPI.unifiedMonitoring.deployer.getRemoteDesignNodesUrls().success(function(data) {
        $scope.designNodes = data;
    }).error(setErrorInScope.bind($scope));

    $scope.load = function () {

        var promises = [];
        promises.push(DataikuAPI.admin.getGeneralSettings());
        promises.push(DataikuAPI.admin.getGlobalVariables());
        promises.push(DataikuAPI.admin.integrationChannels.list());
        promises.push(DataikuAPI.admin.getThemes());

        $scope.promises = $q.all(promises).then(
            function (values) {
                //general settings
                $scope.generalSettings = values[0].data;
                savedGeneralSettings = angular.copy($scope.generalSettings);
                $scope.savedGeneralSettings = angular.copy($scope.generalSettings); // allow to check local dirtiness
                $scope.processFolderSettings($scope.generalSettings.defaultFolderSettings);

                //global variables
                $scope.globalVariables.asJSON = values[1].data;
                savedGlobalVariables = angular.copy($scope.globalVariables.asJSON);
                //messaging channels
                $scope.channels = values[2].data;
                savedChannels = angular.copy($scope.channels);
                //themes
                $scope.themes = values[3].data;

                // fixup customFieldsPluginComponentOrder
                $scope.generalSettings.customFieldsPluginComponentOrder = $scope.generalSettings.customFieldsPluginComponentOrder
                    .filter(ref1 => $scope.appConfig.customFieldsPluginComponentRefs.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId));
                for (let i = 0; i < $scope.appConfig.customFieldsPluginComponentRefs.length; i++) {
                    const ref1 = $scope.appConfig.customFieldsPluginComponentRefs[i];
                    const existingRef = $scope.generalSettings.customFieldsPluginComponentOrder.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId);
                    if (!existingRef) {
                        $scope.generalSettings.customFieldsPluginComponentOrder.push(ref1);
                    }
                }
                // fixup customPolicyHooksPluginComponentOrder
                $scope.generalSettings.customPolicyHooksPluginComponentOrder = $scope.generalSettings.customPolicyHooksPluginComponentOrder
                    .filter(ref1 => $scope.appConfig.customPolicyHooksPluginComponentRefs.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId));
                for (let i = 0; i < $scope.appConfig.customPolicyHooksPluginComponentRefs.length; i++) {
                    const ref1 = $scope.appConfig.customPolicyHooksPluginComponentRefs[i];
                    const existingRef = $scope.generalSettings.customPolicyHooksPluginComponentOrder.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId);
                    if (!existingRef) {
                        $scope.generalSettings.customPolicyHooksPluginComponentOrder.push(ref1);
                    }
                }
            },
            function (errors) {
                setErrorInScope.bind($scope);
            }
        ).then(() => {
            // If URL contains a hash, wait for the settings to be loaded before scrolling to the anchor
            $timeout($anchorScroll);
        });
    };

    $scope.$watchCollection("channels", function (nv) {
        if (nv) {
            $scope.mailChannels = $scope.channels.filter(function (channel) {
                return ["aws-ses-mail", "microsoft-graph-mail", "smtp"].includes(channel.type);
            });
            if ($scope.generalSettings && $scope.mailChannels.map(c => c.id).indexOf($scope.generalSettings.notifications.emailChannelId) < 0) {
                $scope.generalSettings.notifications.emailChannelId = void 0;
            }
        }
    });

    $scope.noneUsersCallToActionBehaviorChoices = [];

    $scope.$watch("generalSettings.licensingSettings.trialBehavior", function(nv , ov) {
        if (nv == "DISABLED") {
            $scope.noneUsersCallToActionBehaviorChoices =[
                ['DISPLAY_MESSAGE', 'None, only display a message'],
                ['ALLOW_REQUEST_ACCESS', 'User can request access']
            ]
            if ($scope.generalSettings.licensingSettings.noneUsersCallToActionBehavior === 'ALLOW_START_TRIAL') {
                $scope.generalSettings.licensingSettings.noneUsersCallToActionBehavior = 'ALLOW_REQUEST_ACCESS';
            }
        } else {
            $scope.noneUsersCallToActionBehaviorChoices =[
                ['DISPLAY_MESSAGE', 'None, only display a message'],
                ['ALLOW_START_TRIAL', 'User can start a trial'],
                ['ALLOW_REQUEST_ACCESS', 'User can request access']
            ]
        }
    });

    $scope.availableUserProfilesForTrials = window.dkuAppConfig.licensing.userProfiles.filter(p => p != "NONE").map(p => [p,p]);

    $scope.load();

    // optionally scroll to a specific place of the page (use scroll-to-me directive)
    // after the promises because data may changes scroll position
    if($stateParams.scrollTo) {
        $scope.promises.then(() => {
            $scope.scrollToMe = $stateParams.scrollTo;
        });
    }

    $scope.autofillStudioUrl = function() {
        $scope.generalSettings.studioExternalUrl = urlWithProtocolAndHost();
    };

    function removeEmptyGitConfigurationOptions() {
        $scope.generalSettings.git.enforcedConfigurationRules.forEach((configRule) => {
            configRule.gitConfigurationOptions = configRule.gitConfigurationOptions.filter(option => !option.$invalid);
        });
    }

    $scope.invalidTabs = new Set();

    $scope.$on("$stateChangeStart", function(event, toState, toParams, fromState) {
        // We do not set 'Resource control' tab as invalid to avoid weird UI behavior. For this tab, a ng-model is not
        // changed if the new input value is not valid. Hence if a user exits the 'Resource control' tab with some
        // invalid fields and then switch back to it, the fields will no longer be invalid, which can be confusing.
        if ($scope.adminGeneralIndexForm.$invalid && fromState.name !== 'admin.general.limits') {
            $scope.invalidTabs.add(fromState.name);
        }
        if ($scope.adminGeneralIndexForm.homeArticlesListInvalid) {
            $scope.invalidTabs.add('admin.general.help');
        }
        if ($scope.adminGeneralIndexForm.homepagePromotedContentInvalid) {
            $scope.invalidTabs.add('admin.general.homepage');
        }
        $timeout(function() {
            $scope.invalidTabs.delete(toState.name);
        });
    });

    $scope.isAdminGeneralIndexFormInvalid = function() {
        return $scope.adminGeneralIndexForm.$invalid || $scope.invalidTabs.size || $scope.adminGeneralIndexForm.homeArticlesListInvalid || $scope.adminGeneralIndexForm.homepagePromotedContentInvalid;
    }

    function fetchGlobalTagsIfChanged() {
        if (!angular.equals($scope.generalSettings.globalTagsCategories, savedGeneralSettings.globalTagsCategories)) {
            TaggingService.fetchGlobalTags(true);
        }
    }

    function checkForDuplicateNames (list, type) {
        let names = [];

        list.forEach(function(element) {
            if (!element.name) {
                throw({message: "Found empty " + type + " name"});
            }
            if (names.includes(element.name)) {
                throw({message: "Found duplicate " + type + " names: " + element.name});
            }
            names.push(element.name);
        });
    }

    $scope.saveGeneralSettings = function() {
        if (savedGeneralSettings.udrMode != $scope.generalSettings.udrMode) {
            WT1.event("usage-reporting-mode-changed", { mode: $scope.generalSettings.udrMode });
        }
        return DataikuAPI.admin.saveGeneralSettings($scope.generalSettings).success(function(data) {
            fetchGlobalTagsIfChanged();
            savedGeneralSettings = angular.copy($scope.generalSettings);
            $scope.savedGeneralSettings = angular.copy($scope.generalSettings); // allow to check local dirtiness
            if (data.anyMessage) {
                Dialogs.infoMessagesDisplayOnly($scope, "Save warnings", data);
            }
            $scope.reloadAppConfig(); // Reload $rootScope.appConfig & friends so that hasGlobalProxy (among other) is correctly (re-)initialized
        }).error(setErrorInScope.bind($scope));
    };

    $scope.updateGlobalTags = function() {
        if (angular.equals($scope.generalSettings.globalTagsCategories, savedGeneralSettings.globalTagsCategories)) return;
        let updatedGlobalTagsMap = {};
        let update = false;
        const toDelete = ['isEdited', 'isNew', 'originalTagName', 'removeUsage'];
        //Map the global tags with their original- and updated name to update the tag on object across the instance
        $scope.generalSettings.globalTagsCategories.forEach(function(category, index) {
            category.globalTags.forEach(function(tag, idx) {
                let originalCategoryName = savedGeneralSettings.globalTagsCategories[index] && savedGeneralSettings.globalTagsCategories[index].name;
                if (originalCategoryName && (tag.originalTagName || originalCategoryName != category.name)) {
                    let oldGlobalTagName = `${originalCategoryName}:${tag.originalTagName || tag.name}`
                    updatedGlobalTagsMap[oldGlobalTagName] = {color: tag.color, updatedTagName: `${category.name}:${tag.name}`, globalTagsCategory: category.name, removeUsage: tag.removeUsage};
                    update = true;
                }
                var it = $scope.generalSettings.globalTagsCategories[index].globalTags[idx];
                toDelete.forEach(k => delete it[k]);
            });
            delete category.isNew;
        });
        if (update) DataikuAPI.admin.globalTags.updateGlobalTags(updatedGlobalTagsMap);
        return updatedGlobalTagsMap;
    };

    $scope.invalidateConfigCache = function() {
        var options = {type: 'text'};
        $scope.cacheInvalidationError = {};
        Dialogs.prompt($scope, "Invalidate cache", "Path to invalidate", "", options)
               .then(function(path) {
                   DataikuAPI.admin.invalidateConfigCache(path).error(setErrorInScope.bind($scope.cacheInvalidationError));
                });
    };

    $scope.disablePermissionsByEmail = () => {
        return $scope.generalSettings && $scope.generalSettings.security && $scope.generalSettings.security.enableEmailAndDisplayNameModification;
    };

    const getModelCacheDetails = () => DataikuAPI.admin.getModelCacheDetails().success((futureRes) => FutureWatcher.watchJobId(futureRes.jobId).success((res) => $scope.modelCacheDetails = res.result));
    getModelCacheDetails();
    $scope.clearModelCache = function() {
        Dialogs.confirmSimple($scope, "Clear model cache?").then(function() {
            DataikuAPI.admin.clearModelCache().error(setErrorInScope.bind($scope))
            .success(function (resp) {
                FutureProgressModal.show($scope, resp, "Clearing model cache").then(function(result) {
                    ActivityIndicator.success("Cleared model cache", 5000);
                    getModelCacheDetails();
                });
            });
        });
    };

    $scope.openDeleteModelModal = function (modelKey) {
        Dialogs.confirmSimple($scope, "Delete " + modelKey + " ?").then(function() {
            DataikuAPI.admin.deleteModelFromCache(modelKey).error(setErrorInScope.bind($scope))
            .success(function (resp) {
                getModelCacheDetails();
            });
        });
    };

    $scope.exportModel = function (modelKey) {
        downloadURL("/dip/api/admin/model-cache/export?modelKey=" + modelKey);
    };

    $scope.openImportModelModal = function() {
        $scope.importData = {
            uploading: false
        };
        CreateModalFromTemplate("/templates/import-file.html", $scope, null, function(modalScope) {
            modalScope.fileName = "Model";
            const parentScope = $scope.$parent.$parent;
            modalScope.import = function() {
                Assert.trueish($scope.importData.files, "No model file");
                $scope.importData.uploading = true;
                DataikuAPI.admin.importModel($scope.importData.files).then(function (data) {
                    modalScope.dismiss();
                    FutureProgressModal.show(parentScope, JSON.parse(data), "Model import", undefined, 'static', false).then(function(result) {
                        Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false);
                        getModelCacheDetails();
                    });
                }, function(payload) {
                    setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
                });
            };
        });
    }

    $scope.fetchOpenIDConfig = function(openIdWellKnown) {
        return DataikuAPI.admin.fetchOpenIDConfig(openIdWellKnown).success(function (config) {
            $scope.generalSettings.ssoSettings.openIDParams.issuer = config.issuer;
            $scope.generalSettings.ssoSettings.openIDParams.authorizationEndpoint = config.authorization_endpoint;
            $scope.generalSettings.ssoSettings.openIDParams.tokenEndpoint = config.token_endpoint;
            $scope.generalSettings.ssoSettings.openIDParams.jwksUri = config.jwks_uri;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.computeWellKnowFromIssuer = function() {
        let issuer = $scope.generalSettings.ssoSettings.openIDParams.issuer;
        $scope.openIdWellKnown = issuer ?
            issuer + '/.well-known/openid-configuration' :
            "";
    }

    /**
    *   Save the generalSettings, global variables and integrations channels and perform all checks prior to that
    *   Needs to return a promise and shouldn't just return, as cmd+s serves a fallback to force save
    */
    $scope.save = function () {
        try {
            removeEmptyGitConfigurationOptions();

            checkForDuplicateNames($scope.generalSettings.containerSettings.executionConfigs, "container configuration");
            checkForDuplicateNames($scope.generalSettings.sparkSettings.executionConfigs, "Spark configuration");
            checkForDuplicateNames($scope.generalSettings.globalTagsCategories, "global category");

            const updatedGlobalTagsMap = $scope.updateGlobalTags();
            const gv = $scope.globalVariables.asJSON || '{}';
            return $scope.saveGeneralSettings().then(function () {
                return DataikuAPI.admin.integrationChannels.saveAll($scope.channels).success(function (data) {
                    $scope.channels = data;
                    savedChannels = angular.copy(data);
                    return DataikuAPI.admin.saveGlobalVariables(gv).success(function () {
                        savedGlobalVariables = angular.copy($scope.globalVariables.asJSON);
                        $scope.$broadcast('generalSettingsSaved', updatedGlobalTagsMap);
                        if ($scope.isAdminGeneralIndexFormInvalid()) {
                            const allInvalidTabs = new Set();
                            $scope.invalidTabs.forEach(tab => allInvalidTabs.add($state.get(tab).pageTitle()));
                            if ($scope.adminGeneralIndexForm.$invalid) {
                                allInvalidTabs.add($state.current.pageTitle());
                            }
                            if ($scope.adminGeneralIndexForm.homepagePromotedContentInvalid) {
                                allInvalidTabs.add($state.get('admin.general.homepage').pageTitle());
                            }
                            if ($scope.adminGeneralIndexForm.homeArticlesListInvalid) {
                                allInvalidTabs.add($state.get('admin.general.help').pageTitle());
                            }
                            const warningMessage = "Saved with some invalid fields in tab" +
                                ($scope.invalidTabs.size + $scope.adminGeneralIndexForm.$invalid > 1 ? "s '" : " '") +
                                [...allInvalidTabs].join("', '") + "'";
                            ActivityIndicator.warning(warningMessage);
                        } else {
                            ActivityIndicator.success("Saved!");
                        }
                        // special cases: flags that need to be in appConfig for the corresponding options to be available
                        // in the frontend: impalaEnabled, etc
                        // note : settings are not sent back from the backend, but that's fine because they are saved as is
                        $scope.appConfig.impalaEnabled = $scope.generalSettings.impalaSettings.enabled && $scope.appConfig.hadoopEnabled;
                        $scope.appConfig.pluginDevExplicitCommit = $scope.generalSettings.pluginDevExplicitCommit;
                        $scope.appConfig.npsSurveyEnabled = $scope.generalSettings.npsSurveyEnabled;
                        $scope.appConfig.nodeName = $scope.generalSettings.nodeName;
                        $scope.appConfig.helpIntegrationEnabled = $scope.generalSettings.helpIntegrationEnabled;
                        $scope.appConfig.openTicketsFromOpalsEnabled = $scope.generalSettings.openTicketsFromOpalsEnabled;
                        $scope.appConfig.projectStatusList = $scope.generalSettings.projectStatusList;
                        if (!$rootScope.featureFlagEnabled('homepageRedesign')) {
                            // TODO @homepage cleanup when removing feature flag
                            $scope.appConfig.homeMessages = $scope.generalSettings.homeMessages;
                        }
                        $scope.appConfig.apiDeployerStages = $scope.generalSettings.apiDeployerServerSettings.stages;
                        $scope.appConfig.projectVisibility = $scope.generalSettings.projectVisibility;
                        $scope.appConfig.appVisibility = $scope.generalSettings.appVisibility;
                        $scope.appConfig.studioForgotPasswordUrl = $scope.generalSettings.studioForgotPasswordUrl;
                        $scope.appConfig.autoAcceptSchemaChangeAtEndOfFlow = $scope.generalSettings.autoAcceptSchemaChangeAtEndOfFlow;
                        $scope.appConfig.uiCustomization = $scope.generalSettings.uiCustomizationSettings;
                        $scope.appConfig.quickSharingElementsEnabled = $scope.generalSettings.quickSharingElementsEnabled;
                        $scope.appConfig.pluginInstallRequestsEnabled = $scope.generalSettings.pluginInstallRequestsEnabled;
                        $scope.appConfig.opalsEnabled = $scope.generalSettings.opalsEnabled;
                        $scope.appConfig.codeEnvInstallRequestsEnabled = $scope.generalSettings.codeEnvInstallRequestsEnabled;
                    }).error(setErrorInScope.bind($scope));
                }).error(setErrorInScope.bind($scope));
            }).catch(setErrorInScope.bind($scope));
        } catch (err) {
            ActivityIndicator.error("Invalid format: "+err.message);
        }
    };

    $scope.registerAlationOpener = function(){
        $scope.save().then(function(){
            DataikuAPI.connections.registerAlationOpener($scope.alationRegister.alationAPIToken).success(function(){
                ActivityIndicator.success("Alation registration done")
            }).error(setErrorInScope.bind($scope));
        })
    }

    DataikuAPI.connections.getNames('SQL')
        .success(function (data) { $scope.sqlConnections = data; })
        .error(setErrorInScope.bind($scope));

    $scope.testLdapSettings = function () {
        $scope.ldapTesting = true;
        $scope.ldapTestResult = null;
        DataikuAPI.admin.testLdapSettings($scope.generalSettings.ldapSettings).success(function (data) {
            $scope.ldapTestResult = data;
        })
            .error(setErrorInScope.bind($scope))
            .finally(() => $scope.ldapTesting = false);
    };


    $scope.testLdapGetUserDetails = function (userName) {
        $scope.ldapUserTesting = true;
        $scope.ldapTestUserDetails = null;
        DataikuAPI.admin.testLdapGetUserDetails({
            settings:$scope.generalSettings.ldapSettings,
            username:userName
        }).success(function (data) {
            $scope.ldapTestUserDetails = data;
        })
            .error(setErrorInScope.bind($scope))
            .finally(() => $scope.ldapUserTesting = false);
    };

    $scope.testAzureADSettings = function (login, email) {
        $scope.azureADTesting = true;
        $scope.azureADTestResult = null;
        DataikuAPI.admin.testAzureADSettings(
        {
            "azureADSettings": $scope.generalSettings.azureADSettings,
            "userIdentity": {
                "login": login || "",
                "email": email || ""
            }
        }).success(function (data) {
            $scope.azureADTestResult = data;
            if ($scope.azureADTestResult.connectionOK) {
                $scope.azureADTestResult.userAttributes = JSON.stringify($scope.azureADTestResult.userAttributes, null, 2);
            }
        })
            .error(setErrorInScope.bind($scope))
            .finally(() => $scope.azureADTesting = false);
    };

    $scope.governIntegrationTest = function() {
        DataikuAPI.admin.govern.test().success(function (data) {
            $scope.governIntegrationTestResult = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.governIntegrationParams = { projectKey : '' };
    $scope.governIntegrationFullSync = function() {
        DataikuAPI.admin.govern.sync($scope.governIntegrationParams.projectKey).success(function (data) {
            FutureProgressModal.show($scope, data, "Synchronizing DSS Items on Dataiku Govern", null, 'static', false, true).then(function(result){
                if (result) {
                    $scope.governIntegrationFullSyncResult = {
                        status: 'SUCCESS',
                        data: result
                    };
                }
            })
        }).error(function(data, status, headers) {
                $scope.governIntegrationFullSyncResult = {
                    status: 'ERROR',
                    error: getErrorDetails(data, status, headers)
                };
        });
    };

    $scope.governIntegrationFullDeployerSync = function() {
        DataikuAPI.admin.govern.deployerSync().success(function (data) {
            FutureProgressModal.show($scope, data, "Synchronizing DSS Deployer Items on Dataiku Govern", null, 'static', false, true).then(function(result){
                if (result) {
                    $scope.governIntegrationFullDeployerSyncResult = {
                        status: 'SUCCESS',
                        data: result
                    };
                }
            })
        }).error(function(data, status, headers) {
                $scope.governIntegrationFullDeployerSyncResult = {
                    status: 'ERROR',
                    error: getErrorDetails(data, status, headers)
                };
        });
    };

    $scope.checkDatastoryIntegration = function() {
        DataikuAPI.admin.datastory.check().success(function (data) {
            $scope.datastoryIntegrationCheckResult = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.addLDAPGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.ldapSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.ldapSettings.groupProfiles = []
        }
        $scope.generalSettings.ldapSettings.groupProfiles.push({key: '', value: $scope.generalSettings.ldapSettings.userProfile});
    };

    $scope.deleteLDAPGroupProfile = function (index) {
        $scope.generalSettings.ldapSettings.groupProfiles.splice(index, 1)
    };

    $scope.addSSOGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.ssoSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.ssoSettings.groupProfiles = []
        }
        $scope.generalSettings.ssoSettings.groupProfiles.push({key: '', value: $scope.generalSettings.ssoSettings.userProfile});
    };

    $scope.deleteSSOGroupProfile = function (index) {
        $scope.generalSettings.ssoSettings.groupProfiles.splice(index, 1)
    };

    $scope.addAzureADGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.azureADSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.azureADSettings.groupProfiles = []
        }
        $scope.generalSettings.azureADSettings.groupProfiles.push({key: '', value: $scope.generalSettings.azureADSettings.userProfile});
    };

    $scope.deleteAzureADGroupProfile = function (index) {
        $scope.generalSettings.azureADSettings.groupProfiles.splice(index, 1)
    };

    $scope.addCustomAuthGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.customAuthSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.customAuthSettings.groupProfiles = []
        }
        $scope.generalSettings.customAuthSettings.groupProfiles.push({key: '', value: $scope.generalSettings.customAuthSettings.userProfile});
    };

    $scope.deleteCustomAuthGroupProfile = function (index) {
        $scope.generalSettings.customAuthSettings.groupProfiles.splice(index, 1)
    };

    $scope.getChannelTypeLabel = function (type) {
        if (!type) {
            return "Unknown";
        } else if (type === 'msft-teams') {
            return "Microsoft Teams";
        } else if (type === 'google-chat') {
            return "Google Chat";
        } else if (type === 'aws-ses-mail') {
            return "Mail (via Amazon SES)";
        } else if(type === 'microsoft-graph-mail') {
            return "Mail (via Microsoft 365 with OAuth)"
        } else if (type === 'smtp') {
            return "Mail (SMTP)";
        } else {
            return type.charAt(0).toUpperCase() + type.slice(1);
        }
    };

    $scope.addChannel = function (type) {
        var definition = {
            type : type,
            configuration : {
                sessionProperties: []
            },
            permissions: [{
                group: ALL_USERS_GROUP_INTERNAL,
                canUse: true,
            }],
            $creation : true //flag to allow id edition only on creation
        };
        if (type === 'slack' || type === 'webhook' || type === 'twilio' || type === 'msft-teams' || type === 'google-chat' ) {
            definition.configuration.useProxy = true;
        }
        if (type === 'slack') {
            definition.configuration.mode = 'WEBHOOK';
        }
        if (type === 'msft-teams') {
            definition.configuration.webhookType = 'WORKFLOWS';
        }
        $scope.channels.push(definition)
    };

    $scope.removeChannel = function (channel) {
        var index = $scope.channels.indexOf(channel);
        if (index >= 0) {
            $scope.channels.splice(index, 1);
        }
    };

    DataikuAPI.admin.clusters.listAccessible('HADOOP').success(function(data){
        $scope.clusters = [{id:null, name:'No override'}].concat(data);
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.admin.clusters.listAccessible('KUBERNETES').success(function(data){
        $scope.k8sClusters = [{id:null, name:'No override'}].concat(data);
    }).error(setErrorInScope.bind($scope));

    $scope.DEPLOYER_MODES = [
        ["DISABLED", "Disabled"],
        ["LOCAL", "Local"],
        ["REMOTE", "Remote"]
    ]
    $scope.DEPLOYER_MODES_DESCRIPTIONS = [
        "Disable ability to publish models and projects on the Deployer",
        "Use this DSS instance as Deployer",
        "Publish models and projects on a remote Deployer"
    ]

    $scope.listOpalsUnsupportedWhiteLabelingSettings = function (whiteLabeling) {
        if (whiteLabeling === undefined) {
            return [];
        }

        let unsupportedSettings;

        if (whiteLabeling.productLongName !== "Dataiku DSS") {
            unsupportedSettings = ["productLongName"];
        } else {
            unsupportedSettings = [];
        }

        const stringSettings = [
            "referenceDocRootUrl",
            "aboutModalTitle",
            "aboutModalLogoUrl",
            "aboutModalText",
            "getHelpModalTitle",
            "getHelpModalText",
        ];

        const booleanSettings = [
            "contextualHelpSearchEnabled",
        ];

        return unsupportedSettings
            .concat(stringSettings.filter(s => whiteLabeling[s]))
            .concat(booleanSettings.filter(s => s in whiteLabeling && !whiteLabeling[s]));
    };

    $scope.syncOnboardingOpals = function() {
        $scope.generalSettings.onboardingExperience = $scope.generalSettings.opalsEnabled;
    }

    $scope.pushBaseImages = function(){
        if($scope.dirtySettings()) {
            $scope.save().then($scope.pushBaseImages_NoCheckDirty);
        } else {
            $scope.pushBaseImages_NoCheckDirty();
        }
    }

    $scope.pushBaseImages_NoCheckDirty = function(){
        DataikuAPI.admin.containerExec.pushBaseImages().success(function(data) {
            // FutureProgressModal.show($scope, data, "Pushing base images");
            FutureProgressModal.show($scope, data, "Pushing base images").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Push result", result.messages, result.futureLog);
                }
            })
        }).error(setErrorInScope.bind($scope));
    }
    const formatDefaults = function(defaults) {
        // Remap missing choice from options set to empty string for proper display
        // NB: No worries, backend does properly the reverse operation
        if (!defaults.cudaVersion) {
            defaults.cudaVersion = '';
        }
        if(!defaults.distrib) {
            defaults.distrib = '';
        }
        return defaults;
    };
    $scope.buildBaseImage = function(buildType, buildOptions) {
        $scope.saveGeneralSettings().then(() => DataikuAPI.admin.containerExec.getBaseImageDefaults(buildType))
        .then(({data}) => DataikuAPI.admin.containerExec.buildBaseImage(buildType, buildOptions || formatDefaults(data)))
        .then(({data}) => {
            FutureProgressModal.show($scope, data, "Building image").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Image built", result, result.futureLog, true);
                }
            });
        }).catch(setErrorInScope.bind($scope));
    };
    $scope.buildCDEPluginsImage = () => {
        $scope.saveGeneralSettings().then(() => DataikuAPI.admin.containerExec.buildBaseImage("CDE_PLUGINS"))
        .then(({data}) => {
            FutureProgressModal.show($scope, data, "Building image").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Image built", result, result.futureLog, true);
                }
            });
        }).catch(setErrorInScope.bind($scope));
    }
    $scope.setupBuildBaseImage = function(buildType) {
        DataikuAPI.admin.containerExec.getBaseImageDefaults(buildType).then(({data}) => {
            CreateModalFromTemplate("/templates/admin/fragments/build-base-image-modal.html", $scope, "BuildBaseImageModalController", function(modalScope) {
                modalScope.availableDistributions = _.concat([["", "Use default"]], data.AVAILABLE_DISTRIBUTIONS.map(x => [x, x.charAt(0).toUpperCase() + x.slice(1)])); // uppercase first letter
                modalScope.availableCudaVersions = _.concat([["", "No CUDA"]], data.AVAILABLE_CUDA_VERSIONS.map(x => [x, "CUDA " + x]));
                modalScope.buildType = buildType;
                modalScope.buildOptions = formatDefaults(data);
                modalScope.buildBaseImage = $scope.buildBaseImage;
            });
        })
        .catch(setErrorInScope.bind($scope));
    };

    $scope.fillInBoilerplateCustomWelcomeEmail = function() {
        if ($scope.generalSettings != null && $scope.generalSettings.welcomeEmailSettings != null && $scope.generalSettings.welcomeEmailSettings.sendAsHTML &&
                !$scope.generalSettings.welcomeEmailSettings.useDefaultTemplate && !$scope.generalSettings.welcomeEmailSettings.message) {
            $http.get('templates/admin/general/inline-boilerplate-custom-welcome-email.html').then(function(response) {
                $scope.generalSettings.welcomeEmailSettings.message = response.data;
            });
        }
    }

    $scope.datasetTypesOptions = (() => {
        const datasetName = $filter('datasetTypeToName');
        const rawDatasetsTiles = GlobalProjectActions.getAllDatasetByTilesNoFilter(); // call without scope is valid because we won't use the clickCallbacks fields.
        const res = [];
        rawDatasetsTiles.forEach(tile => {
            const allTypes = tile.types.concat(tile.types2 || [])
            allTypes.forEach(type => {
                if(type === undefined) return; // ignore the ones hidden by feature-flags.
                res.push({
                    tile: tile.title,
                    value: type.type,
                    displayName: type.label !== undefined ? type.label : datasetName(type.type),
                });
            });
        });

        // this is a special case, the 'search_and_import' behavior doesn't exist in the new dataset page.
        res.push({
            tile: 'Import existing',
            value: 'search_and_import',
            displayName: translate('PROJECT.ACTIONS.SEARCH_AND_IMPORT', 'Search and import\u2026'),
        });
        return res;
    })();


    $scope.highlightedDatasets = {
        // this function returns the list of choices for a list item, eg the full list minus the items already selected elsewhere
        getOptions: (currentSelected) => {
            return $scope.datasetTypesOptions.filter(datasetTypeOption => {
                return datasetTypeOption.value === currentSelected
                    || $scope.generalSettings.uiCustomizationSettings.highlightedDatasets.indexOf(datasetTypeOption.value) === -1;
            });
        },
        // convert the config format into the data format for editable-list
        init: () => {
            $scope.highlightedDatasets.selection = $scope.generalSettings.uiCustomizationSettings.highlightedDatasets.map(type => ({type}));
        },
        // reverse convert the editable-list into the config (and remove empty values)
        onChange: (nv) => {
            $scope.generalSettings.uiCustomizationSettings.highlightedDatasets = nv.map(({type}) => type).filter(x => x);
        },
    };

    // Init the highlightedDatasets values as soon as config is loaded
    $scope.promises.then($scope.highlightedDatasets.init);

    // find available LLMs
    DataikuAPI.codeAssistant.listAvailableLLMs().success(function(data){
        $scope.availableLLMs = data.identifiers;
    }).error(setErrorInScope.bind($scope));

    $scope.openAIServicesTermsOfUseModal = function() {
        CreateModalFromTemplate("/templates/admin/general/ai-services-terms-of-use-modal.html", $scope, null, function(modalScope) {
            modalScope.uiState = {
                termsAccepted: false
            }
            modalScope.confirm = function() {
                $scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAccepted = true;
                $scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAcceptedOn = moment().toISOString(true);
                $scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAcceptedBy = $rootScope.appConfig.login;
                modalScope.dismiss();
            }
        })
    }

    $scope.llmCostLimitingEnabled = window.dkuAppConfig.llmSettings.costLimitingEnabled;
    $scope.llmRateLimitingEnabled = window.dkuAppConfig.llmSettings.rateLimitingEnabled;
});

app.controller("AdminGenAIController", function ($scope, $state, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.useInternalCodeEnvLabel = "Use internal code env";

    DataikuAPI.codeenvs.listNames('PYTHON').then(function({data}) {
        $scope.codeEnvs = data;
        $scope.codeEnvItemsListWithDefault = [{"label": $scope.useInternalCodeEnvLabel, "value": undefined}].concat(data.map(codeEnv => ({"label": codeEnv, "value": codeEnv})));
    }).catch(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("TEXT_EMBEDDING_EXTRACTION").success(function(data){
        $scope.availableEmbeddingLLMs = data["identifiers"];
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("GENERIC_COMPLETION").success(function(data){
        $scope.availableCompletionLLMs = data["identifiers"];
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("IMAGE_INPUT").success(function(data){
        $scope.availableVLMs = data["identifiers"];
        $scope.addNoneOption();
    }).error(setErrorInScope.bind($scope));

    $scope.addNoneOption = function() {
        $scope.availableVLMs.unshift({ // make sure to have it as 1st option
            id: "DSS_NO_SELECTION",
            friendlyName: "No VLM (text-only extraction)",
            type: "" // no group
        });
    }

    let piiDetectionInternalCodeEnvChecked = false;
    DataikuAPI.codeenvs.checkDSSInternalCodeEnv("PII_DETECTION_CODE_ENV").then(function({data}) {
        if (Object.keys(data).length > 0) {
            $scope.piiDetectionInternalCodeEnv = data.value;
        }
        piiDetectionInternalCodeEnvChecked = true;
    }).catch(setErrorInScope.bind($scope));

    let ragInternalCodeEnvChecked = false;
    DataikuAPI.codeenvs.checkDSSInternalCodeEnv("RAG_CODE_ENV").then(function({data}) {
        if (Object.keys(data).length > 0) {
            $scope.ragInternalCodeEnv = data.value;
        }
        ragInternalCodeEnvChecked = true;
    }).catch(setErrorInScope.bind($scope));

    let documentExtractionCodeEnvChecked = false;
    DataikuAPI.codeenvs.checkDSSInternalCodeEnv("DOCUMENT_EXTRACTION_CODE_ENV").then(function({data}) {
        if (Object.keys(data).length > 0) {
            $scope.docextractionInternalCodeEnv = data.value;
        }
        documentExtractionCodeEnvChecked = true;
    }).catch(setErrorInScope.bind($scope));

    $scope.piiDetectionInternalCodeEnvExists = function() {
        return piiDetectionInternalCodeEnvChecked && $scope.piiDetectionInternalCodeEnv != null;
    }

    $scope.ragInternalCodeEnvExists = function() {
        return ragInternalCodeEnvChecked && $scope.ragInternalCodeEnv != null;
    }

    $scope.piiDetectionCodeEnvIsInternal = function () {
        return $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
            $scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv == null || (
                $scope.piiDetectionInternalCodeEnvExists() && $scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv == $scope.piiDetectionInternalCodeEnv.envName
            )
        );
    }

    $scope.ragCodeEnvIsInternal = function () {
        return $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
            $scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv == null || (
                $scope.ragInternalCodeEnvExists() && $scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv == $scope.ragInternalCodeEnv.envName
            )
        );
    }

    $scope.showPiiDetectionCodeEnvWarning = function () {
        return piiDetectionInternalCodeEnvChecked && $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
                (!$scope.piiDetectionInternalCodeEnvExists()) || (!$scope.piiDetectionCodeEnvIsInternal())
            );
    }

    $scope.showRagCodeEnvWarning = function () {
        return ragInternalCodeEnvChecked && $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
                (!$scope.ragInternalCodeEnvExists()) || (!$scope.ragCodeEnvIsInternal())
            );
    }

    $scope.fixupNullishPiiDetectionCodeEnvForDirtynessCheck = function() {
        // if the presidioBasedPIIDetectionCodeEnv is unset in the backend, it will be initialized as non-set (~undefined) in the frontend
        // when selecting "Use internal code env", the code env selector forces this value to null
        // we need to fix this to avoid the dirtyness check failing due to null !== undefined
        if ($scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv == null) {
            $scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv = undefined;
        }
    }

    $scope.fixupNullishEmbedDocumentTextExtractionCodeEnvForDirtynessCheck = function() {
        if ($scope.generalSettings.generativeAISettings.embedDocumentsRecipeSettings.textExtractionCodeEnv == null) {
            $scope.generalSettings.generativeAISettings.embedDocumentsRecipeSettings.textExtractionCodeEnv = undefined;
        }
    }

    $scope.fixupNullishRagCodeEnvForDirtynessCheck = function() {
        // if the defaultRetrievableKnowledgeCodeEnv is unset in the backend, it will be initialized as non-set (~undefined) in the frontend
        // when selecting "Use internal code env", the code env selector forces this value to null
        // we need to fix this to avoid the dirtyness check failing due to null !== undefined
        if ($scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv == null) {
            $scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv = undefined;
        }
    }

    $scope.internalCodeEnvsHRef = function() {
        if ($scope.appConfig.isAutomation) {
            return $state.href("admin.codeenvs-automation.internal");
        } else {
            return $state.href("admin.codeenvs-design.internal");
        }
    }

    $scope.availableTraceExplorerWebApps = [];
    $scope.$watch('generalSettings.generativeAISettings.llmTraceSettings.traceExplorerDefaultWebApp.projectKey', function (newProjectKey, oldProjectKey) {
        if (newProjectKey) {
            DataikuAPI.taggableObjects.listAccessibleObjects(newProjectKey, 'WEB_APP').then(function (response) {
                $scope.availableTraceExplorerWebApps = response.data.filter((object) => object.subtype === 'webapp_traces-explorer_traces-explorer');
            });
        }

        if (newProjectKey === oldProjectKey) {
            return;
        }

        if ($scope.generalSettings.generativeAISettings.llmTraceSettings.traceExplorerDefaultWebApp.webAppId && oldProjectKey) {
            $scope.generalSettings.generativeAISettings.llmTraceSettings.traceExplorerDefaultWebApp.webAppId = null;
        }
        $scope.availableTraceExplorerWebApps = [];
    });

});

app.controller("AdminCostLimitingController", function ($scope, $interval, $state, DataikuAPI, Dialogs, CreateModalFromTemplate, ClipboardUtils) {
    const FALLBACK_ID = "DKU-FALLBACK-QUOTA";
    $scope.getSortExpression = function(expression) {
        let fnExpression = {
            'isFiltered': $scope.isFiltered,
            'getTotalCostSpent': $scope.getTotalCostSpent,
            'getCostProgress': $scope.getCostProgress,
            'periodicityAsMinutes': $scope.periodicityAsMinutes,
        }[expression];
        return fnExpression ? fnExpression : expression;
    };
    $scope.selection = {
        orderQuery: "", orderReversed: false, filterQuery: "",
    }; // used for sorting table

    const llmConnectionTypes = ['Anthropic', 'AzureLLM', 'AzureOpenAI', 'Bedrock', 'Cohere', 'DatabricksLLM', 'HuggingFaceLocal', 'mistral', 'OpenAI',
                                    'SageMaker-GenericLLM', 'SnowflakeCortex', 'StabilityAI', 'VertexAILLM', 'NVIDIA-NIM'];
    DataikuAPI.admin.connections.list().success(function(data) {
        $scope.llmConnections = Object.values(data)
            .filter(c => llmConnectionTypes.includes(c.type))
            .map(c => ({label: c.name, value: c.name}))
            .sort((a, b) => a.label.localeCompare(b.label));
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.projects.list().success(function (projects) {
        $scope.availableProjects = projects
            .map(p => ({label: p.name + " (" + p.projectKey + ")", value: p.projectKey}))
            .sort((a, b) => a.label.localeCompare(b.label));
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.security.listUsers().success(function(data) {
        $scope.allUsers = data
            .map(u => ({label: u.displayName + " (" + u.login + ")", value: u.login}))
            .sort((a, b) => a.label.localeCompare(b.label));
    }).error(setErrorInScope.bind($scope));

    const allCostLimitingPeriodicities = [
        {"label": "Calendar Year", "value": "YEAR", "allowRollingPeriods": false, "asMinutes": 365 * 24 * 60},
        {"label": "Calendar Quarter", "value": "QUARTER", "allowRollingPeriods": false, "asMinutes": 90 * 24 * 60},
        {"label": "Calendar Month", "value": "MONTH", "allowRollingPeriods": false, "asMinutes": 30 * 24 * 60},
        {"label": "Rolling Day(s)", "value": "DAY", "allowRollingPeriods": true, "asMinutes": 24 * 60},
        {"label": "Rolling Minute(s)", "value": "MINUTE", "allowRollingPeriods": true, "asMinutes": 1},
    ]
    $scope.costLimitingPeriodicities = allCostLimitingPeriodicities.filter(p => window.dkuAppConfig.llmSettings.costLimitingPeriodicities.includes(p.value));
    const periodicityMap = $scope.costLimitingPeriodicities.reduce((acc, p) => { acc[p.value] = p; return acc }, {});

    $scope.onCostLimitingPeriodicityChange = function(quota) {
        const periodicity = $scope.costLimitingPeriodicities.find(p => p.value === quota.periodicity);
        if (periodicity && !periodicity.allowRollingPeriods) {
            quota.rollingPeriods = 1;
        }
    };

    $scope.shouldDisplayRollingPeriods = function(quota) {
        const periodicity = periodicityMap[quota.periodicity]
        return periodicity && periodicity.allowRollingPeriods;
    };

    $scope.getPeriodicitySummary = function(quota) {
        const periodicity = periodicityMap[quota.periodicity]
        if (!periodicity) return "";
        if (periodicity.allowRollingPeriods) {
            return quota.rollingPeriods + " " + periodicity.label;
        } else {
            return periodicity.label;
        }
    };

    $scope.periodicityAsMinutes = function(quota) {
        return quota.rollingPeriods * periodicityMap[quota.periodicity].asMinutes;
    };

    $scope.costLimitingQuotaFilterSchema = {
            'columns': [
                {
                    name: 'project',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'user',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'provider',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'connection',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'llmId',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
                {
                    name: 'projectKey',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
                {
                    name: 'userLogin',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
                {
                    name: 'connectionName',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
            ]
        };

    $scope.providers = [
        {
            'label': 'Anthropic',
            'value': 'Anthropic'
        },
        {
            'label': 'Azure LLM',
            'value': 'AzureLLM'
        },
        {
            'label': 'Azure OpenAI',
            'value': 'AzureOpenAI'
        },
        {
            'label': 'AWS Bedrock',
            'value': 'Bedrock'
        },
        {
            'label': 'Cohere',
            'value': 'Cohere'
        },
        // {
        //     'label': 'Databricks Mosaic AI',
        //     'value': 'DatabricksLLM'
        // },
        {
            'label': 'Local Hugging Face',
            'value': 'HuggingFaceLocal'
        },
        {
            'label': 'Mistral AI',
            'value': 'MistralAI'
        },
        {
            'label': 'OpenAI',
            'value': 'OpenAI'
        },
//        {
//            'label': 'Python Agents',
//            'value': 'python-agent'
//        },
        // {
        //     'label': 'Amazon SageMaker LLM',
        //     'value': 'SageMaker-GenericLLM'
        // },
        // {
        //     'label': 'Snowflake Cortex',
        //     'value': 'SnowflakeCortex'
        // },
        {
            'label': 'Stability AI',
            'value': 'StabilityAI'
        },
        // Visual agents can't report direct costs atm
        // {
        //     'label': 'Visual Agents',
        //     'value': 'tools-using-agent'
        // },
        {
            'label': 'Vertex AI',
            'value': 'VertexAILLM'
        },
        {
            'label': 'NVIDIA NIM',
            'value': 'NVIDIA-NIM'
        },
    ];
    const _allCustomLLMPlugins = [...$scope.appConfig.customPythonLLMs, ...$scope.appConfig.customJavaLLMs];
    // Append Custom LLM plugins
    $scope.providers = $scope.providers.concat(
            Array.from(new Set(_allCustomLLMPlugins.map(llm => llm.ownerPluginId)))
            .map(pluginID => $scope.appConfig.loadedPlugins.find(plugin => plugin.id === pluginID))
            .filter(plugin => plugin) // remove any `undefined` entries from plugins that could not be found
            .map(p => ({label: "LLM Plugin: " + p.label, value: "CustomLLM:" + p.id}))
            .sort((a,b) => a.label.localeCompare(b.label))
    );
    // Append Agent plugins
//    $scope.providers = $scope.providers.concat(
//        Array.from(new Set($scope.appConfig.customAgents.map(agent => agent.ownerPluginId)))
//            .map(pluginID => $scope.appConfig.loadedPlugins.find(plugin => plugin.id === pluginID))
//            .filter(plugin => plugin) // remove any `undefined` entries from plugins that could not be found
//            .map(p => ({label: "Plugin Agent: " + p.label, value: "plugin-agent:" + p.id}))
//            .sort((a,b) => a.label.localeCompare(b.label))
//    );

    $scope.getCostLimitingEnumValues = function(columnName) {
        if (columnName == 'provider') {
            return $scope.providers;
        } else if (columnName == 'project') {
            return $scope.availableProjects;
        } else if (columnName == 'connection') {
            return $scope.llmConnections;
        } else if (columnName == 'user') {
            return $scope.allUsers;
        }
        return null;
    }

    $scope.getThresholdAmount = function(quota, reportingAction) {
        if (quota.costLimit && reportingAction.threshold) {
            const thresholdAmount = quota.costLimit * reportingAction.threshold / 100.;
            const formattedAmount = (Math.round(thresholdAmount * 100) / 100).toFixed(2);
            return '$' + formattedAmount;
        }
        return '';
    }
    $scope.getThresholdsLabel = function(quota) {
        return quota.reportingActions.map((action) => action.threshold + "%").join(", ");
    };
    $scope.getReportingTargetsAsString = function(quota) {
        return "Will notify " + quota.reportingTargets.map((t) => t.email).join(", ");
    };

    $scope.quotaHasReportingActions = function(quota) {
        return (quota.reportingActions && quota.reportingActions.length) || quota.blockingLimit;
    };
    $scope.quotaHasRecipients = function(quota) {
        if (!quota.reportingTargets || quota.reportingTargets.length == 0) {
            return false;
        }
        return quota.reportingTargets.filter((t) => t.email && t.email.length > 0).length > 0;
    };
    $scope.validateReportingActions = function(quota) {
        if ($scope.quotaHasReportingActions(quota)) {
            return quota.reportingActions.every((action) => !action.$invalid);
        }
        return true;
    };
    $scope.reorderReportingActions = function(quota) {
        if ($scope.quotaHasReportingActions(quota)) {
            quota.reportingActions.sort((a, b) => parseFloat(a.threshold) - parseFloat(b.threshold));
        }
    };

    $scope.hasReportingActionAndNoQuotaAmount = function(quota) {
        return quota.costLimit === 0 && quota.reportingActions.length;
    }
    $scope.hasNoMailChannelAvailable = function() {
        return $scope.channels && !$scope.mailChannels.length;
    };
    $scope.hasNoMailChannelSelected = function(quota) {
        return $scope.mailChannels.length && $scope.quotaHasReportingActions(quota) && !quota.emailChannelId;
    };
    $scope.hasReportingActionsButNotRecipient = function(quota) {
        return $scope.quotaHasReportingActions(quota) && !$scope.quotaHasRecipients(quota);
    };
    $scope.getNotificationsIssues = function(quota) {
        if ($scope.hasNoMailChannelAvailable())
            return "You must define an email channel in the 'Notifications & Integrations > Messaging channels' section to send alerts.";
        if ($scope.hasNoMailChannelSelected(quota))
            return "You must define a channel for alerts to be sent.";
        if ($scope.hasReportingActionsButNotRecipient(quota))
            return "You must add recipients for alerts to be sent.";
        if ( $scope.hasReportingActionAndNoQuotaAmount(quota))
            return "You must define a quota amount > 0 for alerts to be sent.";
        return "";
    };
    $scope.globalQuotaDescription = "This quota will be applied to all queries.";
    $scope.describeFallbackScope = function() {
        return $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.length ? "This quota will be applied for queries not matching any other quota." : $scope.globalQuotaDescription;
    }

    $scope.costCounters = {};
    let updateCostCounters = function(nospinner) {
        DataikuAPI.admin.costLimiting.getCounters(nospinner).success(function(data){
            $scope.costCounters = data;
        }).error(setErrorInScope.bind($scope));
    };
    updateCostCounters(false);
    const costCounterRefreshIntervalMs = 10000; // 10 seconds
    const costCounterRefresher = $interval(() => updateCostCounters(true), costCounterRefreshIntervalMs);
    $scope.$on('$destroy', function() {
        $interval.cancel(costCounterRefresher);
    });

    $scope.getTotalCostSpent = function(quota) {
        if ($scope.costCounters[quota.id]) {
            return $scope.costCounters[quota.id].accruedCost;
        } else {
            return 0;
        }
    };
    $scope.getAllowedQueries = function(quota) {
        if ($scope.costCounters[quota.id]) {
            return $scope.costCounters[quota.id].allowedQueries;
        } else {
            return 0;
        }
    };
    $scope.getBlockedQueries = function(quota) {
        if ($scope.costCounters[quota.id]) {
            return $scope.costCounters[quota.id].blockedQueries;
        } else {
            return 0;
        }
    };
    $scope.getCostProgress = function(quota) {
        var progress = $scope.getTotalCostSpent(quota);
        if (quota.costLimit === 0) return 100;
        // avoid infinity result when dividing small values
        if (progress < 0.01) return 0;
        return 100 * (progress / quota.costLimit);
    };
    $scope.getProgressColorClass = function(quota) {
        var progress = $scope.getCostProgress(quota);
        if (quota.costLimit === 0 && !quota.blockingLimit) {
            return "gauge--disabled";
        } else if (progress < 75) {
            return "gauge--success";
        } else if (progress < 100) {
            return "gauge--warning";
        } else {
            return "gauge--error";
        }
    };
    $scope.isQuotaBlocked = function(quota) {
        var progress = $scope.getCostProgress(quota);
        return quota.blockingLimit && progress >= 100;
    };
    $scope.isCustomQuota = function(quota) {
        return quota.id !== FALLBACK_ID;
    }

    const randomAlphaNumeric = function(length) {
        let s = '';
        while (s.length < length) {
            s += Math.random().toString(36).slice(2);
        }
        return s.slice(0, length);
    };
    const generateRandomQuotaId = function () {
        return randomAlphaNumeric(8);
    }
    const fillMissingId = function(quota) {
        if (!quota.id) {
            quota.id = generateRandomQuotaId();
        }
        return quota;
    }
    $scope.dirtyCostLimitingSettings = function() {
        return $scope.savedGeneralSettings && !angular.equals($scope.generalSettings.generativeAISettings.costLimitingSettings, $scope.savedGeneralSettings.generativeAISettings.costLimitingSettings);
    };
    const pushNewQuota = function(quota) {
        $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.push(fillMissingId(quota));
    };
    const duplicateQuota = function(quota) {
        let newQuota = angular.copy(quota);
        newQuota.id = generateRandomQuotaId();
        newQuota.name = "Copy of " + newQuota.name;
        return newQuota;
    };
    const deleteQuota = function(quota) {
        // reassign the value so the change is detected and triggers an update of the table
        $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas = $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.filter((q) => q.id != quota.id);
    };

    $scope.customQuotasAllowed = function() {
        return $scope.appConfig.licensedFeatures.advancedLLMMeshAllowed;
    };
    $scope.getCustomQuotasAllowedMessage = function() {
        return !$scope.customQuotasAllowed() ? 'Custom quotas require Advanced LLM Mesh, which is not enabled in your license.' : '';
    };
    const openQuotaModal = function(quota, title, action, confirmCallback) {
        CreateModalFromTemplate("/templates/admin/cost-limiting/quota-modal.html", $scope, null, function(modalScope) {
                modalScope.quota = fillMissingId(quota);
                modalScope.title = title;
                modalScope.action = action;
                modalScope.validate = function(quota) {
                    return (!$scope.isCustomQuota(quota) || quota.name) // mandatory name for only for custom quotas
                        && (quota.costLimit || quota.costLimit === 0)
                        && quota.periodicity && quota.rollingPeriods
                        && $scope.validateReportingActions(quota);
                };
                modalScope.confirm = function(quota) {
                    confirmCallback(quota);
                    modalScope.dismiss();
                };
            });
    };
    $scope.openNewQuotaModal = function() {
        let newQuota = {
              id: generateRandomQuotaId(),
              filter: {
                  enabled: false,
                  distinct: false,
                  uiData: {
                      conditions: [],
                      mode: '&&'
                  }
              },
              periodicity: "MONTH",
              rollingPeriods: 1,
              reportingActions: [],
              reportingTargets: [],
           };
        openQuotaModal(newQuota, "Create a quota", "CREATE", pushNewQuota);
    };
    $scope.openEditQuotaModal = function(currentQuota, title) {
        openQuotaModal(angular.copy(currentQuota), title || "Update a quota", "UPDATE", (quota) => angular.copy(quota, currentQuota));
    };
    $scope.openDuplicateQuotaModal = function(quota) {
        openQuotaModal(duplicateQuota(quota), "Duplicate a quota", "Duplicate", pushNewQuota);
    };
    $scope.openDeleteSingleQuotaDialog = function(quota) {
        Dialogs.confirmInfoMessages($scope, 'Confirm quota deletion', null, '<p>Are you sure you want to delete quota: <strong>'+sanitize(quota.name)+'</strong> (id:&nbsp;<em>'+sanitize(quota.id)+'</em>)</p>', false).then(function() {
            deleteQuota(quota);
        }).catch(() => {}); // needed to avoid exception on abort
    };
    $scope.openDeleteQuotasDialog = function(quotas) {
        Dialogs.confirmInfoMessages($scope, 'Confirm quotas deletion', null, '<p>Are you sure you want to delete the following quotas:</p><ul>'
            + quotas.map((quota) => "<li><strong>"+sanitize(quota.name)+"</strong> (id:&nbsp;<em>"+sanitize(quota.id)+"</em>)</li>").join("") + "</ul>", false)
        .then(function() {
            quotas.forEach((quota) => deleteQuota(quota));
        }).catch(() => {}); // needed to avoid exception on abort
    };
    $scope.duplicateAndPushQuotas = function(quotas) {
        quotas.forEach((quota) => {
            $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.push(duplicateQuota(quota));
        });
    };
    $scope.copyToClipboard = function(value) {
        ClipboardUtils.copyToClipboard(value);
    };
});

app.controller("AdminLLMRateLimitingController", function ($scope, $state, DataikuAPI, Dialogs, CreateModalFromTemplate, localStorageService) {
    $scope.baselineSettings = {};
    $scope.allModelsByProvider = {};
    $scope.filteredModelsByProvider = {};
    $scope.applicableSettings = {};
    $scope.selection = {"orderQuery": "id"}; // apply model sort by default
    const FALLBACK_RPM = 1000; // default value in case we can't get a RPM, not expected to happen

    // Compute applicable settings by merging baseline settings and user settings
    function buildApplicableSettings() {
        let userSettings = $scope.generalSettings.generativeAISettings.rateLimitingSettings;
        let applicableSettings = angular.copy($scope.baselineSettings);

        // keep in sync with LLMRateLimitingSettingsService.buildApplicableProvidersSettings() on java side
        // Note that we do not filter out user settings with the managed flag so the user can retrieve its updated settings later
        for (let providerId in applicableSettings) {
            let userProviderSettings = userSettings.perProviderSettings[providerId] || {"perModelConfigs": {}};
            let applicableProviderSettings = applicableSettings[providerId];

            applicableProviderSettings.completionDefault = userProviderSettings.completionDefault || applicableProviderSettings.completionDefault;
            applicableProviderSettings.embeddingDefault = userProviderSettings.embeddingDefault || applicableProviderSettings.embeddingDefault
            applicableProviderSettings.imageGenerationDefault = userProviderSettings.imageGenerationDefault || applicableProviderSettings.imageGenerationDefault
            for (let modelId in userProviderSettings.perModelConfigs) {
                applicableProviderSettings.perModelConfigs[modelId] = userProviderSettings.perModelConfigs[modelId];
            }
        }
        return applicableSettings;
    }

    // Recompute generalSettings.generativeAISettings.rateLimitingSettings when the local model is updated
    function configIsDirty(defaultConfig, config) {
        return defaultConfig == null || defaultConfig.managed != config.managed || defaultConfig.requestsPerMinute != config.requestsPerMinute;
    }
    $scope.$watch("applicableSettings", function() {
        if (!$scope.generalSettings) return; // nothing to do until generalSettings are available
        let currentCustomSettings = {perProviderSettings: {}};
        // Only initialize provider config when needed to avoid dirty state when nothing was updated by the user
        const initializeProviderIfNeeded = (providerId) => {
            if (!currentCustomSettings.perProviderSettings[providerId]) {
                currentCustomSettings.perProviderSettings[providerId] = {perModelConfigs: {}};
            }
        };
        $scope.applicableSettings.providerList.forEach((provider) => {
            provider.purposes.forEach((purpose) => {
               if (configIsDirty(getDefaultConfigForPurpose(provider.id, purpose.id), purpose.config)) {
                   initializeProviderIfNeeded(provider.id);
                   currentCustomSettings.perProviderSettings[provider.id][purpose.id] = purpose.config;
               }
            });
        });
        for (let providerId in $scope.applicableSettings.modelConfigs) {
            $scope.applicableSettings.modelConfigs[providerId].forEach((model) => {
                if(configIsDirty(getDefaultConfigForModel(providerId, model.id), model.config)) {
                   initializeProviderIfNeeded(providerId);
                   currentCustomSettings.perProviderSettings[providerId].perModelConfigs[model.id] = model.config;
                }
            });
        }
        $scope.generalSettings.generativeAISettings['rateLimitingSettings'] = currentCustomSettings;
    }, true);

    function loadBaselineSettings() {
        // Load default settings from backend and convert applicableSettings as lists that are better suited for sorting and filtering the table rows
        DataikuAPI.admin.rateLimiting.getDefaults().success(function(data) {
            $scope.baselineSettings = data.baselineSettings;
            $scope.allModelsByProvider = data.models;
            let applicableSettings = buildApplicableSettings();

            // keep a reference of applicableSettings for direct access
            // .perProvider field is kept to avoid collisions with .providerList and .modelConfigs
            $scope.applicableSettings.perProvider = applicableSettings;

            // Convert as list so we can sort and filter providers
            $scope.applicableSettings.providerList = [];
            for (let providerId in applicableSettings) {
                let perProviderSettings = applicableSettings[providerId];

                let perPurposeConfigsList = [];
                ["completionDefault", "embeddingDefault", "imageGenerationDefault"].forEach((purposeId) => {
                    if (perProviderSettings[purposeId]) {
                        perPurposeConfigsList.push({id: purposeId, config: perProviderSettings[purposeId]});
                    }
                });
                $scope.applicableSettings.providerList.push({
                    id: providerId,
                    purposes: perPurposeConfigsList,
                    hasModelSortIdx: Object.keys($scope.allModelsByProvider).includes(providerId) ? 1 : 0
                });
            }
            $scope.applicableSettings.providerList = $scope.applicableSettings.providerList.sort((a, b) => {
                return b.hasModelSortIdx - a.hasModelSortIdx || a.id.localeCompare(b.id);
            });

            // Convert applicable settings as a map of lists so we can filter models
            $scope.applicableSettings.modelConfigs = {};
            for (let providerId in applicableSettings) {
                let perProviderSettings = applicableSettings[providerId];
                $scope.applicableSettings.modelConfigs[providerId] = [];
                for (let modelId in perProviderSettings.perModelConfigs) {
                    $scope.applicableSettings.modelConfigs[providerId].push({
                        id: modelId,
                        config: perProviderSettings.perModelConfigs[modelId],
                    });
                }
                filterSuggestedModelsByProvider(providerId);
            }

            initExpandStatus();
        }).error(setErrorInScope.bind($scope));
    }
    // avoid race condition
    let unwatchGeneralSettings = $scope.$watch("generalSettings", function() {
        if ($scope.generalSettings) {
            loadBaselineSettings();
            unwatchGeneralSettings();
        }
    });

    function filterSuggestedModelsByProvider(providerId) {
        if ($scope.allModelsByProvider[providerId]) {
            $scope.filteredModelsByProvider[providerId] = $scope.allModelsByProvider[providerId].filter((modelId) => !$scope.modelAlreadyExists(providerId, modelId));
        }
    }
    function getDefaultConfigForPurpose(providerId, purposeId) {
        return $scope.baselineSettings[providerId][purposeId || "completionDefault"];
    }
    function getDefaultConfigForModel(providerId, modelId) {
        return $scope.baselineSettings[providerId].perModelConfigs[modelId];
    }
    function getDefaultApplicableRPMForProvider(providerId) {
        let providerConfig = $scope.applicableSettings.perProvider[providerId]["completionDefault"] || {};
        return providerConfig.managed ? getDefaultConfigForPurpose(providerId).requestsPerMinute : providerConfig.requestsPerMinute;
    }

    $scope.getDefaultRPMForPurpose = function(providerId, purposeId) {
        let defaultConfig = getDefaultConfigForPurpose(providerId, purposeId) || {};
        return defaultConfig.requestsPerMinute || FALLBACK_RPM;
    }
    $scope.getDefaultRPM = function(providerId, modelId) {
        if (modelId) {
            let defaultModel = getDefaultConfigForModel(providerId, modelId);
            return (defaultModel && defaultModel.requestsPerMinute)
                || getDefaultApplicableRPMForProvider(providerId)
                || 0;
        } else {
            let defaultProvider = getDefaultConfigForPurpose(providerId);
            return (defaultProvider && defaultProvider.requestsPerMinute) || FALLBACK_RPM;
        }
    };
    $scope.modelAlreadyExists = function(providerId, modelId) {
        return $scope.applicableSettings.modelConfigs[providerId].some((model) => model.id == modelId);
    };
    $scope.addModelRateLimit = function(provider) {
        const providerId = provider.id;
        const newModelId = provider.newModelId;
        $scope.applicableSettings.modelConfigs[providerId].push({
            id: newModelId,
            config: {
                managed: false,
                requestsPerMinute: $scope.getDefaultRPM(providerId, newModelId)
            }
        });
        filterSuggestedModelsByProvider(providerId);
        provider.newModelId = "";
    };
    $scope.isDeleteModelAllowed = function(providerId, modelId) {
        return !getDefaultConfigForModel(providerId, modelId);
    };
    function deleteModelConfig(providerId, modelId) {
        $scope.applicableSettings.modelConfigs[providerId] = $scope.applicableSettings.modelConfigs[providerId].filter((model) => model.id != modelId);
        filterSuggestedModelsByProvider(providerId);
    };
    $scope.openDeleteSingleConfigDialog = function(providerId, modelId) {
        Dialogs.confirmInfoMessages($scope, 'Confirm rate limit deletion for '+sanitize(providerId), null, '<p>Are you sure you want to delete the rate limit settings for model: <strong>'+sanitize(modelId)+'</strong></p>', false).then(function() {
            deleteModelConfig(providerId, modelId);
        }).catch(() => {}); // needed to avoid exception on abort
    };
    $scope.getAddRateLimitDisallowedMessage = function(providerId, newModelId) {
        if (!newModelId) {
            return "Model is empty";
        }
        if ($scope.modelAlreadyExists(providerId, newModelId)) {
            return "Settings are already set for this model";
        }
        return "";
    };
    $scope.getSuggestedModelsForProvider = function(providerId) {
        return $scope.filteredModelsByProvider[providerId];
    };
    const purposeLabels = {
        "completionDefault": "Other completion models",
        "embeddingDefault": "Other embedding models",
        "imageGenerationDefault": "Other image generation models"
    };
    $scope.labelForPurpose = function(purposeId) {
        return purposeLabels[purposeId];
    };

    function matchFilter(value, filter) {
        return value.toLowerCase().includes(filter);
    }
    $scope.filterProviders = function(filter) {
        filter = filter && filter.toLowerCase();
        return function(provider) {
            return !filter || matchFilter(provider.id, filter)
            || $scope.applicableSettings.modelConfigs[provider.id].some((model) => matchFilter(model.id, filter));
        };
    };
    $scope.filterModels = function(providerId, filter) {
        filter = filter && filter.toLowerCase();
        return (model) => !filter || matchFilter(providerId, filter) || matchFilter(model.id, filter);
    };

    const localStorageProviderExpandKey = "rateLimitingSettingsProvidersExpand";
    $scope.expandStatus = localStorageService.get(localStorageProviderExpandKey) || {};
    function initExpandStatus() {
        if (!Object.keys($scope.expandStatus).length) {
            let defaultExpandStatus = {};
            Object.keys($scope.allModelsByProvider).forEach((provider) => defaultExpandStatus[provider] = true);
            $scope.expandStatus = defaultExpandStatus;
        }
    }
    $scope.showProvider = function(providerId) {
        return $scope.expandStatus[providerId];
    };
    $scope.toogleShowProvider = function(providerId) {
        $scope.expandStatus[providerId] = !$scope.expandStatus[providerId];
        localStorageService.set(localStorageProviderExpandKey, $scope.expandStatus);
    }
});

app.controller("AdminVariablesController", function ($scope, $state, $stateParams, DataikuAPI, WT1, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.executeVariablesUpdate = function () {
        $scope.save().then(function () {
            DataikuAPI.admin.executeVariablesUpdate().success(function () {
                $state.transitionTo( $state.current, angular.copy($stateParams), { reload: true, inherit: true, notify: true } );
            }).error(setErrorInScope.bind($scope));
        });
    }
});


app.controller("AdminThemeController", function ($scope, $rootScope, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.getThemeThumbnailUrl = function (theme) {
        var uri = $scope.getThemeUri(theme);
        return uri + theme.thumbnail;
    };

    $scope.setCurrentTheme = function (theme) {
        // saving theme content in app config
        if (theme) {
            $scope.generalSettings.themeId = theme.id;
        } else {
            delete $scope.generalSettings.themeId;
        }
        $scope.saveGeneralSettings().then(function () {
            // visualy setting theme
            $rootScope.appConfig.theme = theme;
            $scope.setTheme(theme);
        }).catch(setErrorInScope.bind($scope));
    };

    $scope.removeHomeMessage = function (homeMessage) {
        var index = $scope.generalSettings.homeMessages.indexOf(homeMessage);
        if (index != -1) {
            $scope.generalSettings.homeMessages.splice(index, 1);
        }
    };
});

app.controller("DeleteGlobalCategoryModalController", function ($scope, $rootScope, $controller) {

    function remove(index) {
        let items = $scope.generalSettings.globalTagsCategories[index].globalTags;
        items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = "";
            item.removeUsage = true;
        });
        $scope.updateGlobalTags();
        deleteKeep(index);
    };

    function deleteKeep(index) {
        $scope.generalSettings.globalTagsCategories.splice(index, 1);
        // setTimeout : when the form was invalid but is valid after deleting the category
        // give time to update or it will fail on checkFormValidity
        setTimeout(() => $scope.save());
    };

    function reassign(index, reassignTo) {
        if (reassignTo === undefined) return;
        $scope.generalSettings.globalTagsCategories[index].name = $scope.generalSettings.globalTagsCategories[reassignTo].name;
        $scope.updateGlobalTags();
        const tagsName = $scope.generalSettings.globalTagsCategories[reassignTo].globalTags.map(it => it.name);
        var mergedGlobalTags = angular.copy($scope.generalSettings.globalTagsCategories[reassignTo].globalTags);
        $scope.generalSettings.globalTagsCategories[index].globalTags.forEach(function(tag) {
            if (!tagsName.includes(tag.name)){
                mergedGlobalTags.push(tag);
            }
        });
        $scope.generalSettings.globalTagsCategories[reassignTo].globalTags = mergedGlobalTags;
        deleteKeep(index);
    }

    $scope.doDeleteGlobalCategory = function() {
        switch ($scope.deletionMode) {
            case 'remove':
                remove($scope.index);
                break;
            case 'reassign':
                reassign($scope.index, $scope.reassignTo);
                break;
            case 'keep':
                deleteKeep($scope.index);
                break;
        }
        $scope.dismiss();
    };

});

app.controller("MergeGlobalTagsModalController", function ($scope, $rootScope, $controller) {
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.doMergeTags = function() {
        $scope.items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = $scope.outputTag.name;
            item.color = $scope.outputTag.color;
        });
        if (!$scope.generalSettings.globalTagsCategories[$scope.index].globalTags.find(it => (it.originalTagName || it.name) === $scope.outputTag.name)) {
            $scope.items.shift();
        }
        $scope.updateGlobalTags();
        let indexes = $scope.items.map(it => it.$idx);
        $scope.generalSettings.globalTagsCategories[$scope.index].globalTags = $scope.generalSettings.globalTagsCategories[$scope.index].globalTags.filter((it, index) => !indexes.includes(index));
        $scope.save();
        $scope.dismiss();
    }
});

app.controller("DeleteGlobalTagsModalController", function ($scope, $rootScope) {

    function remove(items) {
        items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = "";
            item.removeUsage = true;
        });
        $scope.updateGlobalTags();
        deleteKeep(items);
    };

    function deleteKeep(items) {
        let indexes = items.map(it => it.$idx);
        $scope.generalSettings.globalTagsCategories[$scope.index].globalTags = $scope.generalSettings.globalTagsCategories[$scope.index].globalTags.filter((it, index) => !indexes.includes(index));
        // setTimeout : when the form was invalid but is valid after deleting the tag(s)
        // give time to update or it will fail on checkFormValidity
        setTimeout(() => $scope.save());
    };

    function reassign(items, reassignTo) {
        if (reassignTo === undefined) return;
        items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = reassignTo;
        });
        $scope.updateGlobalTags();
        deleteKeep(items);
    }

    $scope.doDeleteGlobalTags = function() {
        switch ($scope.deletionMode) {
            case 'remove':
                remove($scope.items);
                break;
            case 'reassign':
                reassign($scope.items, $scope.reassignTo);
                break;
            case 'keep':
                deleteKeep($scope.items);
                break;
        }
        $scope.dismiss();
    };
});

app.controller("GlobalTagsController", function ($scope, $element, $timeout, TaggingService, CreateModalFromTemplate, TAGGABLE_TYPES, DataikuAPI) {
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.taggableTypes = [ // Must be kept in sync with ITaggingService.TaggableType
        'PROJECT',
        'ANALYSIS',
        'SAVED_MODEL',
        'DATASET',
        'RECIPE',
        'LABELING_TASK',
        'MANAGED_FOLDER',
        'STREAMING_ENDPOINT',
        'FLOW_ZONE',
        'SQL_NOTEBOOK',
        'SEARCH_NOTEBOOK',
        'JUPYTER_NOTEBOOK',
        'STATISTICS_WORKSHEET',
        'SCENARIO',
        'DASHBOARD',
        'INSIGHT',
        'ARTICLE',
        'WEB_APP',
        'CODE_STUDIO',
        'REPORT',
        'LAMBDA_SERVICE',
        'RETRIEVABLE_KNOWLEDGE',
        'PROMPT_STUDIO',
        'AGENT_TOOL',
        'DATA_COLLECTION'
    ];

    $scope.computeItemTemplate = function() {
        return {name: '', color: TaggingService.getDefaultColor(Math.random().toString(36).slice(2)), isNew: true};
    };

    $scope.addCategory = function() {
        $scope.generalSettings.globalTagsCategories.push({name: '', globalTags: [], appliesTo: $scope.taggableTypes, isNew: true});
        //focus new category name
        $timeout(function() {
            const focusable = $element.find('input[ng-model="category.name"]').last();
            if (focusable) {
                focusable.focus()
            }
        });
    };

    $scope.calcGlobalTagUsage = function() {
        DataikuAPI.catalog.search("", {scope:['dss'], _type:['ALL']}, true).then((data) => {
            $scope.tagUsageMap = data.data.aggregations["tag.raw"].agg.buckets.reduce(function(map, obj) {
                map[obj.key] = obj.doc_count;
                return map;
            }, {});
        });
    };
    $scope.calcGlobalTagUsage();

    $scope.$on('generalSettingsSaved', $scope.calcGlobalTagUsage);

    $scope.getGlobalTagUsage = function(category, item) {
        if (!$scope.tagUsageMap) return 0;
        let tagName = item.originalTagName || item.name;
        return $scope.tagUsageMap[`${category}:${tagName}`];
    };

    $scope.linkToCatalog = function(category, item) {
        const tagName = item.originalTagName || item.name;
        const globalTagName = (`${category}:${tagName}`).replace(/\\/g,'%5C').replace(/\//g,'~2F').replace(/&/g, '%5C&');
        return `/search-dss-items/scope=all&tag.raw=${globalTagName}`;
    };

    $scope.updateItem = function(category, originalTagName, item) {
        if (!item.isEdited) {
            item.isEdited = true;
            item.originalTagName = originalTagName;
        }
    }

    $scope.mergeTags = function(index, items) {
        CreateModalFromTemplate("/templates/global-tags/merge-global-tags-modal.html", $scope, "MergeGlobalTagsModalController", function(modalScope) {
            modalScope.index = index;
            modalScope.items = items;

            modalScope.outputTag = angular.copy(items[0]);
        });
    };

    $scope.deleteGlobalCategory = function(index, categoryName) {
        if ($scope.generalSettings.globalTagsCategories[index].isNew) {
            $scope.generalSettings.globalTagsCategories.splice(index, 1);
            return;
        }
        CreateModalFromTemplate("/templates/global-tags/delete-global-category-modal.html", $scope, "DeleteGlobalCategoryModalController", function(modalScope) {
            modalScope.deletionMode = "remove";
            modalScope.index = index;
            modalScope.categoryName = categoryName;
            modalScope.assignableCategoryList = $scope.generalSettings.globalTagsCategories.filter(cat => cat.name != categoryName).map((cat, idx) => idx >= index ? idx + 1 : idx);
        });
    };

    $scope.deleteGlobalTags = function(index, items) {
        CreateModalFromTemplate("/templates/global-tags/delete-global-tags-modal.html", $scope, "DeleteGlobalTagsModalController", function(modalScope) {
            modalScope.modalTitle = `Delete global tag${items.length > 1 ? 's' : ''}`;
            modalScope.deletionMode = "remove";
            modalScope.index = index;
            modalScope.items = items;
            modalScope.assignableTagsList = $scope.generalSettings.globalTagsCategories[index].globalTags.filter(it => !items.includes(it)).map(it => it.name);
        });
    };
});

app.controller("ProjectStatusController", function ($scope, $rootScope, TopNav, DKUConstants, TaggingService) {
    $scope.defaultTagColor = TaggingService.getDefaultColor;
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.newStatus = {
        name: '',
        defaultColor: TaggingService.getDefaultColor('')
    };

    //Isolating the archived status so that it does not get deleted nor reorder
    $scope.projectStatusList = [];
    $scope.promises.then(function (values) {
        $scope.projectStatusList = angular.copy($scope.generalSettings.projectStatusList);
        var archivedStatusIndex =  $scope.projectStatusList.findIndex(function (status) {
            return status.name ==  DKUConstants.ARCHIVED_PROJECT_STATUS;
        });
        var archivedStatus = archivedStatusIndex > -1 ? $scope.projectStatusList.splice(archivedStatusIndex, 1) : [];

        //On local projectStatusList change recomputing generalSettings.projectStatusList with the archived status at its end
        $scope.$watch('projectStatusList', function () {
            $scope.generalSettings.projectStatusList = $scope.projectStatusList.concat(archivedStatus);
        }, true);
    });
});

app.controller("BrandColorsController", function ($scope, ColorUtils) {
    $scope.newBrandColor = {
        defaultColor: '#FFFFFF'
    };

    $scope.brandColors = [];
    $scope.brandColorsPalette = ColorUtils.getDiscretePaletteColors('dku_font');
    $scope.promises.then(function () {
      // The scope of the controller is not accessible from the content projected in EditableList.
      // To be able to access the palette, we need to reference it in each item of the palette.
      $scope.brandColors = $scope.generalSettings.brandColors.map(brandColor => ({
          ...brandColor,
          palette: $scope.brandColorsPalette
      }));

      $scope.$watch('brandColors', function () {
          // We don't want to keep the palette in each color entry when saving, so we manually remove them and sync the controller and global models.
          $scope.generalSettings.brandColors = $scope.brandColors.map(brandColor => {
              const cleanedBrandColor = { ...brandColor };
              delete cleanedBrandColor.palette;
              return cleanedBrandColor;
          });
      }, true);
    });
});


app.controller("AdminLicensingController", function ($scope, $state, CreateModalFromTemplate, TopNav, DataikuAPI) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.$state = $state;

    $scope.enterLicense = function () {
        CreateModalFromTemplate("/templates/admin/enter-license.html", $scope, "EnterLicenseController");
    };

    $scope.fetchTrialUsersStatus = function() {
        DataikuAPI.admin.getTrialStatus().success(function (data) {
            $scope.trialStatus = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getLimits = function () {
        if ($scope.isDSSAdmin()) {
            DataikuAPI.admin.getLimitsStatus().success(function (data) {
                $scope.limits = data;
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.getLimits();
});


app.controller("EnterLicenseController", function ($scope, $state, $rootScope, DataikuAPI, Assert, TopNav, WT1) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    Assert.inScope($scope, "appConfig");

    $scope.existingKey = {}

    $scope.reloadMe = function () {
        $scope.dismiss();
        location.reload();
    };

    $scope.setLicense = function () {
        WT1.event("license-update");
        DataikuAPI.registration.setOfflineLicense($scope.existingKey.license).success(function (data) {
            $scope.registrationSuccessful = {};
        }).error(setErrorInScope.bind($scope));
    };
});


//Integration channels to report info outside of dss:


app.controller("AwsChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});

app.controller("MicrosoftGraphChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});

app.controller("SMTPChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});


app.controller("SlackChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});


app.controller("TeamsChannelController", function ($scope, $state, $stateParams, DataikuAPI) {
    if ($scope.channel.configuration.useProxy === undefined) {
        $scope.channel.configuration.useProxy = false;
    }
});

app.controller("GoogleChatChannelController", function ($scope, $state, $stateParams, DataikuAPI) {
});

app.controller("WebhookChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});


app.controller("ShellChannelController", function ($scope, $state, $stateParams, DataikuAPI, TopNav, CodeMirrorSettingService) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.channel.configuration.type = $scope.channel.configuration.type || 'COMMAND';
    $scope.editorOptions = CodeMirrorSettingService.get('text/x-sh', {onLoad: function(cm) {$scope.cm = cm;}});

    $scope.shellSenderTypes = [{type:'FILE',name:'Script file'}, {type:'COMMAND',name:'Command'}];

    $scope.getCommandLine = function () {
        if ( $scope.channel.configuration.type == 'FILE' ) {
            if ( $scope.channel.configuration.command ) {
                return "sh -c " + $scope.channel.configuration.command + " script_file";
            } else {
                return "sh script_file";
            }
        } else {
            return $scope.channel.configuration.command;
        }
    };
});


app.controller("DatasetChannelController", function ($scope, $state, $stateParams, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    DataikuAPI.projects.list().success(function (data) {
        $scope.projectsList = data;
    }).error(setErrorInScope.bind($scope));

    $scope.$watch('channel.configuration.projectKey', function () {
        if ($scope.channel.configuration.projectKey == null ) return;
        DataikuAPI.datasets.list($scope.channel.configuration.projectKey).success(function (data) {
            $scope.datasetsList = data;
        }).error(setErrorInScope.bind($scope));
    });
});

app.controller("ContainerSettingsController", function($scope, DataikuAPI, Dialogs, Assert, TopNav, FutureProgressModal, CodeMirrorSettingService) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    Assert.inScope($scope, "appConfig");
    Assert.inScope($scope, "addLicInfo");
    Assert.inScope($scope, "generalSettings");

    $scope.codeMirrorSettingService = CodeMirrorSettingService;
    $scope.settings = $scope.generalSettings.containerSettings;

    const getConfigurationsByWorkloadType = (configs, workloadType) => {
        if (configs === null){
            return [];
        }
        return [{value: "", label: "Nothing selected"}, ...configs.filter(ec => [workloadType, "ANY"].includes(ec.workloadType) && !!ec.name).map(ec => ({value: ec.name, label: ec.name}))];
    }

    const setConfigurationsByWorkloadType = () => {
        $scope.userCodeContainerConfigs = getConfigurationsByWorkloadType($scope.settings.executionConfigs, "USER_CODE");
        $scope.visualRecipesContainerConfigs = getConfigurationsByWorkloadType($scope.settings.executionConfigs, "VISUAL_RECIPES");
    }

    setConfigurationsByWorkloadType();
    $scope.$watch("settings.executionConfigs", setConfigurationsByWorkloadType, true)

    $scope.getNewContainerConfig = function() {
        return {
            type: 'KUBERNETES',
            usableBy: 'ALL', allowedGroups: [], workloadType: 'ANY',
            dockerNetwork: 'host',
            dockerResources: [],
            kubernetesResources: {
                memRequestMB: -1, memLimitMB: -1,
                cpuRequest: -1, cpuLimit: -1,
                customRequests: [], customLimits: [],
                hostPathVolumes: []
            },
            properties: []
        };
    };

    DataikuAPI.security.listGroups(false)
        .success(data => {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
        })
        .error(setErrorInScope.bind($scope));

    $scope.isBaseImageNameSuspicious = function(baseImage) {
        return /^(?:[\w-_]+\.)+\w+(?::\d+)?\//.test(baseImage);
    };

    $scope.installJupyterSupport = function(){
        DataikuAPI.admin.containerExec.installJupyterSupport().success(function(data) {
            FutureProgressModal.show($scope, data, "(Re)Installing kernels").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.removeJupyterSupport = function(){
        DataikuAPI.admin.containerExec.removeJupyterSupport().success(function(data) {
            FutureProgressModal.show($scope, data, "Removing kernels").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.applyK8SPolicies = function(){
        if($scope.dirtySettings()) {
            $scope.save().then($scope.applyK8SPolicies_NoCheckDirty);
        } else {
            $scope.applyK8SPolicies_NoCheckDirty();
        }
    }
    $scope.applyK8SPolicies_NoCheckDirty = function() {
        DataikuAPI.admin.containerExec.applyK8SPolicies().success(function(data) {
            FutureProgressModal.show($scope, data, "Applying Kubernetes policies").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Policies applied", result, result.futureLog, true);
                }
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("BuildBaseImageModalController", function ($scope) {
    $scope.doBuild = function() {
        $scope.dismiss();
        $scope.buildBaseImage($scope.buildType, $scope.buildOptions);
    };
});


app.controller('AdminHelpSettingsController', function($scope, AccessibleObjectsCacheService) {
    // Cache for object-picker objects
    $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', $scope.setErrorInGeneralSettingsControllerScope).asValue;

    $scope.adminGeneralIndexForm.homeArticlesListInvalid = false;
    $scope.articlesAreValid = () => {
        if ($scope.generalSettings && $scope.generalSettings.personalHomePages.articles) {
            for (let elt of $scope.generalSettings.personalHomePages.articles) {
                if (!Object.keys(elt).includes("projectKey") || !Object.keys(elt).includes("id")) {
                    $scope.adminGeneralIndexForm.homeArticlesListInvalid = true;
                    return false;
                }
            }
        }
        $scope.adminGeneralIndexForm.homeArticlesListInvalid = false;
        return true;
    }
});

app.controller('AdminHomepageSettingsController', function($scope, AccessibleObjectsCacheService, Ng1PromotedContentAdminService, $q, DataikuAPI) {
    // Cache for object-picker objects
    $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', $scope.setErrorInGeneralSettingsControllerScope).asPromise;

    $scope.adminGeneralIndexForm.homepagePromotedContentInvalid = false;
    $scope.promotedContentIsValid = () => {
        if ($scope.generalSettings && $scope.generalSettings.personalHomePages && $scope.generalSettings.personalHomePages.promotedContent) {
            for (let elt of $scope.generalSettings.personalHomePages.promotedContent) {
                if (!$scope.isItemValid(elt)) {
                    $scope.adminGeneralIndexForm.homepagePromotedContentInvalid = true;
                    return false;
                }
            }
        }
        $scope.adminGeneralIndexForm.homepagePromotedContentInvalid = false;
        return true;
    }

    const TYPES_WITHOUT_PROJECT_KEY = ['WORKSPACE', 'DATA_COLLECTION', 'PROJECT', 'APP', 'ADMIN_MESSAGE', 'LINK'];
    const TYPES_WITHOUT_OBJECT = ['ADMIN_MESSAGE', 'LINK'];

    $scope.isItemValid = function(item) {
        if (!item) {
            return false;
        }
        if (!Object.keys(item).includes("type")) {
            return false;
        }
        if (!TYPES_WITHOUT_OBJECT.includes(item.type) && !Object.keys(item).includes("id")) {
            return false;
        }
        if (!TYPES_WITHOUT_PROJECT_KEY.includes(item.type) && !Object.keys(item).includes("projectKey")) {
            return false;
        }
        if (!item.$displayName) {
            // This ensures that the item exists and has not been deleted
            return false;
        }
        return true;
    }

    $scope.invalidItemReason = function(item) {
        if (!item) {
            return "Content is not defined";
        }
        if (!Object.keys(item).includes("type")) {
            return "Type is missing";
        }
        if (!TYPES_WITHOUT_OBJECT.includes(item.type) && !Object.keys(item).includes("id")) {
            return "Id is missing";
        }
        if (!TYPES_WITHOUT_PROJECT_KEY.includes(item.type) && !Object.keys(item).includes("projectKey")) {
            return "Project key is missing";
        }
        if (!item.$displayName) {
            // This ensures that the item exists and has not been deleted
            return "Item does not exist, or has an empty display name";
        }
        return "";
    }

    $scope.editPromotedContent = function(itemToEdit) {
        return Ng1PromotedContentAdminService.openPromotedContentAdminModal({
            promotedContent: itemToEdit,
            getAccessibleObjectsCached: $scope.getAccessibleObjects,
            mode: 'EDIT'
        }).then((output) => {
            if (output?.promotedContent) {
                return setProjectAndDisplayName([output.promotedContent])
                    .then(() => output.promotedContent);
            }
        });
    }

    $scope.addPromotedContent = function() {
        $q.when(Ng1PromotedContentAdminService.openPromotedContentAdminModal({
            getAccessibleObjectsCached: $scope.getAccessibleObjects,
            mode: 'ADD'
        })).then((output) => {
            if (output?.promotedContent) {
                return setProjectAndDisplayName([output.promotedContent]).then(() => {
                    $scope.generalSettings.personalHomePages.promotedContent = [
                        ...$scope.generalSettings.personalHomePages.promotedContent,
                        output.promotedContent
                    ];
                })
            }
        });
    }

    $scope.displayNamesLoaded = false;
    const deregister = $scope.$watch('generalSettings.personalHomePages.promotedContent', function(newValue) {
        if (angular.isDefined(newValue)) {
            $scope.displayNamesLoaded = false;
            setProjectAndDisplayName(newValue).then(() => {
                $scope.displayNamesLoaded = true;
            });
            deregister();
        }
    });

    function setProjectAndDisplayName(items) {
        const taggableObjectRefs = [];
        items.forEach((item) => {
            if(item.type === 'PROJECT') {
                taggableObjectRefs.push({ type: item.type, id: item.id, projectKey: item.id });
            } else if (TYPES_WITHOUT_PROJECT_KEY.includes(item.type)) {
                taggableObjectRefs.push({ type: item.type, id: item.id, projectKey: "" });
            } else {
                taggableObjectRefs.push({ type: 'PROJECT', id: item.projectKey, projectKey: item.projectKey }); // we also need the project display name
                taggableObjectRefs.push({ type: item.type, id: item.id, projectKey: item.projectKey });
            }
        });

        return DataikuAPI.admin.getPromotedContentDisplayNames(taggableObjectRefs).then(function ({ data }) {
            const displayNames = data;
            items.forEach((item) => {
                setProjectName(item, displayNames);
                setDisplayName(item, displayNames);
            });
        }).catch(setErrorInScope.bind($scope));
    }

    function setProjectName(item, displayNames) {
        if (!item) {
            item.$projectName = "";
            return;
        }
        if (TYPES_WITHOUT_PROJECT_KEY.includes(item.type)) {
            item.$projectName = "";
            return;
        }

        const taggableObjectRefKey = `{projectKey:${item.projectKey}, type:PROJECT, id:${item.projectKey}}`;
        if (angular.isDefined(displayNames[taggableObjectRefKey])) {
            item.$projectName = displayNames[taggableObjectRefKey];
        }
    }

    function setDisplayName (item, displayNames) {
        if (!item) {
            item.$displayName = "";
            return;
        }
        if (item.type === 'ADMIN_MESSAGE') {
            item.$displayName = item.customTitle;
            return;
        }
        if (item.type === 'LINK') {
            item.$displayName = item.customTitle || item.url;
            return;
        }

        let projectKey = ""
        if (item.type === 'PROJECT') {
            projectKey = item.id;
        } else if (!TYPES_WITHOUT_PROJECT_KEY.includes(item.type)) {
            projectKey = item.projectKey;
        }

        const taggableObjectRefKey = `{projectKey:${projectKey}, type:${item.type}, id:${item.id}}`;
        if (angular.isDefined(displayNames[taggableObjectRefKey])) {
            item.$displayName = displayNames[taggableObjectRefKey];
        }
    };
});

app.controller('AdminOtherSecurityController', function($scope, DataikuAPI) {
    $scope.updateExistingApiKeysLifetime = function() {
        if($scope.dirtySettings()) {
            $scope.save().then($scope.updateExistingApiKeysLifetime_NoCheckDirty);
        } else {
            $scope.updateExistingApiKeysLifetime_NoCheckDirty();
        }
    }

    $scope.updateExistingApiKeysLifetime_NoCheckDirty = function() {
        DataikuAPI.admin.apiKeys.updateApiKeysLifetime().error(setErrorInScope.bind($scope));
    }
});

    app.controller("AdminAiServicesSettingsController", function($scope, DataikuAPI) {
        DataikuAPI.webapps.listAllBackendsStates().success(function(data) {
            $scope.backends = data
              .filter(backend => backend.backendState && backend.backendState.hasExposedEndpoint)
              .map(backend => ({
                  ...backend,
                  webAppReference: backend.projectKey + "." + backend.webAppId
              }));
        }).error(setErrorInScope.bind($scope));

        $scope.webAppSearch = function(term, webapp) {
            return webapp.webAppReference.toLowerCase().includes(term.toLowerCase()) || webapp.webAppName.includes(term.toLowerCase());
        };

        $scope.defaultAiServerValue = function(localAiIndividualSetting) {
            if (!$scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAccepted) {
                $scope.generalSettings.localAIServerSettings[localAiIndividualSetting + "UseLocal"] = true;
            }
        };

        $scope.updateTelemetryEnabled = function(aiAssistant) {
            // ngChange triggers when the change start so before the value of localAIServerSettings[aiAssistant + "UseLocal"] gets updated. So if it was false we know it will change to true.
            if(!$scope.generalSettings.localAIServerSettings[aiAssistant + "UseLocal"]) {
                $scope.generalSettings.aiDrivenAnalyticsSettings[aiAssistant + "TelemetryEnabled"] = false;
            }
        };

        $scope.$watch('generalSettings.localAIServerSettings.storiesAIUseLocal', function (newVal) {
            if (newVal === true) {
                $scope.generalSettings.storiesVocalEnabled = false;
            }
        });

        $scope.$watchGroup([
        'generalSettings.aiDrivenAnalyticsSettings.storiesAIEnabled',
            function () {
                return $scope.featureFlagEnabled && $scope.featureFlagEnabled('storiesVocal');
            }
        ], function ([aiEnabled, flagEnabled]) {
            if (!(aiEnabled && flagEnabled)) {
                $scope.generalSettings.storiesVocalEnabled = false;
            }
        });
    });

    app.component("integrationChannelPermissions", { // integration-channel-permissions
            bindings: {
                permissions: "<", // IntegrationChannel.PermissionItem[], edited in place
                allUsers: "<",    // readonly UIUser[]
                allGroups: "<",   // readonly string[]
            },
            templateUrl: "templates/admin/general/integration-channel-permissions.html",
            controller: function integrationChannelPermissionsController($scope) {
                const $ctrl = this;

                function updateUnassignedGroups() {
                    $ctrl.unassignedGroupIds = ['$$ALL_USERS$$', ...$ctrl.allGroups].filter(group => !$ctrl.permissions.some(perm => perm.group === group));
                }
                function updateUnassignedUsers() {
                    $ctrl.unassignedUsers = $ctrl.allUsers.filter(user => !$ctrl.permissions.some(perm => perm.user === user.login));
                }

                $ctrl.$onChanges = (changes) => {
                    if(changes.allUsers) {
                        $ctrl.userLoginToDisplayName = new Map($ctrl.allUsers.map(user => [user.login, user.displayName]))
                    }
                    if(changes.allGroups) {
                        $ctrl.groupIdToDisplayName = new Map($ctrl.allGroups.map(group => [group, group]))
                        $ctrl.groupIdToDisplayName.set('$$ALL_USERS$$', 'All Users')
                    }
                    if(changes.permissions || changes.allUsers) {
                        updateUnassignedUsers();
                    }
                    if(changes.permissions || changes.allGroups) {
                        updateUnassignedGroups();
                    }
                }

                $ctrl.onUsersAdded = (users) => {
                    users.forEach(login => {
                        $ctrl.permissions.push({ user: login, canUse: true });
                    });
                    updateUnassignedUsers();
                    $scope.$applyAsync(); // trigger angularjs change detection
                }

                $ctrl.onGroupsAdded = (groups) => {
                    groups.forEach(group => {
                        $ctrl.permissions.push({ group, canUse: true });
                    });
                    updateUnassignedGroups();
                    $scope.$applyAsync(); // trigger angularjs change detection
                }

                $ctrl.removePermission = (index) => {
                    const removed = $ctrl.permissions.splice(index, 1)[0];
                    if(removed.user) {
                        updateUnassignedUsers();
                    } else {
                        updateUnassignedGroups();
                    }
                }

            }
        });
})();
