(function() {
'use strict';

const app = angular.module('dataiku.analysis.script');


app.directive('analysisCharts', function($q, $timeout, Logger, Assert, DataikuAPI, WT1, ActivityIndicator, TopNav, DatasetUtils, ChartsContext, ChartCustomMeasures, ChartTypeChangeHandler, ATSurveyService, DSSVisualizationThemeUtils, ChartNavigationService) {
return {
        scope: true,
        controller: function ($scope, $stateParams, $rootScope){
            /* ********************* Execute Callbacks for chartsCommon ******************* */

            $scope.getDataSpec = function() {
                const currentChart = $scope.charts[$scope.currentChart.index];
                Assert.trueish(currentChart, "No current chart");
                const dataSpec = {
                    datasetProjectKey : $scope.inputDatasetProjectKey,
                    datasetName : $scope.inputDatasetName,
                    script: angular.copy($scope.shaker),
                    copySelectionFromScript: currentChart.copySelectionFromScript,
                    sampleSettings : currentChart.refreshableSelection,
                    engineType : "LINO"
                }
                dataSpec.script.origin = "ANALYSIS";
                return dataSpec;
            }

            $scope.getExecutePromise = function(request, saveShaker = true, noSpinner = false, requiredSampleId = undefined, dataSpec = $scope.getDataSpec()) {
                const currentChart = $scope.charts[$scope.currentChart.index];
                Assert.trueish(currentChart, "No current chart");
                Assert.trueish(currentChart.summary, "Current chart summary is not ready");
                (saveShaker !== false) && $scope.saveShaker();
                if (request) {
                    if (requiredSampleId === undefined) {
                        requiredSampleId = currentChart.summary.requiredSampleId;
                    }
                    let promise = DataikuAPI.shakers.charts.getPivotResponse($stateParams.projectKey, dataSpec, request, requiredSampleId);

                    if (noSpinner === true) {
                        promise = promise.noSpinner();
                    }

                    return promise;
                }
            };

            $scope.saveChart = function(){
                $scope.saveShaker();
            };

            $rootScope.$on("chartSamplingChanged", function(event, opts) {
                if (angular.equals($scope.charts[$scope.currentChart.index], opts.chart)) {
                    $scope.clearCachedSummaries();
                    $scope.fetchColumnsSummaryForCurrentChart().then(function() {
                        Logger.info("Sample reloaded, executing chart");
                        $scope.$broadcast("forceExecuteChart");
                    });
                }
            });

            $scope.getDefaultNewChart = function() {
                var newChart = null;
                const defaultTheme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
                if ($scope.charts.length > 0) {
                    // Copy to retrieve the same sample, copySample and engine settings
                    newChart = angular.copy($scope.charts[$scope.charts.length - 1]);
                    newChart.theme = defaultTheme;
                    newChart.def = ChartTypeChangeHandler.defaultNewChart(newChart.theme);
                } else {
                    newChart = {
                        def : ChartTypeChangeHandler.defaultNewChart(defaultTheme),
                        copySelectionFromScript : true,
                        theme: defaultTheme
                    }
                }
                newChart.engineType = "LINO";
                return newChart;
            }

            function acpIsDirty(ignoreThumbnailChanges) {
                try {
                    var savedACP2 = angular.copy(savedACP);
                    var acp = angular.copy($scope.acp);

                    if (ignoreThumbnailChanges) {
                        acp.charts.forEach(function(chart){
                            chart.def.thumbnailData = null;
                        });
                        savedACP2.charts.forEach(function(chart){
                            chart.def.thumbnailData = null;
                        });
                    }
                    return !angular.equals(acp, savedACP2);
                } catch (e) {
                    Logger.error(e);
                    return true;
                }
            }

            $scope.saveShaker = function() {
                Logger.info("Save ACP");

                var ignoreThumbnailChanges = !$scope.isProjectAnalystRW();
                if (!acpIsDirty(ignoreThumbnailChanges)) {
                    Logger.info("No changes: don't save shaker")
                    return;
                }

                if ($scope.isProjectAnalystRW()){
                    DataikuAPI.analysis.saveCore($scope.acp).success(function(data) {
                        if (!ActivityIndicator.isDisplayed()) {
                            ActivityIndicator.success("Charts saved");
                        }
                    }).error(setErrorInScope.bind($scope)).noSpinner();
                } else {
                    ActivityIndicator.warning("You don't have write access - not saving");
                }
            };

            /* ********************* Load callback ******************* */

            var cachedColumnSummaries = {};

            $scope.clearCachedSummaries = function(){
                $scope.charts.forEach(function(x) {
                    x.summary = null;
                });
                cachedColumnSummaries = {};
            }

            $scope.fetchColumnsSummaryForCurrentChart = function(forceRefresh){
                var currentChart = $scope.charts[$scope.currentChart.index];
                var dataSpec = $scope.getDataSpec();
                var cacheKey = JSON.stringify(dataSpec).dkuHashCode();

                var promise = null;
                if (cachedColumnSummaries[cacheKey] != null && !forceRefresh) {
                    Logger.info("Already cached for", dataSpec);
                    promise = $q.when(cachedColumnSummaries[cacheKey]);
                } else {
                    Logger.info("No cache for", dataSpec);
                    promise = DataikuAPI.shakers.charts.getColumnsSummary($stateParams.projectKey, dataSpec)
                        .error(setErrorInScope.bind($scope))
                        .then(function(response) {
                        cachedColumnSummaries[cacheKey] = response.data;
                        return response.data;
                    })
                }

                return promise.then(function(data){
                    currentChart.summary = data;
                    $scope.makeUsableColumns(data);
                });
            }

            $scope.overrideFormattingWithTheme = function (theme) {
                const currentChart = $scope.charts[$scope.currentChart.index];
                const currentChartCopy = angular.copy(currentChart);
                const formerTheme = currentChart.theme;
                currentChart.theme = theme;

                DSSVisualizationThemeUtils.applyToChart({ chart: currentChart.def, theme, formerTheme });
                DSSVisualizationThemeUtils.showThemeAppliedSnackbar(currentChart, currentChartCopy);
            };

            $scope.$on('$destroy', () => DSSVisualizationThemeUtils.hideThemeAppliedSnackbar());

            /* ********************* Main ******************* */
            var savedACP;
            var main = function(){
                WT1.event("analysis-charts-open");
                // Call ATSurveyService to trigger survey if needed
                ATSurveyService.updateCounter('ChartsOpen');
                TopNav.setLocation(TopNav.TOP_ANALYSES, TopNav.LEFT_ANALYSES, TopNav.TABS_ANALYSIS, "charts");
                TopNav.setItem(TopNav.ITEM_ANALYSIS, $stateParams.analysisId);

                ChartNavigationService.bindCurrentChartWithUrl();

                $scope.$on('$destroy', () => {
                    ChartNavigationService.unbindCurrentChartWithUrl()
                });

                $scope.$watchGroup(
                    ChartNavigationService.getCurrentChartWatchers($scope),
                    function (chartName) {
                        if (chartName) {
                            ChartNavigationService.updateCurrentChart($scope.charts, $scope.currentChart.index)
                        }
                    }
                );

                DataikuAPI.analysis.getCore($stateParams.projectKey, $stateParams.analysisId).success(function(data) {
                    $scope.acp = data;
                    $scope.shaker = data.script;
                    $scope.charts = data.charts;
                    $scope.currentChart.index = ChartNavigationService.getChartIndexFromId($scope.charts.map(chart => chart.def), $stateParams.chartId);

                    savedACP = angular.copy(data);
                    TopNav.setItem(TopNav.ITEM_ANALYSIS, $stateParams.analysisId, {name:data.name, dataset: data.inputDatasetSmartName});
                    TopNav.setPageTitle(data.name);

                    var inputDatasetLoc = DatasetUtils.getLocFromSmart($stateParams.projectKey, data.inputDatasetSmartName);
                    // set the context required for baseInit
                    $scope.analysisDataContext.inputDatasetLoc = inputDatasetLoc;
                    $scope.inputDatasetProjectKey = inputDatasetLoc.projectKey;
                    $scope.inputDatasetName = inputDatasetLoc.name;

                    DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey).success(function(data){
                        $scope.dataset = data;
                    }).error(setErrorInScope.bind($scope));

                    if ($scope.charts.length === 0) {
                        $scope.addChart({ datasetName: $scope.inputDatasetName });
                    }

                    $scope.addCustomMeasuresToScopeAndCache($scope.acp.customMeasures);
                    $scope.addBinnedDimensionToScopeAndCache($scope.acp.reusableDimensions);

                    $scope.$watch("charts[currentChart.index]", function(nv){
                        Logger.info("Chart changed, fetching summary and executing");
                        if (nv) {
                            $scope.fetchColumnsSummaryForCurrentChart()
                            .then(function(){
                                // Fixes a race condition that used to happen sometimes when explores.get returned before the
                                // event listeners in chart_logic.js were properly set up, causing the forceExecuteChart to be missed
                                // and nothing to be drawn.
                                $scope.forceExecuteChartOrWait();
                            })
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            }

            main();
        }
    }
})


app.directive("analysisChartSamplingEditor", function(DataikuAPI, $controller, $rootScope, $stateParams, $timeout, WT1, $q, CreateModalFromTemplate, DatasetUtils) {
    return {
        scope : {
            dataset: '=',
            chart : '=',
            script : '='
        },
        templateUrl : "/templates/analysis/charts-sampling-editor-tab.html",
        controller : function($scope){
            $scope.canCopySelectionFromScript = true;
            $controller("_ChartOnDatasetSamplingEditorBase", {$scope:$scope});

            $scope.save = function() {
                if ($scope.chart.refreshableSelection) {
                    if(!$scope.chart.copySelectionFromScript){
                        WT1.event("chart-sample-setting-update", {
                          chartId: `${$scope.dataset.projectKey.dkuHashCode()}.${$scope.dataset.name.dkuHashCode()}.${$scope.chart.def.name.dkuHashCode()}`,
                          analysis: true,
                          reuseFromExplore: false,
                          samplingMethod: $scope.chart.refreshableSelection.selection.samplingMethod,
                          recordsNumber: $scope.chart.refreshableSelection.selection.maxRecords,
                          targetRatio: $scope.chart.refreshableSelection.selection.targetRatio,
                          filtersNumber:
                            $scope.chart.refreshableSelection.selection.filter &&
                            $scope.chart.refreshableSelection.selection.filter.enabled &&
                            $scope.chart.refreshableSelection.selection.filter.uiData &&
                            $scope.chart.refreshableSelection.selection.filter.uiData.conditions
                              ? $scope.chart.refreshableSelection.selection.filter.uiData.conditions.length
                              : 0
                        });
                    } else {
                        WT1.event('chart-sample-setting-update', {
                            chartId: `${$scope.dataset.projectKey.dkuHashCode()}.${$scope.dataset.name.dkuHashCode()}.${$scope.chart.def.name.dkuHashCode()}`,
                            analysis: true,
                            reuseFromExplore: true
                        });
                    }
                    $scope.chart.refreshableSelection._refreshTrigger = new Date().getTime();
                }
                $rootScope.$emit("chartSamplingChanged", { chart: $scope.chart });
            };
            $scope.saveNoRefresh = function() {
                $rootScope.$emit("chartSamplingChanged", { chart: $scope.chart });
            };

        }
    }
});

app.component('columnAnalysis', {
    bindings: {
        columnAnalysis: '<',
        isNumeric: '<?',
        isDate: '<?',
        asList: '<?'
    },
    templateUrl: '/templates/analysis/column-analysis.html',
    controller: function() {
        const $ctrl = this;
        $ctrl.$onChanges = () => {
            $ctrl.data = null // reset data on any change to prevent from displaying old values
            if (!$ctrl.columnAnalysis) { // not ready
                return;
            }
            if ($ctrl.isNumeric) {
                $ctrl.data = $ctrl.columnAnalysis.numericalAnalysis;
            } else {
                $ctrl.data = $ctrl.columnAnalysis.alphanumFacet;
            }
        }
    }
});

})();
