from typing import Optional
from dataiku.eda.types import Literal

from statsmodels.stats.weightstats import DescrStatsW

from dataiku.doctor.utils import dku_nonaninf
from dataiku.eda.computations.computation import UnivariateComputation
from dataiku.eda.computations.context import Context
from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.types import SumModel, SumResultModel


class Sum(UnivariateComputation):
    def __init__(self, column: str, confidence: Optional[float]):
        super(Sum, self).__init__(column)
        self.confidence = confidence

    @staticmethod
    def get_type() -> Literal["sum"]:
        return "sum"

    @staticmethod
    def build(params: SumModel) -> 'Sum':
        return Sum(params['column'], params.get('confidence'))

    def apply(self, idf: ImmutableDataFrame, ctx: Context) -> SumResultModel:
        series = idf.float_col_no_missing(self.column)

        output: SumResultModel = {"type": Sum.get_type(), "value": series.sum()}
        if self.confidence is not None:
            lower, upper = DescrStatsW(series).tconfint_mean(alpha=1 - self.confidence)
            output["lower"] = dku_nonaninf(lower * len(series))
            output["upper"] = dku_nonaninf(upper * len(series))
        return output
