(function() {
'use strict';

const app = angular.module('dataiku.shaker');


app.constant("NORMALIZATION_MODES", [
    ["EXACT", "Exact"],
    ["LOWERCASE", "Ignore case"],
    ["NORMALIZED", "Normalize (ignore accents)"]
]);

app.controller('ShakerSelectColumnsController', function($scope) {
    $scope.uiState = {
        columnsSelectionMode: "ALL"
    };

    const refreshColumns = () => {
        if ($scope.shaker && $scope.shaker.columnsSelection && $scope.shaker.columnsSelection.list) {
            // get mode and all columns from the saved shaker
            $scope.uiState.columnsSelectionMode = $scope.shaker.columnsSelection.mode;
            $scope.allColumnNames = angular.copy($scope.shaker.columnsSelection.list.map((x) => {
                return { name : x.name, $selected : x.d };
            }));
        } else if ($scope.table) {
            // we don't have the saved selection from shaker yet, get all columns from the table
            $scope.allColumnNames = angular.copy($scope.table.allColumnNames.map((x) => {
                return { name : x, $selected : true };
            }));
        }
    }

    $scope.$watch("shaker", () => {
        if ($scope.shaker) {
            refreshColumns();
        }
    });

    $scope.$watch("table", () => {
        if ($scope.table) {
            refreshColumns();
        }
    });

    $scope.ok = () => {
        if ($scope.shaker) {
            $scope.shaker.columnsSelection.mode = $scope.uiState.columnsSelectionMode;
            if ($scope.shaker.columnsSelection.mode == "ALL") {
                $scope.shaker.columnsSelection.list = null;
            } else if ($scope.selection && $scope.selection.allObjects) {
                $scope.shaker.columnsSelection.list = angular.copy($scope.selection.allObjects.map((x) => {
                    return { name : x.name, d : x.$selected };
                }));
            }
        }
        $scope.dismiss();
        $scope.autoSaveForceRefresh();
    }
});


app.controller('ShakerSelectSortController', function($scope) {
    $scope.uiState = {
        query : ''
    };

    // init the 2 lists
    $scope.choicesLeft = [];
    $scope.choicesMade = [];
    
    $scope.$watch("shaker", () => {
        $scope.choicesMade = $scope.shaker && $scope.shaker.sorting ? angular.copy($scope.shaker.sorting) : [];
        const selectedColumnNames = new Set($scope.choicesMade.map(c => c.column));
        if ($scope.shaker && $scope.shaker.columnsSelection && $scope.shaker.columnsSelection.list) {
            $scope.choicesLeft = $scope.shaker.columnsSelection.list
                .filter(col => col.d) // only keep selected/displayed columns
                .filter(col => !selectedColumnNames.has(col.name)) // remove already selected
                .map(item => ({column: item.name, ascending:true}));
        } else if ($scope.table && $scope.table.headers) {
            $scope.choicesLeft = $scope.table.headers
                .filter(header => !selectedColumnNames.has(header.name)) // remove already selected
                .map(header => ({column: header.name, ascending:true}));
        }
    });

    // utils
    $scope.hasSearch = function() {
        return $scope.uiState.query;
    };
    $scope.resetSearch = function() {
        $scope.uiState.query = null;
    };
    $scope.toggle = function(column) {
        column.ascending = !column.ascending;
    };

    // list operations
    $scope.removeAll = function() {
        $scope.choicesLeft = [...$scope.choicesLeft, ...$scope.choicesMade];
        $scope.choicesMade = [];
    };
    $scope.add = function(column) {
        const i = $scope.choicesLeft.indexOf(column);
        if (i >= 0) {
            $scope.choicesLeft.splice(i, 1);
            $scope.choicesMade.push(column);
        }
    };
    $scope.remove = function(column) {
        const i = $scope.choicesMade.indexOf(column);
        if (i >= 0) {
            $scope.choicesMade.splice(i, 1);
            $scope.choicesLeft.push(column);
        }
    };

    $scope.ok = function() {
        if ($scope.shaker) {
            $scope.shaker.sorting = angular.copy($scope.choicesMade);
        }
        $scope.dismiss();
        $scope.autoSaveForceRefresh();
    }
});

app.controller('RegexBuilderController', function ($scope, $stateParams, DataikuAPI, FutureWatcher, SpinnerService, $filter, WT1) {
    $scope.uiState = {};
    $scope.customRegexError = "";
    $scope.firstSentence = "";
    $scope.sentences = [];
    $scope.selectionPositions = []; // changes when new selections or when new sentences
    $scope.selections = [];
    $scope.excludedSentences = [];
    $scope.patterns = [];
    $scope.columnName = "";
    $scope.onColumnNames = false;
    $scope.selectedPattern = null;
    $scope.wrapLines = false;
    $scope.warningMessage = "";
    $scope.lastRequestNumber = 0;
    const MULTI_OCCURRENCES_THRESHOLD = 0.1;

    $scope.createCustomPattern = function (regex) {
        return {
            oldRegex: regex || "",
            regex: regex || "",
            nbOK: -1,
            nbNOK: -1,
            extractions: [],
            errors: [],
        }
    };

    $scope.customPattern = $scope.createCustomPattern();

    $scope.removeNextStepsFromShaker = function (shaker, step) {
        const stepId = $scope.findStepId(step);
        if (typeof (stepId) !== 'undefined') {
            if (stepId.depth === 0) {
                shaker.steps = shaker.steps.slice(0, stepId.id);
            } else if (stepId.depth === 1) {
                shaker.steps[stepId.id].steps = shaker.steps[stepId.id].steps.slice(0, stepId.subId);
                shaker.steps = shaker.steps.slice(0, stepId.id + 1);
            }
        }
    };

    const findSelections = function (sentence, selections) {
        const foundSelections = [];
        for (const sel of selections) {
            if (sentence === sel.before + sel.selection + sel.after) {
                foundSelections.push({
                    start: sel.before.length,
                    end: sel.before.length + sel.selection.length,
                });
            }
        }
        return foundSelections;
    }

    const computeSelectionPositions = function (sentences, selections) {
        const selectionPositions = [];
        for (const sentence of sentences) {
            selectionPositions.push(findSelections(sentence, selections));
        }
        return selectionPositions;
    }

    const computeErrorPositionsOneRow = function (selectionPositions, extractionPositions, isExcluded) {
        extractionPositions = extractionPositions || [];
        if (isExcluded) {
            return extractionPositions;
        }
        if (selectionPositions.length == 0) {
            return [];
        }
        const errorPositions = [];
        for (const selection of selectionPositions) {
            if (!extractionPositions.some(extr => extr.start === selection.start && extr.end === selection.end)) {
                errorPositions.push(selection);
            }
        }
        for (const extraction of extractionPositions) {
            if (!selectionPositions.some(sel => sel.start === extraction.start && sel.end === extraction.end)) {
                errorPositions.push(extraction);
            }
        }
        return errorPositions;
    }

    $scope.computeErrorPositions = function (pattern) {
        const errorPositions = [];
        for (const [index, sentence] of $scope.sentences.entries()) {
            errorPositions.push(computeErrorPositionsOneRow($scope.selectionPositions[index], pattern.extractions[index], $scope.isUsedAsExclusion(sentence)));
        }
        return errorPositions;
    }

    $scope.selectPattern = function (pattern) {
        $scope.selectedPattern = pattern;
    }

    $scope.setDefaultSelectedPattern = function (onlyCustomComputed) {
        if (onlyCustomComputed) {
            if ($scope.selectedPattern.category !== 'userCustomPattern') {
                // if the custom pattern was not selected before , we should not force select it
                const selectedIndex = $scope.patterns.findIndex(p => p.regex == $scope.selectedPattern.regex);
                if (selectedIndex >= 0) {
                    $scope.selectPattern($scope.patterns[selectedIndex]);
                    return;
                }
            }
            $scope.selectPattern($scope.customPattern);
            return;
        }
        //select a default pattern
        if ($scope.patterns.length >= 1) {
            $scope.selectPattern($scope.patterns[0]);
        } else {
            $scope.selectPattern($scope.customPattern);
        }
    }

    $scope.filterRequestParameter = function () {
        return { "elements": $scope.buildFilterRequest($scope.shaker.explorationFilters) };
    }

    $scope.computePatterns = function (computeOnlyCustom) {
        $scope.lastRequestNumber++;
        const currentRequestNumber = $scope.lastRequestNumber;
        const shakerForQuery = $scope.shakerHooks.shakerForQuery();
        // Remove currently edited step and next steps from query
        if ($scope.editStep) {
            $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
        }
        let selections = [];
        let excludedSentences = [];
        const customRegex = $scope.customPattern.regex;
        if (!computeOnlyCustom) {
            selections = $scope.selections;
            excludedSentences = $scope.excludedSentences;
        }
        DataikuAPI.shakers.smartExtractor($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId,
            $scope.columnName,
            selections,
            excludedSentences,
            customRegex,
            $scope.onColumnNames,
            $scope.firstSentence,
            $scope.filterRequestParameter()
        ).success(function (initialResponse) {
            SpinnerService.lockOnPromise(FutureWatcher.watchJobId(initialResponse.jobId).success(function (data) {
                if (currentRequestNumber !== $scope.lastRequestNumber) return;
                if (!computeOnlyCustom) {
                    $scope.patterns = data.result.categories
                        .filter(c => c.name !== "userCustomPattern")
                        .map(c => c.propositions)
                        .flat();
                }
                $scope.customRegexError = data.result.customRegexError;
                const oldRegex = $scope.customPattern.regex
                const customPatternCategory = data.result.categories.find(c => c.name === "userCustomPattern");
                if (customPatternCategory && customPatternCategory.propositions.length >= 1) {
                    $scope.customPattern = customPatternCategory.propositions[0];
                }
                $scope.customPattern.oldRegex = oldRegex;
                $scope.sentences = data.result.sentences;
                $scope.selectionPositions = computeSelectionPositions($scope.sentences, $scope.selections);
                $scope.patterns = $scope.patterns.map(p => {
                    return {
                        ...p,
                        errors: $scope.computeErrorPositions(p),
                    }
                });
                $scope.customPattern.errors = $scope.computeErrorPositions($scope.customPattern)
                if (!$scope.customPattern) {
                    $scope.customPatterns = $scope.createCustomPattern();
                }
                $scope.setDefaultSelectedPattern(computeOnlyCustom);
            }));
        }).error(setErrorInScope.bind($scope));
    };

    $scope.addSelection = function (sentence, startOff, endOff) {
        if ($scope.isUsedAsExclusion(sentence)) {
            $scope.removeSentenceSelectionExcluded(sentence);
        }
        while (sentence[startOff] === " " && startOff <= sentence.length) {
            startOff++;
        }
        while (sentence[endOff - 1] === " " && endOff >= 0) {
            endOff--;
        }
        if (endOff <= startOff) {
            return;
        }
        const sel = {
            before: sentence.substring(0, startOff),
            selection: sentence.substring(startOff, endOff),
            after: sentence.substring(endOff),
        };
        if ($scope.isExistingSelection(sel)) return;
        $scope.selections.push(sel);
        $scope.computePatterns(false);
    };

    $scope.removeSelection = function (idx) {
        $scope.selections.splice(idx, 1);
        $scope.computePatterns(false);
    };

    $scope.removeSentenceSelectionExcluded = function (sentence) {
        $scope.selections = $scope.selections.filter(sel => sel.before + sel.selection + sel.after !== sentence);
        $scope.excludedSentences = $scope.excludedSentences.filter(excl => excl !== sentence);
    };

    $scope.ignoreErrors = function (sentence) {
        $scope.removeSentenceSelectionExcluded(sentence);
        $scope.computePatterns(false);
    };

    $scope.addExcludedSentence = function (sentence) {
        $scope.removeSentenceSelectionExcluded(sentence);
        $scope.excludedSentences.push(sentence);
        $scope.computePatterns(false);
    };

    $scope.removeExcludedSentence = function (idx) {
        $scope.excludedSentences.splice(idx, 1);
        $scope.computePatterns(false);
    };

    $scope.isExistingSelection = function (selection) {
        return $scope.selections.some(sel => sel.before === selection.before && sel.selection === selection.selection && sel.after === selection.after);
    };

    $scope.isUsedAsExclusion = function (sentence) {
        if ($scope.excludedSentences.includes(sentence)) {
            return true;
        }
        return false;
    };

    $scope.findStartOffset = function (nodes, selection) {
        //selection is not a string, it's a https://developer.mozilla.org/en-US/docs/Web/API/Selection
        //nodes are DOM elements
        let startOff = 0;
        for (const node of nodes) {
            if (node.nodeType == 1 && !selection.containsNode(node, true)) {
                startOff += node.textContent.length;
            }
            if (node.nodeType == 1 && selection.containsNode(node, true)) {
                // the selection is between anchorNode and focusNode, but they can either be at the begining or the end of the selection depending on the selection direction (to the right or to the left)
                const isAnchor = node.isSameNode(selection.anchorNode.parentElement);
                const isFocus = node.isSameNode(selection.focusNode.parentElement);
                if (isAnchor && isFocus) {
                    return startOff + Math.min(selection.anchorOffset, selection.focusOffset);
                }
                if (isAnchor) {
                    return startOff + selection.anchorOffset;
                }
                if (isFocus) {
                    return startOff + selection.focusOffset;
                }
            }
        }
    };

    $scope.isSingleNodeSelection = function (nodes, selection) {
        let containsAnchor = false;
        let containsFocus = false;
        for (const node of nodes) {
            if (node.isSameNode(selection.anchorNode.parentElement)) {
                containsAnchor = true;
            }
            if (node.isSameNode(selection.focusNode.parentElement)) {
                containsFocus = true;
            }
        }
        return containsAnchor && containsFocus;
    };

    $scope.onSelection = function (evt, sentence) {
        evt.stopPropagation();
        var userSelection;
        if (window.getSelection) {
            userSelection = window.getSelection();
        } else if (document.selection) {
            userSelection = document.selection.createRange();
        }
        const rowNodes = evt.currentTarget.childNodes;
        if (!$scope.isSingleNodeSelection(rowNodes, userSelection)) {
            return;
        }
        var selectedText = userSelection + "";
        if (userSelection.text) {
            selectedText = userSelection.text;
        }
        if (selectedText) {
            const startOff = $scope.findStartOffset(rowNodes, userSelection);
            $scope.addSelection(sentence, startOff, startOff + selectedText.length);
        }
    };

    $scope.onCustomRegexChange = function () {
        const pattern = $scope.customPattern;
        if (pattern.regex !== pattern.oldRegex) {
            $scope.computePatterns(true);
        }
    }

    $scope.getWT1Stats = function (action) {
        let stats = {};
        stats.action = action || '';
        stats.nbSelections = $scope.selections.length;
        if ($scope.selectedPattern != null) {
            stats.category = $scope.selectedPattern.category;
            stats.matchOK = $scope.selectedPattern.nbOK;
            stats.matchNOK = $scope.selectedPattern.nbNOK;
        }
        stats.calledFrom = $scope.calledFrom || '';
        return stats;
    };

    $scope.save = function () {
        const WT1stats = $scope.getWT1Stats("save")
        WT1.event("patternbuilder", WT1stats);
        if ($scope.selectedPattern != null) {
            if ($scope.deferred) {
                const extractingLines = $scope.selectedPattern.extractions.filter(extractions => extractions.length > 0).length;
                const multiExtractingLines = $scope.selectedPattern.extractions.filter(extractions => extractions.length > 1).length;
                let multiOccurenceRatio = 0;
                if (extractingLines >= 1) {
                    multiOccurenceRatio = multiExtractingLines / extractingLines;
                }
                const pattern = {
                    regex: $scope.selectedPattern.regex,
                    hasMultiOccurrences: multiOccurenceRatio > MULTI_OCCURRENCES_THRESHOLD,
                };
                $scope.deferred.resolve(pattern);
            }
        }
        $scope.dismiss();
    };

    $scope.cancel = function () {
        const WT1stats = $scope.getWT1Stats("cancel");
        WT1.event("patternbuilder", WT1stats);
        if ($scope.deferred) {
            $scope.deferred.reject();
        }
        $scope.dismiss();
    };

});

app.controller(
    "GrokDebuggerController",
    function ($scope, $stateParams, DataikuAPI) {
        $scope.sentences = [];
        $scope.captureOffsets = [];
        $scope.matchOffsets = [];
        const shakerForQuery = $scope.shakerHooks.shakerForQuery();

        $scope.removeNextStepsFromShaker = function (shaker, step) {
            const stepId = $scope.findStepId(step);
            if (typeof stepId !== "undefined") {
                if (stepId.depth === 0) {
                    shaker.steps = shaker.steps.slice(0, stepId.id);
                } else if (stepId.depth === 1) {
                    shaker.steps[stepId.id].steps = shaker.steps[stepId.id].steps.slice(
                        0,
                        stepId.subId
                    );
                    shaker.steps = shaker.steps.slice(0, stepId.id + 1);
                }
            }
        };

        $scope.loadGrokExpressionSample = function () {
            if ($scope.editStep) {
                $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
            }
            const sourceColumnIndex = $scope.step.$stepState.change.columnsBeforeStep.indexOf($scope.step.params.sourceColumn);
            resetErrorInScope($scope);
            DataikuAPI.shakers
                .getGrokExpressionSample(
                    $stateParams.projectKey,
                    $scope.inputDatasetProjectKey,
                    $scope.inputDatasetName,
                    shakerForQuery,
                    $scope.requestedSampleId,
                    0,
                    100,
                    sourceColumnIndex,
                    1, // number of columns (always one)
                    $scope.grokPattern || $scope.step.params.grokPattern
                )
                .success(function (response) {
                    $scope.sentences = response.table.content;
                    $scope.matchOffsets = response.matchOffsets.map((value) => [value]);
                    $scope.captureOffsets = response.captureOffsets;
                    $scope.matchedLines = response.matchedLines;
                    $scope.processedLines = response.processedLines;
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.handleChange = function () {
            $scope.loadGrokExpressionSample();
        };

        $scope.save = function () {
            if ($scope.deferred) {
                $scope.deferred.resolve($scope.grokPattern);
            }
            $scope.dismiss();
        };

        $scope.cancel = function () {
            if ($scope.deferred) {
                $scope.deferred.reject();
            }
            $scope.dismiss();
        };
    }
);
  

app.directive('overlapUnderlineHighlight', function () {
    const template = '<span ng-repeat="subSentence in subSentences"'
        + 'ng-class="{'
        + '\'overlap-underline-highlight--dark-green\': subSentence.darkGreen,'
        + '\'overlap-underline-highlight--light-green\': subSentence.lightGreen,'
        + '\'overlap-underline-highlight--error\': subSentence.error,'
        + '\'overlap-underline-highlight--crossed\': isCrossed}">'
        + '{{subSentence.value}}'
        + '</span>';
    return {
        template,
        restrict: 'AE',
        scope: {
            sentence: '=',
            isCrossed: '=',
            darkGreenOffsets: '=', // all offsets should look like [{start: 26, end: 41}, {start: 132, end: 138}]
            lightGreenOffsets: '=',
            errorOffsets: '=',
        },
        link: function (scope, element, attrs) {
            scope.$watch('[sentence, darkGreenOffsets, lightGreenOffsets, errorOffsets, isCrossed]', function (ov, nv) {
                if(scope.sentence === null) {
                    return;
                }
                const darkGreenOffsets = scope.darkGreenOffsets || [];
                const lightGreenOffsets = scope.lightGreenOffsets || [];
                const errorOffsets = scope.errorOffsets || [];
                scope.subSentences = [];
                let styleChangingIndexes = new Set([
                    0,
                    ...darkGreenOffsets.flatMap(o => [o.start, o.end]),
                    ...lightGreenOffsets.flatMap(o => [o.start, o.end]),
                    ...errorOffsets.flatMap(o => [o.start, o.end]),
                    scope.sentence.length,
                ]);
                const sortedIndexes = [...styleChangingIndexes].sort((a, b) => a - b);
                for (let i = 0; i < sortedIndexes.length - 1; i++) {
                    const start = sortedIndexes[i];
                    const end = sortedIndexes[i + 1];
                    let subsentence = {
                        value: scope.sentence.substring(start, end),
                        darkGreen: darkGreenOffsets.some(o => o.start <= start && o.end >= end),
                        lightGreen: lightGreenOffsets.some(o => o.start <= start && o.end >= end),
                        error: errorOffsets.some(o => o.start <= start && o.end >= end),
                    };
                    scope.subSentences.push(subsentence);
                };
            }, true)
        }
    }
});

app.controller('SmartDateController', function($scope, $stateParams, $timeout, DataikuAPI, WT1) {
    $scope.uiState = {
        customFormatInput: ""
    };

    $scope.removeNextStepsFromShaker = function(shaker, step) {
        const stepId = $scope.findStepId(step);
        if (typeof(stepId)!=='undefined') {
            if (stepId.depth == 0) {
                shaker.steps = shaker.steps.slice(0, stepId.id);
            } else if (stepId.depth == 1) {
                shaker.steps[stepId.id].steps = shaker.steps[stepId.id].steps.slice(0, stepId.subId);
                shaker.steps = shaker.steps.slice(0, stepId.id + 1);
            }
        }
    };

    $scope.setColumn  = function(name) {
        $scope.columnName = name;
        // Copy of the original shaker for the query
        const shakerForQuery = $scope.shakerHooks.shakerForQuery();
        // Remove currently edited step and next steps from query
        if ($scope.editStep) {
            $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
        }

        DataikuAPI.shakers.smartDateGuess($stateParams.projectKey,
            $scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId, $scope.columnName).success(function(data) {
            $scope.autodetected = data.formats;
            WT1.event("smartdate-guessed", {"nbGuessed" : $scope.autodetected.length});
            let detectedFormatIndex;
            for (const i in $scope.autodetected) {
                let fmt = $scope.autodetected[i];
                fmt.parsesPercentage = 100*fmt.nbOK/(fmt.nbOK + fmt.nbNOK + fmt.nbPartial);
                fmt.partialPercentage = 100*fmt.nbPartial/(fmt.nbOK + fmt.nbNOK + fmt.nbPartial);
                fmt.failsPercentage = 100 - fmt.parsesPercentage - fmt.partialPercentage;
                if ($scope.editFormat == fmt.format) {
                    detectedFormatIndex = i;
                }
            }

            let selectCustom = false
            if (!$scope.editFormat) {
                // New smart date or no format to edit => select first format if found
                if ($scope.autodetected.length > 0) {
                    $scope.selectFormat(0);
                } else { // select the custom format
                    selectCustom = true;
                }
            } else if (detectedFormatIndex >= 0) {
                // Found format in the guess list => select this one
                $scope.selectFormat(detectedFormatIndex);
            } else {
                selectCustom = true;
            }
            // need to validate the empty custom so that we can display examples (they come from backend and depend on the format)
            $scope.validateCustom(selectCustom);
        }).error(setErrorInScope.bind($scope));
    };

    WT1.event("smartdate-open");

    $scope.selectFormat = function(idx) {
        $scope.selectedFormat = $scope.autodetected[idx];
    };

    $scope.validateSeqId = 0;

    $scope.onCustomFormatClick = function() {
        $scope.selectedFormat = $scope.customFormat;
    };

    $scope.validateCustom = function(isSelected) {
        $scope.validateSeqId++;
        const seqId = $scope.validateSeqId;
        WT1.event("smartdate-validate-custom", {"format" : $scope.uiState.customFormatInput});
        // Get a copy of the current shaker for the query
        let shakerForQuery = $scope.shakerHooks.shakerForQuery();
        // Remove currently edited step and next steps from query
        if ($scope.editStep) {
            $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
        }
        DataikuAPI.shakers.smartDateValidate($stateParams.projectKey,
                $scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId,
                $scope.columnName, $scope.uiState.customFormatInput == null ? "" : $scope.uiState.customFormatInput).success(function(data) {
            if (seqId != $scope.validateSeqId) return;
            data.parsesPercentage = 100*data.nbOK/(data.nbOK + data.nbNOK + data.nbPartial);
            data.partialPercentage = 100*data.nbPartial/(data.nbOK + data.nbNOK + data.nbPartial);
            data.failsPercentage = 100 - data.parsesPercentage - data.partialPercentage;
            if (isSelected) {
                $scope.selectedFormat = data;
            }
            $scope.customFormat = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function() {
        if ($scope.selectedFormat != null && $scope.selectedFormat.validFormat) {
            WT1.event("smartdate-accept", {"format" : $scope.selectedFormat.format});
            if ($scope.deferred) {
                $scope.deferred.resolve([$scope.selectedFormat.format, $scope.selectedFormat.language]);
            }
            $scope.dismiss();
        }
    };

    $scope.cancel = function() {
        if ($scope.deferred) {
            $scope.deferred.reject();
        }
        $scope.dismiss();
    };

    $scope.$watch("uiState.customFormatInput", function(ov, nv) {
        if (ov != nv) {
            $timeout(function () {$scope.validateCustom(true);}, 200);
        }
    });
});

app.controller('DateParserController', function($scope, $q, $element, WT1, CreateModalFromTemplate) {
    if (!$scope.step.params.formats) $scope.step.params.formats = [''];
    $scope.formatItemTemplate = {format: ''};
    $scope.formatItems = $scope.step.params.formats.map(function(f) { return {format: f}; });
    $scope.formatsChanged = function() {
        [].splice.apply($scope.step.params.formats,
            [0, $scope.step.params.formats.length].concat($scope.formatItems.map(function(fi) { return fi.format; })));
        $scope.checkAndRefresh();
    };
    $scope.validateFormatList = function(it, itemIndex) {
        if (!it || !it.format || it.format.length == 0) return false;
        for (var i in $scope.formatItems) {
            if ($scope.formatItems[i].format == it.format) return false;
        }
        return true;
    };
    $scope.showSmartDateTool = function() {
        return $scope.columns && $scope.step.params.columns && $scope.step.params.columns[0] && $scope.columns.indexOf($scope.step.params.columns[0]) >= 0;
    };
    $scope.openSmartDateTool = function() {
        if (!$scope.step.params.columns || !$scope.step.params.columns[0] || $scope.columns.indexOf($scope.step.params.columns[0]) == -1) return;
        
        WT1.event("shaker-parsedate-edit-smartdate");
        var deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/smartdate-box.html", $scope, "SmartDateController",
            function(newScope) { newScope.$apply(function() {
                    newScope.deferred = deferred;
                    newScope.editStep = $scope.step;
                    newScope.editFormat = "";
                    newScope.setColumn($scope.step.params.columns[0]);
            }); }, "sd-modal");
        deferred.promise.then(function([newFormat, newFormatLanguage]) {
            if (!newFormat || newFormat.length == 0) return;
            // Checks if the new format is already present in the list and retrieve the old format index
            let isNewFormatThere, newFormatIdx;
            for (let i in $scope.formatItems) {
                if ($scope.formatItems[i].format == newFormat) {
                    isNewFormatThere = true;
                    newFormatIdx = i;
                    break;
                }
            }
            // Edit the new format if not present, otherwise focus on the existing input containing the new format
            if (!isNewFormatThere) {
                if ($scope.formatItems.length > 0) {
                    const lastItem = $scope.formatItems[$scope.formatItems.length - 1];
                    if (lastItem.format == '') {
                        $scope.formatItems.pop();
                    }
                }
                $scope.formatItems = [...$scope.formatItems, { format: newFormat }];
                newFormatIdx = $scope.formatItems.length - 1;
                $scope.formatsChanged();
            }
            if (newFormatLanguage) {
                $scope.step.params.lang = newFormatLanguage;
            }
            setTimeout(() => {
                // element not yet created ion the dom when checking
                $element.find(".dateFormatInput")[newFormatIdx].focus();
            }, 100);
        });
    };
    $scope.storageTemporalTypes = [
        [{name:'foo', type:'date'}, 'Datetime with tz'],
        [{name:'foo', type:'dateonly'}, 'Date only'],
        [{name:'foo', type:'datetimenotz'}, 'Datetime no tz']
    ];
});

app.controller('RenameColumnController', function($scope) {
    $scope.setColumn  = function(name) {
        $scope.columnName = name;
        $scope.renameTarget = name;
    }

    $scope.save = function() {
        if($scope.columnName!=$scope.renameTarget) {
            $scope.addStepNoPreviewAndRefresh("ColumnRenamer", {
                renamings : [
                    {"from" : $scope.columnName, "to" : $scope.renameTarget}
                ]
            });
            $scope.mergeLastColumnRenamers();
        }
        $scope.dismiss();
    }
});

app.controller('MoveColumnController', function($scope) {
    $scope.setColumn = function(name) {
        $scope.columnName = name;
        // Remove the column we want to move from the list of reference columns.
        let index = $scope.referenceColumnList.indexOf(name);
        $scope.referenceColumnList.splice(index, 1);
    };

    $scope.save = function() {
        $scope.addStepNoPreviewAndRefresh("ColumnReorder", {
            appliesTo : "SINGLE_COLUMN",
            columns: [$scope.columnName],
            referenceColumn: $scope.referenceColumn,
            reorderAction: $scope.reorderAction.name
        });
        $scope.mergeLastColumnReorders();
        $scope.dismiss();
    };

    $scope.reorderActions = [
        { name: "AT_START", label: "at beginning", needReferenceColumn: false },
        { name: "AT_END", label: "at end", needReferenceColumn: false },
        { name: "BEFORE_COLUMN", label: "before", needReferenceColumn: true },
        { name: "AFTER_COLUMN", label: "after", needReferenceColumn: true }
    ];

    $scope.reorderAction = $scope.reorderActions[0];
    $scope.referenceColumnList = $scope.tableModel.allColumnNames.slice();
});

app.controller('FillEmptyWithValueController', function($scope, DataikuAPI, MonoFuture, $stateParams) {
    var monoFuture = MonoFuture($scope);
    function analysis(callback) {
        monoFuture.exec(
            DataikuAPI.shakers.multiColumnAnalysis(
                $stateParams.projectKey,
                $scope.inputDatasetProjectKey, $scope.inputDatasetName, $scope.inputStreamingEndpointId, $scope.shakerHooks.shakerForQuery(),
                $scope.requestedSampleId, $scope.columns, $scope.source)
        ).success(function(data) {
            if (data.hasResult) {
                callback(data.result);
            }
        }).error(setErrorInScope.bind($scope));
    }
    $scope.source = 'constant';
    $scope.isNumericOnly = false;
    $scope.setColumns = function(cols) {
        $scope.columns = cols;
        $scope.columnName = cols.length === 1 ? cols[0] : null;
    }
    $scope.save = function() {
        var fn = $scope.columns.length === 1 ? 'addStep' : 'addStepNoPreview';
        if ($scope.source === 'constant') {
            if ($scope.columns.length == 1) {
                $scope[fn]("FillEmptyWithValue", { appliesTo : "SINGLE_COLUMN", columns: $scope.columns, value: $scope.valueToFill });
            } else {
                $scope[fn]("FillEmptyWithValue", { appliesTo : "COLUMNS", columns: $scope.columns, value: $scope.valueToFill });
            }
            $scope.autoSaveForceRefresh();
            $scope.dismiss();
        } else {
            analysis(function(data) {
                for (var c in data) {
                    $scope[fn]("FillEmptyWithValue", { appliesTo : "SINGLE_COLUMN", columns: [c], value: data[c] });
                }
                $scope.autoSaveForceRefresh();
                $scope.dismiss();
            });
        }
    }
});


app.controller('MassRenameColumnsController', function($scope, DataikuAPI) {
    var edits = {
        findReplace: function(c) {
            const pattern = $scope.find;
            if (pattern && pattern !== "") {
                if ($scope.useRegex) {
                    const re = new RegExp(pattern, $scope.ignoreCase ? "ig" : "g");
                    return c.replaceAll(re, $scope.replace);
                } else {
                    if ($scope.ignoreCase) {
                        let index = c.toLowerCase().indexOf(pattern.toLowerCase());
                        if (index >= 0) {
                            let newName = c.substring(0, index) + $scope.replace;
                            let remaining = c.substring(index + pattern.length);
                            while (remaining.length > 0) {
                                index = remaining.toLowerCase().indexOf(pattern.toLowerCase());
                                if (index >= 0) {
                                    newName += remaining.substring(0, index) + $scope.replace;
                                    remaining = remaining.substring(index + pattern.length);
                                } else {
                                    newName += remaining;
                                    remaining = "";
                                }
                            }
                            return newName;
                        }
                    } else {
                        return c.replaceAll(pattern, $scope.replace);
                    }
                }
            }
            return c;
        },
        addPrefix: c => $scope.prefix ? $scope.prefix + c : c,
        addSuffix: c => $scope.suffix ? c + $scope.suffix : c,
        removePrefix: c => {
            const prefix = $scope.prefix;
            return prefix && (c.startsWith(prefix) || $scope.ignoreCase && c.toLowerCase().startsWith(prefix.toLowerCase()))
                ? c.substring(prefix.length) : c;
        },
        removeSuffix: c => {
            const suffix = $scope.suffix;
            return suffix && (c.endsWith(suffix) || $scope.ignoreCase && c.toLowerCase().endsWith(suffix.toLowerCase()))
                ? c.substring(0, c.length - suffix.length) : c;
        },
        lowercase: c => c.toLowerCase(),
        uppercase: c => c.toUpperCase()
    };
    $scope.replace = '';
    $scope.setColumns = function(cols) {
        $scope.columns = cols;
        $scope.columnName = cols.length === 1 ? cols[0] : null;
    }
    $scope.edit = 'findReplace';
    $scope.ignoreCase = true;

    $scope.validateRegex = function() {
        $scope.massRenameForm.$setValidity("regex", true);
        if ($scope.edit === 'findReplace' && $scope.useRegex) {
            try {
                // Try to compile the pattern
                new RegExp($scope.find);
            } catch (error) {
                $scope.massRenameForm.$setValidity("regex", false);
            }
        }
    }

    $scope.$watch('[edit, useRegex]', () => {
        $scope.validateRegex();
    });

    $scope.save = async function() {
        let dirty = false;
        let renamings = [];

        if ($scope.edit === "normalizeSpecialChars") {
            /* This one does an API call, so we batch it */
            const editedList = (await DataikuAPI.shakers.convertToASCII($scope.columns, true, true)).data;
            for (let i = 0; i < $scope.columns.length; i++) {
                const c = $scope.columns[i];
                const c2 = editedList[i];
                if (c2 && c2 !== c) {
                    dirty = true;
                    renamings.push({ from: c, to: c2 });
                }
            }
        } else {
            $scope.columns.forEach(function(c) {
                var c2 = edits[$scope.edit](c);
                if (c2 && c2 !== c) {
                    dirty = true;
                    renamings.push({ from: c, to: c2 });
                }
            });
        }

        if (dirty) {
            $scope.doRenameColumns(renamings);
        }
        $scope.dismiss();
    }
});


app.controller('MultiRangeController', function($scope) {
    $scope.setColumns = function(cols) {
        $scope.columns = cols;
        $scope.columnName = cols.length === 1 ? cols[0] : null;
    }

    $scope.keep = true;
    $scope.save = function() {
        $scope.addStepNoPreview("FilterOnNumericalRange", {
            appliesTo : "COLUMNS",
            columns: $scope.columns,
            min: $scope.min,
            max: $scope.max,
            action : $scope.keep ? "KEEP_ROW" : "REMOVE_ROW"
        });
        $scope.autoSaveForceRefresh();
        $scope.dismiss();
    }
});


app.controller('FindReplaceController', function($scope, translate) {
    const NORMALIZATION_MODES = [
        ["EXACT", translate("SHAKER.NORMALIZATION_MODES.EXACT", "Exact")],
        ["LOWERCASE", translate("SHAKER.NORMALIZATION_MODES.LOWERCASE", "Ignore case")],
        ["NORMALIZED", translate("SHAKER.NORMALIZATION_MODES.NORMALIZED", "Normalize (ignore accents)")]
    ]
    $scope.$watch("step.params.matching", function(nv, ov) {
        if (nv && nv === "FULL_STRING") {
            $scope.normalizationModes = NORMALIZATION_MODES;
        } else if (nv) {
            $scope.normalizationModes = NORMALIZATION_MODES.filter(function(mode) { return mode[0] !== 'NORMALIZED'; });
            if ($scope.step.params.normalization === "NORMALIZED") {
                $scope.step.params.normalization = "EXACT";
            }
        }
    })
});


app.controller('SwitchCaseController', function($scope, translate) {
    $scope.normalizationModes = [
       ["EXACT", translate("SHAKER.NORMALIZATION_MODES.EXACT", "Exact")],
       ["LOWERCASE", translate("SHAKER.NORMALIZATION_MODES.LOWERCASE", "Ignore case")],
       ["NORMALIZED", translate("SHAKER.NORMALIZATION_MODES.NORMALIZED", "Normalize (ignore accents)")]
   ];
});

app.controller('FlagOnValuesController', function($scope, translate) {
    $scope.normalizationModes = [
        ["EXACT", translate("SHAKER.NORMALIZATION_MODES.EXACT", "Exact")],
        ["LOWERCASE", translate("SHAKER.NORMALIZATION_MODES.LOWERCASE", "Ignore case")],
        ["NORMALIZED", translate("SHAKER.NORMALIZATION_MODES.NORMALIZED", "Normalize (ignore accents)")]
    ];
});

app.controller('VisualIfRuleController', function($scope, $controller, CreateCustomElementFromTemplate, DataikuAPI, $stateParams, ShakerPopupRegistry, $rootScope){
    $scope.modalShown = false;
    $scope.parentDismiss = function() {
    	$rootScope.$broadcast("dismissModalInternal_");
    	$scope.modalShown = false;
    }

    $scope.editRule = function(){
        $scope.editRule_(angular.copy($scope.step.params.visualIfDesc));
    }

    $scope.columns = [];
    const updateColumns = () => {
        if ($scope.step.$stepState.change) {
            $scope.columns = $scope.step.$stepState.change.columnsBeforeStep;
        }
    }
    updateColumns();

    const stopWatchingForColumnsCompute = $scope.$watch('[step.preview, shaker.steps]', () => {
        updateColumns();
    });

    $scope.editRule_ = function(visualIfDesc) {
        if ($scope.modalShown) {
            $scope.parentDismiss();
        } else {
            ShakerPopupRegistry.dismissAllAndRegister($scope.parentDismiss);
            $scope.modalShown = true;
            DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey)
                .success(function(data){
                    CreateCustomElementFromTemplate("/static/dataiku/processors/visual-if/visual-if-editor/visual-if-rule.html", $scope, null, function(newScope) {
                        $scope.editing = {};
                        $scope.editing.visualIfDesc = visualIfDesc;
                        $scope.step.columnsCache = $scope.columns;
                    }, $scope.customFormulaEdition.displayCustomFormula);
            }).error(setErrorInScope.bind($scope));
        }
    }

    if ($scope.step.isNew === true) {
        $scope.editRule();
    }

    $scope.$on("resetModalShownInternal_", function() {
        $scope.modalShown = false;
    });

    $scope.$on("$destroy", stopWatchingForColumnsCompute);

});

app.controller('MeaningTranslateController', function($scope) {
    $scope.meanings = $scope.appConfig.meanings.categories
        .filter(function(cat) { return cat.label === "User-defined"; })[0]
        .meanings.filter(function(meaning) { return meaning.type === 'VALUES_MAPPING'; });
});


app.controller('CurrencySplitterController', function($scope) {
    $scope.$watch("step.params.inCol", function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
          $scope.step.params.outColCurrencyCode = $scope.step.params.inCol + "_currency_code";
          $scope.step.params.outColAmount = $scope.step.params.inCol + "_amount";
        }
    }, true);

    /* TODO: This should rather be done by a "default values" section ... */
});


app.controller('ColumnSplitterController', function($scope) {
    $scope.$watch("step.params.inCol", function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
            $scope.step.params.outColPrefix = $scope.step.params.inCol + "_";
        }
    }, true);

    /* TODO: This should rather be done by a "default values" section ... */
    $scope.$watch("step.params.limitOutput", function(nv, ov) {
        if ($scope.step.params.limitOutput && !$scope.step.params.limit) {
            $scope.step.params.limit = 1;
            $scope.step.params.startFrom = "beginning";
        }
    }, true);
});

app.controller('SplitIntoChunksController', function($scope) {
    $scope.$watch("step.params.inCol", function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
            $scope.step.params.outCol = $scope.step.params.inCol + "_chunked";
        }
    }, true);
    $scope.getTemplate = function() {
        return {
            value: "",
            isDefault: false,
            enabled: true
        }
    }
    $scope.disableRemove = function(it) {
        return it.isDefault
    }
});

app.controller('CurrencyConverterController', function($scope, DataikuAPI, translate) {

    $scope.currencies = [
        ["AED", "AED (United Arab Emirates Dirham)"],
        ["AUD", "AUD (Australian Dollar)"],
        ["BGN", "BGN (Bulgarian Lev)"],
        ["BND", "BND (Brunei Dollar)"],
        ["BRL", "BRL (Brazilian Real)"],
        ["BWP", "BWP (Botswana Pula)"],
        ["CAD", "CAD (Canadian Dollar)"],
        ["CHF", "CHF (Swiss Franc)"],
        ["CLP", "CLP (Chilean Peso)"],
        ["CNY", "CNY (Chinese Yuan)"],
        ["CYP", "CYP (Cypriot Pound)"],
        ["CZK", "CZK (Czech Koruna)"],
        ["DKK", "DKK (Danish Krone)"],
        ["DZD", "DZD (Algerian Dinar)"],
        ["EEK", "EEK (Estonian Kroon)"],
        ["EUR", "EUR (Euro)"],
        ["GBP", "GBP (British Pound Sterling)"],
        ["HKD", "HKD (Hong Kong Dollar)"],
        ["HRK", "HRK (Croatian Kuna)"],
        ["HUF", "HUF (Hungarian Forint)"],
        ["IDR", "IDR (Indonesian Rupiah)"],
        ["ILS", "ILS (Israeli Shekel)"],
        ["INR", "INR (Indian Rupee)"],
        ["ISK", "ISK (Icelandic Krona)"],
        ["JPY", "JPY (Japanese Yen)"],
        ["KRW", "KRW (South Korean Won)"],
        ["KWD", "KWD (Kuwaiti Dinar)"],
        ["LTL", "LTL (Lithuanian Litas)"],
        ["LVL", "LVL (Latvian Lats)"],
        ["MTL", "MTL (Maltese Lira)"],
        ["MUR", "MUR (Mauritian Rupee)"],
        ["MXN", "MXN (Mexican Peso)"],
        ["MYR", "MYR (Malaysian Ringgit)"],
        ["NOK", "NOK (Norwegian Krone)"],
        ["NZD", "NZD (New Zealand Dollar)"],
        ["OMR", "OMR (Omani Rial)"],
        ["PEN", "PEN (Peruvian Sol)"],
        ["PHP", "PHP (Philippine Peso)"],
        ["PLN", "PLN (Polish Zloty)"],
        ["QAR", "QAR (Qatari Riyal)"],
        ["ROL", "ROL (Romanian Leu (Old))"],
        ["RON", "RON (Romanian Leu (New))"],
        ["RUB", "RUB (Russian Ruble)"],
        ["SAR", "SAR (Saudi Riyal)"],
        ["SEK", "SEK (Swedish Krona)"],
        ["SGD", "SGD (Singapore Dollar)"],
        ["SIT", "SIT (Slovenian Tolar)"],
        ["SKK", "SKK (Slovak Koruna)"],
        ["THB", "THB (Thai Baht)"],
        ["TRL", "TRL (Turkish Lira (Old))"],
        ["TRY", "TRY (Turkish Lira (New))"],
        ["TTD", "TTD (Trinidad and Tobago Dollar)"],
        ["USD", "USD (United States Dollar)"],
        ["UYU", "UYU (Uruguayan Peso)"],
        ["ZAR", "ZAR (South African Rand)"]
    ]

    $scope.dateInputs = [];
    DataikuAPI.shakers.getLastKnownCurrencyRateDate().success(function(data) {
        $scope.dateInputs = buildDateInputsList(data);
    }).error(function() {
        $scope.dateInputs = buildDateInputsList(translate("SHAKER.PROCESSOR.CurrencyConverter.ERROR.UNKNOWN_DATE", "unknown date"));
    });

    function buildDateInputsList(lastKnownRateDate) {
        return [
            ["LATEST", translate("SHAKER.PROCESSOR.CurrencyConverter.DATE.LATEST", "Last known rates ({{date}})", {date: lastKnownRateDate})],
            ["COLUMN", translate("SHAKER.PROCESSOR.CurrencyConverter.DATE.COLUMN", "From Column (Date)")],
            ["CUSTOM", translate("SHAKER.PROCESSOR.CurrencyConverter.DATE.CUSTOM", "Custom input")]
        ];
    }

    function isColumnValid(col, meaning) {
        if(!$scope.table) return true;
        return $scope.table.headers.some(function(h) { return h.name === col && h.selectedType.name === meaning; });
    }

    $scope.$watch('step.params.inputColumn', function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
            $scope.step.params.outputColumn = $scope.step.params.inputColumn + "_";
        }
    });

    $scope.$watch("step.params.refDateColumn", function(nv, ov) {
        $scope.dateColumnIsValid = isColumnValid(nv, "Date");
        if (nv != ov)
            $scope.processorForm.dateReferenceColumn.$setValidity('columnTypeInvalid', $scope.dateColumnIsValid);
    });

    $scope.$watch("step.params.refCurrencyColumn", function(nv, ov) {
        $scope.currencyColumnIsValid = isColumnValid(nv, "CurrencyMeaning");
        if (nv != ov)
            $scope.processorForm.currencyReferenceColumn.$setValidity('columnTypeInvalid', $scope.currencyColumnIsValid);
    });

    $scope.$watch("step.params.refDateCustom", function(nv, ov) {
        if (nv != ov) {
            var minDate = new Date("1999-01-04");
            /*
             * Date() constructor accepts yyyy or yyyy-MM or yyyy-MM-dd
             * We want to restrict user's entry to yyyy-MM-dd which explains
             * the Regex, checking XXXX-XX-XX, with X any number.
             * Date() constructor will check if it's a valid date
             * If it's not, then "date" will be false
             */
            var dateFormat = new RegExp("^[0-9]{4}(-[0-9]{2}){2}$").test(nv);
            var date = new Date(nv);
            $scope.processorForm.dateReferenceCustom.$setValidity('Date type invalid', dateFormat && date && date > minDate);
            $scope.outOfDateReference = (dateFormat && date && date < minDate);
        }
    });
});

app.controller('DateDifferenceController', function($scope) {
    // [Legacy Compatibility] DateDifference step parameters created before v14 are missing `calendar_id` and `timezone_id`.
    // This fix set the default values ("FR" for calendar_id and "use_preferred_timezone" for timezone_id) to match what is done in the backend
    if ($scope.step.params.calendar_id === undefined) {
        $scope.step.params.calendar_id = "FR";
    }
    if ($scope.step.params.timezone_id === undefined) {
        $scope.step.params.timezone_id = "use_preferred_timezone";
    }
});


app.controller('RegexpExtractorController', function($scope, $q, CreateModalFromTemplate) {
    $scope.validColumn = function() {
        return $scope.step.$stepState.change && $scope.step.$stepState.change.columnsBeforeStep.indexOf($scope.step.params.column) !== -1;
    };
    $scope.openSmartRegexBuilder = function() {
        let deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", $scope, "RegexBuilderController",
            function(newScope) {
                newScope.$apply(function() {
                    newScope.deferred = deferred;
                    newScope.columnName = $scope.step.params.column;
                    newScope.editStep = $scope.step;
                    newScope.calledFrom = "shaker_recipe-regexpextractor";
                    newScope.customPattern = newScope.createCustomPattern($scope.step.params.pattern);
                    newScope.computePatterns(false);
                });
                deferred.promise.then(function(newPattern) {
                    $scope.step.params.pattern = newPattern.regex;
                    $scope.step.params.extractAllOccurrences = newPattern.hasMultiOccurrences;
                    $scope.checkAndRefresh();
                });
            },
            "sd-modal");
    };
});

app.controller('GrokProcessorController', function($scope, $q, CreateModalFromTemplate) {
    $scope.validColumn = function() {
        return $scope.step.$stepState.change && $scope.step.$stepState.change.columnsBeforeStep.indexOf($scope.step.params.sourceColumn) !== -1;
    };
    $scope.openGrokDebugger = function() {
        let deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/grokdebugger-box.html", $scope, "GrokDebuggerController",
            function(newScope) {
                newScope.$apply(function() {
                    newScope.deferred = deferred;
                    newScope.columnName = $scope.step.params.sourceColumn;
                    newScope.editStep = $scope.step;
                    newScope.calledFrom = "shaker_recipe-grokprocessor";
                    newScope.grokPattern = $scope.step.params.grokPattern || '';
                    newScope.debugColumns =  {...$scope.columns};
                    newScope.loadGrokExpressionSample();
                });
                deferred.promise.then(function(newGrokPattern) {
                    $scope.step.params.grokPattern = newGrokPattern;
                    $scope.checkAndRefresh();
                });
            },
            "sd-modal");
    };
});

app.controller('MultiColumnByPrefixFoldController', function($scope, $q, CreateModalFromTemplate) {
    $scope.openSmartRegexBuilder = function() {
        var deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", $scope, "RegexBuilderController",
            function(newScope) {
                newScope.deferred = deferred;
                newScope.$apply(function() {
                    newScope.onColumnNames = true;
                    newScope.editStep = $scope.step;
                    newScope.calledFrom = "shaker_recipe-multi_column_by_prefix_fold";
                    newScope.customPattern = newScope.createCustomPattern($scope.step.params.columnNamePattern);
                    newScope.computePatterns(false);
                });
                deferred.promise.then(function(newPattern) {
                    let columnNamePattern = newPattern.regex;
                    if (!columnNamePattern.startsWith('.*?')) {
                        columnNamePattern = ".*?" + columnNamePattern;
                    }
                    if (!columnNamePattern.endsWith('.*')) {
                        columnNamePattern = columnNamePattern + ".*";
                    }
                    $scope.step.params.columnNamePattern = columnNamePattern;
                    $scope.checkAndRefresh();
                });
            },
            "sd-modal");
    };
});


app.controller('PythonUDFController', function(
       $scope, $timeout, $rootScope, $q, $stateParams,
       CreateModalFromTemplate, CreateCustomElementFromTemplate, DataikuAPI, ShakerPopupRegistry, CodeMirrorSettingService) {

    var pythonSourceCodes = {

"CELL_true": "\
# Modify the process function to fit your needs\n\
import pandas as pd\n\
def process(rows):\n\
    # In 'cell' mode, the process function must return\n\
    # a single Pandas Series for each block of rows,\n\
    # which will be affected to a new column.\n\
    # The 'rows' argument is a dictionary of columns in the\n\
    # block of rows, with values in the dictionary being\n\
    # Pandas Series, which additionally holds an 'index'\n\
    # field.\n\
    return pd.Series(len(rows), index=rows.index)\n"
,
"ROW_true": "\
# Modify the process function to fit your needs\n\
import pandas as pd\n\
def process(rows):\n\
    # In 'row' mode, the process function \n\
    # must return the full rows.\n\
    # The 'rows' argument is a dictionary of columns in the\n\
    # block of rows, with values in the dictionary being\n\
    # Pandas Series, which additionally holds an 'index'\n\
    # field.\n\
    # You may modify the 'rows' in place to\n\
    # keep the previous values of the row.\n\
    # Here, we simply add two new columns.\n\
    rows[\"rowLength\"] = pd.Series(len(rows), index=rows.index)\n\
    rows[\"static_value\"] = pd.Series(42, index=rows.index)\n\
    return rows\n"
,
"MULTI_ROWS_true": "\
# Modify the process function to fit your needs\n\
import numpy as np, pandas as pd\n\
def process(rows):\n\
    # In 'multi rows' mode, the process function\n\
    # must return an indexed dictionary of vectors,\n\
    # either built by modifying the 'rows' \n\
    # parameter, or by returning a pandas DataFrame.\n\
    # To get an input dataframe, use\n\
    # rows.get_dataframe([col_name1, ...])\n\
    input_index = rows.index\n\
    # the values in the output index indicate which\n\
    # row of the input is used as base for the \n\
    # output rows. -1 signals that the new row comes\n\
    # from a blank base\n\
    new_index = np.concatenate([input_index, -1 * np.ones(input_index.shape[0])])\n\
    rows.index = new_index\n\
    # input columns are passed as pandas Series\n\
    existing_column_name = rows.columns[0]\n\
    existing_column = rows[existing_column_name]\n\
    rows[existing_column_name] = pd.concat([existing_column, existing_column])\n\
    # new columns can be numpy arrays\n\
    rows[\"static_value\"] = 42 * np.ones(2 * input_index.shape[0])\n\
    # the index field of the 'rows' parameter\n\
    # is a numpy array\n\
    rows[\"index\"] = np.concatenate([input_index, input_index])\n\
    return rows"
,
"CELL_false": "\
# Modify the process function to fit your needs\n\
def process(row):\n\
    # In 'cell' mode, the process function must return\n\
    # a single cell value for each row,\n\
    # which will be affected to a new column.\n\
    # The 'row' argument is a dictionary of columns of the row\n\
    return len(row)\n"
,
"ROW_false": "\
# Modify the process function to fit your needs\n\
def process(row):\n\
    # In 'row' mode, the process function \n\
    # must return the full row.\n\
    # The 'row' argument is a dictionary of columns of the row\n\
    # You may modify the 'row' in place to\n\
    # keep the previous values of the row.\n\
    # Here, we simply add two new columns.\n\
    row[\"rowLength\"] = len(row)\n\
    row[\"static_value\"] = 42\n\
    return row\n"
,
"MULTI_ROWS_false": "\
# Modify the process function to fit your needs\n\
def process(row):\n\
    # In 'multi rows' mode, the process function\n\
    # must return an iterable list of rows.\n\
    ret = []\n\
    # Here we append a new row with only one column\n\
    newrow1 = { \"previous_row_length\" : len(row) }\n\
    ret.append(newrow1)\n\
    # We can also modify the original row and reappend it\n\
    row[\"i\"] = 3\n\
    ret.append(row)\n\
    \n\
    return ret"
}

    $scope.editorOptions = CodeMirrorSettingService.get('text/x-python', {onLoad: function(cm) {$scope.codeMirror = cm}});

    $scope.$watch("[step.params.mode,step.params.vectorize, step.params.useKernel]", function(nv, ov) {
        if (!nv) return;
        // put defaults if they're not here already
        if ($scope.step.params.vectorize == null) {
            $scope.step.params.vectorize = false;
        }
        if ($scope.step.params.vectorSize == null) {
            $scope.step.params.vectorSize = 256;
        }
        const oldVectorized = ov[1] && ov[2];
        const newVectorized = nv[1] && nv[2];
        const oldDefaultPythonSourceCode = pythonSourceCodes[ov[0] + '_' + oldVectorized];
        const newDefaultPythonSourceCode = pythonSourceCodes[nv[0] + '_' + newVectorized];

        let oldPythonSource = $scope.step.params.pythonSourceCode;

        /* If we have already some code but it was the example code of the previous mode,
        then override it */
        if (oldPythonSource == oldDefaultPythonSourceCode) {
            oldPythonSource = null;
        }

        if ( (!oldPythonSource) || oldPythonSource.trim().length == 0) {
            $scope.step.params.pythonSourceCode = newDefaultPythonSourceCode;
        }
    }, true);

    $scope.parentDismiss = function() {
    	$rootScope.$broadcast("dismissModalInternal_");
    	$scope.modalShown = false;
    }
    $scope.$on("dismissModals", $scope.parentDismiss);

    $scope.hooks= {
            ok : function(){
                throw Error("not implemented");
            },
		    apply : function(){
		        throw Error("not implemented");
		    }
        };

    $scope.editPythonSource = function() {
        if ($scope.modalShown) {
            $scope.parentDismiss();
        } else {
            ShakerPopupRegistry.dismissAllAndRegister($scope.parentDismiss);
            $scope.modalShown = true;
            CreateCustomElementFromTemplate("/templates/shaker/pythonedit-box.html", $scope, null, function(newScope) {
                var pythonSource = $scope.step.params.pythonSourceCode;
                var mode = $scope.step.params.mode;
                var vectorize = $scope.step.params.vectorize;
                newScope.modified = false;
                if ( (!pythonSource) || pythonSource.trim().length == 0) {
                    newScope.pythonSource = pythonSourceCodes[mode + '_' + vectorize];
                    newScope.modified = true;
                }
                else {
                    newScope.pythonSource = $scope.step.params.pythonSourceCode;
                }

                $scope.hooks.apply = function() {
                	if ( newScope.modified ) {
                		$scope.step.params.pythonSourceCode = newScope.pythonSource;
                		$scope.checkAndRefresh();
                	}
                };
                $scope.hooks.ok = function() {
                	$scope.hooks.apply();
                	$scope.parentDismiss();
                };

                newScope.checkSyntax = function(pythonCode) {
                	var stepPosition = $scope.findStepId($scope.step);
                	DataikuAPI.shakers.validateUdf($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName, $scope.shakerHooks.shakerForQuery()
                			, $scope.requestedSampleId, pythonCode, stepPosition.id, stepPosition.subId || 0, stepPosition.depth).success(function(data) {
                				newScope.validationError = data;
                			}).error(setErrorInScope.bind($scope));
                };

                $scope.uiState = { codeSamplesSelectorVisible: false };
                var insertCode = function (codeToInsert) {
                 	//timeout to make sure of an angular safe apply
                  	$timeout(function() {
                   		$scope.codeMirror.replaceSelection(codeToInsert + '\n\n', "around");
                   	});
                   	$scope.codeMirror.focus();
                };
                $scope.insertCodeSnippet = function(snippet) {
                    insertCode(snippet.code);
                };
            } , $scope.customFormulaEdition.displayCustomFormula);
        }
    };
});

/**
 * Formula function hinter directive
 *  @param {string}     name        - Name of the function
 *  @param {boolean}    applied     - Is function applied to column
 *  @param {string}     arguments   - Argument list as string, separated by ","
 *  @param {string}     description - Function description
 *  @param {string}     deprecated  - Is function deprecated
 *  @param {string}     deprecationNotice - Optional additional message displayed when function is deprecated
 *  @param {number}     left        - Absolute CSS position left
 *  @param {number}     top         - Absolute CSS position top
 *  @param {number}     cursor      - Cursor position in the current line
 *  @param {string}     line        - Line of code user actually typing into
 *  @param {number}     start       - Position of the function token in the code line
 */
app.directive('formulaFunctionHinter', function($compile, translate) {
    return {
        template: ``,
        restrict :'E',
        scope : {
            name: "<",
            applied: "<",
            arguments: "<",
            deprecated: "<",
            deprecationNotice: "<",
            description: "<",
            examples: "<",
            left: "<",
            top: "<",
            cursor: '<',
            line: '<',
            start: '<'
        },
        link: function($scope) {
            $scope.translate = translate;
            const removeAnchor = () => {
                const prevAnchor = document.getElementById('formula-function-hinter');
                if(prevAnchor) {
                    prevAnchor.parentNode.removeChild(prevAnchor);
                }
            }

            removeAnchor();
            $scope.$on('$destroy', removeAnchor);

            const formatExamples = function(name, examples) {
                if (!examples) {
                    return "";
                } 
                return examples.map((example) =>
                        `<div class="formula-tooltip__example">
                            <code>${name}(${example.args.join(", ")})</code> returns <code>${example.result}</code>
                        </div>`).join("");
            }

            // Element need to be placed at root level to avoid overflowing issue due to editor container
            $scope.htmlExamples = formatExamples($scope.name, $scope.examples);
            const body = angular.element(document.body);
            const anchor = angular.element(`
                <div class="formula-tooltip" id="formula-function-hinter" ng-style="getStyle()">
                    <strong>{{name}}</strong>(<span ng-bind-html="argsString"></span>)
                    <div ng-if="deprecated" class="formula-tooltip__deprecated">
                        {{::translate('FORMULA.DEPRECATED', '[DEPRECATED]')}}
                        <span ng-if="deprecationNotice" ng-bind-html="deprecationNotice"></span>
                    </div>
                    <p class="formula-tooltip__description" ng-bind-html="description"></p>
                    <span ng-bind-html="htmlExamples"></span>
                </div>
            `);
            body.append(anchor);
            $compile(anchor)($scope);
            
            $scope.getStyle = () => ({
                top: $scope.top + 'px',
                left: $scope.left + 'px'
            });

            const getCurrentArgumentIndex = () => {
                let part = $scope.line.substr($scope.start);
                let pos = 0;
                let parentheses = 0;
                let brackets = 0;
                let quotes = 0;
                let sglQuotes = 0;
                let index = 0;
                while(pos + $scope.start < $scope.cursor) {
                    const char = part.substr(pos, 1);
                    if(char === '\\') {
                        // Escaping character, so we should also skip next one
                        pos += 2;
                        continue;
                    }
                    
                    if(char === ',' && parentheses === 0 && brackets === 0 && sglQuotes % 2 === 0 && quotes % 2 === 0) {
                        // We are not between parentheses or quotes, we can increment the count
                        index ++;
                    } else if(char === '(') {
                        parentheses ++;
                    } else if(char === ')') {
                        parentheses --;
                    } else if(char === '[') {
                        brackets ++;
                    } else if(char === ']') {
                        brackets --;
                    } else if(char === '"' && sglQuotes % 2 === 0) {
                        quotes ++;
                    } else if(char === "'" && quotes % 2 === 0) {
                        sglQuotes --;
                    }
                    pos ++;
                }
                return index;
            };

            $scope.argsString = '';

            $scope.$watch('[arguments,cursor,line,start]', () => {
                const index = getCurrentArgumentIndex();
                $scope.argsString = $scope.arguments.split(', ')
                    .filter((_, id) => id > 0 || !$scope.applied)
                    .map((arg, id) => id === index ? `<strong>${arg}</strong>` : arg)
                    .join(', ');
            });

            $scope.$watch('examples', nv => {
                $scope.htmlExamples = formatExamples($scope.name, nv);
            })
        }       
    };
});

/**
 * Grel formula editor
 *  @param {string}    expression          - Current formula expression
 *  @param {array}     columns             - Array of column objects, formatted as { name: string, type: string, meaning?: string, comment?: string }
 *  @param {object}    recipe              - The recipe object the editor is associated with, if any
 *  @param {object}    scopeVariables      - If set, this will replace the variables coming from API. Must be a key-value pair dictionary.
 *  @param {function}  validator           - Function validating the expression (must return a MonoFuture promise-like object with a success method)
 *  @param {function}  onValidate          - Event fired after complete validation
 *  @param {function}  onExpressionChange  - Event fired when expression changes
 */
app.directive('grelEditor', function($timeout, $stateParams, $filter, $sanitize, DataikuAPI, CachedAPICalls, Debounce, Logger, translate) {
    return {
        template: `<div class="editor-tooltip-anchor h100">
                        <formula-function-hinter
                            ng-if="tooltip.shown"
                            description="tooltip.description"
                            applied="tooltip.applied"
                            deprecated="tooltip.deprecated"
                            deprecation-notice="tooltip.deprecationNotice"
                            left="tooltip.left"
                            top="tooltip.top"
                            arguments="tooltip.arguments"
                            name="tooltip.name"
                            examples="tooltip.examples"
                            line="tooltip.line"
                            cursor="tooltip.cursor"
                            start="tooltip.tokenStart" />
                        <textarea class="h100"></textarea>
                    </div>`,
        restrict :'E',
        replace: true,
        scope : {
            expression: "=",
            columns: "<",
            recipe: "<?",
            scopeVariables: "=?",
            validator: "<",
            onValidate: "<",
            onExpressionChange: "<",
            onError: "<",
            generationsTracker: "<",
            autoFocus: "<?",
            mode: "@",
            onEscape: "&"
        },
        link: function($scope, element) {
            const getRecipeVariables = () => {
                if ($scope.recipe) {
                    return DataikuAPI.flow.recipes.generic.getVariables($scope.recipe);
                }
                return DataikuAPI.flow.recipes.generic.getVariablesWithProjectKey($stateParams.projectKey);
            };

            $scope.tooltip = {
                shown: false,
                applied: false,
                left: 0,
                top: 0,
                arguments: '',
                deprecated: false,
                deprecationNotice: null,
                description: '',
                examples: '',
                name: '',
                line: '',
                cursor: 0,
                tokenStart: 0
            }

            $scope.translate = translate;

            $scope.$watch('scopeVariables', () => {
                if($scope.scopeVariables && Object.keys($scope.scopeVariables).length > 0) {
                    const newVars = {};
                    Object.keys($scope.scopeVariables).forEach(key => {
                        newVars[`variables["${key}"]`] = { value: $scope.scopeVariables[key] };
                        newVars[`\${${key}}`] = { value: $scope.scopeVariables[key] };
                    });
                    $scope.variables = newVars;
                }
            });

            $scope.variables = [];
            getRecipeVariables().success(data => {
                const newVars = {};
                Object.keys(data).forEach(key => {
                    newVars[`variables["${key}"]`] = { value: data[key] };
                    newVars[`\${${key}}`] = { value: data[key] };
                });
                $scope.variables = {
                    ...newVars,
                    ...$scope.variables,
                };
            });

            let helpers = [];
            const formulasReference = $scope.mode === 'udaf' ? CachedAPICalls.udafCustomFormulasReference : CachedAPICalls.customFormulasReference;
            formulasReference.then(data => helpers = data);

            const columnTypeMapping = {
                string: ['String', 'icon-dku-string', 'cm-string'],
                int: ['Integer', 'icon-dku-hexa_view', 'cm-number'],
                double: ['Double', 'icon-dku-hexa_view', 'cm-number'],
                float: ['Float', 'icon-dku-hexa_view', 'cm-number'],
                tinyint: ['Tiny int (8 bits)', 'icon-dku-hexa_view', 'cm-number'],
                smallint: ['Small int (16 bits)', 'icon-dku-hexa_view', 'cm-number'],
                bigint: ['Big int (64 bits)', 'icon-dku-hexa_view', 'cm-number'],
                boolean: ['Boolean', 'icon-dku-true-false', 'cm-bool'],
                date: ['Datetime with tz', 'icon-calendar', 'cm-date'],
                dateonly: ['Date only', 'icon-calendar', 'cm-date'],
                datetimenotz: ['Datetime no tz', 'icon-calendar', 'cm-date'],
                geopoint: ['Geo Point', 'icon-globe', 'cm-date'],
                geometry: ['Geometry/Geography', 'icon-globe', 'cm-date'],
                array: ['Array', 'icon-dku-array', 'cm-date'],
                object: ['Complex object', 'icon-dku-object', 'cm-date'],
                map: ['Map', 'icon-dku-map', 'cm-date'],
                unspecified: ['', 'icon-dku-column', 'cm-table']
            };

            const getColumnNames = () => {
                return $scope.columns.map(c => c.name || c);
            };

            const helperScrollHandler = (e) => {
                const elems = document.getElementsByClassName('helper-display');
                for(let i = 0; i < elems.length; i++) {
                    elems[i].style.top = e.target.scrollTop + 'px';
                }
            };

            const escapeHtml = (html) => {
                return $filter('escapeHtml')(html);
            }
            const sanitizeHtml = (html) => {
                return $sanitize(html);
            }
        
            const hintRenderer = (hint, value, index, type) => {
                return (elt) => {
                    if(!hint || hint === '') {
                        return;
                    }

                    let icon = 'icon-dku-function cm-function';
        
                    const helperElement = document.createElement('div');
                    helperElement.className = 'helper-display';


                    const helper = helpers.find(h => h.name === hint);
                    const helperFooter = '<div class="helper-tip">' + $scope.translate('FORMULA.TAB_TO_COMPLETE', 'Hit tab to complete') + '<div style="float: right;">' + $scope.translate('FORMULA.ESC_TO_HIDE', 'Hit esc to hide') + '</div></div>';
                    if(helper && type==='function') {
                        let htmlTitle = `<div class="helper-title"><strong>${escapeHtml(helper.name)}(${escapeHtml(helper.params)}) ${escapeHtml(helper.returns) || ''}</strong></div>`
                        let htmlDescription = `<div class="helper-description"><p>${sanitizeHtml(helper.description)}</p>`;
                        if (helper.examples) {
                            htmlDescription += helper.examples.map((example) =>
                                `<div class="helper-example">
                                        <code>${escapeHtml(helper.name)}(${escapeHtml(example.args.join(", "))})</code>${$scope.translate('FORMULA.RETURNS', ' returns ')}<code>${escapeHtml(example.result)}</code>
                                    </div>`).join("");
                        }
                        htmlDescription += "</div>";
                        helperElement.innerHTML = htmlTitle + htmlDescription + helperFooter;
                    } else if(Object.keys($scope.variables).includes(hint) && type === 'variable') {
                        const value = escapeHtml(typeof $scope.variables[hint].value === 'object' ?
                                        JSON.stringify($scope.variables[hint].value, null, 2) :
                                        $scope.variables[hint].value);
                        helperElement.innerHTML = `<div class="helper-title">`
                                                        + `<strong>${value}</strong>`
                                                        + `</div>`
                                                        + helperFooter;
                        icon = 'icon-dku-variable cm-variable';
                    } else if(getColumnNames().includes(hint) && type === 'column') {
                        const col = $scope.columns.find(c => c.name === hint || c === hint);
                        const ref = columnTypeMapping[col && col.type ? col.type : 'unspecified'] || ['icon-dku-string', 'cm-default'];
                        if(col) {
                            helperElement.innerHTML = `<div class="helper-title">`
                                                        + (ref[0] ? `${ $scope.translate('FORMULA.TYPE', 'Type: ') }<strong>${ref[0]}</strong>` : '')
                                                        + (col.meaning ? `<br />${ $scope.translate('FORMULA.MEANING', 'Meaning: ') }<strong>${escapeHtml(col.meaning)}</strong>` : '')
                                                        + (col.comment ? `<p>${escapeHtml(col.comment)}</p>` : '<p class="text-weak">' + $scope.translate('FORMULA.NO_DESC', 'No description provided.') + '</p>')
                                                        + `</div>`
                                                        + helperFooter;
                        }
                        icon = `${ref[1]} ${ref[2]}`;
                    }
        
                    elt.innerHTML = `<i class="${icon}"></i>&nbsp;<strong>${hint.substr(0, value.length)}</strong>${hint.substr(value.length)}`;
                    elt.appendChild(helperElement);
        
                    if(index === 0) {
                        elt.parentElement.id = "qa_formula_auto_complete";
                        elt.parentElement.removeEventListener('scroll', helperScrollHandler);
                        elt.parentElement.addEventListener('scroll', helperScrollHandler);
                    }
                }
            }

            const isDeprecated = (hint) => {
                const helper = helpers.find(h => h.name === hint);
                return helper && helper.deprecated;
            }

            const autoCompleteHandler = (hint) => {
                const isColumn = hint.type === 'column' && getColumnNames().includes(hint.text);
                const isFunction = hint.type === 'function' && !Object.keys($scope.variables).includes(hint.text);
                
                // When column has a complex name, the autocompletion should be wrapped inside val('')
                if (isColumn && !hint.text.match(/^[a-z0-9_]+$/i)) {
                    const doc = $scope.cm.getDoc();
                    const cursor = doc.getCursor();
                    const leftPart = cm.getLine(cursor.line).substr(0, cursor.ch - hint.text.length);
                    if(leftPart.endsWith('val(')) {
                        doc.setSelection({line: cursor.line, ch: cursor.ch - hint.text.length});
                        doc.replaceSelection('"');
                        doc.setSelection({line: cursor.line, ch: cursor.ch + 1 });
                        doc.replaceSelection('"');
                    } else if(leftPart.endsWith('val("') || leftPart.endsWith('val(\'')) {
                        doc.setSelection({line: cursor.line, ch: cursor.ch - hint.text.length});
                        doc.setSelection({line: cursor.line, ch: cursor.ch });
                    } else {
                        doc.setSelection({line: cursor.line, ch: cursor.ch - hint.text.length});
                        doc.replaceSelection('val("');
                        doc.setSelection({line: cursor.line, ch: cursor.ch + 5 });
                        doc.replaceSelection('")');
                    }
                }
                else if(isFunction) {
                    const doc = $scope.cm.getDoc();
                    const cursor = doc.getCursor();
                    const line = cm.getLine(cursor.line);
                    if (line.length < cursor.ch || line[cursor.ch] !== '(') {
                        doc.setSelection({ line: cursor.line, ch: cursor.ch });
                        doc.replaceSelection('()');
                    }
                    cursor.ch = cursor.ch + 1;
                    doc.setCursor(cursor);
                }
                else if (!isColumn && !isFunction && hint.text.endsWith('}')) {
                    const doc = $scope.cm.getDoc();
                    const cursor = doc.getCursor();
                    const nextChar = cm.getLine(cursor.line).substr(cursor.ch, 1);
                    if(nextChar === '}') {
                        doc.replaceRange('', { ...cursor }, { line: cursor.line, ch: cursor.ch + 1 });
                    }
                }

                $scope.validateGRELExpression();
            }

            const getLineQuoteState = (line, pos) => {
                let quoteOpened = false;
                let doubleQuoteOpened = false;
                for(let i = 0; i < pos - 1; i++) {
                    if(line[i] == "\\") {
                        i ++;
                    } else if(line[i] === "'" && !doubleQuoteOpened) {
                        quoteOpened = !quoteOpened;
                    } else if(line[i] === '"' && !quoteOpened) {
                        doubleQuoteOpened = !doubleQuoteOpened;
                    }
                }
                return { quoteOpened, doubleQuoteOpened };
            }

            const autoClosingPairs = {'(': ')', '"': '"', "'": "'", '[': ']', '{': '}'};
            const autoCloseCharacter = (cm, key, event) => {
                const doc = cm.getDoc();
                const cursor = doc.getCursor();
                const line = cm.getLine(cursor.line);
                const nextChar = line[cursor.ch];
                const selection = doc.getSelection();
        
                if(nextChar === key) {
                    doc.replaceRange('', { ...cursor }, { line: cursor.line, ch: cursor.ch + 1 });
                    return;
                }

                if(selection.length === 0 && (key === '"' || key === "'") && ![undefined, ']', ')', '}'].includes(nextChar)) {
                    return;
                }

                // Check if we are not currently inside a string definition
                const quoteStatus = getLineQuoteState(line, cursor.ch);
                if(!quoteStatus.doubleQuoteOpened && !quoteStatus.quoteOpened) {
                    if(selection.length > 0) {
                        const endCursor = doc.getCursor(false);
                        doc.replaceSelection(key + selection + autoClosingPairs[key]);
                        doc.setCursor({...endCursor, ch: endCursor.ch + 2});
                    } else {
                        doc.replaceSelection(key + autoClosingPairs[key]);
                        doc.setCursor({...cursor, ch: cursor.ch + 1});
                    }
                    event.preventDefault();
                    event.stopPropagation();
                }
            }
        
            const autoCloseCharacterRemover = (cm) => {
                const doc = cm.getDoc();
                const cursor = doc.getCursor();
                const line = cm.getLine(cursor.line);
                const deletedChar = line[cursor.ch -1];
                const nextChar = line[cursor.ch];
        
                // Check if we are not currently inside a string definition
                const quoteStatus = getLineQuoteState(line, cursor.ch);
                if(quoteStatus.doubleQuoteOpened || quoteStatus.quoteOpened) {
                    return;
                }
        
                if(autoClosingPairs[deletedChar] && autoClosingPairs[deletedChar] === nextChar) {
                    doc.replaceRange('', { ...cursor }, { line: cursor.line, ch: cursor.ch + 1 });
                }
            }

            const getTokenBeforeCursor = (cm) => {
                const cursor = cm.doc.getCursor();
                const tokens = cm.getLineTokens(cursor.line);
                let token = [];
                let parenthese = [];
                let isAppliedToColumn = false;
                for(let i = 0; i < tokens.length; i++) {
                    if(tokens[i].end < cursor.ch + 1) {
                        if(tokens[i].type === 'builtin') {
                            token.push(tokens[i]);
                            if(i > 0 && tokens[i-1].string === '.') {
                                isAppliedToColumn = true;
                            } else {
                                isAppliedToColumn = false;
                            }
                        } else if(tokens[i].string === '(') {
                            parenthese.push(tokens[i].end);
                        } else if(tokens[i].string === ')') {
                            token.pop();
                            parenthese.pop();
                        }
                    }
                }
                if(token.length > 0 && parenthese.length > token.length - 1) {
                    $scope.tooltip.tokenStart = parenthese[token.length - 1];
                    return {...token[token.length - 1], isAppliedToColumn };
                }
                return  null;
            };

            const mode = angular.isDefined($scope.mode) ? $scope.mode : 'grel';
            const editorOptions = {
                value: $scope.expression || "",
                mode: `text/${mode}`,
                theme:'elegant',
                variables: getColumnNames,
                lineNumbers : false,
                lineWrapping : true,
                autofocus: angular.isDefined($scope.autoFocus) ? $scope.autoFocus : true,
                hintOptions: {
                    hint: (editor) => {
                        const grelWordHints = CodeMirror.hint[mode]($scope.cm, { columns: getColumnNames, variables: Object.keys($scope.variables), completeSingle: false });
                        const words = grelWordHints.list;

                        let cursor = editor.getCursor();
                        let curLine = editor.getLine(cursor.line);
                        let start = cursor.ch;
                        let end = start;
                        while (end < curLine.length && /[\w\p{L}$]/u.test(curLine.charAt(end))) ++end;
                        while (start && (/[\w\p{L}.$]/u.test(curLine.charAt(start - 1)) || curLine.charAt(start - 1) == '{')) --start;
                        let curWord = start !== end ? curLine.slice(start, end) : '';
                        // The dot should only be considered a token separator when outside of the ${...} variable syntax
                        const firstDot = curWord.indexOf('.');
                        if(!curWord.startsWith('$') && firstDot > -1) {
                            curWord = curWord.substr(firstDot + 1);
                            start += firstDot + 1;
                        }

                        const list = (!curWord ? words : words.filter(word => {
                            return word.text.toLowerCase().startsWith(curWord.toLowerCase());
                        }))
                        list.sort((a, b) => {
                            const malusA = Object.keys($scope.variables).includes(a.text) ? 2 : getColumnNames().includes(a.text) ? 1 : 0;
                            const malusB = Object.keys($scope.variables).includes(b.text) ? 2 : getColumnNames().includes(b.text) ? 1 : 0;
                            if(malusA === malusB)
                                return a.text - b.text;
                            return malusA - malusB;
                         });

                        const data = {
                            list: list.filter(item => !isDeprecated(item.text)).map((item, index) => ({ text: item.text, type: item.type, render: hintRenderer(item.text, curWord, index, item.type)})),
                            from: CodeMirror.Pos(cursor.line, start),
                            to: CodeMirror.Pos(cursor.line, end)
                        };

                        CodeMirror.on(data, 'pick', autoCompleteHandler);
                            
                        return data;
                    },
                    completeSingle: false
                }
            };

            const prevCursor = { line: 0, ch: 0 };
            const textarea = element.find('textarea')[0];
            const cm = CodeMirror.fromTextArea(textarea, editorOptions);

            $scope.$watch('expression', (newValue, oldValue) => {
                const cursor = cm.doc.getCursor();
                cm.setValue(newValue || '');
                cm.setCursor(cursor);
            });

            $scope.cm = cm;
            cm.on("keydown", function(cm, evt) {
                if(evt.key === "Backspace") { // Backspace
                    autoCloseCharacterRemover(cm);
                }
                else if(Object.keys(autoClosingPairs).includes(evt.key)) {
                    autoCloseCharacter(cm, evt.key, evt);
                }
                if (evt.key === 'Escape' && $scope.onEscape) {
                    $scope.onEscape({state: {completion: cm.state.completionActive && cm.state.completionActive.widget}});
                }
            });

            cm.on("keyup", function(cm, evt) {
                /* Ignore tab, esc, and navigation/arrow keys */
                if (evt.key === "Tab" || evt.key === 'Escape' || (evt.keyCode>= 33 && evt.keyCode <= 40)) {
                    if(evt.key === "Tab") {
                        // We force the expression validation on tab, as autocompletion does not trigger smart change
                        $scope.validateGRELExpression();
                    }
                    CodeMirror.signal(cm, "endCompletion", cm);
                }
                else if(evt.key !== "Enter") {
                    CodeMirror.commands.autocomplete(cm, null, {
                        columns: $scope.columns,
                        completeSingle: false
                    });
                }
            });

            cm.on('cursorActivity', () => {
                const parent = cm.getTextArea().closest('.editor-tooltip-anchor');
                if(!parent) {
                    return;
                }

                const coords = cm.cursorCoords();
                $scope.tooltip.left = coords.left;
                $scope.tooltip.top = coords.top + 16;
                $scope.tooltip.shown = false;

                const doc = cm.getDoc();
                const cursor = doc.getCursor();
                const token = getTokenBeforeCursor(cm);
                const leftPart = cm.getLine(cursor.line).substr(0, cursor.ch);
                if(token && !leftPart.endsWith('val("') && !leftPart.endsWith('val(\'')) {
                    const helper = helpers.find(h => h.name === token.string);
                    if(helper) {
                        $scope.tooltip.shown = true;
                        $scope.tooltip.name = helper.name;
                        $scope.tooltip.arguments = helper.params;
                        $scope.tooltip.deprecated = helper.deprecated;
                        $scope.tooltip.deprecationNotice = helper.deprecationNotice;
                        $scope.tooltip.description = helper.description;
                        $scope.tooltip.examples = helper.examples;
                        $scope.tooltip.line = cm.getLine(cursor.line);
                        $scope.tooltip.cursor = cursor.ch;
                        $scope.tooltip.applied = token.isAppliedToColumn;
                    }
                }

                // We do not want to auto display the autocomplete hinter if the user is just moving to the next line
                // Or if the completion has just ended
                if (cursor.line === prevCursor.line && cm.state.completionActive !== null) {
                    CodeMirror.commands.autocomplete(cm, null, {
                        columns: $scope.columns,
                        completeSingle: false
                    });
                }
                safeApply($scope);

                prevCursor.ch = cursor.ch;
                prevCursor.line = cursor.line;
            });
        
            let errorMarker = null;
            const highlightOffset = (offset) => {
                let current = 0;
                const lineCount = $scope.cm.lineCount();
                for(let i = 0; i < lineCount; i++) {
                    const line = $scope.cm.doc.getLine(i);
                    if(current + line.length > offset) {
                        errorMarker = $scope.cm.doc.markText(
                            { line: i, ch: offset - current },
                            { line: i, ch: offset - current + 1 },
                            { className: 'error-hint', clearOnEnter: true }
                        );
                        return;
                    }
                    current += line.length + 1; // CrLf is counted as 1 offset
                }
            }

            $scope.fireExpressionChange = () => {
                if($scope.onExpressionChange) {
                    $scope.onExpressionChange($scope.expression);
                }
                $scope.validateGRELExpression($scope.expression);
            };
            const debouncedFireExpressionChange = Debounce().withDelay(0, 500).wrap($scope.fireExpressionChange);
        
            $scope.validateGRELExpression = () => {
                const validateStart = performance.now();
                const expr = cm.getValue();
                const contentGenerationBeingValidated = $scope.generationsTracker ? $scope.generationsTracker.currentContentGeneration : 0;
                Logger.info("Starting to validate expression at generation ", contentGenerationBeingValidated);

                $scope.examples = null;

                if($scope.validator) {

                    // prevent from calling the backend if the formula is not complete
                    if (expr.trim().length == 0) {
                        $scope.onValidate({ valid: false, error: false, data: {}, inputExpr: expr });
                        return;
                    }

                    $scope.validator(expr).success((futureResult) => {
                        const data = futureResult.result;

                        const afterBackend = performance.now();
                        Logger.info("Timing: grelEditor.validateBackend: " + (afterBackend-validateStart) + "ms");

                        if($scope.onValidate) {
                            $scope.onValidate({ valid: data.ok, error: data.message, data, inputExpr: expr });
                        }
    
                        if(errorMarker != null) {
                            errorMarker.clear();
                        }
            
                        const matches = (data.message || '').match(/at offset (\d+)/i);
                        if(matches != null) {
                            highlightOffset(matches[1])
                        }

                        const afterApply = performance.now();
                        Logger.info("Timing: grelEditor.totalPerceivedValidationLatency: " + (afterApply-lastCodeMirrorChange) + "ms");

                        if ($scope.generationsTracker && data.ok) {
                            $scope.generationsTracker.validatedContentGeneration = contentGenerationBeingValidated;
                        }

                    }).error($scope.onError ? $scope.onError : setErrorInScope.bind($scope));
                }
            };

            $scope.$on('codemirror-focus-input', () => {
                cm.focus();
            });

            let lastCodeMirrorChange = 0;

            cm.on('change', () => {
                if ($scope.generationsTracker) {
                    $scope.generationsTracker.currentContentGeneration++;
                }
                lastCodeMirrorChange = performance.now();
                $scope.expression = cm.getValue();
                debouncedFireExpressionChange();
            });
        }
    };
});



app.controller('FormulaAwareProcessorController', function($scope, $stateParams, $rootScope, DataikuAPI,
               CreateCustomElementFromTemplate, ShakerPopupRegistry, MonoFuture, DeducedMonoFuture, PrettyPrintDoubleService) {

    $scope.editing = {}
    $scope.columns = [];

    const canGetColumnDetails = () => {
        const stepPosition = $scope.shaker.steps.indexOf($scope.step);

        // If the current step is the last one or if it is in preview mode or if the steps after it are disabled.
        return $scope.shaker && $scope.shaker.steps && $scope.shaker.steps[$scope.shaker.steps.length - 1] === $scope.step 
            || $scope.step.preview === true
            || (!$scope.step.disabled && $scope.shaker.steps.slice(stepPosition + 1).every(step => step.disabled))
    } 

    const computeColumns = () => {
        if (canGetColumnDetails()) {
            $scope.columns = $scope.quickColumns.map(c => ({
                ...(c.recipeSchemaColumn ? c.recipeSchemaColumn.column : {}),
                name: c.name,
                meaning: c.meaningLabel || '',
                comment: c.comment || ''
            }));
        } else {
            $scope.columns = $scope.step.$stepState.change ? $scope.step.$stepState.change.columnsBeforeStep : [];
        }
    }

    const stopWatchingForColumnsCompute = $scope.$watch('[quickColumns, step.preview, shaker.steps]', () => {
        computeColumns();
    });

    let validateExpressionMonoFuture = DeducedMonoFuture($scope, false, 250).wrap(DataikuAPI.shakers.validateExpression);
    $scope.expressionValidator = (expression) => {
    	const stepPosition = $scope.findStepId($scope.step);
        return validateExpressionMonoFuture(
            $stateParams.projectKey,
            $scope.inputDatasetProjectKey,
            $scope.inputDatasetName,
            $scope.shakerHooks.shakerForQuery(),
            $scope.requestedSampleId,
            expression,
            stepPosition.id,
            stepPosition.subId || 0,
            stepPosition.depth,
            true,
            false
        );
    };

    $scope.generationsTracker = {
        currentContentGeneration: 0,
        validatedContentGeneration: -1
    }

    $scope.fixupFormula = (expression, fixName = "plus") => {
        const stepPosition = $scope.findStepId($scope.step);
        return DataikuAPI.shakers.fixExpression(
            $stateParams.projectKey,
            $scope.inputDatasetProjectKey,
            $scope.inputDatasetName,
            $scope.shakerHooks.shakerForQuery(),
            $scope.requestedSampleId,
            expression,
            fixName,
            stepPosition.id,
            stepPosition.subId || 0,
            stepPosition.depth).then(data => {
                $scope.editing.expression = data.data
            }, setErrorInScope.bind($scope));
    }

    $scope.onValidate = (result) => {
        $scope.changeInProgress = false;
        $scope.grelExpressionValid = result.valid;
        $scope.grelExpressionError = result.error;        
        $scope.grelExpressionEmpty = result.inputExpr.trim().length == 0;

        if (result.valid) {
            $scope.grelExpressionData = result.data;
        }

        if($scope.grelExpressionData && $scope.grelExpressionData.table) {
            PrettyPrintDoubleService.patchFormulaPreview(
                $scope.grelExpressionData.table,
                'double', // true output type is unknown but we want the patch to happen for scientific notations values
                $scope.columns.map(c => ({...c, type: c.meaning === 'Decimal' ? 'double' : ''}))
            )
        }
    };

    const checkUnsavedFormulaChanges = () => {
        if ($scope.modalShown && $scope.step.$stepState) {
            $scope.step.$stepState.containsUnsavedFormulaChanges = !angular.equals($scope.step.params.expression, $scope.editing.expression) || !angular.equals($scope.step.params.column, $scope.editing.outputColumnName);
        }
    };

    $scope.onFormulaColumnChange = () => {
        checkUnsavedFormulaChanges();
    };

    $scope.onExpressionChange = (expression) => {
        $scope.changeInProgress = true;
        checkUnsavedFormulaChanges();
    };

    $scope.parentDismiss = function() {
        $rootScope.$broadcast("dismissModalInternal_");
        $scope.modalShown = false;
        if ($scope.step.$stepState) {
            delete $scope.step.$stepState.containsUnsavedFormulaChanges;
        }
    }
    $scope.$on("dismissModals", $scope.parentDismiss);
    $scope.grelExpressionError = false;
    $scope.grelExpressionValid = false;    
    $scope.grelExpressionEmpty = true;
    $scope.changeInProgress = false;

    $scope._edit = function(expression, outputColumnName = null) {
        if ($scope.modalShown) {
            $scope.parentDismiss();
        } else {
            ShakerPopupRegistry.dismissAllAndRegister($scope.parentDismiss);
            $scope.modalShown = true;
            CreateCustomElementFromTemplate("/templates/shaker/formula-editor.html", $scope, null, function(newScope) {
                $scope.editing.expression = expression;
                $scope.editing.outputColumnName = outputColumnName;
                $scope.editing.showColumnNameEditor = outputColumnName !== null; // not all processors have an output column name concept
            }, $scope.customFormulaEdition.displayCustomFormula);
        }
    }


    $scope.hooks= {
        ok : function(){
            throw Error("not implemented");
        }
    }

    $scope.$on("$destroy", stopWatchingForColumnsCompute);
});


app.controller('CreateColumnWithGRELController', function($scope, $controller) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    if (angular.isUndefined($scope.step.params.column)) {
        $scope.step.params.column = "formula_result";
    }
    if (angular.isUndefined($scope.step.params.expression)) {
        $scope.step.params.expression = "";
    }
    if (angular.isUndefined($scope.step.params.errorColumn)) {
        $scope.step.params.errorColumn = "";
    }
    $scope.mode = "create";

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.step.params.column = $scope.editing.outputColumnName || '';
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function() {
        $scope._edit(angular.copy($scope.step.params.expression), $scope.step.params.column);
    }

    $scope.storageTypes = [
    	[null, 'None'],
        [{name:'foo', type:'string'}, 'String'],
        [{name:'foo', type:'int'}, 'Integer'],
        [{name:'foo', type:'double'}, 'Double'],
        [{name:'foo', type:'float'}, 'Float'],
        [{name:'foo', type:'tinyint'}, 'Tiny int (8 bits)'],
        [{name:'foo', type:'smallint'}, 'Small int (16 bits)'],
        [{name:'foo', type:'bigint'}, 'Big int (64 bits)'],
        [{name:'foo', type:'boolean'}, 'Boolean'],
        [{name:'foo', type:'date'}, 'Datetime with tz'],
        [{name:'foo', type:'dateonly'}, 'Date only'],
        [{name:'foo', type:'datetimenotz'}, 'Datetime no tz'],
        [{name:'foo', type:'geopoint'}, "Geo Point"],
        [{name:'foo', type:'geometry'}, "Geometry/Geography"]
    ];

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});

app.controller("CustomJythonProcessorController", function($scope, Assert, DataikuAPI, WT1, $stateParams, TopNav, PluginConfigUtils, Logger){
    $scope.loadedDesc = $scope.appConfig.customJythonProcessors.find(x => x.elementType == $scope.step.type);
    Assert.inScope($scope, 'loadedDesc');
    $scope.pluginDesc = $scope.appConfig.loadedPlugins.find(x => x.id == $scope.loadedDesc.ownerPluginId);
    Assert.inScope($scope, 'pluginDesc');

    // Make a copy to play with roles
    $scope.desc = angular.copy($scope.loadedDesc.desc);
    $scope.desc.params.forEach(function(param) {
        if (param.type == "COLUMNS" || param.type == "COLUMN") {
            param.columnRole = "main";
        }
    });

    if (!$scope.step.params) {
        $scope.step.params = {}
    }
    if (!$scope.step.params.customConfig){
        $scope.step.params.customConfig = {}
    }

    var getCurrentColumns = function() {
        // step info are loaded asynchronously
        if ($scope.step.$stepState.change) {
            return $scope.step.$stepState.change.columnsBeforeStep.map(colName => ({name: colName, type: 'STRING'}));
        } else {
            return [];
        }
    }

    $scope.columnsPerInputRole = {
        "main" : []
    };

    $scope.$watch("step", function(nv, ov) {
        if (nv && nv.$stepState) {
            $scope.columnsPerInputRole = {
                main: getCurrentColumns()
            };
        }
    }, true);

    PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.step.params.customConfig);
});


app.controller('FilterOnCustomFormulaController', function($scope, $controller, $stateParams, $rootScope, DataikuAPI, CreateCustomElementFromTemplate, ShakerPopupRegistry) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function(){
        $scope._edit(angular.copy($scope.step.params.expression));
    }

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});


app.controller('ClearOnCustomFormulaController', function($scope, $controller, $stateParams, $rootScope,
               DataikuAPI, CreateCustomElementFromTemplate, ShakerPopupRegistry) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function(){
        $scope._edit(angular.copy($scope.step.params.expression));
    }

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});


app.controller('FlagOnCustomFormulaController', function($scope, $controller, $stateParams, $rootScope,
               DataikuAPI, CreateCustomElementFromTemplate, ShakerPopupRegistry) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function(){
        $scope._edit(angular.copy($scope.step.params.expression));
    }

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});


app.controller('FilterAndFlagProcessorController', function($scope) {
    // don't show the generic input for clearColumn if the processor already declares a parameter called clearColumn
    $scope.showClearColumn = !$scope.processor.params.some(function(param) { return param.name == 'clearColumn'; });

    $scope.onActionChange = function() {
        // if (['FLAG', 'KEEP_ROW'].indexOf($scope.step.params['action']) != -1) {
        //     $scope.step.params.appliesTo = 'SINGLE_COLUMN';
        //     $scope.$parent.$parent.$parent.appliesToDisabled = true;
        // } else {
        //     $scope.$parent.$parent.$parent.appliesToDisabled = false;
        // }

        $scope.checkAndRefresh();
    }

    if ($scope.processor.filterAndFlagMode == "FLAG") {
        $scope.step.params.action = "FLAG";
    }
});


app.controller('AppliesToProcessorController', function($scope) {
    // Indicate the position of the mouse relative to the add button
    // The flag is immediately set to true when mouse is over the button
    $scope.isMouseOverAddButton = false;

    $scope.clearEmptyColumns = function() {
        if ($scope.step.params.columns.length == 1 && $scope.step.params.columns[0] == '') {
            $scope.step.params.columns = [];
        }
    }

    /**
     * Handle field blur event of the value list
     * @param {function} handler - function that will be triggered when the event occurs
     * @param {boolean} doesTakeIntoAccountOfMouseOverAddButton - set to true to take into account the mouse over event on the value list add button
     */
    $scope.handleValueListFieldBlur = function (event, handler, doesTakeIntoAccountOfMouseOverAddButton) {
        if (
            event.relatedTarget &&
            event.relatedTarget.offsetParent &&
            event.relatedTarget.offsetParent.classList &&
            event.relatedTarget.offsetParent.classList.contains('mat-autocomplete-panel')
        ) return; // story 99882 - prevents shaker refresh table on column selection
        if (!doesTakeIntoAccountOfMouseOverAddButton) {
            handler();
            return;
        }
        if (!$scope.isMouseOverAddButton) {
            // Only handle the blur event when the mouse was not over the add button
            handler();
        }
    }

    /**
     * Handle mouse over event of the value list's add button
     * Immediately switch the scope's isMouseOverAddButton to 'true'
     */
    $scope.handleValueListMouseOverAddButton = function () {
        $scope.isMouseOverAddButton = true;
    }

    /**
     * Handle mouse leave event of the value list's add button
     */
    $scope.handleValueListMouseLeaveAddButton = function () {
        $scope.isMouseOverAddButton = false;
    }
});


app.controller('ConfigureSamplingController', function($scope, DataikuAPI, $stateParams, $timeout,
               WT1, CreateModalFromTemplate, SamplingData, DatasetUtils) {
    $scope.getPartitionsList = function() {
        return DataikuAPI.datasets.listPartitionsWithName($scope.inputDatasetProjectKey, $scope.inputDatasetName)
            .error(setErrorInScope.bind($scope))
            .then(function(ret) { return ret.data; })
    };

    $scope.SamplingData = SamplingData;

    $scope.showFilterModal = function() {
        var newScope = $scope.$new();
        newScope.updateFilter = (filter) => $scope.shaker.explorationSampling.selection.filter = filter;
        if ($scope.inputDatasetName) {
            DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey)
            .success(function(data){
                newScope.dataset = data;
                newScope.schema = data.schema;
                newScope.filter = angular.copy($scope.shaker.explorationSampling.selection.filter);
                CreateModalFromTemplate('/static/dataiku/nested-filters/input-filter-block/filter-modal.component.html', newScope, undefined, false, false, 'static');
            }).error(setErrorInScope.bind($scope));
        } else if ($scope.inputStreamingEndpointId) {
            DataikuAPI.streamingEndpoints.get($scope.inputDatasetProjectKey, $scope.inputStreamingEndpointId)
            .success(function(data){
                newScope.dataset = data;
                newScope.schema = data.schema;
                newScope.filter = angular.copy($scope.shaker.explorationSampling.selection.filter);
                CreateModalFromTemplate('/static/dataiku/nested-filters/input-filter-block/filter-modal.component.html', newScope, undefined, false, false, 'static');
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.datasetIsSQL = function() {
        return $scope.dataset_types && $scope.dataset && $scope.dataset_types[$scope.dataset.type] && $scope.dataset_types[$scope.dataset.type].sql;
    };

    $scope.datasetIsSQLTable = function() {
        return $scope.datasetFullInfo && DatasetUtils.isSQLTable($scope.datasetFullInfo.dataset);
    };

    $scope.datasetSupportsReadOrdering = function() {
        return $scope.datasetFullInfo && DatasetUtils.supportsReadOrdering($scope.datasetFullInfo.dataset);
    };

    $scope.save = function() {
        let evt = {
            analysis: $scope.shaker.origin === 'ANALYSIS',
            samplingMethod: $scope.shaker.explorationSampling.selection.samplingMethod,
            recordsNumber: $scope.shaker.explorationSampling.selection.maxRecords,
            targetRatio: $scope.shaker.explorationSampling.selection.targetRatio,
            filtersNumber: $scope.shaker.explorationSampling.selection.filter &&
                           $scope.shaker.explorationSampling.selection.filter.enabled &&
                           $scope.shaker.explorationSampling.selection.filter.uiData &&
                           $scope.shaker.explorationSampling.selection.filter.uiData.conditions
                ? $scope.shaker.explorationSampling.selection.filter.uiData.conditions.length
                : 0,
            sortingKeyNumber: $scope.shaker.explorationSampling.selection.ordering &&
                              $scope.shaker.explorationSampling.selection.ordering.enabled &&
                              $scope.shaker.explorationSampling.selection.ordering
                ? $scope.shaker.explorationSampling.selection.ordering.rules.length
                : 0,
            maxMemory: $scope.shaker.explorationSampling.selection.maxStoredBytes !== -1 ? $scope.shaker.explorationSampling.selection.maxStoredBytes : null,
            autoRefresh: $scope.shaker.explorationSampling.autoRefreshSample
        };
        if ($scope.inputDatasetName) {
            evt["datasetId"] = `${$scope.inputDatasetProjectKey.dkuHashCode()}.${$scope.inputDatasetName.dkuHashCode()}`;
        }
        if ($scope.inputStreamingEndpointId) {
            evt["streamingEndpointId"] = `${$scope.inputDatasetProjectKey.dkuHashCode()}.${$scope.inputStreamingEndpointId.dkuHashCode()}`;
        }
        WT1.event('dataset-sample-settings-update', evt);
        $scope.shaker.explorationSampling._refreshTrigger = new Date().getTime();
        $scope.forgetSample();
        $scope.autoSaveForceRefresh();
    };
});

app.controller('DateRangeShakerController', function($scope, $timeout, DataikuAPI, Debounce, DateUtilsService, translate, ChartFilterUtils) {
    $scope.step.params.part = $scope.step.params.part != null ? $scope.step.params.part : "YEAR";
    $scope.dateRelativeFilterComputedStart = '-';
    $scope.dateRelativeFilterComputedEnd = '';
    $scope.displayed = { values: [] };
    $scope.valuesCount = 0;
    // Firefox always displays the milliseconds in the time picker, so give more space to the time picker
    $scope.timeInputStyle = { 'width': (navigator.userAgent.includes("Gecko/") ? '144px' : '104px') };

    if($scope.step.params.values) {
        $scope.displayed.values = [...$scope.step.params.values];
    }

    if ($scope.step.params.min) {
        // If date was written as UTC (old format), convert it to the corresponding time zone
        if ($scope.step.params.min.slice(-1).toUpperCase() === "Z") {
            $scope.step.params.min = DateUtilsService.formatDateToISOLocalDateTime(DateUtilsService.convertDateToTimezone(new Date($scope.step.params.min), $scope.step.params.timezone_id));
        }
        $scope.displayed.min = new Date($scope.step.params.min);
        $scope.displayed.min.setSeconds($scope.displayed.min.getSeconds(), 0);
    }

    if ($scope.step.params.max) {
        // If date was written as UTC (old format), convert it to the corresponding time zone
        if ($scope.step.params.max.slice(-1).toUpperCase() === "Z") {
            $scope.step.params.max = DateUtilsService.formatDateToISOLocalDateTime(DateUtilsService.convertDateToTimezone(new Date($scope.step.params.max), $scope.step.params.timezone_id));
        }
        $scope.displayed.max = new Date($scope.step.params.max);
        $scope.displayed.max.setSeconds($scope.displayed.max.getSeconds(), 0);
    }

    // This is a fix for firefox, not firing blur event if input cleared with cross icon
    $scope.updateBoundariesWithDelay = function(boundary) {
        setTimeout($scope.updateBoundaries, 200, boundary);
    }

    $scope.updateBoundaries = function(boundary) {
        if (boundary === 'min') {
            if ($scope.displayed.min) {
                $scope.step.params.min = DateUtilsService.formatDateToISOLocalDateTime($scope.displayed.min);
            } else {
                $scope.step.params.min = null;
            }
        } else if (boundary === 'max') {
            if ($scope.displayed.max) {
                $scope.step.params.max = DateUtilsService.formatDateToISOLocalDateTime($scope.displayed.max);
            } else {
                $scope.step.params.max = null;
            }
        }
        $scope.checkAndRefresh();
    };

    const computeRelativeDateIntervalDebounced = Debounce().withDelay(100,100).withScope($scope).wrap(function() {
        DataikuAPI.shakers.computeRelativeDateInterval({
            part: $scope.step.params.part,
            option: $scope.step.params.option,
        }).success(function(interval) {
            $scope.dateRelativeFilterComputedStart = interval.start;
            $scope.dateRelativeFilterComputedEnd = interval.end;
        }).error(function() {
            $scope.dateRelativeFilterComputedStart = '-';
            $scope.dateRelativeFilterComputedEnd = '-';
        });
    });
    const refreshRelativeIntervalHint = function() {
        $scope.isRelativeFilterEffective = ChartFilterUtils.isRelativeDateFilterEffective($scope.step.params.part, $scope.step.params.option);
        if ($scope.step.params.filterType === 'RELATIVE' && $scope.isRelativeFilterEffective) {
            computeRelativeDateIntervalDebounced();
        }
    }

    $scope.$watchGroup(["step.params.filterType", "step.params.part"], refreshRelativeIntervalHint);

    $scope.handleRelativeDateOptionChange = (relativeOption) => {
        $scope.step.params.option = relativeOption;
        refreshRelativeIntervalHint();
        $scope.updateBoundaries()
    }

    $scope.$watch("displayed.values", function() {
        $scope.step.params.values = [...$scope.displayed.values];
    });

    $scope.$watch("step.params.timezone_id", $scope.updateBoundaries);

    $scope.$watch("step.params.filterType", function(nv) {
        if (nv === "RELATIVE") {
            if (["INDIVIDUAL", "DAY_OF_WEEK"].includes($scope.step.params.part)) {
                $scope.step.params.part = "YEAR";
            }
        }
    });
});

app.controller('UnfoldController', function($scope, translate){

    $scope.overflowActions = [
        ["KEEP", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.KEEP", "Keep all columns")],
        ["WARNING", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.WARNING", "Add warnings")],
        ["ERROR", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.ERROR", "Raise an error")],
        ["CLIP", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.CLIP", "Clip further data")],
    ];

    $scope.overflowActionsDesc = [
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.KEEP.DESC", "Will keep all the created columns. Warning, this may create a huge amount of columns and slow the whole computation.")],
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.WARNING.DESC", "Will raise warning during the computation but continues to process all the columns. It may create a huge amount of columns and slow the whole computation.")],
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.ERROR.DESC", "Will raise an error and make the computation fail as soon as the maximum number of created columns is exceeded.")],
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.CLIP.DESC", "Will silently drop the remaining columns when the maximum number of columns is reached.")]
    ];

    $scope.modifyOverflowAction = function() {
        if ($scope.step.params.limit === 0) {
            $scope.step.params.overflowAction = "KEEP";
        }
    };

});

app.controller('ColumnRenamerController', function(
    $scope, $timeout, $rootScope, $q, $stateParams, CreateModalFromTemplate) {

    $scope.massRenameColumns = function() {
        CreateModalFromTemplate('/templates/shaker/modals/shaker-rename-columns.html', $scope, 'MassRenameColumnsController', function(newScope) {
            let columnNames = [...computeColumns()];
            $scope.step.params.renamings.forEach(function(renaming) {
                let index = columnNames.indexOf(renaming.from);
                if (index >= 0) {
                    columnNames[index] = renaming.to;
                }
            });

            newScope.setColumns(columnNames);
            newScope.doRenameColumns = function(renamings) {
                $scope.step.params.renamings = $scope.step.params.renamings.concat(renamings);
            };
        });
    };

    const canGetColumnDetails = () => {
        const stepPosition = $scope.shaker.steps.indexOf($scope.step);

        // If the current step is the last one or if it is in preview mode or if the steps after it are disabled.
        return $scope.shaker && $scope.shaker.steps && $scope.shaker.steps[$scope.shaker.steps.length - 1] === $scope.step
            || $scope.step.preview === true
            || (!$scope.step.disabled && $scope.shaker.steps.slice(stepPosition + 1).every(step => step.disabled))
    }

    const computeColumns = () => {
        if (canGetColumnDetails()) {
            return $scope.quickColumns
                .filter(c => c.recipeSchemaColumn ? !c.recipeSchemaColumn.deleted : true)
                .map(c => ({
                    ...(c.recipeSchemaColumn ? c.recipeSchemaColumn.column : {}),
                    name: c.name
                }))
                .map(c => c.name);
        } else {
            return $scope.step.$stepState.change ? $scope.step.$stepState.change.columnsBeforeStep : [];
        }
    }
});

}());
