(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('ChartTypeChangeHandler', chartTypeChangeHandler);

    /**
     * ChartTypeChangeHandler service
     * Defines what to do when switching from a chart type to another.
     * (!) This file previously was in static/dataiku/js/simple_report/chart_change.js
     */
    function chartTypeChangeHandler(translate, Assert, LoggerProvider, ChartsStaticData, ChartAxesUtils, ChartTypeChangeUtils, ChartColumnTypeUtils, ChartColorUtils, VALUES_DISPLAY_MODES, DKU_PALETTE_NAMES,
        ChartDimension, ChartUADimension, ChartFeatures, WebAppsService, PluginConfigUtils, ChartLabels, $rootScope, CHART_TYPES, CHART_VARIANTS, ChartColorSelection, ValuesInChartPlacementMode,
        ChartFilters, RegressionTypes, ValuesInChartOverlappingStrategy, PolygonSources, LineStyleTypes, AxisTicksConfigMode, AxisTicksFormatting, AxisTitleFormatting, GridlinesAxisType, ValueType, DSSVisualizationThemeUtils) {

        const Logger = LoggerProvider.getLogger('charts');
        const COLUMN_TYPE_ORDER = {
            color: ['colorMeasure', 'uaColor'],
            size: ['sizeMeasure', 'uaSize'],
            tooltip: ['tooltipMeasures', 'uaTooltip']
        };

        const DEFAULT_FONT_FORMATTING = {
            fontSize: 11,
            fontColor: '#333',
            hasBackground: false
        };

        const DEFAULT_TABLE_HEADER_FONT_FORMATTING = {
            ...DEFAULT_FONT_FORMATTING,
            fontSize: 12
        };

        const DEFAULT_AUTO_FONT_FORMATTING = {
            fontSize: 11,
            fontColor: 'AUTO'
        };

        const DEFAULT_VALUES_DISPLAY_IN_CHART_TEXT_FORMATTING = {
            ...DEFAULT_AUTO_FONT_FORMATTING,
            hasBackground: false,
            backgroundColor: '#D9D9D9BF'
        };

        const DEFAULT_GAUGE_MIN_MAX = {
            sourceType: ValueType.Constant,
            aggregatedColumn: null,
            datasetColumn: null,
            customAggregation: null,
            aggregation: null
        };

        const accept = function(message) {
            return {
                accept: true,
                message: message
            };
        };

        const reject = function(message) {
            return {
                accept: false,
                message: message
            };
        };

        function setSingleIfHasEnough(srcArr, tgtArr, srcIdx) {
            tgtArr.length = 0;
            if (srcArr.length > srcIdx) {
                tgtArr.push(srcArr[srcIdx]);
            }
        };

        function retrieveGeoFromGeomMap(chartDef) {
            if (chartDef.geometry.length === 0 && chartDef.geoLayers && chartDef.geoLayers.length > 1) {
                chartDef.geometry = angular.copy(chartDef.geoLayers[0].geometry);
            }
        };

        function migrateToGeomMap(chartDef) {
            const uaColor = angular.copy(chartDef.uaColor);
            const colorOptions = angular.copy(chartDef.colorOptions);
            if (chartDef.geometry.length > 0) {
                chartDef.geoLayers.unshift({
                    geometry: angular.copy(chartDef.geometry),
                    uaColor: uaColor,
                    colorOptions: colorOptions
                });
                chartDef.geometry = [];
            } else {
                chartDef.geoLayers[0].colorOptions = colorOptions;
                chartDef.geoLayers[0].uaColor = uaColor;
            }
        }

        function retrieveOptionsFromGeomMap(chartDef) {
            if (chartDef.geoLayers && chartDef.geoLayers.length > 0) {
                chartDef.colorOptions = angular.copy(chartDef.geoLayers[0].colorOptions);
                if (chartDef.geoLayers[0].uaColor) {
                    chartDef.uaColor = angular.copy(chartDef.geoLayers[0].uaColor);
                }
            }
        }

        function computeGenericMeasureInitialValuesInChartDisplayTextFormattingOptions(chartDef) {
            // retrieve all common text formatting properties among displayed measures
            const commonProperties = {};

            const properties = ['fontSize', 'fontColor', 'hasBackground', 'backgroundColor'];
            if (chartDef.genericMeasures) {
                // filter on already initialized measures
                chartDef.genericMeasures.filter(m => m.isA === 'measure').forEach(m => {
                    const initializedProperties = Object.keys(commonProperties);
                    properties.forEach(prop => {
                        if (commonProperties[prop] !== null
                            && m.valuesInChartDisplayOptions
                            && m.valuesInChartDisplayOptions.displayValues
                            && m.valuesInChartDisplayOptions.textFormatting
                            && !_.isNil(m.valuesInChartDisplayOptions.textFormatting[prop])) {
                            if (!initializedProperties.includes(prop)) {
                                commonProperties[prop] = m.valuesInChartDisplayOptions.textFormatting[prop];
                            } else if (commonProperties[prop] !== m.valuesInChartDisplayOptions.textFormatting[prop]) {
                                // set it to null if at least 2 different values
                                commonProperties[prop] = null;
                            }
                        }
                    });
                });
            }

            return {
                ...DEFAULT_VALUES_DISPLAY_IN_CHART_TEXT_FORMATTING,
                ..._.omitBy(commonProperties, _.isNil)
            };
        }

        function computeGenericMeasureInitialValuesInChartDisplayOptions(chartDef) {
            // retrieve all common text formatting properties among displayed measures
            const properties = ['placementMode', 'spacing'];
            const commonProperties = {};

            if (!chartDef.genericMeasures) {
                return commonProperties;
            }

            for (const measure of chartDef.genericMeasures) {
                if (measure.isA !== 'measure' || !measure.valuesInChartDisplayOptions) {
                    continue;
                }

                for (const prop of properties) {
                    const value = measure.valuesInChartDisplayOptions[prop];

                    if (_.isNil(value)) {
                        continue;
                    }
                    if (!(prop in commonProperties)) {
                        commonProperties[prop] = value;
                    } else if (commonProperties[prop] !== value) {
                        // set it to null if at least 2 different values
                        commonProperties[prop] = null;
                    }
                }
            }

            return _.omitBy(commonProperties, _.isNil);
        }

        /**
         * Return one or more attribute(s) based on the chartDefAttributeName and removes them from the "chartDefAttributes" array
         * We use the following method because all non-returned measures are moved in tooltip
         * @param {object[]} chartDefAttributes
         * @param {string[]} chartDefAttributeNames
         * @param {boolean} multiple
         * @returns {*[]}
         */
        function getAttributeAndRemove(chartDefAttributes, chartDefAttributeNames = [], multiple = false) {
            if (Array.isArray(chartDefAttributes) && Array.isArray(chartDefAttributeNames)) {
                const response = [];
                const indexes = [];

                for (const chartDefAttributeName of chartDefAttributeNames) {
                    for (const [index, column] of chartDefAttributes.entries()) {
                        if (column && column.chartDefAttributeName === chartDefAttributeName) {
                            response.push(column);
                            indexes.push(index);
                            if (!multiple) {
                                break;
                            }
                        }
                    }
                }

                // Mandatory to delete several array element without indexes problems
                indexes.sort(descendingNumericSort).forEach((index) => chartDefAttributes.splice(index, 1));

                return response;
            }
            return [];
        }

        function autocompleteUA(ua) {
            if (ua.isA !== 'ua') {
                ua.sortBy = 'NATURAL';
                ua.isA = 'ua';
                ua.multiplier = ChartsStaticData.DEFAULT_MULTIPLIER;
                if (ua.type === 'DATE') {
                    ua.dateMode = 'RANGE';
                }
                if (ua.type === 'NUMERICAL') {
                    ua.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                    ua.useParenthesesForNegativeValues = false;
                    ua.shouldFormatInPercentage = false;
                    ua.hideTrailingZeros = getDefaultTrailingZeros(ua);
                }
            }
        }

        function newColorOptions(index) {
            const colorOptions = getDefaultColorOption(index);
            initializeColorPalettes(colorOptions);
            return colorOptions;
        }

        function getDefaultColorOption(index) {
            const circularColorPalette = window.dkuColorPalettes.discrete[0].sample;
            return { singleColor: circularColorPalette[index % circularColorPalette.length], transparency: 0.75 };
        }

        function initializeColorPalettes(colorOptions) {
            colorOptions.customPalette = ChartColorUtils.getDefaultCustomPalette();
            colorOptions.ccScaleMode = 'NORMAL';
            colorOptions.paletteType = 'CONTINUOUS';
            colorOptions.quantizationMode = 'NONE';
            colorOptions.numQuantizeSteps = 5;
            colorOptions.paletteMiddleValue = 0;
            if (colorOptions.paletteMiddleValue <= 0 && colorOptions.paletteType === 'DIVERGING' && colorOptions.ccScaleMode === 'LOG') {
                colorOptions.paletteMiddleValue = 1;
            }
            colorOptions.customColors = {};
        }

        // Initializing the necessary properties
        function fixupAxes(chartDef, theme) {

            // x axis
            if (_.isNil(chartDef.xAxisFormatting) || !Object.keys(chartDef.xAxisFormatting).length) {
                chartDef.xAxisFormatting = {
                    axisTitleFormatting: { ...AxisTitleFormatting },
                    customExtent: { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT }
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisTitleFormatting(theme, chartDef.xAxisFormatting.axisTitleFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.displayAxis)) {
                chartDef.xAxisFormatting.displayAxis = true;
            }
            if (_.isNil(chartDef.xAxisFormatting.showAxisTitle)) {
                chartDef.xAxisFormatting.showAxisTitle = true;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting)) {
                chartDef.xAxisFormatting.axisValuesFormatting = {
                    axisTicksFormatting: { ...AxisTicksFormatting },
                    numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.xAxisFormatting.axisValuesFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting)) {
                chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting = { ...AxisTicksFormatting };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.xAxisFormatting.axisValuesFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting = { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS };
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.digitGrouping)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues = false;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage = false;
            }
            if (_.isNil(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros)) {
                chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros = getDefaultTrailingZeros(chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting);
            }
            if (_.isNil(chartDef.xAxisFormatting.axisTitleFormatting)) {
                chartDef.xAxisFormatting.axisTitleFormatting = { ...AxisTitleFormatting };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisTitleFormatting(theme, chartDef.xAxisFormatting.axisTitleFormatting);
                }
            }
            if (_.isNil(chartDef.xAxisFormatting.ticksConfig)) {
                chartDef.xAxisFormatting.ticksConfig = { mode: AxisTicksConfigMode.INTERVAL, number: null };
            }
            if (_.isNil(chartDef.xAxisFormatting.customExtent) || !Object.keys(chartDef.xAxisFormatting.customExtent).length) {
                chartDef.xAxisFormatting.customExtent = { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT };
            }


            // y axis
            if (_.isNil(chartDef.yAxesFormatting) || !chartDef.yAxesFormatting.length) {
                chartDef.yAxesFormatting = [];
            }

            const leftAxisFormatting = chartDef.yAxesFormatting.find(formatting => formatting.id === ChartsStaticData.LEFT_AXIS_ID || _.isNil(formatting.id));
            const rightAxisFormatting = chartDef.yAxesFormatting.find(formatting => formatting.id === ChartsStaticData.RIGHT_AXIS_ID);
            if (!leftAxisFormatting || !Object.keys(leftAxisFormatting).length) {
                chartDef.yAxesFormatting.push({
                    id: ChartsStaticData.LEFT_AXIS_ID
                });
            }
            if (!rightAxisFormatting || !Object.keys(rightAxisFormatting).length) {
                chartDef.yAxesFormatting.push({
                    id: ChartsStaticData.RIGHT_AXIS_ID
                });
            }

            chartDef.yAxesFormatting.forEach(axisFormatting => {
                if (_.isNil(axisFormatting.id)) {
                    // left axis is the only one that can have no id at initialization due to mapping from old charts
                    axisFormatting.id = ChartsStaticData.LEFT_AXIS_ID;
                }
                if (_.isNil(axisFormatting.displayAxis)) {
                    axisFormatting.displayAxis = true;
                }
                if (_.isNil(axisFormatting.showAxisTitle)) {
                    axisFormatting.showAxisTitle = true;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting)) {
                    axisFormatting.axisValuesFormatting = {
                        axisTicksFormatting: { ...AxisTicksFormatting },
                        numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, axisFormatting.axisValuesFormatting);
                    }

                }
                if (_.isNil(axisFormatting.axisValuesFormatting.axisTicksFormatting)) {
                    axisFormatting.axisValuesFormatting.axisTicksFormatting = { ...AxisTicksFormatting };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, axisFormatting.axisValuesFormatting);
                    }
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting)) {
                    axisFormatting.axisValuesFormatting.numberFormatting = { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS };
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.digitGrouping)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues = false;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.shouldFormatInPercentage = false;
                }
                if (_.isNil(axisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros)) {
                    axisFormatting.axisValuesFormatting.numberFormatting.hideTrailingZeros = getDefaultTrailingZeros(axisFormatting.axisValuesFormatting.numberFormatting);
                }
                if (_.isNil(axisFormatting.axisTitleFormatting)) {
                    axisFormatting.axisTitleFormatting = { ...AxisTitleFormatting };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyAxisTitleFormatting(theme, axisFormatting.axisTitleFormatting);
                    }
                }
                if (_.isNil(axisFormatting.ticksConfig)) {
                    axisFormatting.ticksConfig = { mode: AxisTicksConfigMode.INTERVAL, number: null };
                }
                if (_.isNil(axisFormatting.customExtent) || !Object.keys(axisFormatting.customExtent).length) {
                    axisFormatting.customExtent = { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT };
                }
                if (_.isNil(axisFormatting.includeZero)) {
                    axisFormatting.includeZero = true;
                }
            });


            // radial axis
            if (!chartDef.radialAxisFormatting || !chartDef.radialAxisFormatting.axisValuesFormatting || !chartDef.radialAxisFormatting.axisValuesFormatting.axisTicksFormatting) {
                // before we were storing the values formatting in the titles formatting
                const legacyFormatting = chartDef.radialAxisFormatting && chartDef.radialAxisFormatting.axisTitleFormatting && _.cloneDeep(chartDef.radialAxisFormatting.axisTitleFormatting);
                chartDef.radialAxisFormatting = {
                    axisValuesFormatting: {
                        axisTicksFormatting: {
                            ...AxisTitleFormatting,
                            ...(legacyFormatting || {})
                        }
                    }
                };
                if (theme && !legacyFormatting) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.radialAxisFormatting.axisValuesFormatting);
                };
            }
            if (_.isNil(chartDef.tooltipOptions)) {
                chartDef.tooltipOptions = { display: true };
            }

            // gauge options
            if (_.isNil(chartDef.gaugeOptions)) {
                chartDef.gaugeOptions = {
                    min: { ...DEFAULT_GAUGE_MIN_MAX },
                    max: { ...DEFAULT_GAUGE_MIN_MAX },
                    displayPointer: false
                };
            }
            if (_.isNil(chartDef.gaugeOptions.min)) {
                chartDef.gaugeOptions.min = { ...DEFAULT_GAUGE_MIN_MAX };
            }
            if (_.isNil(chartDef.gaugeOptions.max)) {
                chartDef.gaugeOptions.max = { ...DEFAULT_GAUGE_MIN_MAX };
            }
            if (_.isNil(chartDef.gaugeOptions.min.aggregatedColumn)) {
                chartDef.gaugeOptions.min.aggregatedColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.min.datasetColumn)) {
                chartDef.gaugeOptions.min.datasetColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.min.customAggregation)) {
                chartDef.gaugeOptions.min.customAggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.min.aggregation)) {
                chartDef.gaugeOptions.min.aggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.aggregatedColumn)) {
                chartDef.gaugeOptions.max.aggregatedColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.datasetColumn)) {
                chartDef.gaugeOptions.max.datasetColumn = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.customAggregation)) {
                chartDef.gaugeOptions.max.customAggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.max.aggregation)) {
                chartDef.gaugeOptions.max.aggregation = null;
            }
            if (_.isNil(chartDef.gaugeOptions.axis)) {
                chartDef.gaugeOptions.axis = {
                    ticksConfig: {
                        mode: AxisTicksConfigMode.INTERVAL,
                        number: null
                    },
                    axisValuesFormatting: {
                        axisTicksFormatting: {
                            ...AxisTicksFormatting,
                            fontSize: 16
                        },
                        numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                    },
                    thickness: 30
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.gaugeOptions.axis?.axisValuesFormatting);
                }
            }
            if (_.isNil(chartDef.gaugeOptions.axis.ticksConfig)) {
                chartDef.gaugeOptions.axis.ticksConfig = { mode: AxisTicksConfigMode.INTERVAL, number: null };
            }
            if (_.isNil(chartDef.gaugeOptions.axis.ticksConfigMode)) {
                chartDef.gaugeOptions.axis.ticksConfigMode = AxisTicksConfigMode.INTERVAL;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting = {
                    axisTicksFormatting: {
                        ...AxisTicksFormatting,
                        fontSize: 12
                    },
                    numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyAxisValueFormatting(theme, chartDef.gaugeOptions.axis.axisValuesFormatting);
                }

            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting = { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS };
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.digitGrouping)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.useParenthesesForNegativeValues = false;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.shouldFormatInPercentage)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.shouldFormatInPercentage = false;
            }
            if (_.isNil(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.hideTrailingZeros)) {
                chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting.hideTrailingZeros = getDefaultTrailingZeros(chartDef.gaugeOptions.axis.axisValuesFormatting.numberFormatting);
            }
        }

        function fixupNumberFormatting(formattableOptions) {
            if (!_.isNil(formattableOptions)) {
                formattableOptions.forEach(formattableOption => {

                    if (_.isNil(formattableOption.numberFormatting)) {
                        formattableOption.numberFormatting = { ...ChartsStaticData.DEFAULT_REFERENCE_LINES_NUMBER_FORMATTING_OPTIONS };
                    }

                    if (!_.isNil(formattableOption.prefix)) {
                        formattableOption.numberFormatting.prefix = formattableOption.prefix;
                        delete formattableOption.prefix;
                    }

                    if (!_.isNil(formattableOption.suffix)) {
                        formattableOption.numberFormatting.suffix = formattableOption.suffix;
                        delete formattableOption.suffix;
                    }

                    if (!_.isNil(formattableOption.multiplier)) {
                        formattableOption.numberFormatting.multiplier = formattableOption.multiplier;
                        delete formattableOption.multiplier;
                    }
                });
            }
        }

        function ensureReferenceLinesCompatibility(chartType, referenceLines) {
            if (!_.isNil(referenceLines)) {
                referenceLines.forEach(referenceLine => {
                    // revert type to constant for reference line of type AggregatedColumn on Unnaggregated charts
                    if ((chartType === CHART_TYPES.SCATTER || chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS)
                        && referenceLine.sourceType === ValueType.AggregatedColumn) {
                        referenceLine.sourceType = ValueType.Constant;
                        referenceLine.constantValue = undefined;
                    }
                });
            }
        }

        function fixupReferenceLines(chartDef) {
            // Number Formatting in reference lines was not using NumberFormattingOptions before 13.2.0
            fixupNumberFormatting(chartDef.referenceLines);
        }

        function fixupGaugeTargets(chartDef) {
            // Number Formatting in gauge targets was not using NumberFormattingOptions before 13.2.0
            fixupNumberFormatting(chartDef.gaugeOptions.targets);
        }

        function fixupDynamicMeasure(dynamicMeasure) {
            if (!dynamicMeasure) {
                return;
            }

            dynamicMeasure.aggregatedColumn && fixPercentile(dynamicMeasure.aggregatedColumn);
            dynamicMeasure.datasetColumn && fixPercentile(dynamicMeasure.datasetColumn);
            dynamicMeasure.customAggregation && fixPercentile(dynamicMeasure.customAggregation);
        }

        function fixPercentile(value) {
            if (_.isNil(value.percentile) || value.percentile === 0) {
                value.percentile = 50;
            }
        }

        function fixupPivotTableFormatting(tableFormatting, theme) {

            if (!_.isNil(tableFormatting.rowSubheaders)) {
                return; // Fixup already done, 13.4 structure already there
            }

            // Headers formatting has been splitted into main headers and subheaders in 13.4.0
            if (!_.isNil(tableFormatting.rowHeaders)) {
                tableFormatting.rowSubheaders = { ...tableFormatting.rowHeaders };
                // Conserve < 13.4.0 row headers unfreezed status
                tableFormatting.freezeRowHeaders = false;
            } else {
                tableFormatting.rowSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                tableFormatting.freezeRowHeaders = true; // Starting 13.4.0, row headers will be frozen by default
            }

            if (!_.isNil(tableFormatting.columnSubheaders)) {
                return; // Fixup already done, 13.4 structure already there
            }

            if (!_.isNil(tableFormatting.columnHeaders)) {
                tableFormatting.columnMainHeaders = { ...tableFormatting.columnHeaders };
                tableFormatting.columnSubheaders = { ...tableFormatting.columnHeaders };
            } else {
                tableFormatting.rowMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                tableFormatting.columnMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                tableFormatting.columnSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
            }
        }

        // By default, we trail zeros if no decimal places are specified, else we do
        function getDefaultTrailingZeros(numberFormattingOptions) {
            return !_.isNil(numberFormattingOptions.decimalPlaces) && numberFormattingOptions.decimalPlaces > 0 ? false : true;
        }

        function autocompleteGenericMeasure({ contextualMenuMeasureType, chartDef, defaultTextFormatting, theme }, measure, index, array) {
            const chartType = chartDef.type;
            if (!ChartLabels.aggregationIsSupported(measure.function, measure.type, chartType, contextualMenuMeasureType)) {
                measure.function = 'COUNT';
            }

            if (measure.isA !== 'measure') {
                Assert.trueish(measure.type, 'no measure type');
                const col = measure.column;
                const type = measure.type;
                const measureFn = measure.function;
                const measureInferredType = measure.inferredType;
                clear(measure);
                if (col === '__COUNT__') {
                    measure.column = null;
                    measure.function = 'COUNT';
                } else if (type === 'ALPHANUM') {
                    measure.column = col;
                    measure.function = 'COUNTD';
                } else if (type === 'DATE') {
                    measure.column = col;
                    measure.function = 'COUNT';
                } else if (type === 'CUSTOM') {
                    measure.column = col;
                    measure.function = 'CUSTOM';
                    measure.customFunction = measureFn;
                    measure.inferredType = measureInferredType;
                } else {
                    measure.column = col;
                    measure.function = 'AVG';
                }
                measure.type = type;
                measure.displayed = true;
                measure.displayAxis = 'axis1';
                measure.displayType = chartDef.type == CHART_TYPES.MULTI_COLUMNS_LINES && array.some(m => m.displayType === 'column') ? 'line' : 'column';

                measure.isA = 'measure';
                measure.percentile = 50;
                measure.valueTextFormatting = { ...DEFAULT_FONT_FORMATTING };
                measure.labelTextFormatting = { fontColor: '#333333', fontSize: 15 };

                measure.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                measure.useParenthesesForNegativeValues = false;
                measure.shouldFormatInPercentage = false;
                measure.hideTrailingZeros = getDefaultTrailingZeros(measure);
                measure.multiplier = ChartsStaticData.DEFAULT_MULTIPLIER;
                const textFormatting = defaultTextFormatting || computeGenericMeasureInitialValuesInChartDisplayTextFormattingOptions(chartDef);
                let spacing = 12;
                switch (chartDef.type) {
                    case CHART_TYPES.GROUPED_COLUMNS :
                        spacing = 2;
                        break;
                    case CHART_TYPES.MULTI_COLUMNS_LINES :
                        spacing = 5;
                        break;
                    case CHART_TYPES.LINES :
                        spacing = 12;
                        break;

                    default:
                        break;
                }
                const otherComputedProperties = computeGenericMeasureInitialValuesInChartDisplayOptions(chartDef);

                measure.valuesInChartDisplayOptions = {
                    displayValues: chartType !== CHART_TYPES.TREEMAP,
                    textFormatting,
                    additionalMeasures: [],
                    addDetails: false,
                    placementMode: ValuesInChartPlacementMode.AUTO,
                    spacing,
                    ...otherComputedProperties
                };
                if (theme) {
                    DSSVisualizationThemeUtils.applyToMeasure(chartType, measure, theme);
                }
            }

            fixPercentile(measure);

            // Measure-compute modes
            const measureComputationModes = ChartFeatures.getMeasureComputationModes(chartDef);
            const modeEnabled = measureComputationModes.some(m => m[0] === measure.computeMode);

            // Revert to 'NORMAL' if the compute mode is invalid
            if (!modeEnabled) {
                measure.computeMode = 'NORMAL';
            }
        }

        function fixupStackedColumnsOptions(chartDef, theme) {
            if (ChartFeatures.canDisplayTotalValues(chartDef)) {
                if (_.isNil(chartDef.stackedColumnsOptions)) {
                    chartDef.stackedColumnsOptions = {
                        totalsInChartDisplayOptions: {
                            textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                            additionalMeasures: [],
                            addDetails: false
                        }
                    };
                }
                if (_.isNil(chartDef.stackedColumnsOptions.totalsInChartDisplayOptions)) {
                    chartDef.stackedColumnsOptions.totalsInChartDisplayOptions = {
                        textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                        additionalMeasures: [],
                        addDetails: false
                    };
                }
                if (_.isNil(chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures)) {
                    chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures = [];
                }

                if (_.isNil(chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.spacing)) {
                    chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.spacing = 5;
                }
                chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'valuesInChart', chartDef, defaultTextFormatting: chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.textFormatting, theme }));
            }
        }

        function autocompleteGenericDimension(chartDef, dimension, listKey) {
            if (dimension.isA !== 'dimension') {
                Assert.trueish(dimension.column, 'no dimension column');
                Assert.trueish(dimension.type, 'no dimension type');
                const col = dimension.column;
                const type = dimension.type;
                clear(dimension);
                dimension.column = col;
                dimension.type = type;
                if (dimension.type === 'DATE') {
                    dimension.numParams = { emptyBinsMode: 'AVERAGE' };
                    dimension.dateParams = { mode: ChartsStaticData.defaultDateMode.value };
                } else if (dimension.type === 'NUMERICAL') {
                    /*
                     * Compute a proper number of bins :
                     * - Second dimension of a stack or group : 5
                     * - First dimension of an area or lines : 30
                     * - Scatter or first dimension of an histogram: 10
                     */
                    const cdt = chartDef.type;
                    const isMainDimension = [CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.STACKED_COLUMNS].includes(cdt) && dimension === chartDef.genericDimension0[0];
                    const nbBins = ChartDimension.getNumericalBinNumber(cdt, isMainDimension);
                    dimension.numParams = {
                        mode: 'FIXED_NB',
                        nbBins: nbBins,
                        binSize: 100,
                        emptyBinsMode: 'ZEROS'
                    };
                    dimension.maxValues = 100; // Will only be used if no binning
                } else if (dimension.type === 'ALPHANUM') {
                    dimension.numParams = { emptyBinsMode: 'ZEROS' };
                    dimension.maxValues = 20;
                }
                dimension.generateOthersCategory = true;
                dimension.filters = [];
                dimension.isA = 'dimension';

                if (dimension.type === 'GEOPOINT' && chartDef.type.find('_map') < 0) {
                    chartDef.type = CHART_TYPES.GRID_MAP;
                }
                if (dimension.type === 'GEOMETRY' && chartDef.type.find('_map') < 0) {
                    chartDef.type = CHART_TYPES.GRID_MAP;
                }

                if (listKey === 'genericDimension1'
                    && ChartFeatures.canIgnoreEmptyBinsForColorDimension(chartDef)
                    && chartDef.genericDimension0.length === 1
                    && chartDef.genericDimension0[0].column === dimension.column) {
                    dimension.ignoreEmptyBins = true;
                }

                dimension.digitGrouping = ChartsStaticData.DEFAULT_DIGIT_GROUPING;
                dimension.useParenthesesForNegativeValues = false;
                dimension.shouldFormatInPercentage = false;
                dimension.hideTrailingZeros = getDefaultTrailingZeros(dimension);
                dimension.multiplier = ChartsStaticData.DEFAULT_MULTIPLIER;
                dimension.oneTickPerBin = 'AUTO';
            }
        };


        const svc = {
            autocompleteUA,
            autocompleteGenericDimension: autocompleteGenericDimension,
            onChartTypeChange: function(chartDef, newType, newWebAppType) {
                Logger.info('Start type change:' + JSON.stringify(chartDef));

                let existingMeasures;
                let existingDimensions;
                let allMeasures;
                let allDimensions;
                let allUA;
                const oldType = chartDef.type;

                /*
                 * Reset user defined custom range on chart type change:
                 * Based on chart type, dimensions can move from x or y axes or other measures and break the intented range presentation (ex: from Scatter Plot to Grouped Bubbles)
                 */
                ChartAxesUtils.resetCustomExtents([chartDef.xAxisFormatting]);
                ChartAxesUtils.resetCustomExtents(chartDef.yAxesFormatting);

                if (oldType === CHART_TYPES.GEOMETRY_MAP) {
                    retrieveOptionsFromGeomMap(chartDef);
                    retrieveGeoFromGeomMap(chartDef);
                    chartDef.geoLayers = [svc.newEmptyGeoPlaceholder(0)];
                }

                ensureReferenceLinesCompatibility(newType, chartDef.referenceLines);

                switch (newType) {
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        if (existingDimensions.length) {
                            chartDef.genericDimension0 = existingDimensions.splice(0, 1);
                        }
                        if (existingDimensions.length) {
                            chartDef.genericDimension1 = existingDimensions.splice(0, 1);
                        }


                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        chartDef.tooltipMeasures = getAttributeAndRemove(existingMeasures, COLUMN_TYPE_ORDER.tooltip, true);
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.genericDimension0) && !has(chartDef.genericDimension1)) {
                            // 1D chart, get all measures
                            chartDef.genericMeasures = existingMeasures;
                            // eslint-disable-next-line no-undef
                        } else if (has(chartDef.genericDimension0) && has(chartDef.genericDimension1)) {
                            // 2D chart, get first measure and drop all others ...
                            chartDef.genericMeasures = existingMeasures.slice(0, 1);
                        }
                        break;
                    case CHART_TYPES.PIE:
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        if (existingMeasures.length >= 1) {
                            chartDef.genericMeasures = existingMeasures.slice(0, 1);
                        }
                        if (existingDimensions.length >= 1) {
                            chartDef.genericDimension0 = [existingDimensions[0]];
                        }
                        break;
                    case CHART_TYPES.SCATTER:
                    case CHART_TYPES.HEATMAP:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        setSingleIfHasEnough(existingDimensions, chartDef.uaXDimension, 0);
                        setSingleIfHasEnough(existingDimensions, chartDef.uaYDimension, 1);

                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef);
                        // Using the strict assignment just on the same kind of chart
                        if ([CHART_TYPES.SCATTER, CHART_TYPES.HEATMAP].includes(chartDef.type)) {
                            chartDef.uaColor = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.color);
                            chartDef.uaSize = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.size);
                            chartDef.uaTooltip = allUA;
                        } else {
                            // puts the dimensions that can be accepted as uaColor at the front of the list for easier selection
                            const acceptedScaleFirst = allUA.sort((a, b) => svc.scatterAcceptScaleMeasure(b).accept - svc.scatterAcceptScaleMeasure(a).accept);
                            if (acceptedScaleFirst.length && svc.scatterAcceptScaleMeasure(acceptedScaleFirst[0]).accept) {
                                setSingleIfHasEnough(acceptedScaleFirst, chartDef.uaSize, 0);
                                setSingleIfHasEnough(acceptedScaleFirst, chartDef.uaColor, 1);
                            } else {
                                setSingleIfHasEnough(allUA, chartDef.uaColor, 0);
                            }

                            chartDef.uaTooltip = allUA.slice(2);
                        }
                        break;
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        if (existingDimensions.length) {
                            chartDef.uaDimensionPair = [
                                {
                                    uaXDimension: [],
                                    uaYDimension: []
                                }
                            ];
                            const possibleXDimensions = existingDimensions.filter(d => svc.scatterMPAccept(d, chartDef, 'x', 0).accept);
                            const possibleYDimensions = existingDimensions.filter(d => svc.scatterMPAccept(d, chartDef, 'y', 0).accept
                                && (!possibleXDimensions[0] || possibleXDimensions[0].column !== d.column));
                            setSingleIfHasEnough(possibleXDimensions, chartDef.uaDimensionPair[0].uaXDimension, 0);
                            setSingleIfHasEnough(possibleYDimensions, chartDef.uaDimensionPair[0].uaYDimension, 0);
                        }
                        //we don't need them, but we need to purge them
                        ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef);
                        break;
                    case CHART_TYPES.GROUPED_XY:
                        /*
                         * TODO: If we come from scatter, maybe we could make a smarter
                         * choice by taking dimX -> AVG -> measX, dimY -> AVG -> measY
                         */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.groupDimension)) {
                            const existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                            if (existingDimensions.length >= 1) {
                                chartDef.groupDimension = [existingDimensions[0]];
                            }
                        }
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);

                        setSingleIfHasEnough(allMeasures, chartDef.xMeasure, 0);
                        setSingleIfHasEnough(allMeasures, chartDef.yMeasure, 1);
                        setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 2);
                        setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 3);
                        break;

                    case CHART_TYPES.BINNED_XY:
                        // xDim, yDim, colorMeasure, sizeM
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        setSingleIfHasEnough(existingDimensions, chartDef.xDimension, 0);
                        setSingleIfHasEnough(existingDimensions, chartDef.yDimension, 1);

                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if (newType === oldType) {
                            chartDef.colorMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.color);
                            chartDef.sizeMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.size);
                            chartDef.tooltipMeasures = allMeasures;
                        } else {
                            setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 0);
                            setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 1);
                            chartDef.tooltipMeasures = allMeasures.slice(2);
                        }
                        break;

                    case CHART_TYPES.PIVOT_TABLE:
                        // xDim, yDim, genericMeasures
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        setSingleIfHasEnough(existingDimensions, chartDef.xDimension, 0);
                        setSingleIfHasEnough(existingDimensions, chartDef.yDimension, 1);

                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        chartDef.genericMeasures = allMeasures;
                        break;

                    case CHART_TYPES.SCATTER_MAP:
                    case CHART_TYPES.DENSITY_HEAT_MAP:
                    case CHART_TYPES.HEATMAP_MAP:
                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if ([CHART_TYPES.SCATTER_MAP, CHART_TYPES.DENSITY_HEAT_MAP, CHART_TYPES.HEATMAP_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.GRID_MAP].includes(oldType)) {
                            chartDef.uaColor = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.color);
                            chartDef.uaSize = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.size);
                            chartDef.uaTooltip = allUA;
                        } else {
                            setSingleIfHasEnough(allUA, chartDef.uaSize, 0);
                            setSingleIfHasEnough(allUA, chartDef.uaColor, 1);
                            chartDef.uaTooltip = allUA.slice(2);
                        }
                        break;
                    case CHART_TYPES.GEOMETRY_MAP:
                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        chartDef.uaColor = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.color);
                        chartDef.uaSize = getAttributeAndRemove(allUA, COLUMN_TYPE_ORDER.size);
                        chartDef.uaTooltip = allUA;
                        migrateToGeomMap(chartDef);
                        break;
                    case CHART_TYPES.GROUPED_SCATTER_MAP:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.groupDimension)) {
                            allDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                            setSingleIfHasEnough(allDimensions, chartDef.groupDimension, 0);
                        }
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if (newType === oldType) {
                            chartDef.colorMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.color);
                            chartDef.sizeMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.size);
                            chartDef.tooltipMeasures = allMeasures;
                        } else {
                            setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 0);
                            setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 1);
                            chartDef.tooltipMeasures = allMeasures.slice(2);
                        }
                        break;

                    case CHART_TYPES.GRID_MAP:
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 0);
                        chartDef.tooltipMeasures = allMeasures.slice(1);
                        break;

                    case CHART_TYPES.ADMINISTRATIVE_MAP:
                        /*
                         * TODO: for admin map if we are in FILLED mode then we only
                         * have a color measure, no size, so we should probably inverse
                         */
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        // Using the strict assignment just on the same kind of chart
                        if (newType === oldType) {
                            chartDef.colorMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.color);
                            chartDef.sizeMeasure = getAttributeAndRemove(allMeasures, COLUMN_TYPE_ORDER.size);
                            chartDef.tooltipMeasures = allMeasures;
                        } else {
                            setSingleIfHasEnough(allMeasures, chartDef.sizeMeasure, 0);
                            setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 1);
                            chartDef.tooltipMeasures = allMeasures.slice(2);
                        }
                        break;
                    case CHART_TYPES.TREEMAP:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef).filter(d => svc.stdAggregatedAcceptDimension(d).accept);
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasure(m).accept);
                        setSingleIfHasEnough(allMeasures, chartDef.genericMeasures, 0);
                        setSingleIfHasEnough(allMeasures, chartDef.colorMeasure, 1);
                        chartDef.yDimension = existingDimensions;
                        break;
                    case CHART_TYPES.DENSITY_2D:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef).filter(d => svc.density2dAccept(d).accept);
                        setSingleIfHasEnough(existingDimensions, chartDef.xDimension, 0);
                        setSingleIfHasEnough(existingDimensions, chartDef.yDimension, 1);
                        break;

                    case CHART_TYPES.GAUGE:
                        ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef);
                        existingMeasures = existingMeasures.filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        setSingleIfHasEnough(existingMeasures, chartDef.genericMeasures, 0);
                        break;

                    case CHART_TYPES.KPI:
                        ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef);
                        chartDef.genericMeasures = existingMeasures.filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        break;

                    case CHART_TYPES.RADAR:
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        chartDef.genericMeasures = existingMeasures;

                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        setSingleIfHasEnough(existingDimensions, chartDef.genericDimension0, 0);
                        break;

                    case CHART_TYPES.SANKEY:
                        existingMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        setSingleIfHasEnough(existingMeasures, chartDef.genericMeasures, 0);

                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        chartDef.yDimension = existingDimensions;
                        break;

                    case CHART_TYPES.BOXPLOTS:
                        existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                        if (existingDimensions.length) {
                            chartDef.boxplotBreakdownDim = existingDimensions.splice(0, 1);
                        }
                        if (existingDimensions.length) {
                            chartDef.genericDimension1 = existingDimensions.splice(0, 1);
                        }
                        allUA = ChartTypeChangeUtils.takeAllUAWithMeasures(chartDef).filter(m => svc.boxplotsAcceptMeasure(m).accept);
                        setSingleIfHasEnough(allUA, chartDef.boxplotValue, 0);
                        break;
                    case CHART_TYPES.LIFT:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.groupDimension)) {
                            existingDimensions = ChartTypeChangeUtils.takeAllExistingDimensions(chartDef);
                            if (existingDimensions.length >= 1) {
                                chartDef.groupDimension = [existingDimensions[0]];
                            }
                        }
                        allMeasures = ChartTypeChangeUtils.takeAllMeasuresWithUA(chartDef).filter(m => svc.stdAggregatedAcceptMeasureWithAlphanumResults(m).accept);
                        setSingleIfHasEnough(allMeasures, chartDef.xMeasure, 0);
                        setSingleIfHasEnough(allMeasures, chartDef.yMeasure, 1);
                        break;

                    case CHART_TYPES.WEBAPP:
                        // all custom
                        break;

                    case CHART_TYPES.NUMERICAL_HEATMAP:
                    default:
                        throw Error('unimplemented chart type : ' + newType);
                }

                if (!ChartFeatures.canAnimate(newType)) {
                    chartDef.animationDimension.length = 0;
                }

                if (!ChartFeatures.canFacet(newType, newWebAppType)) {
                    chartDef.facetDimension.length = 0;
                }
            },

            /* ********************** ACCEPT / REJECT drops per type ********************* */

            stdAggregatedAcceptDimension: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {

                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                // Reject count of records
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NO_AGGR_ON_COUNT_REC);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return accept();
            },

            stdAggregatedAcceptMeasure: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isDateColumnType(data.type)) {
                    return accept();
                }
                if (ChartColumnTypeUtils.isCustomMeasureAndNonNumerical(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_CUSTOM_COL);
                }
                if (!ChartColumnTypeUtils.isNumericalColumnType(data.type) && !ChartColumnTypeUtils.isAlphanumColumnType(data.type) && !ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_ALPHA);
                }
                return accept();
            },

            stdAggregatedAcceptMeasureWithAlphanumResults: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isDateColumnType(data.type)) {
                    return accept();
                }
                if (!ChartColumnTypeUtils.isNumericalColumnType(data.type) && !ChartColumnTypeUtils.isAlphanumColumnType(data.type) && !ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_ALPHA);
                }
                return accept();
            },

            uaTooltipAccept: function(data) {
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(`${ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC} in tooltips`);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return accept();
            },


            scatterAccept: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return accept();
            },

            scatterMPAccept: function(data, chartDef, dimension, pairIndex) {
                const index = parseInt(pairIndex);
                const { firstXDim, firstXDimIndex } = svc.getScatterFirstXDims(chartDef.uaDimensionPair, index);
                if (dimension === 'x') {
                    if (!_.isNil(firstXDim) && firstXDimIndex !== index) {
                        const sameType = ChartColumnTypeUtils.isSameColumnType(data, firstXDim);
                        if (!sameType) {
                            return reject(ChartLabels.CHART_ERROR_MESSAGES.COL_TYPE_X);
                        }
                    }
                    if (ChartColumnTypeUtils.isAlphanumColumnType(data.type)) {
                        return reject(ChartLabels.CHART_ERROR_MESSAGES.SCATTER_MP_ALPHANUM_X);
                    }
                    return svc.scatterAccept(data);
                }

                return svc.scatterAccept(data);
            },

            scatterAcceptScaleMeasure: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isAlphanumColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.SCATTER_SCALE_NON_NUM);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return svc.scatterAccept(data);
            },

            densityMapAcceptScaleMeasure: function(data) {
                return svc.scatterAcceptScaleMeasure(data);
            },

            acceptGeo: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return accept();
                } else {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.ONLY_GEO);
                }
            },

            density2dAccept: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isNumericalColumnType(data.type)
                    || ChartColumnTypeUtils.isDateColumnType(data.type)) {
                    return accept();
                } else {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.DENSITY_NUM_OR_DATE);
                }
            },

            boxplotsAcceptBreakdown: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isGeoColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.GEO_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                return accept();
            },
            boxplotsAcceptMeasure: function(data) {
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                if (!ChartColumnTypeUtils.isNumericalColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_NUM_COL);
                }
                if (ChartColumnTypeUtils.isCount(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.COUNT_REC);
                }
                if (ChartColumnTypeUtils.isRBND(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_RBND);
                }
                return accept();
            },
            binnedXYAcceptDimension: function(chartDef, data) {
                Assert.trueish(chartDef.variant, 'no chartDef variant');
                if (ChartColumnTypeUtils.isNonColumnData(data)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_COL_DATA);
                }
                if (chartDef.variant === CHART_VARIANTS.binnedXYHexagon && !ChartColumnTypeUtils.isNumericalColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.HEX_BINNING_DIM);
                }
                if (ChartColumnTypeUtils.isCustomColumnType(data.type)) {
                    return reject(ChartLabels.CHART_ERROR_MESSAGES.NON_CUSTOM_COL);
                }
                return svc.stdAggregatedAcceptDimension(data);
            },

            getScatterFirstXDims(pairs, pairIndex) {
                let firstXDim;
                let firstXDimIndex;
                for (let i = 0; i < pairs.length; i++) {
                    if (i !== pairIndex) {
                        if (pairs[i].uaXDimension && pairs[i].uaXDimension.length && _.isNil(firstXDim)) {
                            firstXDim = pairs[i].uaXDimension[0];
                            firstXDimIndex = i;
                        }
                        if (!_.isNil(firstXDim)) {
                            return { firstXDim, firstXDimIndex };
                        }
                    }
                }
                return { firstXDim, firstXDimIndex };
            },

            computeAutoName: function(chartDef) {
                // If returns null or empty, don't use the new name.

                function getAutoNameMeasureLabel(measure) {
                    return measure.function === 'COUNT' ? 'Count' : ChartLabels.getShortMeasureLabel(measure, false);
                }

                function addGM0L() {
                    // eslint-disable-next-line no-undef
                    if (has(chartDef.genericMeasures)) {
                        newName += getAutoNameMeasureLabel(chartDef.genericMeasures[0]);
                    }
                }

                function sdl(d) {
                    return d[0].column;
                }

                function sml(m) {
                    return getAutoNameMeasureLabel(m[0]);
                }

                let dimensionPairs;
                let newName = '';
                switch (chartDef.type) {
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                    case CHART_TYPES.PIE:
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.genericDimension0) && !has(chartDef.genericDimension1)) {
                            addGM0L();
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(chartDef.genericDimension0[0], false);
                            // eslint-disable-next-line no-undef
                        } else if (has(chartDef.genericDimension0) && has(chartDef.genericDimension1)) {
                            addGM0L();
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(chartDef.genericDimension0[0], false);
                            newName += ' and ';
                            newName += ChartLabels.getDimensionLabel(chartDef.genericDimension1[0], false);
                        }
                        break;
                    case CHART_TYPES.PIVOT_TABLE:
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.xDimension) && has(chartDef.yDimension) && has(chartDef.genericMeasures)) {
                            addGM0L();
                            newName += ' by ';
                            newName += sdl(chartDef.xDimension);
                            newName += ' and ';
                            newName += sdl(chartDef.yDimension);
                        }
                        break;
                    case CHART_TYPES.RADAR:
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.genericDimension0) && has(chartDef.genericMeasures)) {
                            chartDef.genericMeasures.forEach((dim, i) => {
                                newName += dim.column;
                                if (i === chartDef.genericMeasures.length - 2) {
                                    newName += ' and ';
                                } else if (i < chartDef.genericMeasures.length - 1) {
                                    newName += ', ';
                                }
                            });
                            newName += ' by ' + ChartLabels.getDimensionLabel(chartDef.genericDimension0[0], false);
                        }
                        break;
                    case CHART_TYPES.TREEMAP:
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.yDimension) && has(chartDef.genericMeasures)) {
                            addGM0L();
                            newName += ' by ';
                            chartDef.yDimension.forEach((dim, i) => {
                                newName += dim.column;
                                if (i === chartDef.yDimension.length - 2) {
                                    newName += ' and ';
                                } else if (i < chartDef.yDimension.length - 1) {
                                    newName += ', ';
                                }
                            });
                        }
                        break;
                    case CHART_TYPES.BINNED_XY:
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.xDimension) && has(chartDef.yDimension)) {
                            newName += sdl(chartDef.xDimension) + ' vs ' + sdl(chartDef.yDimension) + ' (aggregated)';
                        }
                        break;
                    case CHART_TYPES.GROUPED_XY:
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.xMeasure) && has(chartDef.yMeasure) && has(chartDef.groupDimension)) {
                            newName += sml(chartDef.xMeasure) + ' / ' + sml(chartDef.yMeasure) + ' by ' +
                                sdl(chartDef.groupDimension);
                        }
                        break;
                    case CHART_TYPES.SCATTER:
                        newName = `${sdl(chartDef.uaXDimension)} vs ${sdl(chartDef.uaYDimension)}`;
                        break;
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        dimensionPairs = ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
                        newName = `${sdl(ChartUADimension.getPairUaXDimension(dimensionPairs, dimensionPairs[0]))} vs ${sdl(dimensionPairs[0].uaYDimension)}`;
                        break;
                    case CHART_TYPES.GAUGE:
                    case CHART_TYPES.KPI:
                        addGM0L();
                        break;
                    case CHART_TYPES.WEBAPP:
                        newName = chartDef.$loadedDesc.desc.meta.label || CHART_TYPES.WEBAPP;
                        break;
                    case CHART_TYPES.BOXPLOTS:
                        newName += ChartLabels.getDimensionLabel(chartDef.boxplotValue[0], false);
                        // eslint-disable-next-line no-undef
                        if (has(chartDef.boxplotBreakdownDim) && !has(chartDef.genericDimension1)) {
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(chartDef.boxplotBreakdownDim[0], false);
                            // eslint-disable-next-line no-undef
                        } else if (has(chartDef.boxplotBreakdownDim) && has(chartDef.genericDimension1)) {
                            addGM0L();
                            newName += ' by ';
                            newName += ChartLabels.getDimensionLabel(chartDef.boxplotBreakdownDim[0], false);
                            newName += ' and ';
                            newName += ChartLabels.getDimensionLabel(chartDef.genericDimension1[0], false);
                        }
                        break;
                }
                return newName;
            },

            /* ********************** Indicates chart validity ********************* */

            getValidity: function(chart) {
                const chartDef = chart.def;

                function ok() {
                    return {
                        valid: true
                    };
                }

                function incomplete(message) {
                    return {
                        valid: false,
                        type: 'INCOMPLETE',
                        message: message
                    };
                }

                function invalid(message) {
                    return {
                        valid: false,
                        type: 'INVALID',
                        message: message
                    };
                }

                function isGeometryMapComplete(geoLayers) {
                    return geoLayers.some(geoLayer => geoLayer.geometry && geoLayer.geometry.length > 0);
                }

                switch (chartDef.type) {
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                    case CHART_TYPES.PIE:
                        /* Minimal validity condition: first dimension, 1 measure */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericDimension0)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        /* Check for invalidities */
                        if (chartDef.type === CHART_TYPES.STACKED_COLUMNS || chartDef.type === CHART_TYPES.STACKED_BARS) {
                            if (chartDef.variant === 'stacked_100') {
                                // Stack 100% needs two dimensions to be meaningful
                                // eslint-disable-next-line no-undef
                                if (!has(chartDef.genericDimension1) && chartDef.genericMeasures.length === 1) {
                                    return invalid(ChartLabels.CHART_ERROR_MESSAGES.STACKED_100_COLS);
                                }
                            }
                        }
                        return ok();

                    case CHART_TYPES.PIVOT_TABLE:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.BINNED_XY:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.xDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                        }

                        if (chartDef.variant === CHART_VARIANTS.binnedXYHexagon) {
                            if (!ChartDimension.isTrueNumerical(chartDef.xDimension[0]) ||
                                !ChartDimension.isTrueNumerical(chartDef.yDimension[0])) {
                                return invalid(ChartLabels.CHART_ERROR_MESSAGES.HEX_BINNING_DIM);
                            }
                        }
                        return ok();

                    case CHART_TYPES.GROUPED_XY:
                    case CHART_TYPES.LIFT:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.groupDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.xMeasure)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.LIFT_GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yMeasure)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.LIFT_GIVE_Y_AXIS);
                        }
                        return ok();


                    case CHART_TYPES.DENSITY_2D:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.xDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                        }
                        return ok();

                    case CHART_TYPES.SCATTER:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.uaXDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.uaYDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                        }
                        return ok();

                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        if (!ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair).length) {
                            // eslint-disable-next-line no-undef
                            if (!chartDef.uaDimensionPair || !chartDef.uaDimensionPair.length || !has(chartDef.uaDimensionPair[0].uaXDimension)) {
                                return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_X_AXIS);
                            }
                            // eslint-disable-next-line no-undef
                            if (!has(chartDef.uaDimensionPair[0].uaYDimension)) {
                                return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_Y_AXIS);
                            }
                        }
                        return ok();

                    case CHART_TYPES.BOXPLOTS:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.boxplotValue)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.GAUGE:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        // eslint-disable-next-line no-undef
                        if (chartDef.genericMeasures.length > 1) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GAUGE_ONLY_ONE);
                        }
                        return ok();
                    case CHART_TYPES.KPI:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.RADAR:
                        /* Minimal validity condition: first dimension, 1 measure */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericDimension0)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.WHAT_TO_SHOW);
                        }
                        return ok();

                    case CHART_TYPES.SANKEY:
                        /* Minimal validity condition: 1 dimension, 2 measures */
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension) || chartDef.yDimension.length < 2) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.TWO_LEVELS);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.SANKEY_GIVE_COL);
                        }
                        return ok();

                    case CHART_TYPES.SCATTER_MAP:
                    case CHART_TYPES.DENSITY_HEAT_MAP:
                    case CHART_TYPES.ADMINISTRATIVE_MAP:
                    case CHART_TYPES.GRID_MAP:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.geometry)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GEO_COL);
                        }
                        return ok();
                    case CHART_TYPES.GEOMETRY_MAP:
                        if (!isGeometryMapComplete(chartDef.geoLayers)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GEO_COL);
                        }
                        return ok();
                    case CHART_TYPES.TREEMAP:
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.yDimension)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.GIVE_GROUP);
                        }
                        // eslint-disable-next-line no-undef
                        if (!has(chartDef.genericMeasures)) {
                            return incomplete(ChartLabels.INCOMPLETE_CHART_LABELS.TREEMAP_GIVE_NUM_COL);
                        }
                        return ok();
                    case CHART_TYPES.WEBAPP:
                        // all custom
                        return ok();

                    default:
                        throw Error('unimplemented handling of ' + chartDef.type);
                }
            },

            /************** Indicates charts warnings **************/

            /**
             * Indicates if pivot table threshold has been reached
             */
            hasPivotTableReachThreshold: function(chartDef, data) {
                return chartDef.type === CHART_TYPES.PIVOT_TABLE && data && data.errorType === 'com.dataiku.dip.pivot.backend.model.SecurityAbortedException';
            },

            /**
             * Indicates if pivot table threshold has been reached
             */
            hasCorruptedLinoCacheWarning: function(data) {
                return data && data.errorType === 'com.dataiku.dip.pivot.backend.model.CorruptedDataException';
            },

            /**
             * Indicates if pivot table threshold has been reached
             */
            hasEmptyCustomBins: function(data) {
                return data && data.errorType === 'com.dataiku.dip.pivot.backend.model.EmptyCustomBinsException';
            },

            /**
             * Returns true if there is a waning to display
             */
            hasRequestResponseWarning: function(chartDef, data) {
                return this.hasPivotTableReachThreshold(chartDef, data) || this.hasCorruptedLinoCacheWarning(data) || this.hasEmptyCustomBins(data);
            },

            /**
             * Returns validity warning
             */
            getRequestResponseWarning: function(chartDef, data) {
                if (this.hasPivotTableReachThreshold(chartDef, data)) {
                    return {
                        valid: false,
                        type: 'PIVOT_TABLE_TOO_MUCH_DATA',
                        message: data.message
                    };
                } else if (this.hasCorruptedLinoCacheWarning(data)) {
                    return {
                        valid: false,
                        type: 'CORRUPTED_LINO_CACHE',
                        message: data.message
                    };
                } else if (this.hasEmptyCustomBins(data)) {
                    return {
                        valid: false,
                        type: 'EMPTY_CUSTOM_BINS',
                        message: data.message
                    };
                }
            },

            /* ********************** Fixup, autocomplete, handle sort ********************* */

            /* Fixup everything that needs to */
            fixupSpec: function(chart, theme, changedDefinition) {
                const chartDef = chart.def;
                const usableColumns = chart.summary && chart.summary.usableColumns;
                Logger.info('Fixing up the spec: ' + JSON.stringify(chartDef));

                if (chartDef.type === CHART_TYPES.WEBAPP) {
                    chartDef.webAppConfig = chartDef.webAppConfig || {};
                    chartDef.$loadedDesc = WebAppsService.getWebAppLoadedDesc(chartDef.webAppType) || {};
                    chartDef.$pluginDesc = WebAppsService.getOwnerPluginDesc(chartDef.webAppType);
                    chartDef.$pluginChartDesc = chartDef.$loadedDesc.desc.chart;
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.leftBarParams, chartDef.webAppConfig);
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.topBarParams, chartDef.webAppConfig);
                } else {
                    chartDef.$loadedDesc = null;
                    chartDef.$pluginDesc = null;
                    chartDef.$pluginChartDesc = null;
                }

                if (ChartFeatures.canDisplayDimensionValuesInChart(chartDef.type)) {
                    chartDef.yDimension.forEach((dim, i) => {
                        if (dim.showDimensionValuesInChart !== false) {
                            dim.showDimensionValuesInChart = true;
                        }
                        if (chartDef.yDimension.length - 1 === i) {
                            dim.$canChangeShowValuesInChart = false;
                            dim.showDimensionValuesInChart = true;
                        } else {
                            dim.$canChangeShowValuesInChart = true;
                        }
                        if (!dim.textFormatting) {
                            dim.textFormatting = { ...DEFAULT_FONT_FORMATTING };
                        }
                    });
                }


                // resetting values display mode to default if selected mode is unsuported
                const valuesDisplayMode = chartDef.valuesInChartDisplayOptions.displayMode;
                if (chartDef.type === CHART_TYPES.STACKED_COLUMNS && (valuesDisplayMode === VALUES_DISPLAY_MODES.LABELS || valuesDisplayMode === VALUES_DISPLAY_MODES.VALUES_AND_LABELS)
                    || chartDef.type === CHART_TYPES.PIE && valuesDisplayMode === VALUES_DISPLAY_MODES.VALUES_AND_TOTALS) {
                    chartDef.valuesInChartDisplayOptions.displayMode = VALUES_DISPLAY_MODES.VALUES;
                }

                /* If log scale is enabled on X axis but not possible, disable it */
                if (chartDef.xAxisFormatting.isLogScale && !ChartFeatures.canSetLogScale(chartDef, 'x')) {
                    chartDef.xAxisFormatting.isLogScale = false;
                }

                chartDef.yAxesFormatting.forEach(axisFormatting => {
                    /* If log scale is enabled on Y axis but not possible, disable it */
                    if (axisFormatting.isLogScale && !ChartFeatures.canSetLogScale(chartDef, 'y')) {
                        axisFormatting.isLogScale = false;
                    }
                });

                if (chartDef.gridlinesOptions.$default) {
                    const isOnlyRightAxis = chartDef.genericMeasures && chartDef.genericMeasures.length === 1 && chartDef.genericMeasures[0].displayAxis === 'axis2';
                    chartDef.gridlinesOptions.vertical.show = chartDef.type === CHART_TYPES.STACKED_BARS;
                    chartDef.gridlinesOptions.horizontal.show = ![CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.BINNED_XY, CHART_TYPES.STACKED_BARS].includes(chartDef.type);
                    chartDef.gridlinesOptions.horizontal.displayAxis.type = isOnlyRightAxis ? GridlinesAxisType.RIGHT_Y_AXIS : GridlinesAxisType.LEFT_Y_AXIS;
                }


                chartDef.genericDimension0.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.genericDimension1.forEach(dimension => autocompleteGenericDimension(chartDef, dimension, 'genericDimension1'));
                chartDef.facetDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.animationDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.boxplotBreakdownDim.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.xDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.yDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));
                chartDef.groupDimension.forEach(dimension => autocompleteGenericDimension(chartDef, dimension));

                // Reset custom range settings when there are no dimensions
                // eslint-disable-next-line no-undef
                if (areAllEmpty(ChartTypeChangeUtils.takeAllInX(chartDef))) {
                    ChartAxesUtils.resetCustomExtents([chartDef.xAxisFormatting]);
                }
                // eslint-disable-next-line no-undef
                if (areAllEmpty(ChartTypeChangeUtils.takeAllInY(chartDef))) {
                    ChartAxesUtils.resetCustomExtents(chartDef.yAxesFormatting);
                }

                if (!chartDef.filters) {
                    chartDef.filters = [];
                }

                chartDef.filters.forEach(filter => ChartFilters.autocompleteFilter(filter, usableColumns, false));


                chartDef.genericMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'generic', chartDef, theme }));
                chartDef.genericMeasures.forEach(measure => {
                    if (measure.valuesInChartDisplayOptions && measure.valuesInChartDisplayOptions.additionalMeasures) {
                        measure.valuesInChartDisplayOptions.additionalMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'valuesInChart', chartDef, defaultTextFormatting: measure.valuesInChartDisplayOptions.textFormatting, theme }));
                    }
                });
                chartDef.sizeMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'size', chartDef, theme }));
                chartDef.colorMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'color', chartDef, theme }));
                if (Array.isArray(chartDef.colorGroups)) {
                    chartDef.colorGroups.forEach(colorGroup => {
                        if (_.isNil(colorGroup.colorMeasure)) {
                            colorGroup.colorMeasure = [];
                        } else {
                            colorGroup.colorMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'colorGroup', chartDef, theme }));
                        }

                        if (chartDef.type === CHART_TYPES.KPI) {
                            colorGroup.colorGroupMode = 'RULES';
                        }
                    });
                } else {
                    chartDef.colorGroups = [];
                }
                chartDef.tooltipMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'tooltip', chartDef, theme }));
                chartDef.xMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'x', chartDef, theme }));
                chartDef.yMeasure.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'y', chartDef, theme }));
                chartDef.uaXDimension.forEach(autocompleteUA);
                chartDef.uaYDimension.forEach(autocompleteUA);
                chartDef.uaDimensionPair.forEach(pair => {
                    pair.uaXDimension.forEach(autocompleteUA);
                    pair.uaYDimension.forEach(autocompleteUA);
                });
                chartDef.uaYDimension.forEach(autocompleteUA);
                chartDef.uaSize.forEach(autocompleteUA);
                chartDef.uaColor.forEach(autocompleteUA);
                chartDef.uaTooltip.forEach(autocompleteUA);
                chartDef.geometry.forEach(autocompleteUA);
                chartDef.boxplotValue.forEach(autocompleteUA);

                function handleSortsOnStdDimension(dimension) {
                    const oldSort = dimension.sort;
                    let firstNonNaturalSort = null;
                    dimension.possibleSorts = [{ type: 'NATURAL', label: translate('CHARTS.SORTING_METHOD.NATURAL_ORDERING', 'Natural ordering'), sortAscending: true }];

                    function addMeasureSorts(measure, resultingIdx) {
                        let sort = null;

                        if (measure) {
                            sort = { type: 'AGGREGATION', measureIdx: resultingIdx, label: ChartLabels.getLongMeasureLabel(measure, true, false) };
                        }

                        if (firstNonNaturalSort == null) {
                            firstNonNaturalSort = sort;
                        }

                        if (sort) {
                            dimension.possibleSorts.push(sort);

                            // Generate the sort ascending version
                            const ascSort = { type: 'AGGREGATION', measureIdx: resultingIdx, sortAscending: true, label: ChartLabels.getLongMeasureLabel(measure, true, true) };
                            dimension.possibleSorts.push(ascSort);
                        }
                    }

                    chartDef.genericMeasures.forEach(addMeasureSorts);
                    chartDef.tooltipMeasures.forEach(function(x, i) {
                        addMeasureSorts(x, chartDef.genericMeasures.length + i);
                    });

                    // Put back the old sort if possible ...
                    dimension.sort = null;
                    for (let i = 0; i < dimension.possibleSorts.length; i++) {
                        if (angular.equals(oldSort, dimension.possibleSorts[i])) {
                            dimension.sort = dimension.possibleSorts[i];
                            break;
                        }
                    }
                    if (dimension.sort == null) {
                        if (dimension.type === 'ALPHANUM' && firstNonNaturalSort != null) {
                            dimension.sort = firstNonNaturalSort;
                        } else {
                            dimension.sort = dimension.possibleSorts[0];
                            dimension.isSortDefault = true;
                        }
                    }
                }

                chartDef.genericDimension0.forEach(handleSortsOnStdDimension);
                chartDef.facetDimension.forEach(handleSortsOnStdDimension);
                chartDef.animationDimension.forEach(handleSortsOnStdDimension);
                chartDef.xDimension.forEach(handleSortsOnStdDimension);
                chartDef.yDimension.forEach(handleSortsOnStdDimension);
                chartDef.groupDimension.forEach(handleSortsOnStdDimension);

                let colorColumnData;
                const currColorColumn = ChartColorSelection.getColorDimensionOrMeasure(chartDef, chartDef.geoLayers[0]);
                if (usableColumns && currColorColumn) {
                    colorColumnData = usableColumns.find(c => c.column === currColorColumn.column);
                }
                // rollback to default palette if there is no color column or no meaning and the chart is using a meaning palette
                if ((!colorColumnData || !colorColumnData.meaningInfo) && (chartDef.colorOptions.colorPalette === DKU_PALETTE_NAMES.MEANING)) {
                    chartDef.colorOptions.colorPalette = 'default';
                }

                function handleSortsOnBoxplotDimension(dimension) {
                    const oldSort = dimension.sort;
                    dimension.possibleSorts = [
                        { type: 'NATURAL', label: 'Natural ordering', sortAscending: true },
                        { type: 'COUNT', label: ChartLabels.DESCENDING_COUNT_OF_RECORDS_LABEL, sortAscending: false },
                        { type: 'COUNT', label: ChartLabels.ASCENDING_COUNT_OF_RECORDS_LABEL, sortAscending: true }
                    ];

                    // Put back the old sort if possible ...
                    dimension.sort = null;
                    for (let i = 0; i < dimension.possibleSorts.length; i++) {
                        if (angular.equals(oldSort, dimension.possibleSorts[i])) {
                            dimension.sort = dimension.possibleSorts[i];
                            break;
                        }
                    }
                    /* Default to count desc */
                    if (dimension.sort == null) {
                        dimension.sort = dimension.possibleSorts[1];
                    }
                }

                chartDef.boxplotBreakdownDim.forEach(handleSortsOnBoxplotDimension);
                if (chartDef.type === CHART_TYPES.BOXPLOTS) {
                    chartDef.genericDimension1.forEach(handleSortsOnBoxplotDimension);
                } else {
                    chartDef.genericDimension1.forEach(handleSortsOnStdDimension);
                }

                let resetAllColumns = false;
                let indexChanged = -1;
                if (changedDefinition && chartDef.type == CHART_TYPES.MULTI_COLUMNS_LINES) {
                    if (changedDefinition.name === 'genericMeasures' && chartDef.genericDimension1.length) {
                        indexChanged = changedDefinition.nv.findIndex((v, index) => !_.isEqual(v, changedDefinition.ov[index]));
                    }
                    if (changedDefinition.name === 'genericDimension1' && changedDefinition.nv.length) {
                        resetAllColumns = true;
                    }
                }

                let columnFound = false;
                chartDef.genericMeasures.forEach((gm, index, array) => {
                    if (gm.displayed && gm.displayAxis == null) {
                        gm.displayAxis = 'axis1';
                    }
                    if (gm.displayed && gm.displayType == null) {
                        /*
                         * TODO: Reuse the logic that makes the default
                         * display type according to the dimension type
                         */
                        gm.displayType = chartDef.type == CHART_TYPES.MULTI_COLUMNS_LINES && array.length > 1 ? 'line' : 'column';
                    }

                    if (!gm.labelPosition) {
                        gm.labelPosition = ChartsStaticData.LABEL_POSITIONS.BOTTOM.value;
                    }

                    if (gm.showDisplayLabel === undefined) {
                        gm.showDisplayLabel = true;
                    }

                    if (gm.showValue === undefined) {
                        gm.showValue = true;
                    }

                    if (!gm.colorRules) {
                        gm.colorRules = [];
                    }
                    if (!gm.kpiTextAlign) {
                        gm.kpiTextAlign = 'CENTER';
                    }
                    if (!gm.labelTextFormatting) {
                        gm.labelTextFormatting = { fontColor: '#333333', fontSize: 15 };
                    }
                    if (gm.labelFontSize) {
                        //ensuring retrocompability - labelFontSize is legacy
                        gm.labelTextFormatting.fontSize = gm.labelFontSize;
                        delete gm.labelFontSize;
                    }
                    if (!gm.kpiValueFontSizeMode) {
                        gm.kpiValueFontSizeMode = ChartsStaticData.availableFontSizeModes.RESPONSIVE.value;
                    }
                    if (!gm.kpiValueFontSize) {
                        gm.kpiValueFontSize = 32;
                    }
                    if (!gm.responsiveTextAreaFill) {
                        gm.responsiveTextAreaFill = 100;
                    }
                    if (resetAllColumns && columnFound) {
                        gm.displayType = 'line';
                    }
                    if (gm.displayType === 'column') {
                        columnFound = true;
                    }
                    if (indexChanged == index && gm.displayType === 'column') {
                        for (let oi = 0; oi < array.length; oi++) {
                            if (oi === indexChanged) {
                                continue;
                            }
                            if (array[oi].displayType === 'column') {
                                array[oi].displayType = 'line';
                            }
                        }
                    }
                });
                /*
                 * For std-aggregated charts that only support 1 measure when there
                 * are two dimensions, move additional measures to tooltip
                 */
                // eslint-disable-next-line no-undef
                if (has(chartDef.genericDimension0) && has(chartDef.genericDimension1) && chartDef.type !== CHART_TYPES.MULTI_COLUMNS_LINES) {
                    chartDef.genericMeasures.splice(1, chartDef.genericMeasures.length - 1).forEach(function(x) {
                        chartDef.tooltipMeasures.push(x);
                    });
                }

                /* Force column display of all measures (not line) if chart type is grouped_columns */
                if (chartDef.type === CHART_TYPES.GROUPED_COLUMNS) {
                    chartDef.genericMeasures.forEach(function(m) {
                        m.displayType = 'column';
                    });
                }

                /* Force Auto mode for all axes if scatter equalScales mode is enabled */
                if (chartDef.type === CHART_TYPES.SCATTER && chartDef.scatterOptions && chartDef.scatterOptions.equalScales) {
                    chartDef.xAxisFormatting.customExtent = { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT };
                    ChartAxesUtils.setYAxisCustomExtent(chartDef.yAxesFormatting, { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT });
                    chartDef.xAxisFormatting.customExtent.editMode = ChartsStaticData.AUTO_EXTENT_MODE;
                }

                /* Map fixup */
                if (!chartDef.mapOptions) {
                    chartDef.mapOptions = {
                        tilesLayer: 'cartodb-positron'
                    };
                }
                if (!chartDef.mapGridOptions) {
                    chartDef.mapGridOptions = {};
                }
                if (_.isNil(chartDef.mapGridOptions.gridLonDeg)) {
                    chartDef.mapGridOptions.gridLonDeg = 0.5;
                }
                if (_.isNil(chartDef.mapGridOptions.gridLatDeg)) {
                    chartDef.mapGridOptions.gridLatDeg = 0.5;
                }

                // Gauge fixup (reset default min and max)
                if (
                    changedDefinition &&
                    (
                        changedDefinition.name === 'genericMeasures' &&
                            changedDefinition.nv[0] &&
                            changedDefinition.ov[0] &&
                            (
                                changedDefinition.nv[0].column !== changedDefinition.ov[0].column ||
                                changedDefinition.nv[0].function !== changedDefinition.ov[0].function
                            )
                        || changedDefinition.name === 'type'
                    )
                ) {
                    chartDef.gaugeOptions = {
                        ...chartDef.gaugeOptions,
                        min: { ...DEFAULT_GAUGE_MIN_MAX },
                        max: { ...DEFAULT_GAUGE_MIN_MAX }
                    };
                }

                // Treemap fixup
                if (chartDef.type === CHART_TYPES.TREEMAP
                    && chartDef.valuesInChartDisplayOptions
                    && !_.isNil(chartDef.valuesInChartDisplayOptions.displayValues)
                    && !!chartDef.genericMeasures
                    && chartDef.genericMeasures.length > 0
                    && chartDef.genericMeasures[0].valuesInChartDisplayOptions
                    && !_.isNil(chartDef.genericMeasures[0].valuesInChartDisplayOptions.displayValues)
                ) {
                    // for treemap, ensure that the global "values in chart" option is always enabled, and if it's explicitely set to false, set it to false on the measure
                    if (chartDef.valuesInChartDisplayOptions.displayValues === false) {
                        chartDef.genericMeasures[0].valuesInChartDisplayOptions.displayValues = false;
                    }
                    chartDef.valuesInChartDisplayOptions.displayValues = true;
                }

                // Pivot table fixup (disable hiding row headers when more than one row dimension)
                if (chartDef.type === CHART_TYPES.PIVOT_TABLE && changedDefinition &&
                    changedDefinition.name === 'yDimension' &&
                    chartDef.yDimension.length > 1
                ) {
                    chartDef.pivotTableOptions.tableFormatting.showRowHeaders = true;
                }

                //purging multiple y axes formatting from scattersMP
                if (chartDef.type !== CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                    chartDef.yAxesFormatting = chartDef.yAxesFormatting.filter(v => [ChartsStaticData.LEFT_AXIS_ID, ChartsStaticData.RIGHT_AXIS_ID].includes(v.id));
                }

                // fixup additional measures for stacked columns
                fixupStackedColumnsOptions(chartDef, theme);

                Logger.info('ChartSpec fixup done', chartDef);
            },

            fixupChart: function(chartDef, theme) {
                //  Echarts beta test
                chartDef.hasEchart = ChartFeatures.hasEChartsDefinition(chartDef.type); //  Transient, we don't need to register it in ChartDef, but it is used for dashboard/insights/editor
                chartDef.hasD3 = ChartFeatures.hasD3Definition(chartDef.type); // Transient, we don't need to register it in ChartDef, but it is used for dashboard/insights/editor

                if (ChartFeatures.hasEChartsDefinition(chartDef.type) && chartDef.displayWithECharts === undefined && $rootScope.featureFlagEnabled('echarts')) {
                    chartDef.displayWithECharts = $rootScope.featureFlagEnabled('echarts');
                }

                if (chartDef.displayWithEChartsByDefault === null || chartDef.displayWithEChartsByDefault === undefined) {
                    chartDef.displayWithEChartsByDefault = true;
                }

                // "Auto-migration"

                if (!chartDef.id) {
                    chartDef.id = window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16);
                }

                if (!chartDef.hexbinRadius) {
                    chartDef.hexbinRadius = 20;
                }
                if (!chartDef.hexbinRadiusMode) {
                    chartDef.hexbinRadiusMode = 'NUM_HEXAGONS';
                }
                if (!chartDef.hexbinNumber) {
                    chartDef.hexbinNumber = 20;
                }
                if (!chartDef.yAxisMode) {
                    chartDef.yAxisMode = 'NORMAL';
                }
                if (!chartDef.xAxisMode) {
                    chartDef.xAxisMode = 'NORMAL';
                }
                if (!chartDef.computeMode) {
                    chartDef.computeMode = 'NONE';
                }
                if (chartDef.smoothing === undefined) {
                    chartDef.smoothing = true;
                }
                if (!chartDef.strokeWidth) {
                    chartDef.strokeWidth = ChartsStaticData.DEFAULT_STROKE_WIDTH;
                }
                if (!chartDef.fillOpacity) {
                    chartDef.fillOpacity = ChartsStaticData.DEFAULT_FILL_OPACITY;
                }
                if (!chartDef.chartHeight) {
                    chartDef.chartHeight = 200;
                }
                if (chartDef.showLegend === undefined) {
                    chartDef.showLegend = true;
                }
                if (_.isNil(chartDef.legendFormatting)) {
                    chartDef.legendFormatting = {
                        ...DEFAULT_FONT_FORMATTING
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToLegend(theme, chartDef.legendFormatting);
                    }
                }
                if (chartDef.colorPaletteType === undefined) {
                    chartDef.colorPaletteType = 'LINEAR';
                }
                if (chartDef.colorMode === undefined) {
                    chartDef.colorMode = 'UNIQUE_SCALE';
                }

                fixupAxes(chartDef, theme);
                fixupReferenceLines(chartDef);
                fixupGaugeTargets(chartDef);

                fixupDynamicMeasure(chartDef.gaugeOptions.range);
                fixupDynamicMeasure(chartDef.gaugeOptions.min);
                fixupDynamicMeasure(chartDef.gaugeOptions.max);
                chartDef.gaugeOptions.targets?.map(dynMeasure => fixupDynamicMeasure(dynMeasure));
                chartDef.referenceLines?.map(dynMeasure => fixupDynamicMeasure(dynMeasure));

                if (chartDef.singleXAxis === undefined) {
                    chartDef.singleXAxis = true;
                }
                if (!chartDef.animationFrameDuration) {
                    chartDef.animationFrameDuration = 3000;
                }
                if (!chartDef.legendPlacement) {
                    chartDef.legendPlacement = 'OUTER_RIGHT';
                }

                if (!chartDef.gridlinesOptions) {
                    chartDef.gridlinesOptions = {
                        $default: true,
                        vertical: {
                            lineFormatting: {
                                type: 'FILLED',
                                size: '1',
                                color: '#d9d9d9'
                            }
                        },
                        horizontal: {
                            displayAxis: {},
                            lineFormatting: {
                                type: 'FILLED',
                                size: '1',
                                color: '#d9d9d9'
                            }
                        }
                    };
                }

                // Since DSS 11, the colored pivot table has been merged with pivot table
                if (chartDef.type === CHART_TYPES.PIVOT_TABLE && chartDef.variant === CHART_VARIANTS.colored) {
                    chartDef.variant = undefined;
                    if (chartDef.colorMeasure.length === 0 && chartDef.genericMeasures.length > 0) {
                        chartDef.colorMeasure.push(chartDef.genericMeasures[0]);
                    }
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions)) {
                    chartDef.valuesInChartDisplayOptions = {};
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.displayValues)) {
                    chartDef.valuesInChartDisplayOptions.displayValues = false;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.displayPieLabelsOrValues)) {
                    chartDef.valuesInChartDisplayOptions.displayPieLabelsOrValues = true;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.displayMode)) {
                    chartDef.valuesInChartDisplayOptions.displayMode = VALUES_DISPLAY_MODES.LABELS;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.overlappingStrategy)) {
                    chartDef.valuesInChartDisplayOptions.overlappingStrategy = ValuesInChartOverlappingStrategy.AUTO;
                }

                if (_.isNil(chartDef.valuesInChartDisplayOptions.textFormatting)) {
                    chartDef.valuesInChartDisplayOptions.textFormatting = DEFAULT_VALUES_DISPLAY_IN_CHART_TEXT_FORMATTING;
                }

                fixupStackedColumnsOptions(chartDef, theme);
                if (!_.isNil(chartDef.valuesInChartDisplayOptions.placementMode) && !_.isNil(chartDef.genericMeasures) && chartDef.genericMeasures.length > 0 && chartDef.valuesInChartDisplayOptions.placementMode !== ValuesInChartPlacementMode.AUTO) {
                    // the default value has been changed, we have to copy it on measures
                    chartDef.genericMeasures.forEach(m => {
                        if (!_.isNil(m.valuesInChartDisplayOptions)) {
                            m.valuesInChartDisplayOptions.placementMode = chartDef.valuesInChartDisplayOptions.placementMode;
                        }
                    });
                    chartDef.valuesInChartDisplayOptions.placementMode = null;
                }

                if (!_.isNil(chartDef.valuesInChartDisplayOptions.spacing) && !_.isNil(chartDef.genericMeasures) && chartDef.genericMeasures.length > 0) {
                    chartDef.genericMeasures.forEach(m => {
                        if (!_.isNil(m.valuesInChartDisplayOptions)) {
                            m.valuesInChartDisplayOptions.spacing = chartDef.valuesInChartDisplayOptions.spacing;
                        }
                    });

                    if (ChartFeatures.canDisplayTotalValues(chartDef)) {
                        chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.spacing = chartDef.valuesInChartDisplayOptions.spacing;
                    }
                    chartDef.valuesInChartDisplayOptions.spacing = null;
                }


                if (_.isNil(chartDef.pivotTableOptions) || _.isEmpty(chartDef.pivotTableOptions)) {
                    chartDef.pivotTableOptions = {
                        measureDisplayMode: ChartsStaticData.pivotTableMeasureDisplayMode.ROWS,
                        displayEmptyValues: false,
                        showSidebar: true,
                        areRowsExpandedByDefault: true,
                        areColumnExpandedByDefault: true,
                        rowIdByCustomExpandedStatus: {},
                        columnIdByCustomExpandedStatus: {},
                        columnIdByCustomWidth: {},
                        displayTotals: angular.copy(ChartsStaticData.defaultPivotDisplayTotals)
                    };
                }

                if (_.isNil(chartDef.pivotTableOptions.measureDisplayMode)) {
                    chartDef.pivotTableOptions.measureDisplayMode = ChartsStaticData.pivotTableMeasureDisplayMode.ROWS;
                }

                if (_.isNil(chartDef.pivotTableOptions.displayTotals)) {
                    chartDef.pivotTableOptions.displayTotals = angular.copy(ChartsStaticData.defaultPivotDisplayTotals);
                }

                if (_.isNil(chartDef.pivotTableOptions.showSidebar)) {
                    chartDef.pivotTableOptions.showSidebar = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting)) {
                    chartDef.pivotTableOptions.tableFormatting = {
                        showRowHeaders: true,
                        showRowMainHeaders: true,
                        rowMainHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        rowSubheaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        freezeRowHeaders: true,

                        showColumnHeaders: true,
                        showColumnMainHeaders: true,
                        columnMainHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        columnSubheaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },

                        values: { ...DEFAULT_FONT_FORMATTING },

                        // Obsolete since 13.4 but keeping for retrocompatibility
                        rowHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING },
                        columnHeaders: { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToTableFormatting(theme, chartDef.pivotTableOptions.tableFormatting);
                    }
                } else {
                    fixupPivotTableFormatting(chartDef.pivotTableOptions.tableFormatting, theme);
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showRowHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showRowHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.rowMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.rowMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.rowSubheaders)) {
                    chartDef.pivotTableOptions.tableFormatting.rowSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showColumnHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showColumnHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.showColumnMainHeaders = true;
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.columnMainHeaders)) {
                    chartDef.pivotTableOptions.tableFormatting.columnMainHeaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.columnSubheaders)) {
                    chartDef.pivotTableOptions.tableFormatting.columnSubheaders = { ...DEFAULT_TABLE_HEADER_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_TABLE_HEADER_FONT_FORMATTING.fontColor };
                }

                if (_.isNil(chartDef.pivotTableOptions.tableFormatting.values)) {
                    chartDef.pivotTableOptions.tableFormatting.values = { ...DEFAULT_FONT_FORMATTING, fontColor: theme ? theme.generalFormatting.fontColor : DEFAULT_FONT_FORMATTING.fontColor };
                }

                if (!chartDef.genericDimension0) {
                    chartDef.genericDimension0 = [];
                }
                if (!chartDef.genericDimension1) {
                    chartDef.genericDimension1 = [];
                }
                if (!chartDef.facetDimension) {
                    chartDef.facetDimension = [];
                }
                if (!chartDef.animationDimension) {
                    chartDef.animationDimension = [];
                }
                if (!chartDef.genericMeasures) {
                    chartDef.genericMeasures = [];
                }
                if (!chartDef.xDimension) {
                    chartDef.xDimension = [];
                }
                if (!chartDef.yDimension) {
                    chartDef.yDimension = [];
                }
                if (!chartDef.groupDimension) {
                    chartDef.groupDimension = [];
                }
                if (!chartDef.uaXDimension) {
                    chartDef.uaXDimension = [];
                }
                if (!chartDef.uaYDimension) {
                    chartDef.uaYDimension = [];
                }
                if (!chartDef.uaDimensionPair) {
                    chartDef.uaDimensionPair = [];
                }
                if (!chartDef.xMeasure) {
                    chartDef.xMeasure = [];
                }
                if (!chartDef.yMeasure) {
                    chartDef.yMeasure = [];
                }
                if (!chartDef.sizeMeasure) {
                    chartDef.sizeMeasure = [];
                }
                if (!chartDef.colorMeasure) {
                    chartDef.colorMeasure = [];
                }
                if (!chartDef.tooltipMeasures) {
                    chartDef.tooltipMeasures = [];
                }
                if (!chartDef.uaSize) {
                    chartDef.uaSize = [];
                }
                if (!chartDef.uaColor) {
                    chartDef.uaColor = [];
                }
                if (!chartDef.uaShape) {
                    chartDef.uaShape = [];
                }
                if (!chartDef.uaTooltip) {
                    chartDef.uaTooltip = [];
                }
                if (!chartDef.boxplotBreakdownDim) {
                    chartDef.boxplotBreakdownDim = [];
                }
                if (!chartDef.boxplotValue) {
                    chartDef.boxplotValue = [];
                }
                if (!chartDef.geometry) {
                    chartDef.geometry = [];
                }

                if (!chartDef.colorOptions) {
                    chartDef.colorOptions = ChartsStaticData.DEFAULT_COLOR_OPTIONS;
                }

                if (!chartDef.colorOptions.colorPalette) {
                    chartDef.colorOptions.colorPalette = theme.palettes.discrete ?? ChartsStaticData.DEFAULT_DISCRETE_PALETTE;
                }

                if (!chartDef.colorOptions.customPalette) {
                    chartDef.colorOptions.customPalette = ChartColorUtils.getDefaultCustomPalette();
                }

                if (!chartDef.colorOptions.ccScaleMode) {
                    chartDef.colorOptions.ccScaleMode = 'NORMAL';
                }
                if (!chartDef.colorOptions.paletteType) {
                    chartDef.colorOptions.paletteType = 'CONTINUOUS';
                }
                if (!chartDef.colorOptions.quantizationMode) {
                    chartDef.colorOptions.quantizationMode = 'NONE';
                }
                if (!chartDef.colorOptions.numQuantizeSteps) {
                    chartDef.colorOptions.numQuantizeSteps = 5;
                }
                if (!chartDef.colorOptions.paletteMiddleValue) {
                    chartDef.colorOptions.paletteMiddleValue = 0;
                }
                if (chartDef.colorOptions.paletteMiddleValue <= 0 && chartDef.colorOptions.paletteType === 'DIVERGING' && chartDef.colorOptions.ccScaleMode === 'LOG') {
                    chartDef.colorOptions.paletteMiddleValue = 1;
                }
                if (!chartDef.colorOptions.customColors) {
                    chartDef.colorOptions.customColors = {};
                }

                if (!chartDef.geoLayers) {
                    chartDef.geoLayers = [{ 'geometry': angular.copy(chartDef.geometry), 'uaColor': angular.copy(chartDef.uaColor), colorOptions: angular.copy(chartDef.colorOptions) }];
                }

                if (!chartDef.bubblesOptions) {
                    chartDef.bubblesOptions = {
                        defaultRadius: 5,
                        singleShape: 'FILLED_CIRCLE'
                    };
                }
                if (!chartDef.pieOptions) {
                    chartDef.pieOptions = {
                        donutHoleSize: 54
                    };
                }

                const defaultConnectPointsOpts = {
                    enabled: false,
                    lineFormatting: {
                        size: 1
                    }
                };

                if (!chartDef.scatterOptions) {
                    chartDef.scatterOptions = {
                        regression: {
                            show: false,
                            type: RegressionTypes.LINEAR,
                            lineFormatting: {
                                size: 1,
                                color: '#000'
                            },
                            labelPosition: 'INSIDE_END',
                            displayFormula: false,
                            textFormatting: {
                                ...DEFAULT_FONT_FORMATTING,
                                backgroundColor: '#D9D9D9BF'
                            }
                        }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToScatterOptions(theme, chartDef.scatterOptions);
                    }
                }

                if (!chartDef.scatterOptions.connectPoints) {
                    chartDef.scatterOptions.connectPoints = angular.copy(defaultConnectPointsOpts);
                }

                if (!chartDef.sankeyOptions) {
                    chartDef.sankeyOptions = {
                        linkColorVariant: 'GRADIENT',
                        curveness: 0.5
                    };
                }

                if (!chartDef.sankeyOptions.nodeLabelFormatting) {
                    chartDef.sankeyOptions = {
                        ...chartDef.sankeyOptions,
                        nodeLabelFormatting: {
                            ...DEFAULT_AUTO_FONT_FORMATTING,
                            hasBackground: false,
                            backgroundColor: '#D9D9D9BF'
                        }
                    };
                    if (theme) {
                        DSSVisualizationThemeUtils.applyToSankeyNodeLabels(theme, chartDef.sankeyOptions.nodeLabelFormatting);
                    }
                }

                if (!chartDef.scatterMPOptions) {
                    chartDef.scatterMPOptions = {
                        pairColorOptions: {
                            transparency: 0.75,
                            colorPalette: theme && theme.palettes ? theme.palettes.discrete : DKU_PALETTE_NAMES.THEME,
                            customPalette: ChartColorUtils.getDefaultCustomPalette(),
                            customColors: {}
                        }
                    };
                }

                if (!chartDef.scatterMPOptions.connectPoints) {
                    chartDef.scatterMPOptions.connectPoints = angular.copy(defaultConnectPointsOpts);
                }

                if (!chartDef.radarOptions) {
                    chartDef.radarOptions = {
                        filled: false,
                        polygonsSource: PolygonSources.MEASURES,
                        lineStyle: {
                            width: 2,
                            type: LineStyleTypes.SOLID
                        }
                    };
                }

                if (!chartDef.gaugeOptions) {
                    chartDef.gaugeOptions = {
                        min: null,
                        max: null,
                        displayPointer: false
                    };
                }

                if (!chartDef.scatterZoomOptions) {
                    chartDef.scatterZoomOptions = {
                        enabled: true,
                        persisted: true,
                        scale: [1, 1],
                        translate: [0, 0]
                    };
                }

                if (!chartDef.linesZoomOptions) {
                    chartDef.linesZoomOptions = {
                        enabled: true,
                        persisted: true,
                        displayBrush: true
                    };
                }

                if (chartDef.type === CHART_TYPES.WEBAPP) {
                    chartDef.webAppConfig = chartDef.webAppConfig || {};
                    chartDef.$loadedDesc = WebAppsService.getWebAppLoadedDesc(chartDef.webAppType) || {};
                    chartDef.$pluginDesc = WebAppsService.getOwnerPluginDesc(chartDef.webAppType);
                    chartDef.$pluginChartDesc = chartDef.$loadedDesc.desc.chart;
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.leftBarParams, chartDef.webAppConfig);
                    PluginConfigUtils.setDefaultValues(chartDef.$pluginChartDesc.topBarParams, chartDef.webAppConfig);
                } else {
                    chartDef.$loadedDesc = null;
                    chartDef.$pluginDesc = null;
                    chartDef.$pluginChartDesc = null;
                }


                // fixup display values options on measures
                chartDef.genericMeasures.forEach(measure => {
                    if (!measure.valuesInChartDisplayOptions) {
                        measure.valuesInChartDisplayOptions = {
                            displayValues: true,
                            textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                            additionalMeasures: [],
                            addDetails: false
                        };
                    }
                    if (!measure.valuesInChartDisplayOptions.additionalMeasures) {
                        measure.valuesInChartDisplayOptions.additionalMeasures = [];
                    }
                    measure.valuesInChartDisplayOptions.additionalMeasures.forEach(autocompleteGenericMeasure.bind(this, { contextualMenuMeasureType: 'valuesInChart', chartDef, defaultTextFormatting: measure.valuesInChartDisplayOptions.textFormatting, theme }));
                });

                const formattableMeasuresAndDimensions = [
                    'genericMeasures',
                    'colorMeasure',
                    'uaColor',
                    'uaSize',
                    'uaShape',
                    'boxplotValue',
                    'uaTooltip',
                    'tooltipMeasures',
                    'genericDimension0',
                    'genericDimension1',
                    'facetDimension',
                    'animationDimension',
                    'xDimension',
                    'yDimension',
                    'uaXDimension',
                    'uaYDimension',
                    'groupDimension'
                ];

                const getFormattableMeasureOrDimensionDefaultsByName = (measureOrDimensionName) => {
                    if (measureOrDimensionName === 'genericMeasures') {
                        return {
                            labelPosition: ChartsStaticData.LABEL_POSITIONS.BOTTOM.value,
                            showDisplayLabel: true
                        };
                    }
                    return {};
                };

                for (const measureOrDimension of formattableMeasuresAndDimensions) {
                    if (chartDef[measureOrDimension]) {
                        chartDef[measureOrDimension] = chartDef[measureOrDimension].map(measureOrDimensionDef => ({
                            ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS,
                            hideTrailingZeros: getDefaultTrailingZeros(measureOrDimensionDef),
                            ...getFormattableMeasureOrDimensionDefaultsByName(measureOrDimension),
                            ...measureOrDimensionDef
                        }));
                    }
                }
                $rootScope.$emit('chart-fixed-up');
                return chartDef;
            },

            defaultNewChart: function(themeToBeUsed) {
                const theme = DSSVisualizationThemeUtils.getThemeOrDefault(themeToBeUsed ? themeToBeUsed : $rootScope.appConfig.selectedDSSVisualizationTheme);
                // Correctly initialize the new chart so that the "blank" chart that is saved is correct.
                const newChart = this.fixupChart({
                    name: 'Chart',
                    type: CHART_TYPES.GROUPED_COLUMNS,
                    variant: 'normal',
                    showLegend: true
                }, theme);
                // Even if the theme is passed to fixup, some structures can be already defined so they need to be themed.
                DSSVisualizationThemeUtils.applyToChart({ chart: newChart, theme });
                return newChart;
            },

            newEmptyGeoPlaceholder: function(index) {
                return { geometry: [], uaColor: [], colorOptions: newColorOptions(index) };
            }
        };
        return svc;
    };

})();
