#! /bin/bash -e

echo "*********************************************"
echo "*      Dataiku Fleet Manager installer      *"
echo "*********************************************"

MYDIR=$(cd "$(dirname "$0")" && pwd -P)

# Initial sanity checks
if [ $(id -u) -eq 0 ]; then
  echo >&2 "[-] Installing or running FM as root is not supported."
  exit 1
fi

dssArch=$(cat "$MYDIR/dss-arch.txt")
case "$(uname)" in
	Linux)
		if [ "$dssArch" != "linux" ]; then
			echo >&2 "[-] This is the installer for the $dssArch version of FM."
			echo >&2 "[-] Please download the Linux version from www.dataiku.com."
			exit 1
		fi
		;;
	Darwin)
		if [ "$dssArch" != "osx" ]; then
			echo >&2 "[-] This is the installer for the $dssArch version of FM."
			echo >&2 "[-] Please download the macOS version from www.dataiku.com."
			exit 1
		fi
		;;
	*)
		echo >&2 "[-] Unsupported system: $(uname)"
		exit 1
		;;
esac

# Parse command line options
usage() {
	prog=$(basename "$0")
	echo >&2 "
*** Usage:
New installation:    $prog -d DATADIR -p BASE_PORT -c CLOUD [-l LICENSE_FILE] [-n]
Upgrade:             $prog -u -d DATADIR [-y] [-n]
Print this help:     $prog -h

  -d DATADIR : specify data directory
  -p BASE_PORT : specify base port
  -c CLOUD : Cloud provider to use {AWS,AZURE,GCP}
  -l LICENSE_FILE : specify license file
  -n : do not check for required dependencies
  -u : upgrade an existing data directory
  -y : do not prompt, assume answer 'yes'
  -h : prints this help
  "
	exit 1
}

DIP_HOME=
CLOUD=
LICENSE=
PORT=
noDeps=
upgrade=
yes=
while getopts c:d:p:l:hnuy OPT; do
	case "$OPT" in
	c)
		CLOUD="$OPTARG"
		;;
	d)
		DIP_HOME="$OPTARG"
		;;
	p)
		PORT="$OPTARG"
		;;
	l)
		LICENSE="$OPTARG"
		;;
	h)
		usage
		;;
	n)
		noDeps=1
		;;
	u)
		upgrade=1
		;;
	y)
		yes=1
		export DKU_MIGRATE_YES=1
		;;
	*)
		usage
		;;
	esac
done

if [ $OPTIND -le $# ]; then
	echo >&2 "[-] Bad usage: invalid argument : ${!OPTIND}"
	usage
fi

#
# Check arguments
#

if [ -n "$upgrade" -a \( -n "$PORT" -o -n "$LICENSE" -o -n "$CLOUD" \) ]; then
    echo "[-] Bad usage: cannot specify port, cloud or license file while upgrading" >&2
	usage
fi

if [ -z "$upgrade" -a -z "$CLOUD" ]; then
	echo "[-] Bad usage : -c CLOUD argument is mandatory" >&2
	usage
fi

if [ -z "$upgrade" -a "$CLOUD" != "AWS" -a "$CLOUD" != "AZURE" -a "$CLOUD" != "GCP" ]; then
  echo "[-] Bad usage : -c CLOUD argument must be either AWS or AZURE or GCP" >&2
	usage
fi

if [ -z "$DIP_HOME" ]; then
	echo "[-] Bad usage : -d DATADIR argument is mandatory" >&2
	usage
fi
# Sanity check, strip trailing / if any
case "$DIP_HOME" in
	. | ./ | .. | ../ | / )
		echo "[-] Invalid value for DATADIR : $DIP_HOME" >&2
		usage
		;;
	*/ )
		DIP_HOME=$(echo "$DIP_HOME" | sed 's_/$__')
		;;
esac

if [ -n "$upgrade" ]; then
	if [ -d "$DIP_HOME" -a -f "$DIP_HOME"/dss-version.json ]; then
		echo "[+] $DIP_HOME already exists and is a FM installation"
	else
		echo >&2 "[-] $DIP_HOME does not appear to be a valid FM installation"
		exit 1
	fi
else
	if [ -z "$PORT" ]; then
		echo "Bad usage : -p BASE_PORT argument is mandatory" >&2
		usage
	fi

	if [ -f "$DIP_HOME"/FM-INITIAL-INSTALL-IN-PROGRESS.txt ]; then
		echo "[!] *****************************************************" >&2
		echo "[!] $DIP_HOME contains a previously failed install of FM" >&2
		echo "[!] Moving it out of the way and proceeding" >&2
		echo "[!] *****************************************************" >&2
		mv "$DIP_HOME" "$DIP_HOME.fm_failed_install.`date +%Y%m%d-%H%M%S`"
	fi

	if [ -f "$DIP_HOME"/config/dip.properties ]; then
	    echo "[-] $DIP_HOME appears to already contain a FM installation" >&2
	    echo "[-] If you want to upgrade it, rerun this installer with the -u flag" >&2
        exit 1
    fi
	if [ ! -e "$DIP_HOME" ]; then
		echo "[+] Creating data directory: $DIP_HOME" >&2
		mkdir -p "$DIP_HOME"
		touch "$DIP_HOME"/FM-INITIAL-INSTALL-IN-PROGRESS.txt

	elif [ -d "$DIP_HOME" -a -z "$(ls "$DIP_HOME" 2>/dev/null)" ]; then
		echo "[+] Using data directory: $DIP_HOME" >&2
		touch "$DIP_HOME"/FM-INITIAL-INSTALL-IN-PROGRESS.txt
	else
		echo "[-] Directory $DIP_HOME already exists, but is not empty. Aborting !" >&2
		exit 1
	fi
fi

DIP_HOME=$(cd "$DIP_HOME" && pwd -P)
umask 22

# Start logging script output
mkdir -p "$DIP_HOME"/run
logFile="$DIP_HOME"/run/install.log
echo "[+] Saving installation log to $logFile"
cat >>"$logFile" <<EOF
*********************************************************
Dataiku FM installer: $(date +%Y/%m/%d-%H:%M:%S)
Command line: $0 $@
Version: $(cat "$MYDIR"/dss-version.json)
DIP_HOME: $DIP_HOME

EOF

# Start block | tee -a "$logFile"
set -o pipefail
(
set +o pipefail

#
# Check for running instance
#
if "$DIP_HOME"/bin/fm status >/dev/null 2>/dev/null; then
	echo "[!] *********************************************************"
	echo "[!] There seem to be an already running instance of FM"
	echo "[!] using this data directory:"
	"$DIP_HOME"/bin/fm status
	echo "[!]"
	echo "[!] It is STRONGLY RECOMMENDED to stop it before upgrade."
	echo "[!] If you continue, you will not be able to use bin/fm stop"
	echo "[!] and will need to kill the processes manually"
	if [ -z "$yes" ]; then
		echo "[!] Press Enter to continue at your own risk, Ctrl+C to abort"
   		read
   	else
   		echo "[!] Non-interactive mode, continuing anyway"
   	fi
fi

#
# Check for DATADIR inside INSTALLDIR
#
# this check is enough because both DIP_HOME and MYDIR have been cleaned with (cd ... && pwd -P)
if [[ "$DIP_HOME" == "$MYDIR"/* ]]; then
	echo "[!] *********************************************************"
	echo "[!] Warning: the data directory you specified:"
	echo "[!]     $DIP_HOME"
	echo "[!] appears to be a subdirectory of the installation directory:"
	echo "[!]     $MYDIR"
	echo "[!] This is NOT RECOMMENDED for production environment as it complexifies subsequent FM upgrades."
	if [ -z "$yes" ]; then
		echo "[!] Press Enter to continue, Ctrl+C to abort"
		read
	else
		echo "[!] Non-interactive mode, continuing anyway"
	fi
fi

#
# Check system dependencies
#
deps_flags=
if [ -n "$DKUJAVABIN" ]; then
	echo "[+] Using custom Java environment \"$DKUJAVABIN\""
	javaBin="$DKUJAVABIN"
	deps_flags="$deps_flags -without-java"
elif javaBin=$("$MYDIR"/scripts/_find-java.sh); then
	deps_flags="$deps_flags -without-java"
else
	javaBin=
fi
if [ -n "$DKUPYTHONBIN" ]; then
	echo "[+] Using custom Python environment \"$DKUPYTHONBIN\""
	deps_flags="$deps_flags -without-python"
fi

if [ -z "$noDeps" ]; then
	echo "[+] Checking required dependencies"
	"$MYDIR"/scripts/install/install-deps.sh -check $deps_flags || {
		echo >&2 "
[-] Dependency check failed
[-] You can install required dependencies with:
[-]    sudo -i \"$MYDIR/scripts/install/install-deps.sh\"$deps_flags
[-] You can also disable this check with the -n installer flag
"
		exit 1
	}
fi
if [ -z "$javaBin" ]; then
	echo "[-] Could not find suitable Java installation. Aborting!" >&2
	exit 1
elif "$javaBin" -version 2>&1 >/dev/null | grep -qE '^(java|openjdk) version "1\.7\.'; then
	echo "[!] *****************************************************" >&2
	echo "[!] WARNING: FM is not supported on Java 7             " >&2
	echo "[!] It is strongly advised to upgrade to Java 8 or later " >&2
	echo "[!] *****************************************************" >&2
fi

function precompile_python {
	pythonBin="$1"
	precompilePackages="$2"

	if [ ! -w "$MYDIR" ]; then
		echo "[-] Read-only installation directory $MYDIR, skipping Python precompilation"
		return
	fi

	echo "[+] Precompiling Dataiku Python code"
	"$pythonBin" -m compileall -q "$MYDIR"/python ||
		echo "[-] Error precompiling Dataiku Python code (ignored)"
}

if [ -n "$upgrade" ]
then
	######################################################################################
	#                             Upgrade
	######################################################################################

	# Create or upgrade Python environment unless overridden
	if [ -n "$DKUPYTHONBIN" ]; then
		pythonBin="$DKUPYTHONBIN"

	elif [ -d "$DIP_HOME"/pyenv ]; then
		echo "[+] Migrating Python environment"
		"$MYDIR"/scripts/_install-pyenv.sh --rebuild-if-needed -f -u "$DIP_HOME"
		pythonBin="$DIP_HOME/bin/python"
		precompile_python "$pythonBin" "yes"

	else
		echo "[+] Initializing Python environment"
		"$MYDIR"/scripts/_install-pyenv.sh -f "$DIP_HOME"
		pythonBin="$DIP_HOME/bin/python"
		precompile_python "$pythonBin" "yes"
	fi

	# Perform migration
	echo "[+] Migrating data directory"
	PYTHONIOENCODING=UTF-8 PYTHONUNBUFFERED=1 DKUINSTALLDIR="$MYDIR" DKUJAVABIN="$javaBin" \
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/migrate_auto.py "$DIP_HOME"

	NODETYPE=$("$pythonBin" "$MYDIR"/scripts/dkuinstall/install_config.py -d "$DIP_HOME" -get nodetype)
	echo "Node type : $NODETYPE"
else
	######################################################################################
	#                             Fresh install
	######################################################################################

	echo "[+] Installation starting"
	cp -p "$MYDIR"/dss-version.json "$DIP_HOME"/
	chmod u+w "$DIP_HOME"/dss-version.json   # should not be necessary
	mkdir -p "$DIP_HOME"/bin "$DIP_HOME"/config

	# Create empty env-site.sh
	cat <<EOF >"$DIP_HOME"/bin/env-site.sh
# This file is sourced last by FM startup scripts
# You can add local customizations to it
EOF

	if [ -n "$LICENSE" ]; then
		echo "[+] Installing license file"
		cp -p "$LICENSE" "$DIP_HOME"/config/license.json
	fi

	# Create Python environment unless overridden
	if [ -n "$DKUPYTHONBIN" ]; then
		pythonBin="$DKUPYTHONBIN"
		precompile_python "$pythonBin" ""

	else
		echo "[+] Initializing Python environment"
		"$MYDIR"/scripts/_install-pyenv.sh -f "$DIP_HOME"
		pythonBin="$DIP_HOME"/bin/python
		precompile_python "$pythonBin" "yes"
	fi

	DKUINSTALLDIR="$MYDIR" DKUJAVABIN="$javaBin" \
		"$pythonBin" "$MYDIR"/scripts/dkuinstall/install_fm.py "$DIP_HOME" $PORT "$CLOUD"
fi


"$DIP_HOME"/bin/fmadmin -noLog regenerate-config

# Final steps that we always do
mkdir -p "$DIP_HOME"/lib/jdbc  "$DIP_HOME"/lib/java "$DIP_HOME"/lib/python

rm -f "$DIP_HOME"/FM-INITIAL-INSTALL-IN-PROGRESS.txt

echo "$DIP_HOME" > "$DIP_HOME"/install-support/expected-dip-home.txt

echo "***************************************************************"
echo "* Installation complete."
echo "* Edit '$DIP_HOME/config/user-data.json' to configure"
echo "* the connection to the PostgreSQL database, then"
if [ ! -n "$upgrade" ]; then
  echo "* you need to initialize your setup before the first boot using:"
  echo "*         '$DIP_HOME/bin/fmadmin initial-setup USER_ID PASSWORD'"
  echo "* then you can"
fi
echo "* start FM using:"
echo "*         '$DIP_HOME/bin/fm start'"
echo "***************************************************************"

) 2>&1 | tee -a "$logFile"
