(function() {
    'use strict';

    const app = angular.module("dataiku.fm.virtualnetworks", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs"]);

    app.controller('DeleteVirtualNetworkConfirmDialogController', function($scope) {
        $scope.confirm = function() {
            if($scope.acceptDeferred) {
                $scope.acceptDeferred.resolve($scope.options);
            }
            $scope.acceptDeferred = null;
            $scope.dismiss();
        };
        $scope.cancel = function() {
            if($scope.acceptDeferred) {
                $scope.acceptDeferred.reject();
            }
            $scope.acceptDeferred = null;
            $scope.dismiss();
        };
        $scope.$on("$destroy", function() {
            if ($scope.acceptDeferred) {
                $scope.acceptDeferred.reject();
            }
            $scope.acceptDeferred = null;
        });
    });

    app.controller("VirtualNetworksListController", function ($scope, $state, $location, $q, FMAPI, Dialogs, CreateModalFromTemplate, FutureProgressModal, WT1, TaggingService) {
        $scope.activeProjectTagColor = TaggingService.getDefaultColor;
        $scope.refreshVirtualNetworkList = function() {
            FMAPI.virtualNetworks.list().success(function(data) {
                $scope.virtualNetworks = data;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.deleteVirtualNetwork = function(virtualNetwork) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/app/virtual-networks/virtualnetwork-confirm-delete.html", $scope, "DeleteVirtualNetworkConfirmDialogController", function(newScope) {
                newScope.acceptDeferred = deferred;
                newScope.virtualNetwork = virtualNetwork;
                newScope.options = {deleteAutogeneratedSecurityGroups: true, deleteAutogeneratedPeering: true};
            });
            deferred.promise.then(function(options) {
                $scope.deleteVirtualNetworkNoConfirmation(virtualNetwork, options);
            });
        };

        $scope.deleteVirtualNetworkNoConfirmation = function(virtualNetwork, options) {
            WT1.event("fm-virtual-networks-delete", {});

            FMAPI.virtualNetworks.startDelete(virtualNetwork.id, options).success(function(response) {
                FutureProgressModal.show($scope, response, "Deleting virtual network", undefined, true, 'static', true).then(function(result) {
                    if (result.statusMessages.anyMessage) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Deletion result", result.statusMessages);
                    }
                    $scope.refreshVirtualNetworkList();
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.refreshVirtualNetworkList();
    });

    app.controller("VirtualNetworkCreationController", function($scope, $state, FMAPI, Logger, FutureProgressModal, WT1) {
        $scope.newProtoVN = {
            cloudTags: []
        };
        $scope.addAwsSecurityGroup = function() {
            if (!$scope.newProtoVN.awsSecurityGroups) {
                $scope.newProtoVN.awsSecurityGroups = [];
            }
            $scope.newProtoVN.awsSecurityGroups.push("");
        };

        $scope.removeAwsSecurityGroup = function(index) {
            if ($scope.newProtoVN.awsSecurityGroups) {
                $scope.newProtoVN.awsSecurityGroups.splice(index, 1);
            }
        };
        $scope.addGcpNetworkTag = function() {
            if (!$scope.newProtoVN.gcpNetworkTags) {
                $scope.newProtoVN.gcpNetworkTags = [];
            }
            $scope.newProtoVN.gcpNetworkTags.push("");
        };

        $scope.removeGcpNetworkTag = function(index) {
            if ($scope.newProtoVN.gcpNetworkTags) {
                $scope.newProtoVN.gcpNetworkTags.splice(index, 1);
            }
        };

        $scope.accountChanged  = function () {
            if ($scope.newProtoVN.accountId === 'main') {
                $scope.mainAccountChange()
                return;
            }
            $scope.newProtoVN.awsVpcId = "";
            $scope.newProtoVN.awsSubnetId = "";
            $scope.newProtoVN.azureVnId  = "";
            $scope.newProtoVN.azureSubnetId = "";
            $scope.newProtoVN.gcpNetwork = "";
            $scope.newProtoVN.gcpSubnetwork = "";
        }

        $scope.mainAccountChange  = function () {
            $scope.newProtoVN.awsVpcId = $scope.defaultValues.awsVpcId;
            $scope.newProtoVN.awsSubnetId = $scope.defaultValues.awsSubnetId;
            $scope.newProtoVN.azureVnId  = $scope.defaultValues.azureVnId;
            $scope.newProtoVN.azureSubnetId = $scope.defaultValues.azureSubnetId;
            $scope.newProtoVN.gcpNetwork = $scope.defaultValues.gcpNetwork;
            $scope.newProtoVN.gcpSubnetwork = $scope.defaultValues.gcpSubnetwork;
        }

        $scope.create = function() {
            WT1.event("fm-virtual-networks-create", {});
    
            // Pre-process newProtoVN before sending it to the backend
            if ($scope.newProtoVN.awsSecurityGroups) {
                $scope.newProtoVN.awsSecurityGroups = $scope.newProtoVN.awsSecurityGroups.filter(item => item.length > 0)
            }
            if ($scope.newProtoVN.gcpNetworkTags) {
                $scope.newProtoVN.gcpNetworkTags = $scope.newProtoVN.gcpNetworkTags.filter(item => item.length > 0)
            }
            // Do the actual job
            FMAPI.virtualNetworks.startCreate($scope.newProtoVN).success(function(response) {
                FutureProgressModal.show($scope, response, "Provisioning virtual network").then((result) => {
                    $state.go("virtualnetworks.virtualnetwork.dashboard", {virtualNetworkId: result.id});
                });
            }).error(setErrorInScope.bind($scope));
        };

        FMAPI.accounts.getCreationData().success(function(data) {
            $scope.creationData = data;
        });
        // Initialize
        FMAPI.virtualNetworks.getDefaultValues().success(function(protoVN) {
            $scope.defaultValues = protoVN;
            Object.assign($scope.newProtoVN, protoVN);
        }).error(setErrorInScope.bind($scope));
    });

    app.controller("VirtualNetworkBaseController", function($scope, $state, $stateParams, FMAPI) {
        $scope.SharedState = $scope.SharedState || {virtualNetworkSettingsFormIsValid : false, virtualNetworkSettingsFormIsDirty : false};

        $scope.labelFromItemInArray = function(itemId, array, idField /* default = "id" */, labelField /* default = "label" */) {
            const _idField = idField ? idField : "id";
            const _labelField = labelField ? labelField : "label";
            const item = array.find(s => s[_idField] === itemId);
            return item ? item[_labelField] : "";
        };

        $scope.refresh = function() {
            FMAPI.virtualNetworks.get($stateParams.virtualNetworkId).success(function(virtualNetwork) {
                $scope.virtualNetwork = virtualNetwork;
                updateAzureURLs(virtualNetwork);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.onSaveButtonClicked = function() {
            $scope.$broadcast("saveButtonClicked")
        };

        $scope.$on("dirtyChanged", function(event, dirtyValue) {
            $scope.vnEditionState.dirty = dirtyValue;
        });


        $scope.$on("dirtyChanged", function(event, dirtyValue) {
            $scope.vnEditionState.dirty = dirtyValue;
        });

        $scope.dnsStrategies = [{id: "NONE", label: "Don't assign domain names (IP only)"}];
        if ($scope.appConfig.cloud === "AWS") {
            $scope.dnsStrategies.push({id: "VN_SPECIFIC_CLOUD_DNS_SERVICE", label: "Assign a Route53 domain name that you manage"});
            if ($scope.appConfig.tenancy === "MULTITENANT") {
                $scope.dnsStrategies.push({id: "FM_MANAGED_CLOUD_DNS_SERVICE", label: "Assign a Dataiku-managed domain name"});
            }
        } else if ($scope.appConfig.cloud === "AZURE") {
            $scope.dnsStrategies.push({id: "VN_SPECIFIC_CLOUD_DNS_SERVICE", label: "Assign a Azure DNS domain name that you manage"});
            if ($scope.appConfig.tenancy === "MULTITENANT") {
                $scope.dnsStrategies.push({id: "FM_MANAGED_CLOUD_DNS_SERVICE", label: "Assign a Dataiku-managed domain name"});
            }
        } else if ($scope.appConfig.cloud === "GCP") {
            $scope.dnsStrategies.push({id: "VN_SPECIFIC_CLOUD_DNS_SERVICE", label: "Assign a GCP DNS domain name that you manage"});
            if ($scope.appConfig.tenancy === "MULTITENANT") {
                $scope.dnsStrategies.push({id: "FM_MANAGED_CLOUD_DNS_SERVICE", label: "Assign a Dataiku-managed domain name"});
            }
        }
        $scope.httpsStrategies = [
            {id: 'NONE', label: 'None (HTTP only)'},
            {id: 'SELF_SIGNED', label: 'Self-signed certificates'},
            {id: 'CUSTOM_CERTIFICATE', label: 'Enter a certificate/key for each instance'},
            {id: 'LETSENCRYPT', label: 'Generate certificates using Let\'s Encrypt'}
        ];
        $scope.httpStrategies = [
            {id: 'DISABLE', label: 'HTTP port is disabled'},
            {id: 'REDIRECT', label: 'HTTP port redirects to HTTPS'}
        ];
        $scope.letsEncryptHttpStrategies = [
            {id: 'REDIRECT', label: 'HTTP port redirects to HTTPS'}
        ];
        $scope.vnEditionState = {
            dirty: false
        };

        function updateAzureURLs(virtualNetwork) {
            if ($scope.appConfig.cloud === 'AZURE') {
                // We need the tenandID to build the link to the cloud resources
                FMAPI.tenant.getCloudCredentials().success(function(cloudCredentials) {
                    const azureTenantURL = `https://app.azure.com/${virtualNetwork.azureTenantId}`;
                    const azureSubscriptionURL = `${azureTenantURL}/subscriptions/${virtualNetwork.azureSubscription}`;
                    $scope.azureVirtualNetworkName = virtualNetwork.azureVnId.substring(virtualNetwork.azureVnId.lastIndexOf('/') + 1);
                    $scope.azureResourceGroupURL = `${azureSubscriptionURL}/resourceGroups/${virtualNetwork.azureRgNameForCreatedResources || virtualNetwork.azureRgName}`;
                    $scope.azureVirtualNetworkURL = `${azureTenantURL}${virtualNetwork.azureVnId}`;
                    $scope.azureSubnetURL = `${$scope.azureVirtualNetworkURL}/subnets`;
                }).error(setErrorInScope.bind($scope));
            }
        }
    });

    app.controller("VirtualNetworkDashboardController", function ($scope, TaggingService) {
        $scope.activeProjectTagColor = TaggingService.getDefaultColor;
        $scope.refresh();
    });

    app.controller("VirtualNetworkSettingsController", function($scope, $state, $stateParams, FMAPI, Logger, ActivityIndicator, FeatureFlagsService) {
        $scope.refresh = function() {
            FMAPI.virtualNetworks.get($stateParams.virtualNetworkId).success(function(virtualNetwork) {
                $scope.virtualNetwork = virtualNetwork;
                // eslint-disable-next-line no-undef
                $scope.editedVN = dkuDeepCopy($scope.virtualNetwork, filterDollarKey);
                $scope.$emit('vnUpdated', virtualNetwork);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.$on("saveButtonClicked", function() {
            FMAPI.virtualNetworks.save($scope.editedVN).success(function(virtualNetwork) {
                resetErrorInScope($scope);
                $scope.virtualNetwork = virtualNetwork;
                // eslint-disable-next-line no-undef
                $scope.editedVN = dkuDeepCopy($scope.virtualNetwork, filterDollarKey);
                $scope.$emit('vnUpdated', virtualNetwork);
                $scope.$emit('dirtyChanged', false);
                ActivityIndicator.success("Saved");
            }).error(setErrorInScope.bind($scope));
        });

        $scope.$watch("editedVN", function(nv) {
            if (nv) {
                if ($scope.editedVN.httpsStrategy === "LETSENCRYPT") {
                    $scope.editedVN.httpStrategy = "REDIRECT";
                }
                const dirtyValue = !angular.equals($scope.editedVN, $scope.virtualNetwork);
                $scope.$emit('dirtyChanged', dirtyValue);
            }
        }, true);

        $scope.$watch('virtualNetworkSettingsForm.$valid', function(newVal) {
            $scope.SharedState.virtualNetworkSettingsFormIsValid = newVal;
        })

        $scope.$watch('virtualNetworkSettingsForm.$dirty', function(newVal) {
            $scope.SharedState.virtualNetworkSettingsFormIsDirty = newVal;
        })

        $scope.$on("dirtyChanged", function(event, dirtyValue) {
            $scope.SharedState.virtualNetworkSettingsFormIsDirty = dirtyValue;
        });

        // Initialize
        $scope.refresh();
    });

}());
