(function() {
    'use strict';

    const app = angular.module('dataiku.ml.explainability');

    app.factory('WhatIfFeaturesSortingFactory', function() {
        /**
         * The purpose of this factory is to facilitate the use of the `filtered-multi-select-rows` directive across
         * the "What If?"-related pages.
         */
        class FeaturesSortingService {
            constructor() {
                this._selection = {};
                this._perFeatureIndex = null;
                this._perFeatureDistributionType = null;
                this._perFeatureImportance = null;
            }

            cleanSelection() {
                const keysToKeep = ['orderQuery', 'filterParams', 'orderReversed'];
                // remove other automatically generated attributes
                Object.keys(this._selection)
                    .filter(key => !keysToKeep.includes(key))
                    .forEach(key => { delete this._selection[key]; });
                // if orderQuery uses a non-standard attribute, reset it
                if (this._selection.orderQuery && !['name', 'index', 'importance', 'distributionType'].includes(this._selection.orderQuery)) {
                    angular.extend(this._selection, this.getDefaultSelection()); // reset the
                }
                return this._selection;
            }

            setPerFeatureIndex(schemaColumns) { // `schemaColumns` can be found in `splitDesc` response's `data.schema.columns`
                this._perFeatureIndex = Object.fromEntries(
                    schemaColumns.map((x, i) => [x.name, i])
                );
            }

            setPerFeatureDistributionType(distributions) { // `distributions` come from `FeaturesDistributionAdapter`
                this._perFeatureDistributionType = Object.fromEntries(
                    distributions.getNames().map(name => [name, distributions.getSingle(name).type])
                );
            }

            setPerFeatureImportance(perFeatureImportance) { // warning: column importance is not always available
                this._perFeatureImportance = perFeatureImportance;
            }

            hasIndex() {
                return !!this._perFeatureIndex;
            }

            hasDistributionType() {
                return !!this._perFeatureDistributionType;
            }

            hasImportance() {
                return !!this._perFeatureImportance;
            }

            enrichWithSortingAttributes(features) {
                // `features` should be an array of objects that's linked to a `filtered-multi-select-rows` directive.
                // The features should contain at least a `name` attribute.
                // This function will add additional attributes in the array's objects for sorting purposes.
                if (this.hasIndex()) {
                    features.forEach(feature => feature.index = this._perFeatureIndex[feature.name]);
                }
                if (this.hasDistributionType()) {
                    features.forEach(feature => feature.distributionType = this._perFeatureDistributionType[feature.name]);
                }
                if (this.hasImportance()) {
                    features.forEach(feature => feature.importance = this._perFeatureImportance[feature.name]);
                }
            }

            getSortOptions() { // these should be used as options of the `filtered-multi-select-rows`'s associated `select`
                return [{ value: 'name', label: 'Name' }, { value: 'index', label: "Dataset" }, { value: 'distributionType', label: "Type" }]
                    .concat(this.hasImportance() ? [{ value: 'importance', label: "Importance" }] : []);
            }

            getDefaultSelection() { // this returns a valid `$scope.selection` for `filtered-multi-select-rows` to interact with
                return {
                    orderQuery: this.hasImportance() ? 'importance' : 'index',
                    filterParams: {
                        userQueryTargets: ["name"]
                    },
                    orderReversed: this.hasImportance()
                }
            }

            updateSortOrderAfterOptionChange(nv, ov) {
                // By default, sort order should be reversed when sorting by importance.
                // This function can be used as a watcher callback to handle this logic.
                if (ov == "importance" && nv != "importance") {
                    this._selection.orderReversed = false;
                } else if (ov != "importance" && nv == "importance") {
                    this._selection.orderReversed = true;
                }
            }
        }
        return {
            init: () => new FeaturesSortingService()
        };
    });

})();
