(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('PieChartDrawer', PieChartDrawer);

    // (!) This service previously was in static/dataiku/js/simple_report/pies/pie.js
    function PieChartDrawer(PieChartUtils, d3Utils, ChartLabels, VALUES_DISPLAY_MODES, ChartHierarchyDimension, SVGUtils, ChartDrilldown) {
        return function(g, chartDef, chartBase, facetData, numberFormattingOptions) {

            const displayValuesOrLabels = chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayPieLabelsOrValues;

            let outerR = Math.min(chartBase.vizWidth, chartBase.vizHeight) / 2;
            let r = displayValuesOrLabels ? (outerR - 40) : outerR;
            const center = { 'x': chartBase.vizWidth / 2, 'y': chartBase.vizHeight / 2 };

            const viz = g.selectAll('g.viz').data([null]);
            viz.enter().append('g').attr('class', 'viz');
            viz.attr('transform', 'translate(' + center.x + ',' + center.y + ')');

            const arc = d3.svg.arc().outerRadius(r);

            const wrappers = viz.selectAll('g.wrapper')
                .data(facetData.pie || [], function(d) {
                    return d.data.color + '-' + d.data.measure;
                });

            const newWrappers = wrappers.enter().append('g').attr('class', 'wrapper');


            const drawHole = function() {
                //--------------- Donut hole if necessary -----------------

                const hole = g.selectAll('circle.hole').data([null]);
                if (chartDef.variant === 'donut') {
                    let holeRadius = r / 2;
                    if (chartDef.pieOptions && chartDef.pieOptions.donutHoleSize && 0 < chartDef.pieOptions.donutHoleSize && chartDef.pieOptions.donutHoleSize < 100) {
                        holeRadius = r * chartDef.pieOptions.donutHoleSize / 100;
                    }
                    hole.enter().append('circle').attr('class', 'hole')
                        .attr('cx', center.x)
                        .attr('cy', center.y)
                        .attr('r', holeRadius)
                        .style('fill', 'white');
                    hole.attr('r', holeRadius);
                } else {
                    hole.remove();
                }
            };

            //--------------- Draw labels -----------------
            if (displayValuesOrLabels) {
                //  Retrieves min/max/numValues for color dimension to fit current implementation in tooltips
                const colorDimensionMin = chartBase.chartData.getMinValue('color');
                const colorDimensionMax = chartBase.chartData.getMaxValue('color');
                const colorDimensionNumValues = chartBase.chartData.getNumValues('color');

                const outerArc = d3.svg.arc()
                    .innerRadius(r)
                    .outerRadius(outerR);

                const svgBoundingBox = $(g.node()).closest('svg').get(0).getBoundingClientRect();
                let maxOverflow = 0;

                const transformLabels = function(sel) {
                    return sel.attr('transform', function(d) {
                        return 'translate(' + outerArc.centroid(d) + ')';
                    });
                };

                newWrappers.append('text').attr('class', 'pie-chart-label')
                    .call(transformLabels);

                const texts = wrappers.select('text.pie-chart-label');

                texts
                    .text(function(d) {
                        let text = '';
                        switch (chartDef.valuesInChartDisplayOptions.displayMode) {
                            case VALUES_DISPLAY_MODES.VALUES_AND_LABELS:
                                /*
                                 *  This formatting is set to be the same used for tooltips but is not relevant
                                 *  @TODO : https://app.shortcut.com/dataiku/story/110162/enhance-formatting-for-dimensions-and-measures-in-auto-mode
                                 */
                                text += ChartLabels.getFormattedLabel(chartBase.chartData.getAxisLabels('color')[d.data.color], numberFormattingOptions, colorDimensionMin, colorDimensionMax, colorDimensionNumValues);
                                text += ' - ' + chartBase.measureFormatters[d.data.measure](d.data.value);
                                break;
                            case VALUES_DISPLAY_MODES.LABELS:
                                text += ChartLabels.getFormattedLabel(chartBase.chartData.getAxisLabels('color')[d.data.color], numberFormattingOptions, colorDimensionMin, colorDimensionMax, colorDimensionNumValues);
                                break;
                            case VALUES_DISPLAY_MODES.VALUES:
                                text += chartBase.measureFormatters[d.data.measure](d.data.value);
                                break;
                        }

                        return text;
                    })
                    .attr('text-anchor', function(d) {
                        const middleAngle = d.startAngle + (d.endAngle - d.startAngle) / 2;
                        if (middleAngle > Math.PI) {
                            return 'end';
                        } else {
                            return 'start';
                        }
                    })
                    .attr('fill', '#666')
                    .style('display', function(d) {
                        return d.data.value > 0 ? 'block' : 'none';
                    })
                    .interrupt('updateLabels')
                    .transition('updateLabels')
                    .call(transformLabels)
                    .call(d3Utils.endAll, function() {
                        if (maxOverflow > 0) {
                            outerR = Math.max(75, outerR - maxOverflow);
                            r = outerR - 40;
                            arc.outerRadius(r);
                            outerArc.innerRadius(r)
                                .outerRadius(outerR);
                        }
                        texts.call(transformLabels);
                        PieChartUtils.hideOverlappingLabels(wrappers[0], (facetData.pie || []).map(function(d) {
                            return d.data.value;
                        }), facetData.total);
                        drawHole();
                    }, function(d, i) {
                        if (d.data.value === 0) {
                            return;
                        }
                        const boundingBox = this.getBoundingClientRect();
                        maxOverflow = Math.max(maxOverflow, svgBoundingBox.left - boundingBox.left, boundingBox.right - svgBoundingBox.right);
                    });
            } else {
                drawHole();
            }


            //--------------- Draw arcs -----------------

            const noData = viz.selectAll('text.no-data').data([null]);
            noData.enter().append('text')
                .attr('class', 'no-data')
                .attr('text-anchor', 'middle')
                .style('pointer-events', 'none')
                .style('font-size', '20px')
                .text('No data');


            newWrappers.append('path').attr('class', 'slice')
                .each(function(d) {
                    this._current = d;
                })
                .attr('fill', function(d) {
                    return chartBase.colorScale(d.data.color + d.data.measure);
                })
                .attr('opacity', chartDef.colorOptions.transparency)
                .each(function(d) {
                    chartBase.tooltips.registerEl(this, {
                        measure: d.data.measure,
                        color: d.data.color,
                        animation: d.data.animation,
                        facet: d.data.facet
                    }, 'fill');
                    chartBase.contextualMenu.addContextualMenuHandler(this, {
                        color: d.data.color,
                        animation: d.data.animation,
                        facet: d.data.facet
                    });
                });

            const slices = wrappers.select('path.slice');

            function arcTween(a) {
                const i = d3.interpolate(this._current, a);
                this._current = i(0);
                return function(t) {
                    return arc(i(t));
                };
            }

            if (facetData.total > 0) {
                wrappers
                    .style('pointer-events', 'none')
                    .interrupt('fade')
                    .transition('fade')
                    .attr('opacity', 1);

                slices
                    .style('pointer-events', 'all')
                    .interrupt('fade')
                    .transition('fade')
                    .attrTween('d', arcTween);
                noData
                    .interrupt('fade')
                    .transition('fade')
                    .attr('opacity', 0);

            } else {
                noData.interrupt('fade').transition('fade')
                    .attr('opacity', 1);
                wrappers.exit()
                    .style('pointer-events', 'none')
                    .interrupt('fade')
                    .transition('fade')
                    .attr('opacity', 0);
            }

            drawHole();

            if (ChartHierarchyDimension.getCurrentHierarchyLevel(chartDef) > 0) {
                const anchor = SVGUtils.addPlotAreaContextMenuAnchor(g, chartBase.vizWidth, chartBase.vizHeight);
                chartBase.contextualMenu.addChartContextualMenuHandler(anchor.node(), undefined, () => ChartDrilldown.getDrillupActions(chartDef));
            }
        };
    }
})();
