import logging
from contextlib import contextmanager
from pathlib import Path

from dataiku.base.utils import is_os_windows

logger = logging.getLogger(__name__)


if not is_os_windows():
    import fcntl
else:
    logger.debug("Windows OS detected, file locking disabled")


@contextmanager
def acquire_lockfile(lockfile_path: str, operation=fcntl.LOCK_EX):
    """
    Gets a lock over a file to help synchronizing multiple processes.
    By default, the lock is exclusive, and the call is blocking.
    """
    Path(lockfile_path).touch()
    with open(lockfile_path, "w") as fd:
        if not is_os_windows():
            logger.debug("acquire lockfile: {}".format(lockfile_path))
            fcntl.flock(fd, operation)

        try:
            yield
        finally:
            if not is_os_windows():
                logger.debug("release lockfile: {}".format(lockfile_path))
                fcntl.flock(fd, fcntl.LOCK_UN)


def try_acquire_lockfile(lockfile_path: str, operation=fcntl.LOCK_EX|fcntl.LOCK_NB):
    """
    Tries to acquire a lock over a file. By default, tries to acquire an
    exclusive lock, in a non-blocking mode.

    Returns (file_descriptor, True) when locking is successful, otherwise
    returns (None, False).

    :rtype: tuple(io.TextIOWrapper, boolean)
    """
    Path(lockfile_path).touch()
    fd = open(lockfile_path, "w")

    try:
        if not is_os_windows():
            logger.debug("try acquire lockfile: {}".format(lockfile_path))
            fcntl.flock(fd, operation)

        return fd, True

    except BlockingIOError:
        # the lock is already held by another thread/process
        # close the file descriptor, no longer used
        fd.close()
        return None, False


def release_lockfile(lockfile_path: str, fd):
    """
    Releases a previously acquired lock. Also closes the provided file
    descriptor.
    """
    try:
        if not is_os_windows():
            logger.debug("release lockfile: {}".format(lockfile_path))
            fcntl.flock(fd, fcntl.LOCK_UN)

    finally:
        # close the file descriptor, no longer used
        fd.close()
