(function() {
    'use strict';

    const app = angular.module("dataiku.fm.accounts", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs"]);

    app.controller("AccountsListController", function ($scope, $state, $location, $q, FMAPI, Dialogs, CreateModalFromTemplate, FutureProgressModal, WT1, TaggingService) {
        $scope.activeProjectTagColor = TaggingService.getDefaultColor;
        $scope.authenticationModes = [
            ['DEFAULT_INSTANCE_CREDENTIALS', 'Same as Fleet Manager'],
            ['MANAGED_IDENTITY', 'Managed Identity'],
            ['OAUTH2_AUTHENTICATION_METHOD_CLIENT_SECRET', 'Application'],
            ['OAUTH2_AUTHENTICATION_METHOD_CERTIFICATE', 'Application'],
            ['IAM_ROLE', 'IAM Role'],
            ['KEYPAIR', 'Keypair'],
            ['JSON_KEY', 'JSON key']
        ];
    
        $scope.getLabelForType = function(mode) {
            var label = 'Unknown'; 
            angular.forEach($scope.authenticationModes, function(value) {
                if (value[0] === mode) {
                    label = value[1];
                }
            });
            return label;
        };

        $scope.refreshAccountList = function() {
            FMAPI.accounts.list().success(function(data) {
                $scope.accounts = data;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.newAccount = function() {
            $state.go("accounts.edit", { accountId: "new" });
        };

        $scope.isDeletable = function (account) {
            return account.id !== "main";
        }

        $scope.canBeDeleted = function (account) {
            return account.nbNetworks == 0;
        }

        $scope.deleteAccount = function(account) {
            if (account.nbNetworks === 0) {
                Dialogs.confirmInfoMessages($scope, 'Confirm account deletion', null, 'Are you sure you want to delete the account "' + sanitize(account.label) + '"?', false).then(function() {
                    $scope.deleteAccountNoConfirmation(account);
                });
            } else {
                // For accounts where a grey icon is displayed, no confirmation message as deletion will fail.
                // This way users will get the reason why this template cannot be deleted.
                $scope.deleteAccountNoConfirmation(account);
            }
        };

        $scope.deleteAccountNoConfirmation = function(template) {
            WT1.event("fm-account-delete", {});
            FMAPI.accounts.delete(template.id).success(function(commandResult) {
                resetErrorInScope($scope);
                if (commandResult.success) {
                    $scope.refreshAccountList();
                } else {
                    CreateModalFromTemplate("/templates/dialogs/info-messages-dialog.html", $scope, null, function(newScope) {
                        newScope.modalTitle = "Cannot delete account";
                        newScope.data = commandResult.statusMessages;
                    }, true, 'static', true);
                }
            }).error(setErrorInScope.bind($scope));
        };


        $scope.refreshAccountList();
    });

    app.controller("AccountBaseController", function($scope, $state, $stateParams, FMAPI) {
        $scope.refresh = function() {
            FMAPI.accounts.get($stateParams.accountId).success(function(account) {
                $scope.account = account;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.refresh();
    });

    app.controller("AccountEditController", function($scope, $stateParams, $state, FMAPI, Dialogs, CreateModalFromTemplate, ActivityIndicator, WT1) {
        $scope.accountEditionState = {
            dirty: false
        };

        $scope.isMain = function (account) {
            return account.id === "main";
        }
        $scope.isLegacy = function (account) {
            if ($scope.appConfig.cloud == 'AWS') {
                return account.awsAuthenticationMode === "LEGACY";
            }
            if ($scope.appConfig.cloud == 'AZURE') {
                return account.azureAuthenticationMode === "LEGACY";
            }
            if ($scope.appConfig.cloud == 'GCP') {
                return account.gcpAuthenticationMode === "LEGACY";
            }
            return false;
        }
        $scope.isReadOnly = function (account) {
            return (account.id === "main") && !$scope.isLegacy(account);
        }

        $scope.randomString = function(length) {
            var randomChars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
            var result = '';
            for (let i = 0; i < length; i++ ) {
                result += randomChars.charAt(Math.floor(Math.random() * randomChars.length));
            }
            return result;
        }

        $scope.loadAccount = function(accountId) {
            if (accountId === "new") {
                // Creation mode
                $scope.account = { id: "", label: "" };
                $scope.account.awsIAMRoleExternalId = $scope.randomString(16);
                $scope.account.cloudTags = [];
                $scope.originalAccount = $scope.account;
            } else {
                // Edition mode
                FMAPI.accounts.get(accountId).success(function(account) {
                    resetErrorInScope($scope);
                    $scope.account = account;
                    // eslint-disable-next-line no-undef
                    $scope.originalAccount = dkuDeepCopy($scope.account, filterDollarKey);
                }).error(setErrorInScope.bind($scope));
            }
            // to retrieve encryption state
            FMAPI.tenant.getCloudCredentials().success(function(data) {
                $scope.cloudCredentials = data;
            }).error(setErrorInScope.bind($scope));
        };
        $scope.$on("dirtyChanged", function(event, dirtyValue) {
            $scope.accountEditionState.dirty = dirtyValue;
        });
        $scope.refresh = function() {
            FMAPI.accounts.get($stateParams.accountId).success(function(account) {
                $scope.account = account;
                // eslint-disable-next-line no-undef
                $scope.$emit('accountUpdated', account);
            }).error(setErrorInScope.bind($scope));
        };
        $scope.saveAccount = function() {
            if ($scope.account.id) {
                FMAPI.accounts.save($scope.account).success(function(account) {
                    resetErrorInScope($scope);
                    $scope.account = account;
                    // eslint-disable-next-line no-undef
                    $scope.originalAccount = dkuDeepCopy($scope.account, filterDollarKey);
                    $scope.$emit('dirtyChanged', false);
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind($scope));
            } else {

                WT1.event("fm-instance-template-create", {});

                // Create new account
                FMAPI.accounts.create($scope.account).success(function(newAccount) {
                    ActivityIndicator.success("Account created");
                    $state.go("accounts.edit", { accountId: newAccount.id });
                }).error(setErrorInScope.bind($scope));
            }
        };
        $scope.$watch("account", function(nv) {
            if (nv) {
                // eslint-disable-next-line no-undef
                const dirtyValue = !angular.equals($scope.originalAccount, dkuDeepCopy($scope.account, filterDollarKey));
                $scope.$emit('dirtyChanged', dirtyValue);
            }
        }, true);

        $scope.loadAccount($stateParams.accountId);
    });
}());

