(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Utility functions for dashboard pages.
     */
    app.factory('DashboardPageUtils', function(TileLoadingState, FILTERABLE_INSIGHT_TYPES, DataikuAPI, ColorUtils, DashboardUtils, $q, translate) {
        const isInExport = getCookie('dku_graphics_export') === 'true';

        const ADJACENT_SQUARES_TO_LOAD_AROUND_VIEWPORT = 3;

        const columnSummaryByProjectKeyAndDataSpec = new Map(); // store the column summary for each pair of {projectKey, dataSpec}
        const columnSummaryRequestsQueues = new Map(); // used to handle simultaneous requests to fetch the column summary of the same pair of {projectKey, dataSpec}

        function isTileAdjacent(tile, { top, bottom }) {
            if (_.isNil(top) || _.isNil(bottom) || _.isNil(tile) || _.isNil(tile.box) || _.isNil(tile.box.top) || _.isNil(tile.box.height)) {
                return false;
            }
            const tileBottom = tile.box.top + tile.box.height;
            const tileTop = tile.box.top;
            return (
                (tileBottom >= top - ADJACENT_SQUARES_TO_LOAD_AROUND_VIEWPORT && tileBottom <= top) ||
                (tileTop >= bottom && tileTop <= bottom + ADJACENT_SQUARES_TO_LOAD_AROUND_VIEWPORT)
            );

        }

        function getViewportTopAndBottomCoords(visibleTileIndex, tileById) {
            return Object.entries(visibleTileIndex).reduce((acc, [tileId, isVisible]) => {
                if (isVisible) {
                    const tile = tileById[tileId];
                    if (_.isNil(tile) || _.isNil(tile.box)) {
                        return;
                    }
                    let { top, bottom } = acc;
                    if (_.isNil(top) || _.isNil(bottom)) {
                        return { top: tile.box.top, bottom: tile.box.top + tile.box.height };
                    } else {
                        const tileBottom = tile.box.top + tile.box.height;
                        const tileTop = tile.box.top;

                        if (tileBottom > bottom) {
                            bottom = tileBottom;
                        }
                        if (tileTop < top) {
                            top = tileTop;
                        }
                        return { top, bottom };
                    }
                }
                return acc;
            }, { top: null, bottom: null });
        }

        function shadeColor(color, delta) {
            let R = parseInt(color.substring(1, 3), 16);
            let G = parseInt(color.substring(3, 5), 16);
            let B = parseInt(color.substring(5, 7), 16);

            R = parseInt(R + delta);
            G = parseInt(G + delta);
            B = parseInt(B + delta);

            R = R < 255 ? R : 255;
            G = G < 255 ? G : 255;
            B = B < 255 ? B : 255;

            const RR = R.toString(16).length == 1 ? '0' + R.toString(16) : R.toString(16);
            const GG = G.toString(16).length == 1 ? '0' + G.toString(16) : G.toString(16);
            const BB = B.toString(16).length == 1 ? '0' + B.toString(16) : B.toString(16);

            return '#' + RR + GG + BB;
        }

        const svc = {
            isTileVisible: (tileId, hook) => {
                if (isInExport || _.isNil(hook)) {
                    return true;
                } else {
                    return !!(hook.visibleStates[tileId]);
                }
            },
            isTileVisibleOrAdjacent: (tileId, hook) => {
                if (isInExport || _.isNil(hook) || (_.isNil(hook.visibleStates) && _.isNil(hook.adjacentStates))) {
                    return true;
                } else {
                    return !!(hook.visibleStates[tileId] || hook.adjacentStates[tileId]);
                }
            },
            getTilesToLoadOrderedList: (tileLoadingPromiseByTileId, hook, tileById) => {
                const tilesToLoad = svc.getTilesToLoad(tileLoadingPromiseByTileId, hook);
                /*
                 * Load tiles in the following order of priority:
                 * 1. Visible over adjacent
                 * 2. Unfilterable tiles first
                 * 3. Tiles at the top-most position (smallest top value)
                 * 4. Tiles at the left-most position (smallest left value)
                 */
                tilesToLoad.sort((tileId1, tileId2) => {
                    const isTile1Visible = svc.isTileVisible(tileId1, hook);
                    const isTile2Visible = svc.isTileVisible(tileId2, hook);
                    const isTile1Filterable = FILTERABLE_INSIGHT_TYPES.includes(tileById[tileId1].insightType);
                    const isTile2Filterable = FILTERABLE_INSIGHT_TYPES.includes(tileById[tileId2].insightType);
                    const tile1TopPosition = tileById[tileId1].box.top;
                    const tile2TopPosition = tileById[tileId2].box.top;
                    const tile1LeftPosition = tileById[tileId1].box.left;
                    const tile2LeftPosition = tileById[tileId2].box.left;

                    if (isTile1Visible && !isTile2Visible) {
                        return -1;
                    }
                    if (!isTile1Visible && isTile2Visible) {
                        return 1;
                    }

                    if (isTile1Filterable && !isTile2Filterable) {
                        return 1;
                    }
                    if (!isTile1Filterable && isTile2Filterable) {
                        return -1;
                    }

                    if (tile1TopPosition < tile2TopPosition) {
                        return -1;
                    }
                    if (tile1TopPosition > tile2TopPosition) {
                        return 1;
                    }

                    if (tile1LeftPosition < tile2LeftPosition) {
                        return -1;
                    }
                    if (tile1LeftPosition > tile2LeftPosition) {
                        return 1;
                    }

                    return 0;
                });
                return tilesToLoad;
            },
            getIsTileAdjacentToVisibleTilesByTileId: (visibleTileIndex, tileById = {}) => {
                const viewportTopAndBottomCoords = getViewportTopAndBottomCoords(visibleTileIndex, tileById);
                return Object.fromEntries(Object.entries(visibleTileIndex).map(([tileId, isVisible]) => {
                    const tile = tileById[tileId];
                    return [tileId, !isVisible && isTileAdjacent(tile, viewportTopAndBottomCoords)];
                }));
            },
            areAllVisibleAndAdjacentInsightsLoaded: (tiles, hook) => {
                return !tiles.some(tile =>
                    tile.tileType === 'INSIGHT' && svc.getLoadingState(tile.$tileId, hook) !== TileLoadingState.COMPLETE && svc.isTileVisibleOrAdjacent(tile.$tileId, hook));
            },
            getLoadingState: (tileId, hook) => {
                return (tileId in hook.loadStates) ? hook.loadStates[tileId] : TileLoadingState.WAITING;
            },
            hasLoadingTiles: (tileLoadingPromiseByTileId, hook) => {
                return Object.keys(tileLoadingPromiseByTileId).some(tileId =>
                    svc.getLoadingState(tileId, hook) !== TileLoadingState.COMPLETE && svc.isTileVisibleOrAdjacent(tileId, hook));
            },
            getTilesToLoad: (tileLoadingPromiseByTileId, hook) => {
                return Object.keys(tileLoadingPromiseByTileId).filter((tileId) => {
                    return svc.getLoadingState(tileId, hook) === TileLoadingState.WAITING || svc.getLoadingState(tileId, hook) === TileLoadingState.WAITING_FOR_RELOAD;
                });
            },
            getNumberOfLoadingTiles: (hook) => {
                return Object.keys(hook.loadStates).map(tileId => svc.getLoadingState(tileId, hook)).filter(state =>
                    state === TileLoadingState.LOADING).length;
            },
            getVisibleTilesToLoadOrReloadPromiseByTileId: (hook) => {
                return Object.fromEntries(Object.entries(hook.loadStates)
                    .filter(([key, loadState]) => svc.isTileVisibleOrAdjacent(key, hook) &&
                            (loadState === TileLoadingState.WAITING || loadState === TileLoadingState.WAITING_FOR_RELOAD))
                    .map(([key]) => [key, hook.loadStates[key] === TileLoadingState.WAITING ? hook.loadPromises[key] : hook.reloadPromises[key]]));
            },
            getColumnSummary: (projectKey, dataSpec) => {
                const key = `${projectKey}/${JSON.stringify(dataSpec)}`;
                if (columnSummaryByProjectKeyAndDataSpec.has(key)) {
                    return Promise.resolve(columnSummaryByProjectKeyAndDataSpec.get(key));
                }
                return new Promise((resolve, reject) => {
                    if (columnSummaryRequestsQueues.has(key)) {
                        columnSummaryRequestsQueues.get(key).push({ queuedResolve: resolve, queuedReject: reject });
                    } else {
                        columnSummaryRequestsQueues.set(key, [{ queuedResolve: resolve, queuedReject: reject }]);
                        DataikuAPI.shakers.charts.getColumnsSummary(projectKey, dataSpec)
                            .noSpinner()
                            .success(data => {
                                columnSummaryByProjectKeyAndDataSpec.set(key, data);
                                columnSummaryRequestsQueues.get(key).forEach(({ queuedResolve }) => queuedResolve(data));
                                columnSummaryRequestsQueues.delete(key);
                            }).error((data, status, headers, config, statusText, xhrStatus) => {
                                columnSummaryRequestsQueues.get(key).forEach(({ queuedReject }) => {
                                    queuedReject({ data, status, headers, config, statusText, xhrStatus });
                                });
                                columnSummaryRequestsQueues.delete(key);
                            });
                    }
                });
            },
            flushCachedColumnSummaries: () => {
                columnSummaryByProjectKeyAndDataSpec.clear();
            },
            getGridColor: (backgroundColor, backgroundOpacity) => {
                if (backgroundColor == null) {
                    backgroundColor = '#ffffff';
                }
                let contrastColor = shadeColor(backgroundColor, +20);
                if (contrastColor === '#ffffff') {
                    contrastColor = shadeColor(backgroundColor, -5);
                }
                return ColorUtils.getHexWithAlpha(contrastColor, backgroundOpacity);
            },
            positionResizeHandles: function(tileWrapper, tileSpacing) {
                if (tileWrapper == null) {
                    return;
                }
                const tileMargin = tileSpacing / 2;
                const handleMargin = `${tileMargin - 3}px`;
                tileWrapper.find('.ui-resizable-nw').css({
                    left: handleMargin,
                    top: handleMargin
                });

                tileWrapper.find('.ui-resizable-ne').css({
                    top: handleMargin,
                    right: handleMargin
                });

                tileWrapper.find('.ui-resizable-sw').css({
                    bottom: handleMargin,
                    left: handleMargin
                });

                tileWrapper.find('.ui-resizable-se').css({
                    bottom: handleMargin,
                    right: handleMargin
                });

                tileWrapper.find('.ui-resizable-n').css({
                    top: handleMargin,
                    left: '50%'
                });

                tileWrapper.find('.ui-resizable-s').css({
                    bottom: handleMargin,
                    left: '50%'
                });

                tileWrapper.find('.ui-resizable-w').css({
                    left: handleMargin,
                    top: '50%'
                });

                tileWrapper.find('.ui-resizable-e').css({
                    right: handleMargin,
                    top: '50%'
                });
            },
            isOverflowingVertically: function(element) {
                return element.scrollHeight > element.clientHeight;
            },
            getTileElementFromTileId: function(container, tileId) {
                return container.find('[data-id = ' + tileId+ ']');
            },
            getTileWrappers: function(container) {
                return container.find('.tile-wrapper');
            },
            getTileWrapperFromElement: function(element) {
                return $(element).closest('.tile-wrapper');
            },
            getTileByElement(tiles, el) {
                const tileId = $(el).data('id');
                for (let i = 0; i< tiles.length; i++) {
                    const tile = tiles[i];
                    if (tileId == tile.$tileId) {
                        return tile;
                    }
                }
            },
            isPerformingDradAngDropWithinTile(element) {
                return element.closest('[dashboard-no-drag]') != null;
            },
            computeGridContainerBackgroundColor(showGrid, editable, backgroundColor, gridColor) {
                return showGrid && editable ? gridColor : backgroundColor ? backgroundColor : '#ffffff';
            },
            getNextPageIndex: function(dashboard, currentPageIdx) {
                if (currentPageIdx < dashboard.pages.length - 1) {
                    return currentPageIdx + 1;
                } else if (dashboard.circularNavigation) {
                    return 0;
                }
                return currentPageIdx;
            },
            getPreviousPageIndex: function(dashboard, currentPageIdx) {
                if (currentPageIdx > 0) {
                    return currentPageIdx - 1;
                } else if (dashboard.circularNavigation) {
                    return dashboard.pages.length - 1;
                }
                return currentPageIdx;
            },
            getPageTitle(title, index) {
                if (title && title.length) {
                    return DashboardUtils.expandVariables(title);
                } else {
                    return $q.resolve({ data: translate('DASHBOARD.NAVIGATION_DRAWER.PAGE_INDEX', 'Page ' + (index + 1), { index: index + 1 }) });
                }
            }
        };

        return svc;
    });
})();
