(function(){
'use strict';
const app = angular.module('dataiku.agents', ['dataiku.ml.report', 'dataiku.lambda']);

app.controller("AgentSavedModelController", function($scope, $rootScope, $state, $controller, $stateParams, DataikuAPI, ActiveProjectKey, FullModelLikeIdUtils, $q,
    DKUtils, Dialogs, WT1, Debounce) {
    $controller("_SavedModelReportController", {$scope});

    DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.fillVersionSelectorStuff(data);
    });

    const currentVersionId = FullModelLikeIdUtils.parse($stateParams.fullModelId).versionId;
    $scope.currentVersionId = currentVersionId;
    $scope.warningMsg = '';

    $scope.$watch("savedModel", function(nv, ov) {
        if (!nv) return;

        $scope.currentVersionIdx = nv.inlineVersions.findIndex(v => v.versionId === currentVersionId);
        if ($scope.currentVersionIdx < 0) {
            return;
        }
        $scope.currentlySavedInlineModel = angular.copy(nv.inlineVersions[$scope.currentVersionIdx]);
        $scope.currentVersionData = nv.inlineVersions[$scope.currentVersionIdx];
    });

    // to trigger change detection in child components
    $scope.currentVersionUpdateCounter = 0;
    $scope.$watch('currentVersionData', Debounce().withDelay(200, 200).wrap(() => $scope.currentVersionUpdateCounter++), true);

    $scope.save = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent not dirty');
        }

        const deferred = $q.defer();
        const saveAfterConflictCheck = function() {
            DataikuAPI.savedmodels.agents.save($scope.savedModel, $scope.currentVersionId).success(function(data) {
                WT1.event(
                    'agent-save', {
                        savedModelType: data.savedModelType,
                    });
                $scope.savedModel = data;
                $scope.currentlySavedInlineModel = angular.copy($scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
                deferred.resolve('agent saved');
            }, () => deferred.reject());
        };

        DataikuAPI.savedmodels.agents.checkSaveConflict($scope.savedModel).success(function(conflictResult) {
            if (!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope, conflictResult).then(
                    function(resolutionMethod) {
                        if (resolutionMethod === 'erase') {
                            saveAfterConflictCheck();
                        } else if (resolutionMethod === 'ignore') {
                            deferred.reject();
                            DKUtils.reloadState();
                        }
                    }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));

        return deferred.promise;
    };
    $rootScope.saveAgentModel = $scope.save;

    $scope.internalCodeEnvsHRef = function() {
        if ($scope.appConfig.isAutomation) {
            return $state.href("admin.codeenvs-automation.internal");
        } else {
            return $state.href("admin.codeenvs-design.internal");
        }
    }

    $scope.saveIgnoringQuickTestQuery = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent not dirty');
        }

        if (!$scope.isConfigDirty()) {
            return $q.when('only difference is the quicktest query, not saving');
        }

        return $scope.save();
    };

    $scope.isConfigDirty = function() {
        let frankenVersion = angular.copy($scope.currentlySavedInlineModel);
        frankenVersion["quickTestQueryStr"] = $scope.savedModel.inlineVersions[$scope.currentVersionIdx]["quickTestQueryStr"];

        return !angular.equals(frankenVersion, $scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
    };

    $scope.isDirty = function() {
        if (!$scope.savedModel) return false;
        return !angular.equals($scope.savedModel.inlineVersions[$scope.currentVersionIdx], $scope.currentlySavedInlineModel);
    };
    $scope.getCurrentFile = function() {
        if (!$scope.savedModel) return false;
        return $scope.savedModel.id + "_" + $scope.currentVersionId + ".py";
    };
    $rootScope.agentModelIsDirty = $scope.isDirty;
    $rootScope.agentModelCurrentFile = $scope.getCurrentFile;

    function allowedTransitionsFn(data) {
        return data.toState?.name?.startsWith('projects.project.savedmodels.savedmodel.agent') 
            && data.fromState?.name?.startsWith('projects.project.savedmodels.savedmodel.agent') 
            && data.toParams?.fullModelId === data.fromParams?.fullModelId;
    }

    checkChangesBeforeLeaving($scope, $scope.isDirty, null, allowedTransitionsFn);
});

app.controller("AgentSavedModelDesignController", function($scope, $rootScope, $state, $stateParams, Debounce, ActivityIndicator, CodeBasedEditorUtils,
                                                           CreateModalFromTemplate, DataikuAPI, PluginConfigUtils, PluginsService, TopNav, SmartId) {
    $scope.editorOptions = CodeBasedEditorUtils.editorOptions('text/x-python', $scope, true);
    CodeBasedEditorUtils.registerBroadcastSelectionHandler($scope.editorOptions);
    CodeBasedEditorUtils.setRecipeScript($scope.script);

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        if ($scope.savedModel.savedModelType === "PLUGIN_AGENT") {
            $scope.loadedDesc =  $rootScope.appConfig.customAgents.find(x => x.desc.id === $scope.currentVersionData.pluginAgentType);
            $scope.desc = $scope.loadedDesc.desc;
            $scope.pluginDesc = PluginsService.getOwnerPluginDesc($scope.loadedDesc);

            if ($scope.pluginDesc && $scope.desc && $scope.desc.params) {
                PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.currentVersionData.pluginAgentConfig);
            }
        }

        if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            $scope.onToolLLMChange = function() {
                if ($scope.currentVersionData.toolsUsingAgentSettings.llmId && $scope.availableLLMs) {
                    $scope.activeLLM = $scope.availableLLMs.find(l => l.id === $scope.currentVersionData.toolsUsingAgentSettings.llmId);
                }
            };

            DataikuAPI.agentTools.listAvailable($stateParams.projectKey).success(function(data) {
                $scope.availableTools = data;
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.codeenvs.checkAgentsCodeEnv($stateParams.projectKey)
                .then(function ({data}) {
                    $scope.codeEnvType = data["codeEnvType"];
                    $scope.codeEnvError = data["codeEnvError"];
                }).catch(setErrorInScope.bind($scope));
        }

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
            $scope.availableLLMs = data.identifiers;

            if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
                $scope.availableLLMs = $scope.availableLLMs.filter(llm => (
                    !llm.id.startsWith("agent:" + $scope.savedModel.id)  // Don't allow selecting myself as the LLM for this agent, to avoid infinite loop
                    && llm.type !== "RETRIEVAL_AUGMENTED"
                    && (
                        $scope.currentVersionData.toolsUsingAgentSettings.allowAgentsAsLLM 
                        || (llm.type !== "SAVED_MODEL_AGENT")
                    )
                ));
                $scope.onToolLLMChange();
            }
        }).error(setErrorInScope.bind($scope));

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "design");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "design");
            }
        }
    });

    $scope.$watch('currentVersionData.toolsUsingAgentSettings.tools', Debounce().withDelay(500, 500).wrap(() => {
        if ($scope.availableTools && $scope.currentVersionData.toolsUsingAgentSettings.tools.length) {
            const kbTools = $scope.availableTools.filter(tool => tool.type === 'VectorStoreSearch').map(tool => SmartId.fromTor(tool));
            $scope.hasKbTools = $scope.currentVersionData.toolsUsingAgentSettings.tools.some(tool => !tool.disabled && kbTools.includes(tool.toolRef));
        }
    }), true);

    $scope.isToolAvailable = function(usedTool) {
        if ($scope.availableTools && usedTool?.toolRef) {
            return $scope.availableTools.some((l) => SmartId.fromTor(l) === usedTool.toolRef);
        }
        return false;
    };

    $scope.createNewTool = function ($event) {
        $event.preventDefault();
        const stateName = 'projects.project.agenttools.list';
        if ($scope.isDirty()) {
            $scope.save().then(function () {
                ActivityIndicator.success("Saved");
                $scope.$state.go(stateName, { createTool: true });
            }).catch(setErrorInScope.bind($scope));
        } else {
            $scope.$state.go(stateName, { createTool: true });
        }
    };

    $scope.toggleTool = function(usedTool) {
        usedTool.disabled = !usedTool.disabled;
    };

    $scope.editToolDescriptionModal = function(usedTool) {
        if (!usedTool.toolRef) {
            return;
        }
        const smartId = SmartId.resolve(usedTool.toolRef, $stateParams.projectKey);

        DataikuAPI.agentTools.get(smartId.projectKey, smartId.id).success(agentTool => {
            const modalScope = $scope.$new(true);
            modalScope.tool = agentTool;
            modalScope.usedTool = usedTool;
            modalScope.toolAdditionalDescription = usedTool.additionalDescription;

            CreateModalFromTemplate("/templates/savedmodels/agents/agent-design-edit-tool-description-modal.html", modalScope, null, function(modalScope) {
                modalScope.accept = function() {
                    if (modalScope.toolAdditionalDescription) {
                        usedTool.additionalDescription = modalScope.toolAdditionalDescription;
                    } else {
                        delete usedTool.additionalDescription;
                    }
                };

                modalScope.showToolDescriptionModal = function() {
                    DataikuAPI.agentTools.getDescriptor(agentTool.projectKey, agentTool.id).success(descriptor => {
                        const innerModalScope = $scope.$new(true);
                        innerModalScope.toolName = agentTool.name;
                        // remove the global description that was appended by the back-end in get-descriptor to show it in another field
                        innerModalScope.internalDescription = descriptor.description.replace(agentTool.additionalDescriptionForLLM, "").trimEnd();
                        innerModalScope.globalDescription = agentTool.additionalDescriptionForLLM;
                        innerModalScope.agentSpecificDescription = modalScope.toolAdditionalDescription || "";
                        innerModalScope.agentToolURL = $state.href("projects.project.agenttools.agenttool", {projectKey: agentTool.projectKey, agentToolId: agentTool.id});
                        CreateModalFromTemplate('/templates/savedmodels/agents/agent-tool-description-modal.html', innerModalScope);
                    }).error(setErrorInScope.bind($scope));
                };
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AgentSavedModelSettingsController", function($scope, $controller, $stateParams, DataikuAPI, TopNav, FLOW_COMPUTABLE_DEPENDENCIES, SavedModelHelperService) {
    $scope.availableDependencies = undefined;
    $scope.currentDependencies = undefined;

    $scope.isDependenciesEnabled = function(){
        return $scope.savedModel?.savedModelType === 'PYTHON_AGENT';
    }

    $scope.$watch("savedModel", (nv, ov) => {
        if (!nv) return;

        switch ($scope.savedModel.savedModelType) {
            case "PYTHON_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.pythonAgentSettings;
                break;
            case "PLUGIN_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.pluginAgentSettings;
                break;
            case "TOOLS_USING_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.toolsUsingAgentSettings;
                break;
        }

        if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
                if ($scope.currentAgentSettings.llmId && data.identifiers) {
                    $scope.activeLLM = data.identifiers.find(l => l.id === $scope.currentAgentSettings.llmId);
                }
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.codeenvs.checkAgentsCodeEnv($stateParams.projectKey)
                .then(function ({data}) {
                    $scope.codeEnvType = data["codeEnvType"];
                    $scope.codeEnvError = data["codeEnvError"];
                }).catch(setErrorInScope.bind($scope));
        }

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "settings");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "settings");
            }
        }

        $scope.addDependency = function(dependencyToAdd){
            const pythonSettings = $scope.currentVersionData.pythonAgentSettings;
            if (!pythonSettings.dependencies) {
                pythonSettings.dependencies = [];
            }

            const isAlreadyAdded = pythonSettings.dependencies.some(dep => dep.ref === dependencyToAdd.smartId);
            if (isAlreadyAdded) {
                $scope.warningMsg = "Object already added as dependency.";
                return;
            }

            const newDependency = {
                type: dependencyToAdd.model?.savedModelType || dependencyToAdd.type,
                ref: dependencyToAdd.smartId
            };

            pythonSettings.dependencies.push(newDependency);
            $scope.currentDependencies.push(dependencyToAdd);
            $scope.warningMsg = "";
        };

        $scope.removeDependency = function(index) {
            if (index < 0) { return; }
            const pythonSettings = $scope.currentVersionData.pythonAgentSettings;
            if (pythonSettings?.dependencies) {
                pythonSettings.dependencies.splice(index, 1);
                $scope.currentDependencies.splice(index, 1);
            }
        };

        // only load this once as it depends on the project not the model
        if ($scope.isDependenciesEnabled() && $scope.availableDependencies === undefined) {
            $scope.availableDependencies = [];
            $scope.currentDependencies = [];
            DataikuAPI.taggableObjects.listAccessibleObjects($stateParams.projectKey).then(function(resp) {
                const dependencies = resp.data;
                $scope.currentDependencies = SavedModelHelperService.enrichDependencies($scope.currentAgentSettings?.dependencies, dependencies)
                $scope.availableDependencies = SavedModelHelperService.filterDependencies(dependencies, FLOW_COMPUTABLE_DEPENDENCIES);
            }).catch(setErrorInScope.bind($scope));
        }
    });
});

app.controller("AgentVersionHistoryController", function($scope, $stateParams, $state, $controller, TopNav) {
    $scope.redirectCallback = function() {
        $state.go('projects.project.savedmodels.savedmodel.versions', {projectKey:$stateParams.projectKey, savedModelId:$scope.savedModel.id});
    };
    TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "history");
});


app.controller("AgentSavedModelLogsController", function($scope, $stateParams, $controller, DataikuAPI, Dialogs, TopNav) {
    $scope.listLogs = function() {
        DataikuAPI.savedmodels.agents.listLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId).success(function(data) {
            $scope.agentLogs = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.clearLogs = function() {
        const title = 'Confirm logs deletion';
        const message = 'Are you sure you want to clear the logs for this agent?';

        Dialogs.confirm($scope, title, message).then(() => {
            DataikuAPI.savedmodels.agents.clearLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId)
                .error(setErrorInScope.bind($scope))
                .finally($scope.listLogs);
        });
    }

    $scope.deleteLog = function(projectKey, savedModelId, versionId, logName) {
        return DataikuAPI.savedmodels.agents.deleteLog(projectKey, savedModelId, versionId, logName)
                .error(setErrorInScope.bind($scope))
                .finally($scope.listLogs);
    }

    $scope.getLog = DataikuAPI.savedmodels.agents.getLog;

    $scope.logDownloadURL = (projectKey, savedModelId, versionId, logName) => {
        const params = new URLSearchParams({
            projectKey,
            smId: savedModelId,
            versionId,
            logName
        });
        return `/dip/api/savedmodels/agents/stream-log?${params}`;
    };

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        $scope.listLogs();

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "logs");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "logs");
            }
        }
    });
});

app.component('agentArtifacts', {
    bindings: {
        artifacts: '<',
        expanded: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifacts.html',

    controller: function() {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            if ($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }
        };

        $ctrl.hasDisplayableArtifacts = function() {
            // If the agent returns no results, we could have a sources section with empty `items`
            return $ctrl.artifacts?.some(artifact => artifact.parts.length > 0);
        };

        $ctrl.shouldExpandArtifact = function(artifact) {
            // Always expand artifacts that do not have a hierarchy field
            // Expand thinking artifacts to level 1
            // Expand all other artifacts to level 2
            if (!artifact.hierarchy || !artifact.hierarchy.length) {
                return true;
            }
            if (artifact.type === "REASONING") {
                return artifact.hierarchy.length <= 1;
            }
            return artifact.hierarchy.length <= 2;
        };
    }
});

app.component('agentArtifact', {
    bindings: {
        artifact: '<',
        expanded: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifact.html',

    controller: function($dkuSanitize, Logger) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            if ($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }
        };

        $ctrl.convertToMarkdown = function(reasoningParts) {
            for (let reasoning of reasoningParts) {
                if (reasoning.type === "TEXT") {
                    try {
                        reasoning.reasoning  = $dkuSanitize(marked(reasoning.text, { breaks: true}));
                    }
                    catch(e) {
                        Logger.error('Error parsing markdown HTML, switching to plain text', e);
                    }
                }
            }
            return reasoningParts;
        };

        $ctrl.hasHierarchy = function() {
            return $ctrl.artifact.hierarchy && $ctrl.artifact.hierarchy.length;
        };
    },
});

app.component('agentHierarchy', {
    bindings: {
        hierarchy: '<',
    },
    templateUrl: '/templates/savedmodels/agents/hierarchy.html',
});

app.component('artifactRecordsTable', {
    bindings: {
        records: '<',
    },
    template: `
        <div class="export-btn pull-right">
            <button class="btn btn--secondary" ng-click="$ctrl.exportData()">Export</button>
        </div>
        <div id="artifact-records-table" class="agent-artifact-records-table ag-theme-alpine" style="clear: both;"></div>
    `,
    controller: function(AgGrid, $element, ExportUtils, $scope) {
        const $ctrl = this;
        let gridApi;

        $ctrl.$onInit = function() {
            const $container = $element.find('#artifact-records-table')[0];
            const {headers, rows} = formatData($ctrl.records);
            gridApi = AgGrid.createGrid($container, initGridOptions(headers, rows));
        };

        $ctrl.$onChanges = (changes) => {
            if (!gridApi) return;

            if (changes && changes.records) {
                const {headers, rows} = formatData($ctrl.records);
                gridApi.setGridOption('columnDefs', headers);
                gridApi.setGridOption('rowData', rows);
            }
        };

        $ctrl.exportData = function() {
            ExportUtils.exportUIData($scope, {
                name : "Export artifact records table data",
                columns: $ctrl.records.columns.map(colName => { return { name: colName, type: "String" } }),
                data : $ctrl.records.data,
            }, "Export data", { hideAdvancedParameters: true });
        };

        function formatData(records) {
            const headers = records.columns.map(colName => { return { field: colName} });
            const rows = records.data.map(rowData => {
                const row = {};
                for (let i = 0; i < rowData.length; i++) {
                    row[headers[i].field] = rowData[i];
                }
                return row;
            });
            return {headers, rows}
        }

        function initGridOptions(headers, rows) {
            return {
                rowData: rows,
                columnDefs: headers,
                defaultColDef: {
                    sortable: true,
                    resizable: true,
                    filter: false
                },
                sideBar: {
                    toolPanels: [
                        {
                            id: 'columns',
                            labelDefault: 'Columns',
                            labelKey: 'columns',
                            iconKey: 'columns',
                            toolPanel: 'agColumnsToolPanel',
                            toolPanelParams: {
                                suppressRowGroups: true,
                                suppressValues: true,
                                suppressPivots: true,
                                suppressPivotMode: true,
                            }
                        }
                    ]
                },
                suppressPropertyNamesCheck: true,
                alwaysMultiSort: false,
                suppressCsvExport: true,
                suppressExcelExport: true,
                pinnedBottomRowData: [],
                tooltipShowDelay: 0,
                enableCellTextSelection: true,
                onFirstDataRendered: () => {
                    setTimeout(() => {
                        gridApi.autoSizeAllColumns();

                        const MAX_WIDTH = 600;
                        gridApi.getColumns().forEach(col => {
                            const width = col.getActualWidth();
                            if (width > MAX_WIDTH) {
                                gridApi.setColumnWidths([{
                                    key: col,
                                    newWidth: MAX_WIDTH
                                }]);
                            }
                        });
                    })
                },
            }
        }
    }
});

app.component('artifactDocument', {
    bindings: {
        part: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifact-document.html',
    controller: function (AnyLoc) {
        const $ctrl = this;
        $ctrl.fullFolderId = null;
        $ctrl.anyLocFolder = null;
        $ctrl.$onChanges = function (changes) {
            if (changes.part) {
                if ($ctrl.part.type === 'FILE_BASED_DOCUMENT') {
                    if ($ctrl.part.imageRefs && $ctrl.part.imageRefs.length) {
                        $ctrl.fullFolderId = $ctrl.part.imageRefs[0].folderId;
                        $ctrl.anyLocFolder = AnyLoc.getLocFromFull($ctrl.fullFolderId);
                        $ctrl.imagePaths = $ctrl.part.imageRefs.map(image => image['path']);
                    }
                }
                if ($ctrl.part.jsonSnippet) {
                    try {
                        $ctrl.parsedJsonSnippet = JSON.parse($ctrl.part.jsonSnippet);
                    } catch (error) {
                        $ctrl.parsedJsonSnippet = {
                            parsingError: "Invalid JSON format.",
                            originalValue: $ctrl.part.jsonSnippet
                        };
                    }
                }
            }
        };
    },
});

})();
