from abc import ABC, abstractmethod
from typing import Dict, Generic, Type, TypeVar
from dataiku.eda.types import Final

from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.exceptions import UnknownObjectType
from dataiku.eda.types import FilterModel, FilterTypeLiteral


FilterModelType = TypeVar("FilterModelType", bound=FilterModel)


class Filter(ABC, Generic[FilterModelType]):
    REGISTRY: Final[Dict[FilterTypeLiteral, Type['Filter']]] = {}

    @staticmethod
    @abstractmethod
    def get_type() -> FilterTypeLiteral:
        raise NotImplementedError

    @abstractmethod
    def apply(self, idf: ImmutableDataFrame, inverse: bool = False) -> ImmutableDataFrame:
        raise NotImplementedError

    @abstractmethod
    def serialize(self) -> FilterModel:
        raise NotImplementedError

    @staticmethod
    @abstractmethod
    def build(params: FilterModelType) -> 'Filter':
        try:
            filter_class = Filter.REGISTRY[params["type"]]
        except KeyError:
            raise UnknownObjectType("Unknown filter type: %s" % params.get("type"))
        return filter_class.build(params)

    @staticmethod
    def define(filter_class: Type['Filter']) -> None:
        Filter.REGISTRY[filter_class.get_type()] = filter_class
