#!/bin/bash -e
# Configures Dataiku Fleet Manager to start at boot

MYDIR=$(cd "$(dirname "$0")" && pwd -P)

Usage() {
	echo >&2 "Usage: $0 [-n INSTANCE_NAME] FM_DATADIR FM_USER"
	exit 1
}

SERVICE_NAME="dataiku"
if [ $# -ge 2 -a "$1" = "-n" ]; then
	SERVICE_NAME="dataiku.$2"
	shift 2
fi
if [ $# -ne 2 ]; then
	Usage
fi
FM_DATADIR="$1"
FM_USER="$2"

if [ -f "/etc/init.d/$SERVICE_NAME" ]; then
	cat >&2 <<==EOF==
* ERROR : legacy sysv-based init script for the same service found at "/etc/init.d/$SERVICE_NAME"
* Please remove it first
* This script will replace it with an equivalent systemd-based one
* You can also remove the legacy configuration file at "/etc/sysconfig/$SERVICE_NAME" or "/etc/default/$SERVICE_NAME"
==EOF==
	exit 1
fi

INSTALLID="$(sed -n 's/^ *installid *= *\([^ #]*\).*/\1/p' "$FM_DATADIR/install.ini")"
if [ -z "$INSTALLID" ]; then
	echo >&2 "$0 : ERROR : installid not found in $FM_DATADIR/install.ini"
	exit 1
fi

# Startup script and configuration file need to be accessible by the DSS user account
umask 22

echo "[+] Installing startup script in /etc/dataiku/$INSTALLID"
mkdir -p "/etc/dataiku/$INSTALLID"
install -o root -g root -p "$MYDIR/../dataiku-boot" "/etc/dataiku/$INSTALLID"

if [ -f "/etc/dataiku/$INSTALLID/dataiku-boot.conf" ]; then
	echo "[+] Preserving existing startup configuration file /etc/dataiku/$INSTALLID/dataiku-boot.conf"
else
    echo "[+] Installing default startup configuration file /etc/dataiku/$INSTALLID/dataiku-boot.conf"
	cat >>"/etc/dataiku/$INSTALLID/dataiku-boot.conf" <<==EOF==
# Service configuration file for Dataiku Fleet Manager instance $INSTALLID
FM_DATADIR="$FM_DATADIR"
FM_USER="$FM_USER"
==EOF==
fi

if selinux=$(/usr/sbin/getenforce 2>/dev/null) && [ "$selinux" != "Disabled" ]; then
	{
		semanage fcontext -a -t bin_t "/etc/dataiku/$INSTALLID/dataiku-boot" 2>/dev/null ||
		semanage fcontext -m -t bin_t "/etc/dataiku/$INSTALLID/dataiku-boot" &&
		restorecon -R /etc/dataiku/"$INSTALLID"
	} || echo >&2 "$0 : Warning : could not set SELinux policy for /etc/dataiku/$INSTALLID"
fi

echo "[+] Installing service configuration file /etc/systemd/system/$SERVICE_NAME.service"
# systemd < 235 does not support the 'infinity' timeout specification
systemd_version=$(systemctl --version | awk '/^systemd/{print $2}')
if [ "$systemd_version" -ge 235 ]; then
	infinity="infinity"
else
	infinity="0"
fi

cat >"/etc/systemd/system/$SERVICE_NAME.service" <<==EOF==
[Unit]
Description=Dataiku Fleet Manager instance at $FM_DATADIR
After=network-online.target nss-lookup.target
Wants=network-online.target

[Service]
Type=forking
TimeoutStartSec=$infinity
TimeoutStopSec=5min
User=$FM_USER
ExecStart=/etc/dataiku/$INSTALLID/dataiku-boot start
ExecStop=/etc/dataiku/$INSTALLID/dataiku-boot stop

[Install]
WantedBy=multi-user.target
==EOF==

systemctl daemon-reload
systemctl enable "$SERVICE_NAME"

echo "[+] Done"
