(function(){
'use strict';

var app = angular.module('dataiku.admin.monitoring', []);

app.controller("AdminMonitoringSummaryController", function($scope, $rootScope, $state, DataikuAPI, $filter, $anchorScroll, $timeout){
    $scope.refresh = function refresh(){
        $scope.isLoading = true;
        DataikuAPI.admin.monitoring.getGlobalUsageSummary().success(function(data){
            $scope.globalSummary = data;
            $scope.globalSummary.datasets.allByTypeArray = $filter("toKVArray")($scope.globalSummary.datasets.allByType);
            $scope.globalSummary.recipes.byTypeArray = $filter("toKVArray")($scope.globalSummary.recipes.byType);
            $timeout(() => $anchorScroll());
            $scope.isLoading = false;
        }).error(() => {
            setErrorInScope.bind($scope);
            $scope.isLoading = false;
        });
    }
    $scope.isLoading = false;
});

app.controller("AdminMonitoringWebAppBackendsController", function($scope, $rootScope, $state, DataikuAPI, ActivityIndicator) {
    $scope.refreshList = function(){
        DataikuAPI.webapps.listAllBackendsStates().success(function(data){
            $scope.backends = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.stopBackend = function(backend){
        DataikuAPI.webapps.stopBackend({projectKey:backend.projectKey, id:backend.webAppId}).success(function(data){
            ActivityIndicator.success("Backend stopped")
        }).error(setErrorInScope.bind($scope));
    }

    $scope.restartBackend = function(backend){
        DataikuAPI.webapps.restartBackend({projectKey:backend.projectKey, id:backend.webAppId}).success(function(data){
            ActivityIndicator.success("Backend start command sent")
        }).error(setErrorInScope.bind($scope));
    }
    $scope.refreshList();
});

app.controller("AdminMonitoringIntegrationsController", function($scope, DataikuAPI, ActivityIndicator, Dialogs, WT1) {
    $scope.hidePreviewColumn = true;
    $scope.noTags = true;
    $scope.noStar = true;
    $scope.noWatch = true;
    $scope.massDelete = true;
    $scope.massIntegrations = true;
    $scope.noDelete = true;

    $scope.sortBy = [
        { value: 'projectKey', label: 'Project name' },
        { value: 'integrationName', label: 'Integration type' },
        { value: 'integrationActive', label: 'Active' }
    ];
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            interest: {
                starred: '',
            },
        },
        filterParams: {
            userQueryTargets: ["projectKey", "name", "integrationName", "integrationProperties"],
            propertyRules: { },
        },
        orderQuery: "projectKey",
        orderReversed: false,
    }, $scope.selection || {});
    $scope.sortCookieKey = 'project-integrations';
    $scope.maxItems = 100;

    $scope.list = () => {
        WT1.event("refresh-project-integrations-list");
        DataikuAPI.admin.monitoring.getProjectsIntegrations().success((data) => {
            $scope.integrations = data;
            $scope.listItems = [];
            data.forEach(item => {
                item.integrations.forEach(integration => {
                    $scope.listItems.push({
                        ...integration,
                        integrationName: formatIntegrationName(integration.hook.type),
                        integrationActive: integration.active,
                        integrationDetails: integration.hook.configuration.webhookUrl || undefined,
                        integrationProperties: formatIntegrationProperties(integration),
                        projectKey: item.projectKey
                    });
                });
            });
        }).error(setErrorInScope.bind($scope));
    };
    $scope.list();

    $scope.toggleActive = function(item) {
        WT1.event("integration-save-active");
        DataikuAPI.admin.monitoring.saveProjectIntegration(item.projectKey, item).success(function(data){
            ActivityIndicator.success("Saved");
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteIntegration = (item) => {
        WT1.event("integration-delete");
        DataikuAPI.admin.monitoring.deleteProjectIntegration(item.projectKey, item).success(() => {
            ActivityIndicator.success("Saved");
        }).error(setErrorInScope.bind($scope));
    };

    $scope.massDeletion = (items) => {
        if(items.length < 1) return;
        Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the selected integrations?").then(function() {
            items.forEach((item) => {
                item.active = status;
                $scope.deleteIntegration(item);
                $scope.listItems = $scope.listItems.filter(s => s !== item)
            })
        });
    };

    $scope.allIntegrations = function(objects) {
        if (!objects) return;
        return objects.map(o => o.active).reduce(function(a,b){return a&&b;},true);
    };

    $scope.autoIntegrationsObjects = function(autoIntegrationsStatus, objects) {
        objects.forEach(function(object) {
            if (object.active === autoIntegrationsStatus) return;
            object.active = autoIntegrationsStatus;
            $scope.toggleActive(object);
        })
    };

    const formatIntegrationProperties = (integration) => {
        if(integration.hook.type === "github") {
            return "Repository: " + integration.hook.configuration.repository;
        }

        const labels = [];
        if(integration.hook.configuration.selection.commits)                     { labels.push("Git commits"); }
        if(integration.hook.configuration.selection.discussions)                 { labels.push("Discussions messages"); }
        if(integration.hook.configuration.selection.jobEnds)                     { labels.push("Build jobs ends"); }
        if(integration.hook.configuration.selection.jobStarts)                   { labels.push("Build jobs beginnings"); }
        if(integration.hook.configuration.selection.mlEnds)                      { labels.push("Analysis ML training ends"); }
        if(integration.hook.configuration.selection.mlStarts)                    { labels.push("Analysis ML training beginnings"); }
        if(integration.hook.configuration.selection.scenarioEnds)                { labels.push("Scenario ends"); }
        if(integration.hook.configuration.selection.scenarioStarts)              { labels.push("Scenario starts"); }
        if(integration.hook.configuration.selection.timelineEditionItems)        { labels.push("Objects editions"); }
        if(integration.hook.configuration.selection.timelineItemsExceptEditions) { labels.push("Objects creation / deletion / ..."); }
        if(integration.hook.configuration.selection.watchStar)                   { labels.push("Watch / Star"); }
        return labels.length > 0 ? "Sends on " + labels.join(", ") : "";
    };

    const  formatIntegrationName = (type) => {
        const typeMapping = {
            "msft-teams-project": "Microsoft Teams",
            "google-chat-project": "Google Chat",
            "github": "Github",
            "slack-project": "Slack"
        };
        return typeMapping[type] || type;
    }
});


app.controller("AdminMonitoringClusterTasksController", function($scope, $rootScope, $state, DataikuAPI, Fn, $filter, ConnectionsService){

	$scope.uiState = {}

    $scope.searchConnectionsWithDescriptions = ConnectionsService.searchConnectionsWithDescriptions;

    DataikuAPI.admin.connections.list().success(function(data) {
        var array = $filter("toArray")(data);
        var hasHDFS = array.filter(Fn.compose(Fn.prop("type"), Fn.eq("HDFS"))).length > 0;
        $scope.connections = array.filter(function(x){
            return x.type != "HDFS";
        }).map(function(x){
            return { "name"  :x.name , type  : x.type , "id" : x.name, "description": x.description }
        });
        if (hasHDFS) {
            $scope.connections.push({
                "name" : "Hadoop",
                "type" : "Hadoop",
                "id" : "HADOOP"
            })
        }
    }).error(setErrorInScope.bind($scope));

    $scope.fetchData = function fetchData(connectionId){
        DataikuAPI.admin.monitoring.getConnectionTasksHistory(connectionId).success(function(data){
            $scope.connectionData = data;

            data.lastTasks.forEach(function(t) {
                t.elapsedTime = t.endTime - t.startTime;
            })

            $scope.projectData = [];
            data.perProject.forEach(function(p) {
                p.types.forEach(function(t, i){
                    $scope.projectData.push(angular.extend(t, {projectKey : p.project, l : p.types.length}));
                });
            });

            $scope.userData = [];
            data.perUser.forEach(function(p) {
                p.types.forEach(function(t, i){
                    $scope.userData.push(angular.extend(t, {initiator : p.user, l : p.types.length}));
                });
            });

        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AdminMonitoringBackgroundTasksController", function ($scope, DataikuAPI, $rootScope, Dialogs, ProgressStackMessageBuilder, $timeout) {
    $scope.Math = window.Math; // for the display of the running time
    $scope.isLoading = false;

    $scope.uiState = {
        query: {task: {$:''}}, // with the filter pipe, search any sub-field of 'task'
        taskType: [], // task type filter
        sortColumn: 'type', sortDescending: false
    }

    $scope.tasks = [], // {type, task, since, canAbort, abortLabel, abortModalTitle, abort}

    $scope.taskTypeOptions = [ // value must match the associated $scope.uiState.tasks types
        { value: 'future', label: 'Long task' },
        { value: 'notebook', label: 'Notebook' },
        { value: 'job', label: 'Job' },
        { value: 'scenario', label: 'Scenario' },
        { value: 'clusterKernel', label: 'Hive & Scala validation' },
    ];

    const isScenarioFuture = (future) => future?.payload?.action === 'run_scenario';
    $scope.filterOnType = (value) => $scope.uiState.taskType.length === 0 || $scope.uiState.taskType.includes(value.type);
    $scope.labelForType = (value) => $scope.taskTypeOptions.find(t => t.value === value)?.label;

    $scope.refreshList = function() {
        $scope.isLoading = true;
        $scope.tasks = [];
        DataikuAPI.running.listAll().success(function(data) {
            $scope.tasks = [
                ...data.futures.map(future => ({
                    type: isScenarioFuture(future) ? 'scenario' : 'future',
                    task: future,
                    since: future.alive && !future.aborted ? future.startTime : undefined,
                    abortLabel: 'Abort',
                    canAbort: !future.aborted && future.alive && future.abortable,
                    abortModalTitle: isScenarioFuture(future) ? 'Abort the scenario?' : 'Abort the task?',
                    abort: () => DataikuAPI.futures.abort(future.jobId)
                })),
                ...data.notebooks.map(notebook => ({
                    type: 'notebook',
                    task: notebook,
                    since: notebook.activeSessions?.[0].sessionStartTime,
                    abortLabel: 'Unload',
                    canAbort: notebook.activeSessions.length,
                    abortModalTitle: 'Unload the notebook?',
                    abort: () => DataikuAPI.jupyterNotebooks.unload(notebook.activeSessions[0]?.sessionId)
                })),
                ...data.jobs.map(job => ({
                    type: 'job',
                    task: job,
                    since: job.state === 'PENDING' ? job.def.initiationTimestamp : job.startTime,
                    abortLabel: 'Abort',
                    canAbort: !job.stableState,
                    abortModalTitle: 'Abort the job?',
                    abort: () => DataikuAPI.flow.jobs.abort(job.def.projectKey, job.def.id)
                })),
                ...data.clusterKernels.map(clusterKernel => ({
                    type: 'clusterKernel',
                    task: clusterKernel,
                    since: clusterKernel.aborted ? undefined : clusterKernel.lastTouched,
                    abortLabel: 'Abort',
                    canAbort: !clusterKernel.aborted,
                    abortModalTitle: 'Abort the validation kernel?',
                    abort: () => {
                        clusterKernel.aborted = true;
                        return DataikuAPI.admin.clusters.abortKernel(clusterKernel.prefix, clusterKernel.kernelId)
                    }
                }))
            ];
        }).error(setErrorInScope.bind($scope))
        .finally(() => $scope.isLoading = false);
    };

    $scope.abort = (item) => {
        Dialogs.confirm($scope, item.abortModalTitle)
            .then(() => {
                item.canAbort = false; // disables the button while the abort / refresh is running
                item.abort()
                    .success($scope.refreshList)
                    .error(setErrorInScope.bind($scope))
            });
    }

    $scope.refreshList();
});


app.controller("AdminMonitoringConnectionDataController", function($scope, $rootScope, $state, DataikuAPI, Fn, $filter, CreateModalFromTemplate, FutureProgressModal, InfoMessagesModal, Dialogs, DatasetsService, ConnectionsService, ManagedFoldersService){

    $scope.uiState = {}

    $scope.searchConnectionsWithDescriptions = ConnectionsService.searchConnectionsWithDescriptions;

    DataikuAPI.admin.connections.list().success(function(data) {
        var array = $filter("toArray")(data);
        $scope.connections = array.map(function(x){
            return { "name"  :x.name , type  : x.type , "id" : x.name, "description": x.description }
        });
    }).error(setErrorInScope.bind($scope));

    $scope.fetchData = function fetchData(connectionId){
        DataikuAPI.admin.monitoring.connectionData.get(connectionId).success(function(data){
            $scope.connectionData = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.focusOnProject = function(connectionId, projectKey) {
         DataikuAPI.admin.monitoring.connectionData.getForProject(connectionId, projectKey).success(function(data){
            $scope.projectData = data;
        }).error(setErrorInScope.bind($scope));
    }
    $scope.clearProjectData = function(){
        $scope.projectData = null;
    }

    $scope.updateForProject = function(connectionId, projectKey){
        CreateModalFromTemplate("/templates/admin/monitoring/connection-data-update-confirm.html", $scope, null, function(newScope){
            newScope.settings = {
                computeRecords : false,
                forceRecompute : false
            }
            newScope.go = function(){
                DataikuAPI.admin.monitoring.connectionData.updateForProject(
                    connectionId, projectKey, newScope.settings.computeRecords, newScope.settings.forceRecompute).success(function(data){
                    FutureProgressModal.show($scope, data, "Datasets update").then(function(result){
                        Dialogs.infoMessagesDisplayOnly($scope, "Update result", result);
                        $scope.fetchData(connectionId);
                    })
                    newScope.dismiss();
                });
            }
        });
    }

    $scope.updateSingleDataset = function(connectionId, projectKey, datasetName) {
        CreateModalFromTemplate("/templates/admin/monitoring/connection-data-update-confirm.html", $scope, null, function(newScope){
            newScope.settings = {
                computeRecords : false,
                forceRecompute : false
            }
            newScope.go = function(){
                DataikuAPI.admin.monitoring.connectionData.updateForDataset(
                    projectKey, datasetName, newScope.settings.computeRecords, newScope.settings.forceRecompute).success(function(data){
                    FutureProgressModal.show($scope, data, "Dataset update").then(function(result){
                        InfoMessagesModal.showIfNeeded($scope, result, "Update result");
                        $scope.focusOnProject(connectionId, projectKey);
                    })
                    newScope.dismiss();
                });
            }
        });
    };

    $scope.updateSingleManagedFolder = function(connectionId, projectKey, managedFolderName) {
        CreateModalFromTemplate("/templates/admin/monitoring/connection-data-update-managed-folder-confirm.html", $scope, null, function(newScope){
            newScope.settings = {
                forceRecompute : false
            }
            newScope.go = function(){
                DataikuAPI.admin.monitoring.connectionData.updateForManagedFolder(
                    projectKey, managedFolderName, newScope.settings.forceRecompute).success(function(data){
                    FutureProgressModal.show($scope, data, "Managed Folder update").then(function(result){
                        InfoMessagesModal.showIfNeeded($scope, result, "Update result");
                        $scope.focusOnProject(connectionId, projectKey);
                    })
                    newScope.dismiss();
                });
            }
        });
    };

    $scope.clearDataset = function(connectionId, projectKey, datasetName) {
        DatasetsService.clear($scope, projectKey, datasetName).then(function() {
            $scope.focusOnProject(connectionId, projectKey);
        });
    };

    $scope.clearManagedFolder = function(connectionId, projectKey, managedFolderId, managedFolderName) {
        ManagedFoldersService.clear($scope, projectKey, managedFolderId, managedFolderName).then(function() {
            $scope.focusOnProject(connectionId, projectKey);
        });
    };

});


})();
