(function(){
'use strict';

const app = angular.module('dataiku.analysis.mlcore');


app.factory("DeepHubParamsService", function() {
    const defaultOptimizers = [{
                                  value: "ADAM",
                                  description: "Adam"
                               },
                               {
                                  value: "SGD",
                                  description: "Stochastic Gradient Descent"
                               },
                               {
                                  value: "RMSPROP",
                                  description: "RMSprop"
                               },
                               {
                                  value: "ADAMAX",
                                  description: "Adamax"
                               },
                               {
                                  value: "ADAGRAD",
                                  description: "Adagrad"
                               },
                               {
                                  value: "ADADELTA",
                                  description: "Adadelta"
                               }]
    const defaultLRSchedulers = [{
                                    value: "PLATEAU",
                                    description: "On plateau"
                                 },
                                 {
                                    value: "STEP",
                                    description: "Step"
                                 },
                                 {
                                    value: "EXPONENTIAL",
                                    description: "Exponential"
                                 }]
    const paramsOptionsPerPredictionType = {
        "DEEP_HUB_IMAGE_OBJECT_DETECTION": {  // See  `ObjectDetectionPreTrainModelingParams`
            OPTIMIZERS: defaultOptimizers,
            LR_SCHEDULERS: defaultLRSchedulers,
            PRETRAINED_MODELS: [
                {
                    value: "FASTERRCNN",
                    description: "Faster R-CNN",
                    nbLayers: 5
                }
            ]
        },
        "DEEP_HUB_IMAGE_CLASSIFICATION": {  // See  `ImageClassificationPreTrainModelingParams`
            OPTIMIZERS: defaultOptimizers,
            LR_SCHEDULERS: defaultLRSchedulers,
            PRETRAINED_MODELS: [
                {
                    value: "EFFICIENTNET_B0",
                    description: "Lighter / Faster (EfficientNet B0)",
                    nbLayers: 8
                },
                {
                    value: "EFFICIENTNET_B4",
                    description: "Balanced (EfficientNet B4)",
                    nbLayers: 8
                },
                {
                    value: "EFFICIENTNET_B7",
                    description: "Performance-oriented (EfficientNet B7)",
                    nbLayers: 8
                }
            ]
        }
    };
    return {
        getParamsOptions: (predictionType) => {
           if (!(predictionType in paramsOptionsPerPredictionType)) {
               throw new Error("Unsupported prediction type: " + predictionType);
           }
           return paramsOptionsPerPredictionType[predictionType];
       },
       getNbTrainableLayers: (predictionType) => {
            if (!(predictionType in paramsOptionsPerPredictionType)) {
                throw new Error("Unsupported prediction type: " + predictionType);
            }
            return paramsOptionsPerPredictionType[predictionType]['PRETRAINED_MODELS'].reduce(function(map, obj) {
                       map[obj.value] = obj.nbLayers;
                       return map;
                   }, {});
       },
       getParamDescription: (predictionType, paramType, paramValue) => {

           if (!(predictionType in paramsOptionsPerPredictionType)) {
               throw new Error("Unsupported prediction type: " + predictionType);
           }
           if (!(paramType in paramsOptionsPerPredictionType[predictionType])) {
              throw new Error("Unsupported param type: " + paramType + " in prediction type " + predictionType);
           }
           for (const paramOption of paramsOptionsPerPredictionType[predictionType][paramType]) {
               if (paramOption.value === paramValue){
                    return paramOption.description;
               }
           }
           throw new Error(`Unsupported param value ${paramValue} for type: ${paramType} in prediction type ${predictionType}`);
       }
   }
})

app.controller("DeepHubPMLTaskDesignController", function($scope, $controller, CreateModalFromTemplate,
        PMLTrainTestPolicies, DeepHubMetricsService, DeepHubParamsService, $timeout) {
    $controller("_MLTaskDesignController",{$scope:$scope});

    $scope.evaluationMetrics = Object.entries(DeepHubMetricsService.metricNameToDescriptionMapPerPrediction($scope.mlTaskDesign.predictionType));
    $scope.confidenceScoreThresholdOptimMetrics = DeepHubMetricsService.confidenceScoreThresholdOptimMetricNameToDescription();
    $scope.trainTestPolicies = PMLTrainTestPolicies.trainTestPolicies;
    $scope.trainTestPoliciesDesc = PMLTrainTestPolicies.trainTestPoliciesDesc;

    $scope.uiState = $scope.uiState || {};
    $scope.uiState.selectedCategories = [];

    $scope.deephubParamsOptions = DeepHubParamsService.getParamsOptions($scope.mlTaskDesign.predictionType);
    $scope.deephubNbLayersPerModel = DeepHubParamsService.getNbTrainableLayers($scope.mlTaskDesign.predictionType);

    $scope.setSelectedCategories = function(category) {
        $scope.uiState.selectedCategories = [category];
    }

    $scope.onChangeTargetFeature = function() {
        if ($scope.dirtySettings()) {
            $scope.saveSettings();
        }
        CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
            newScope.paramKey = "targetVariable";
        });
    };

    // comes from Angular
    $scope.augmentationParamsChange = function(augmentationParams) {
        $timeout(function() {
            Object.assign($scope.mlTaskDesign.modeling.augmentationParams, augmentationParams);
        });
    };

    $scope.$watch('mlTaskDesign', function(nv, ov){
        if (nv) {
            $scope.uiState.targetVariable = nv.targetVariable;
            $scope.uiState.splitMethodDesc = nv.splitParams.ssdSplitMode==="SORTED" ? "Based on time variable" : "Randomly";
            $scope.retrieveCodeEnvsInfo();
        }
    });
});

app.controller("DeepHubPMLTargetRemappingController", function($scope, $controller, COLOR_PALETTES) {
    $scope.colors = COLOR_PALETTES.highContrast;
    $controller("_PMLTargetRemappingController",{ $scope: $scope });
});

app.controller("DeepHubPMLTaskPreTrainModal", function($scope, $stateParams, $controller, DataikuAPI, WT1, Logger) {
    $controller("PMLTaskPreTrainModal", { $scope });

    // For deep hub the settings can be updated from inside the modal (if you activate the GPU training), so we save
    // first before training
    $scope.train = () => { $scope.saveSettings().then($scope._doTrainThenResolveModal); };
    $scope._doTrain = function () {
        try {
            WT1.event("deephub-prediction-train", {
                backendType: $scope.mlTaskDesign.backendType,
                taskType: $scope.mlTaskDesign.taskType,
                predictionType: $scope.mlTaskDesign.predictionType,
                metrics: $scope.mlTaskDesign.modeling.metrics,
                hasSessionName: !!$scope.uiState.userSessionName,
                hasSessionDescription: !!$scope.uiState.userSessionDescription,
                dataAugmentationParams: $scope.mlTaskDesign.modeling.augmentationParams
            });
        } catch (e) {
            Logger.error('Failed to report mltask info', e);
        }
        return DataikuAPI.analysis.pml.trainStart($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId,
            $scope.uiState.userSessionName, $scope.uiState.userSessionDescription, $scope.uiState.forceRefresh, false).error(setErrorInScope.bind($scope));
    };
});


app.controller("DeepHubPMLTaskResultController", function($scope, $controller, PMLSettings, PMLFilteringService) {

    angular.extend($scope, PMLSettings.taskF($scope.mlTasksContext.activeMLTask.backendType));
    angular.extend($scope, PMLSettings.task);
    $scope.metricMap = PMLFilteringService.metricMap;
    $controller("_MLTaskResultsController",{$scope:$scope});
    $controller("_PMLTaskResultController",{$scope:$scope});
    $controller("_DeepLearningPMLTaskResultController",{$scope:$scope});

    $scope.canShowTensorboard = function () {
        return false;
    };

    $scope.anySessionModelHasOptimizationResults = function () {
        return ($scope.selection.sessionModels || []).some(function (x) {
            return x.modelTrainingInfo;
        });
    };
});



app.controller("DeepHubPMLTaskBaseController", function($scope, $controller) {
    $controller("_PMLTrainSessionController", { $scope: $scope });
});

app.controller("DeepHubPMLModelReportController", function($scope) {
    $scope.uiState = $scope.uiState || {};
    $scope.uiState.iou = 0.5;  // for object detection
});

app.controller("DeepHubPMLReportSummaryController", function($scope, $controller, DeepHubParamsService) {
    $controller("_PMLReportSummaryController", {$scope:$scope});

    $scope.getParamDescription = DeepHubParamsService.getParamDescription;
    $scope.deephubNbLayersPerModel = DeepHubParamsService.getNbTrainableLayers($scope.modelData.coreParams.prediction_type);
});

app.component("objectDetectionMetricsReport", {
    bindings: {
        "perf": "<",
        "targetRemapping": "<"
    },
    templateUrl: "/templates/ml/prediction-model/deephub/object-detection-metrics.html",
    controller: function($scope, DeepHubMetricsService) {
        const ctrl = this;
        let categories;

        function buildMetricsTabData(perf) {
            const metricsResultTabData = [];
            metricsResultTabData.push(
                {
                    allClasses: true,
                    category: "All classes",
                    metrics: perf.metrics
                });

            perf.perCategoryMetrics.forEach((categoryMetrics, categoryIndex) => {
                metricsResultTabData.push({
                    category: categories[categoryIndex],
                    metrics: categoryMetrics
                })
            })
            return metricsResultTabData;
        }

        $scope.isSortMetric = metricInfo => $scope.uiState.sortMetric === metricInfo.name;

        $scope.sortByMetric = metricInfo => {
            const ss = $scope.selection;

            if ($scope.isSortMetric(metricInfo)) {
                ss.orderReversed = !ss.orderReversed;
            } else {
                ss.orderReversed = false;
                $scope.uiState.sortMetric = metricInfo.name;

                $scope.metricsResultTabData.forEach(categoryData => {
                    categoryData.sortMetric = categoryData.metrics[metricInfo.fieldName];
                });
            }
            $scope.$emit('refresh-list');
        }

        ctrl.$onInit = () => {
            categories = ctrl.targetRemapping.map(mapping => mapping.sourceValue);
            $scope.metricsResultTabData = buildMetricsTabData(ctrl.perf);
            $scope.metricsInfo = DeepHubMetricsService.objectDetectionMetricsInfo();

            $scope.uiState = {};

            // SETTINGS FOR FILTERING RESULTS

            $scope.selection = $.extend({
                filterParams: {
                    userQueryTargets: ["category"]
                },
                orderQuery: "-sortMetric"
            }, $scope.selection || {});
        }
    }
});

})();