(function () {
    'use strict';

    const app = angular.module('dataiku.analysis.mlcore');

    app.controller("PerForecastDistanceMetricsController", function ($scope, $stateParams, PerTimeseriesService, DataikuAPI, PMLSettings, PMLFilteringService, TimeseriesForecastingUtils, TimeseriesTableService) {
        $scope.loading = true;
        $scope.timeseriesFilters = {};
        $scope.uiState.timeseriesTableFoldFilters = [];
        $scope.uiState.showPerFold = false;
        $scope.metricsPerForecast = {};
        $scope.predictionLength = $scope.modelData.coreParams.predictionLength;
        const isDashboardTile = !!$stateParams.dashboardId;
        $scope.objectId = isDashboardTile ? `insightId.${$scope.insight.id}` : `fullModelId.${$scope.modelData.fullModelId}`;

        $scope.getPerforecastDistanceMetrics = function () {
            if ($scope.uiState.showPerFold) {
                $scope.getPerFoldData();
            } else {
                $scope.getPerTimeseriesData();
            }
        }

        $scope.getPerTimeseriesData = function () {
            if (!$scope.perTimeseriesMetrics) {
                DataikuAPI.ml.prediction.getPerTimeseriesMetrics($scope.modelData.fullModelId).then(({data}) => {
                    $scope.perTimeseriesMetrics = data.perTimeseries;
                    $scope.unparsedTimeseriesIdentifiers = Object.keys($scope.perTimeseriesMetrics);
                    $scope.identifierNb = Object.keys($scope.perTimeseriesMetrics).length;
                    $scope.parseTimeseriesIdentifiers();
                }).finally(() => {
                    $scope.loading = false;
                    $scope.hasPerForecastDistanceMetrics = "perForecastDistanceMetrics" in Object.values($scope.perTimeseriesMetrics)[0];
                    $scope.updateDisplayedMetrics()
                }).catch(setErrorInScope.bind($scope));
            } else {
                $scope.unparsedTimeseriesIdentifiers = Object.keys($scope.perTimeseriesMetrics);
                $scope.parseTimeseriesIdentifiers();
                $scope.updateDisplayedMetrics();
            }
        }

        $scope.getPerFoldData = function () {
            if (!$scope.perFoldData) {
                DataikuAPI.ml.prediction.getKfoldPerfs($scope.modelData.fullModelId).then(({data}) => {
                    $scope.perFoldData = data.perFoldMetrics;
                    $scope.foldIds = (Object.values($scope.perFoldData)[0].map((x) => x["foldId"]));
                    $scope.unparsedTimeseriesIdentifiers = Object.keys($scope.perFoldData);
                    $scope.parseTimeseriesIdentifiers();
                }).finally(() => {
                    $scope.hasPerForecastDistanceMetrics = "perForecastDistanceMetrics" in Object.values($scope.perFoldData)[0][0]['metrics'];
                    $scope.updateDisplayedMetrics();
                });
            } else {
                $scope.unparsedTimeseriesIdentifiers = Object.keys($scope.perFoldData);
                $scope.parseTimeseriesIdentifiers();
                $scope.updateDisplayedMetrics();
            }
        }

        $scope.parseTimeseriesIdentifiers = function () {
            if ($scope.unparsedTimeseriesIdentifiers[0] !== "__single_timeseries_identifier") {
                $scope.allTimeseriesIdentifierValuesMap = PerTimeseriesService.initTimeseriesIdentifiersValues($scope.modelData.coreParams.timeseriesIdentifiers);
                $scope.unparsedTimeseriesIdentifiers.forEach(function (unparsedTimeseriesIdentifier) {
                    const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
                    PerTimeseriesService.addIdentifierValues(parsedTimeseriesIdentifier, $scope.modelData.coreParams.timeseriesIdentifiers, $scope.allTimeseriesIdentifierValuesMap);
                });
                PerTimeseriesService.removeDuplicatesAndSortIdentifierValuesForFilterDropdowns($scope.allTimeseriesIdentifierValuesMap);
            }
        }

        $scope.updateDisplayedMetrics = function () {
            if (!$scope.hasPerForecastDistanceMetrics) {
                return;
            }
            let perIdentifierMetrics;
            if ($scope.uiState.showPerFold) {
                // aggregate per fold if needed
                perIdentifierMetrics = angular.copy($scope.perFoldData);
                perIdentifierMetrics = Object.fromEntries(Object.entries(perIdentifierMetrics).map(
                    ([identifier, values]) => [identifier, $scope.aggregateActiveFoldsMetrics(values)]));
            } else {
                perIdentifierMetrics = angular.copy($scope.perTimeseriesMetrics);
            }
            for (const [identifier, values] of Object.entries($scope.timeseriesFilters)) {
                // keep only the selected identifiers if they are some
                if (values.length > 0) {
                    perIdentifierMetrics = Object.fromEntries(
                        Object.entries(perIdentifierMetrics).filter(([key]) => {
                            const entry = JSON.parse(key);
                            return values.includes(entry[identifier]);
                        })
                    );
                }
            }
            let perIdentifierPerForecastDistanceMetrics = Object.fromEntries(Object.entries(perIdentifierMetrics)
                .map(([key, value]) => [key, value["perForecastDistanceMetrics"]]));

            // aggregate per identifier
            let perForecastDistanceMetrics = $scope.aggregateActiveIdentifiersMetrics(perIdentifierPerForecastDistanceMetrics);
            $scope.uiState.perForecastDistanceMetrics = $scope.formatPerForecastDistanceMetrics(perForecastDistanceMetrics)


            if (!$scope.uiState.tableHeaders) {
                $scope.getHeaders();
            }

            // update the export data
            $scope.$emit('timeseriesTableEvent', {
                headers: $scope.uiState.tableHeaders,
                rows: $scope.uiState.perForecastDistanceMetrics
            });
            if (!$scope.uiState.chartSelectableMetrics) {
                $scope.uiState.chartSelectableMetrics = Object.keys($scope.uiState.perForecastDistanceMetrics[0]).filter(key => !["computePointNb", "forecastDistance", "usedInEvaluation"].includes(key))
                    .map(metricRawField => ({
                        rawField: metricRawField,
                        label: $scope.getMetricHeaderFromRawField(metricRawField)
                    }));
                $scope.uiState.chartSelectedMetric = $scope.uiState.chartSelectableMetrics[0].rawField;
            }
            $scope.updateChart();
        }

        $scope.updateChart = function () {
            $scope.uiState.chartSeries = {
                data: $scope.uiState.perForecastDistanceMetrics.map(x => [x["forecastDistance"].rawValue, x[$scope.uiState.chartSelectedMetric].rawValue]),
                emphasis: {disabled: true},
                lineStyle: {type: 'solid'},
                name: $scope.getMetricHeaderFromRawField($scope.uiState.chartSelectedMetric),
                symbol: 'circle',
                symbolSize: 10,
                type: 'line',
            };
            $scope.chartOptions = null;
            $scope.chartOptions = {
                tooltip: {
                    trigger: 'axis',
                    formatter: (params) => {
                        let html = '<table class="global-explanations-scatter__tooltip-table">';
                        html += '<tr><td><strong>Forecast Distance</strong></td><td>' + params[0].data[0] +
                            (params[0].data[0] - 1 < $scope.modelData.coreParams.evaluationParams.gapSize ? " (ignored for evaluation)" : "") + '</td></tr>'
                        html += '<tr><td><strong>' + params[0].seriesName + '</strong></td><td>' + PerTimeseriesService.createMetricRow(params[0].data[1], null, params[0].seriesName) + '</td></tr>'
                        html += '<tr><td><strong> Sample size </strong></td><td>' + $scope.uiState.perForecastDistanceMetrics[params[0].data[0] - 1].computePointNb.displayValue + '</td></tr>'
                        html += "</table>";
                        return html;
                    },

                },
                xAxis: {name: "Forecast Distance", nameLocation: "middle", scale: true, nameGap: 32},
                yAxis: {
                    axisLabel: { formatter: val => (Math.abs(val) >= 1000 || Math.abs(val) < 0.001 && val !== 0) ? val.toExponential(2) : val },
                    name: $scope.getMetricHeaderFromRawField($scope.uiState.chartSelectedMetric),
                    nameLocation: "middle",
                    scale: true,
                    nameGap: 48
                },
                animation: true,
                grid: {bottom: 64, top: 64, left: 64, right: 64},
                textStyle: {fontFamily: 'SourceSansPro'},
                series: [$scope.uiState.chartSeries],
                title: {
                    text: $scope.getMetricHeaderFromRawField($scope.uiState.chartSelectedMetric) + " over forecast distance",
                    textStyle: {fontWeight: 'bold', fontSize: 16},
                },
            };
        }

        $scope.aggregateActiveFoldsMetrics = function (values) {
            let active_folds_metrics = values.filter(fold_metric => $scope.uiState.timeseriesTableFoldFilters.length === 0 || $scope.uiState.timeseriesTableFoldFilters.includes(fold_metric["foldId"])).map(fold_metric => fold_metric);
            let metric_names = Array.from(new Set(
                active_folds_metrics.flatMap(fold =>
                    Object.keys(fold.metrics.perForecastDistanceMetrics[$scope.predictionLength])
                )
            ));
            let perForecastDistanceMetrics = {};
            for (let forecastDistance = 1; forecastDistance < $scope.predictionLength + 1; forecastDistance++) {
                for (let metric_name of metric_names) {
                    if (metric_name !== "customMetricsResults") {
                        let points = active_folds_metrics.map(fold_metrics => fold_metrics["metrics"]["perForecastDistanceMetrics"][forecastDistance][metric_name]);
                        points = points.filter(val => val !== undefined && val !== null);
                        if (!(forecastDistance in perForecastDistanceMetrics)) {
                            perForecastDistanceMetrics[forecastDistance] = {};
                        }
                        perForecastDistanceMetrics[forecastDistance][metric_name] = points.reduce((acc, val) => acc + val, 0);
                        if (metric_name !== "computePointNb") {
                            perForecastDistanceMetrics[forecastDistance][metric_name] = perForecastDistanceMetrics[forecastDistance][metric_name] / points.length;
                        }
                    }
                }
                $scope.addFoldCustomMetrics(perForecastDistanceMetrics[forecastDistance], forecastDistance, active_folds_metrics);
            }
            return {"perForecastDistanceMetrics": perForecastDistanceMetrics};
        }

        $scope.addFoldCustomMetrics = function (row, forecastDistance, active_folds_metrics) {
            const customMetrics = active_folds_metrics[0]["metrics"]["perForecastDistanceMetrics"][forecastDistance]["customMetricsResults"];
            if (!customMetrics || !customMetrics.length > 0) {
                return;
            }
            row["customMetricsResults"] = []
            for (let customMetricIdx = 0; customMetricIdx < customMetrics.length; customMetricIdx++) {
                let customMetricName = active_folds_metrics[0]["metrics"]["perForecastDistanceMetrics"][forecastDistance]["customMetricsResults"][customMetricIdx]["metric"]["name"]
                let points = active_folds_metrics.map(metrics => metrics["metrics"]["perForecastDistanceMetrics"][forecastDistance]["customMetricsResults"][customMetricIdx].value);
                points = points.filter(val => val !== undefined && val !== null);
                const sum = points.reduce((acc, val) => acc + val, 0);
                row["customMetricsResults"].push({metric: {name: customMetricName}, value: sum / points.length})
            }
        }

        $scope.aggregateActiveIdentifiersMetrics = function (perIdentifierPerForecastDistanceMetrics) {
            let metric_names = Object.keys(Object.values(perIdentifierPerForecastDistanceMetrics)[0][$scope.predictionLength]);
            let res = {};
            for (let forecastDistance = 1; forecastDistance < $scope.predictionLength + 1; forecastDistance++) {
                let row = {};
                for (let metric_name of metric_names) {
                    if (metric_name !== "customMetricsResults") {
                        let points = Object.values(perIdentifierPerForecastDistanceMetrics).map(metrics =>
                            ({
                                "value": metrics[forecastDistance][metric_name],
                                "weight": metrics[forecastDistance]["computePointNb"]
                            }));
                        points = points.filter(point => point.value !== undefined && point.value !== null && !isNaN(point.value));

                        if (metric_name !== "computePointNb") {
                            let weightedSum = 0;
                            let totalWeight = 0;
                            for (let point of points) {
                                weightedSum += point.value * point.weight;
                                totalWeight += point.weight;
                            }
                            row[metric_name] = weightedSum
                            row[metric_name] = weightedSum / totalWeight;
                        } else {
                            row[metric_name] = points.reduce((acc, val) => acc + val.value, 0);
                        }
                    }
                }
                row["forecastDistance"] = forecastDistance;
                $scope.addCustomMetricsToRow(row, forecastDistance, perIdentifierPerForecastDistanceMetrics);
                res[forecastDistance] = row;
            }
            return res;
        }

        $scope.formatPerForecastDistanceMetrics = function (perForecastDistanceMetrics) {
            let perForecastMetricsDistanceRows = []
            let metric_names = Object.keys(perForecastDistanceMetrics[$scope.predictionLength]);

            for (let forecastDistance in perForecastDistanceMetrics) {
                let row = {};
                for (let metric_name of metric_names) {
                    row[metric_name] = {
                        rawValue: perForecastDistanceMetrics[forecastDistance][metric_name],
                        displayValue: perForecastDistanceMetrics[forecastDistance][metric_name]
                    }
                }
                row["forecastDistance"] = {rawValue: parseInt(forecastDistance), displayValue: parseInt(forecastDistance)};
                perForecastMetricsDistanceRows.push(row);
            }
            return perForecastMetricsDistanceRows;
        }

        $scope.addCustomMetricsToRow = function (row, forecastDistance, perIdentifierPerForecastDistanceMetrics) {
            const customMetrics = Object.values(perIdentifierPerForecastDistanceMetrics)[0][forecastDistance]["customMetricsResults"];
            if (!customMetrics || !customMetrics.length > 0) {
                return;
            }
            for (let customMetricIdx = 0; customMetricIdx < customMetrics.length; customMetricIdx++) {
                let metricName = "custom-" + sanitize(Object.values(perIdentifierPerForecastDistanceMetrics)[0][forecastDistance]["customMetricsResults"][customMetricIdx].metric.name)
                let points = Object.values(perIdentifierPerForecastDistanceMetrics).map(metrics => metrics[forecastDistance]["customMetricsResults"][customMetricIdx].value);
                points = points.filter(val => val !== undefined && val !== null);
                const sum = points.reduce((acc, val) => acc + val, 0);
                row[metricName] = sum / points.length;
            }
        }

        $scope.getHeaders = function () {
            if ($scope.uiState.tableHeaders) {
                return $scope.uiState.tableHeaders;
            }
            let timeseriesEvaluationMetrics = PMLSettings.taskF().timeseriesEvaluationMetrics;
            timeseriesEvaluationMetrics.push(['Input points for the metric calculation', 'computePointNb'], ['Forecast Distance', 'forecastDistance']);

            // first columns
            const firsts = ['forecastDistance', 'computePointNb']
            let fields = Object.keys($scope.uiState.perForecastDistanceMetrics[0]).filter(field => !firsts.includes(field));
            fields = [...firsts, ...fields]

            $scope.uiState.tableHeaders = fields.map(metric => {
                let rawField = metric;
                let pinned;
                let valueFormatter = $scope.formatValue;
                let type;
                switch (metric) {
                    case "computePointNb":
                        pinned = "left";
                        // does not display on the aggregation row
                        valueFormatter = p => p.node.rowPinned === "bottom" ? "" : p.value;
                        type = "rightAligned";
                        break;
                    case "usedInEvaluation":
                        pinned = "left";
                        // does not display on the aggregation row
                        valueFormatter = p => p.value;
                        break;
                    case "forecastDistance":
                        pinned = "left";
                        // does not display on the aggregation row
                        type = "rightAligned";
                        valueFormatter = p => p.node.rowPinned === "bottom" ? "" : TimeseriesForecastingUtils.prettyTimeSteps(p.value, $scope.modelData.coreParams.timestepParams.timeunit);
                        break;
                    default:
                        type = "rightAligned";
                        if (!metric.startsWith("custom-")) {
                            rawField = PMLFilteringService.metricMap[metric.toUpperCase()];
                        } else {
                            valueFormatter = p => $scope.formatValue(p, true);
                        }
                }
                return {
                    headerName: $scope.getMetricHeaderFromRawField(metric), // used by ag-grid for the column name
                    field: rawField + ".rawValue", // used by ag-grid for the column value
                    rawField: rawField, // used by us to export
                    rawName: metric.toUpperCase(),
                    filter: 'agNumberColumnFilter',
                    pinned: pinned,
                    type: type,
                    valueFormatter: valueFormatter
                };
            })
            return $scope.uiState.tableHeaders;
        }

        $scope.getMetricHeaderFromRawField = function (rawField) {
            switch (rawField) {
                case "computePointNb":
                    return "Sample size";
                case "forecastDistance":
                    return "Forecast distance";
                case "usedInEvaluation":
                    return "Used in evaluation";
                default:
                    if (rawField.startsWith("custom-")) {
                        return rawField.slice(7);
                    }
                    return PMLSettings.names.evaluationMetrics[rawField.toUpperCase()];
            }
        }

        $scope.formatValue = function (p, isCustomMetric = false) {
            if (p.node.rowPinned === "bottom") {
                return p.data[p.colDef.rawField].displayValue;
            }
            if (isCustomMetric) {
                // for custom metrics, we directly retrieve the value to prevent issues with custom metrics containing "."
                return PerTimeseriesService.createMetricRow(p.data[p.colDef.rawField].displayValue, null, p.colDef.rawName);
            }
            return PerTimeseriesService.createMetricRow(p.value, null, p.colDef.rawName);
        }

        $scope.$on('timeseriesIdentifiersFiltersUpdated', function (event, filters) {
            if ($scope.modelData.coreParams.timeseriesIdentifiers.length > 0) {
                $scope.timeseriesFilters = filters;
                $scope.updateDisplayedMetrics();
            }
        });

        $scope.onDisplayedRowUpdate = function (gridApi, headers) {
            let bottomRow = TimeseriesTableService.calculateAverageRow(gridApi, headers);
            gridApi.setGridOption('pinnedBottomRowData', bottomRow || []);
            headers.forEach(header => {
                header.tooltipValueGetter = (p) => {
                    if (p.node.rowPinned && header.headerName && p.colDef.pinned !== "left") {
                        return "Average (± stddev) value of " + header.headerName;
                    }
                    if (p.colDef.rawField === "computePointNb" && p.data.computePointNb.rawValue === 0) {
                        return "Ignored for evaluation (skipped)";
                    }
                }
            })
            gridApi.setGridOption('columnDefs', headers);
        }
    });
})();
