(function() {
'use strict';

const app = angular.module('dataiku.bundles.design', []);


app.controller("DesignBundleDetailsModalController", function($scope, $stateParams, $controller, Assert, DataikuAPI, ProgressStackMessageBuilder, FutureProgressModal) {
    $scope.uiState = {
        activeTab: "content"
    };

    $controller("_BundleGovernanceStatusController", {$scope:$scope});

    function fetch() {
        Assert.inScope($scope, 'bundleId');
        DataikuAPI.projects.design.getBundleDetails($stateParams.projectKey, $scope.bundleId).success(function(data) {
            $scope.bundleDetails = data;
        }).error(setErrorInScope.bind($scope))
    }

    $scope.$watch("bundleId", function(nv, ov) {
        if (!nv) return;
        fetch();
        $scope.getGovernanceStatus(nv);
    });

    $scope.$watch("bundleTab", function(nv, ov) {
        if (!nv) return;
        $scope.uiState.activeTab = nv;
    });
});


app.controller("DesignBundlesListController", function($scope, $controller, $stateParams, DataikuAPI, Dialogs, $state,
    $q, TopNav, Fn, CreateModalFromTemplate, ProgressStackMessageBuilder, FutureProgressModal, Assert) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    TopNav.setLocation(TopNav.TOP_MORE, "bundlesdesign", TopNav.TABS_NONE, null);
    TopNav.setNoItem();

    function exportedOn(bundle) { return ((bundle.exportManifest || {}).exportUserInfo || {}).exportedOn; }

    $scope.noTags = true;
    $scope.noWatch = true;
    $scope.noStar = true;
    $scope.sortBy = [
        { value: 'exportManifest.exportUserInfo.exportedOn', label: 'Date' },
        { value: 'bundleId', label: 'Name' },
    ];
    $scope.sortCookieKey = 'designbundles';
    $scope.selection = $.extend({
        filterQuery: {
            q: '',
        },
        filterParams: {userQueryTargets: "bundleId"},
        orderQuery: 'exportManifest.exportUserInfo.exportedOn',
        orderReversed: false,
    }, $scope.selection || {});
    $scope.noTags = true;

    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.projects.design.listBundles($stateParams.projectKey).success(function(data) {
            $scope.listItems = data.bundles;
            $scope.$broadcast('clearMultiSelect');

            $scope.listItems.forEach(function(x) {
                if (x.futureItem && x.futureItem.lastResponse && x.futureItem.lastResponse.progress) {
                    // not ProgressStackMessageBuilder.build(...), so pass the states instead of the progress
                    x.progressMsg = ProgressStackMessageBuilder.buildFull(x.futureItem.lastResponse.progress.states);
                }
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.list();

    if ($stateParams.showProgressModalFor) {
        FutureProgressModal.showPeekOnlyIfRunning($scope,
                    $stateParams.showProgressModalFor, "Exporting bundle ...").then($scope.list);
    }

    $scope.canNotPublishBundleMessage = function() {
        if (!$scope.appConfig.deployerClientEnabled) {
            return "Deployer is not enabled";
        }
        if (!$scope.appConfig.globalPermissions.mayCreatePublishedProjects) {
            return "You do not have the permission to create a published project";
        }
    }

    $scope.canPublishBundle = function() {
        return $scope.appConfig.deployerClientEnabled && $scope.appConfig.globalPermissions.mayCreatePublishedProjects;
    }

    // /* Specific actions */
    $scope.goToItem = function(data) {
        data.state === "BUILT" && $scope.showBundleDetails(data);
    };

    $scope.showBundleDetails = function(data) {
        CreateModalFromTemplate("/templates/bundles/design/details-modal.html", $scope, null, function(modalScope) {
            modalScope.bundleId = data.bundleId;
            modalScope.bundleTab = data.bundleTab;
            modalScope.$apply();
        });
    };

    if ($stateParams.bundleId) {
        $scope.showBundleDetails($stateParams);
    }

    $scope.downloadBundleArchive = function(bundle) {
        downloadURL(DataikuAPI.projects.design.getBundleDownloadURL($stateParams.projectKey, bundle.bundleId));
    };

    $scope.publishOnDeployer = function(bundle) {
        if ($scope.appConfig.deployerClientEnabled) {
            CreateModalFromTemplate("/templates/bundles/design/publish-on-deployer-modal.html", $scope, "PublishBundleOnDeployerModalController", function(newScope) {
                newScope.uploadParams = {bundleId: bundle.bundleId};

                newScope.refreshBundle = function(bundleId, publishedBundleState) {
                    const latestBundle = $scope.listItems.find(item => item.bundleId === bundleId);
                    latestBundle.publishedBundleState = publishedBundleState;
                }
            });
        }
    };

    $scope.setBundleExporterSettings = function() {
        CreateModalFromTemplate("/templates/bundles/design/exporter-settings-modal.html", $scope);
    };

    $scope.startRevert = function(bundleId) {
        DataikuAPI.projects.design.checkBundleReversion($stateParams.projectKey, bundleId).success(function(data) {
            CreateModalFromTemplate("/templates/bundles/design/reversion-check-result.html", $scope, null, function(modalScope) {
                modalScope.checkResult = data;
                modalScope.doRevert = function() {
                    Assert.inScope(modalScope, 'checkResult');
                    DataikuAPI.projects.design.revertBundle($stateParams.projectKey, modalScope.checkResult.bundleId, modalScope.importOptions).success(function(data) {
                        modalScope.dismiss();

                        FutureProgressModal.show($scope, data, "Activating bundle").then(function(activateResult) {
                            if (activateResult.anyMessage) {
                                Dialogs.infoMessagesDisplayOnly($scope, "Activation report", activateResult);
                            }

                            $scope.refreshProjectData();
                        });

                    }).error(setErrorInScope.bind(modalScope));
                }
            })
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteBundle = function(bundle) {
        Dialogs.confirmSimple($scope, "Delete bundle <strong>" + bundle.bundleId +"</strong>?").then(function() {
            DataikuAPI.projects.design.deleteBundle($stateParams.projectKey, bundle.bundleId)
                .success($scope.list.bind(null))
                .error(setErrorInScope.bind($scope));
        });
    };

    $scope.deleteFailedBundle = function(bundle) {
        if ($scope.selection.none) {
            return;
        } else if ($scope.selection.selectedObjects.length == 1) {
            let bundle = $scope.selection.selectedObject;
            Dialogs.confirmSimple($scope, "Delete failed bundle <strong>" + bundle.bundleId +"</strong>?").then(function() {
                DataikuAPI.projects.design.deleteFailedBundle($stateParams.projectKey, bundle.bundleId)
                    .success($scope.list.bind(null))
                    .error(setErrorInScope.bind($scope));
            });
        }
    };    

    $scope.deleteSelected = function() {
        if ($scope.selection.none) {
            return;
        } else if ($scope.selection.selectedObjects.length == 1) {
            if ($scope.selection.selectedObject.state ==='BUILT') {
                $scope.deleteBundle($scope.selection.selectedObject);
            } else {
                $scope.deleteFailedBundle($scope.selection.selectedObject);
            }
        } else {
            Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the selected bundles? Only successfully created bundles will be deleted.").then(function() {
                $q.all($scope.selection.selectedObjects.filter(bundle => bundle.state === 'BUILT').map(Fn.prop('bundleId'))
                    .map(DataikuAPI.projects.design.deleteBundle.bind(null, $stateParams.projectKey))
                ).then($scope.list.bind(null), setErrorInScope.bind($scope));
                $q.all($scope.selection.selectedObjects.filter(function(bundle){return bundle.state === 'FAILED' || bundle.state === 'ABORTED'})
                    .map(Fn.prop('bundleId')).map(DataikuAPI.projects.design.deleteFailedBundle.bind(null, $stateParams.projectKey))
                ).then($scope.list.bind(null), setErrorInScope.bind($scope));
            });
        }
    };
});

app.directive('bundlesDesignRightColumnSummary', function(){
    return {
        templateUrl :'/templates/bundles/design/right-column-summary.html'
    }
});

app.controller("PublishBundleOnDeployerModalController", function($scope, $stateParams, DataikuAPI, StringUtils, WT1) {
    $scope.publishedProjectKeys = [];
    $scope.isAlreadyDeployed = false;
    $scope.step = "confirmPublish";

    let suggestedProjectKey;
    $scope.$watch('uploadParams.targetProject.createProjectMessage', function(nv, ov) {
        if (nv) {
            $scope.uploadParams.targetProject.id = suggestedProjectKey;
            $scope.isAlreadyDeployed = suggestedProjectKey !== $stateParams.projectKey && $scope.targetProjectHasBundleId;
        } else {
            $scope.isAlreadyDeployed = false;
        }
    });

    DataikuAPI.projectdeployer.client.listPublishedProjects()
        .success(function(response) {
            $scope.publishedProjects = response.filter(projectStatus => projectStatus.canWrite).sort((a, b) => a.projectBasicInfo.name.localeCompare(b.projectBasicInfo.name));
            suggestedProjectKey = StringUtils.transmogrify($stateParams.projectKey,
                                                          $scope.publishedProjects.map(_ => _.projectBasicInfo.id),
                                                          (count, name) => `${name}_${count}`);
            $scope.publishedProjects.unshift({createProjectMessage: "Create a new project...", packages: [], projectBasicInfo: {id: suggestedProjectKey}});
            $scope.publishedProjectKeys = $scope.publishedProjects.map(function(projectStatus) {
                if (projectStatus.createProjectMessage || (projectStatus.projectBasicInfo.id === projectStatus.projectBasicInfo.name)) return "";
                return projectStatus.projectBasicInfo.id;
            });
            $scope.uploadParams.targetProject = $scope.publishedProjects.find(project => project.projectBasicInfo.id === $stateParams.projectKey);
            $scope.targetProjectHasBundleId = $scope.uploadParams.targetProject.packages.some(bundle => bundle.id === $scope.uploadParams.bundleId);
            if (!$scope.uploadParams.targetProject || $scope.targetProjectHasBundleId) {
                $scope.uploadParams.targetProject = $scope.publishedProjects[0];
            }
        })
        .error(setErrorInScope.bind($scope));

    $scope.ok = function() {
        $scope.step = "publishing";
        DataikuAPI.projects.design.publishToDeployer(
                $stateParams.projectKey,
                $scope.uploadParams.bundleId,
                $scope.uploadParams.targetProject.projectBasicInfo.id)
            .success(function(response) {
                $scope.step = "published";
                $scope.refreshBundle($scope.uploadParams.bundleId, response);
                WT1.event('project-deployer-publish-to-deployer');
            })
            .error(setErrorInScope.bind($scope));
    };
});

app.directive("bundleContentEditor", function(Collections, DataikuAPI, $stateParams, FeatureFlagsService, $rootScope, SavedModelsService) {
    return {
        templateUrl: "/templates/bundles/design/bundle-content-editor.html",
        scope: {
            "exporterSettings": "="
        },
        link: function($scope) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.featureFlagEnabled = FeatureFlagsService.featureFlagEnabled;
            function rebuildAvailableDatasets() {
                $scope.availableDatasets = $scope.rawHeaders.filter(function(x) {
                    return x.type != "JobsDB" && x.type != "StatsDB" && x.type != "Labels" && !x.type.startsWith("Sample_")
                }).map(function(dataset) {
                    return {
                        localProject : $stateParams.projectKey,
                        datasetType : dataset.type,
                        type : dataset.type,
                        usable: true,
                        smartName: dataset.name,
                        name : dataset.name
                    };
                });
                $scope.datasetsMap = Collections.indexByField($scope.availableDatasets, "name");
                // filter exported datasets to only have existing datasets
                $scope.exporterSettings.exportOptions.includedDatasetsData = $scope.exporterSettings.exportOptions.includedDatasetsData.filter(function(dataset) {
                    return dataset.name in $scope.datasetsMap;
                });
                $scope.exporterSettings.exportOptions.includedDatasetsData.forEach(function(dataset) {
                    $scope.datasetsMap[dataset.name].usable = false;
                });
            }

            DataikuAPI.datasets.listHeaders($stateParams.projectKey).success(function(data) {
                $scope.rawHeaders = data;
                rebuildAvailableDatasets();
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.labelingtasks.list($stateParams.projectKey).success(function(data) {
                $scope.availableLabelingTasks = data.map(function(labelingTask) {
                    return {
                        localProject : $stateParams.projectKey,
                        type : labelingTask.type,
                        datasetType: "",
                        usable: true,
                        smartName: labelingTask.id,
                        name : labelingTask.name,
                        id: labelingTask.id
                    };
                });
                $scope.labelingTasksMap = Collections.indexByField($scope.availableLabelingTasks, "id");
                $scope.exporterSettings.exportOptions.includedLabelingTasks = $scope.exporterSettings.exportOptions.includedLabelingTasks.filter(function(labelingTask) {
                    return labelingTask.id in $scope.labelingTasksMap;
                });
                $scope.exporterSettings.exportOptions.includedLabelingTasks.forEach(function(labelingTask) {
                    $scope.labelingTasksMap[labelingTask.id].usable = false;
                });
            }).error(setErrorInScope.bind($scope));
            
            DataikuAPI.managedfolder.list($stateParams.projectKey).success(function(data) {
                $scope.availableManagedFolders = data.map(function(folder) {
                    return {
                        localProject : $stateParams.projectKey,
                        type : folder.type,
                        datasetType: "",
                        usable: true,
                        smartName: folder.id,
                        name : folder.name,
                        id: folder.id
                    };
                });
                $scope.foldersMap = Collections.indexByField($scope.availableManagedFolders, "id");
                $scope.exporterSettings.exportOptions.includedManagedFolders = $scope.exporterSettings.exportOptions.includedManagedFolders.filter(function(folder) {
                    return folder.id in $scope.foldersMap;
                });
                $scope.exporterSettings.exportOptions.includedManagedFolders.forEach(function(folder) {
                    $scope.foldersMap[folder.id].usable = false;
                });
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.savedmodels.list($stateParams.projectKey).success(function(data) {
                $scope.availableSavedModels = data
                    .filter((sm) => SavedModelsService.hasModelData(sm))
                    .map(function (model) {
                        return {
                            localProject: $stateParams.projectKey,
                            type: (model.miniTask || {}).taskType || '',
                            datasetType: '',
                            usable: true,
                            smartName: model.id,
                            name: model.name,
                            id: model.id,
                            model: model,
                        };
                    });
                $scope.modelsMap = Collections.indexByField($scope.availableSavedModels, "id");
                $scope.exporterSettings.exportOptions.includedSavedModels = $scope.exporterSettings.exportOptions.includedSavedModels.filter(function(model) {
                    return model.id in $scope.modelsMap;
                });
                $scope.exporterSettings.exportOptions.includedSavedModels.forEach(function(model) {
                   $scope.modelsMap[model.id].usable = false;
                });
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.modelevaluationstores.list($stateParams.projectKey).success(function(data) {
                $scope.availableModelEvaluationStores = data.map(function(store) {
                    return {
                        localProject : $stateParams.projectKey,
                        type : "",
                        datasetType: "",
                        usable: true,
                        smartName: store.id,
                        name : store.name,
                        id: store.id,
                        store: store
                    };
                });
                $scope.storesMap = Collections.indexByField($scope.availableModelEvaluationStores, "id");
                $scope.exporterSettings.exportOptions.includedModelEvaluationStores = $scope.exporterSettings.exportOptions.includedModelEvaluationStores.filter(function(store) {
                    return store.id in $scope.storesMap;
                });
                $scope.exporterSettings.exportOptions.includedModelEvaluationStores.forEach(function(store) {
                   $scope.storesMap[store.id].usable = false;
                });
            }).error(setErrorInScope.bind($scope));
            
            DataikuAPI.codeStudioObjects.list($stateParams.projectKey).success(function(data) {
                $scope.availableCodeStudios = data.map(function( codeStudio) {
                    return {
                        localProject : $stateParams.projectKey,
                        type : codeStudio.templateId,
                        datasetType: "",
                        usable: true,
                        smartName: codeStudio.id,
                        name : codeStudio.name,
                        id: codeStudio.id,
                        codeStudio: codeStudio
                    };
                });
                $scope.codeStudiosMap = Collections.indexByField($scope.availableCodeStudios, "id");
                $scope.exporterSettings.exportOptions.includedCodeStudios = $scope.exporterSettings.exportOptions.includedCodeStudios.filter(function( codeStudio) {
                    return codeStudio.id in $scope.codeStudiosMap;
                });
                $scope.exporterSettings.exportOptions.includedCodeStudios.forEach(function( codeStudio) {
                   $scope.codeStudiosMap[codeStudio.id].usable = false;
                });
            }).error(setErrorInScope.bind($scope));
            
            DataikuAPI.retrievableknowledge.list($stateParams.projectKey).success(function(data) {
                $scope.availableKnowledgeBanks = data.map(function(knowledgeBank) {
                    return {
                        localProject : $stateParams.projectKey,
                        type : "",
                        datasetType: "",
                        usable: true,
                        smartName: knowledgeBank.id,
                        name : knowledgeBank.name,
                        id: knowledgeBank.id,
                        knowledgeBank: knowledgeBank
                    };
                });
                $scope.knowledgeBankMap = Collections.indexByField($scope.availableKnowledgeBanks, "id");
                $scope.exporterSettings.exportOptions.includedKnowledgeBanks = $scope.exporterSettings.exportOptions.includedKnowledgeBanks.filter(function(knowledgeBank) {
                    return knowledgeBank.id in $scope.knowledgeBankMap;
                });
                $scope.exporterSettings.exportOptions.includedKnowledgeBanks.forEach(function(knowledgeBank) {
                   $scope.knowledgeBankMap[knowledgeBank.id].usable = false;
                });
            }).error(setErrorInScope.bind($scope));

            $scope.addDataset = {};
            $scope.addSavedModel = {};
            $scope.addManagedFolder = {};
            $scope.addModelEvaluationStore = {};
            $scope.addCodeStudio = {};
            $scope.addLabelingTask = {};
            $scope.addKnowledgeBank = {};

            $scope.$watch("addDataset.dataset", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedDatasetsData.unshift($scope.datasetsMap[nv]);
                    rebuildAvailableDatasets();
                    $scope.datasetsMap[nv].usable = false;
                    $scope.addDataset.dataset = null;
                }
            });

            $scope.$watch("addManagedFolder.folder", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedManagedFolders.unshift($scope.foldersMap[nv]);
                    $scope.foldersMap[nv].usable = false;
                    $scope.addManagedFolder.folder = null;
                }
            });
            $scope.$watch("addLabelingTask.labelingTask", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedLabelingTasks.unshift($scope.labelingTasksMap[nv]);
                    $scope.labelingTasksMap[nv].usable = false;
                    $scope.addLabelingTask.labelingTask = null;
                }
            });

            $scope.$watch("addSavedModel.model", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedSavedModels.unshift($scope.modelsMap[nv]);
                    $scope.modelsMap[nv].usable = false;
                    $scope.addSavedModel.model = null;
                }
            });
            
            $scope.$watch("addModelEvaluationStore.store", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedModelEvaluationStores.unshift($scope.storesMap[nv]);
                    $scope.storesMap[nv].usable = false;
                    $scope.addModelEvaluationStore.store = null;
                }
            });

            $scope.$watch("addCodeStudio.codeStudio", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedCodeStudios.unshift($scope.codeStudiosMap[nv]);
                    $scope.codeStudiosMap[nv].usable = false;
                    $scope.addCodeStudio.codeStudio = null;
                }
            });

            $scope.$watch("addKnowledgeBank.knowledgeBank", function(nv) {
                if (nv) {
                    $scope.exporterSettings.exportOptions.includedKnowledgeBanks.unshift($scope.knowledgeBankMap[nv]);
                    $scope.knowledgeBankMap[nv].usable = false;
                    $scope.addKnowledgeBank.knowledgeBank = null;
                }
            });
            
        }
    }
});


app.controller("DesignBundleContentModalController", function($scope, DataikuAPI, $stateParams, $controller) {

    $controller("_BundleNotebookExportController", {$scope:$scope});

    DataikuAPI.projects.design.getBundleExporterSettings($stateParams.projectKey).success(function(data) {
        $scope.exporterSettings = data;
        $scope.uiState.notebookExportCurrentOption = $scope.getNotebookExport(data.exportOptions);
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.datasets.listHeaders($stateParams.projectKey).success(function(data) {
        $scope.availableDatasets = data.map(function(dataset) {
            return {
                localProject: $stateParams.projectKey,
                datasetType: dataset.type,
                type: dataset.type
            };
        });
    }).error(setErrorInScope.bind($scope));

    $scope.save = function() {
        DataikuAPI.projects.design.saveBundleExporterSettings($stateParams.projectKey,  $scope.exporterSettings).success(function(data) {
            $scope.dismiss();
        }).error(setErrorInScope.bind($scope));
    }
});


app.controller("DesignBundlesNewController", function($scope, $state, $stateParams,  Assert, DataikuAPI, TopNav, $controller, MonoFuture, $rootScope, Dialogs) {

    $controller("_BundleNotebookExportController", {$scope:$scope});

    TopNav.setLocation(TopNav.TOP_HOME, "bundlesdesign", TopNav.TABS_NONE, null);
    TopNav.setNoItem();

    Assert.trueish($stateParams.projectKey, "Not in a project");

    $scope.newBundle = {};
    function listBundlesIds() {
        DataikuAPI.projects.design.listBundles($stateParams.projectKey).success(function(data) {
            $scope.bundlesIDs = data.bundles.filter(b => (b.state === 'BUILT' || b.state === 'BUILDING')).map( bundle => bundle.bundleId);
        }).error(setErrorInScope.bind($scope));
    };
    listBundlesIds();
    
    $scope.dssExternalURL = $rootScope.appConfig.dssExternalURL;

    function realCreateBundle(evaluateProjectStandardsChecks) {
        DataikuAPI.projects.design.createBundle($stateParams.projectKey, $scope.newBundle.bundleId, $scope.preparationResult, evaluateProjectStandardsChecks).success(function(data) {
            $state.go("projects.project.bundlesdesign.list", {
                showProgressModalFor : data.jobId
            });
        }).error(function (data, status, headers) {
            listBundlesIds();
            $scope.newBundleForm.bundleId.$setValidity('unique', false);
            setErrorInScope.bind($scope)(data, status, headers);
        });
    }

    $scope.createBundle = function(evaluateProjectStandardsChecks) {
        if (!evaluateProjectStandardsChecks && $scope.appConfig.licensedFeatures.projectStandardsAllowed) {
            Dialogs.confirm($scope, "Bundle Project", "Are you sure that you want to bundle the project without computing the Project Standards Checks?")
                    .then(function() {
                        realCreateBundle(false);
                    });
        } else {
            realCreateBundle(evaluateProjectStandardsChecks);
        }
    };

    MonoFuture($scope).wrap(DataikuAPI.projects.design.prepareBundleCreation)($stateParams.projectKey).success(function(data) {
        $scope.preparationResult = data.result;
        $scope.uiState.notebookExportCurrentOption = $scope.getNotebookExport(data.result.exporterSettings.exportOptions);
        $scope.preparingFuture = null;
    }).update(function(data) {
        $scope.preparingFuture = data;
    }).error(function (data, status, headers) {
        $scope.preparingFuture = null;
        setErrorInScope.bind($scope)(data, status, headers);
    });
});


app.controller("DesignBundleCheckResultModalController", function($scope, DiffFormatter) {
    $scope.importOptions = {
        meaningsToImport: {}
    };

    $scope.$watch('checkResult', function(nv) {
        if (!nv) return;
        nv.messages.forEach(function(message) {
            if (message.udmId) {
                $scope.hasUDMConflict = true;
                $scope.importOptions.meaningsToImport[message.udmId] = false;
                message.formattedDiff = DiffFormatter.formatChange(message.diff);
            }
        });
    });
});

app.controller("_BundleGovernanceStatusController", function($scope, $stateParams, $rootScope, DataikuAPI) {
    $scope.getGovernanceStatus = function(bundleId) {
        $scope.bundleGovernanceStatus = undefined;
        if (!$rootScope.appConfig.governEnabled) return;
        if (!bundleId) return;
        $scope.bundleGovernanceStatus = { loading: true };
        DataikuAPI.projects.design.getBundleGovernanceStatus($stateParams.projectKey, bundleId).success(function(data) {
            $scope.bundleGovernanceStatus = { loading: false, data: data };
        }).error(function(a,b,c,d) {
            const fatalAPIError = getErrorDetails(a,b,c,d);
            fatalAPIError.html = getErrorHTMLFromDetails(fatalAPIError);
            $scope.bundleGovernanceStatus = { loading: false, error: fatalAPIError };
        });
    };
});

app.controller("_BundleNotebookExportController", function($scope) {
    $scope.uiState = {
        notebookExportOptions: [
            // FULL option
            { name: "Export with output", desc: "Export notebook definitions and outputs", exportNotebooks: true, exportNotebooksWithOutputs: true }, 
            // LIMITED option
            { name: "Export without output", desc: "Export only notebook definitions", exportNotebooks: true, exportNotebooksWithOutputs: false }, 
            // NONE option - should remain the last option for getNotebookExport to work as expected
            { name: "Do not export", desc: "Entirely exclude notebooks (outputs and definitions) from the bundle", exportNotebooks: false, exportNotebooksWithOutputs: false },
        ]
    }
    $scope.uiState.notebookExportCurrentOption = $scope.uiState.notebookExportOptions[0];

    $scope.updateNotebookExport = function( exportOptions) {
        exportOptions.exportNotebooks = $scope.uiState.notebookExportCurrentOption.exportNotebooks;
        exportOptions.exportNotebooksWithOutputs = $scope.uiState.notebookExportCurrentOption.exportNotebooksWithOutputs;
    }

    $scope.getNotebookExport = function(exportOptions) {
        return $scope.uiState.notebookExportOptions.find(o => o.exportNotebooks === exportOptions.exportNotebooks &&
             o.exportNotebooksWithOutputs === exportOptions.exportNotebooksWithOutputs) ?? $scope.uiState.notebookExportOptions.at(-1);
         
    }
});
})();
