(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Everything to store and manipulate dashboard filters.
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardsListController', function($scope, $controller, $stateParams, DataikuAPI, CreateModalFromTemplate, Dialogs, $state, $q, TopNav, Fn, $filter, ActivityIndicator, DashboardUtils, StateUtils, translate, DashboardTileSettingsService, DashboardsExportFormModalOpenService) {

        $controller('_TaggableObjectsListPageCommon', { $scope: $scope });
        $controller('DashboardsCommonController', { $scope: $scope });

        $scope.DashboardUtils = DashboardUtils;
        $scope.DashboardsExportFormModalOpenService = DashboardsExportFormModalOpenService;

        $scope.listHeads = DataikuAPI.dashboards.listHeads;
        $scope.getDashboardPromotionTitle = DashboardUtils.getTooltipPromotionTitleFunction('dashboard');

        $scope.sortBy = [
            { value: 'name', label: translate('PROJECT.DATASETS_LIST.SORT.NAME', 'Name') },
            { value: '-lastModifiedOn', label: translate('PROJECT.DATASETS_LIST.SORT.LAST_MODIFIED', 'Last modified') }
        ];

        $scope.selection = $.extend({
            filterQuery: {
                userQuery: '',
                tags: [],
                listed: '',
                interest: {
                    starred: ''
                }
            },
            filterParams: {
                userQueryTargets: ['name', 'tags'],
                propertyRules: { tag: 'tags' }
            },
            inclusiveFilter: {
                owner: []
            },
            customFilterWatch: 'selection.inclusiveFilter',
            customFilter: function(objList) {
                return objList.filter(function(obj) {
                    if ($scope.selection.inclusiveFilter.owner.length > 0) {
                        if ($scope.selection.inclusiveFilter.owner.indexOf(obj.owner) > -1) {
                            return true;
                        }
                        return false;
                    } else {
                        return true;
                    }
                });
            },
            orderQuery: '-lastModifiedOn',
            orderReversed: false
        }, $scope.selection || {});
        $scope.sortCookieKey = 'dashboards';
        $scope.maxItems = 20;

        $scope.setOwnerFilter = function(owner) {
            if (!owner) {
                $scope.selection.inclusiveFilter.owner = [];
                return;
            }

            const arr = $scope.selection.inclusiveFilter.owner;
            const index = arr.indexOf(owner);

            if (index > -1) {
                arr.splice(index, 1);
            } else {
                arr.push(owner);
            }
        };

        $scope.setListedFilterQuery = function(value) {
            $scope.selection.filterQuery.listed = value ? 'true' : '';
        };

        if ($state.current.name.indexOf('dashboards') != -1) {
            TopNav.setLocation(TopNav.TOP_DASHBOARD, 'dashboards', TopNav.TABS_NONE, null);
            TopNav.setNoItem();
        }

        $scope.list();

        $scope.$watch('selection.selectedObject', function(nv) {
            if (!nv) {
                return;
            }

            DataikuAPI.dashboards.getSummary($stateParams.projectKey, nv.id).success(function(data) {
                $scope.dashboard = data.object;
                $scope.dashboard._tileSpacing = $scope.dashboard.tileSpacing;
                Object.defineProperty($scope.dashboard, 'tileSpacing', {
                    get: function() {
                        return this._tileSpacing;
                    },
                    set: function(newTileSpacing) {
                        DashboardTileSettingsService.updateTileSpacing({
                            dashboard: $scope.dashboard,
                            newTileSpacing
                        });
                    }
                });
                $scope.dashboard.pages.forEach(page => {
                    /*
                     * client id or cid is a unique identifier automatically assigned
                     * when they're first created.
                     * Client ids are handy when the model has not yet been saved to the server,
                     * and does not yet have its eventual true id, but already needs to be visible in the UI.
                     */
                    page.cid = _.uniqueId();
                });
                $scope.origDashboard = angular.copy($scope.dashboard);
            }).error(setErrorInScope.bind($scope));
        });

        /* Specific actions */
        $scope.goToItem = function(data) {
            $state.go('projects.project.analyses.analysis.script', { projectKey: $stateParams.projectKey, analysisId: data.id });
        };

        $scope.newDashboard = function(language) {
            CreateModalFromTemplate('/templates/dashboards/new-dashboard-modal.html', $scope)
                .then(function(dashboard) {
                    StateUtils.go.dashboard(dashboard.id, dashboard.projectKey, { name: dashboard.name, tab: 'edit' });
                });
        };

        $scope.isAllListed = function(items) {
            if (!items) {
                return true;
            }
            return items.map(Fn.prop('listed')).reduce(function(a, b) {
                return a && b;
            }, true);
        };

        $scope.canMassMakeListed = true;
        $scope.massMakeListed = function(items, listed) {
            const apiCall = function() {
                DataikuAPI.dashboards.makeListed(items[0].projectKey, ids, listed)
                    .success(function(data) {
                        ActivityIndicator.success('Saved!');
                        $scope.list();
                    }).error(setErrorInScope.bind($scope));
            };

            if (!(items && items.length > 0)) {
                return;
            }
            var ids = [];
            const hasMissingReaderAuthorizationsItems = [];
            items.forEach(function(item) {
                if (item.listed != listed) {
                    ids.push(item.id);
                }
                if (item.hasMissingReaderAuthorizations) {
                    hasMissingReaderAuthorizationsItems.push(item);
                }
            });

            if (listed && hasMissingReaderAuthorizationsItems.length > 0) {
                CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                    newScope.initForDashboards(hasMissingReaderAuthorizationsItems, true);
                }).then(apiCall);
            } else {
                apiCall();
            }
        };

        // Used in the list
        $scope.canMassExportDashboard = true;

        $scope.owners = [];
        $scope.list = function() {
            $scope.listHeads($stateParams.projectKey, $scope.tagFilter).success(function(data) {
                $scope.filteredOut = data.filteredOut;
                $scope.listItems = data.items;
                $scope.restoreOriginalSelection();

                const ownersMap = {};
                data.items.forEach(function(dashboard) {
                    ownersMap[dashboard.owner] = {
                        login: dashboard.owner,
                        displayName: dashboard.ownerDisplayName
                    };
                });
                $scope.owners.length = 0;
                for (const login in ownersMap) {
                    $scope.owners.push(ownersMap[login]);
                }
                $scope.owners.sort(function(a, b) {
                    if (a.displayName < b.displayName) {
                        return -1;
                    }
                    if (a.displayName > b.displayName) {
                        return 1;
                    }
                    return 0;
                });

            }).error(setErrorInScope.bind($scope));
        };
        $scope.list();

        $scope.getNbListedDashboards = function(dashboards) {
            if (dashboards && dashboards.length > 0) {
                return dashboards.filter(function(dashboard) {
                    return dashboard.listed;
                }).length;
            }
            return 0;
        };
    });

})();
