(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('SavedModelReportInsightHandler', {
        name: 'Saved model report',
        desc: 'Full report of a model',
        i18nNameKey: 'INSIGHTS.SAVED_MODEL_REPORT.NAME',
        i18nDescKey: 'INSIGHTS.SAVED_MODEL_REPORT.DESC',
        icon: 'icon-dku-modelize',
        color: 'saved-model',

        getSourceId: function(insight) {
            return insight.params.savedModelSmartId;
        },
        sourceType: 'SAVED_MODEL',
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'summary'
        },
        defaultTileDimensions: [24, 12],
        filterSavedModel: function(taggableSavedModel) {
            return taggableSavedModel.subtype != 'LLM_GENERIC_RAW';
        }

    });

    app.controller('SavedModelReportViewCommon', function($scope, DataikuAPI, $controller, FullModelLikeIdUtils, WebAppsService,$state, $stateParams) {
        $scope.noMlReportTourHere = true; // the tabs needed for the tour are not present
        $scope.readOnly = true;
        $scope.noUrlChange = true;

        $scope._getSkins = function(versionId, contentType, algorithm, coreParams) {
            if (algorithm) {
                if (contentType && !contentType.endsWith('/')) {
                    contentType = contentType + '/';
                }
                contentType += algorithm.toLowerCase();
            }
            const skins = WebAppsService.getSkins(
                'SAVED_MODEL', versionId,
                { predictionType: coreParams.prediction_type, backendType: coreParams.backendType, contentType },
                $scope.staticModelSkins
            );
            return skins;
        };

        $scope.getModel = function(onLoadError) {
            const p = DataikuAPI.savedmodels.get($scope.insight.projectKey, $scope.insight.params.savedModelSmartId)
                .success(function(data) {
                    $scope.insight.$savedModel = data;
                    if (!$scope.insight.$savedModel.miniTask) return;
                    const version = $scope.insight.params.version || data.activeVersion;

                    $scope.insight.$fullModelId = FullModelLikeIdUtils.buildSavedModelFmi({
                        projectKey: data.projectKey,
                        savedModelId: data.id,
                        versionId: version
                    });
                    $scope.fullModelId = $scope.insight.$fullModelId;
                    const getDetailsP = DataikuAPI.ml[data.miniTask.taskType.toLowerCase()].getModelDetails($scope.fullModelId).success(function(modelData) {
                    /*
                     * For example, in the case of MLflow, coreParams may be undefined.
                     * The user will still be able to manually provide data.contentType through the model "settings" screen
                     */
                        const defaultValue = modelData.coreParams ? `${modelData.coreParams.taskType}/${modelData.coreParams.backendType}`.toLowerCase() : '';
                        const contentType = data.contentType ? data.contentType : defaultValue;
                        $scope.modelSkins = $scope._getSkins(
                            version, contentType, modelData.modeling.algorithm, modelData.coreParams
                        );
                    }).error(setErrorInScope.bind($scope));
                    if ($scope.noSpinner) {
                        getDetailsP.noSpinner();
                    }
                    /*
                     * TODO(jfy): is that good... ?
                     * set smId for saved models usage
                     * if using $state.go(".", ..., {reload: false}), there are exception due to backend calls because $stateParams.smId is not yet set
                     */
                    $stateParams.smId = data.id;
                    $scope.noSetLoc = true;
                    $scope.versionsContext = {};

                    switch (data.miniTask.taskType) {
                    // TODO: This could be better factorized with saved models to call the controller in the template or the route
                        case 'PREDICTION':
                            if ($scope.insight.$savedModel.miniTask.partitionedModel
                            && $scope.insight.$savedModel.miniTask.partitionedModel.enabled) {
                                $controller('PMLPartModelReportController', { $scope:$scope });
                            }
                            $controller('PredictionSavedModelReportController', { $scope:$scope });
                            break;
                        case 'CLUSTERING':
                            $controller('ClusteringSavedModelReportController', { $scope:$scope });
                            break;
                    }
                })
                .error(function(data, status, headers, config, statusText) {
                    if (typeof(onLoadError) === 'function') {
                        onLoadError(data, status, headers, config, statusText);
                    }
                });

            if ($scope.noSpinner) {
                p.noSpinner();
            }
        };

        $scope.isDeepHubPrediction = function(){
            return $scope.insight.$savedModel.miniTask.backendType === 'DEEP_HUB';
        };

        $scope.isCausalPrediction = function(){
            return ['CAUSAL_REGRESSION', 'CAUSAL_BINARY_CLASSIFICATION'].includes($scope.insight.$savedModel.miniTask.predictionType);
        };

    });

    app.directive('savedModelReportInsightTile', function($controller, DashboardUtils, TileLoadingState, SavedModelsService) {
        return {
            templateUrl: '/templates/dashboards/insights/saved-model_report/saved-model_report_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope){

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.onLoadSuccess = DashboardUtils.setLoaded.bind([$scope, resolve]);
                    $scope.onLoadError = DashboardUtils.setError.bind([$scope, reject]);
                    $scope.noSpinner = true;
                    $scope.getModel($scope.onLoadError);
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                $scope.isPartitionedModel = function() {
                    return $scope.insight.$modelData
                    && $scope.insight.$modelData.coreParams
                    && $scope.insight.$modelData.coreParams.partitionedModel
                    && $scope.insight.$modelData.coreParams.partitionedModel.enabled;
                };

                $scope.isExternalMLflowModel = function() {
                    return SavedModelsService.isExternalMLflowModel($scope.insight.$modelDat);
                };
                $scope.isMLflowModel = function() {
                    return SavedModelsService.isMLflowModel($scope.insight.$modelDat);
                };

                $controller('SavedModelReportViewCommon', { $scope:$scope });

                // Expose model data for savedModelReportInsightTileParams to display the appropriate tabs
                $scope.$watch('modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.$modelData = nv;
                });

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                $scope.noSkinControls = true; // no need to display controls widget in dashboards view
            }
        };
    });

    app.controller('SavedModelReportInsightController', function($scope, $controller, $stateParams, DataikuAPI, $state, MLModelsUIRouterStates, GoToStateNameSuffixIfBase, ComputeTransitionStateName) {
        const baseStateName = 'insight.view.savedmodels';

        $controller('SavedModelReportViewCommon', { $scope:$scope });


        const getSummarySuffix = () => {
            if ($scope.modelData.coreParams.taskType === 'PREDICTION') {
                return MLModelsUIRouterStates.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false);
            } else {
                return 'summary';
            }
        };

        const getRouteType = () => {
            return $scope.modelData.coreParams.taskType === 'PREDICTION' ? 'prediction' : 'clustering';
        };

        const updateUIRouterStateWithTarget = ({ stateName, stateParams }) => {
            const transitionStateName = ComputeTransitionStateName(baseStateName, stateName);

            if (transitionStateName || !angular.equals($state.current.params, stateParams)) {
                $state.go(transitionStateName, stateParams, { location: 'replace' });
            }
        };

        const getUIRouterStateNameTarget = () => {
            if (!$state.current.name.includes(baseStateName) || !$scope.modelData) {
                return baseStateName;
            }

            // if stateName is 'insight.view.savedmodels' redirection on 'insight.view.savedmodels.summary'
            if ($state.current.name.endsWith(baseStateName)) {
                return `${baseStateName}.${getRouteType()}.${getSummarySuffix()}`;
            }

            return $state.current.name.slice($state.current.name.lastIndexOf(baseStateName));
        };

        const getUIRouterStateNameAndParamsTargetFromTile = (tile) => {
            const stateNameTypePart = getRouteType();
            let stateParams = null;
            let stateNameTabPart;

            if (tile.tileParams.displayMode === 'skins') {

                if (tile.tileParams.advancedOptions && tile.tileParams.advancedOptions.customViews) {
                    stateNameTabPart = 'modelview';
                    stateParams = { skinId: tile.tileParams.advancedOptions.customViews.viewId };
                } else {
                    stateNameTabPart = getSummarySuffix();
                }

            } else {
                const pane = MLModelsUIRouterStates.dashboardTileToSavedModelPane(tile, $scope.modelData.coreParams.taskType === 'CLUSTERING', $scope);
                stateNameTabPart = pane.route;
                stateParams = pane.routeParams;
            }

            return {
                stateName: `${baseStateName}.${stateNameTypePart}.${stateNameTabPart}`,
                stateParams: stateParams
            };
        };


        $scope.$on('$stateChangeStart', function(event, toState, toParams, fromState, fromParams) {
            if (!$scope.modelData) {
                return;
            }
            GoToStateNameSuffixIfBase($state, toState, toParams, fromState, fromParams, '.insight.view', `savedmodels.${getRouteType()}.${getSummarySuffix()}`, event);
        });

        $scope.getModel(setErrorInScope.bind($scope));

        $scope.$watch('modelData', function(nv, ov) {
            if (!nv) {
                return;
            }
            if ($state.current.name.endsWith(baseStateName)) {
                updateUIRouterStateWithTarget({ stateName: getUIRouterStateNameTarget() });
            }

            if ($stateParams.originDashboard) {
                DataikuAPI.dashboards.getFullInfo($stateParams.projectKey, $stateParams.originDashboard.id).success(function(data) {
                    const tile = data.dashboard.pages.find(page => page.id === $stateParams.originDashboard.pageId).grid.tiles.find(tile => tile.insightId === $stateParams.insightId);
                    updateUIRouterStateWithTarget(getUIRouterStateNameAndParamsTargetFromTile(tile));
                });
            }
        });

        $scope.onLoadError = function(data, status, headers, config, statusText) {
            setErrorInScope.bind($scope)(data, status, headers, config, statusText);
        };

    });

    app.directive('savedModelReportInsightView', function($controller, $stateParams, DataikuAPI, $state) {
        return {
            scope: true,
            link: function($scope, element, attrs) {
                if ($state.current.name.endsWith('.insight.view')) {
                    $state.go('.savedmodels', null, { location: 'replace' });
                }
            }
        };
    });

    app.directive('savedModelReportInsightTileParams', function($controller, $timeout, DataikuAPI, FullModelLikeIdUtils, $q, ModelDataUtils){
        return {
            templateUrl: '/templates/dashboards/insights/saved-model_report/saved-model_report_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope, $element, attrs) {
                $scope.ModelDataUtils = ModelDataUtils;

                function setDefaultUpliftCurveDisplayMode() {
                    $scope.tileParams.advancedOptions = $scope.tileParams.advancedOptions || {};
                    $scope.tileParams.advancedOptions.upliftCurve = $scope.tileParams.advancedOptions.upliftCurve || {};
                    $scope.tileParams.advancedOptions.upliftCurve.displayMode = $scope.tileParams.advancedOptions.upliftCurve.displayMode || 'cumulativeUplift';
                };

                const setDefaultOverridesMetricsDisplayMode = function() {
                    // Setting default value of `tileParams.advancedOptions.overridesMetrics.displayMode` to "sankey" if not already set
                    $scope.tileParams.advancedOptions = $scope.tileParams.advancedOptions || {};
                    $scope.tileParams.advancedOptions.overridesMetrics = $scope.tileParams.advancedOptions.overridesMetrics || {};
                    $scope.tileParams.advancedOptions.overridesMetrics.displayMode = $scope.tileParams.advancedOptions.overridesMetrics.displayMode || 'sankey';
                };

                $scope.$watch('insight.$modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.modelData = nv;
                    $scope.fullModelId = $scope.insight.$fullModelId;

                    $controller('SavedModelReportViewCommon', { $scope:$scope });

                    function getVersionSkins(projectKey, smartId) {
                        const deferred = $q.defer();
                        DataikuAPI.savedmodels.get(projectKey, smartId)
                            .success(function(sm) {
                                const version = $scope.insight.params.version || sm.activeVersion;
                                if (!$scope.fullModelId) {
                                    $scope.fullModelId = FullModelLikeIdUtils.buildSavedModelFmi({
                                        projectKey: sm.projectKey,
                                        savedModelId: sm.id,
                                        versionId: version
                                    });
                                }
                                DataikuAPI.ml[sm.miniTask.taskType.toLowerCase()].getModelDetails($scope.fullModelId).success(function(modelDetails) {
                                    let contentType = sm.contentType;
                                    if (!contentType) {
                                        contentType = modelDetails.coreParams ? `${modelDetails.coreParams.taskType}/${modelDetails.coreParams.backendType}`.toLowerCase() : '';
                                    }
                                    // _getSkins() defined in SavedModelReportViewCommon
                                    deferred.resolve($scope._getSkins(
                                        version, contentType, modelDetails.modeling.algorithm, modelDetails.coreParams
                                    ));
                                }).error(setErrorInScope.bind($scope));
                            });
                        return deferred.promise;
                    };

                    getVersionSkins($scope.insight.projectKey, $scope.insight.params.savedModelSmartId).then(function(modelSkins) {
                        $scope.modelSkins = modelSkins;
                        $timeout(function() {
                            $scope.$broadcast('selectPickerRefresh');
                        });
                    });

                    // set default for tileParams.advancedOptions.interactiveScoring
                    if (!($scope.tileParams.advancedOptions && $scope.tileParams.advancedOptions.interactiveScoring) && $scope.insight.$savedModel.miniTask.taskType == 'PREDICTION') {
                        Promise.all([DataikuAPI.ml.prediction.getColumnImportance($scope.fullModelId),
                            DataikuAPI.ml.prediction.getSplitDesc($scope.fullModelId),
                            DataikuAPI.ml.prediction.getPreparationScript($scope.fullModelId),
                            DataikuAPI.ml.prediction.getInputDatasetSchema($scope.fullModelId).catch(e => e)]).then(
                            ([columnImportanceResp, splitDescResp, preparationScriptResp, inputDatasetSchemaResp]) => {
                                let featuresOrder;
                                if (columnImportanceResp.data) { // sort by importance
                                    const importances = columnImportanceResp.data.importances;
                                    const columns = columnImportanceResp.data.columns;
                                    featuresOrder = columns.sort((c1, c2) => importances[columns.indexOf(c2)] - importances[columns.indexOf(c1)]);
                                } else { // same order as in dataset
                                    const perFeature = $scope.modelData.preprocessing.per_feature;
                                    const inputColumns = Object.keys(perFeature).filter(featureName => perFeature[featureName].role === 'INPUT');
                                    featuresOrder = splitDescResp.data.schema.columns.map(c => c.name).filter(c => inputColumns.includes(c));
                                }
                                const hasPreparationSteps = preparationScriptResp.data.steps.some(step => !step.disabled);
                                if (hasPreparationSteps) {
                                    if (inputDatasetSchemaResp.data.columns) {
                                        const preScriptFeatures = inputDatasetSchemaResp.data.columns.map((col) => col.name);
                                        if (columnImportanceResp.data) {
                                            featuresOrder.push(...preScriptFeatures.filter(f => !featuresOrder.includes(f)));
                                        } else {
                                            featuresOrder = [...preScriptFeatures, ...featuresOrder.filter(f => !preScriptFeatures.includes(f))];
                                        }
                                    } else if (inputDatasetSchemaResp.status !== 404) {
                                    /*
                                     * 404 is expected when the model has no `input_dataset_schema.json` (old model)
                                     * and has no more origin analysis (deleted)
                                     */
                                        setErrorInScope.call($scope, inputDatasetSchemaResp);
                                    }
                                }

                                $scope.tileParams.advancedOptions = {
                                    ...$scope.tileParams.advancedOptions,
                                    interactiveScoring: {
                                        featuresOrder
                                    }
                                };
                            }
                        ).catch(setErrorInScope.bind($scope));
                    }

                    switch ($scope.insight.$savedModel.miniTask.taskType) {
                        case 'PREDICTION':
                            $controller('_PredictionModelReportController', { $scope:$scope });
                            break;
                        case 'CLUSTERING':
                            $controller('_ClusteringModelReportController', { $scope:$scope });
                            break;
                    }

                    $timeout(function() {
                        $element.find('.view-select').selectpicker('refresh');
                    });

                    /*
                     * tileParams.$displayModeExtended is used for models except deephub
                     * deephub uses regular displayMode
                     */
                    if ($scope.insight.$modelData.backendType !== 'DEEP_HUB') {
                        /**
                         * Returns the displayModeExtended to use for the given skin
                         */
                        $scope.displayModeExtendedForSkin = function(skinId) {
                            return 'skins_' + skinId;
                        };

                        /**
                         * Translates the displayModeExtended to the displayMode and the viewId
                         */
                        $scope.onDisplayModeExtendedChange = function() {
                            if (!$scope.tileParams || !$scope.tileParams.$displayModeExtended) {
                                return;
                            }
                            const displayModeExtended = $scope.tileParams.$displayModeExtended;
                            if (displayModeExtended.startsWith('skins_')) {
                                if (!$scope.tileParams.advancedOptions) {
                                    $scope.tileParams.advancedOptions = {};
                                }
                                if (!$scope.tileParams.advancedOptions.customViews) {
                                    $scope.tileParams.advancedOptions.customViews = {};
                                }
                                $scope.tileParams.advancedOptions.customViews.viewId = displayModeExtended.substring('skins_'.length);
                                $scope.tileParams.displayMode = 'skins';
                            } else {
                                $scope.tileParams.displayMode = displayModeExtended;
                            }
                            if (displayModeExtended === 'overrides_metrics') {
                                setDefaultOverridesMetricsDisplayMode();
                            }
                            if (displayModeExtended === 'uplift_curve') {
                                setDefaultUpliftCurveDisplayMode();
                            }
                        };
                        // Used for initializing the select
                        const deregister = $scope.$watch('tileParams', function(nv) {
                            if (!nv) {
                                return;
                            }
                            if ($scope.tileParams.displayMode == 'skins' && $scope.tileParams.advancedOptions && $scope.tileParams.advancedOptions.customViews) {
                                $scope.tileParams.$displayModeExtended = $scope.displayModeExtendedForSkin($scope.tileParams.advancedOptions.customViews.viewId);
                            } else {
                                $scope.tileParams.$displayModeExtended = $scope.tileParams.displayMode || 'summary';
                            }
                            deregister();
                        });
                    }

                    if ($scope.tileParams.displayMode === 'overrides_metrics') {
                        setDefaultOverridesMetricsDisplayMode();
                    }
                    setDefaultImportanceDisplayMode($scope.hasVariableImportance(), $scope.hasGlobalExplanations());
                });

                if ($scope.tileParams.displayMode === 'uplift_curve') {
                    setDefaultUpliftCurveDisplayMode();
                }

                if ($scope.tileParams.displayMode === 'overrides_metrics') {
                    setDefaultOverridesMetricsDisplayMode();
                }

                const setDefaultImportanceDisplayMode = function(hasVariableImportance, hasGlobalExplanations) {
                    let importanceDisplayMode = 'globalExplanations';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (hasVariableImportance) {
                        if (!tileParamsAdvancedOptions || !tileParamsAdvancedOptions.featureImportance) {
                            if (!hasGlobalExplanations) {
                                importanceDisplayMode = 'variableImportance';
                            }
                        } else {
                            importanceDisplayMode = tileParamsAdvancedOptions.featureImportance.importanceDisplayMode;
                        }
                    }
                    const advancedOptions = $scope.tileParams.advancedOptions || { featureImportance: {} };
                    $scope.tileParams.advancedOptions = {
                        ...advancedOptions,
                        featureImportance: {
                            ...advancedOptions.featureImportance,
                            importanceDisplayMode
                        }
                    };
                    setDefaultGraphType();
                };

                const setDefaultGraphType = function() {
                    let graphType = 'absoluteFeatureImportance';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (tileParamsAdvancedOptions && tileParamsAdvancedOptions.featureImportance.graphType) {
                        graphType = tileParamsAdvancedOptions.featureImportance.graphType;
                    }

                    $scope.tileParams.advancedOptions = {
                        ...tileParamsAdvancedOptions,
                        featureImportance: {
                            ...tileParamsAdvancedOptions.featureImportance,
                            graphType
                        }
                    };
                };
            }
        };
    });


    app.directive('savedModelReportInsightCreateForm', function(translate) {
        return {
            templateUrl: '/templates/dashboards/insights/saved-model_report/saved-model_report_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.defaultName = translate('DASHBOARD.NEW_TILE_MODAL.SAVED_MODEL', 'Saved model');
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = translate(
                        'DASHBOARD.NEW_TILE_MODAL.MODEL_EVALUATION_TABLE',
                        nv.label + ' table',
                        { label: nv.label }
                    );
                });

                /*
                 *$scope.$watch("insight.params.savedModelSmartId", function(nv) {
                 *    $scope.versions = [];
                 *    if (!nv) return;
                 *    DataikuAPI.savedmodels.listVersionIds($scope.insight.projectKey, $scope.insight.params.savedModelSmartId).success(function() {
                 *        $scope.versions = data;
                 *    });
                 *})
                 */
            }
        };
    });


})();
