(function() {
'use strict';

const app = angular.module('dataiku.logs', []);

const simpleLogsLink = function($interval, Dialogs) {
    return function($scope, element, attrs) {
        $scope.uiState = {
                logsQuery : ''
            };

        $scope.currentLogName = null;
        $scope.currentLog = null;
        $scope.cancelFetch = null;

        $scope.fetchLog = function(logName) {
            $scope.getLog()(...$scope.objId, logName).success(function(data) {
                $scope.currentLogName = logName;
                $scope.currentLog = data;
                $scope.$apply(); // necessary to refresh the logs when using the upgraded Angular component
            }).error(function(a, b, c) {
                $scope.stopPoll();
                setErrorInScope.bind($scope)(a, b, c);
            });
        }

        $scope.fetchLogRepeat = function(logName) {
            if ($scope.cancelFetch) {
                $interval.cancel($scope.cancelFetch);
            }
            $scope.fetchLog(logName);
            if (!$scope.preventPolling) {
                $scope.cancelFetch = $interval(function() {
                    $scope.fetchLog(logName);
                }, 10 * 1000);
            }
        };
        $scope.stopPoll = function() {
            if ($scope.cancelFetch) {
                $interval.cancel($scope.cancelFetch);
            }
            $scope.cancelFetch = null;
        }
        $scope.$on('$destroy', function () {
            $scope.stopPoll();
        });
        $scope.downloadLink = function(logName) {
            return $scope.downloadUrl()(...$scope.objId, logName);
        }

        $scope.downloadAllLink = function() {
            return $scope.downloadAllUrl()(...$scope.objId);
        }

        $scope.deleteLogFile = function(logName) {
            const title = 'Confirm log deletion';
            const message = 'Are you sure you want to delete this log file?';

            Dialogs.confirm($scope, title, message).then(() => {
                if (logName === $scope.currentLogName) {
                    $scope.currentLog = null;
                    $scope.currentLogName = null;
                }
                $scope.deleteLog()(...$scope.objId, logName);
            });
        }

        $scope.$watch("logs", function(nv,ov) {
            if (!$scope.logs) {
                $scope.currentLog = null;
                $scope.currentLogName = null;
            } else if ($scope.currentLogName && !$scope.logs.map(l => l.name).includes($scope.currentLogName)) {
                $scope.stopPoll();
                $scope.currentLogName = null;
                $scope.currentLog = null;
            } else if ($scope.logs.length === 1) {
                $scope.fetchLogRepeat($scope.logs[0].name);
            }
        });

        $scope.$watch('preventPolling', function(nv) {
            if (nv === true) {
                $scope.stopPoll();
            }
        })
    }
}

app.directive('simpleLogs', function($interval, Dialogs) {
    return {
        restrict : 'A',
        templateUrl : '/templates/simple-logs.html',
        replace : true,
        scope : {
                objId  : '=',
                logs : '=',
                getLog : '&',
                downloadUrl : '&',
                downloadAllUrl : '&?',
                deleteLog : '&?',
                hideFilter : '<?',
                preventPolling : '<?',
        },
        link : simpleLogsLink($interval, Dialogs)
    };
});

app.directive('simpleLogsTwoPanes', function($interval, Dialogs) {
    return {
        restrict : 'A',
        templateUrl : '/templates/simple-logs-two-panes.html',
        replace : true,
        scope : {
                objId  : '=',
                logs : '=',
                getLog : '&',
                downloadUrl : '&',
                downloadAllUrl : '&?',
                deleteLog : '&?'
        },
        link :  simpleLogsLink($interval, Dialogs)
    };
});

app.component('runLogs', {
    bindings: {
        api: '@',
        projectKey: '<',
        objectId: '<'
    },
    templateUrl: '/templates/run-logs.html',
    controller: function($scope, $timeout, DataikuAPI, ActivityIndicator) {
        const $ctrl = this;

        function reset() {
            $ctrl.runs = [];
            $ctrl.runsCount = 0;
            $ctrl.currentRun = null;
            $ctrl.currentLog = null;
            $ctrl.listLogs();
        }

        function scrollToLogsEnd() {
            var smartLogTailContent = $('.run-logs .smart-log-tail-content');
            if (smartLogTailContent.length > 0) {
                smartLogTailContent.scrollTop(smartLogTailContent[0].scrollHeight);
            }
        }

        $ctrl.$onInit = reset;
        const refreshRunLogsAndUpdateCurrentListenerStopFn = $scope.$on('refreshRunLogsAndUpdateCurrent', () => $ctrl.listLogs(true));
        const refreshRunLogsListenerStopFn = $scope.$on('refreshRunLogs', () => $ctrl.listLogs(false));
        $scope.$on("$destroy", () => {refreshRunLogsListenerStopFn(); refreshRunLogsAndUpdateCurrentListenerStopFn();});

        $ctrl.setupCurrentRun = function(){
            if ($ctrl.currentRun && $ctrl.currentRun.logFiles.length > 0) {
                let logFile;
                if ($ctrl.currentLog) {
                    // we stay on the same previous log file if available
                    const previousLogFilename = $ctrl.currentLog.filename;
                    logFile = $ctrl.currentRun.logFiles.find(log => log.filename === previousLogFilename);
                }
                if (!logFile) {
                    // we default on backend.log if available
                    logFile = $ctrl.currentRun.logFiles.find(log => log.filename === 'backend.log') || $ctrl.currentRun.logFiles[0];
                }
                $ctrl.setupCurrentLog(logFile);
            } else {
                $ctrl.currentLog = null;
            }
        };

        $ctrl.setupCurrentLog = function(log){
            $ctrl.currentLog = log;
            DataikuAPI[$ctrl.api].getLogTail($ctrl.projectKey, $ctrl.objectId, $ctrl.currentRun.runId, log.filename)
                .success(function(result) {
                    $ctrl.currentLog.logTail= result.logTail;
                    $timeout(scrollToLogsEnd);
                })
                .error(setErrorInScope.bind($scope));
        };

        $ctrl.listLogs = function(updateCurrent) {
            DataikuAPI[$ctrl.api].listLogs($ctrl.projectKey, $ctrl.objectId)
                .success(function(result) {
                    const previousRunId = $ctrl.currentRun && !updateCurrent ? $ctrl.currentRun.runId : null;
                    $ctrl.currentRun = null;
                    $ctrl.runs = result.runs.map(run => {
                        const [,y,m,d,h,min,sec,ms] = run.runId.split(/[_-]/);
                        run.label = y ? `${y}-${m}-${d} ${h}:${min}:${sec}.${ms}` : run.runId;
                        if (result.activeRunId && result.activeRunId === run.runId) {
                            run.label += ' (current)';
                        }
                        return run;
                    });
                    $ctrl.runsCount = result.runsCount;
                    if (previousRunId) {
                        // we stay on the same previous run if available
                        $ctrl.currentRun = $ctrl.runs.find(run => run.runId === previousRunId);
                    }
                    if (!$ctrl.currentRun) {
                        // else we default to the last run
                        $ctrl.currentRun = result.length == 0 ? null : $ctrl.runs[0];
                    }
                    $ctrl.setupCurrentRun();
                })
                .error(setErrorInScope.bind($scope));
        };

        $ctrl.getRawLogURL = log => $ctrl.currentLog && $ctrl.currentRun
            ? DataikuAPI[$ctrl.api].getRawLogURL($ctrl.projectKey, $ctrl.objectId, $ctrl.currentRun.runId, $ctrl.currentLog.filename)
            : '';

        $ctrl.downloadDiagnosis = () => {
            ActivityIndicator.success("Preparing diagnosis ...");
            downloadURL(DataikuAPI[$ctrl.api].getDiagnosisURL($ctrl.projectKey, $ctrl.objectId, $ctrl.currentRun.runId));
        };
    }
});

})();