(function() {
    'use strict';

    const app = angular.module('dataiku.ml.explainability');

    app.constant('ExplorationChartsConfig', {
        // If there are more than this number of categories, group them under "Other"
        MAX_NB_CATEGORIES: 8,
        // That's the name that will be displayed for the "Other" bar
        OTHER_CATEGORY_NAME: 'Other',
        // That's how we will secretly call the group containing the hidden categories if there are too many
        OTHER_CATEGORY_INTERNAL_NAME: '__DKU__OTHER_HIDDEN_CATEGORIES__'
    });

    app.constant('InteractiveModelCommand', {
        COUNTERFACTUALS: 'COUNTERFACTUALS',
        OUTCOME_OPTIMIZATION: 'OUTCOME_OPTIMIZATION',
        SCORING: 'SCORING',
        EXPLANATIONS: 'EXPLANATIONS'
    });

    app.constant('EmuType', {
        NUMERICAL: 'NUMERICAL',
        CATEGORICAL: 'CATEGORICAL',
        FROZEN: 'FROZEN'
    });

    app.constant('DistributionType', {
        NUMERIC: 'NUMERIC',
        CATEGORY: 'CATEGORY',
        VECTOR: 'VECTOR',
        TEXT: 'TEXT',
    });

    app.constant('OutcomeOptimizationSpecialTarget', {
        MIN: 'MIN',
        MAX: 'MAX'
    });

    app.constant('NO_TARGET_CLASS', '__DKU_NO_TARGET_CLASS__');

    app.constant('WhatIfView', {
        MAIN: 'MAIN',
        COMPARATOR: 'COMPARATOR',
        EXPLORATION_CONSTRAINTS: 'EXPLORATION_CONSTRAINTS',
        EXPLORATION_RESULTS: 'EXPLORATION_RESULTS'
    });

    app.factory('WhatIfRouter', function(WhatIfView) {
        class WhatIfRouter {
            constructor() {
                this._currentView = WhatIfView.MAIN;
            }
            openMainView() { this._currentView = WhatIfView.MAIN; }
            openComparator() { this._currentView = WhatIfView.COMPARATOR; }
            openConstraints() { this._currentView = WhatIfView.EXPLORATION_CONSTRAINTS; }
            openResults() { this._currentView = WhatIfView.EXPLORATION_RESULTS; }
            getCurrentView() { return this._currentView; }
        }
        return {
            build: () => new WhatIfRouter()
        };
    });

    // The following factory shares the features inputted by the user with the exploration-related controllers.
    app.factory('ExplorationLocalStore', ['LocalStorage', 'NO_TARGET_CLASS', 'OutcomeOptimizationSpecialTarget',
        (LocalStorage) => ((fmi, insight) => {
        const storeId = insight ? insight.id : fmi;
        // If some keys are missing in the local storage, some pages won't work
        const mandatoryVariablesForConstraintsPage = ['reference', 'score'];
        const mandatoryVariablesForResultsPage = ['featureDomains', 'reference', 'score'];
        const areVariablesStored = keys => keys
            .map(key => LocalStorage.get(getKey(key)))
            .every(e => e !== undefined);
        const noVersion = 'v2'; // update this const when you modify the format of the data to invalidate previous caches
        const getKey = variable => `dku.ml.interactivescoring.exploration.${noVersion}.${storeId}.${variable}`;
        return {
            setPredictionColorMap: predictionColorMap => LocalStorage.set(getKey('predictionColorMap'), JSON.stringify(predictionColorMap)),
            setTarget: target => LocalStorage.set(getKey('target'), target),
            setReference: reference => LocalStorage.set(getKey('reference'), reference),
            setFeatureDomains: featureDomains => LocalStorage.set(getKey('featureDomains'), featureDomains),
            setScore: score => LocalStorage.set(getKey('score'), JSON.stringify(score)),

            getPredictionColorMap: () => JSON.parse(LocalStorage.get(getKey('predictionColorMap')) || '{}'),
            getTarget: () => LocalStorage.get(getKey('target')), // no fallback if target not found
            getReference: () => LocalStorage.get(getKey('reference')) || {},
            getFeatureDomains: () => LocalStorage.get(getKey('featureDomains')) || [],
            getScore: () => JSON.parse(LocalStorage.get(getKey('score'))),

            canUseConstraintsPage: () => areVariablesStored(mandatoryVariablesForConstraintsPage),
            canUseResultsPage: () => areVariablesStored(mandatoryVariablesForResultsPage)
        };
    })]);

    app.factory('ConstraintHelper', (EmuType, DistributionType, $filter, DatasetTypesService) => ({
        init: (distribution, storageType) => { // the param's definition should match FeaturesDistributionAdapter.init.getSingle's
            // Returns a wrapper that represents the distribution of a single variable.
            // The object contains:
            //   - emuType: EmuType (numerical or categorical)
            //   - getDefaultConstraint: returns a default featureDomain
            const emuType = distribution.type === DistributionType.NUMERIC ? EmuType.NUMERICAL : EmuType.CATEGORICAL; // (here, we don't want type='FROZEN')
            const shouldAlwaysBeFrozen = () => {
                // some features should remain frozen (eg. TEXT)
                const isNumericOrCategorical = [DistributionType.NUMERIC, DistributionType.CATEGORY].includes(distribution.type);
                return DatasetTypesService.isTemporalType(storageType) || !isNumericOrCategorical;
            }
            const getDefaultConstraint = () => {
                // Create an array containing default featureDomain
                const featureDomain = { name: distribution.featureName };
                if (shouldAlwaysBeFrozen()) {
                    featureDomain.type = EmuType.FROZEN;
                    if (distribution.type === DistributionType.NUMERIC) {
                        featureDomain.minValue = 0;
                        featureDomain.maxValue = 0;
                    } else {
                        featureDomain.categories = [];
                    }
                }
                else if (distribution.type === DistributionType.NUMERIC) {
                    featureDomain.type = EmuType.NUMERICAL;
                    featureDomain.minValue = Math.min(...distribution.scale);
                    featureDomain.maxValue = Math.max(...distribution.scale);
                } else {
                    featureDomain.type = EmuType.CATEGORICAL;
                    featureDomain.categories = [...distribution.allCategories];
                }
                return featureDomain;
            };
            const getConstraintDescription = featureDomain => {
                switch (emuType) {
                    case EmuType.NUMERICAL:
                        return `Between ${$filter('smartNumber')(featureDomain.minValue)} and ${$filter('smartNumber')(featureDomain.maxValue)}`;
                    case EmuType.CATEGORICAL: {
                        const {nbDistinct, missingCount} = distribution.rawData;
                        const nbCategories = nbDistinct - (missingCount > 0 ? 1 : 0);
                        const nbEnabled = featureDomain.categories.length;
                        return (nbEnabled === nbCategories) ? 'All enabled' : `${nbEnabled}/${nbCategories} enabled`;
                    }
                }
            }
            return {
                shouldAlwaysBeFrozen,
                getDefaultConstraint,
                getConstraintDescription,
                emuType
            }
        }
    }));

    app.factory('FeaturesDistributionAdapter', (ExplorationChartsConfig, DataikuAPI, DistributionType) => {
        const _getTopNValuesAndIndices = (values, n) => {
            // Since there can be many values, we want to avoid a O(mlog(m)) algorithm, with `m` being the
            // length or the array. So instead of sorting the values and returning the last `n` elements,
            // we save the best `n` values as we iterate over the values. The resulting algo runs in O(m).
            const topValues = values.slice(0, n).map((v, i) => ({value: v, index: i}));
            topValues.sort((a, b) => a.value - b.value);
            for (let i = n; i < values.length; i++) {
                if (values[i] > topValues[0].value){
                    topValues.shift();
                    const insertAt = topValues.findIndex(v => values[i] < v.value);
                    topValues.splice(insertAt === -1 ? topValues.length : insertAt, 0, {value: values[i], index: i});
                }
            }
            return topValues;
        }

        let featuresDistributionCache = {};
        return {
            init: (fmi, storageTypes=undefined) => new Promise((resolve, reject) => {

                // Returns a wrapper that represents the distribution of a single variable.
                // The object contains:
                //   - rawData: data for the feature as returned by the Java
                //   - type: DistributionType (as written in the JSON file)
                //   - featureName: name of the corresponding feature
                //   - scale: marks on the X axis - after application of a filter to handle different kinds of features correctly (eg. few modalities, etc.)
                //   - distribution: number of occurrences for each group of the `scale`
                //   - other attributes specific to numerical or categorical data (shouldBeRepresentedWithHistograms, getBinIndex and allCategories)
                const getSingle = (featureName, otherCategorySecretName=ExplorationChartsConfig.OTHER_CATEGORY_INTERNAL_NAME) => {
                    const rawData = featuresDistributionCache[fmi][featureName]; // data as it was sent by Java
                    const type = rawData.type;
                    let adapter = {featureName, type, rawData};

                    if (type === DistributionType.NUMERIC) {
                        const shouldBeRepresentedWithHistograms = () => {
                            if (storageTypes && ['float', 'double'].includes(storageTypes[featureName])) {
                                return true;
                            }
                            const { nbDistinct, histogram } = rawData;
                            const minValue = histogram.scale[0];
                            const maxValue = histogram.scale[histogram.scale.length-1];
                            const rangeSize = 1 + maxValue - minValue;
                            return (nbDistinct > 50) || (rangeSize > 50);
                        };
                        adapter.getBinIndex = x => {
                            // Histograms are split in bins. Returns the index of the bin that contains the given number `x`
                            const bins = adapter.scale;
                            const index = bins.findIndex(bin => x < bin);
                            return index > -1 ? index - 1 : x ? bins.length - 1 : undefined;
                        };
                        adapter.shouldBeRepresentedWithHistograms = shouldBeRepresentedWithHistograms();
                    }
                    else {
                        adapter.allCategories = rawData.values.scale;
                    }

                    function _getCleanScaleAndDistribution() {
                        if (type === DistributionType.NUMERIC) {
                            // return histogram if many sparse categories, else, return top values
                            const { histogram, topValues } = rawData;
                            if (adapter.shouldBeRepresentedWithHistograms) {
                                return histogram;
                            }
                            const minValue = histogram.scale[0];
                            const maxValue = histogram.scale[histogram.scale.length - 1];
                            const rangeSize = 1 + maxValue - minValue;
                            const distribution = _.range(0, rangeSize, 0);
                            topValues.forEach(({ value, count }) => {
                                distribution[value - minValue] += count;
                            });
                            return {
                                scale: _.range(minValue, maxValue + 1, 1),
                                distribution: distribution
                            };
                        } else {
                            const categories = rawData.values;
                            if (categories.scale.length <= ExplorationChartsConfig.MAX_NB_CATEGORIES) {
                                return categories;
                            }
                            // We only keep the most common categories.
                            const topValues = _getTopNValuesAndIndices(categories.distribution, ExplorationChartsConfig.MAX_NB_CATEGORIES - 1);
                            // Sum of remaining distributions
                            const sumOfDisplayedDistributions = topValues.map(e => e.value).reduce((a, b) => a + b, 0);
                            return {
                                scale: topValues.map(e => categories.scale[e.index]).concat(otherCategorySecretName),
                                distribution: topValues.map(e => e.value).concat(1 - sumOfDisplayedDistributions)
                            };
                        }
                    }
                    adapter = {...adapter, ..._getCleanScaleAndDistribution()};
                    return adapter;
                }

                const methods = {
                    getNames: () => Object.keys(featuresDistributionCache[fmi]),
                    getSingle
                };
                if (featuresDistributionCache[fmi]) {
                    // featuresDistribution is cached already
                    resolve(methods);
                } else if (featuresDistributionCache[fmi] === null) {
                    // we cached the fact that featuresDistribution isn't available
                    reject();
                } else {
                    // not cached -> retrieve it
                    DataikuAPI.ml.prediction.getFeaturesDistribution(fmi).success(function(data) {
                        if (data) {
                            featuresDistributionCache[fmi] = data.featuresDistribution;
                            resolve(methods);
                        } else {
                            featuresDistributionCache[fmi] = null;
                            reject();
                        }
                    });
                }
            })
        };
    });

    app.factory('ExplorationWT1Service', function(WT1, EmuType, InteractiveModelCommand, OutcomeOptimizationSpecialTarget) {
        const _getExplorationType = isClassification => {
            return isClassification ? InteractiveModelCommand.COUNTERFACTUALS : InteractiveModelCommand.OUTCOME_OPTIMIZATION;
        }

        const init = (isClassification, nFeatures) => {
            const explorationType = _getExplorationType(isClassification);
            return {
                emitConstraintsOpened: () => {
                    WT1.event('ml-exploration-constraints-opened', { explorationType, nFeatures });
                },
                emitComputationStarted: (explorationParams, unlockedFeatures) => {
                    // Anonymize the target
                    let targetDescription;
                    if (explorationType === InteractiveModelCommand.COUNTERFACTUALS) {
                        targetDescription = (explorationParams.target !== undefined) ? "SPECIFIC_CLASS" : "NO_SPECIFIC_CLASS";
                    } else {
                        targetDescription = Object.values(OutcomeOptimizationSpecialTarget).includes(explorationParams.target) ? explorationParams.target : "SPECIFIC_TARGET";
                    }

                    // Count the number of features with non-default constraints
                    let nFeaturesWithCustomConstraints = 0;
                    for (const unlockedFeature of unlockedFeatures) {
                        const defaultConstraint = unlockedFeature.constraintHelper.getDefaultConstraint();
                        const actualConstraint = angular.copy(explorationParams.featureDomains.find(featureDomain => featureDomain.name === unlockedFeature.name));
                        // when we compare both objects, we ignore the order of the arrays (i.e. categories)
                        for (const key in defaultConstraint) {
                            if (angular.isArray(defaultConstraint[key])) {
                                actualConstraint[key].sort();
                                defaultConstraint[key].sort();
                            }
                        }
                        if (!angular.equals(actualConstraint, defaultConstraint)) {
                            nFeaturesWithCustomConstraints++;
                        }
                    }

                    const countNFeaturesForType = type => explorationParams.featureDomains.filter(featureDomain => featureDomain.type === type).length;
                    WT1.event('ml-exploration-computation-started', {
                        nFrozenFeatures: countNFeaturesForType(EmuType.FROZEN),
                        nNumericalFeatures: countNFeaturesForType(EmuType.NUMERICAL),
                        nCategoricalFeatures: countNFeaturesForType(EmuType.CATEGORICAL),
                        target: targetDescription,
                        explorationType,
                        nFeatures,
                        nFeaturesWithCustomConstraints
                    });
                },
                emitComputationSucceeded: (nResults, duration) => {
                    WT1.event('ml-exploration-computation-succeeded', { explorationType, nFeatures, nResults, duration });
                },
                emitExportClicked: (nRows) => {
                    WT1.event('ml-exploration-export', { explorationType, nFeatures, nRows });
                }
            }
        };

        return { init };
    });

    app.service('WhatIfFormattingService', function(ModelDataUtils) {
        const getSmartNumberDigits = function(min, max) {
            if (min === max) {
                return 0;
            }
            const absoluteMin = Math.min(Math.abs(min), Math.abs(max));
            const differenceOrderOfMagnitude = Math.floor(Math.log10(max - min));
            if (differenceOrderOfMagnitude < 0) {
                return -differenceOrderOfMagnitude + 1;
            } else {
                return Math.max(max - min < 10 ? 1 : 0, -Math.floor(Math.log10(absoluteMin || 10)));
            }
        };
        const getComplexPredictionFormatForRegression = function(modelData, prediction) {
            let min, max;
            if(modelData.predictionInfo && modelData.predictionInfo.predictions && modelData.predictionInfo.predictions.length > 0) {
                const predictions = modelData.predictionInfo.predictions;
                min = d3.min(predictions);
                max = d3.max(predictions);
            } else if (modelData.perf &&  modelData.perf.scatterPlotData && modelData.perf.scatterPlotData.y && modelData.perf.scatterPlotData.y.length > 0) {
                // Overall partitioned models do not have prediction statistics.
                // Instead, and because this code is only reachable for regression models, we use the scatter plot if possible
                const predictions = modelData.perf.scatterPlotData.y;
                min = d3.min(predictions);
                max = d3.max(predictions);
            } else {
                if (prediction === undefined) {
                    return "f";
                }
                // Edge case when we don't have access to prediction statistics nor scatter plot data
                min = prediction - Math.abs(prediction);
                max = prediction + Math.abs(prediction);
            }
            return `.${getSmartNumberDigits(min, max)}f`;
        };

        return {
            formatPrediction: (modelData, prediction) => {
                if (ModelDataUtils.isRegression(modelData)) {
                    if (modelData.perf && modelData.perf.scatterPlotData && modelData.perf.scatterPlotData.y) {
                        return d3.format(getComplexPredictionFormatForRegression(modelData, prediction))(prediction);
                    }
                    return prediction.toPrecision(5);
                }
                return prediction;
            },
            formatProba: proba => d3.format(",.2f")(proba * 100),
            getSmartNumberDigits: getSmartNumberDigits,
            getComplexPredictionFormatForRegression: getComplexPredictionFormatForRegression
        };
    });

})();
