(function(){
    "use strict";
    var app = angular.module('dataiku.ml.hyperparameters', ['platypus.utils']);

    /**
     * Generic range/explicit values input component
     * @param {function} changeModeCallback - callback to be called when the mode is changing. Useful for ml-hp-numerical wrapper
     * @param {int[]} explicitValues - list of explicit values
     * @param {boolean} gridStrategy - whether it uses grid strategy or not
     * @param {string} helpInline - optional inline help
     * @param {boolean} isDecimalInput - whether the input values are decimal or not
     * @param {string} label - The control label
     * @param {object} limit - object containing `min` and `max` range values
     * @param {object} range - The range object { min, max, nbValues }
     * @param {boolean} required - optional add css class "required" to the directive
     * @param {boolean} useRange - whether we use range or explicit mode
     * @param {function} validateExplicitValue - optional function to check if new explicit values are valid
     */
    app.component("numericalMultiInput", {
        bindings: {
            changeModeCallback: '<?',
            explicitValues: '=',
            gridStrategy: '<',
            helpInline: '@',
            isDecimalInput: '<?',
            label: '@',
            limit: '<',
            range: '=',
            required: '<?',
            useRange: '=',
            validateExplicitValue: '<?',
        },
        templateUrl: 'templates/ml/ml-numerical-input.html',
        controller: function() {
            const $ctrl = this;

            function isDecimalValue(v) {
                return v % 1 !== 0;
            }

            $ctrl.$onInit = function() {
                $ctrl.isDecimalInput = !angular.isUndefined($ctrl.isDecimalInput) ? $ctrl.isDecimalInput : false; // Default to integer inputs
                $ctrl.required = !angular.isUndefined($ctrl.required) ? $ctrl.required : false; // Defaults to not required
            };

            $ctrl.isExplicitValueValid = function(newValue) {
                const limit = $ctrl.limit || {};
                const inRange = (angular.isUndefined(limit.min) || newValue >= limit.min)
                    && (angular.isUndefined(limit.max) || newValue <= limit.max);
                if (!inRange) return false;
                if (!$ctrl.isDecimalInput && newValue !== parseInt(newValue)) return false;
                return angular.isFunction($ctrl.validateExplicitValue) ? $ctrl.validateExplicitValue(newValue) : true;
            };

            $ctrl.setUseRange = function(useRange) {
                $ctrl.useRange = useRange;
                if (angular.isFunction($ctrl.changeModeCallback)) {
                    $ctrl.changeModeCallback(useRange);
                }
            };

            $ctrl.isInvalidMinMax = function() {
                if (angular.isUndefined($ctrl.range)) {
                    return false;
                }

                const { min, max } = $ctrl.range;

                if (angular.isUndefined(min) || angular.isUndefined(max)) {
                    return false;
                }
                return min >= max;
            };

            $ctrl.isValidRange = function() {
                const { min, max, nbValues } = $ctrl.range;

                // Validate range number of values
                if (nbValues < 2 || isDecimalValue(nbValues)) return false;

                // If the input is decimal, we don't care about `nbValues` itself since we ll always have a correct range
                // Otherwise, we check the nb of values in the range is large enough regarding min and max values
                return $ctrl.isDecimalInput ? true : nbValues <= max - min + 1;
            };
        }
    });


    /**
     * Controls for numeric grid search (range and explicit) -> Wrapper around numericalMultiInput
     * @param {boolean} decimal - optional to allow decimal values
     * @param {boolean} gridStrategy - whether it uses grid strategy or not
     * @param {string} helpInline - optional inline help
     * @param {object} hpDesc - inner dict containing the hyperparameter search description
     * @param {string} label - The control label
     * @param {boolean} required - optional add css class "required" to the directive
     * @param {function} validateExplicitValue - optional function to check if new explicit values are valid
     */
    app.directive('mlHpNumerical', function() {
        return {
            restrict: 'E',
            scope: {
                decimal: '<',
                gridStrategy: '<',
                helpInline: '@',
                hpDesc: '=ngModel',
                label: '@',
                required: '<?',
                validateExplicitValue: '<?',
            },
            templateUrl: 'templates/ml/ml-hp-numerical.html',
            link: function($scope) {
                function initUseRange() {
                    if ($scope.gridStrategy) {
                        $scope.useRange = $scope.hpDesc.gridMode === "RANGE";
                    } else {
                        $scope.useRange = $scope.hpDesc.randomMode === "RANGE";
                    }
                }

                $scope.changeMode = function(useRange) {
                    if ($scope.gridStrategy) {
                        $scope.hpDesc.gridMode = useRange ? "RANGE" : "EXPLICIT";
                    } else {
                        $scope.hpDesc.randomMode = useRange ? "RANGE" : "EXPLICIT";
                    }
                };

                initUseRange();
            }
        }
    })

    /**
     * Built on top of Suggestions directive to display numeric suggestions (for grid search).
     * @param {array} tags - The list of selected suggestions displayed as tags in the input.
     * @param {string} placeholder - Optional placeholder of the input.
     * @param {array} lockedValues - Optional array of the values that must remain in the gsField and therefore cannot be removed.
     * @param {function} validateValue - Optional function to check whether the potential new value is valid, otherwise it is discarded and not added to the others.
    */
    app.directive('gsField', function($timeout, $q) {
        return {
            restrict:'A',
            scope: {
                tags: '=ngModel',
                placeholder: '@',
                lockedValues: '<?',
                validateValue:'<?',
            },
            templateUrl: 'templates/ml/gs-field.html',
            link: function(scope, element, attrs) {
                if (!angular.isFunction(scope.validateValue)) {
                    scope.validateValue = _ => true;
                }
                const allowedMultiples = [2,5,10,100];
                const allowedAdditives = [1,2,5,10,20,30,50,100];
                scope.getClassName = function() {
                    let className = 'gsField';

                    if (attrs.notGrid !== undefined) {
                        className += ' gsField--not-grid';
                    }
                    return className;
                };

                function filterSuggestedGS(ls) {
                    return ls.filter(function(o){
                        return scope.tags.indexOf(o) === -1 && o > 0;
                    });
                }

                var getSuggestedGS = function() {
                    if (scope.tags.length > 0) {
                        var a = Math.min(scope.tags[scope.tags.length-1],scope.tags[scope.tags.length-2] || 1),
                            b = Math.max(scope.tags[scope.tags.length-1],scope.tags[scope.tags.length-2] || 1),
                            rev = scope.tags[scope.tags.length-1] < (scope.tags[scope.tags.length-2] || 1),
                            m;
                        if (a<0 || b<0) {
                            return filterSuggestedGS([scope.placeholder || 1])
                        }

                        for (m=0;m<allowedMultiples.length;m++) {
                            var muldif = Math.log(b/a) / Math.log(allowedMultiples[m]);
                            muldif = Math.round(muldif * 1000) / 1000;
                            if (isInteger(muldif)) {
                                if (!rev) {
                                    return filterSuggestedGS([b*allowedMultiples[m], a/allowedMultiples[m]])
                                }
                                else {
                                    return filterSuggestedGS([a/allowedMultiples[m], b*allowedMultiples[m]])
                                }
                            }
                        }

                        for (m=allowedAdditives.length;m>=0;m--) {
                            if ((b-a) % allowedAdditives[m] === 0 && a % allowedAdditives[m] === 0) {
                                if (!rev) {
                                    return filterSuggestedGS([b+allowedAdditives[m], a-allowedAdditives[m]])
                                }
                                else {
                                    return filterSuggestedGS([a-allowedAdditives[m], b+allowedAdditives[m]])
                                }
                            }
                        }

                        return filterSuggestedGS([scope.placeholder || 1])
                    } else {
                        return filterSuggestedGS([scope.placeholder || 1])
                    }
                }

                scope.suggestGS = function(q) {
                    const deferred = $q.defer();
                    if (attrs.notGrid !== undefined) {
                        deferred.resolve([]);
                    } else {
                        deferred.resolve(getSuggestedGS());
                    }
                    return deferred.promise;
                };

                scope.tagIndex = undefined;
                var input = element.find('.suggestions input');
                scope.hasFocus = function(){
                    return input.is(":focus") || element.find(".fake").is(":focus");
                };
                scope.newTag = '';

                scope.addSuggestion = function(value, e){
                    scope.newTag = value;
                    e.stopPropagation();
                    if (scope.addTag()) {
                        e.preventDefault();
                    }
                };

                scope.selectTag = function(e, idx){
                    e.stopPropagation();
                    scope.tagIndex = idx;
                };

                scope.deleteTag = function(e, idx){
                    if (e){ e.originalEvent.stopPropagation() }

                    var index = idx;
                    if (index === null || index === undefined) {
                        index = scope.tagIndex;
                    }

                    if(!angular.isUndefined(index)) {
                        if (!( scope.lockedValues || [] ).includes(scope.tags[index])) {
                            scope.tags.splice(index, 1);
                        }
                        $timeout(function(){ scope.$broadcast('showSuggestions') });
                        if(scope.tags.length) {
                            // set tagIndex to former tag
                            scope.tagIndex = Math.max(index - 1, 0);
                        } else {
                            // otherwise set focus to input, but only if this was from a backspace deletion
                            if (!e) { input.focus() }
                        }
                    }
                };

                scope.addTag = function(){
                    var added = false;
                    if(scope.newTag && !isNaN(scope.newTag)){
                        scope.newTag = parseFloat(scope.newTag);
                        const noDuplicatingIssue = attrs.allowDubs || scope.tags.indexOf(scope.newTag) === -1;
                        if (scope.validateValue(scope.newTag) && noDuplicatingIssue) {
                            // add tag
                            scope.tags.push(scope.newTag);
                            added = true;
                            if (!angular.isUndefined(attrs.sortValues)) {
                                scope.tags.sort((a,b) => a-b);
                            }
                        }
                        // empty field
                        scope.newTag = '';

                        if(! scope.$root.$$phase) scope.$apply();
                        $timeout(function(){ scope.$broadcast('showSuggestions') });
                    }
                    return added;
                };

                scope.$watch('tagIndex', function(){
                    if (!angular.isUndefined(scope.tagIndex)){
                        input.blur();
                        element.find(".fake").focus();
                    }
                });

                input.on('focus', function(){
                    scope.tagIndex = undefined;
                });

                scope.setFocus = function(e){
                    input.focus();
                    e.stopPropagation();
                };

                scope.$on("$destroy", function(){
                    $(element).off("keydown.tags");
                });
                scope.inputBlur = function(e) {
                    if (e) {
                        e.stopPropagation();
                        if (scope.addTag()) {
                            e.preventDefault();
                        }
                    }
                }
                $(element).on('keydown.tags', function(e){
                    if(scope.hasFocus()){
                        if (e.keyCode == 37){ // left arrow
                            if(!angular.isUndefined(scope.tagIndex)){
                                scope.tagIndex = Math.max(scope.tagIndex - 1, 0);
                                scope.$apply();
                            } else {
                                if(scope.newTag.length === 0){
                                    scope.tagIndex = scope.tags.length - 1;
                                    scope.$apply();
                                }
                            }
                        } else if (e.keyCode == 39){ // right arrow
                            if(!angular.isUndefined(scope.tagIndex)){
                                scope.tagIndex = scope.tagIndex + 1;
                                if(scope.tagIndex >= scope.tags.length){
                                    scope.tagIndex = undefined;
                                    input.focus();
                                }
                                scope.$apply();
                            }
                        } else if (e.keyCode == 8){ // delete
                            if(angular.isUndefined(scope.tagIndex)){
                                if(scope.newTag.length === 0){
                                    scope.tagIndex = scope.tags.length - 1;
                                    scope.$apply();
                                }
                            } else {
                                e.preventDefault();
                                scope.deleteTag();
                                scope.$apply();
                            }
                        } else if (e.keyCode == 13 || e.keyCode == 32){ // enter & space : If we added a tag, don't let the "enter" key trigger a form submit
                            e.stopPropagation();
                            e.preventDefault();
                            if (!scope.newTag) {
                                if (attrs.noAutoFill !== undefined) return;
                                scope.newTag = getSuggestedGS()[0];
                            }
                           scope.addTag();
                        }
                    }
                });

                scope.tags = scope.tags || [];

                scope.$watch('tags', function(nv, ov) {
                    // Sometimes someone rebinds the ngModel to null, in our case the API...
                    if (nv === null || nv === undefined) {
                        scope.tags = [];
                    }
                });
            }
        };
    });
})();
