(function() {
'use strict';

const app = angular.module('dataiku.recipes');


app.controller("NLPLLMTextClassificationRecipeCreationController", function($scope, $stateParams, CreateModalFromTemplate, DatasetUtils, DatasetLabUtils,
                                                                            DataikuAPI, CachedAPICalls) {
    // Properties for `modelDefined` & `userDefined` are:
    // - `displayName` & `desc`: Name and description for the use case group
    // - `useCases`: an array of objects, if objects having the following properties:
    //     * `name` & `desc`: Name & description for the specific use case
    //     * `action`: a function with no args, called when selecting the use case
    //     * `icon` (optional): CSS class(es) for a potential icon in front of the use case name
    const modelDefined = {
        displayName: "Task-specific classes",
        desc: "Use predefined classes to perform specific text classification tasks",
        useCases: [
            {
                name: "Sentiment analysis",
                desc: "Analyze sentiment (positive, negative, neutral...) in a text",
                inputDatasetRequired: false,
                action: newSentimentAnalysis,
                icon: "dku-icon-sentiment-20 universe-color recipe-nlp",
            },
            {
                name: "Emotion analysis",
                desc: "Analyze emotions in a text",
                inputDatasetRequired: false,
                action: newEmotionAnalysis,
                icon: "dku-icon-heart-fill-20 universe-color recipe-nlp",
            },
            {
                name: "Other classification task",
                desc: "Perform a specific task with model-defined classes, using a fine-tuned model",
                inputDatasetRequired: false,
                action: newBuiltinClassesClassification,
                icon: "dku-icon-llm-text-classification-other-20 universe-color recipe-nlp",
            },
        ],
    };

    const userDefined = {
        displayName: "User-specified classes",
        desc: "Provide the classes to use for a text classification task",
        useCases: [
            {
                name: "LLM classification",
                desc: "Classify text with a LLM using classes you specify, optionally provide a few examples to help the model. Labeled dataset is not required",
                inputDatasetRequired: false,
                action: newCustomClassesClassification,
                icon: "dku-icon-edit-note-20 universe-color recipe-nlp",
            },
            {
                name: "AutoML Prediction",
                desc: "Train a classification model based on many labeled rows. Labeled dataset is required",
                inputDatasetRequired: true,
                action: newPrediction,
                icon: "dku-icon-automl-prediction-20 universe-color analysis",
            },
        ],
    };

    $scope.textClassificationTypes = [ modelDefined, userDefined ];

    $scope.getUseCaseDisabledMessage = function(useCase) {
        if (useCase.inputDatasetRequired && !$scope.recipePresets.preselectedInput) {
            return 'Requires an input dataset';
        }
        return '';
    };

    function copyToRecipeScope(recipePresets, modalScope) {
        modalScope.zone = recipePresets.zone;
        modalScope.initialPayload = recipePresets.initialPayload;
        if (recipePresets.preselectedInput) {
            modalScope.io.inputDataset = recipePresets.preselectedInput;
        }
    }

    function newSentimentAnalysis() {
        CreateModalFromTemplate('/templates/recipes/nlp-based-recipe-creation.html', $scope.$parent, "NLPLLMModelProvidedClassificationRecipeCreationController", function(modalScope) {
            $scope.recipePresets.initialPayload = {
                task: 'SENTIMENT_ANALYSIS',
                outputMode: 'FIRST',
            };

            copyToRecipeScope($scope.recipePresets, modalScope);
        });
    }

    function newEmotionAnalysis() {
        CreateModalFromTemplate('/templates/recipes/nlp-based-recipe-creation.html', $scope.$parent, "NLPLLMModelProvidedClassificationRecipeCreationController", function(modalScope) {
            $scope.recipePresets.initialPayload = {
                task: 'EMOTION_ANALYSIS',
                outputMode: 'FIRST',
            };

            copyToRecipeScope($scope.recipePresets, modalScope);
        });
    }

    function newCustomClassesClassification() {
        CreateModalFromTemplate('/templates/recipes/nlp-based-recipe-creation.html', $scope.$parent, "NLPLLMUserProvidedClassificationRecipeCreationController", function(modalScope) {
            copyToRecipeScope($scope.recipePresets, modalScope);
        });
    }

    function newBuiltinClassesClassification() {
        CreateModalFromTemplate('/templates/recipes/nlp-based-recipe-creation.html', $scope.$parent, "NLPLLMModelProvidedClassificationRecipeCreationController", function(modalScope) {
            $scope.recipePresets.initialPayload = {
                task: 'OTHER',
                outputMode: 'FIRST',
            };

            copyToRecipeScope($scope.recipePresets, modalScope);
        });
    }

    // This and displayNewPredictionModal are simplified from `controllers.js`, might be further factorized
    function newPrediction() {
        $scope.predictionTaskData = {};
        $scope.datasetSmartName = $scope.recipePresets.preselectedInput;
        let backends;
        DataikuAPI.analysis.mlcommon.listBackends($stateParams.projectKey, $scope.datasetSmartName, '')
            .then(function({data}) {
               backends = data;
               const loc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.datasetSmartName);
                return DataikuAPI.datasets.get(loc.projectKey, loc.name, $stateParams.projectKey);
            }).then(function({data}) {
                $scope.possibleColumns = data.schema.columns.map(it => it.name);
                return CachedAPICalls.pmlGuessPolicies
            }).then(pmlGuessPolicies => displayNewPredictionModal(pmlGuessPolicies, backends)).catch(setErrorInScope.bind($scope));
    }

    function displayNewPredictionModal(policies, backends) {
        $scope.predictionTaskData.guessPolicies = policies;
        DatasetLabUtils.prepareGuessPolicies($scope.predictionTaskData, backends);

        const policyGroup = "auto";
        DatasetLabUtils.updateTaskWithDefaultPolicy($scope.predictionTaskData, policyGroup, $scope.datasetSmartName);

        CreateModalFromTemplate("/templates/datasets/create-prediction-modal.html", $scope, "NewPMLTaskModalController", (modalScope) => {
            modalScope.useCurrentAnalysisForNewMlTask = false;
            modalScope.selectPolicy = function(taskData, policy) {
                DatasetLabUtils.updateTaskWithNewPolicy(taskData, policy, $scope.datasetSmartName)
            };
            modalScope.policyGroup = policyGroup;
        });
    }
});


app.controller("NLPLLMUserProvidedClassificationRecipeCreationController", function(
    $scope, $stateParams, $controller, DataikuAPI, RecipesUtils,
    RecipeDescService, RecipeComputablesService, PromptUtils) {

    $scope.recipeType = 'nlp_llm_user_provided_classification';

    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});
    $scope.showOutputPane = _ => !!$scope.io.inputDataset;

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            const niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./, "");
            $scope.maybeSetNewDatasetName(niceInputName + "_classified");
        }
    };

    $scope.getCreationSettings = function() {
        const inputs = [];
        if ($scope.io.inputDataset) {
            inputs.push($scope.io.inputDataset);
        }
        const creationSettings = { virtualInputs: inputs };

        if ($scope.initialPayload) {
            creationSettings.initialPayload = $scope.initialPayload;
        }

        return creationSettings;
    };

    addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);
    fetchManagedDatasetConnections($scope, DataikuAPI);
});


app.controller("NLPLLMUserProvidedClassificationRecipeEditor", function($scope, $controller, $state, $stateParams, DataikuAPI, CreateModalFromTemplate) {
    $controller("_NLPLLMRecipeControllerBase", {$scope: $scope});

    $scope.loadLLMs("CLASSIFICATION_WITH_USER_PROVIDED_CLASSES");

    $scope.openConvertToPromptModal = function() {
        if (!$scope.recipe || !$scope.recipe.name) return;

        $scope.saveRecipe().then(function() {
            CreateModalFromTemplate("/templates/recipes/nlp/convert-to-prompt-recipe.html", $scope, "NLPRecipeConvertToPromptRecipeModal", function(modalScope) {
            });
        });
    };
});



app.controller("NLPLLMModelProvidedClassificationRecipeCreationController", function(
    $scope, $stateParams, $controller, DataikuAPI, RecipesUtils,
    RecipeDescService, RecipeComputablesService, PromptUtils) {

    $scope.recipeType = 'nlp_llm_model_provided_classification';

    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});
    $scope.showOutputPane = _ => !!$scope.io.inputDataset;

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            const niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./, "");
            $scope.maybeSetNewDatasetName(niceInputName + "_classified");
        }
    };

    $scope.getCreationSettings = function() {
        const inputs = [];
        if ($scope.io.inputDataset) {
            inputs.push($scope.io.inputDataset);
        }
        const creationSettings = { virtualInputs: inputs };

        if ($scope.initialPayload) {
            creationSettings.initialPayload = $scope.initialPayload;
        }

        return creationSettings;
    };

    addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);
    fetchManagedDatasetConnections($scope, DataikuAPI);
});


app.controller("NLPLLMModelProvidedClassificationRecipeEditor", function($scope, $controller, $state, $stateParams, DataikuAPI, CreateModalFromTemplate) {
    $controller("_NLPLLMRecipeControllerBase", {$scope: $scope});

    $scope.taskOptions = [
        ['SENTIMENT_ANALYSIS', 'Sentiment analysis (+/=/-)'],
        ['EMOTION_ANALYSIS', 'Emotion analysis'],
        ['OTHER', 'Other']
    ];

    const deregister = $scope.$watch('desc', function() {
        if (!$scope.desc) return;

        $scope.onTaskChange();
        deregister();
    });

    const genericOutputModes = [
        { label: "Output the most likely class", value: "FIRST"},
        { label: "Output all the classes", value: "ALL"},
    ];

    const emotionAnalysisOutputModes = [
        { label: "Output the most likely class", value: "FIRST"},
        { label: "Output the top most likely classes", value: "MOST_RELEVANT"},
        { label: "Output all the classes", value: "ALL"},
    ];

    $scope.onTaskChange = function() {
        if (!$scope.desc) return;

        switch ($scope.desc.task) {
        case 'SENTIMENT_ANALYSIS':
            $scope.loadLLMs("SENTIMENT_ANALYSIS");
            if ($scope.desc.outputMode === "MOST_RELEVANT") {
                $scope.desc.outputMode = "FIRST";
            }
            $scope.outputModes = genericOutputModes;
            break;
        case 'EMOTION_ANALYSIS':
            $scope.loadLLMs("EMOTION_ANALYSIS");
            $scope.outputModes = emotionAnalysisOutputModes;
            break;
        case 'OTHER':
            $scope.loadLLMs("CLASSIFICATION_WITH_OTHER_MODEL_PROVIDED_CLASSES");
            if ($scope.desc.outputMode === "MOST_RELEVANT") {
                $scope.desc.outputMode = "FIRST";
            }
            $scope.outputModes = genericOutputModes;
            break;
        }
    };

    $scope.openConvertToPromptModal = function() {
        if (!$scope.recipe || !$scope.recipe.name) return;
        $scope.saveRecipe().then(function() {
            CreateModalFromTemplate("/templates/recipes/nlp/convert-to-prompt-recipe.html", $scope, "NLPRecipeConvertToPromptRecipeModal", function(modalScope) {
            });
        });
    };
});


}());
