(function(){
'use strict';

const app = angular.module('dataiku.savedmodels', ['dataiku.ml.report', 'dataiku.lambda']);

app.service("SavedModelsService", function($q, DataikuAPI, $stateParams, $state, CreateModalFromTemplate, SmartId, $rootScope, MLModelsUIRouterStates,
    ActiveProjectKey, FullModelLikeIdUtils, translate, Dialogs, WT1, PluginConfigUtils) {
    const listModels = function(projectKey, filter) {
        const deferred = $q.defer();
        DataikuAPI.savedmodels.listWithAccessible(projectKey).success(function(data){
            const savedModels = data.filter(filter);
            savedModels.forEach(function(sm) {
                if (sm.projectKey !== projectKey) {
                    sm.name = sm.projectKey + "." + sm.name;
                    sm.id = sm.projectKey + "." + sm.id;
                }
            });
            deferred.resolve(savedModels);
        });
        return deferred.promise;
    };

    const svc = {
        nonAPIServiceableMsg: function(sm) {
            if (!sm) {
                return null;
            }
            if (sm.savedModelType === "PROXY_MODEL") {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_EXTERNAL_MODELS", "An API can not be created from External Models");
            } else if (this.isAgent(sm)) {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_AGENTS", "An API can not be created from an Agent");
            } else if (this.isRetrievalAugmentedLLM(sm)) {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_RA_LLM", "An API can not be created from a Retrieval-Augmented LLM");
            } else if (this.isLLMGeneric(sm)) {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_LLM_MODELS", "An API can not be created from LLM models");
            }
            if (sm.miniTask?.backendType === 'VERTICA') {
                return translate("PROJECT.PERMISSIONS.VERTICA_NOT_SUPPORTED", "Vertica ML backend is no longer supported");
            }
            return null;
        },
        // enrich a saved model object (saved model, agent or Retrieval Augmented LLM) with status and summary data (that is used for display in the RHP)
        enrichSavedModelObject: function(objectData, contextProjectKey, scope) {
            if (!objectData.model) objectData.model = objectData.saved_model;
            if (!objectData.status) {
                switch (objectData.model.miniTask.taskType) {
                    case 'CLUSTERING':
                        DataikuAPI.savedmodels.clustering.getStatus(objectData.model.projectKey, SmartId.create(objectData.model.id, objectData.model.projectKey , contextProjectKey))
                            .success(function (data) {
                                objectData.status = data;
                            })
                            .error(setErrorInScope.bind(scope));
                        break;
                    case 'PREDICTION':
                        DataikuAPI.savedmodels.prediction.getStatus(objectData.model.projectKey, SmartId.create(objectData.model.id, objectData.model.projectKey , contextProjectKey))
                            .success(function (data) {
                                objectData.status = data;
                            })
                            .error(setErrorInScope.bind(scope));
                        break;
                }
            }

            const isRetrievalAugmentedLLM = this.isRetrievalAugmentedLLM; // forward method so it can be used within functions
            function findAgentConnectUsage(objectData, scope) {
                const raModelFields = {modelType: "retrieval-augmented-llm", agentConnectField: "augmented_llms"};
                const agentFields = {modelType: "agent", agentConnectField: "agents_ids"};
                const fields = isRetrievalAugmentedLLM(objectData.model) ? raModelFields : agentFields;
                const modelId = fields.modelType + ":" + objectData.model.id;
                function filterWebappUsage(webapp) {
                    return webapp.type === "webapp_agent-connect_portal"
                            && ((webapp.config.llm_id === modelId)
                                || ((webapp.config[fields.agentConnectField] || []).includes(objectData.model.projectKey + ":" + modelId)));
                }
                DataikuAPI.webapps.list(objectData.model.projectKey).success(function(data) {
                    objectData.model.summary.webappsUsage = objectData.model.summary.webappsUsage.concat(data.filter(filterWebappUsage));
                }).error(setErrorInScope.bind(scope));
            }

            function findAgentHubUsage(objectData, scope) {
                const modelIdPrefix = isRetrievalAugmentedLLM(objectData.model) ? ":agent:retrieval-augmented-llm:" : ":agent:";
                const objectFullId = objectData.model.projectKey + modelIdPrefix + objectData.model.id;
                DataikuAPI.webapps.list(objectData.model.projectKey)
                    .then((resp) => {
                       resp.data.filter((webapp) => webapp.type === "webapp_agent-hub_agent-hub")
                                .forEach((webapp) => {
                                    DataikuAPI.webappsBackends
                                        .get(objectData.model.projectKey, webapp.id, "api/agents", false)
                                        .then((resp) => {
                                          if (resp?.data?.enterpriseAgents.some((agent) => agent.id === objectFullId)) {
                                              // Reassign instead of push to ensure the change is detected
                                              objectData.model.summary.webappsUsage = objectData.model.summary.webappsUsage.concat([webapp]);
                                          }
                                        }).catch(err => {
                                            if (err?.status == 404) {
                                                return; // do not show error if the backend is not running
                                            }
                                            setErrorInScope.bind(scope);
                                        });
                                });
                    }).catch(setErrorInScope.bind(scope));
            }

            // Agents display a Summary section
            if (this.isAgent(objectData.model)) {
                objectData.model.summary = {
                    webappsUsage: [],
                    // visual agents specific fields
                    prompt: undefined,
                    llm: undefined,
                    tools: []
                };

                if (objectData.model.savedModelType === "TOOLS_USING_AGENT") {
                    // Visual agents can use a LLM and some tools
                    const agentVersion = objectData.model.inlineVersions.find(v => v.versionId === objectData.model.activeVersion);
                    if (agentVersion) {
                        objectData.model.summary.mode = agentVersion.toolsUsingAgentSettings.mode
                        objectData.model.summary.prompt = agentVersion.toolsUsingAgentSettings.systemPromptAppend;

                        // find the LLM used by the agent
                        DataikuAPI.pretrainedModels.listAvailableLLMs(objectData.model.projectKey, "GENERIC_COMPLETION").success(function(data) {
                            objectData.model.summary.llm = data.identifiers.find(llm => llm.id === agentVersion.toolsUsingAgentSettings.llmId);
                        }).error(setErrorInScope.bind(scope));

                        // enrich each tool object with their type, name and more
                        DataikuAPI.agentTools.listAvailable(objectData.model.projectKey).success(function(data) {
                            objectData.model.summary.tools = agentVersion.toolsUsingAgentSettings.tools.map(tool => {
                                const usedTool = SmartId.resolve(tool.toolRef, objectData.model.projectKey);
                                const projectTool = data.find(t => t.id === usedTool.id && t.projectKey === usedTool.projectKey);
                                if (projectTool) {
                                    Object.assign(tool, projectTool);
                                }
                                if (objectData.model.projectKey == usedTool.projectKey) {
                                    tool.isLocal = true;
                                    tool.fullName = tool.name;
                                } else {
                                    tool.isLocal = false;
                                    tool.fullName = `${usedTool.projectKey}.${tool.name}`;
                                }
                                tool.toolPageURL = $state.href("projects.project.agenttools.agenttool", {
                                    projectKey: usedTool.projectKey,
                                    agentToolId: usedTool.id,
                                });
                                return tool;
                            });
                        }).error(setErrorInScope.bind(scope));
                    }
                }
                // find webapps usage (current project only), all agents
                findAgentConnectUsage(objectData, scope);
                findAgentHubUsage(objectData, scope);
            }
            // Retrieval Augmented LLMs display a Summary section
            if (this.isRetrievalAugmentedLLM(objectData.model)) {
                const ragllmVersion = objectData.model.inlineVersions.find(v => v.versionId === objectData.model.activeVersion);
                objectData.model.summary = {
                    version: ragllmVersion,
                    settings: ragllmVersion.ragllmSettings,
                    llm: undefined,
                    kb: undefined,
                    webappsUsage: [],
                };
                // find the LLM used by the ra-llm
                DataikuAPI.pretrainedModels.listAvailableLLMs(objectData.model.projectKey, "GENERIC_COMPLETION").success(function(data) {
                    objectData.model.summary.llm = data.identifiers.find(llm => llm.id === ragllmVersion.ragllmSettings.llmId);
                }).error(setErrorInScope.bind(scope));
                // find knowledge bank
                DataikuAPI.retrievableknowledge.get(objectData.model.projectKey, ragllmVersion.ragllmSettings.kbRef).success(knowledgeBank => {
                    objectData.model.summary.kb = knowledgeBank;
                }).error(setErrorInScope.bind(scope));
                // find webapps usage (current project only)
                findAgentConnectUsage(objectData, scope);
                findAgentHubUsage(objectData, scope);
            }
        },
        listAPIServiceableModels: function(projectKey) {
            return listModels(projectKey, sm => !svc.nonAPIServiceableMsg(sm));
        },
        listPredictionModels: function(projectKey) {
            return listModels(projectKey, sm => sm.miniTask && sm.miniTask.taskType === 'PREDICTION');
        },
        listEvaluablePredictionModels: function(projectKey) {
            return listModels(projectKey, sm => sm.miniTask && sm.miniTask.taskType === 'PREDICTION' && !['VERTICA', 'DEEP_HUB'].includes(sm.miniTask.backendType));
        },
        listProxyModels: function(projectKey) {
            return listModels(projectKey, sm => sm.savedModelType === 'PROXY_MODEL');
        },
        listClusteringModels: function(projectKey) {
            return listModels(projectKey, sm => sm.miniTask && sm.miniTask.taskType === 'CLUSTERING');
        },
        isActiveVersion: function(fullModelId, savedModel) {
            if (!fullModelId || !savedModel) return;
            return FullModelLikeIdUtils.parse(fullModelId).versionId === savedModel.activeVersion;
        },
        isPartition: function(fullModelId) {
            if (!fullModelId) return;
            return !!FullModelLikeIdUtils.parse(fullModelId).partitionName;
        },
        isExternalMLflowModel: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["MLFLOW_PYFUNC", "PROXY_MODEL"].includes(model.savedModelType);
            }
            return model && model.modeling && model.modeling.algorithm && (
                ["VIRTUAL_MLFLOW_PYFUNC", "VIRTUAL_PROXY_MODEL"].includes(model.modeling.algorithm))
        },
        isMLflowModel: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return "MLFLOW_PYFUNC" === model.savedModelType;
            }
            return model && model.modeling && model.modeling.algorithm && (
                "VIRTUAL_MLFLOW_PYFUNC" === model.modeling.algorithm)
        },
        isProxyModel: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["PROXY_MODEL"].includes(model.savedModelType);
            }
            return model && model.modeling && model.modeling.algorithm && (
                ["VIRTUAL_PROXY_MODEL"].includes(model.modeling.algorithm))
        },
        isPartitionedModel: function(model) {
           return (model && model.partitioning && model.partitioning.dimensions && model.partitioning.dimensions.length > 0);
        },
        isAgent: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["PYTHON_AGENT", "PLUGIN_AGENT", "TOOLS_USING_AGENT"].includes(model.savedModelType);
            }
        },
        isRetrievalAugmentedLLM: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["RETRIEVAL_AUGMENTED_LLM"].includes(model.savedModelType);
            }
        },
        isLLMSavedModel: function (model) {
            return this.isAgent(model) || this.isRetrievalAugmentedLLM(model)
        },
        isLLMGeneric: function(model) {
            if (model && model.savedModelType) {
                return model.savedModelType.startsWith("LLM_GENERIC");
            }
        },
        isLLM: function(model) {
            return this.isLLMGeneric(model) || this.isAgent(model) || this.isRetrievalAugmentedLLM(model);
        },
        isVisualMLModel: function(model) {
            if (model && model.savedModelType){
                return "DSS_MANAGED" === model.savedModelType;
            }
            return model && model.modeling && model.modeling.algorithm && (
                !["VIRTUAL_MLFLOW_PYFUNC", "VIRTUAL_PROXY_MODEL"].includes(model.modeling.algorithm));
        },
        hasModelData: function(model) {
            return !this.isAgent(model) && !this.isRetrievalAugmentedLLM(model);
        },
        createAndPinInsight: function(model, settingsPane) {
            const insight = {
                projectKey: ActiveProjectKey.get(),
                type: 'saved-model_report',
                params: {savedModelSmartId: SmartId.create(model.id, model.projectKey)},
                name: "Full report of model " + model.name
            };
            let params;
            if (settingsPane) {
                params = MLModelsUIRouterStates.savedModelPaneToDashboardTile(settingsPane, $stateParams);
            }

            CreateModalFromTemplate("/templates/dashboards/insights/create-and-pin-insight-modal.html", $rootScope, "CreateAndPinInsightModalController", function (newScope) {
                newScope.init(insight, params);
            });
        },
        // keep in sync with com.dataiku.dip.server.services.AccessibleObjectsService.AccessibleObject#fromTaggableObject
        asAccessibleObjects: function(models, contextProjectKey) {
            if (!models || !models.length) {
                return [];
            }
            return models.map(m => {
                const isLocal = m.projectKey == contextProjectKey;
                const obj = {
                    type: "SAVED_MODEL",
                    projectKey: m.projectKey,
                    id: m.id,
                    localProject: isLocal,
                    smartId: isLocal?m.id:m.projectKey+"."+m.id,
                    label: m.name,
                    isReaderAccessible: true,
                    subType: m.miniTask.taskType
                };
                obj.object = obj;
                return obj;
            });
        },
        newCodeAgent: function(agentName, zoneId, from) {
            return DataikuAPI.savedmodels.agents.createPython($stateParams.projectKey, agentName, zoneId).success(function (data) {
                WT1.event(
                    'agent-create', {
                        savedModelType: 'PYTHON_AGENT',
                        from: from
                    });
                $state.go('projects.project.savedmodels.savedmodel.versions', { projectKey: $stateParams.projectKey, smId: data.id });
            });
        },
        newCodeAgentPrompt: function(zoneId, from) {
            Dialogs.prompt($rootScope, 'New Code Agent', 'Agent name').then(function (agentName) {
                svc.newCodeAgent(agentName, zoneId, from).error(setErrorInScope.bind($rootScope));
            });
        },
        newVisualAgent: function(agentName,  agentMode,zoneId, from) {
            return DataikuAPI.savedmodels.agents.createToolsUsing($stateParams.projectKey, agentName, agentMode, zoneId).success(function (data) {
                WT1.event(
                    'agent-create', {
                        savedModelType: 'TOOLS_USING_AGENT',
                        from: from
                    });
                $state.go('projects.project.savedmodels.savedmodel.agent.design', { smId: data.id, fullModelId: `S-${data.projectKey}-${data.id}-${data.activeVersion}` });
            });
        },
        newVisualAgentPrompt: function(zoneId, from) {
            DataikuAPI.savedmodels.listHeads($stateParams.projectKey).success(function(data) {
                const newScope = $rootScope.$new();
                newScope.smNames = data.filter(sm => sm?.type === 'LLM_GENERIC_RAW' && sm?.projectKey === $stateParams.projectKey).map(sm => sm.name);
                newScope.showOnlyVisualAgents = true;
                CreateModalFromTemplate('/templates/savedmodels/agents/new-agent-selector-modal.html', newScope, 'AgentSelectorModalController');
            }).error(setErrorInScope.bind($rootScope));

        },
        newPluginAgent: function(agentId, agentName, zoneId, from) {
            return DataikuAPI.savedmodels.agents.createPlugin($stateParams.projectKey, agentId, agentName, zoneId).success(function (data) {
                WT1.event(
                    'agent-create', {
                        savedModelType: 'PLUGIN_AGENT',
                        from: from
                    });
                $state.go('projects.project.savedmodels.savedmodel.versions', { smId: data.id });
            });
        },
        newPluginAgentPrompt: function(agentLabel, agentId, zoneId, from) {
            Dialogs.prompt($rootScope, 'New ' + agentLabel, 'Agent name').then(function (agentName) {
                svc.newPluginAgent(agentId, agentName, zoneId, from).error(setErrorInScope.bind($rootScope));
            });
        },
        getAllPluginAgents: function() {
            const visibilityFilter = PluginConfigUtils.shouldComponentBeVisible($rootScope.appConfig.loadedPlugins);
            return $rootScope.appConfig.customAgents
                .filter(visibilityFilter)
                .map(agent => ({
                        label: agent.desc.meta?.label ?? "Custom agent " + agent.desc.id,
                        agentId: agent.desc.id,
                        description: agent.desc.meta?.description ?? ""
                    })
                );
        },
        getLlmEndpoint: function(savedModelType) {
            return savedModelType === 'RETRIEVAL_AUGMENTED_LLM' ? 'retrievalAugmentedLLMs' : 'agents';
        },
        getAgentReviewUrl: function(agentId, agentName, activeVersion, currentVersion) {
            const projectKey = $state.params.projectKey;

            return DataikuAPI.agentreviews.listHeads(projectKey).then(function(response) {
                const versionId = currentVersion?.versionId;
                const sameAgentARs = response.data.filter(a => a.agentId === agentId);
                let url;
                if (!sameAgentARs.length) {
                    url = $state.href("projects.project.agentreviews.list", {agentId, versionId, projectKey});
                }
                else if (sameAgentARs.length === 1 &&
                    (sameAgentARs[0].agentVersion === versionId || !sameAgentARs[0].agentVersion && versionId === activeVersion)) {
                    // has same version, or if AR version is set to active, versionId is the active one
                    url = $state.href("projects.project.agentreviews.agentreview.results", {agentReviewId: sameAgentARs[0].id, projectKey});
                }
                else {
                    url = $state.href("projects.project.agentreviews.list", {filterOn: agentId, projectKey});
                }
                return url;
            }).catch(() => {
                return $state.href("projects.project.agentreviews.list", {projectKey});
            });
        }
    };

    return svc;
});

/* ************************************ List / Right column  *************************** */

// this controller is only for the right panel of the main saved-model page (saved-model right panel from the flow or from the saved model list don't use this controller). If you want to reuse it elsewhere, make sure the dependency on SavedModelRefService is satisfied.
app.controller("SavedModelPageRightColumnActions", function($controller, $scope, $state, DataikuAPI, SavedModelRefService, ActivityIndicator, SavedModelRenameService, SmartId) {
    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.savedmodels.get(SavedModelRefService.contextProjectKey, SavedModelRefService.smartId).success((data) => {
        data.description = data.shortDesc;
        data.nodeType = SavedModelRefService.isForeign ? 'FOREIGN_SAVEDMODEL' : 'LOCAL_SAVEDMODEL';
        data.smartName = SmartId.create(data.name, SavedModelRefService.projectKey, SavedModelRefService.contextProjectKey);
        data.name = data.id;
        data.interest = {};

        $scope.selection = {
            selectedObject : data,
            confirmedItem : data,
        };
    }).error(setErrorInScope.bind($scope));

    $scope.renameSavedModel = function () {
        const savedModel = $scope.savedModel;

        SavedModelRenameService.renameSavedModel({
            scope: $scope,
            state: $state,
            projectKey: savedModel.projectKey,
            savedModelId: savedModel.id,
            savedModelName: savedModel.name,
            onSave: () => { ActivityIndicator.success("Saved"); }
        });
    };
});


app.directive('savedModelRightColumnSummary', function($controller, $state, $stateParams, SavedModelCustomFieldsService, $rootScope, FlowGraphSelection, SmartId,
    DataikuAPI, CreateModalFromTemplate, QuickView, TaggableObjectsUtils, SavedModelsService, LambdaServicesService, ActiveProjectKey, ActivityIndicator,
    FlowBuildService, AnyLoc, SelectablePluginsService, Logger, translate, TypeMappingService, SavedModelRenameService, PluginCategoryService, StateUtils) {

    return {
        templateUrl :'/templates/savedmodels/right-column-summary.html',

        link : function(scope, element, attrs) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;
            scope.LambdaServicesService = LambdaServicesService;

            scope.createAndPinInsight = SavedModelsService.createAndPinInsight;
            scope.isLLM = (sm) => SavedModelsService.isLLM(sm);

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.refreshData = function() {
                var projectKey = scope.selection.selectedObject.projectKey;
                var name = scope.selection.selectedObject.name;
                scope.canAccessObject = false;

                DataikuAPI.savedmodels.getFullInfo(ActiveProjectKey.get(), SmartId.create(name, projectKey)).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return; // too late!
                    }
                    scope.savedModelData = data;
                    scope.savedModel = data.model;
                    scope.savedModel.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;
                    scope.selection.selectedObject.interest = data.interest;
                    scope.isLocalSavedModel = projectKey == ActiveProjectKey.get();
                    scope.isMLSavedModel = SavedModelsService.isVisualMLModel(data.model) || SavedModelsService.isLLMGeneric(data.model);
                    scope.objectAuthorizations = data.objectAuthorizations;
                    scope.isRetrainableSavedModel = scope.isLocalSavedModel && !SavedModelsService.isExternalMLflowModel(scope.savedModel) && !SavedModelsService.isAgent(scope.savedModel) && !SavedModelsService.isRetrievalAugmentedLLM(scope.savedModel);
                    scope.isAgent = SavedModelsService.isAgent(data.model);
                    scope.isPromptableModel = SavedModelsService.isAgent(data.model) || SavedModelsService.isRetrievalAugmentedLLM(data.model) || SavedModelsService.isLLMGeneric(data.model);
                    scope.canAccessObject = true;

                    scope.selectablePlugins = SelectablePluginsService.listSelectablePlugins(scope.isPromptableModel ? {'PROMPTABLE_MODEL': 1} : {'SAVED_MODEL' : 1});
                    scope.noRecipesCategoryPlugins = PluginCategoryService.standardCategoryPlugins(scope.selectablePlugins, ['code'])
                }).catch(setErrorInScope.bind(scope));
            };

            scope.publishEnabled = function() {
                if (!$state.is('projects.project.savedmodels.savedmodel.prediction.report')
                    && !$state.is('projects.project.savedmodels.savedmodel.clustering.report')) {
                    return true;
                }
                if (SavedModelsService.isPartition($stateParams.fullModelId)) {
                    scope.publishDisabledReason = "Only the overall model can be published";
                    return false;
                }
                if (!SavedModelsService.isActiveVersion($stateParams.fullModelId, scope.smContext.savedModel)) {
                    scope.publishDisabledReason = "Only the active version can be published";
                    return false;
                }
                return true;
            }

            scope.$on("objectSummaryEdited", function() {
                DataikuAPI.savedmodels.save(scope.savedModel, {summaryOnly: true})
                .success(function(data) {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            });

            scope.$watch("selection.selectedObject",function(selectedObject) {
                if (selectedObject) {
                    const {projectKey, name} = selectedObject;
                    const smartId = SmartId.create(name, projectKey);

                    scope.isInSelectedObjectPage = $state.includes('projects.project.savedmodels.savedmodel');
                    scope.selectedObjectPageUrl = StateUtils.href.savedModel(smartId);
                    SavedModelsService.getAgentReviewUrl(smartId, selectedObject.interest?.details?.objectDisplayName).then((url)=> {
                        scope.agentReviewUrl = url;
                    });
                    scope.selectedObjectIcon = getIcon(selectedObject);
                    scope.viewInFlowUrl = StateUtils.href.flowLinkFromProps('SAVED_MODEL', projectKey, name, ActiveProjectKey.get());

                    // TODO sm-foreign-view epic sc-273588 to be removed once foreign view is fully implemented and feature-flag is removed
                    if (!$rootScope.featureFlagEnabled('enableForeignSavedModelView')) {
                        scope.foreignSavedModelView = {
                            get hideOpenAction() {
                                return !scope.objectAuthorizations.directAccessOnOriginal
                            }
                        }
                    } else {
                        scope.foreignSavedModelView = {
                            hideOpenAction: false,
                        }
                    }
                }

                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.savedModel = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.zoomToOtherZoneNode = function(zoneId) {
                const otherNodeId = scope.selection.selectedObject.id.replace(/zone__.+?__saved/, "zone__" + zoneId + "__saved");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId, id: graphVizUnescape(otherNodeId) }))
                }
                else {
                    scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[otherNodeId]);
                }
            }

            scope.renameSavedModel = function() {
                const savedModel = scope.savedModel;

                SavedModelRenameService.renameSavedModel({
                    scope: scope,
                    state: $state,
                    projectKey: savedModel.projectKey,
                    savedModelId: savedModel.id,
                    savedModelName: savedModel.name
                });
            }

            scope.trainModel = function() {
                const modalOptions = {
                    upstreamBuildable: scope.savedModelData.upstreamBuildable,
                    downstreamBuildable: scope.savedModelData.downstreamBuildable,
                };
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "SAVED_MODEL",
                            AnyLoc.makeLoc(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name), modalOptions);
            };

            scope.editCustomFields = function(editingTabIndex = 0) {
                if (!scope.selection.selectedObject) {
                    return;
                }
                DataikuAPI.savedmodels.getSummary(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name).success(function(data) {
                    let savedModel = data.object;
                    let modalScope = angular.extend(scope, {objectType: 'SAVED_MODEL', objectName: savedModel.name, objectCustomFields: savedModel.customFields, editingTabIndex});
                    CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                        SavedModelCustomFieldsService.saveCustomFields(savedModel, customFields);
                    });
                }).error(setErrorInScope.bind(scope));
            };

            scope.saveCustomFields = SavedModelCustomFieldsService.saveCustomFields;

            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);

            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "SAVED_MODEL", ActiveProjectKey.get(), scope.selection.selectedObject.name).success(function(data) {

                    scope.selection.selectedObject.interest = data;
                    scope.savedModelData.interest = data;

                }).error(setErrorInScope.bind(scope));
            }

            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            scope.$on("$destroy", interestsListener);

            scope.cannotCreateAPIServiceMsg = function() {
                if (!scope.canWriteProject()) {
                    return translate("PROJECT.PERMISSIONS.WRITE_ERROR", "You don't have write permissions for this project");
                }
                const msg = SavedModelsService.nonAPIServiceableMsg(scope.savedModel);
                if (msg) {
                    return msg;
                }
                return null;
            }

            function getIcon(selectedObject) {
                return TypeMappingService.mapSavedModelSubtypeToIcon(
                    selectedObject.taskType || (selectedObject.miniTask || {}).taskType,
                    selectedObject.backendType,
                    selectedObject.predictionType || (selectedObject.miniTask || {}).predictionType,
                    selectedObject.savedModelType,
                    selectedObject.externalSavedModelType || (selectedObject.proxyModelConfiguration && selectedObject.proxyModelConfiguration.protocol),
                    24
                );
            }
        }
    }
});

app.service("SavedModelCustomFieldsService", function($rootScope, TopNav, DataikuAPI, ActivityIndicator, WT1){
    let svc = {};

    svc.saveCustomFields = function(savedModel, newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'SAVED_MODEL'});
        let oldCustomFields = angular.copy(savedModel.customFields);
        savedModel.customFields = newCustomFields;
        return DataikuAPI.savedmodels.save(savedModel, {summaryOnly: true})
            .success(function(data) {
                ActivityIndicator.success("Saved");
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), savedModel.customFields);
                $rootScope.$broadcast('reloadGraph');
            })
            .error(function(a, b, c) {
                savedModel.customFields = oldCustomFields;
                setErrorInScope.bind($rootScope)(a, b, c);
            });
    };

    return svc;
});

app.controller('_ModelListController', function($scope, $controller, $stateParams, CreateModalFromComponent, createExternalSavedModelModalDirective, createExternalSavedModelSelectorModalDirective, $state, TypeMappingService, SmartId, StateUtils) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'smartName', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["smartName", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.maxItems = 20;

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.savedmodels.savedmodel.versions", {projectKey : $stateParams.projectKey, smId : data.id});
    };

    $scope.prepareListItems = function(listItems) {
        // dirty things to handle the discrepancy between the types of selected objects
        // which can have info displayed in the right panel
        listItems.forEach(sm => {
            // handles foreign objects
            sm.foreign = sm.projectKey !== $stateParams.projectKey;
            sm.smartId = SmartId.create(sm.id, sm.projectKey);
            sm.smartName = SmartId.create(sm.name, sm.projectKey);
            sm.href = StateUtils.href.savedModel(sm.smartId, $stateParams.projectKey);
            sm.taskType = sm.type; // the right panel icon is decided by the taskType field (in part), but that info is stored on type for the list item

            sm.name = sm.id;
            let proxyModelProtocol = null;
            if (sm.proxyModelConfiguration) {
                proxyModelProtocol = sm.proxyModelConfiguration.protocol;
            }
            sm.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(sm.type, sm.backendType, sm.predictionType, sm.savedModelType, proxyModelProtocol, 24);
            sm.colorClassName = sm.foreign ? 'shared' : TypeMappingService.mapSavedModelTypeToClassColor(sm.savedModelType);
        });

        return listItems;
    };

    $scope.newExternalSavedModel = function(externalModelTypeName) {
        if (externalModelTypeName) {
            CreateModalFromComponent(createExternalSavedModelModalDirective, { externalModelTypeName }, ['modal-wide']);
        }
        else {
            CreateModalFromComponent(createExternalSavedModelSelectorModalDirective, {}, ['modal-wide']);
        }
    };
});

app.controller('SavedModelListController', function($scope, $controller, $stateParams, DataikuAPI, TopNav, $rootScope, translate) {
    $controller('_ModelListController', {$scope: $scope});

    const showForeignFilter = $rootScope.featureFlagEnabled('allowForeignInProjectObjectLists');
    if(showForeignFilter) {
        $scope.enableForeignFilterWithLabels({
            all: translate('PROJECT.SAVED_MODEL_LIST.SHOW_FOREIGN.ALL', 'All models'),
            onlyLocal: translate('PROJECT.SAVED_MODEL_LIST.SHOW_FOREIGN.ONLY_LOCAL', 'Only models from the project'),
            onlyForeign: translate('PROJECT.SAVED_MODEL_LIST.SHOW_FOREIGN.ONLY_FOREIGN', 'Only foreign models'),
        });
    }

    $scope.list = function() {
        const needsForeign = showForeignFilter && $scope.selection.filterQuery.foreign !== 'false';
        DataikuAPI.savedmodels.listHeads($stateParams.projectKey, needsForeign).success(function(data) {
            $scope.listItems = $scope.prepareListItems(data.filter(sm => sm?.type !== 'LLM_GENERIC_RAW'));
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();
});

app.controller('GenAIListController', function($scope, $controller, $rootScope, $stateParams, DataikuAPI, TopNav, CreateModalFromTemplate, SavedModelsService) {
    $controller('_ModelListController', {$scope: $scope});

    $scope.list = function() {
        DataikuAPI.savedmodels.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = $scope.prepareListItems(data.filter(sm => sm?.type === 'LLM_GENERIC_RAW'));
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    $scope.pluginAgentsTypes = $rootScope.appConfig.customAgents.map(ca => ca.desc);

    $scope.newAgent = function() {
        const newScope = $scope.$new();
        newScope.smNames = $scope.listItems.filter(sm => sm?.type === 'LLM_GENERIC_RAW' && sm?.projectKey === $stateParams.projectKey).map(sm => sm.smartName);
        CreateModalFromTemplate('/templates/savedmodels/agents/new-agent-selector-modal.html', newScope, 'AgentSelectorModalController');
    };
});

app.filter("niceGenAIModelType", function($rootScope) {
    return function(sm) {
        if (sm.savedModelType === 'PYTHON_AGENT') {
            return "Code Agent";
        } else if (sm.savedModelType === 'TOOLS_USING_AGENT') {
            return "Visual Agent";
        } else if (sm.savedModelType === 'PLUGIN_AGENT') {
            const pluginAgent = $rootScope.appConfig.customAgents.find(ca => ca.agentType === sm.pluginAgentType);
            return pluginAgent?.desc.meta?.label ?? "Custom Agent";
        } else if (sm.savedModelType === 'RETRIEVAL_AUGMENTED_LLM') {
            return "Retrieval-augmented LLM";
        } else if (sm.savedModelType === 'LLM_GENERIC') {
            return "Fine-tuned LLM";
        } else {
            return sm.savedModelType;
        }
    };
});

/**
 * This service places a layer between the raw $stateParams and the saved model reference info
 * It aims to make handling local and foreign saved models easily.
 * Every place that may support foreign saved models should use SavedModelRefService.something instead of $stateParams directly
 *
 * SavedModelRefService also offers a method to test if a route is supported for foreign view (used by StateUtils.*.savedModel, probably not required elsewhere)
 *
 * SavedModelRefService watches route changes when inside the context of a saved model page in order to:
 * - redirect to the source project automatically when following a link that is not supported for foreign models (means links inside the saved model page scope don't need to handle this explicitly)
 * - keep the resolved projectKey / id values up to date
 *
 * Provides the following values:
 * - contextProjectKey: the project we are in ($stateParams.projectKey)
 * - projectKey: the source project key for the model (=contextProjectKey for local models)
 * - smId the model id (without the project key)
 * - smartId the model smart id (=smId for a local mode, and =sourceProjectKey.smId for a foreign model)
 * - isForeign, booleans marking if the model is local or foreign
 *
 * Since it's based on the stateParams, make sure the refresh method is called every time the route is changed
 */
app.service("SavedModelRefService", function(SmartId, $state, $stateParams, $location) {
    const SavedModelRefService = {
        bindToScope,
        supportsForeignView,
        forceRedirectToSourceProject,
    };

    const supportedRoutes = new Set([
        'projects.project.savedmodels.savedmodel',
        'projects.project.savedmodels.savedmodel.versions',
    ]);

    function refresh() {
        const {projectKey: contextProjectKey, smId: smartId} = $stateParams;
        const {projectKey , id} = SmartId.resolve(smartId, contextProjectKey);
        SavedModelRefService.contextProjectKey = contextProjectKey;
        SavedModelRefService.projectKey = projectKey;
        SavedModelRefService.smartId = smartId;
        SavedModelRefService.fullId = projectKey + '.' + id;
        SavedModelRefService.smId = id;
        SavedModelRefService.isForeign = projectKey !== contextProjectKey;
    }

    function isModelPage(route) {
        return route.startsWith('projects.project.savedmodels.savedmodel')
    }

    function supportsForeignView(route) {
        return supportedRoutes.has(route);
    }

    function bindToScope($scope) {
        // watch route change to check if we are trying to go to a savedmodel route that doesn't support foreign view. If yes, and if the current params match foreign view, then redirect to the source project
        $scope.$on('$stateChangeStart', (ev, toState, toParams, fromState, fromParams, options) => {
            const toRoute = toState.name;
            if(isModelPage(toRoute) && !supportsForeignView(toRoute) && toParams.smId.includes('.')) {
                // If the state change was triggered by a href, the url has already changed, which means we need to use {location: 'replace'} to prevent an additional history entry
                // If the state change was triggered by a ui-sref or a state.go, the url hasn't already changed, so we must not use {location: 'replace'}... except if the call was explicitly done with {location: 'replace'}
                // If we let the transition finish then redirect (as done in redirectState in app.js), the controller & templates would try to load & make api calls and errors.
                // We really want to be able to cancel the transition here.
                ev.preventDefault();
                const {projectKey , id: smId} = SmartId.resolve(toParams.smId);
                const shouldReplace = options?.location === 'replace' || $state.href(toState, toParams) === $location.url();
                $state.go(toState, {...toParams, projectKey, smId}, shouldReplace ? {location: 'replace'} : undefined);
            }
        });

        // make sure the ref info are updated when the state changes - most those refreshes are probably redundant as the main controller is probably re-instantiated on model change, but let's be safe
        $scope.$on('$stateChangeSuccess', () => refresh());
        refresh();
    }

    function forceRedirectToSourceProject() {
        $state.go($state.current.name, {
            ...$stateParams,
            projectKey: SavedModelRefService.projectKey,
            smId: SavedModelRefService.smId
        }, {location: 'replace'});
    }

    return SavedModelRefService;
});

app.controller("SavedModelController", function($scope, $rootScope, Assert, DataikuAPI, CreateModalFromTemplate, $state,
    $stateParams, SavedModelsService, MLExportService, ActiveProjectKey, WebAppsService, FullModelLikeIdUtils,
    createOrAppendMELikeToModelComparisonModalDirective, CreateModalFromComponent, MLModelsUIRouterStates, ExportModelDatasetService,
    AnyLoc, FlowBuildService, SmartId, TopNav, CodeStudiosService, SavedModelRefService) {
    $scope.versionsContext = {}
    $scope.smContext = {};
    $scope.uiState = {};

    SavedModelRefService.bindToScope($scope);
    $scope.smRef = SavedModelRefService;

    $scope.clearVersionsContext = function(){
        // eslint-disable-next-line no-undef
        clear($scope.versionsContext);
    };
    $scope.isAgent = function(snippetData) {
        return SavedModelsService.isAgent(snippetData);
    };
    $scope.agentModelIsDirty = function() {
        return $rootScope.agentModelIsDirty();
    };
    $scope.saveAgentModel = function() {
        return $rootScope.saveAgentModel();
    };

    $scope.isRetrievalAugmentedLLM = function(snippetData) {
        return SavedModelsService.isRetrievalAugmentedLLM(snippetData);
    };
    $scope.retrievalAugmentedModelIsDirty = function() {
        return $rootScope.retrievalAugmentedModelIsDirty();
    };
    $scope.saveRetrievalAugmentedModel = function() {
        return $rootScope.saveRetrievalAugmentedModel();
    };

    $scope.isLLM = function(snippetData) {
        return SavedModelsService.isLLM(snippetData);
    };

    $scope.showMainPanel = function(tabsType){
        return ['SAVED_MODEL', 'PLUGIN_AGENT-SAVED_MODEL-VERSION', 'RETRIEVAL-AUGMENTED_LLM-SAVED_MODEL-VERSION'].includes(tabsType);
    };

    $scope.trainModel = function() {
        FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "SAVED_MODEL",
                            AnyLoc.makeLoc(SavedModelRefService.projectKey, SavedModelRefService.smId));
    };

    $scope.goToAnalysisModelFromVersion = function(){
        Assert.trueish($scope.smContext.model, 'no model data');
        Assert.trueish($scope.smContext.model.smOrigin, 'no origin analysis');

        const id = $scope.smContext.model.smOrigin.fullModelId;
        const elements = FullModelLikeIdUtils.parse(id);

        var params =  {
            projectKey: elements.projectKey, // ProjectKey from SavedModels is updated when reading it
            analysisId: elements.analysisId,
            mlTaskId: elements.mlTaskId,
            fullModelId: id
        }

        if ($scope.smContext.model.smOrigin.origin == "EXPORTED_FROM_ANALYSIS") {
           $scope.goToAnalysis(params);
         } else {
            CreateModalFromTemplate("/templates/savedmodels/go-to-analysis-model-modal.html", $scope, null, function(newScope){
                newScope.go = function(){
                    newScope.dismiss();
                    $scope.goToAnalysis(params);
                }
            })
        }
    }

    $scope.goToAnalysis = function(params) {
        let state = "projects.project.analyses.analysis.ml.";
        let api;
        if ($state.includes("projects.project.savedmodels.savedmodel.prediction")) {
            state += "predmltask";
            api = DataikuAPI.analysis.pml;
        } else {
            state += "clustmltask";
            api = DataikuAPI.analysis.cml;
        }
        api.getTaskStatus(params.projectKey, params.analysisId, params.mlTaskId)
            .success(function (data) {
                if (data.fullModelIds.some(model => model.fullModelId.fullId === params.fullModelId)) {
                    state += ".model.report"
                }
                else {
                    state += ".list.results"
                }
                $state.go(state, params);
            })
            .error(function () {
                setErrorInScope.bind($scope);
                $state.go(`${state}.list.results`, params);
            });
    }

    $scope.goToExperimentRunFromVersion = function() {
        Assert.trueish($scope.smContext.model, 'no model data');
        Assert.trueish($scope.smContext.model.mlflowOrigin, 'no MLflow origin');
        Assert.trueish($scope.smContext.model.mlflowOrigin.type === 'EXPERIMENT_TRACKING_RUN', 'MLflow origin not from experiment/run');
        Assert.trueish($scope.smContext.model.mlflowOrigin.runId, 'MLflow origin malformed (no runId)');

        const origin = $scope.smContext.model.mlflowOrigin;
        const params = {
            projectKey: $stateParams.projectKey,
            runId: origin.runId
        };
        $state.go("projects.project.experiment-tracking.run-details", params);
    }

    $scope.showPredictedData = function() {
        let model = $scope.smContext.model;
        if (!model) {
            return false;
        }

        if ($scope.isExternalMLflowModel(model)) {
            // MLFlow and external models do not support predicted data straighforwardly
            return false;
        }

        // For partitioned models only show predicted data tab for individual partitions
        const isPartitionedModel = model.coreParams.partitionedModel && model.coreParams.partitionedModel.enabled;
        const isIndividualPartitionedModel = SavedModelsService.isPartition(model.fullModelId);
        if (isPartitionedModel && !isIndividualPartitionedModel) {
            return false;
        }

        const algoName = model.actualParams.resolved.algorithm;
        return !/^(MLLIB|SPARKLING|VERTICA|PYTHON_ENSEMBLE|SPARK_ENSEMBLE|KERAS)/i.test(algoName);
    }

    $scope.showExportModel = function() {
        return $scope.smContext.model && MLExportService.showExportModel($scope.appConfig);
    };
    $scope.mayExportModel = function(type) {
        return MLExportService.mayExportModel($scope.smContext.model, type);
    };
    $scope.downloadDocForbiddenReason = function() {
        return MLExportService.downloadDocForbiddenReason($scope.appConfig, $scope.smContext.model);
    }
    $scope.downloadDoc = function() {
        return MLExportService.downloadDoc($scope);
    }
    $scope.exportModelModal = function() {
        MLExportService.exportModelModal($scope, $scope.smContext.model)
    }
    $scope.disableExportModelModalReason = function() {
        return MLExportService.disableExportModelModalReason($scope.smContext.model);
    }

    $scope.exportTrainTestSets = function() {
        ExportModelDatasetService.exportTrainTestSets($scope, $scope.smContext.model.fullModelId);
    };
    $scope.exportTrainTestSetsForbiddenReason = function() {
        return ExportModelDatasetService.exportTrainTestSetsForbiddenReason($scope.smContext.model, $scope.canWriteProject());
    };

    $scope.exportPredictedDataForbiddenReason = function() {
        return ExportModelDatasetService.exportPredictedDataForbiddenReason($scope.smContext.model);
    };

    $scope.comparisonForbiddenReason = function() {
        if (!$scope.smContext.model) {
            return null;
        }
        if($scope.smContext.model.coreParams.backendType === "DEEP_HUB") {
            return "Computer vision model comparison is not supported";
        }
        if($scope.smContext.model.coreParams.taskType !== "PREDICTION") {
            return "Only prediction models can be compared";
        }
        if($scope.smContext.model.trainInfo.state !== 'DONE') {
            return "Cannot compare a model being trained or that failed";
        }
        if($scope.smContext.model.coreParams.partitionedModel && $scope.smContext.model.coreParams.partitionedModel.enabled) {
            return "Cannot compare a partitioned model";
        }
        if($scope.smContext.model.modeling.ensemble_params) {
            return "Cannot compare an ensembled model";
        }
        if(($scope.smContext.model.modeling.algorithm === "VIRTUAL_MLFLOW_PYFUNC") &&
            ($scope.smContext.model.coreParams.prediction_type === undefined)) {
            return "Non tabular MLflow models cannot be compared";
        }
        if(($scope.smContext.model.modeling.algorithm === "VIRTUAL_PROXY_MODEL") &&
            ($scope.smContext.model.coreParams.prediction_type === undefined)) {
            return "Non tabular External Models cannot be compared";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.compareModel = function(type) {
        CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
            fullIds: [$scope.smContext.model.fullModelId],
            modelTaskType: $scope.smContext.model.coreParams.prediction_type, // ModelTaskType values encompass PredictionType possible values. So this "cast" will work.
            allowImportOfRelatedEvaluations: true,
            suggestedMCName: `Compare 1 model version from ${$scope.smContext.model.userMeta.name}`,
            projectKey: $stateParams.projectKey,
            trackFrom: 'saved-model-page'
        });
    };

    $scope.createAndPinInsight = SavedModelsService.createAndPinInsight;
    $scope.getPublishDisabledReason = () => {
        for (const pane of ['exploration-constraints', 'exploration-results']) {
            if ($scope.uiState && $scope.uiState.settingsPane && $scope.uiState.settingsPane.includes(pane)) {
                return 'Publishing this page on dashboards is not supported';
            }
        }
        return null;
    }

    $scope.isActiveVersion = SavedModelsService.isActiveVersion;
    $scope.isPartition = SavedModelsService.isPartition;
    $scope.isExternalMLflowModel = SavedModelsService.isExternalMLflowModel;
    $scope.isMLflowModel = SavedModelsService.isMLflowModel;
    $scope.isVisualMLModel = SavedModelsService.isVisualMLModel;

    $scope.$on("$destroy", $scope.clearVersionsContext);

    $scope.canEditCodeAgentInCodeStudio = false;
    $scope.editCodeAgentInCodeStudio = function() {
        $scope.saveAgentModel().then(function() {
            CodeStudiosService.editFileInCodeStudio($scope, "code_agents", $rootScope.agentModelCurrentFile());
        }).catch(setErrorInScope.bind($scope));
    }

    DataikuAPI.savedmodels.get(SavedModelRefService.contextProjectKey, SavedModelRefService.smartId).success(function(sm) {
        $scope.savedModel = sm;
        if ($scope.savedModel.savedModelType === "LLM_GENERIC") { // TODO sm-foreign-page sc-273588 something to do here when we support llms
             DataikuAPI.savedmodels.getFullInfo(ActiveProjectKey.get(), SmartId.create(sm.id, sm.projectKey)).then(function({data}) {
                $scope.savedModelFullInfo = data;  // to have the back to Fine tune recipe button
             });
        }

        if ($scope.savedModel.savedModelType === "PYTHON_AGENT") { // TODO sm-foreign-page sc-273588 something to do here when we support python agent
            DataikuAPI.codeStudioTemplates.canEditInCodeStudio($stateParams.projectKey).success(function(data) {
                $scope.canEditCodeAgentInCodeStudio = data.canEdit;
            })
        }

        TopNav.setPageTitle($scope.savedModel.name);
    }).error(setErrorInScope.bind($scope));

    // controls if we can see the 'go to origin' link for foreign model, and if we show the go to analysis button.
    if(SavedModelRefService.isForeign) {
        DataikuAPI.projects.getProjectAccessInfo(SavedModelRefService.projectKey)
            .then(({data}) => $scope.uiState.canReadOriginProject = data.hasReadConf)
            .catch(() => {}) // having an error here means no access, no need to show the error to the user
    } else {
        $scope.uiState.canReadOriginProject = true;
    }
});

app.controller("_SavedModelGovernanceStatusController", function($scope, $rootScope, FullModelLikeIdUtils, DataikuAPI) {
    $scope.getGovernanceStatus = function(fullModelId, partitionedModel) {
        $scope.modelGovernanceStatus = undefined;
        if (!$rootScope.appConfig.governEnabled) return;
        if (!fullModelId) return;
        // ignore non-saved model version
        if (!FullModelLikeIdUtils.isSavedModel(fullModelId)) return;
        // ignore partition model versions
        if (FullModelLikeIdUtils.isPartition(fullModelId)) return;
        // ignore partitioned model versions
        if (partitionedModel && partitionedModel.enabled) return;

        const fmi = FullModelLikeIdUtils.parse(fullModelId);
        $scope.modelGovernanceStatus = { loading: true };
        DataikuAPI.savedmodels.getModelVersionGovernanceStatus(fmi.projectKey, fmi.savedModelId, fullModelId).success(function(data) {
            $scope.modelGovernanceStatus = { loading: false, data: data };
        }).error(function(a,b,c,d) {
            const fatalAPIError = getErrorDetails(a,b,c,d);
            fatalAPIError.html = getErrorHTMLFromDetails(fatalAPIError);
            $scope.modelGovernanceStatus = { loading: false, error: fatalAPIError };
        });
    };
});

app.filter('savedModelMLTaskHref', function($state, $stateParams, ActiveProjectKey, FullModelLikeIdUtils) {
    return function(sm) {
        if (!sm || !sm.lastExportedFrom) return;

        const elements = FullModelLikeIdUtils.parse(sm.lastExportedFrom);

        const params =  {
            projectKey: elements.projectKey,  // ProjectKey from SavedModels is updated when reading it
            analysisId: elements.analysisId,
            mlTaskId: elements.mlTaskId
        };

        const type = sm.type || sm.miniTask.taskType;
        let state = "projects.project.analyses.analysis.ml.";
        if (type == "PREDICTION") {
            state += "predmltask.list.results";
        } else {
            state += "clustmltask.list.results";
        }
        return $state.href(state, params);
    };
});

/* ************************************ Versions listing *************************** */



app.controller("SavedModelVersionsController", function($scope, Assert, DataikuAPI, $state, $stateParams, SavedModelRefService, TopNav, $controller, Logger,
                                                        GraphZoomTrackerService, MLDiagnosticsService,
                                                        createOrAppendMELikeToModelComparisonModalDirective, createProxySavedModelVersionModalDirective,
                                                        CreateModalFromComponent, SavedModelsService, RecipeComputablesService, WT1, MetricsUtils, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS, SmartId,
                                                        CreateModalFromTemplate, MlflowImportModelsService){
    TopNav.setItem(TopNav.ITEM_SAVED_MODEL, SavedModelRefService.smartId);
    angular.extend($scope, MLDiagnosticsService);

    if (!$stateParams.fromFlow) { // Do not change the focus item zoom if coming from flow
        GraphZoomTrackerService.setFocusItemByFullId("savedmodel", SavedModelRefService.fullId);
    }
    $scope.initialized = false;
    $scope.snippetSource = 'SAVED';
    $scope.smRef = SavedModelRefService;

    // TODO sm-foreign-page sc-284835 make sure this works well with the future RA-LLM multi-version screen
    $scope.mainPanelHideOptions = SavedModelRefService.isForeign ? {selectedCount: true, massActions: true, newVersion: true} : {};
    $scope.modelSnippetHideOptions = SavedModelRefService.isForeign ? {selectCheckbox: true, makeActive: true, deleteModel: true} : {};

    $scope.isModelDone = function() {
        return true;
    };
    $scope.isModelRunning = function() {
        return false;
    };
    $scope.isSessionRunning = function() {
        return false;
    };

    $scope.isMetricFailed = MetricsUtils.isMetricFailed;
    $scope.getSpecificCustomMetricResult = MetricsUtils.getSpecificCustomMetricResult;

    $scope.isExternalMLflowModel = function(snippetData) {
        return SavedModelsService.isExternalMLflowModel(snippetData);
    }
    $scope.getExternalModelDetails = function(model) {
        return AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === model.proxyModelConfiguration.protocol);
    }
    $scope.isProxyModel = function(snippetData) {
        return SavedModelsService.isProxyModel(snippetData);
    }
     $scope.isMLflowModel = function(snippetData) {
        return SavedModelsService.isMLflowModel(snippetData);
    }
    $scope.isAgent = function(snippetData) {
        return SavedModelsService.isAgent(snippetData);
    };
    $scope.isRetrievalAugmentedLLM = function(snippetData) {
        return SavedModelsService.isRetrievalAugmentedLLM(snippetData);
    };
    $scope.isLLMSavedModel = function(snippetData) {
        return SavedModelsService.isLLMSavedModel(snippetData)
    };
    $scope.isLLMGeneric = function(snippetData) {
        return SavedModelsService.isLLMGeneric(snippetData);
    };

    $scope.isARecipeOutput = false;

    $scope.newProxySavedModelVersion = function() {
        CreateModalFromComponent(createProxySavedModelVersionModalDirective, { savedModel: $scope.savedModel, smStatus: $scope.smStatus }, ['modal-wide']);
    }

    $scope.openImportMlflowModelVersionModal = function () {
        MlflowImportModelsService.openImportMlflowModelVersionModal({ savedModel: $scope.savedModel, predictionType: $scope.savedModel.miniTask.predictionType });
    }

    $scope.isTimeseriesPrediction = function () {
        return $scope.savedModel &&
               $scope.savedModel.miniTask &&
               $scope.savedModel.miniTask.predictionType == "TIMESERIES_FORECAST";
    }

    $scope.getInitialOrder = function (snippetData) {
        if ($scope.isLLMSavedModel(snippetData)) {
            $scope.selection.orderQuery = 'snippet.creationTag.lastModifiedOn';
        } else {
            $scope.selection.orderQuery = 'versionId';
        }
    }

    DataikuAPI.savedmodels.get(SavedModelRefService.contextProjectKey, SavedModelRefService.smartId).success(function(savedModel) {
        WT1.event(
            'saved-model-open', {
                savedModelType: savedModel.savedModelType,
                predictionType: (savedModel.miniTask || {}).predictionType,
                taskType: (savedModel.miniTask || {}).taskType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            });

        $scope.savedModel = savedModel;
        $scope.smContext.savedModel = savedModel;
        if (SavedModelsService.isLLM(savedModel)) {
            TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_SAVED_MODEL, "versions");
        } else {
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, TopNav.TABS_SAVED_MODEL, "versions");
        }
        TopNav.setItem(
            TopNav.ITEM_SAVED_MODEL,
            SavedModelRefService.smartId,
            {
                name: SmartId.create(savedModel.name, savedModel.projectKey),
                smartId: SavedModelRefService.smartId,
                taskType: (savedModel.miniTask || {}).taskType,
                backendType: (savedModel.miniTask || {}).backendType,
                savedModelType: savedModel.savedModelType,
                predictionType: (savedModel.miniTask || {}).predictionType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            }
        );

        if ($scope.savedModel.miniTask) {
            const taskType = $scope.savedModel.miniTask.taskType;
            Assert.trueish(['PREDICTION', 'CLUSTERING'].includes(taskType), 'Unknown task type');
            if (taskType === 'PREDICTION') {
                if ($stateParams.redirectToActiveVersion) {
                    $state.go('projects.project.savedmodels.savedmodel.prediction.report', {
                        fullModelId: `S-${$scope.savedModel.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                    }, {location: 'replace'});
                } else {
                    $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.prediction';
                    $controller("PredictionSavedModelVersionsController", { $scope });
                }
            } else if (taskType === "CLUSTERING") {
                if ($stateParams.redirectToActiveVersion) {
                    $state.go('projects.project.savedmodels.savedmodel.clustering.report', {
                        fullModelId: `S-${$scope.savedModel.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                    }, {location: 'replace'});
                } else {
                    $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.clustering';
                    $controller("ClusteringSavedModelVersionsController" , {$scope});
                }
            }
        } else if ($scope.savedModel.savedModelType === "LLM_GENERIC") {
            // TODO sm-foreign-page sc-284835 support foreign view
            if(SavedModelRefService.isForeign) {
                return SavedModelRefService.forceRedirectToSourceProject();
            }

            if ($stateParams.redirectToActiveVersion) {
                $state.go('projects.project.savedmodels.savedmodel.llmGeneric.report', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.llmGeneric';
                $controller("LLMGenericModelVersionsController", {$scope, $stateParams});
            }
        } else if ($scope.savedModel.savedModelType === "PYTHON_AGENT") {
            //TODO sm-foreign-page sc-273588 support foreign view
            if(SavedModelRefService.isForeign) {
                return SavedModelRefService.forceRedirectToSourceProject();
            }

            if ($stateParams.redirectToActiveVersion || ($stateParams.fromFlow && $scope.savedModel.activeVersion)) {
                $state.go('projects.project.savedmodels.savedmodel.agent.design', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.agent';
                $controller("PythonAgentModelVersionsController", { $scope });
            }
        } else if ($scope.savedModel.savedModelType === "PLUGIN_AGENT") {
            //TODO sm-foreign-page sc-273588 support foreign view
            if(SavedModelRefService.isForeign) {
                return SavedModelRefService.forceRedirectToSourceProject();
            }

            $state.go('projects.project.savedmodels.savedmodel.agent.design', {
                fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`
            }, { location: 'replace' });
        } else if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            //TODO sm-foreign-page sc-273588 support foreign view
            if(SavedModelRefService.isForeign) {
                return SavedModelRefService.forceRedirectToSourceProject();
            }

            if ($stateParams.redirectToActiveVersion || ($stateParams.fromFlow && $scope.savedModel.activeVersion)) {
                $state.go('projects.project.savedmodels.savedmodel.agent.design', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.agent';
                $controller("ToolsUsingAgentModelVersionsController", { $scope });
            }
        } else if ($scope.savedModel.savedModelType === "RETRIEVAL_AUGMENTED_LLM") {
            // TODO sm-foreign-page sc-284835 support foreign view
            if(SavedModelRefService.isForeign) {
                return SavedModelRefService.forceRedirectToSourceProject();
            }

            if ($stateParams.redirectToActiveVersion || ($stateParams.fromFlow && $scope.savedModel.activeVersion)) {
                $state.go('projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.retrievalaugmentedllm';
                $controller("RetrievalAugmentedLLMModelVersionsController", { $scope });
            }

        } else {
            Logger.error("Unknown saved model type: " + $scope.savedModel.savedModelType);
        }

        if(!SavedModelRefService.isForeign) { // used to show the train button, no point making an api query if model is foreign, we want it hidden
            RecipeComputablesService.getComputablesMap({type:"python"}).then((map) => {
                $scope.isARecipeOutput = map[SavedModelRefService.smartId].alreadyUsedAsOutputOf || null;
            });
        }

    }).error(setErrorInScope.bind($scope));

    $scope.canDeleteSelectedModels = function() {
        return $scope.selection?.selectedObjects?.some(version => !version.active);
    };

    $scope.differentSampleWeights = function() {
        if ($scope.allSnippets?.length > 0 &&
            $scope.savedModel?.miniTask && // irrelevant for agents which do not have miniTask
            ["BINARY_CLASSIFICATION", "REGRESSION", "MULTICLASS"].includes($scope.savedModel.miniTask.predictionType)) {
            let firstSampleWeight = $scope.allSnippets[0].sampleWeightsVariable;
            return !$scope.allSnippets.every(model => (model.sampleWeightsVariable === firstSampleWeight));
        }
    }

    $scope.differentClassAveraging = function() {
        if ($scope.allSnippets?.length > 0 &&
            $scope.savedModel?.miniTask && // irrelevant for agents which do not have miniTask
            $scope.savedModel.miniTask.predictionType === "MULTICLASS") {
            let firstClassAveraging = $scope.allSnippets[0].classAveragingMethod;
            return !$scope.allSnippets.every(model => (model.classAveragingMethod === firstClassAveraging));
        }
    }

    $scope.differentCausalWeights = function() {
        if ($scope.allSnippets?.length > 0 &&
            $scope.savedModel?.miniTask && // irrelevant for agents which do not have miniTask
            ["CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION"].includes($scope.savedModel.miniTask.predictionType)) {
            let firstCausalWeight = $scope.allSnippets[0].causalWeighting;
            return !$scope.allSnippets.every(model => (model.causalWeighting === firstCausalWeight));
        }
    }

    $scope.comparisonForbiddenReason = function() {
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length < 1) {
            return "At least one model version must be selected";
        }
        if ($scope.savedModel.savedModelType === 'LLM_GENERIC' 
            || $scope.savedModel.savedModelType === 'PYTHON_AGENT' 
            || $scope.savedModel.savedModelType === 'PLUGIN_AGENT'
            || $scope.savedModel.savedModelType === 'TOOLS_USING_AGENT'
            || $scope.savedModel.savedModelType === 'RETRIEVAL_AUGMENTED_LLM') {
            return 'Large Language Models cannot be compared';
        }
        if($scope.savedModel.miniTask.backendType === "DEEP_HUB") {
            return "Computer vision model comparison is not supported";
        }
        if($scope.savedModel.miniTask.taskType !== "PREDICTION") {
            return "Only prediction models can be compared";
        }
        if(!$scope.selection.selectedObjects.every(model => model.snippet.predictionType === $scope.selection.selectedObjects[0].snippet.predictionType)) {
            return "Model versions must have the same prediction type";
        }
        if($scope.selection.selectedObjects.some(model => model.snippet.partitionedModelEnabled)) {
            return "Partitioned models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => model.snippet.isEnsembled)) {
            return "Ensembled models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => (model.snippet.algorithm === "VIRTUAL_MLFLOW_PYFUNC") && (model.snippet.predictionType === undefined))) {
            return "Non tabular MLflow models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => (model.snippet.algorithm === "VIRTUAL_PROXY_MODEL") && (model.snippet.predictionType === undefined))) {
            return "Non tabular External models cannot be compared";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.compareSelectedModels = function() {
        const nbModels = $scope.selection.selectedObjects.length;
        const smName = $scope.savedModel.name;

        CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
            fullIds: $scope.selection.selectedObjects.map(me => me.snippet.fullModelId),
            modelTaskType: $scope.selection.selectedObjects[0].snippet.predictionType, // ModelTaskType values encompass PredictionType possible values. So this "cast" will work.
            allowImportOfRelatedEvaluations: true,
            suggestedMCName: `Compare ${nbModels} model versions from ${smName}`,
            projectKey: $stateParams.projectKey,
            trackFrom: 'saved-model-versions-list'
        });
    }

    $scope.duplicationForbiddenReason = function() {
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length > 1) {
            return "Only one model version must be selected";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.duplicateSelectedModel = function() {
        CreateModalFromTemplate("/templates/savedmodels/llm/duplicate-llm-sm-version-modal.html", $scope);
    }
});


app.controller("DuplicateLlmSmVersionController", function($scope, $state, $stateParams, SavedModelHelperService, ActiveProjectKey) {
    $scope.newVersion = {
        versionId : SavedModelHelperService.suggestedVersionId($scope.smStatus),
    }

    $scope.modelType = $scope.isAgent($scope.savedModel) ? "agent" : "retrieval-augmented LLM";
    const designPath = $scope.isAgent($scope.savedModel) ? "projects.project.savedmodels.savedmodel.agent.design" : "projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design";

    $scope.create = function() {
        const selectedModel = $scope.selection.selectedObjects[0];

        $scope.baseAPI.duplicateVersion(ActiveProjectKey.get(), $stateParams.smId, selectedModel.versionId, $scope.newVersion.versionId)
            .success(function(data) {
                $scope.dismiss();
                $state.go(designPath, {
                    fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId
                });
            })
            .error(setErrorInScope.bind($scope));
    }
});

app.controller("PredictionSavedModelVersionsController", function($scope, DataikuAPI, Fn, PMLFilteringService, Dialogs,
                                                                  PartitionedModelsService, CustomMetricIDService, SavedModelRefService){
    angular.extend($scope, PartitionedModelsService);
    $scope.initialized = false;
    $scope.uiState.currentMetricIsCustom = false;
    function filterIntermediatePartitionedModels(status) {
        if (status.task && status.task.partitionedModel && status.task.partitionedModel.enabled) {
            /* Keeping all models exported from analysis (they don't have intermediate versions) */
            const analysisModels = status.versions
                .filter(model => model.smOrigin && model.smOrigin.origin === 'EXPORTED_FROM_ANALYSIS');

            /* Grouping models trained from recipe by their JobId */
            const recipeModelsByJobId = status.versions
                .filter(model => model.smOrigin && model.smOrigin.origin === 'TRAINED_FROM_RECIPE')
                .reduce((map, model) => {
                    map[model.smOrigin.jobId] = (map[model.smOrigin.jobId] || []).concat(model);
                    return map;
                }, {});

            /* Keeping most recent or active models in those groups */
            const recipeMostRecentModels = Object.entries(recipeModelsByJobId)
                .map((jobEntries) =>
                    jobEntries[1].reduce((mostRecentModel, currentModel) => {
                        if (!mostRecentModel || currentModel.active) {
                            return currentModel;
                        }

                        return (currentModel.snippet.trainDate > mostRecentModel.snippet.trainDate) ? currentModel : mostRecentModel;
                    }, null));

            status.versions = analysisModels.concat(recipeMostRecentModels);
        }
    }

    $scope.refreshStatus = function(){
        DataikuAPI.savedmodels.prediction.getStatus(SavedModelRefService.contextProjectKey, SavedModelRefService.smartId)
            .then(({data}) => {
                data.versions.map(function(v) { v.snippet.versionRank = +v.versionId || 0; });
                $scope.smStatus = data;
                $scope.setMainMetric();
                $scope.possibleMetrics = PMLFilteringService.getPossibleMetrics($scope.smStatus.task);
                $scope.allSnippets = data.versions.map(item => item.snippet);

                $scope.allMetrics = $scope.possibleMetrics;

                PMLFilteringService.getPossibleCustomMetrics($scope.allSnippets).map(item => {
                    $scope.allMetrics.push([item.id, item.name]);
                });

                $scope.allMetricsHooks = $scope.allMetrics.map((m) => m[0]);

                if ($scope.smStatus.task.modeling && !$scope.uiState.currentMetric) {
                    const modelingMetrics = $scope.smStatus.task.modeling.metrics;
                    $scope.uiState.currentMetric = modelingMetrics.evaluationMetric === "CUSTOM" ? CustomMetricIDService.getCustomMetricId(modelingMetrics.customEvaluationMetricName) : modelingMetrics.evaluationMetric;
                    $scope.uiState.currentMetricIsCustom = false;
                }
            })
            .catch(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.initialized = true;
            });
    }

    $scope.refreshStatus();

    $scope.setMainMetric = function() {
        if(!$scope.smStatus || !$scope.smStatus.versions || !$scope.uiState.currentMetric) { return; }

        $scope.uiState.currentMetricIsCustom = CustomMetricIDService.checkMetricIsCustom($scope.uiState.currentMetric);

        PMLFilteringService.setMainMetric($scope.smStatus.versions,
            ["snippet"],
            $scope.uiState.currentMetricIsCustom ? CustomMetricIDService.getCustomMetricName($scope.uiState.currentMetric) : $scope.uiState.currentMetric,
            $scope.smContext.savedModel.miniTask.modeling.metrics.customMetrics,
            $scope.uiState.currentMetricIsCustom
        );
    };
    $scope.$watch('uiState.currentMetric', $scope.setMainMetric);

    $scope.makeActive = function(data) {
        Dialogs.confirmPositive($scope, "Set model as active", "Do you want to set this model version as the active scoring version ?").then(function(){
            DataikuAPI.savedmodels.prediction.setActive(SavedModelRefService.projectKey, SavedModelRefService.smId, data.versionId)
                .success(function(data) {
                    $scope.refreshStatus();
                    if (data.schemaChanged) {
                        let warningMessage;
                        if ($scope.isTimeseriesPrediction()) {
                            warningMessage = "The newly selected model has a different preparation script schema, custom metric configuration or time series quantile configuration compared to the previous version.\n"
                        } else {
                            warningMessage = "The newly selected model has a different preparation script schema or custom metric configuration compared to the previous version.\n"
                        }
                        Dialogs.ackMarkdown($scope, "Schema changed", warningMessage +
                            "This change may affect the output schema of any downstream scoring and evaluation recipes."
                        );
                    }
                })
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteSelectedModels = function() {
        const deletableModels = $scope.selection.selectedObjects.filter(model => !model.active);
        const plural = deletableModels.length > 1 ? 's' : '';
        Dialogs.confirmAlert($scope, "Delete model" + plural, "Are you sure you want to delete " + deletableModels.length + " version" + plural + "? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.prediction.deleteVersions(SavedModelRefService.projectKey, SavedModelRefService.smId, deletableModels.map(Fn.prop('versionId')))
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteModel = function(model) {
        Dialogs.confirmAlert($scope, "Delete model", "Are you sure you want to delete \"" + model.snippet.userMeta.name + "\"? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.prediction.deleteVersions(SavedModelRefService.projectKey, SavedModelRefService.smId, [model.versionId])
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };
});


app.controller("ClusteringSavedModelVersionsController", function($scope, DataikuAPI, Fn, CMLFilteringService, Dialogs, SavedModelRefService){
    $scope.initialized = false;
    $scope.refreshStatus = function(){
        return DataikuAPI.savedmodels.clustering.getStatus(SavedModelRefService.contextProjectKey, SavedModelRefService.smartId)
            .then(({data}) => {
                $scope.smStatus = data;
                $scope.setMainMetric();
                $scope.possibleMetrics = CMLFilteringService.getPossibleMetrics($scope.smStatus.task);
                if (!$scope.uiState.currentMetric) {
                    $scope.uiState.currentMetric = "SILHOUETTE"; // Dirty tmp
                }
            })
            .catch(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.initialized = true;
            });
    }

    $scope.setMainMetric = function() {
        if(!$scope.smStatus || !$scope.smStatus.versions) { return; }
        CMLFilteringService.setMainMetric($scope.smStatus.versions,
            ["snippet"],
            $scope.uiState.currentMetric,
            $scope.smContext.savedModel.miniTask.modeling.metrics.customMetrics);
    };

    $scope.makeActive = function(data) {
        Dialogs.confirmPositive($scope, "Set model as active", "Do you want to set this model version as the active scoring version ?").then(function(){
            DataikuAPI.savedmodels.clustering.setActive(SavedModelRefService.projectKey, SavedModelRefService.smId, data.versionId)
                .success(function(data) {
                    $scope.refreshStatus();
                    if (data.schemaChanged) {
                        Dialogs.ack($scope, "Schema changed", "The preparation script schema of the selected version is different than " +
                            "the previously selected version, this may affect the ouput schema of downstream scoring recipes.");
                    }
                })
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteSelectedModels = function() {
        const deletableModels = $scope.selection.selectedObjects.filter(model => !model.active);
        const plural = deletableModels.length > 1 ? 's' : '';
        Dialogs.confirmAlert($scope, "Delete model" + plural, "Are you sure you want to delete " + deletableModels.length + " version" + plural + "? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.clustering.deleteVersions(SavedModelRefService.projectKey, SavedModelRefService.smId, deletableModels.map(Fn.prop('versionId')))
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteModel = function(model) {
        Dialogs.confirmAlert($scope, "Delete model", "Are you sure you want to delete \"" + model.snippet.userMeta.name + "\"? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.clustering.deleteVersions(SavedModelRefService.projectKey, SavedModelRefService.smId, [model.versionId])
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    // Watchers & init

    $scope.$watch('uiState.currentMetric', $scope.setMainMetric);
    $scope.refreshStatus();
});

app.controller("_CommonLLMModelVersionsController", function($scope, DataikuAPI, Fn, $stateParams, $controller, Dialogs, ActiveProjectKey, CreateModalFromComponent, makeActiveLLMGenericModalDirective) {
    $scope.refreshDeployments = function() {
        DataikuAPI.savedmodels.llmGeneric.deployments.list(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
            $scope.deployments = data;
        }).error(setErrorInScope.bind($scope));
    }
    $scope.refreshStatus = function(){
        DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data) {
            $scope.smStatus = data;
            $scope.allSnippets = data.versions.map(item => item.snippet);
            $scope.initialized = true;
        }).error(setErrorInScope.bind($scope));
        $scope.refreshDeployments();
    };
    $scope.refreshStatus();

    $scope.makeActive = function(data) {
        if (data.snippet.savedModelType === "LLM_GENERIC") {
            CreateModalFromComponent(makeActiveLLMGenericModalDirective, { deployments: $scope.deployments, savedModel: $scope.savedModel, newVersionId: data.versionId, llmType: data.snippet.llmSMInfo.llmType })
                .then($scope.refreshStatus);
        } else if (["PYTHON_AGENT", "TOOLS_USING_AGENT", "RETRIEVAL_AUGMENTED_LLM"].includes(data.snippet.savedModelType)) {
            Dialogs.confirmPositive($scope, "Set agent version as active", "Do you want to set this agent version as the active version when running this model?")
                .then(function() {
                    $scope.baseAPI.setActive($scope.savedModel.projectKey, $scope.savedModel.id, data.versionId)
                        .success($scope.refreshStatus)
                        .error(setErrorInScope.bind($scope))
                }, function() {
                    // Dialog closed
                });
        }
    };

    $scope.deleteSelectedModels = function() {
        const deletableModels = $scope.selection.selectedObjects.filter(model => !model.active);
        const plural = deletableModels.length > 1 ? 's' : '';
        Dialogs.confirmAlert($scope, "Delete model" + plural, "Are you sure you want to delete " + deletableModels.length + " version" + plural + "? This action is irreversible. Any remote resource will be deleted as well.").then(function(){
            $scope.baseAPI.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, deletableModels.map(Fn.prop('versionId')))
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteModel = function(model) {
        Dialogs.confirmAlert($scope, "Delete model", "Are you sure you want to delete \"" + model.snippet.userMeta.name + "\"? This action is irreversible. Any remote resource will be deleted as well.").then(function(){
            $scope.baseAPI.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, [model.versionId])
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };
});


app.controller("NewPluginAgentModalController", function($scope, DataikuAPI, $stateParams, $state) {
    $scope.newAgent = {

    }

    $scope.chooseType = function(id) {
        $scope.newAgent.type = id;
    }

    $scope.create = function() {
        DataikuAPI.savedmodels.agents.createPlugin($stateParams.projectKey, $scope.newAgent.type, $scope.newAgent.name)
            .success(function(data) {
                $scope.dismiss();
                $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                    fullModelId: "S-" + $stateParams.projectKey + "-" + $stateParams.smId + "-v1" // TODO
                });
            })
            .error(setErrorInScope.bind($scope))
    }

});


app.controller("NewPythonAgentVersionController", function($scope, DataikuAPI, $state, $stateParams, Dialogs, AgentCodeTemplates, SavedModelHelperService) {
    $scope.VERSION_FROM_SCRATCH = "VERSION_FROM_SCRATCH";
    $scope.VERSION_FROM_EXISTING = "VERSION_FROM_EXISTING";
    $scope.versionCreationModes = [
        {
            "id": $scope.VERSION_FROM_SCRATCH,
            "icon": "dku-icon-plus-circle-outline-24",
            "title": "From scratch",
            "desc": "Start with a blank slate; templates are available for a quick setup.",
        },
        {
            "id": $scope.VERSION_FROM_EXISTING,
            "icon": "dku-icon-copy-24",
            "title": "From existing version",
            "desc": "Start from an existing version of this agent:",
        },
    ]
    $scope.uiState = {
        versionId: SavedModelHelperService.suggestedVersionId($scope.smStatus),
        selectedMode: $scope.VERSION_FROM_SCRATCH,
        currentTabIndex: $scope.savedModel.inlineVersions.length ? 0 : 1,
        templateId: 'simple-llm-mesh-proxy-streamed',
        versionIdToCopy: $scope.smStatus.activeVersionId,
    };

    $scope.agentCodeTemplates = AgentCodeTemplates;

    $scope.selectModeKeydown = function(event, modeId) {
        if (event.key === "Enter") {
            $scope.uiState.selectedMode = modeId
        }
    }
    $scope.selectCodeTemplateKeydown = function(event, codeTemplateId) {
        if (event.key === "Enter") {
            $scope.uiState.templateId = codeTemplateId
        }
    }
    $scope.showNext = function() {
        return $scope.uiState.currentTabIndex === 0 && $scope.uiState.selectedMode === $scope.VERSION_FROM_SCRATCH;
    }
    $scope.showPrevious = function() {
        if ($scope.savedModel.inlineVersions.length) {
            return $scope.uiState.currentTabIndex === 1;
        } else {
            // First version of this agent, we don't show the first tab at all.
            return false;
        }
    }
    $scope.previousTab = function() {
        $scope.uiState.currentTabIndex = 0;
    }
    $scope.nextTab = function() {
        $scope.uiState.currentTabIndex = 1;
    }
    $scope.create = function() {
        if ($scope.uiState.selectedMode === $scope.VERSION_FROM_SCRATCH) {
            const smiv = {
                code: AgentCodeTemplates[$scope.uiState.templateId].codeSample,
                pythonAgentSettings: {
                    supportsImageInputs: false,
                }
            }

            if (AgentCodeTemplates[$scope.uiState.templateId].quickTestQuery) {
                smiv.quickTestQueryStr = JSON.stringify(AgentCodeTemplates[$scope.uiState.templateId].quickTestQuery, null, 2);
            }

            if (AgentCodeTemplates[$scope.uiState.templateId].supportsImageInputs) {
                smiv.pythonAgentSettings.supportsImageInputs = true;
            }

            DataikuAPI.savedmodels.agents.createVersion($stateParams.projectKey, $stateParams.smId, $scope.uiState.versionId, smiv)
                .success(function(data) {
                    $scope.dismiss();
                    $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                        fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId
                    });
                })
                .error(setErrorInScope.bind($scope))
        } else if ($scope.uiState.selectedMode === $scope.VERSION_FROM_EXISTING) {
            DataikuAPI.savedmodels.agents.duplicateVersion($stateParams.projectKey, $stateParams.smId, $scope.uiState.versionIdToCopy, $scope.uiState.versionId)
                .success(function(data) {
                    $scope.dismiss();
                    $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                        fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId
                    });
                })
                .error(setErrorInScope.bind($scope));
        }
    }
});

app.controller("PythonAgentModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, Dialogs,  CreateModalFromTemplate){
    $controller("_CommonLLMModelVersionsController", {$scope});

    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.agents;
    $scope.newAgentVersion = function() {
        CreateModalFromTemplate("/templates/savedmodels/agents/new-python-agent-version-modal.html", $scope);
    }

    const unwatchSmStatus = $scope.$watch('smStatus', function (newSmStatus) {
        if (newSmStatus) {
            if (newSmStatus.versions.length === 0) {
                $scope.newAgentVersion();
            }
            unwatchSmStatus();
        }
    });
});

app.controller("NewLlmSmVersionController", function($scope, DataikuAPI, $state, $stateParams, Dialogs, SavedModelHelperService) {
    $scope.VERSION_FROM_SCRATCH = "VERSION_FROM_SCRATCH";
    $scope.VERSION_FROM_EXISTING = "VERSION_FROM_EXISTING";
    $scope.modelType = $scope.isAgent($scope.savedModel) ? "agent" : "retrieval augmented LLM";
    $scope.designPath = $scope.isAgent($scope.savedModel) ? "projects.project.savedmodels.savedmodel.agent.design" : "projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design";


    $scope.versionCreationModes = [
        {
            "id": $scope.VERSION_FROM_SCRATCH,
            "icon": "dku-icon-plus-circle-outline-24",
            "title": "From scratch",
            "desc": "Start with a blank slate.",
        },
        {
            "id": $scope.VERSION_FROM_EXISTING,
            "icon": "dku-icon-copy-24",
            "title": "From existing version",
            "desc": "Start from an existing version of this "+ $scope.modelType +":",
        },
    ]
    $scope.uiState = {
        versionId : SavedModelHelperService.suggestedVersionId($scope.smStatus),
        selectedMode: $scope.VERSION_FROM_SCRATCH,
        versionIdToCopy: $scope.smStatus.activeVersionId,
    }

    $scope.selectModeKeydown = function(event, modeId) {
        if (event.key === "Enter") {
            $scope.uiState.selectedMode = modeId
        }
    }
    $scope.create = function() {
        if ($scope.uiState.selectedMode === $scope.VERSION_FROM_SCRATCH) {
            const smiv = {};
            if ($scope.isAgent($scope.savedModel)) {
                smiv.toolsUsingAgentSettings = {
                    mode: $scope.savedModel.inlineVersions[0].toolsUsingAgentSettings?.mode,
                };
            } else if ($scope.isRetrievalAugmentedLLM($scope.savedModel)) {
                smiv.ragllmSettings = {kbRef: $scope.savedModel.inlineVersions[0].ragllmSettings?.kbRef || null};
            } else {
                throw new Error("Unsupported saved model type for LLM version creation modal: " + $scope.savedModel.savedModelType);
            }


            $scope.baseAPI.createVersion($stateParams.projectKey, $stateParams.smId, $scope.uiState.versionId, smiv)
                .success(function (data) {
                    $scope.dismiss();
                    $state.go($scope.designPath, {fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId});
                })
                .error(setErrorInScope.bind($scope))
        } else if ($scope.uiState.selectedMode === $scope.VERSION_FROM_EXISTING) {
            $scope.baseAPI.duplicateVersion($stateParams.projectKey, $stateParams.smId, $scope.uiState.versionIdToCopy, $scope.uiState.versionId)
                .success(function (data) {
                    $scope.dismiss();
                    $state.go($scope.designPath, {fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId});
                })
                .error(setErrorInScope.bind($scope));
        }
    }
});

app.controller("ToolsUsingAgentModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, Dialogs,  CreateModalFromTemplate){
    $controller("_CommonLLMModelVersionsController", {$scope});
    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.agents;
    $scope.newAgentVersion = function() {
        CreateModalFromTemplate("/templates/savedmodels/llm/new-llm-sm-version-modal.html", $scope);
    }

    if ($scope.savedModel.inlineVersions.length === 0) {
        $scope.newAgentVersion();
    }
});

app.controller("LLMGenericModelVersionsController", function($scope, $controller, DataikuAPI) {
    $controller("_CommonLLMModelVersionsController", {$scope});

    $scope.baseAPI = DataikuAPI.savedmodels.llmGeneric;
});

app.controller("RetrievalAugmentedLLMModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, CreateModalFromTemplate){
    $controller("_CommonLLMModelVersionsController", {$scope});
    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.retrievalAugmentedLLMs;

    $scope.newRaLlmVersion = function() {
        CreateModalFromTemplate("/templates/savedmodels/llm/new-llm-sm-version-modal.html", $scope);
    }

    if ($scope.savedModel.inlineVersions.length === 0) {
        $scope.newRaLlmVersion();
    }
});

app.controller("AgentSavedModelHistoryController", function($scope, $stateParams, DataikuAPI, ActiveProjectKey, TopNav) {
    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(savedModel) {
        $scope.savedModel = savedModel;
        TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_SAVED_MODEL, "history");
        TopNav.setItem(
            TopNav.ITEM_SAVED_MODEL,
            $stateParams.smId,
            {
                name: savedModel.name,
                taskType: (savedModel.miniTask || {}).taskType,
                backendType: (savedModel.miniTask || {}).backendType,
                savedModelType: savedModel.savedModelType,
                predictionType: (savedModel.miniTask || {}).predictionType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            }
        );
    });
});

app.component("llmQuicktest", {
    bindings: {
        projectKey: "<",
        savedModelId: "<",
        savedModelType: "<",
        currentVersionId: "<",
        refreshEditor: "<",
        quickTestQueryStr: "=",
        saveModelFunction: "&",
        onTest: "&",
    },
    templateUrl : '/templates/savedmodels/llm-quicktest.html',
    controller: function($rootScope, $scope, $timeout, $q, DataikuAPI, ClipboardUtils, CodeMirrorSettingService, ConnectionsService, SavedModelHelperService, TraceExplorerService, WT1) {
        const $ctrl = this;
        $ctrl.codeMirrorSettingService = CodeMirrorSettingService;
        $ctrl.quickTestResponse = null;
        $ctrl.quickTestEnabled = false;
        $ctrl.quickTestError = null;
        $ctrl.quickTestRequestCancellable = null;

        $scope.activeTab = 'response';
        $scope.uiState = {
            response: 'TEXT', // or FULL
            trace: 'LLM', // or FULL or GRAPHICAL
        };

        $ctrl.$onChanges = function(changes) {
            if (changes.savedModelType?.currentValue) {
                const currentModelType = changes.savedModelType.currentValue;

                if (currentModelType === "RETRIEVAL_AUGMENTED_LLM") {
                    $ctrl.api = DataikuAPI.savedmodels.retrievalAugmentedLLMs;
                } else {
                    $ctrl.api = DataikuAPI.savedmodels.agents;
                }
                $ctrl.quickTestEnabled = true;
            }
            if (changes.currentVersionId?.currentValue) {
                $ctrl.validateQuicktestQuery();
            }
        };

        $ctrl.validateQuicktestQuery = function() {
            try {
                JSON.parse($ctrl.quickTestQueryStr);
                $ctrl.quickTestError = null;
            } catch (e) {
                $ctrl.quickTestError = "Invalid test query, it should be a JSON object: " + e;
            }
        };

        $ctrl.quickTest = function() {
            if (!$ctrl.quickTestEnabled) {
                return;
            }
            $ctrl.quickTestEnabled = false;

            $ctrl.saveModelFunction().then(() => {
                $ctrl.onTest();
                $ctrl.quickTestRequestCancellable = $q.defer();

                $ctrl.api.test($ctrl.projectKey, $ctrl.savedModelId, $ctrl.currentVersionId, $ctrl.quickTestQueryStr, $ctrl.quickTestRequestCancellable.promise).then((response) => {
                    if ($ctrl.savedModelType == "RETRIEVAL_AUGMENTED_LLM") {
                        WT1.event(
                            'ra-llm-test-run', {
                                savedModelType: $ctrl.savedModelType,
                            });
                    }
                    else {
                        WT1.event(
                            'agent-test-run', {
                                savedModelType: $ctrl.savedModelType,
                            });
                    }
                    $ctrl.quickTestResponse = response.data;
                    $ctrl.quickTestResponse.response.credentialsError = ConnectionsService.getCredentialsError($ctrl.quickTestResponse.response.errorMessage);
                    $ctrl.graphicalTrace = SavedModelHelperService.generateAsciiTreeFromTrace(response.data.fullTrace);
                    $scope.activeTab = 'response';
                    $rootScope.lastTrace = response.data.fullTrace;
                    resetErrorInScope($scope);
                }).catch((error) => {
                    if (error && error.status === -1) {
                        // request was canceled
                    } else {
                        setErrorInScope.bind($scope)(error);
                    }
                    $ctrl.quickTestResponse = null;
                    $rootScope.lastTrace = null;
                }).finally(() => {
                    $ctrl.quickTestEnabled = true;
                    $ctrl.quickTestRequestCancellable = null;
                });
            });
        };

        $ctrl.copyTraceToClipboard = function() {
            $scope.traceCopied = true;
            ClipboardUtils.copyToClipboard(JSON.stringify($scope.showFullTrace ? $ctrl.quickTestResponse.fullTrace : $ctrl.quickTestResponse.traceOfPython));
            $timeout(() => { $scope.traceCopied = false}, 5000);
        };

        $ctrl.openTraceExplorer = function() {
            TraceExplorerService.openTraceExplorer($ctrl.quickTestResponse.fullTrace)
        };

        const llmStopDevKernelUnsubscribe = $rootScope.$on('llmStopDevKernel', () => {
            if ($ctrl.quickTestRequestCancellable) {
                $ctrl.quickTestRequestCancellable.resolve("Request canceled by user");
            }
        });
        $scope.$on('$destroy', llmStopDevKernelUnsubscribe);
    },
});

app.component('llmQuickchat', {
    bindings: {
        projectKey: '<',
        savedModelId: '<',
        savedModelType: '<',
        currentVersion: '<',
        currentVersionUpdate: '<',
        hasActiveLlm: '<',
        showContextSelector: '<',
        onSave: '&',
        onChat: '&',
        isDirty: '&'
    },
    templateUrl : '/templates/savedmodels/llm-quickchat.html',
    controller: function($scope, $q, DataikuAPI, PromptUtils, $stateParams, PromptChatService, SavedModelsService, Dialogs, RALLM_SEARCH_INPUT_STRATEGY_MAP) {
        const $ctrl = this;
        $ctrl.activeChatTab = 'chat';
        $ctrl.warnAboutRaLlmSearchMode = false;
        let prevVersionId;

        $ctrl.$onChanges = function (changes) {
            if (changes.currentVersionUpdate?.currentValue != null) {
                if ($ctrl.currentVersion?.versionId !== prevVersionId) {
                    $ctrl.sessionId = $stateParams.fullModelId;
                    $ctrl.cachedChat = PromptChatService.getChat($ctrl.sessionId);
                    $ctrl.chatLog = $ctrl.getLog();
                    prevVersionId = $ctrl.currentVersion.versionId;
                }
                $ctrl.warnAboutRaLlmSearchMode = false;
                if ($ctrl.savedModelType === "RETRIEVAL_AUGMENTED_LLM") {
                    const ragllmSettings = $ctrl.currentVersion?.ragllmSettings;
                    if (ragllmSettings) {
                        $ctrl.warnAboutRaLlmSearchMode = RALLM_SEARCH_INPUT_STRATEGY_MAP[ragllmSettings.searchInputStrategySettings.strategy].warnIfUsedInChat;
                    }
                }
            }
        };

        $ctrl.resetChatWithDialog = () => {
            Dialogs.confirm($scope, 'Reset chat', 'Are you sure you want to reset this chat session? All current messages will be deleted.')
                .then(() => {
                    $ctrl.resetChat();
                })
                .catch(setErrorInScope.bind($scope));
        };

        $ctrl.resetChat = () => {
            $scope.$broadcast('prompt-chat--resetChat');
            PromptChatService.resetChat($ctrl.sessionId);
        };

        $ctrl.getLog = () => {
            return PromptChatService.getLog($ctrl.sessionId);
        };

        $ctrl.sendChatMessage = (newMessage, abortController, chatMessages, callback) => {
            const lastUserMessage = newMessage;
            const context = PromptChatService.getChatContextInitializeIfNeeded($ctrl.sessionId);

            $ctrl.activeLLMIcon = {
                name: PromptUtils.getLLMIcon($ctrl.savedModelType),
                style: PromptUtils.getLLMColorStyle('agent')
            };

            return $ctrl.save().then(() => {
                $ctrl.onChat();
                return DataikuAPI.savedmodels[SavedModelsService.getLlmEndpoint($ctrl.savedModelType)].chat($ctrl.projectKey, $ctrl.savedModelId, $ctrl.currentVersion.versionId, { chatMessages, lastUserMessage, context }, callback, abortController).catch(setErrorInScope.bind($scope));
            }).catch(setErrorInScope.bind($scope));
        };

        $ctrl.onChatResponse = (lastMessageId, messages, enrichedMessages) => {
            $ctrl.$lastMessageId = lastMessageId;
            $ctrl.chatLog = $ctrl.getLog();
            PromptChatService.setChat($ctrl.sessionId, lastMessageId, enrichedMessages);
        };

        $ctrl.onStopStreaming = (sessionId, activeSession, messages) => {
            const stoppedMessage = {
                id: generateRandomId(7),
                parentId: activeSession.runData?.parentId,
                version: activeSession.runData?.version,
                message: {
                    role: 'assistant',
                    content: activeSession.rawMessage
                },
                llmStructuredRef: {
                    id: buildSavedModelLlmId(),
                    type: $ctrl.savedModelType,
                    savedModelSmartId: $ctrl.savedModelId, 
                    savedModelVersionId: $ctrl.currentVersion.versionId
                },
                completionSettings: {}
            };
            // if no raw message, throw an error
            if (activeSession.rawMessage === '') {
                stoppedMessage.error = true;
                stoppedMessage.llmError = 'Response generation was interrupted.';
            }

            const chatMessages = angular.copy(messages);
            chatMessages[stoppedMessage.id] = stoppedMessage;
            const response = {
                lastMessageId: stoppedMessage.id,
                chatMessages
            };
            return $q.when(response);
        };

        $ctrl.save = () => {
            return $ctrl.onSave();
        };

        $ctrl.setContext = (context) => {
            PromptChatService.setChatContext($ctrl.sessionId, context);
        };

        $ctrl.settingsComparator = (newMessage, oldMessage) => {
            const changes = [];
            switch ($ctrl.savedModelType) {
                case 'RETRIEVAL_AUGMENTED_LLM': {
                    const ragSettings = {
                        'llmId': 'LLM',
                        'enforceDocumentLevelSecurity': 'Enforce document-level security',
                        'filter': 'Static filtering',
                        'performFiltering': 'Filters',
                        'retrievalColumns': 'Columns to retrieve',
                        'retrievalSource': 'Retrieval source',
                        'searchType': 'Search type',
                        'similarityThreshold': 'Similarity threshold',
                        'useAdvancedReranking': 'Advanced reranking',
                        'rrfRankWindowSize': 'Rank window size',
                        'maxDocuments': 'Number of documents',
                        'rrfRankConstant': 'Rank constant',
                        'mmrK': 'Diversity selection documents',
                        'mmrDiversity': 'Diversity vs relevancy factor',
                        'contextMessage': 'Custom retrieval prompt',
                        'searchInputStrategySettings': 'Search input strategy settings',
                        'sourcesSettings': 'Source settings',
                        'outputFormat': 'Legacy source output format',
                        'includeContentInSources': 'Print document sources',
                        'ragSpecificGuardrails': 'Guardrail settings',
                    };
                    const newRagSettings = newMessage.$customSettings?.ragllmSettings;
                    const oldRagSettings = oldMessage.$customSettings?.ragllmSettings;
                    PromptChatService.compareCompletionSettings(newRagSettings?.completionSettings, oldRagSettings?.completionSettings, changes);
                    Object.entries(ragSettings).forEach(([key, label]) => {
                        if (!angular.equals(newRagSettings?.[key], oldRagSettings?.[key])) {
                            if (['enforceDocumentLevelSecurity', 'filter', 'printSources'].includes(key)) {
                                changes.push(`${label} ${newRagSettings?.[key] ? 'enabled' : 'disabled' }`);
                            } else if (['llmId', 'similarityThreshold', 'maxDocuments', 'rrfRankWindowSize', 'rrfRankConstnat', 'mmrK', 'mmrDiversity'].includes(key)) {
                                changes.push(`${label} changed from ${oldRagSettings?.[key] ?? 'not defined'} to ${newRagSettings?.[key] ?? 'not defined'}`);
                            } else {
                                changes.push(`${label} modified`);
                            }
                        }
                    });

                    break;
                }
                case 'TOOLS_USING_AGENT': {
                    const newAgentSettings = newMessage.$customSettings?.toolsUsingAgentSettings;
                    const oldAgentSettings = oldMessage.$customSettings?.toolsUsingAgentSettings;
                    PromptChatService.compareCompletionSettings(newAgentSettings?.completionSettings, oldAgentSettings?.completionSettings, changes);
                    if (newAgentSettings?.llmId !== oldAgentSettings?.llmId) {
                        changes.push(`LLM has changed from ${oldAgentSettings?.llmId ?? 'not defined'} to ${newAgentSettings?.llmId ?? 'not defined'}`);
                    }
                    if (newAgentSettings?.systemPromptAppend !== oldAgentSettings?.systemPromptAppend) {
                        changes.push('Instructions have been modified');
                    }
                    if (!angular.equals(newAgentSettings?.tools, oldAgentSettings?.tools)) {
                        changes.push('Tools have been modified');
                    }
                    break;
                }
                case 'PYTHON_AGENT':
                    if (newMessage.$customSettings?.code !== oldMessage.$customSettings?.code) {
                        changes.push('Code has been modified');
                    }
                    break;
                case 'PLUGIN_AGENT':
                    if (!angular.equals(newMessage.$customSettings?.pluginAgentConfig, oldMessage.$customSettings?.pluginAgentConfig)) {
                        changes.push('Plugin settings have been modified');
                    }
                    break;
            }

            return changes;
        };

        function buildSavedModelLlmId() {
            let llmIdRoot = $ctrl.savedModelType === 'RETRIEVAL_AUGMENTED_LLM' ? `retrieval-augmented-llm:${$ctrl.projectKey}.` : 'agent:';

            return llmIdRoot + $ctrl.savedModelId + ':' + $ctrl.currentVersion.versionId;
        }
    }
});

app.component('llmTestPanel', {
    bindings: {
        projectKey: '<',
        savedModel: '<',
        currentVersion: '<',
        currentVersionUpdate: '<',
        availableLlms: '<',
        availableTools: '<',
        availableAgents: '<',
        hasActiveLlm: '<',
        showContextSelector: '<',
        displayMode: '<',
        selectedBlockId: '<',
        onSave: "&",
        onChat: "&",
        onSelectBlock: "&",
        onDiagramUpdate: "&",
        onToggleHighlight: "&",
        onChangeDisplayMode: '&',
        isDirty: '&'
    },
    templateUrl : '/templates/savedmodels/llm-test-panel.html',
    controller: function($rootScope, $element, $scope, $state, DataikuAPI, SavedModelsService, localStorageService, CreateModalFromTemplate, ActivityIndicator) {
        const $ctrl = this;
        $ctrl.stopDevKernelEnabled = true;
        $ctrl.tabs = [{
            id: 'chat',
            name: 'Chat',
        }, {
            id: 'test',
            name: 'Test Query'
        }];
        let tabsSet = false;

        $ctrl.save = () => {
            return $ctrl.onSave();
        };

        $ctrl.$onChanges = (changes) => {
            if (!tabsSet && $ctrl.savedModel && $ctrl.savedModel.savedModelType) {
                if ($ctrl.savedModel.savedModelType === 'TOOLS_USING_AGENT') {
                    $ctrl.tabs.push({
                        id: 'view',
                        name: 'View'
                    });
                }
                const displayMode = localStorageService.get("llmPreferredTestingMode");
                $ctrl.setDisplayMode($ctrl.tabs.find(tab => tab.id === displayMode) ? displayMode : 'chat');
                tabsSet = true;
            }
            if (changes.currentVersionUpdate) {
                if ($ctrl.savedModel?.savedModelType === 'TOOLS_USING_AGENT' && $ctrl.currentVersion?.toolsUsingAgentSettings) {
                    $ctrl.agentDiagramSettings = angular.copy($ctrl.currentVersion.toolsUsingAgentSettings);
                }
            }
        }

        $ctrl.computeSuperDiagram = function(){
            DataikuAPI.savedmodels.agents.getBlocksGraphDiagram($ctrl.projectKey, $ctrl.savedModel.id,
                     $ctrl.currentVersion.versionId, $ctrl.agentDiagramSettings).success(function(data){
                        $ctrl.superDiagram = data;

                        const newSVGElement = $($ctrl.superDiagram.svg);
                        const svgParent = $element.find(".the_svg_parent");
                        svgParent.find('svg').remove();
                        svgParent.append(newSVGElement);
                     }).error(setErrorInScope.bind($scope));
        };

        $ctrl.setDisplayMode = function(mode) {
            $ctrl.displayMode = mode;
            $ctrl.onChangeDisplayMode && $ctrl.onChangeDisplayMode({ mode });
        };

        $ctrl.stopDevKernel = function() {
            $rootScope.$emit('llmStopDevKernel', $ctrl.savedModel.id);
            DataikuAPI.savedmodels[SavedModelsService.getLlmEndpoint($ctrl.savedModel.savedModelType)].stopDevKernel($ctrl.projectKey, $ctrl.savedModel.id, $ctrl.currentVersion.versionId).then(() => {
                $ctrl.stopDevKernelEnabled = false;
            })
            .catch((error) => {
                if (error && error.status === 404) {
                    $ctrl.stopDevKernelEnabled = false;
                } else {
                    $ctrl.stopDevKernelEnabled = true;
                    setErrorInScope.bind($scope)(error);
                }
            });
        };

        $ctrl.downloadDiagnosis = () => {
            ActivityIndicator.success("Preparing diagnosis files ...");
            let payload = JSON.stringify({
                "trace": $rootScope.lastTrace ? JSON.stringify($rootScope.lastTrace) : "",
            });
            DataikuAPI.savedmodels.agents.preloadDiagnosis($ctrl.projectKey, $ctrl.savedModel.id, $ctrl.currentVersion.versionId, payload).then((response) => {
                downloadURL(DataikuAPI.savedmodels.agents.downloadDiagnosis($ctrl.projectKey, $ctrl.savedModel.id, $ctrl.currentVersion.versionId, response.data.name));
            })
            .catch((error) => {
                if (error && (error.status === 401 || error.status === 403)) {
                    setErrorInScope.bind($scope)(error);
                } else {
                    // we can download the archive without preloaded data
                    downloadURL(DataikuAPI.savedmodels.agents.downloadDiagnosis($ctrl.projectKey, $ctrl.savedModel.id, $ctrl.currentVersion.versionId, ""));
                }
            });
        };

        $ctrl.selectBlock = function(blockId) {
            $ctrl.onSelectBlock({ blockId });
        };

        $ctrl.handleDiagramUpdate = function(blocks) {
            $ctrl.onDiagramUpdate({ blocks });
        };

        $scope.$watch('$ctrl.displayMode', function(displayMode) {
            localStorageService.set("llmPreferredTestingMode", displayMode);
        });

        $scope.openTestInPromptStudioModal = function () {
            const modalScope = $scope.$new();
            modalScope.savedModel = $ctrl.savedModel;
            modalScope.save = $ctrl.save;
            CreateModalFromTemplate('/templates/savedmodels/retrieval-augmented-llm/test-in-prompt-studio.html', modalScope, 'TestInPromptStudioModalController');
        };

        $scope.launchAgentReview = function () {
            SavedModelsService.getAgentReviewUrl($ctrl.savedModel.id, $ctrl.savedModel.name, $ctrl.savedModel.activeVersion, $ctrl.currentVersion).then((url) => {
                window.open(url, '_blank');
            });
        }

        $scope.advancedLLMMeshAllowed = function () {
            return $rootScope.appConfig.licensedFeatures.advancedLLMMeshAllowed;
        }
    }
});

/* ************************************ Settings *************************** */

app.controller("SavedModelSettingsController", function($scope, DataikuAPI, $q, CreateModalFromTemplate, $stateParams, TopNav, ComputableSchemaRecipeSave,
        SavedModelsService, WT1, ActiveProjectKey, Logger, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS){

    let savedSettings;
    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(data) {
        $scope.savedModel = data;

        if (SavedModelsService.isLLM($scope.savedModel)) {
            TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_SAVED_MODEL, "settings");
        } else {
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, TopNav.TABS_SAVED_MODEL, "settings");
        }

        $scope.canHaveConditionalOutput = data.miniTask && data.miniTask.taskType === 'PREDICTION' && data.miniTask.predictionType === 'BINARY_CLASSIFICATION' && !SavedModelsService.isExternalMLflowModel(data);
        savedSettings = angular.copy(data);
        TopNav.setItem(
            TopNav.ITEM_SAVED_MODEL,
            $stateParams.smId,
            {
                name: data.name,
                taskType: (data.miniTask || {}).taskType,
                backendType: (data.miniTask || {}).backendType,
                predictionType: (data.miniTask || {}).predictionType,
                savedModelType: data.savedModelType,
                proxyModelProtocol: (data.proxyModelConfiguration || {}).protocol
            }
            );

        if (!$scope.canHaveConditionalOutput) return;
        $scope.targetRemapping = ['0', '1'];
        DataikuAPI.ml.prediction.getModelDetails(['S', data.projectKey, data.id, data.activeVersion].join('-')).success(function(data){
            $scope.targetRemapping = [];
            data.preprocessing.target_remapping.forEach(function(r){ $scope.targetRemapping[r.mappedValue] = r.sourceValue; });
        }).error(setErrorInScope.bind($scope));
    }).error(setErrorInScope.bind($scope));

    $scope.isProxyModel = function() {
        return SavedModelsService.isProxyModel($scope.savedModel);
    }

    $scope.isPartitionedModel = function() {
        return SavedModelsService.isPartitionedModel($scope.savedModel);
    }

    $scope.isMLflowModel = function() {
        return SavedModelsService.isMLflowModel($scope.savedModel);
    }

    $scope.$watch("savedModel", function() {
        if (!$scope.savedModel) {
            return;
        }

        if ($scope.savedModel.proxyModelConfiguration) {
            $scope.uiState.proxyModelConnection = $scope.savedModel.proxyModelConfiguration.connection === undefined ? null : $scope.savedModel.proxyModelConfiguration.connection;
            const fullProtocol = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === $scope.savedModel.proxyModelConfiguration.protocol);
            if (fullProtocol && fullProtocol.connectionType) {
                DataikuAPI.connections.getNames(fullProtocol.connectionType).success(function (data) {
                    const baseList = fullProtocol.canAuthenticateFromEnvironment?[{ name: null, label: "Environment" }]:[];
                    $scope.uiState.availableCompatibleConnections = baseList.concat(data.map(n => { return { name: n, label: n } }));
                }).error(setErrorInScope.bind($scope));
            }
        }
    });

    $scope.$watch("uiState.proxyModelConnection", function() {
        if (!$scope.savedModel || !$scope.savedModel.proxyModelConfiguration) {
            return;
        }
        $scope.savedModel.proxyModelConfiguration.connection = $scope.uiState.proxyModelConnection;
    });

    let oldNumberChecksOnAssertionsMetrics;
    let oldNumberChecks;
    $scope.save = function() {
        try {
            let numberChecksOnAssertionsMetrics = 0;
            let numberChecks = 0;
            if ($scope.savedModel && $scope.savedModel.metricsChecks && $scope.savedModel.metricsChecks.checks) {
                numberChecksOnAssertionsMetrics = $scope.savedModel.metricsChecks.checks.filter(m => m.metricId).filter(
                    m => m.metricId.startsWith("model_perf:ASSERTION_") ||
                        m.metricId === "model_perf:PASSING_ASSERTIONS_RATIO"
                ).length;
                numberChecks = $scope.savedModel.metricsChecks.checks.length || 0;
            }
            if (numberChecksOnAssertionsMetrics !== oldNumberChecksOnAssertionsMetrics ||
                numberChecks !== oldNumberChecks) {

                WT1.event("checks-save", {
                    numberChecksOnAssertionsMetrics: numberChecksOnAssertionsMetrics,
                    numberChecks: numberChecks
                });
            }
            oldNumberChecksOnAssertionsMetrics = numberChecksOnAssertionsMetrics;
            oldNumberChecks = numberChecks;
        }  catch (e) {
            Logger.error('Failed to report checks info', e);
        }
        DataikuAPI.savedmodels.save($scope.savedModel).success(function(data) {
            savedSettings = angular.copy($scope.savedModel);
            if ($scope.canHaveConditionalOutput && data && 'recipes' in data) {
                if (data.recipes.length) {
                    DataikuAPI.flow.recipes.getComputableSaveImpacts($scope.savedModel.projectKey, data.recipes, data.payloads).success(function(data){
                        if (!data.totalIncompatibilities) return;
                        CreateModalFromTemplate("/templates/recipes/fragments/recipe-incompatible-schema-multi.html", $scope, null,
                            function(newScope) {
                                ComputableSchemaRecipeSave.decorateChangedDatasets(data.computables, false);

                                newScope.schemaChanges = data;
                                newScope.customMessage = "The output datasets of scoring recipes using this model have incompatible schemas.";
                                newScope.noCancel = true;
                                function done(){ newScope.dismiss(); };
                                newScope.ignoreSchemaChangeSuggestion = done;
                                newScope.updateSchemaFromSuggestion = function() {
                                    $q.all(ComputableSchemaRecipeSave.getUpdatePromises(data.computables))
                                        .then(done).catch(setErrorInScope.bind($scope));
                                }
                            }
                        );
                    });
                } else if (data.hiddenRecipes) {    // TODO warn?
                }
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.dirtySettings = function() {
        return !angular.equals(savedSettings, $scope.savedModel);
    }
    checkChangesBeforeLeaving($scope, $scope.dirtySettings);
});


/* ************************************ Report *************************** */

app.controller("_SavedModelReportController", function($scope, TopNav, $stateParams, DataikuAPI, ActiveProjectKey, WebAppsService){
    if (!$scope.noSetLoc) {
        TopNav.setItem(TopNav.ITEM_SAVED_MODEL, $stateParams.smId);
    }

    const p = DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(data) {
        $scope.savedModel = data;
        if ($scope.smContext) $scope.smContext.savedModel = data;
        if (!$scope.noSetLoc) {
            TopNav.setItem(TopNav.ITEM_SAVED_MODEL, $stateParams.smId, {
                name: data.name,
                taskType: (data.miniTask || {}).taskType,
                backendType: (data.miniTask || {}).backendType,
                predictionType: (data.miniTask || {}).predictionType,
                savedModelType: data.savedModelType,
                proxyModelProtocol: (data.proxyModelConfiguration || {}).protocol
            });
        }
    });
    if ($scope.noSpinner) {
        p.noSpinner();
    }

    $scope.fillVersionSelectorStuff = function(statusData, needsMetric){
        if (!$scope.versionsContext.activeMetric && needsMetric) {
            $scope.versionsContext.activeMetric = statusData.task.modeling.metrics.evaluationMetric;
        }
        $scope.versionsContext.versions = statusData.versions.filter(function(m){
            return m.snippet.trainInfo.state == "DONE";
        });
        $scope.versionsContext.currentVersion = statusData.versions.filter(function(m){
            return m.snippet.fullModelId === $stateParams.fullModelId;
        })[0] || {}; // (partitioned models) ensure watch on versionsContext.currentVersion is fired (see ch45900)
        $scope.versionsContext.versions.sort(function(a, b) {
            var stardiff = (0+b.snippet.userMeta.starred) - (0+a.snippet.userMeta.starred)
            if (stardiff !=0) return stardiff;
            return b.snippet.sessionDate - a.snippet.sessionDate;
        });

        statusData.versions.forEach(function(version) {
            if (version.active) {
                $scope.versionsContext.activeVersion = version;
            }
        });

        if ($scope.versionsContext.currentVersion.snippet && $scope.savedModel) {
            let contentType = $scope.savedModel.contentType;
            if (!contentType) {
                if ($scope.savedModel.miniTask) {
                    contentType = `${$scope.savedModel.miniTask.taskType}/${$scope.savedModel.miniTask.backendType}`.toLowerCase();
                } else if ($scope.savedModel.savedModelType === "PYTHON_AGENT") {
                    contentType = "python";
                } else if ($scope.savedModel.savedModelType === "PLUGIN_AGENT") {
                    contentType = "plugin";
                } else if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
                    contentType = "tools-using";

                } else if ($scope.savedModel.savedModelType === "LLM_GENERIC") {
                    contentType = "fine-tuned";
                } else if ($scope.savedModel.savedModelType === "RETRIEVAL_AUGMENTED_LLM") {
                    contentType = "retrieval-augmented-llm";
                }
            }
            if (!contentType.endsWith("/")) {
                contentType += '/';
            }
            if ($scope.versionsContext.currentVersion.snippet.algorithm) {
                contentType += $scope.versionsContext.currentVersion.snippet.algorithm.toLowerCase();
            } else if ($scope.savedModel.miniTask && $scope.savedModel.miniTask.backendType === "DEEP_HUB") {
                contentType += $scope.savedModel.miniTask.predictionType.toLowerCase();
            }
            if ($scope.savedModel.miniTask) {
                $scope.modelSkins = WebAppsService.getSkins(
                    'SAVED_MODEL', $scope.versionsContext.currentVersion.versionId,
                    { predictionType: $scope.savedModel.miniTask.predictionType, backendType: $scope.savedModel.miniTask.backendType, contentType },
                    $scope.staticModelSkins
                );
            }
        }
    }
})

app.controller("PredictionSavedModelReportController", function($scope, DataikuAPI, $stateParams, TopNav, $controller, PMLFilteringService, ActiveProjectKey, GoToStateNameSuffixIfBase, MLModelsUIRouterStates, $state){
    $scope.noMlReportTourHere = true; // the tabs needed for the tour are not present
    $controller("_PredictionModelReportController",{$scope:$scope});
    $controller("_SavedModelReportController", {$scope:$scope});
    $controller("_SavedModelGovernanceStatusController", {$scope:$scope});

    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "PREDICTION-SAVED_MODEL-VERSION", "report");
    }

    // Fill the version selector
    const getStatusP = DataikuAPI.savedmodels.prediction.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.getGovernanceStatus($stateParams.fullModelId, data.task.partitionedModel);
        $scope.fillVersionSelectorStuff(data, true);
        $scope.versionsContext.versions.forEach(function(m){
            m.snippet.mainMetric = PMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
            m.snippet.mainMetricStd = PMLFilteringService.getMetricStdFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
        });
    });
    if ($scope.noSpinner) {
        getStatusP.noSpinner();
    }

    $scope.getPredictionDesignTabPrefix = () => MLModelsUIRouterStates.getPredictionDesignTabPrefix($scope);

    $scope.isClassicalPrediction = function() {
        if (!$scope.modelData) return;
        return ["BINARY_CLASSIFICATION", "REGRESSION", "MULTICLASS"].includes($scope.modelData.coreParams.prediction_type);
    };

    $scope.isTimeseriesPrediction = function() {
        if (!$scope.modelData) return;
        return $scope.modelData.coreParams.prediction_type === 'TIMESERIES_FORECAST';
    };

    $scope.isMLBackendType = function(mlBackendType) {
        if ($scope.modelData && $scope.modelData.coreParams) {
            return $scope.modelData.coreParams.backendType === mlBackendType;
        }
    };

    $scope.isDeepHubPrediction = function() {
        return $scope.isMLBackendType("DEEP_HUB");
    };

    $scope.isCausalPrediction = function() {
        if (!$scope.modelData) return;
        return ["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.modelData.coreParams.prediction_type);
    };

    const baseStateName = "projects.project.savedmodels.savedmodel.prediction.report";
    if ($state.current.name === baseStateName) {
        const deregister = $scope.$watch("modelData", (nv, ov) => {
            if (!nv) {
                return;
            }
            // redirect to summary page when arriving on "report"
            const route = MLModelsUIRouterStates.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false);
            $state.go(`.${route}`, null, {location: 'replace'});
            deregister();
        });
    }

    $scope.$on('$stateChangeSuccess', function (event, toState, toParams, fromState, fromParams) {
        if (!$scope.modelData) {
            return;
        }
        const suffix = MLModelsUIRouterStates.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false);
        GoToStateNameSuffixIfBase($state, toState, toParams, fromState, fromParams, baseStateName, suffix, event);
    });
});


app.controller("ClusteringSavedModelReportController", function($scope, $controller, $state, $stateParams, $q, DataikuAPI, CreateModalFromTemplate, TopNav, CMLFilteringService, ActiveProjectKey){
    $controller("_ClusteringModelReportController",{$scope:$scope});
    $controller("_SavedModelReportController", {$scope:$scope});
    $scope.clusteringResultsInitDone = false;

    const smId = $stateParams.smId // by the time getModelStatus is called in async logic, smId is no longer present in $stateParams, so needs to be cached
    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "CLUSTERING-SAVED_MODEL-VERSION", "report");
    }

    // Fills the version selector
    const getModelStatus = function() {
        return DataikuAPI.savedmodels.clustering.getStatus(ActiveProjectKey.get(), smId)
            .then(({data}) => {
                $scope.fillVersionSelectorStuff(data, true);
                $scope.versionsContext.versions.forEach(function(m){
                    m.snippet.mainMetric = CMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                });
        });
    }

    $scope.deferredAfterInitCModelReportDataFetch
        .then(() => {
            if (!$scope.noSetLoc) {
                TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.ITEM_SAVED_MODEL, "CLUSTERING-SAVED_MODEL-VERSION", "report");
            }
        })
        .then(getModelStatus)
        .then(() => {
            $scope.clusteringResultsInitDone = true;
        })
        .catch(setErrorInScope.bind($scope));

});



app.controller("LLMGenericSavedModelReportController", function($scope, $controller, $stateParams, DataikuAPI, ActiveProjectKey, TopNav, FinetuningUtilsService) {
    $controller("_SavedModelReportController", {$scope});
    $controller("_MLReportSummaryController", {$scope});

    // TODO @llm: adapt for dashboard (cf. PML/CML controller structure)
    DataikuAPI.ml.llm.getModelDetails($stateParams.fullModelId).success(function(data) {
        $scope.modelData = data;
        if (data.llmSMInfo.connection) {
            DataikuAPI.admin.connections.get(data.llmSMInfo.connection).success(function (data) {
                $scope.connection = data;
            });
        }
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.fillVersionSelectorStuff(data);
    });

    const smId = $stateParams.smId // by the time getModelStatus is called in async logic, smId is no longer present in $stateParams, so needs to be cached
    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "LLM-SAVED_MODEL-VERSION", "report");
    }

    $scope.supportsModelDeployment = FinetuningUtilsService.supportsModelDeployment;
});

app.controller("LLMGenericReportSummaryController", function($scope, $controller, $stateParams, DataikuAPI, Debounce, ActivityIndicator) {
    $controller("_MLReportSummaryController", {$scope});

    // Maybe can be factorized: see _ClusteringModelReportController / _PredictionModelReportController
    function saveMeta() {
        if ($scope.readOnly) return;
        DataikuAPI.ml.saveModelUserMeta($stateParams.fullModelId || $scope.fullModelId, $scope.modelData.userMeta).success(function(){
            ActivityIndicator.success("Saved")
        }).error(setErrorInScope.bind($scope));
    }

    const debouncedSaveMeta = Debounce().withDelay(400,1000).wrap(saveMeta);

    $scope.$watch("modelData.userMeta", function(nv, ov) {
        // Equality check here is needed here as something (not sure what) is updating the modelData scope object with an identical copy
        if (!nv || !ov || _.isEqual(nv, ov)) return;
        debouncedSaveMeta();
    }, true);
});

app.controller("LLMGenericReportTrainingInformationController", function($scope, $controller) {
    $controller("_MLReportSummaryController", {$scope});

    $scope.buildChartOptions = function(llmStepwiseTrainingMetrics) {
        const trainingLossData = [];
        const validationLossData = [];
        const fullValidationLossData = [];
        Object.entries(llmStepwiseTrainingMetrics.metrics).forEach(([step, metric]) => {
            if (metric.trainingMetric) {
                trainingLossData.push([step, metric.trainingMetric.loss])
            }
            if (metric.validationMetric) {
                validationLossData.push([step, metric.validationMetric.loss])
            }
            if (metric.fullValidationMetric) {
                fullValidationLossData.push([step, metric.fullValidationMetric.loss])
            }
        });
        const series = [];
        if (trainingLossData.length > 0) {
            series.push({
                data: trainingLossData, type: 'line', name: 'Training loss'
            })
        }
        if (validationLossData.length > 0) {
            series.push({
                data: validationLossData, type: 'line', name: 'Validation loss'
            })
        }
        if (fullValidationLossData.length > 0) {
            series.push({
                data: fullValidationLossData, type: 'line', name: 'Full validation loss'
            })
        }
        $scope.chartOptions = {
            xAxis: { name: "Step"},
            yAxis: { name: "Loss", scale: true },
            series,
            tooltip: {
                trigger: 'axis'
            },
            legend: {
                data: ['Training loss', 'Validation loss', 'Full validation loss']
            },
        }
    }
});

// use createOrAttachDeploymentModalDirective when calling CreateModalFromComponent
app.component("createOrAttachDeploymentModal", {
    bindings: {
        attachOnly: '<',
        llmType: '<',
        returnData: '=',
        modalControl: '<',
    },
    templateUrl: '/templates/ml/llm-generic/create-or-attach-deployment-modal.html',
    controller: function($scope, DataikuAPI, $stateParams, ActiveProjectKey) {
        const $ctrl = this;

        $ctrl.createOrAttach = function() {
            const apiCreateFn = $ctrl.attachOnly ? DataikuAPI.savedmodels.llmGeneric.deployments.attach : DataikuAPI.savedmodels.llmGeneric.deployments.create;
            apiCreateFn(ActiveProjectKey.get(), $stateParams.fullModelId, $ctrl.newDeploymentId)
                .success(function(data) {
                    $ctrl.returnData.deploymentWithStatus = data;
                    $ctrl.modalControl.resolve();
                })
                .error(setErrorInScope.bind($scope));
        }
    }
});

// use makeActiveLLMGenericModalDirective when calling CreateModalFromComponent
app.component("makeActiveLLMGenericModal", {
    bindings: {
        deployments: '<',
        savedModel: '<',
        newVersionId: '<',
        llmType: '<',
        modalControl: '<',
    },
    templateUrl: '/templates/ml/llm-generic/make-active-modal.html',
    controller: function($scope, FinetuningUtilsService, DataikuAPI) {
        const $ctrl = this;

        $ctrl.makeActive = function() {
            DataikuAPI.savedmodels.llmGeneric.setActive($ctrl.savedModel.projectKey, $ctrl.savedModel.id, $ctrl.newVersionId, $ctrl.deployNewActiveModel, $ctrl.deleteInactiveDeployments)
                .then($ctrl.modalControl.resolve)
                .catch(setErrorInScope.bind($scope).bind($scope))
        }

        $ctrl.showDeployNewModelCheckbox = function() {
            if (!['SAVED_MODEL_FINETUNED_AZURE_OPENAI', 'SAVED_MODEL_FINETUNED_BEDROCK'].includes($ctrl.llmType)) {
                return false; // Deployments are only relevant for Azure Open AI / Bedrock models
            }
            // Ignore the checkbox if a model is already deployed
            return !$ctrl.deployments.find(d => d.versionId === $ctrl.newVersionId)
        }

        $ctrl.initDeployNewModel = function() {
            $ctrl.deployNewActiveModel = $ctrl.showDeployNewModelCheckbox();
        }

        $ctrl.showDeleteInactiveModelsCheckbox = function() {
            if ($ctrl.deployments.length === 0) return false; // No deployments to delete
            if ($ctrl.deployments.length === 1) {
                // Ignore the checkbox as the deployed model is attached to the new active model version
                return !$ctrl.deployments.find(d => d.versionId === $ctrl.newVersionId);
            }
            return true;
        }

        $ctrl.getDeletedDeploymentsCountMessage = function() {
            return FinetuningUtilsService.getDeletedDeploymentsCountMessage($ctrl.deployments, $ctrl.savedModel, $ctrl.newVersionId, true, $ctrl.deleteInactiveDeployments);
        }
    }
});

app.controller("LLMSavedModelDeploymentController", function($scope, $controller, $stateParams, ActiveProjectKey, DataikuAPI, CreateModalFromTemplate, CreateModalFromComponent, createOrAttachDeploymentModalDirective, Dialogs) {
    $controller("_MLReportSummaryController", {$scope});

    $scope.loadDeployment = function() {
        DataikuAPI.savedmodels.llmGeneric.deployments.get(ActiveProjectKey.get(), $stateParams.fullModelId)
            .success(function(data){
                $scope.deploymentWithStatus = data;
            })
            .error((data, status, headers) => {
                $scope.disabledMessage = $scope.modelData.llmSMInfo.llmType === 'SAVED_MODEL_FINETUNED_AZURE_OPENAI'
                    ? 'Make sure you have a valid Azure ML connection in your Azure Open AI connection'
                    : 'Make sure your credentials in your Bedrock connection are properly set up.';
                setErrorInScope.call(this, data, status, headers)
            });
    }

    $scope.getConnectionName = function () {
        return ($scope.modelData.llmSMInfo.llmType === 'SAVED_MODEL_FINETUNED_AZURE_OPENAI' ? 'Azure' : 'Bedrock');
    }

    $scope.detachDeployment = function() {
        if (!$scope.deploymentWithStatus) return;
        Dialogs.confirmAlert(
            $scope,
            'Confirm model detachment',
            'Are you sure you want to detach your deployment ' + $scope.deploymentWithStatus.deploymentId + '?',
            "Your model will remain active on " + $scope.getConnectionName() + " and will continue to incur costs.",
            "WARNING"
        ).then(function() {
            DataikuAPI.savedmodels.llmGeneric.deployments.detach(ActiveProjectKey.get(), $stateParams.fullModelId)
                .success(function() {
                    $scope.deploymentWithStatus = undefined;
                    $scope.modelData.deployment = undefined;
                })
                .error(setErrorInScope.bind($scope));
        });

    }

    $scope.newDeploymentModal = function(attachOnly) {
        const dataHolder = {}
        CreateModalFromComponent(createOrAttachDeploymentModalDirective, { attachOnly: attachOnly, returnData: dataHolder, llmType: $scope.modelData.llmSMInfo.llmType }).then(() => {
            $scope.deploymentWithStatus = dataHolder.deploymentWithStatus;
            $scope.modelData.deployment = dataHolder.deploymentWithStatus;
        });
    }

    $scope.deleteDeployment = function() {
        if (!$scope.deploymentWithStatus) return;
        // $scope.loading = true;
        Dialogs.confirmAlert(
            $scope,
            'Confirm model deletion',
            'Are you sure you want to delete your deployment ' + $scope.deploymentWithStatus.deploymentId + '?',
            "This action is irreversible.",
            "WARNING"
        ).then(function() {
            DataikuAPI.savedmodels.llmGeneric.deployments.delete(ActiveProjectKey.get(), $stateParams.fullModelId)
                .success(function() {
                    $scope.deploymentWithStatus = undefined;
                    $scope.modelData.deployment = undefined;
                })
                .error(setErrorInScope.bind($scope));
        });
    }

});

/* ***************************** Scoring recipe creation ************************** */
app.service("RecipeModalCheckSavedModelBackendService", function() {
    return {
        getCustomInputRequirements: function(smId, inputDataset, needsInputDataFolder, managedFolderId) {
            const ret = [];

            // Recipe must always have an input dataset
            if (!inputDataset) {
                ret.push("an input dataset");
            }

            // Modal does not have a model set
            if (!smId) {
                ret.push("a model");
            }

            // Modal has a model selected that lacks a managedFolder input.
            if (needsInputDataFolder && !managedFolderId) {
                    ret.push("a managed folder");
            }

            if (ret.length === 0) {
                return null;
            }
            return ret.join(", ");
        }
    };
});

app.controller("NewPredictionScoringRecipeModalController", function($scope, $stateParams, $controller, DataikuAPI, Fn,
                                                                     SavedModelsService, ActiveProjectKey,
                                                                     RecipeModalCheckSavedModelBackendService) {
    $scope.recipe = { // needed for updateRecipeDesc()
        projectKey : ActiveProjectKey.get(),
        type: "prediction_scoring",
        inputs : {},
        outputs : {},
        params: {}
    };
    $scope.recipeType = "prediction_scoring";
    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

    $scope.scoringRecipe = {};

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_scored");
        }
    };

    $scope.managedFolder = {id: null};
    let selectedSavedModel = null;

    $scope.getCustomInputRequirements = function() {
        return RecipeModalCheckSavedModelBackendService.getCustomInputRequirements($scope.smId, $scope.io.inputDataset,
                                                                                   $scope.isManagedFolderRequired(), $scope.managedFolder.id);
    }

    $scope.isManagedFolderRequired = function(){
        const dataRole = $scope.recipeDesc.inputRoles.find(role => role.name == 'data');
        return dataRole && dataRole.required && $scope.isInputRoleAvailableForPayload(dataRole);
    }

    $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
        switch(role.name){
            case "data":
                // Data role is only available for deephub & classical task with image preprocessing on scoring recipes
                return selectedSavedModel && selectedSavedModel.needsInputDataFolder;
            default:
                throw new Error(`Rules for availability of input role "${role.name}" not implemented`);
        }
    };

    $scope.doCreateRecipe = function() {
        var createOutput = $scope.io.newOutputTypeRadio == 'create';

        var finalRecipe = angular.copy($scope.scoringRecipe);
        finalRecipe.inputDatasetSmartName = $scope.io.inputDataset;
        finalRecipe.managedFolderSmartId = $scope.managedFolder.id;
        finalRecipe.savedModelSmartName = $scope.smId;
        finalRecipe.createOutput = createOutput;
        finalRecipe.outputDatasetSmartName = createOutput ? $scope.newOutputDataset.name : $scope.io.existingOutputDataset;
        finalRecipe.outputDatasetCreationSettings = $scope.getDatasetCreationSettings();
        finalRecipe.zone = $scope.zone;

        return DataikuAPI.savedmodels.prediction.deployScoring(ActiveProjectKey.get(), finalRecipe);
    };

    $scope.subFormIsValid = function() {
        return !$scope.getCustomInputRequirements();
    };

    $scope.$watch("smId", function(nv) {
        if (!nv) return;
        selectedSavedModel = $scope.savedModels.find(sm => sm.id === $scope.smId);

        // Remove image folder from inputs of recipe if currently selected SM do not need the managedFolder input
        if (!$scope.isInputRoleAvailableForPayload({name: 'data'})) {
            $scope.managedFolder.id = null;
        }
    });
});


app.controller("NewClusteringScoringRecipeModalController", function($scope, Fn, $stateParams, $controller, DataikuAPI, SavedModelsService, ActiveProjectKey) {
    $scope.recipeType = "prediction_scoring";
    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

    $scope.scoringRecipe = {};

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_scored");
        }
    };

    $scope.doCreateRecipe = function() {
        var createOutput = $scope.io.newOutputTypeRadio == 'create';
        return DataikuAPI.savedmodels.clustering.deployScoring(
            ActiveProjectKey.get(),
            $scope.smId,
            $scope.io.inputDataset,
            createOutput,
            createOutput ? $scope.newOutputDataset.name : $scope.io.existingOutputDataset,
            $scope.getDatasetCreationSettings());
    };

    $scope.subFormIsValid = function() { return !!$scope.smId; };
});

/* ***************************** Evaluation recipe creation ************************** */

app.controller('NewEvaluationRecipeModalController', function($scope, $controller, $stateParams, $state, DataikuAPI,
                                                              DatasetUtils, RecipeComputablesService,
                                                              PartitionDeps, ActiveProjectKey, $rootScope,
                                                              RecipeModalCheckSavedModelBackendService) {
    $scope.recipeType = "evaluation";
    $scope.forceMainLabel = true;
    $controller("_RecipeCreationControllerBase", {$scope:$scope});

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_evaluated");
        }
    };

    addDatasetUniquenessCheck($scope, DataikuAPI, ActiveProjectKey.get());
    fetchManagedDatasetConnections($scope, DataikuAPI);

    let selectedSavedModel = null;

    $scope.isManagedFolderRequired = function(){
        const dataRole = $scope.recipeDesc.inputRoles.find(role => role.name == 'data');
        return dataRole && dataRole.required && $scope.isInputRoleAvailableForPayload(dataRole);
    }

    $scope.getCustomInputRequirements = function() {
        return RecipeModalCheckSavedModelBackendService.getCustomInputRequirements($scope.recipeParams.smId, $scope.recipeParams.inputDs,
            $scope.isManagedFolderRequired(), $scope.recipeParams.managedFolderSmartId);
    }

    $scope.recipe = {
        projectKey : ActiveProjectKey.get(),
        type: "evaluation",
        inputs : {},
        outputs : {},
        params: {}
    };
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.recipeParams.inputDs = preselectedInputDataset;
    });

    $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
        if (role.name == 'data'){
            return selectedSavedModel && selectedSavedModel.needsInputDataFolder;
        }
        return true;
    };

    $scope.isOutputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!

        if (!selectedSavedModel || !selectedSavedModel.miniTask) return false;
        const minitask = selectedSavedModel.miniTask;

        switch(role.name){
            case 'evaluationStore':
                // Only not partitionned classical models (binary classification, multiclass, regression) can have an evaluation recipe with a MES as output
                // TODO @causal @deephub handle MES as evaluation output
                return ["BINARY_CLASSIFICATION", "MULTICLASS", "REGRESSION", "TIMESERIES_FORECAST"].includes(minitask.predictionType)
                    && (!minitask.partitionedModel || !minitask.partitionedModel.enabled);
            default:
                throw new Error(`Rules for availability of output role "${role.name}" not implemented`);
        }
    };

    $scope.$watch("recipeParams.inputDs", function(nv, ov) {
        if (nv) {
            $scope.recipe.name = "evaluate_" + nv;
        }
        if ($scope.recipeParams.inputDs) {
            $scope.recipe.inputs.main = {items:[{ref:$scope.recipeParams.inputDs}]}; // for the managed dataset creation options
        } else {
            $scope.recipe.inputs.main = {items:[]}; // for the managed dataset creation options
        }
    }, true);

    $scope.$watch("recipeParams.smId", function(nv) {
        if (nv) {
            selectedSavedModel = $scope.savedModels.find(sm => sm.id === nv);

            $scope.recipe.inputs.model = {items:[{ref:nv}]}; // for the managed dataset creation options

            // Remove MES from outputs of recipe if currently selected SM does not support MES as eval output
            if (!$scope.isOutputRoleAvailableForPayload({name: "evaluationStore"})) {
                delete $scope.recipe.outputs["evaluationStore"];
            }

            // Remove image folder from inputs of recipe if currently selected SM do not need the managedFolder input
            if (!$scope.isInputRoleAvailableForPayload({name: 'data'})) {
                $scope.recipeParams.managedFolderSmartId = null;
            }
        } else {
            $scope.recipe.inputs.model = {items:[]}; // for the managed dataset creation options
        }
    }, true);

    DatasetUtils.listDatasetsUsabilityInAndOut(ActiveProjectKey.get(), "evaluation").then(function(data){
        $scope.availableInputDatasets = data[0];
    });

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
        $scope.setComputablesMap(map);
    });

    $scope.hasMain = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.main && outputs.main.items && outputs.main.items.length > 0 && outputs.main.items[0].ref
    }
    $scope.hasMetrics = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.metrics && outputs.metrics.items && outputs.metrics.items.length > 0 && outputs.metrics.items[0].ref
    }
    $scope.hasEvaluationStore = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.evaluationStore && outputs.evaluationStore.items && outputs.evaluationStore.items.length > 0 && outputs.evaluationStore.items[0].ref
    }

    $scope.canCreate = function(){
        return $scope.recipe.name
            && $scope.recipe.name.length > 0
            && $scope.recipe.outputs
            && !$scope.shouldDisplayOutputExplanation()
            && !($scope.newRecipeForm.$invalid)
            && !$scope.getCustomInputRequirements();
    }

    $scope.shouldDisplayOutputExplanation = function () {
        return !$scope.hasMain() && !$scope.hasMetrics() && !$scope.hasEvaluationStore();
    };

    $scope.generateOutputExplanation = function () {
        const requiredOutputRoles = [];
        $scope.recipeDesc.outputRoles.forEach((role, outputRoleidx) => {
            if (role.availabilityDependsOnPayload && !$scope.isOutputRoleAvailableForPayload(role)) return;
            requiredOutputRoles.push(role.name === "main" ? "main output" : '"' + (role.label || role.name) + '"');
        });
        const message = "This recipe requires at least one output in: "
            + requiredOutputRoles.slice(0, -1).join(', ')
            + (requiredOutputRoles.length === 2 ? ' or ' : ', or ')
            + requiredOutputRoles.slice(-1) + ".";
        return message;
    };

    $scope.createRecipe = function() {
        $scope.creatingRecipe = true;
        var finalRecipe = {};

        finalRecipe.inputDatasetSmartName = $scope.recipeParams.inputDs;
        finalRecipe.savedModelSmartName = $scope.recipeParams.smId;
        finalRecipe.managedFolderSmartId = $scope.recipeParams.managedFolderSmartId;
        finalRecipe.scoredDatasetSmartName = $scope.recipe.outputs.main && $scope.recipe.outputs.main.items && $scope.recipe.outputs.main.items.length>0 ? $scope.recipe.outputs.main.items[0].ref : null;
        finalRecipe.metricsDatasetSmartName = $scope.recipe.outputs.metrics && $scope.recipe.outputs.metrics.items && $scope.recipe.outputs.metrics.items.length>0 ? $scope.recipe.outputs.metrics.items[0].ref : null;
        finalRecipe.evaluationStoreSmartName = $scope.recipe.outputs.evaluationStore && $scope.recipe.outputs.evaluationStore.items && $scope.recipe.outputs.evaluationStore.items.length>0 ? $scope.recipe.outputs.evaluationStore.items[0].ref : null;
        finalRecipe.zone = $scope.zone;

        DataikuAPI.savedmodels.prediction.deployEvaluation(ActiveProjectKey.get(), finalRecipe)
            .success(function(data) {
                $scope.creatingRecipe = false;
                $scope.dismiss();
                $scope.$state.go('projects.project.recipes.recipe', {
                    recipeName: data.id
                });
            }).error(function(a, b, c) {
                $scope.creatingRecipe = false;
                setErrorInScope.bind($scope)(a,b,c);
            });

    };
});

app.controller('NewStandaloneEvaluationRecipeModalController', function($scope, $controller, $stateParams, $state, DataikuAPI,
DatasetUtils, RecipeComputablesService, PartitionDeps, ActiveProjectKey){
    $scope.recipeType = "standalone_evaluation";
    $controller("_RecipeCreationControllerBase", {$scope:$scope});

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_scored");
        }
    };

    addDatasetUniquenessCheck($scope, DataikuAPI, ActiveProjectKey.get());

    $scope.recipeParams = {
        inputDs: "",
        referenceDs : ""
    };

    $scope.recipe = {
        projectKey : ActiveProjectKey.get(),
        type: "standalone_evaluation",
        inputs : {},
        outputs : {},
        params: {}
    };
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.recipeParams.inputDs = preselectedInputDataset;
    });
    $scope.$on("preselectReferenceDataset", function(scope, preselectReferenceDataset) {
        $scope.recipeParams.referenceDs = preselectReferenceDataset;
    });

    $scope.$watch("recipeParams.inputDs", function(nv, ov) {
        if (nv) {
            $scope.recipe.name = "standalone_evaluate_" + nv;
        }
        if ($scope.recipeParams.inputDs) {
            $scope.recipe.inputs.main = {items:[{ref:$scope.recipeParams.inputDs}]}; // for the managed dataset creation options
        } else {
            $scope.recipe.inputs.main = {items:[]}; // for the managed dataset creation options
        }
    }, true);

    DatasetUtils.listDatasetsUsabilityInAndOut(ActiveProjectKey.get(), "standalone_evaluation").then(function(data){
        $scope.availableInputDatasets = data[0];
    });

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
        $scope.setComputablesMap(map);
    });

    $scope.hasMain = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.main && outputs.main.items && outputs.main.items.length > 0 && outputs.main.items[0].ref
    }

    $scope.canCreate = function(){
        return $scope.recipe.name
            && $scope.recipe.name.length > 0
            && $scope.recipe.outputs
            && !$scope.shouldDisplayOutputExplanation()
            && !($scope.newRecipeForm.$invalid)
    }

    $scope.shouldDisplayOutputExplanation = function () { return !$scope.hasMain(); };

    $scope.createRecipe = function() {
        $scope.creatingRecipe = true;
        var finalRecipe = {};

        finalRecipe.inputDatasetSmartName = $scope.recipeParams.inputDs;
        finalRecipe.referenceDatasetSmartName = $scope.recipeParams.referenceDs;
        finalRecipe.evaluationStoreSmartName = $scope.recipe.outputs.main.items[0].ref;
        finalRecipe.zone = $scope.zone;

        DataikuAPI.savedmodels.prediction.deployStandaloneEvaluation(ActiveProjectKey.get(), finalRecipe)
            .success(function(data) {
                $scope.creatingRecipe = false;
                $scope.dismiss();
                $scope.$state.go('projects.project.recipes.recipe', {
                    recipeName: data.id
                });
            }).error(function(a, b, c) {
                $scope.creatingRecipe = false;
                setErrorInScope.bind($scope)(a,b,c);
            });;

    };
});

app.controller("SavedModelVersionSkinsController", function($scope, $state, VirtualWebApp, $rootScope, $timeout, Logger) {
    function setSkinFromTile() {
        // if insight or dashboard tile, we select the skin defined in the params
        if ($scope.tile.tileParams && $scope.tile.tileParams.advancedOptions && $scope.tile.tileParams.advancedOptions.customViews) {
            const viewId = $scope.tile.tileParams.advancedOptions.customViews.viewId;
            $scope.uiState.skin = $scope.modelSkins.find(s => s.id === viewId);
        }
    }

    let modelId = '';
    let version = '';
    if ($scope.savedModel && $scope.savedModel.id) {
        // when called from sm
        modelId = $scope.savedModel.id;
    } else if ($scope.insight &&  $scope.insight.$savedModel && $scope.insight.$savedModel.id) {
        // when called from insight
        modelId = $scope.insight.$savedModel.id;
    } else {
        Logger.error("Skin missing model's modelId");
    }
    if ($scope.versionsContext && $scope.versionsContext.currentVersion && $scope.versionsContext.currentVersion.versionId) {
        // when called from sm
        version = $scope.versionsContext.currentVersion.versionId
    } else if ($scope.insight && $scope.insight.$savedModel && $scope.insight.$savedModel.activeVersion) {
        // when called from insight
        version = $scope.insight.$savedModel.activeVersion;
    } else {
        Logger.error("Skin missing model's version");
    }
    if ($scope.tile && $scope.tile.insightId) {
        $scope.skinHolderClass = "skin-holder-insight-" + $scope.tile.insightId;
        const deregister = $scope.$watch('modelSkins', function (nv, ov) {
            if (!nv) {return}
            // make sure modelSkins is defined before calling setSkinFromTile
            $scope.$watch('tile.tileParams.advancedOptions.customViews.viewId', function () {
                setSkinFromTile(); // changes uiState.skin accordingly
            });
            deregister();
        });
    } else {
        $scope.skinHolderClass = "skin-holder"
    }

    $scope.$watch('uiState.skin', function() {
        if (!$scope.uiState.skin) {return;}
        if ($scope.tile && $scope.tile.tileParams && $scope.tile.tileParams.displayMode === 'skins'
            && $scope.tile.tileParams.advancedOptions && $scope.tile.tileParams.advancedOptions.customViews) {
            // we are in a dashboard tile and the tile has a custom config
            const tileView = $scope.tile.tileParams.advancedOptions.customViews;
            $scope.webAppCustomConfig = {
                ...tileView.viewParams
            }
        }

        // ng-class="skinHolderClass"  needs to be evaluated before changing skin
        $timeout(() =>
            VirtualWebApp.changeSkin($scope, 'SAVED_MODEL', $scope.uiState.skin, $scope.uiState, $scope.skinHolderClass, modelId,
                version, false)
        );
    }, true);
});

app.service("CreateSavedModelVersionService", function(PMLSettings, ClipboardReadWriteService, DatasetUtils, AnyLoc, DataikuAPI) {
    return {
        onInit: function($scope, predictionType) {
            $scope.predictionTypes = PMLSettings.task.predictionTypes.filter(type => type.classical);

            $scope.binaryIncorrectNumClasses = false;
            $scope.multiclassIncorrectNumClasses = false;

            $scope.hasInvalidClasses = function () {
                return $scope.binaryIncorrectNumClasses || $scope.multiclassIncorrectNumClasses;
            }

            $scope.deployingSmv = false;

            $scope.newSavedModelVersion = {
                versionId: null,
                predictionType: predictionType,
                datasetSmartName: null,
                targetColumn: null,
                classes: [],
                smvFuture: null,
                samplingParam: {"samplingMethod": "HEAD_SEQUENTIAL", "maxRecords": 10000, "ascending": true},
                binaryClassificationThreshold: 0.5,
                useOptimalThreshold: true,
                skipExpensiveReports: true,
                activate: true
            };

            $scope.uiState = {
                evaluateModel: $scope.newSavedModelVersion.predictionType !== "OTHER",
                activeModelVersion: null,
                progress: null,
            };
        },
        validateClasses: function($scope) {
            $scope.binaryIncorrectNumClasses = false;
            $scope.multiclassIncorrectNumClasses = false;
            if ($scope.isClassification()) {
                const classes = $scope.newSavedModelVersion.classes.filter(aClass => aClass && aClass.length > 0);
                if ($scope.newSavedModelVersion.predictionType === 'BINARY_CLASSIFICATION') {
                    if (classes.length != 2) {
                        $scope.binaryIncorrectNumClasses = true;
                    }
                } else if ($scope.newSavedModelVersion.predictionType === 'MULTICLASS') {
                    if (classes.length < 2) {
                        $scope.multiclassIncorrectNumClasses = true;
                    }
                }
            }
        },
        validateTarget: function($scope) {
            if (!$scope.uiState.evaluateModel) {
                $scope.invalidTarget = false;
            } else {
                $scope.invalidTarget = !$scope.targetColumnNames.includes($scope.newSavedModelVersion.targetColumnName);
                if ($scope.invalidTarget) {
                    $scope.currentInvalidTargetColumnName = $scope.newSavedModelVersion.targetColumnName;
                }
            }
        },
        resetErrorDisplay: function($scope) {
            $scope.invalidTarget = false;
            $scope.binaryIncorrectNumClasses = false;
            $scope.multiclassIncorrectNumClasses = false;
        },
        copyClasses: function($scope) {
            ClipboardReadWriteService.writeItemsToClipboard($scope.newSavedModelVersion.classes);
            $scope.$applyAsync()
        },
        pasteClasses: async function($scope) {
            try {
                let classes = await ClipboardReadWriteService.readItemsFromClipboard($scope, "The following classes will be used")
                $scope.newSavedModelVersion.classes = classes
                $scope.$applyAsync()
            } catch (error) {
                // do nothing
            }
        },
        isClassification: function($scope) {
            if (!$scope.newSavedModelVersion || !$scope.newSavedModelVersion.predictionType) {
                return false;
            }
            return $scope.newSavedModelVersion.predictionType === 'BINARY_CLASSIFICATION' || $scope.newSavedModelVersion.predictionType === 'MULTICLASS';
        },
        afterInit: function($scope, $stateParams, versions) {
            if (versions && versions.length) {
                const activeVersions = versions.filter(v => v.active);
                if (activeVersions && activeVersions.length) {
                    const activeVersion = activeVersions[0];
                    const activeVersionSnippet = activeVersion.snippet;
                    $scope.uiState.activeModelVersion = activeVersion.versionId;
                    if (activeVersionSnippet.proxyModelConfiguration) {
                        switch (activeVersionSnippet.proxyModelConfiguration.protocol) {
                            case "sagemaker":
                                $scope.newSavedModelVersion.sageMakerEndpointName = activeVersionSnippet.proxyModelConfiguration.endpoint_name;
                                break;
                            case "vertex-ai":
                                $scope.newSavedModelVersion.vertexEndpointId = activeVersionSnippet.proxyModelConfiguration.endpoint_id;
                                $scope.newSavedModelVersion.vertexProjectId = activeVersionSnippet.proxyModelConfiguration.project_id;
                                break;
                            case "azure-ml":
                                $scope.newSavedModelVersion.azureEndpointName = activeVersionSnippet.proxyModelConfiguration.endpoint_name;
                                $scope.newSavedModelVersion.azureResourceGroup = activeVersionSnippet.proxyModelConfiguration.resource_group;
                                $scope.newSavedModelVersion.azureSubscriptionId = activeVersionSnippet.proxyModelConfiguration.subscription_id;
                                $scope.newSavedModelVersion.azureWorkspace = activeVersionSnippet.proxyModelConfiguration.workspace;
                                break;
                            case "databricks":
                                $scope.newSavedModelVersion.endpointName = activeVersionSnippet.proxyModelConfiguration.endpointName;
                                break;
                        }
                    }
                    if (activeVersionSnippet.mlflowClassLabels && activeVersionSnippet.mlflowClassLabels.length) {
                        $scope.newSavedModelVersion.classes = activeVersionSnippet.mlflowClassLabels.map(l => l.label);
                    }
                    $scope.newSavedModelVersion.datasetSmartName = activeVersionSnippet.mlflowEvaluationDatasetSmartName;
                    $scope.uiState.signatureAndFormatsGuessingDatasetSmartName = activeVersionSnippet.mlflowSignatureAndFormatsGuessingDatasetSmartName;
                    $scope.newSavedModelVersion.samplingParam = activeVersionSnippet.mlflowEvaluationSamplingParam;
                    $scope.newSavedModelVersion.guessingDatasetSamplingParam = activeVersionSnippet.mlflowGuessingDatasetSamplingParam;
                    $scope.uiState.evaluateModel = !!$scope.newSavedModelVersion.datasetSmartName;
                    if ($scope.uiState.evaluateModel) {
                        $scope.newSavedModelVersion.targetColumnName = activeVersionSnippet.mlflowEvaluationTargetColumnName;
                    } else {
                        $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName = activeVersionSnippet.mlflowEvaluationTargetColumnName;
                    }
                }
            }

            DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(function(data){
                $scope.availableDatasets = data;
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("newSavedModelVersion.datasetSmartName", function(newDatasetSmartName, oldDatasetSmartName) {
                if (newDatasetSmartName) {
                    $scope.uiState.signatureAndFormatsGuessingDatasetSmartName = newDatasetSmartName;
                    const datasetLoc = AnyLoc.getLocFromSmart($stateParams.projectKey, newDatasetSmartName);
                    DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.localId, $stateParams.projectKey).success(function(dataset){
                        const columns = dataset && dataset.schema && dataset.schema.columns ? dataset.schema.columns : [];
                        $scope.targetColumnNames = columns.map(function(col) { return col.name });
                        if (!$scope.targetColumnNames.includes($scope.newSavedModelVersion.targetColumnName)) {
                            $scope.newSavedModelVersion.targetColumnName = null;
                        }
                    }).error(setErrorInScope.bind($scope));
                }
            });

            $scope.$watch("uiState.evaluateModel", function(newEvaluateModel, oldEvaluateModel) {
                if (newEvaluateModel === oldEvaluateModel || !$scope.uiState.guessFormat) {
                    return;
                }

                if (newEvaluateModel) {
                    $scope.newSavedModelVersion.datasetSmartName = $scope.uiState.signatureAndFormatsGuessingDatasetSmartName ? $scope.uiState.signatureAndFormatsGuessingDatasetSmartName : $scope.newSavedModelVersion.datasetSmartName;
                    $scope.targetColumnNames = $scope.signatureAndFormatsGuessingDatasetColumnNames ? $scope.signatureAndFormatsGuessingDatasetColumnNames : $scope.targetColumnNames;
                    $scope.newSavedModelVersion.targetColumnName = $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName ? $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName : $scope.newSavedModelVersion.targetColumnName;
                } else {
                    $scope.uiState.signatureAndFormatsGuessingDatasetSmartName = $scope.newSavedModelVersion.datasetSmartName ? $scope.newSavedModelVersion.datasetSmartName : $scope.uiState.signatureAndFormatsGuessingDatasetSmartName;
                    $scope.signatureAndFormatsGuessingDatasetColumnNames = $scope.targetColumnNames ? $scope.targetColumnNames : $scope.signatureAndFormatsGuessingDatasetColumnNames;
                    $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName = $scope.newSavedModelVersion.targetColumnName ? $scope.newSavedModelVersion.targetColumnName : $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName;
                }
            });
        }
    };
});

// use createProxySavedModelVersionModalDirective when calling CreateModalFromComponent
app.component("createProxySavedModelVersionModal", {
    bindings: {
        savedModel: '<',
        smStatus: '<',
        modalControl: '<'
    },
    templateUrl: '/templates/savedmodels/new-proxy-saved-model-version-modal.html',

    controller: function($scope, $state, Assert, DataikuAPI, WT1, $stateParams, SavedModelsService, AnyLoc, PMLSettings, DatasetUtils, MonoFuture, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS, ClipboardReadWriteService, FutureWatcher, CreateSavedModelVersionService) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            Assert.trueish(SavedModelsService.isProxyModel($ctrl.savedModel), 'Cannot create a saved model version with the GUI for a non-external model.');
            const details = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === $ctrl.savedModel.proxyModelConfiguration.protocol);
            if (!details) {
                return;
            }
            $scope.externalModelDetails = details;

            CreateSavedModelVersionService.onInit($scope, $ctrl.savedModel.miniTask.predictionType);

            $scope.isSageMaker = $ctrl.savedModel.proxyModelConfiguration.protocol == 'sagemaker';
            $scope.isVertex = $ctrl.savedModel.proxyModelConfiguration.protocol == 'vertex-ai';
            $scope.hasRegion = $scope.isSageMaker || $scope.isVertex;
            $scope.deployingSmv = false;

            $scope.newSavedModelVersion = {
                ...$scope.newSavedModelVersion,
                sageMakerEndpointName: null,
                vertexProjectId: $ctrl.savedModel.proxyModelConfiguration.project_id,
                vertexEndpointId: null,
                azureSubscriptionId: $ctrl.savedModel.proxyModelConfiguration.subscription_id,
                azureResourceGroup: $ctrl.savedModel.proxyModelConfiguration.resource_group,
                azureWorkspace: $ctrl.savedModel.proxyModelConfiguration.workspace,
                azureEndpointName: null,
            };

            $scope.uiState = {
                ...$scope.uiState,
                loadingSageMakerEndpointList: false,
                vertexProjectDisplayName: null,
                vertexInformationRetrievalError: null,
                guessFormat: true,
                signatureAndFormatsGuessingDatasetSmartName: null,
                signatureAndFormatsGuessingDatasetTargetColumnName: null,
                inputFormat: null,
                outputFormat: null
            };

            CreateSavedModelVersionService.afterInit($scope, $stateParams, $ctrl.smStatus.versions);

            if ($ctrl.savedModel.proxyModelConfiguration.protocol === 'vertex-ai') {
                retrieveVertexProjectInformation($scope.newSavedModelVersion.vertexProjectId, $ctrl.savedModel.proxyModelConfiguration.connection);
            }
        }

        $scope.$watch("uiState.signatureAndFormatsGuessingDatasetSmartName", function(newSignatureAndFormatsGuessingDatasetSmartName, oldSignatureAndFormatsGuessingDatasetSmartName) {
            if (newSignatureAndFormatsGuessingDatasetSmartName && $scope.uiState.guessFormat) {
                const signatureAndFormatsGuessingDatasetLoc = AnyLoc.getLocFromSmart($stateParams.projectKey, newSignatureAndFormatsGuessingDatasetSmartName);
                DataikuAPI.datasets.get(signatureAndFormatsGuessingDatasetLoc.projectKey, signatureAndFormatsGuessingDatasetLoc.localId, $stateParams.projectKey).success(function(dataset){
                    const columns = dataset && dataset.schema && dataset.schema.columns ? dataset.schema.columns : [];
                    $scope.signatureAndFormatsGuessingDatasetColumnNames = columns.map(function(col) { return col.name });
                    if (!$scope.signatureAndFormatsGuessingDatasetColumnNames.includes($scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName)) {
                        $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName = null;
                    }
                }).error(setErrorInScope.bind($scope));
            }
        });

        async function retrieveVertexProjectInformation(vertexProjectId, connectionName) {
            const project = await getVertexAIProject(vertexProjectId, connectionName);
            if (!project) {
                return;
            }
            $scope.uiState.vertexProjectDisplayName = project.displayName;
        }

        async function getVertexAIProject(vertexProjectId, connectionName) {
            try {
                const resp = await DataikuAPI.externalinfras.infos.getVertexAIProject(vertexProjectId, connectionName);
                const resp2 = await FutureWatcher.watchJobId(resp.data.jobId).update(data => {
                    if (data.progress && data.progress.states) {
                        $scope.uiState.progress = "Retrieving project information...";
                    } else {
                        $scope.uiState.progress = null;
                    }
                });
                return resp2.data.result;
            } catch (e) {
                $scope.uiState.vertexInformationRetrievalError = "Failed to retrieve project information.";
            }
        }

        $scope.$watch("newSavedModelVersion.targetColumnName", function(nv, ov) {
            CreateSavedModelVersionService.resetErrorDisplay($scope);
        });

        $scope.$watch("newSavedModelVersion.classes", function(nv, ov) {
            CreateSavedModelVersionService.resetErrorDisplay($scope);
        }, true);

        function getProxyModelVersionConfiguration() {
            const proxyModelConfiguration = {
                protocol: $ctrl.savedModel.proxyModelConfiguration.protocol,
                connection: $ctrl.savedModel.proxyModelConfiguration.connection
            };
            const proxyModelVersionConfiguration = {
                protocol: $ctrl.savedModel.proxyModelConfiguration.protocol
            };

            if (proxyModelConfiguration.protocol === "sagemaker") {
                proxyModelConfiguration.region = $ctrl.savedModel.proxyModelConfiguration.region;

                proxyModelVersionConfiguration.endpoint_name = $scope.newSavedModelVersion.sageMakerEndpointName;
            } else if (proxyModelConfiguration.protocol === "vertex-ai") {
                proxyModelConfiguration.project_id = $scope.newSavedModelVersion.vertexProjectId;
                proxyModelConfiguration.region = $ctrl.savedModel.proxyModelConfiguration.region;

                proxyModelVersionConfiguration.endpoint_id = $scope.newSavedModelVersion.vertexEndpointId;
            } else if (proxyModelConfiguration.protocol === "azure-ml") {
                proxyModelConfiguration.subscription_id = $scope.newSavedModelVersion.azureSubscriptionId;
                proxyModelConfiguration.resource_group = $scope.newSavedModelVersion.azureResourceGroup;
                proxyModelConfiguration.workspace = $scope.newSavedModelVersion.azureWorkspace;
                proxyModelVersionConfiguration.endpoint_name = $scope.newSavedModelVersion.azureEndpointName;
            } else if (proxyModelConfiguration.protocol === "databricks") {
                proxyModelVersionConfiguration.endpointName = $scope.newSavedModelVersion.endpointName;
            }
            proxyModelVersionConfiguration.proxyModelConfiguration = proxyModelConfiguration;
            return proxyModelVersionConfiguration;
        }

        function getModelVersionInfo() {
            let signatureAndFormatsGuessingDatasetSmartName, inputFormat, outputFormat;
            if ($ctrl.savedModel.proxyModelConfiguration.protocol === 'vertex-ai') {
                signatureAndFormatsGuessingDatasetSmartName = null;
                inputFormat = 'INPUT_VERTEX_DEFAULT';
                outputFormat = 'OUTPUT_VERTEX_DEFAULT';
            } else if ($scope.uiState.guessFormat) {
                signatureAndFormatsGuessingDatasetSmartName = $scope.uiState.signatureAndFormatsGuessingDatasetSmartName;
                inputFormat = "GUESS";
                outputFormat = "GUESS";
            } else {
                signatureAndFormatsGuessingDatasetSmartName = null;
                inputFormat = $scope.uiState.inputFormat;
                outputFormat = $scope.uiState.outputFormat;
            }
            const targetColumnName = $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.targetColumnName : $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName;
            const gatherFeaturesFromDataset = $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.datasetSmartName : $scope.uiState.signatureAndFormatsGuessingDatasetSmartName;
            const guessingDatasetSamplingParam = $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.samplingParam : $scope.newSavedModelVersion.guessingDatasetSamplingParam;
            const evaluationDatasetSmartName =  $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.datasetSmartName : null;
            const modelVersionInfo = {
                ...(evaluationDatasetSmartName && {evaluationDatasetSmartName}),
                ...(gatherFeaturesFromDataset && {gatherFeaturesFromDataset}),
                ...(guessingDatasetSamplingParam && {guessingDatasetSamplingParam}),
                predictionType: $scope.newSavedModelVersion.predictionType,
                classLabels: [],
                targetColumnName: targetColumnName,
                signatureAndFormatsGuessingDataset: signatureAndFormatsGuessingDatasetSmartName,
                inputFormat,
                outputFormat,
            };
            if ($scope.isClassification()) {
                modelVersionInfo.classLabels = $scope.newSavedModelVersion.classes.map(classLabel => { return { label: classLabel }; });
            }
            modelVersionInfo.proxyModelVersionConfiguration = getProxyModelVersionConfiguration();
            return modelVersionInfo;
        }

        $scope.create = function() {
            CreateSavedModelVersionService.validateClasses($scope);
            CreateSavedModelVersionService.validateTarget($scope);
            if ($scope.hasInvalidClasses() || $scope.invalidTarget) {
                return;
            }
            $scope.deployingSmv = true;

            const modelVersionInfo = getModelVersionInfo();

            resetErrorInScope($scope);
            WT1.event("saved-model-create-proxy", { from: 'sm-list', predictionType: $scope.newSavedModelVersion.predictionType });
            MonoFuture($scope).wrap(DataikuAPI.savedmodels.prediction.createProxySavedModelVersion)(
                $stateParams.projectKey,
                $ctrl.savedModel.id,
                $scope.newSavedModelVersion.versionId,
                modelVersionInfo,
                $scope.newSavedModelVersion.samplingParam,
                $scope.newSavedModelVersion.binaryClassificationThreshold,
                $scope.newSavedModelVersion.useOptimalThreshold,
                $scope.newSavedModelVersion.activate,
                $scope.newSavedModelVersion.skipExpensiveReports,
                {"containerMode": "INHERIT"}
            ).success(function (futureResponse) {
                const smvData = futureResponse.result;
                $scope.smvFuture = null;
                $scope.deployingSmv = false;
                $ctrl.modalControl.dismiss();
                $state.go('projects.project.savedmodels.savedmodel.prediction.report', Object.assign({}, $stateParams, { fullModelId: smvData.fullModelId }));
            }).update(function (data) {
                $scope.smvFuture = data;
            }).error($scope.onSVMCreationError);
        }

        $scope.onSVMCreationError = function (data, status, headers) {
            $scope.smvFuture = null;
            $scope.deployingSmv = false;
            const elem = document.getElementById('new-smv-modal-body');
            if (elem) {
                elem.scrollTop = 0;
            }
            setErrorInScope.bind($scope)(data, status, headers);
        }

        $scope.isClassification = CreateSavedModelVersionService.isClassification.bind(this, $scope);
        $scope.copyClasses = CreateSavedModelVersionService.copyClasses.bind(this, $scope);
        $scope.pasteClasses = CreateSavedModelVersionService.pasteClasses.bind(this, $scope);
    }
});

// use createExternalSavedModelSelectorModalDirective when calling CreateModalFromComponent
app.component('createExternalSavedModelSelectorModal', {
    templateUrl: '/templates/savedmodels/new-external-saved-model-selector-modal.html',
    controller: function($scope, CreateModalFromComponent, createExternalSavedModelModalDirective) {
        const $ctrl = this;
        $scope.newExternalSavedModel = function(externalModelTypeName) {
            CreateModalFromComponent(createExternalSavedModelModalDirective, { externalModelTypeName }, ['modal-wide']);
        }
    }
});

app.controller('AgentSelectorModalController', function ($scope, SavedModelsService, StringUtils) {
    $scope.pluginAgents = SavedModelsService.getAllPluginAgents();
    if ($scope.showAdvancedToolsUsingAgent === undefined) {
        $scope.showAdvancedToolsUsingAgent = true;
    }
    if (!$scope.smNames) {
        $scope.smNames = [];
    }
    $scope.newAgent = {
        agentType: null,
        pluginAgent: null,
        name: null,
        defaultName: StringUtils.transmogrify('Visual Agent', $scope.smNames, null, 1),
    };

    $scope.isNameUnique = function (value) {
        for (let k in $scope.smNames) {
            let name = $scope.smNames[k];
            if ((name || '').toLowerCase() === (value || '').toLowerCase()) {
                return false;
            }
        }
        return true;
    };

    $scope.selectAgentType = function (agentType, mode) {
        $scope.newAgent.agentType = agentType;
        $scope.newAgent.agentMode = mode;
        let defaultNameStart = 'Agent';
        switch (agentType) {
            case 'TOOLS_USING_AGENT':
                defaultNameStart = 'Visual Agent';
                break;
            case 'PYTHON_AGENT':
                defaultNameStart = 'Code Agent';
                break;
            case 'PLUGIN_AGENT':
                defaultNameStart = $scope.newAgent.pluginAgent.label;
                break;

        }
        $scope.newAgent.defaultName = StringUtils.transmogrify(defaultNameStart, $scope.smNames, null, 1);
    };

    $scope.selectPluginAgent = function (pluginAgent) {
        $scope.newAgent.pluginAgent = pluginAgent;
        $scope.selectAgentType('PLUGIN_AGENT');
    };

    $scope.createAgent = function () {
        let agentName = $scope.newAgent.defaultName;
        if ($scope.newAgent.name) {
            agentName = $scope.newAgent.name;
        }

        if ($scope.newAgent.agentType === 'TOOLS_USING_AGENT') {
            SavedModelsService.newVisualAgent(agentName, $scope.newAgent.agentMode, null, 'agent-list').error(setErrorInScope.bind($scope));
        } else if ($scope.newAgent.agentType === 'PYTHON_AGENT') {
            SavedModelsService.newCodeAgent(agentName, null, 'agent-list').error(setErrorInScope.bind($scope));
        } else if ($scope.newAgent.agentType === 'PLUGIN_AGENT' && $scope.newAgent.pluginAgent) {
            SavedModelsService.newPluginAgent($scope.newAgent.pluginAgent.agentId, agentName, null, 'agent-list').error(setErrorInScope.bind($scope));
        }
    };
});

// use createExternalSavedModelModalDirective when calling CreateModalFromComponent
app.component('createExternalSavedModelModal', {
    bindings: {
        externalModelTypeName: '<',
        modalControl: '<'
    },
    templateUrl: '/templates/savedmodels/new-external-saved-model-modal.html',
    controller: function($scope, $state, $stateParams, DataikuAPI, WT1, AnyLoc, PMLSettings, Assert, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS, FutureWatcher) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $scope.creatingSavedModel = false;

            $scope.uiState = {
                availableCompatibleConnections: null,
                advancedMode: false,
            };

            const details = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === $ctrl.externalModelTypeName);

            Assert.trueish(details, 'Invalid or empty externalModelTypeName: ' + $ctrl.externalModelTypeName);

            const isMLflow = details.name == 'mlflow';
            const isFinetuned = details.name == 'finetuned';
            const isProxyModel = !isMLflow && !isFinetuned;

            $scope.predictionTypes = PMLSettings.task.predictionTypes.filter(type => (type.classical || isMLflow && type.other) && !isFinetuned);

            $scope.isSageMaker = details.name == 'sagemaker';
            $scope.isVertex = details.name == 'vertex-ai';

            $scope.externalModelDetails = details;

            $scope.newExternalSavedModel = {
                name : null,
                predictionType : isFinetuned ? null : $scope.predictionTypes[0].type,
                protocol: isProxyModel ? details.name : null,
                connection: "",  // null is used to represent the Environment connection
                region: null,
                savedModelType: details.savedModelType,
                isMLflow: isMLflow,
                isFinetuned: isFinetuned,
                azureSubscriptionId: null,
                azureResourceGroup: null,
                azureWorkspace: null,
                vertexProjectId: null
            };

            $scope.$watch("newExternalSavedModel.protocol", function(protocol) {
                $scope.uiState.availableCompatibleConnections = null;
                if (protocol) {
                    const fullProtocol = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === protocol);
                    if (fullProtocol && fullProtocol.connectionType) {
                        DataikuAPI.connections.getNames(fullProtocol.connectionType).success(function (data) {
                            $scope.uiState.availableCompatibleConnections =
                                (($scope.newExternalSavedModel.protocol == 'databricks')?[]:[{ name: null, label: "Environment" }]).concat(data.map(n => { return { name: n, label: n } }));
                            if ($scope.uiState.availableCompatibleConnections.length == 1) {
                                // the only available connection is environment => select it
                                $scope.newExternalSavedModel.connection = null;
                            }
                        }).error(setErrorInScope.bind($scope));
                    }
                }
            });

            function getProxyModelConfiguration() {
                if (!$scope.newExternalSavedModel.protocol) {
                    return null;
                }

                const proxyModelConfiguration = {
                    protocol: $scope.newExternalSavedModel.protocol,
                    connection: $scope.newExternalSavedModel.connection,
                    region: $scope.newExternalSavedModel.region
                };
                if ($scope.newExternalSavedModel.protocol === "azure-ml") {
                    proxyModelConfiguration.subscription_id = $scope.newExternalSavedModel.azureSubscriptionId;
                    proxyModelConfiguration.resource_group = $scope.newExternalSavedModel.azureResourceGroup;
                    proxyModelConfiguration.workspace = $scope.newExternalSavedModel.azureWorkspace;
                } else if ($scope.newExternalSavedModel.protocol === "vertex-ai") {
                    proxyModelConfiguration.project_id = $scope.newExternalSavedModel.vertexProjectId;
                }
                // No additional fields necessary for SageMaker
                return proxyModelConfiguration;
            }

            $scope.create = function() {
                $scope.creatingSavedModel = true;
                resetErrorInScope($scope);
                WT1.event("saved-model-create-external", { from: 'sm-list', externalModelTypeName: $ctrl.externalModelTypeName, predictionType: $scope.newExternalSavedModel.predictionType });

                const proxyModelConfiguration = getProxyModelConfiguration();
                DataikuAPI.savedmodels.prediction.createExternal(
                    $stateParams.projectKey,
                    $scope.newExternalSavedModel.savedModelType,
                    $scope.newExternalSavedModel.predictionType,
                    $scope.newExternalSavedModel.name,
                    proxyModelConfiguration
                ).success(function(smData) {
                    $scope.creatingSavedModel = false;
                    $state.go('projects.project.savedmodels.savedmodel.versions', Object.assign({}, $stateParams, { smId: smData.id }));
                }).error(function(data, status, headers) {
                    $scope.creatingSavedModel = false;
                    setErrorInScope.bind($scope)(data, status, headers);
                });
            }
        }
    }
});

app.component("sagemakerVertexRegionCodeSelector", {
    bindings: {
        protocol: "<",          // 'vertex-ai' or 'sagemaker'
        value: '=',             // 2-way binding value of the selector (ex: 'eu-west1')
        required: '<',          // Adds a "*" to the title if set
        disabledMessage: "<",   // if set, disable input and tooltip this message
        additionalHelp: "<?"
    },
    template: `
    <div class="region-selector">
        <div block-api-error />
        <label class="control-label">Region{{$ctrl.required ? '*' : ''}}</label>
        <div class="controls"
             toggle="tooltip-left"
             title="{{$ctrl.disabledMessage ? $ctrl.disabledMessage : ''}}">
            <div ng-if="uiState.availableRegions != null"
                toggle="tooltip-left"
                container="body"
                title="{{uiState.selectedRegion.descWithDefault}}">
                <select
                    data-qa-new-model-form-region
                    dku-bs-select
                    ng-model="$ctrl.value"
                    ng-options="r.name as r.nameWithDefault for r in uiState.availableRegions"
                    data-live-search="true"
                    ng-disabled="$ctrl.disabledMessage"
                    ng-required="$ctrl.required"
                />
                <button class="btn btn--secondary" ng-click="uiState.availableRegions = null" ng-disabled="$ctrl.disabledMessage">
                    Enter custom
                </button>
                <span class="help-inline">Region name (eg: "{{ $ctrl.protocol === 'sagemaker' ? 'eu-west-3' : 'europe-west9' }}")</span>
            </div>
            <div ng-if="uiState.availableRegions == null">
                <input
                    type="text"
                    ng-model="$ctrl.value"
                    placeholder="Region"
                    ng-pattern="/^[a-zA-Z][a-zA-Z0-9-]{0,30}\\d$/"
                    ng-disabled="$ctrl.disabledMessage"
                    ng-required="$ctrl.required"
                    data-qa-new-model-form-region
                />
                <button type="button" class="btn btn--secondary" ng-click="fetchAvailableRegions()" ng-disabled="$ctrl.disabledMessage">Get regions</button>
                <span class="help-inline">Region name (eg: "{{ $ctrl.protocol === 'sagemaker' ? 'eu-west-3' : 'europe-west9' }}"){{$ctrl.additionalHelp?('. ' + $ctrl.additionalHelp):''}}</span>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, DataikuAPI) {
        const $ctrl = this;

        $scope.fetchAvailableRegions = function() {
            let listRegions;
            if ($ctrl.protocol === "sagemaker") {
                listRegions = DataikuAPI.externalinfras.infos.listSagemakerRegions;
            } else if ($ctrl.protocol === "vertex-ai") {
                listRegions = DataikuAPI.externalinfras.infos.listVertexAIRegions;
            }
            else {
                throw new Error('Fetch available regions is not supported for ' + $ctrl.protocol);
            }

            listRegions().then(function(resp) {
                $scope.uiState.availableRegions = resp.data;
                $scope.uiState.availableRegions.forEach(r => {
                    r.descWithDefault = r.description + (!r.isDefault?"":" (default from environment)");
                })
                $scope.uiState.availableRegions.forEach(r => r.nameWithDefault = r.name + (r.isDefault?" (default)":""));
                if ($ctrl.value) {
                    const matches = $scope.uiState.availableRegions.filter(r => r.name === $ctrl.value);
                    if (!matches || !matches.length) {
                        $ctrl.value = null;
                    }
                }
                if (!$ctrl.value) {
                    const defaultRegions = $scope.uiState.availableRegions.filter(r => r.isDefault);
                    if (defaultRegions && defaultRegions.length) {
                        // should really only be one at most...
                        $ctrl.value = defaultRegions[0].name;
                    }
                }
            }).catch(setErrorInScope.bind($scope));
        }

        $ctrl.$onInit = function() {
            $scope.uiState = {
                selectedRegion: null
            };

            $scope.$watch("$ctrl.value", function(nv) {
                if (!nv || !$scope.uiState.availableRegions || !$scope.uiState.availableRegions.length) {
                    $scope.uiState.selectedRegion = null;
                } else {
                    const selectedRegions = $scope.uiState.availableRegions.filter(r => r.name === nv);
                    if (selectedRegions && selectedRegions.length) {
                        $scope.uiState.selectedRegion = selectedRegions[0];
                    } else {
                        $scope.uiState.selectedRegion = null;
                    }
                }
            });
        }
    }
});

app.component("vertexProjectSelector", {
    bindings: {
        value: '=',                 // 2-way binding value of the selector (ex: 'team-miel-pops')
        disabledMessage: '<',       // if set, disable input and tooltip this message
        connectionName: '<',        // if set, use the credentials from the connection to log into GCP, else use environment
        required: '<',              // Adds a "*" to the title if set
        disableAutoFetch: '<',      // If projects list should be fetched on component initialization or on connection change (false by default)
    },
    template: `
    <div class="vertex-project-selector">
        <label class="control-label">{{uiState.existingVertexAIProjects ? "Project Name" : "Project ID"}}{{$ctrl.required ? '*' : ''}}</label>
        <div class="controls horizontal-flex"  style="line-height: 24px;"
             toggle="tooltip-left"
             title="{{$ctrl.disabledMessage ? $ctrl.disabledMessage : ''}}">
            <div ng-if="!uiState.loadingVertexAIProjectList && uiState.existingVertexAIProjects != null">
                <select
                        dku-bs-select
                        ng-model="$ctrl.value"
                        ng-options="ep.id as ep.displayName for ep in uiState.existingVertexAIProjects"
                        data-live-search="true"
                        ng-required="$ctrl.required"
                />
                <button class="btn btn--secondary" ng-click="uiState.existingVertexAIProjects = null;">
                    Enter custom
                </button>
            </div>
            <div ng-if="uiState.loadingVertexAIProjectList || uiState.existingVertexAIProjects == null">
                <input type="text"
                       ng-model="$ctrl.value"
                       placeholder="GCP Project ID"
                       ng-disabled="$ctrl.disabledMessage"
                       ng-pattern="/^[a-z][a-z0-9-]{4,28}[a-z0-9]$/"
                       ng-required="$ctrl.required"
                       data-qa-new-version-form-project-name/>
                <div class="dib" toggle="tooltip-right" title="{{$ctrl.isValidConnection($ctrl.connectionName) ? '' : 'Select a connection to fetch the project list'}}">
                    <button ng-if="!uiState.loadingVertexAIProjectList"
                            type="button"
                            class="btn btn--secondary"
                            ng-disabled="$ctrl.disabledMessage || !$ctrl.isValidConnection($ctrl.connectionName)"
                            ng-click="fetchAvailableProjects()">Get projects list</button>
                </div>
                <span ng-if="uiState.loadingVertexAIProjectList"><i class="icon-spin icon-spinner"></i> {{uiState.progress || "Listing projects..."}}</span>
                <div ng-if="uiState.fetchProjectsFailed !== null" class="alert alert-error mtop8">
                    <div><i class="icon-warning-sign alert-error"></i>&nbsp;An error occurred when fetching available projects.</div>
                    <div ng-if="uiState.fetchEndpointsFailed.length > 0">{{uiState.fetchEndpointsFailed}} <a ng-click="$ctrl.copyErrorToClipboard()"><i class="dku-icon-copy-step-16"/></a></div>
                </div>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, DataikuAPI, FutureWatcher, ClipboardUtils) {
        const $ctrl = this;

        $ctrl.$onInit = function () {
            $scope.uiState = {
                selectedProject: null,
                fetchProjectsFailed: null
            };

            if (!$ctrl.value && !$ctrl.disabledMessage && !$ctrl.disableAutoFetch) {
                $scope.fetchAvailableProjects();
            }
        }

        $scope.fetchAvailableProjects = function () {
            $scope.uiState.fetchProjectsFailed = null;
            $ctrl.value = null;
            DataikuAPI.externalinfras.infos.listVertexAIProjects($ctrl.connectionName)
                .then(function (resp) {
                    $scope.$applyAsync(() => {
                        $scope.uiState.loadingVertexAIProjectList = true;
                        $scope.uiState.progress = null;
                    });
                    FutureWatcher.watchJobId(resp.data.jobId)
                        .update(function (data) {
                            if (data.progress && data.progress.states) {
                                $scope.uiState.progress = "Fetching projects list...";
                            } else {
                                $scope.uiState.progress = null;
                            }
                        }).then(function (resp2) {
                        $scope.uiState.existingVertexAIProjects = resp2.data.result;
                        const defaultProject = $scope.uiState.existingVertexAIProjects.filter(r => r.isDefault);
                        if (defaultProject && defaultProject.length) {
                            // should really only be one at most...
                            $ctrl.value = defaultProject[0].id;
                        }

                    }).catch((err) => {
                        $scope.uiState.fetchProjectsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                        setErrorInScope.bind($scope);
                    }).finally(() => {
                        $scope.uiState.loadingVertexAIProjectList = false;
                    });
                })
                .catch((err) => {
                    $scope.uiState.fetchProjectsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                    setErrorInScope.bind($scope);
                }).finally(() => {
                $scope.uiState.loadingVertexAIProjectList = false;
            });
        }

        $scope.$watch("$ctrl.value", function (nv) {
            $scope.uiState.fetchProjectsFailed = null;
            if (!nv || !$scope.uiState.existingVertexAIProjects || !$scope.uiState.existingVertexAIProjects.length) {
                $scope.uiState.selectedProject = null;
            } else {
                const selectedProject = $scope.uiState.existingVertexAIProjects.filter(r => r.id === nv);
                if (selectedProject && selectedProject.length) {
                    $scope.uiState.selectedProject = selectedProject[0];
                } else {
                    $scope.uiState.selectedProject = null;
                }
            }
        });

        $scope.$watch("$ctrl.connectionName", function (nv) {
            if (!$ctrl.value && !$ctrl.disabledMessage && !["", undefined].includes(nv) && !$ctrl.disableAutoFetch) {
                $scope.fetchAvailableProjects();
            } else {
                $scope.uiState.existingVertexAIProjects = null;
            }
        });

        $ctrl.copyErrorToClipboard = function() {
            ClipboardUtils.copyToClipboard($scope.uiState.fetchEndpointsFailed);
        };

        $ctrl.isValidConnection = function(connectionName) {
            return ![undefined, ""].includes(connectionName);
        };
    }
});

app.component("externalPlatformEndpointSelector", {
    bindings: {
        protocol: "<",  // 'vertex-ai', 'sagemaker', 'azure-ml' or 'databricks'
        proxyModelConnection: "<", //
        projectId: "<", // project id input (ex "team-miel-pops")
        region: "<",    // region input. (ex: "europe-west1")
        canFetch: "<",  // enables the "Get Endpoint List" button
        value: "=",      // 2-way binding value of the selector (an endpoint id)
        azureWorkspace: "<", // $scope.newSavedModelVersion.azureWorkspace,
        azureResourceGroup: "<", // $scope.newSavedModelVersion.azureResourceGroup,
        azureSubscriptionId: "<"
    },
    template: `
    <div class="external-platform-endpoint-selector">
        <label class="control-label">{{uiState.existingEndpoints ? "Endpoint Name*" : "Endpoint ID*"}}</label>
        <div class="controls horizontal-flex"  style="line-height: 24px;">
            <div ng-if="!uiState.loadingEndpointList && uiState.existingEndpoints != null">
                <select
                        ng-if="$ctrl.protocol === 'databricks'"
                        dku-bs-select
                        ng-model="$ctrl.value"
                        ng-options="ep.name as ep.name for ep in uiState.existingEndpoints"
                        data-live-search="true"
                        required
                />
                <select
                        ng-if="$ctrl.protocol !== 'databricks'"
                        dku-bs-select
                        ng-model="$ctrl.value"
                        ng-options="ep.id || ep.name as ep.name for ep in uiState.existingEndpoints"
                        data-live-search="true"
                        required
                />
                <button class="btn btn--secondary" ng-click="uiState.existingEndpoints = null;">
                    Enter custom
                </button>
            </div>
            <div ng-if="uiState.loadingEndpointList || uiState.existingEndpoints == null">
                <input
                    type="text"
                    ng-model="$ctrl.value"
                    ng-pattern="endpointValidationPattern"
                    placeholder="Endpoint ID"
                    required
                    data-qa-new-version-form-endpoint-name
                />
                <button ng-if="!uiState.loadingEndpointList" ng-disabled="!$ctrl.canFetch" type="button" class="btn btn--secondary" ng-click="fetchAvailableEndpoints()">Get endpoints list</button>
                <span ng-if="uiState.loadingEndpointList"><i class="icon-spin icon-spinner"></i> {{uiState.progress || "Listing endpoints..."}}</span>
                <div ng-if="uiState.fetchEndpointsFailed !== null" class="alert alert-error mtop8">
                    <i class="icon-warning-sign alert-error"></i>&nbsp;An error occurred when fetching available Endpoints.
                    <div ng-if="uiState.fetchEndpointsFailed.length > 0">{{uiState.fetchEndpointsFailed}} <a ng-click="$ctrl.copyErrorToClipboard()"><i class="dku-icon-copy-step-16"/></a></div>
                </div>
            </div>
        </div>
        <div ng-if="uiState.selectedEndpoint.fullId">
            <label class="control-label"></label>
            <div class="controls" style="line-height: 24px;">
                <span class="help-inline" style="width: 100%">Endpoint id: {{uiState.selectedEndpoint.fullId}}</span>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, DataikuAPI, FutureWatcher, ActiveProjectKey, ClipboardUtils) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $scope.uiState = {
                fetchEndpointsFailed: null
            };

            $scope.endpointValidationPattern = getEndpointValidationPattern();
        };

        $scope.fetchAvailableEndpoints = function() {
            $scope.uiState.fetchEndpointsFailed = null;

            let listEndpoints;

            if ($ctrl.protocol === "vertex-ai") {
                listEndpoints = DataikuAPI.externalinfras.infos.listVertexAIEndpoints.bind(this, $ctrl.projectId, $ctrl.region, $ctrl.proxyModelConnection);
            }
            else if ($ctrl.protocol === "sagemaker") {
                listEndpoints = DataikuAPI.externalinfras.infos.listSagemakerEndpointSummaries.bind(this, ActiveProjectKey.get(), $ctrl.region, $ctrl.proxyModelConnection);
            }
            else if ($ctrl.protocol === "azure-ml") {
                listEndpoints = DataikuAPI.externalinfras.infos.listAzureMLEndpoints.bind(this,
                    ActiveProjectKey.get(), $ctrl.azureWorkspace, $ctrl.azureResourceGroup, $ctrl.azureSubscriptionId, $ctrl.proxyModelConnection);
            }
            else if ($ctrl.protocol === "databricks") {
                listEndpoints = DataikuAPI.externalinfras.infos.listDatabricksEndpoints.bind(this,
                    ActiveProjectKey.get(), $ctrl.proxyModelConnection);
            }
            else {
                throw new Error('Fetch available endpoints is not supported for ' + $ctrl.protocol);
            }

            listEndpoints().then(function(resp) {
                $scope.$applyAsync(() => {
                    $scope.uiState.loadingEndpointList = true;
                    $scope.uiState.progress = null;
                });
                FutureWatcher.watchJobId(resp.data.jobId)
                    .update(function(data) {
                        if (data.progress && data.progress.states) {
                            $scope.uiState.progress = "Fetching endpoint list...";
                        } else {
                            $scope.uiState.progress = null;
                        }
                    }).then(function(resp2) {
                    if (resp2.data.aborted) {
                        $scope.uiState.fetchEndpointsFailed = "Fetch endpoint list aborted";
                    }
                    else {
                        $scope.uiState.existingEndpoints = resp2.data.result;
                        if (!$scope.uiState.existingEndpoints || !$scope.uiState.existingEndpoints.length) {
                            $ctrl.value = null;
                        } else {
                            $ctrl.value = $scope.uiState.existingEndpoints.map(e => e.name).find(n => n===$ctrl.value);
                        }
                    }
                }).catch((err) => {
                    $scope.uiState.fetchEndpointsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                    setErrorInScope.bind($scope);
                }).finally(() => { $scope.uiState.loadingEndpointList = false;});
            }).catch((err) => {
                    $scope.uiState.fetchEndpointsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                    setErrorInScope.bind($scope);
            }).finally(() => { $scope.uiState.loadingEndpointList = false;});
        };

        function getEndpointValidationPattern () {
            if ($ctrl.protocol === "sagemaker" || $ctrl.protocol === "vertex-ai") {
                return /^(?!-)(?!.*-$)[a-zA-Z0-9-]{1,63}$/
            }
            if ($ctrl.protocol === "azure-ml") {
                return /^[a-z][a-z0-9-]{1,30}[a-z0-9]$/;
            }
            if ($ctrl.protocol === "databricks") {
                return /^[a-zA-Z0-9]([\w-]{0,61}[a-zA-Z0-9])?$/;
            }
            throw new Error('Invalid protocol: ' + $ctrl.protocol);
        }

        $scope.$watch("$ctrl.projectId", function() {
            $scope.uiState.selectedEndpoint = null;
        });

        $scope.$watch("$ctrl.name", function() {
            $scope.uiState.selectedEndpoint = null;
        });

        $scope.$watch("$ctrl.value", function(nv) {
            $scope.uiState.fetchEndpointsFailed = null;
            if (!nv || !$scope.uiState.existingEndpoints || !$scope.uiState.existingEndpoints.length) {
                $scope.uiState.selectedEndpoint = null;
            } else {
                let selectedEndpoint;
                if ($ctrl.protocol === "databricks") {
                    selectedEndpoint = $scope.uiState.existingEndpoints.filter(e => e.name === nv);
                } else {
                    selectedEndpoint = $scope.uiState.existingEndpoints.filter(e => e.id ? e.id === nv : e.name === nv);
                }
                if (selectedEndpoint && selectedEndpoint.length) {
                    $scope.uiState.selectedEndpoint = selectedEndpoint[0];
                } else {
                    $scope.uiState.selectedEndpoint = null;
                }
            }
        });

        $ctrl.copyErrorToClipboard = function() {
            ClipboardUtils.copyToClipboard($scope.uiState.fetchEndpointsFailed);
        };
    }

});

app.controller('ExternalModelSetupMonitoringModalController', function($scope, $stateParams, DataikuAPI, ActivityIndicator, ActiveProjectKey, $state, WT1, StringUtils){

    DataikuAPI.datasets.listNames($stateParams.projectKey)
        .success(function(datasetNames) {
            $scope.datasetName = StringUtils.transmogrify($scope.smName + "_logs", datasetNames)
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.modelevaluationstores.listHeads($stateParams.projectKey)
        .success(function(heads) {
            $scope.mesName = StringUtils.transmogrify($scope.smName + "_evaluation_store", heads.map(head => head.name))
        })
        .error(setErrorInScope.bind($scope));

    const cloudStorageConnectionType = 'EC2';

    DataikuAPI.connections.listCloudConnectionsHDFSRoot(cloudStorageConnectionType)
        .success(function (data) {
            const cloudStorageConnections = data;
            cloudStorageConnections.sort(function (a,b) {
                const comparison = isLikelyConnection(b, $scope.predictionLogsUri) - isLikelyConnection(a, $scope.predictionLogsUri);
                if (comparison === 0) {
                    return a.name.localeCompare(b.name); // sort alphabetically otherwise
                } else {
                    return comparison;
                }
            });
            $scope.cloudStorageConnections = cloudStorageConnections.map(connection => connection.name);

            if ($scope.cloudStorageConnections.length > 0){
                $scope.connection = $scope.cloudStorageConnections[0];
            }
        })
        .error(setErrorInScope.bind($scope));

    const smId = $stateParams.smId;
    const projectKey = ActiveProjectKey.get();

    /***
     * This methods, used in a sort, will put first in the list the connections that match the bucket and root of the logs URI,
     * second the conenctions that have no root/bucket, and last the connections that don't match it and won't work
     */
    function isLikelyConnection(connection, predictionLogsUri) {
        if (connection.predictionLogsRoot && predictionLogsUri.startsWith(connection.predictionLogsRoot)) {
            return connection.predictionLogsRoot.length;
        } else {
            return -1;
        }
    }

    $scope.ok = function() {
        const params = {
            createMes: $scope.createMes,
            createOutputDataset: $scope.createOutputDataset
        }
        DataikuAPI.savedmodels.prediction.setupProxyModelMonitoring(projectKey, smId, $scope.versionId, $scope.connection, params)
            .success(function(data) {
                ActivityIndicator.success(`External model monitoring created ! <a href="${$state.href('projects.project.flow', {id : 'dataset_' + $stateParams.projectKey + "." + data.inputDatasetRef.objectId})}">
                    View in flow
                </a>`, 5000);
                $scope.resolveModal();
                WT1.event('external-model-monitoring-created', {type: $scope.connection != null ? $scope.connection.type : null});
            })
            .catch((err) => {
                WT1.event('external-model-monitoring-failure', {type: $scope.connection != null ? $scope.connection.type : null});
                setErrorInScope.bind($scope)(err);
            });
    }
    })


app.controller("_AbstractSavedModelPredictedTableController", function($scope, $stateParams, $state, $controller, $q, Assert, DataikuAPI, MonoFuture, Logger, ExportModelDatasetService){
    $scope.exportPredictedData = function() {
        ExportModelDatasetService.exportPredictedData($scope, $scope.smContext.model.fullModelId);
    };

    $scope.shakerWithSteps = false;

    /* ********************* Callbacks for shakerExploreBase ******************* */

    // Nothing to save
    $scope.shakerHooks.saveForAuto = function() {
        return Promise.resolve({});
    }

    var monoFuturizedRefresh = MonoFuture($scope).wrap(DataikuAPI.savedmodels.predicted.predictedRefreshTable);

    $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
        return monoFuturizedRefresh($stateParams.fullModelId, $scope.shaker, filtersOnly, filterRequest);
    }

    $scope.shakerHooks.shakerForQuery = function(){
        var queryObj = angular.copy($scope.shaker);
        if ($scope.isRecipe) {
            queryObj.recipeSchema = $scope.recipeOutputSchema;
        }
        queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
        return queryObj;
    }

    $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
        // withFullSampleStatistics, fullSamplePartitionId are not relevant in this context
        DataikuAPI.savedmodels.predicted.detailedColumnAnalysis($stateParams.fullModelId, $scope.shakerHooks.shakerForQuery(), columnName, alphanumMaxResults).success(function(data){
                    setAnalysis(data);
        }).error(function(a, b, c) {
            if (handleError) {
                handleError(a, b, c);
            }
            setErrorInScope.bind($scope)(a, b, c);
        });
    };

    $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
        return DataikuAPI.savedmodels.predicted.predictedGetTableChunk($stateParams.fullModelId, $scope.shaker,
            firstRow, nbRows, firstCol, nbCols, filterRequest)
    }

    // Load shaker
    DataikuAPI.savedmodels.predicted.getPredictionDisplayScript($stateParams.fullModelId).success(function(data){
        $scope.baseInit();
        $scope.shaker = data;
        $scope.originalShaker = angular.copy($scope.shaker);
        $scope.fixupShaker();
        $scope.refreshTable(false);
    }).error(setErrorInScope.bind($scope));

});


app.directive("predictionSavedModelPredictedTable", function() {
    return {
        scope: true,
        controller: function($scope, $stateParams, $controller, DataikuAPI, ActiveProjectKey, PMLFilteringService, TopNav, WT1) {
            WT1.event("savedmodel-pml-predicted-table-open");
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "PREDICTION-SAVED_MODEL-VERSION", "predictedtable");
            $controller("_PredictionModelReportController",{$scope:$scope});
            $controller("_SavedModelReportController", {$scope:$scope});
            $controller("_SavedModelGovernanceStatusController", {$scope:$scope});

            // Fill the version selector
            const getStatusP = DataikuAPI.savedmodels.prediction.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
                $scope.getGovernanceStatus($stateParams.fullModelId, data.task.partitionedModel);
                $scope.fillVersionSelectorStuff(data, true);
                $scope.versionsContext.versions.forEach(function(m){
                    m.snippet.mainMetric = PMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                    m.snippet.mainMetricStd = PMLFilteringService.getMetricStdFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                });
            });

            $controller("_AbstractSavedModelPredictedTableController", {$scope:$scope});

        }
    }
});

app.directive("clusteringSavedModelPredictedTable", function() {
    return {
        scope: true,
        controller: function($scope, $stateParams, $controller, DataikuAPI, ActiveProjectKey, CMLFilteringService, TopNav, WT1) {
            WT1.event("savedmodel-cml-predicted-table-open");
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "CLUSTERING-SAVED_MODEL-VERSION", "predictedtable");
            $controller("_ClusteringModelReportController",{$scope:$scope});
            $controller("_SavedModelReportController", {$scope:$scope});
            $scope.clusteringResultsInitDone = false;

            // Fills the version selector
            const getStatusC = DataikuAPI.savedmodels.clustering.getStatus(ActiveProjectKey.get(), $stateParams.smId)
                .then(({data}) => {
                    $scope.fillVersionSelectorStuff(data, true);
                    $scope.versionsContext.versions.forEach(function(m){
                        m.snippet.mainMetric = CMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                    });
            });

            $controller("_AbstractSavedModelPredictedTableController", {$scope:$scope});
        }
    }
});

/* ***************************** Retrieval Augmented LLM creation ************************** */

app.controller('CreateRetrievalAugmentedLLMModalController', function ($scope, DataikuAPI, $state, $stateParams, SmartId, StringUtils) {
    $scope.input = {
        preselectedInput: null,
    };
    $scope.params = {
        name: null,
        knowledgeBank: null,
        knowledgeBankRef: null,
        llmId: null,
    };

    DataikuAPI.savedmodels.retrievalAugmentedLLMs
        .list($stateParams.projectKey)
        .success(function (data) {
            $scope.retrievalAugmentedLLMNames = data.map((sm) => sm.name);
            $scope.$watch('input.preselectedInput', function (preselectedInput) {
                if (Array.isArray(preselectedInput) && preselectedInput.length === 1 && preselectedInput[0].type === 'RETRIEVABLE_KNOWLEDGE') {
                    $scope.params.knowledgeBankRef = SmartId.create(preselectedInput[0].id, preselectedInput[0].projectKey ?? $stateParams.projectKey);
                }
            });
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels
        .listAvailableLLMs($stateParams.projectKey, 'GENERIC_COMPLETION')
        .then(function ({ data }) {
            $scope.availableAugmentableLLMs = data.identifiers.filter((id) => id.type !== 'RETRIEVAL_AUGMENTED' && id.type !== 'SAVED_MODEL_AGENT');
        })
        .catch(setErrorInScope.bind($scope));

    const defaultNameStart = 'Retrieval of ';


    $scope.$watch('params.knowledgeBankTO', function () {
        if ((!$scope.params.name || $scope.params.name.startsWith(defaultNameStart)) && $scope.params.knowledgeBankTO) {
            $scope.params.name = StringUtils.transmogrify(defaultNameStart + $scope.params.knowledgeBankTO.label, $scope.retrievalAugmentedLLMNames, null, 1);
        }
    });

    $scope.isNameUnique = function (value) {
        for (let k in $scope.retrievalAugmentedLLMNames) {
            let name = $scope.retrievalAugmentedLLMNames[k];
            if ((name || '').toLowerCase() === (value || '').toLowerCase()) {
                return false;
            }
        }
        return true;
    };

    $scope.createRetrievalAugmentedLLM = function () {
        DataikuAPI.savedmodels.retrievalAugmentedLLMs
            .create($stateParams.projectKey, $scope.params.name, $scope.params.knowledgeBankRef, $scope.params.llmId)
            .success(function (data) {
                $state.go('projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design', {
                    fullModelId: `S-${$stateParams.projectKey}-${data.id}-${data.activeVersion}`,
                    smId: data.id,
                });
            })
            .catch(setErrorInScope.bind($scope));
    };
});

/* ***************************** Test In Prompt Studio Modal ************************** */

app.controller('TestInPromptStudioModalController', function ($scope, $stateParams, $state, DataikuAPI) {
    $scope.prompt = {};

    DataikuAPI.promptStudios
        .listHeads($stateParams.projectKey)
        .then(function ({ data }) {
            const createOption = { createOption: true, name: 'Create a new Prompt studio...' };
            $scope.promptStudios = data;
            $scope.promptStudios.unshift(createOption);
            $scope.prompt.studio = createOption;
            $scope.prompt.newPromptName = '';
            return;
        })
        .catch(setErrorInScope.bind($scope));

    $scope.editInStudio = function () {
        const promptCreationSettings = {};
        if ($scope.prompt.studio.createOption) {
            promptCreationSettings.newPromptStudioName = $scope.prompt.newPromptName;
        } else {
            promptCreationSettings.promptStudioId = $scope.prompt.studio.id;
        }

        $scope.save(true).then(function () {
            DataikuAPI.promptStudios
                .createFromSM($stateParams.projectKey, $scope.savedModel.id, promptCreationSettings)
                .then(function ({ data }) {
                    $state.go('projects.project.promptstudios.promptstudio', { promptStudioId: data.promptStudioId, promptId: data.promptStudioPromptId });
                })
                .catch(setErrorInScope.bind($scope));
        });
    };
});

app.directive('changeSmVersionDropdown', function () {
    return {
        restrict: 'E',
        templateUrl: '/templates/savedmodels/change-sm-version-dropdown.html',
        scope: {
            showMetrics: '<',
            showDateOnly: '<',
            withVersionName: '<',
            uiSref: '@?',
            versionsContext: '<',
        },
        controller: function ($scope) {
            const DEFAULT_SREF = 'projects.project.savedmodels.savedmodel.prediction.report';
            $scope.uiSrefFinal = $scope.uiSref || DEFAULT_SREF;
        }
    };
});

})();
