(function() {
'use strict';

/* Base directives for the "exploration-only" part of shaker */

// Base_explore is the first loaded and creates the module.
const app = angular.module('dataiku.shaker', ['dataiku.filters', 'dataiku.shared', 'platypus.utils']);

app.directive("shakerExploreBase", function(Logger, $filter, $rootScope, $translate, translate) {
    return {
        scope: true,
        priority : 100,
        controller : function($scope, $rootScope, $stateParams, $state, DataikuAPI, CachedAPICalls, $filter, CreateModalFromTemplate,
                              WT1, ActivityIndicator, $timeout, $q, Debounce, MonoFuture, GraphZoomTrackerService, ActiveProjectKey,
                              Dialogs, FutureProgressModal, computeColumnWidths, SmartId, SamplingData, ChartsStaticData, RatingFeedbackParams, 
                              ConditionalFormattingEditorService, ClipboardUtils) {
            $scope.isRecipe = false;
            let lastValidatedSteps;

            $scope.ratingFeedbackParams = RatingFeedbackParams;

            //This is necessary to make sure  the banner disappears when we route to another page (otherwise, it appears again after we open a recipe)
            $scope.$on('$stateChangeStart', (event, toState, toParams, fromState) => {
                if (fromState.name !== 'projects.project.recipes.recipe') {
                    // Only alter the rating feedback banner if what we are leaving is NOT a recipe (mainly because the prepare recipe also use this directive)
                    // For the recipes the alteration is done accordingly in the RecipeEditorController
                    $scope.ratingFeedbackParams.showRatingFeedback = false;
                }
            });

            if (!$stateParams.fromFlow) {
                // Do not change the focus item zoom if coming from flow
                GraphZoomTrackerService.setFocusItemById("dataset", $stateParams.datasetName);
            }

            $scope.RIGHT_PANE_VIEW = {
                NONE: 0,
                QUICK_COLUMNS_VIEW: 1,
                QUICK_CONDITIONAL_FORMATTING_VIEW: 2
            };

            $scope.isQuickColumnsViewOpened = function () {
                return $scope.shakerState.rightPaneView === $scope.RIGHT_PANE_VIEW.QUICK_COLUMNS_VIEW;
            }
            $scope.isQuickConditionalFormattingViewOpened = function () {
                return $scope.shakerState.rightPaneView === $scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW;
            }
            // rightPaneView is a value from the "enum" $scope.RIGHT_PANE_VIEW
            $scope.openRightPaneWith = function (rightPaneView) {
                $scope.shakerState.rightPaneView = rightPaneView;
            }

            $scope.conditionalFormattingState = {
                columns: [],
                isWholeData: false,
                openedFromColumnHeader: null,
                tableHeaders: [],
            };

            $scope.shakerState = {
                activeView: 'table',
                // For now only 1 view can be opened on the right panel
                rightPaneView: $scope.RIGHT_PANE_VIEW.NONE,

                lockedHighlighting : [],
                // this is a row selected, now used for the compare feature, the row is highlighted in blue
                selectedRow: null,
            };

            updateColoringStates();

            // Real controller inserts its hooks here
            $scope.shakerHooks = {
                isMonoFuturizedRefreshActive: function(){}, // NOSONAR: OK to have an empty function

                // Returns a promise when save is done
                saveForAuto: undefined,
                // Returns a promise that resolves with a future for the refresh
                getRefreshTablePromise: undefined,

                // Sets the meaning of a column
                setColumnMeaning : undefined,

                // Sets the storage type of a column
                getSetColumnStorageTypeImpact : undefined,
                setColumnStorageType : undefined,

                // Should open a box to edit the details of the column
                // (meaning, storage type, description)
                editColumnDetails : undefined,

                // Hook called in parallel to the table refresh
                onTableRefresh : function(){}, // NOSONAR: OK to have an empty function that does nothing by default
                // Hook called after the table refresh
                afterTableRefresh : function(){}, // NOSONAR: OK to have an empty function that does nothing by default

                // analysis modal :
                // - fetch the detailed analysis of a column
                fetchDetailedAnalysis : undefined,
                // - get clusters
                fetchClusters : undefined,
                // - compute text analysis
                fetchTextAnalysis : undefined
            };

            Mousetrap.bind("r s", function(){
                DataikuAPI.shakers.randomizeColors();
                $scope.refreshTable(false);
            })

            Mousetrap.bind("alt+a", function(){
                $scope.$apply(function(){
                    $scope.shaker.exploreUIParams.autoRefresh = !$scope.shaker.exploreUIParams.autoRefresh;
                    ActivityIndicator.success("Auto refresh is now " +
                        ($scope.shaker.exploreUIParams.autoRefresh ? "enabled" : "disabled"));
                    if ($scope.shaker.exploreUIParams.autoRefresh) {
                        $scope.autoSaveAutoRefresh();
                    }
                });
            })

            $scope.$on("$destroy", function(){
                Mousetrap.unbind("r s");
                Mousetrap.unbind("alt+a")
            })

            function id(dataset) {
                //if the current dataset is foreign, force the use of full dataset names (equi-joiner for example requires it)
                if ($scope.inputDatasetProjectKey != $stateParams.projectKey) {
                    return dataset.projectKey + '.' + dataset.name;
                } else {
                    return dataset.smartName;
                }
            }

            $scope.datasetHref = function() {
                if (!$scope.dataset) {return ''}
                return $state.href('projects.project.datasets.dataset.explore', {datasetName: $scope.dataset.name});
            }
            /** Called by the real controller to fetch required data once context has been set */
            $scope.baseInit = function(displaySpinner = true) {
                if ($scope.inputDatasetName) {
                    if ($rootScope.topNav.isProjectAnalystRO) {
                        DataikuAPI.datasets.getFullInfo($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName).spinner(displaySpinner).success(function(data){
                            $scope.datasetFullInfo = data;
                        }).error(setErrorInScope.bind($scope));

                        DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey).spinner(displaySpinner)
                        .success(function(data) {
                            $scope.dataset = data;
                        }).error(setErrorInScope.bind($scope));
                        var opts = {
                            datasetsOnly : true
                        };
                        DataikuAPI.flow.listUsableComputables($stateParams.projectKey, opts).spinner(displaySpinner).success(function(computables) {
                            $scope.datasetNames = $.map(computables, function(val) {
                                return id(val);
                            });
                        }).error(setErrorInScope.bind($scope));

                        CachedAPICalls.datasetTypes.then(function(datasetTypes) {
                            $scope.dataset_types = datasetTypes;
                        }).catch(setErrorInScope.bind($scope));
                        $scope.datasetColumns = {};
                        $scope.getDatasetColumns = function(datasetId) { // datasetId is something id() would return
                            // only for input datasets. Only once (we don't care if the schema is changing while we edit the shaker)
                            if ($scope.datasetNames && $scope.datasetNames.indexOf(datasetId) >= 0 && !(datasetId in $scope.datasetColumns)) {
                                let resolvedSmartId = SmartId.resolve(datasetId, $stateParams.projectKey);
                                $scope.datasetColumns[datasetId] = [];
                                DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, $stateParams.projectKey).spinner(displaySpinner).success(function(dataset){
                                    $scope.datasetColumns[datasetId] = $.map(dataset.schema.columns, function(el) {
                                        return el.name;
                                    });
                                    // and let the digest update the UI...
                                }).error(setErrorInScope.bind($scope));
                            }
                            return $scope.datasetColumns[datasetId];
                        };
                    }
                }

                CachedAPICalls.processorsLibrary.success(function(processors){
                    $scope.processors = processors;
                }).error(setErrorInScope.bind($scope));

            }


            /** Real handler calls this once $scope.shaker is set to fixup incomplete scripts */

            $scope.fixupShaker = function(shaker) {
                shaker = shaker || $scope.shaker;
                if (shaker.exploreUIParams == null) {
                    shaker.exploreUIParams = {};
                }
                if (shaker.exploreUIParams.autoRefresh == null) {
                    shaker.exploreUIParams.autoRefresh = true;
                }
                if (shaker.explorationFilters == null) {
                    shaker.explorationFilters = [];
                }
            }

            $scope.$watch("shaker.steps", function(nv, ov){
                if (!nv) return;
                 function _addChange(s) {
                    if (s.metaType == "GROUP") {
                        if (s.steps) {
                            s.steps.forEach(_addChange);
                        }
                    }
                    if (s.$stepState == null) {
                        s.$stepState = {}
                    }
                }
                $scope.shaker.steps.forEach(_addChange);
            }, true);

            $scope.invalidScriptError = {};

            $scope.forgetSample = function() {
                $scope.requestedSampleId = null;
            };

            $scope.computeFullRowCount = function() {
                // Only force recomputing the whole row count if we are sure we are not on a partitioned dataset.
                // On a partitioned dataset, we only want to recompute the partitions that may have been updated.
                const forceRecompute = Boolean($scope.datasetFullInfo && !$scope.datasetFullInfo.partitioned);
                DataikuAPI.datasets.getRefreshedSummaryStatus($scope.inputDatasetProjectKey, $scope.inputDatasetName, true, forceRecompute).success(function(data) {
                    FutureProgressModal.show($scope, data, "Computing row count").then(function(result){
                        if ($scope.datasetFullInfo) {
                            $scope.datasetFullInfo.dataset.status = result;
                        }
                        if (result) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Row count updated", result.messages);
                            if (result.records && result.records.hasData && !result.records.incomplete && $scope.table && $scope.table.sampleMetadata) {
                                $scope.table.sampleMetadata.datasetRecordCount = result.records.totalValue;
                                $scope.table.sampleMetadata.recordCountIsObsolete = false;
                                $scope.table.sampleMetadata.recordCountIsApproximate = false;
                                $rootScope.$broadcast('rightPanelSummary.triggerFullInfoUpdate'); // Ensure that the right panel updates itself with the latest information
                            }
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $scope.showComputeFullRowCountButton = function() {
                return $scope.table
                    && $scope.table.sampleMetadata
                    && !$scope.table.sampleMetadata.sampleIsWholeDataset
                    && ($scope.table.sampleMetadata.datasetRecordCount === -1
                        || $scope.table.sampleMetadata.recordCountIsApproximate
                        || $scope.table.sampleMetadata.recordCountIsObsolete)
                    && (ActiveProjectKey.get() === $scope.inputDatasetProjectKey); // No refresh button on foreign datasets because user might not have the permissions to compute metrics on this dataset
            }

            $scope.saveOnly = function() {
                $scope.shakerHooks.saveForAuto().then(function(){
                    ActivityIndicator.success("Changes saved");
                });
            };

            $scope.getSampleConfigDesc = function() {
                if ($scope.shaker && $scope.shaker.explorationSampling && $scope.shaker.explorationSampling.selection) {
                    const selection = $scope.shaker.explorationSampling.selection;
                    return SamplingData.formatSamplingConfig(selection.samplingMethod, selection.maxRecords, selection.targetRatio);
                }
                return "-";
            }

            $scope.switchToSamplingParamsTab = function() {
                $rootScope.$broadcast('tabSelect', 'sample-settings')
            }

            $scope.getSampleDesc = function() {
                function formatSamplingRowCountRestrictions(sampleMetadata) {
                    const spacer = " ";
                    if (sampleMetadata.partitionCount > 0) {
                        const params = { partitionCount: sampleMetadata.partitionCount };
                        if (sampleMetadata.hasFilter) {
                            return translate("SHAKER.SAMPLING.ACTUAL.FILTERED_WITH_PARTITION", "(filtered & {{partitionCount}} {{partitionCount == 1 ? 'partition' : 'partitions'}})", params) + spacer;
                        } else {
                            return translate("SHAKER.SAMPLING.ACTUAL.WITH_PARTITION", "({{partitionCount}} {{partitionCount == 1 ? 'partition' : 'partitions'}})", params) + spacer;
                        }
                    } else if (sampleMetadata.hasFilter) {
                        return translate("SHAKER.SAMPLING.ACTUAL.FILTERED", "(filtered)") + spacer;
                    }
                    return "";
                }

                const table = $scope.table;
                if (!table) {
                    return "";
                }

                const sampleMetadata = table.sampleMetadata;
                let strRow = 'row' + (table.initialRows === 1 ? '': 's');
                if (sampleMetadata && !sampleMetadata.sampleIsWholeDataset) {
                    const params = {
                        rowCount: table.initialRows,
                        smartRowCount: $filter('longSmartNumber')(table.initialRows),
                        rowCountRestrictions: formatSamplingRowCountRestrictions(sampleMetadata),
                        totalRowCount: sampleMetadata.datasetRecordCount === -1
                            ? translate('SHAKER.SAMPLING.ACTUAL.TOTAL_NOT_COMPUTED', "not computed")
                            : $filter('longSmartNumber')(Math.max(table.initialRows, sampleMetadata.datasetRecordCount)),
                        totalRowCountEstimated: sampleMetadata.recordCountIsApproximate
                            ? (" " + translate('SHAKER.SAMPLING.ACTUAL.TOTAL_ESTIMATED', "(estimated)"))
                            : ""
                    };
                    switch (sampleMetadata.samplingMethod) {
                        case "HEAD_SEQUENTIAL":
                            return translate("SHAKER.SAMPLING.ACTUAL.HEAD_SEQUENTIAL", `First <strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "TAIL_SEQUENTIAL":
                            return translate("SHAKER.SAMPLING.ACTUAL.TAIL_SEQUENTIAL", `Last <strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "RANDOM_FIXED_NB":
                        case "COLUMN_BASED":
                        case "STRATIFIED_TARGET_NB_EXACT":
                        case "CLASS_REBALANCE_TARGET_NB_APPROX":
                        case "RANDOM_FIXED_NB_EXACT":
                            return translate("SHAKER.SAMPLING.ACTUAL.RANDOM", `<strong>{{smartRowCount}}</strong> random {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "RANDOM_FIXED_RATIO":
                        case "STRATIFIED_TARGET_RATIO_EXACT":
                        case "CLASS_REBALANCE_TARGET_RATIO_APPROX":
                        case "RANDOM_FIXED_RATIO_EXACT":
                            return translate("SHAKER.SAMPLING.ACTUAL.RANDOM", `<strong>{{smartRowCount}}</strong> random {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "FULL":
                        default:
                            return translate("SHAKER.SAMPLING.ACTUAL.FULL", `<strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                    }
                } else {
                    const params = {
                        rowCount: table.initialRows,
                        smartRowCount: $filter('longSmartNumber')(table.initialRows),
                        rowCountRestrictions: formatSamplingRowCountRestrictions(sampleMetadata)
                    };
                    return translate("SHAKER.SAMPLING.ACTUAL.WHOLE", `<strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}`, params);
                }
            }

            $scope.getSampleTooltip = function() {
                const table = $scope.table;
                if (!table) {
                    return "";
                }
                let result = $scope.getSampleDesc();
                if (table.sampleMetadata && table.sampleMetadata.memoryLimitReached) {
                    result += `<br><strong>${table.sampleMetadata.memoryLimitInMB}</strong> MB memory limit reached`
                }
                return result;
            }

            $scope.getSampleDescSimple = function() {
                const table = $scope.table;
                if (!table) {
                    return "";
                }
                return SamplingData.formatSamplingConfig("FULL", table.totalRows);
            }

            $scope.$on('refresh-table',function() {
                $scope.autoSaveForceRefresh();
            });

            /* Save if auto-save is enabled and force a refresh */
            $scope.autoSaveForceRefresh = function() {
                if (!$scope.isRecipe && ($scope.canWriteProject() || $scope.isLocalSave)) {
                    $scope.shakerHooks.saveForAuto().catch(error => {
                        Logger.error("Could not save", { error });
                    });
                }
                $scope.refreshTable(false);
            };

            $scope.autoSave = function() {
                if (!$scope.isRecipe && ($scope.canWriteProject() || $scope.isLocalSave)) {
                    $scope.shakerHooks.saveForAuto();
                }
            };

            // returns relevant shaker data, a fat-free data only object
            // without the change information.
            $scope.getShakerData = function() {
                // get only own property stuff.
                if ($scope.shaker == undefined)  {
                    return undefined;
                }

                function clearOne(step) {
                    if (step.metaType == "GROUP") {
                        step.steps.forEach(clearOne);
                    } else {
                        delete step.$stepState;
                        delete step.$$hashKey;
                    }
                }

                var shakerData = JSON.parse(JSON.stringify($scope.shaker));
                shakerData.steps.forEach(clearOne);
                return shakerData;
            };

            // formerShakerData is supposed to hold the last shaker state for which we updated
            // the table.
            $scope.setFormerShakerData = function() {
                $scope.formerShakerData = $scope.getShakerData();
            }

            /* Save if auto-save is enabled and refresh if auto-refresh is enabled */
            $scope.autoSaveAutoRefresh = function() {
                var shakerData = $scope.getShakerData();

                if (angular.equals(shakerData, $scope.formerShakerData)) {
                    // nothing has changed, we don't have to do this.
                    return;
                }

                $scope.autoRefreshDirty = true;
                if ($scope.isRecipe){
                    if ($scope.shaker.exploreUIParams.autoRefresh && $scope.recipeOutputSchema) {
                        // Only call refreshTable when recipeOutputSchema is set.
                        // If it is not set yet, it will be set later when the DataikuAPI.datasets.get() call in recipe.js will succeed
                        // and refreshTable will called then.
                        $scope.refreshTable(false);
                    }
                } else {
                    if ($scope.shaker.exploreUIParams.autoRefresh) {
                        $scope.shakerHooks.saveForAuto();
                        $scope.refreshTable(false);
                    } else {
                        $scope.saveOnly();
                        $scope.autoRefreshDirty = true;
                        $scope.setFormerShakerData();
                    }
                }
            };

            $scope.$on("overrideTableUpdated", function(){
                $scope.autoSaveAutoRefresh();
            });

            function clearBackendErrors(step) {
                step.$stepState.change = null;
                step.$stepState.backendError = null;
                if (step.metaType == "GROUP") {
                    step.steps.forEach(clearBackendErrors);
                }
            }

            function mergeChanges(step, change) {
                step.$stepState.change = change;
                step.designTimeReport = change.recordedReport;
                if (step.metaType == "GROUP") {
                    step.steps.forEach(function(substep, i){
                        if (change.groupStepsChanges && change.groupStepsChanges[i]){
                            mergeChanges(substep, change.groupStepsChanges[i]);
                        } else {
                            substep.$stepState.change = null;
                            step.designTimeReport = null;
                        }
                    });
                }
            }
            function mergeBackendErrors(step, errHolder) {
                if (errHolder.error) {
                    step.$stepState.backendError = errHolder.error;
                } else {
                    step.$stepState.backendError = null;
                }
                if (step.metaType === "GROUP") {
                    step.steps.forEach(function(substep, i) {
                        if (errHolder.children && errHolder.children.length > i && errHolder.children[i] != null) {
                            mergeBackendErrors(substep, errHolder.children[i]);
                            step.$stepState.backendError = substep.$stepState.backendError;
                        } else {
                            substep.$stepState.backendError = null;
                        }
                    });
                }
            }

            $scope.onRefreshFutureDone = function(filtersOnly) {
                $scope.shakerState.runError = null;
                $scope.shakerState.initialRefreshDone = true;
                $scope.requestedSampleId = $scope.future.result.usedSampleId;
                $scope.invalidScriptError = {};

                $scope.shakerState.lockedHighlighting = [];
                $scope.shakerState.selectedRow = null;

                $scope.table = $scope.future.result;

                $scope.setSpinnerPosition(undefined);
                $scope.lastRefreshCallTime = (new Date().getTime()-$scope.refreshCallBeg);
                if ($scope.updateFacetData) {
                    $scope.updateFacetData();
                }

                $scope.shaker.columnsSelection = $scope.table.newColumnsSelection;

                $scope.shaker.steps.forEach(function(step, i){
                    if ($scope.table.scriptChange.groupStepsChanges[i] != null) {
                        mergeChanges(step, $scope.table.scriptChange.groupStepsChanges[i]);
                    }
                })

                $scope.shakerState.hasAnyComment = false;
                $scope.shakerState.hasAnyCustomFields = false;

                var getNoFakeExtremeDoubleDecimalPercentage = function(numerator, denominator) {
                    var result = numerator * 10000 / denominator;
                    switch (Math.round(result)) {
                        case 0:
                            result = result == 0 ? 0 : 1;
                            break;
                        case 10000:
                            result = result == 10000 ? 10000 : 9999;
                            break
                        default:
                            result = Math.round(result);
                    }
                    return result / 100;
                }

                $scope.columns = $.map($scope.table.headers, function(header) {
                    if (header.selectedType) {
                        header.selectedType.totalCount = (header.selectedType.nbOK + header.selectedType.nbNOK + header.selectedType.nbEmpty);
                        header.okPercentage = getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.nbOK, header.selectedType.totalCount);
                        header.emptyPercentage = !header.selectedType.nbEmpty ? 0 : getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.nbEmpty, header.selectedType.totalCount);
                        header.nonemptyPercentage = header.selectedType.nbEmpty == null ? 0 : getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.totalCount - header.selectedType.nbEmpty, header.selectedType.totalCount);
                        header.nokPercentage = !header.selectedType.nbNOK ? 0 : getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.nbNOK, header.selectedType.totalCount);

                        if (header.deletedMeaningName) {
                            header.meaningLabel = header.deletedMeaningName + ' (deleted)';
                        } else {
                            header.meaningLabel = $filter('meaningLabel')(header.selectedType.name);
                        }
                    }

                    /* Check if this column has a comment */
                    if (header.recipeSchemaColumn && header.recipeSchemaColumn.column.comment) {
                        $scope.shakerState.hasAnyComment = true;
                        header.comment = header.recipeSchemaColumn.column.comment
                    }
                    if (header.datasetSchemaColumn && header.datasetSchemaColumn.comment) {
                        $scope.shakerState.hasAnyComment = true;
                        header.comment = header.datasetSchemaColumn.comment
                    }
                    if ($scope.shaker.origin == "ANALYSIS" &&
                       $scope.shaker.analysisColumnData[header.name] &&
                       $scope.shaker.analysisColumnData[header.name].comment) {
                        $scope.shakerState.hasAnyComment = true;
                        header.comment = $scope.shaker.analysisColumnData[header.name].comment;
                    }

                    // This part propagates the column comment coming from the input dataset
                    if (header.recipeSchemaColumn?.column != null && !header.recipeSchemaColumn.column.isColumnEdited) {
                        const columns = $scope.dataset?.schema?.columns || null;
                        if (columns) {
                            const columnFromInput = columns.find(col => col.name === header.name);
                            const newComment = columnFromInput?.comment;
                            if (newComment && header.comment !== newComment) {
                                $scope.shakerState.hasAnyComment = true;
                                header.comment = newComment;
                                header.recipeSchemaColumn.column.comment = newComment;
                                $scope.schemaDirtiness.dirty = true;
                            }
                        }
                    }

                    /* Check if this column has preview custom fields */
                    function addCustomFieldsPreviews(customFields) {
                        const ret = [];
                        const customFieldsMap = $rootScope.appConfig.customFieldsMap['COLUMN'];
                        for (let i = 0; i < customFieldsMap.length; i++) {
                            const selectCFList = (customFieldsMap[i].customFields || []).filter(cf => cf.type == 'SELECT');
                            for (let j = 0; j < selectCFList.length; j++) {
                                const cfDef = selectCFList[j];
                                const value = (cfDef.selectChoices || []).find(choice => choice.value == (customFields && customFields[cfDef.name] || cfDef.defaultValue));
                                if (value && value.showInColumnPreview) {
                                    ret.push({definition: cfDef, value: value});
                                }
                            }
                        }
                        return ret;
                    }
                    $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap['COLUMN'];
                    if (header.recipeSchemaColumn) {
                        header.customFields = header.recipeSchemaColumn.column.customFields;
                    }
                    if (header.datasetSchemaColumn) {
                        header.customFields = header.datasetSchemaColumn.customFields;
                    }
                    if ($scope.shaker.origin == "ANALYSIS" &&
                        $scope.shaker.analysisColumnData[header.name]) {
                        header.customFields = $scope.shaker.analysisColumnData[header.name].customFields;
                    }
                    const cfPreviews = addCustomFieldsPreviews(header.customFields);
                    if (cfPreviews.length > 0) {
                        $scope.shakerState.hasAnyCustomFields = true;
                        header.customFieldsPreview = cfPreviews;
                    }

                    return header.name;
                });
                if ($scope.shakerState.activeView === 'table') {
                    $scope.setQuickColumns();
                    $scope.clearQuickColumnsCache();
                }
                if ($scope.isRecipe && $scope.table.newRecipeSchema) {
                    $scope.recipeOutputSchema = $scope.table.newRecipeSchema;
                }
                $scope.$broadcast("shakerTableChanged");
                $rootScope.$broadcast("shakerTableChangedGlobal", $scope.shaker);

                getDigestTime($scope, function(time) {
                    $scope.lastRefreshDigestTime = time;
                    updateColoringStates();
                    $scope.$broadcast("reflow");
                    WT1.event("shaker-table-refreshed", {
                        "activeFFs" : $scope.shaker.explorationFilters.length,
                        "backendTime" : $scope.lastRefreshCallTime,
                        "digestTime" : time,
                        "numCols" : $scope.table.headers.length,
                        "totalKeptRows" : $scope.table.totalKeptRows,
                        "totalRows" : $scope.table.totalRows
                    });
                });
            };
            $scope.onRefreshFutureFailed = function(data, status, headers) {
                $scope.shakerState.runError = null;
                $scope.shakerState.initialRefreshDone = true;
                $scope.setSpinnerPosition(undefined);
                if(data && data.hasResult && data.aborted) {
                    $rootScope.$broadcast('shakerTableChangeAbortedGlobal');
                    return; // Abortion is not an error
                }
                var apiErr = getErrorDetails(data, status, headers);
                $scope.shakerState.runError = apiErr;

                if (apiErr.errorType == "ApplicativeException" && apiErr.code == "STEP_RUN_EXCEPTION" && apiErr.payload) {
                    $scope.shaker.steps.forEach(function(step, i){
                        if (apiErr.payload.children[i] != null) {
                            mergeBackendErrors(step, apiErr.payload.children[i]);
                        }
                    })
                }
                $rootScope.$broadcast('shakerTableChangeFailedGlobal');
            };

            $scope.showWarningsDetails = function(){
                CreateModalFromTemplate("/templates/widgets/warnings-details.html", $scope, null, function($newScope) {
                    if ($scope.table && $scope.table.warnings && $scope.table.warnings.warnings) {
                        $newScope.warnings = $scope.table.warnings.warnings
                    }
                });
            }

            $scope.markSoftDisabled = function(){
                function _mark(s, isAfterPreview) {
                    if (isAfterPreview) {
                        s.$stepState.softDisabled = true;
                    }
                    if (s.metaType == "GROUP") {
                        if (s.steps) {
                            for (var i = 0; i < s.steps.length; i++) {
                                isAfterPreview = _mark(s.steps[i], isAfterPreview);
                            }
                        }
                    }
                    if (s.preview) {
                        $scope.stepBeingPreviewed = s;
                        return true;
                    }
                    return isAfterPreview
                }
                $scope.stepBeingPreviewed = null;
                var isAfterPreview = false;
                for (var i = 0; i < $scope.shaker.steps.length; i++) {
                    isAfterPreview = _mark($scope.shaker.steps[i], isAfterPreview);
                }
            }

            $scope.hasAnySoftDisabled = function(){
                var hasAny = false;
                function _visit(s) {
                    if (s.metaType == "GROUP") {
                        s.steps.forEach(_visit);
                    }
                    if (s.$stepState.softDisabled) hasAny = true;
                }
                $scope.shaker.steps.forEach(_visit);
                return hasAny;
            }

            // Make sure that every step after a preview is marked soft-disabled
            $scope.fixPreview = function() {
                $scope.markSoftDisabled();
                // var disable = false;
                // for (var i = 0; i < $scope.shaker.steps.length; i++) {
                //     if(disable) {
                //         $scope.shaker.steps[i].disabled = true;
                //     }
                //     if($scope.shaker.steps[i].preview) {
                //         disable=true;
                //     }
                // }
                // #2459
                if ($scope.dataset && $scope.dataset.partitioning && $scope.dataset.partitioning.dimensions){
                    if (!$scope.dataset.partitioning.dimensions.length && $scope.shaker.explorationSampling.selection.partitionSelectionMethod != "ALL") {
                        Logger.warn("Partition-based sampling requested on non partitioned dataset. Force non-partitioned sample.")
                        $scope.shaker.explorationSampling.selection.partitionSelectionMethod = "ALL";
                        delete $scope.shaker.explorationSampling.selection.selectedPartitions;
                    }
                }

            };

            /**
            * Refreshes the whole table
            * Set "filtersOnly" to true if this refresh is only for a change of filters / facets
            */
            $scope.refreshTable = function(filtersOnly) {
                const refreshArguments = arguments;
                return CachedAPICalls.processorsLibrary.success(function(){
                    if ($scope.validateScript){
                        if (!$scope.validateScript()) {
                            Logger.info("Aborted refresh: script is invalid");
                            ActivityIndicator.error("Not refreshing: script is invalid !");
                            return;
                        }
                    }
                    lastValidatedSteps = angular.copy($scope.shaker.steps);
                    // Mark the table as longer relevant until the new & refreshed table is displayed.
                    // This allows to stop displaying some info about the table while it is being refreshed.
                    if ($scope.table) {
                        $scope.table.$invalidated = true;
                    }
                    $scope.refreshTable_.apply(this, refreshArguments);
                });
            };

            const refreshDebounce = Debounce();
            $scope.refreshTable_ = refreshDebounce
                .withDelay(200, 500) // delay should be long enough to debounce dataset preview (single-clicking a dataset) when the user wants to explore a dataset (double-clicking a dataset)
                .withSpinner(!$scope.refreshNoSpinner)
                .withScope($scope)
                .wrap(function(filtersOnly){
                    if (!angular.isDefined(filtersOnly)) throw new Error();

                    // Because of the delay, it might happen that the parameters from the steps changed
                    // between $scope.validateScript() and $scope.refreshTable_().
                    // That leads to the fact that the steps could not be validated anymore when sending data to the server.
                    // In order to avoid this problem, we check that the validated data in steps did not change during the delay
                    if (!angular.equals(lastValidatedSteps, $scope.shaker.steps)) {
                        return;
                    }
                    $scope.fixPreview();
                    var filterRequest = $scope.buildFilterRequest ? $scope.buildFilterRequest($scope.shaker.explorationFilters) : [];
                    $scope.setFormerShakerData();

                    $scope.shaker.steps.forEach(clearBackendErrors);

                    $scope.$broadcast("scrollToLine", 0);

                    $scope.refreshCallBeg  = new Date().getTime();
                    $scope.future = null;

                    $scope.shakerHooks.onTableRefresh();
                    // Offer a chance to the right panel to be informed of the new state of the shaker
                    $rootScope.$broadcast("shakerTableRefresh", $scope.shaker);

                    $scope.shakerHooks.getRefreshTablePromise(filtersOnly, {"elements": filterRequest})
                    .update(function(future) {
                        $scope.autoRefreshDirty = true;
                        $scope.future = future;
                    }).success(function(future) {
                        $scope.autoRefreshDirty = false;
                        Logger.info("Got table data");
                        $scope.future = future;
                        $scope.onRefreshFutureDone(filtersOnly);
                        $scope.shakerHooks.afterTableRefresh();
                    }).error(function(data,status,headers) {
                        $scope.future = null;
                        $scope.onRefreshFutureFailed(data,status,headers);
                        
                        if ($scope.setDashboardTileError) {
                            $scope.setDashboardTileError(data, status, headers);
                        }

                        $scope.shakerHooks.afterTableRefresh();
                    });
            });

            /**
             * Checks weather there is a pending debounced refresh. and if the MonoFuturizedRefresh has an empty refresh queue
             */
            $scope.allRefreshesDone = function() {
                return !refreshDebounce.active() && !$scope.shakerHooks.isMonoFuturizedRefreshActive();
            };

            /**
            * Waits for all RefreshTable calls to be resolved. Returns a promise.
            */
            $scope.waitAllRefreshesDone = function () {
                const deferred = $q.defer();
                const inter = setInterval(
                    function () {
                        if ($scope.allRefreshesDone()) {
                            clearInterval(inter);
                            deferred.resolve();
                        }
                    }, 25);
                return deferred.promise;
            }

            /**
            * Fetches a chunk of the table. Returns a promise.
            * Out of bound is NOT handled, and will throw.
            */
            $scope.getTableChunk = function(firstRow, nbRows, firstCol, nbCols) {
                var deferred = $q.defer();
                var filterRequest = $scope.buildFilterRequest ? $scope.buildFilterRequest($scope.shaker.explorationFilters) : [];
                $scope.shakerHooks.getTableChunk(firstRow, nbRows, firstCol, nbCols,
                        {"elements":filterRequest})
                .then(({data}) => deferred.resolve(data))
                .catch(err => {
                    deferred.reject();
                    setErrorInScope.bind($scope)(err)
                });
                return deferred.promise;
            };

            $scope.analyseColumn = function(column, columns) {
                CreateModalFromTemplate("/templates/shaker/analyse-box.html",
                    $scope, "ColumnAnalysisController", function(newScope) {
                        newScope.setColumn(column, columns || $scope.table.headers);
                    }, "analyse-box");
            };

            $scope.copyColumnName = function(column) {
                if (!column?.name) {
                    return;
                }
                ClipboardUtils.copyToClipboard(column?.name, translate("SHAKER.HEADER.CONTEXTUAL_MENU.COPY_COLUMN_NAME.NOTIF", `Copied "${column?.name}" to clipboard`, {column: column?.name}));
            }

            $scope.editColumnDetails = function(column) {
                CreateModalFromTemplate("/templates/shaker/modals/shaker-edit-column.html",
                    $scope, null, function(newScope) {
                        newScope.setColumn(column);
                    });
            }

            $scope.hideColumn = function(columnName) {
                if (columnName) {
                    if ($scope.shaker.columnsSelection && $scope.shaker.columnsSelection.mode === "SELECT" && $scope.shaker.columnsSelection.list) {
                        let displayedColumn = $scope.shaker.columnsSelection.list.find(column => column.name === columnName);
                        if (displayedColumn) {
                            displayedColumn.d = false;
                        } else {
                            $scope.shaker.columnsSelection.list.push({name: columnName, d:false});
                        }
                    } else {
                        $scope.shaker.columnsSelection = {
                            mode: "SELECT",
                            list: $scope.columns.map(colName => ({ name: colName, d: (colName !== columnName) }))
                        }
                    }
                }
                $scope.autoSaveForceRefresh();
            }

            $scope.scrollToColumn = $scope.$broadcast.bind($scope, 'scrollToColumn'); // broadcast to child fattable
            $scope.$watch('shakerState.activeView', function(nv) {
                if ($scope.shakerState.activeView === 'table') {
                    $scope.setQuickColumns();
                }
            });
            $scope.setQuickColumns = function(qc) {
                $scope.quickColumns = qc || ($scope.table && $scope.table.headers || []);
            };
            $scope.clearQuickColumnsCache = function () {
                $scope.quickColumnsCache = {};
            };
            $scope.quickColumns = [];
            $scope.quickColumnsCache = {};

            /**
             * Returns true if coloring.scheme is MEANING_AND_STATUS otherwise false
             */
            $scope.isColoringSchemeByMeaning = function () {
                return (
                    $scope.shaker && $scope.shaker.coloring && $scope.shaker.coloring.scheme === "MEANING_AND_STATUS"
                );
            };
            /**
             * Returns true if coloring.scheme is ALL_COLUMNS_VALUES otherwise false
             */
            $scope.isColoringSchemeByScale = function () {
                return (
                    $scope.shaker && $scope.shaker.coloring && $scope.shaker.coloring.scheme === "ALL_COLUMNS_VALUES"
                );
            };
            /**
             * Returns true if coloring.scheme is COLORING_GROUPS or legacy (XXX_COLUMNS_RULES or XXX_COLUMNS_VALUES) otherwise false
             */
            $scope.isColoringSchemeByRules = function () {
                return (
                    $scope.shaker &&
                    $scope.shaker.coloring &&
                    ($scope.shaker.coloring.scheme === "COLORING_GROUPS" ||
                        $scope.shaker.coloring.scheme === "INDIVIDUAL_COLUMNS_RULES" ||
                        $scope.shaker.coloring.scheme === "SINGLE_COLUMN_RULES" ||
                        $scope.shaker.coloring.scheme === "INDIVIDUAL_COLUMNS_VALUES" ||
                        $scope.shaker.coloring.scheme === "SINGLE_COLUMN_VALUES")
                );
            };

            function getColoringModeString(visible, text) {
                return (
                    '<i class="' +
                    (visible ? "dku-icon-checkmark-12 mright4" : "mright16") +
                    '"></i><a>' +
                    text +
                    "</a>"
                );
            }

            /**
             * Updates the coloring display string applied and the Conditional Formatting pane
             */
            function updateColoringStates() {
                let isColoringSchemeByMeaning = false;
                let isColoringSchemeByRules = false;
                let isColoringSchemeByScale = false;

                if ($scope.shaker && $scope.shaker.coloring) {
                    if ($scope.isColoringSchemeByScale()) {
                        isColoringSchemeByScale = true;
                        $scope.coloredByText = translate("FLOW.DATASET.DISPLAY_MENU.COLORED_BY.SCALE", "scale");
                    } else if ($scope.isColoringSchemeByRules()) {
                        const table = $scope.table;
                        isColoringSchemeByRules = true;
                        $scope.conditionalFormattingState.columns = $scope.shaker.columnsSelection.list;
                        $scope.conditionalFormattingState.isWholeData = table.sampleMetadata && table.sampleMetadata.sampleIsWholeDataset;
                        $scope.conditionalFormattingState.tableHeaders = table.headers;
                        $scope.setColoredByRules();
                    } else {
                        // Nothing to display for default
                        isColoringSchemeByMeaning = true;
                        $scope.coloredByText = "";
                    }
                } else {
                    // Nothing to display for default
                    isColoringSchemeByMeaning = true;
                    $scope.coloredByText = "";
                }

                // Updates DISPLAY menus
                $scope.displayColoringMeaning = getColoringModeString(
                    isColoringSchemeByMeaning,
                    translate("FLOW.DATASET.DISPLAY_MENU.COLOR.BY_MEANING_VALIDITY", "Meaning validity (all columns)")
                );
                $scope.displayColoringRules = getColoringModeString(
                    isColoringSchemeByRules,
                    translate("FLOW.DATASET.DISPLAY_MENU.COLOR.BY_RULES", "Conditional formatting (can be slow)")
                );
                $scope.displayColoringScale = getColoringModeString(
                    isColoringSchemeByScale,
                    translate("FLOW.DATASET.DISPLAY_MENU.COLOR.SCALE", "Color scale (all columns - can be slow)")
                );
            }

            /**
             * Sets the coloring display string by rules
             */
            $scope.setColoredByRules = function () {
                $scope.coloredByText = translate('FLOW.DATASET.DISPLAY_MENU.COLORED_BY.RULES', 'rules');
            };

            /**
             * Closes the right panel if the Conditional formatting panel is opened
             */
            $scope.closeRightPaneIfConditionalFormattingOpened = function () {
                if ($scope.isQuickConditionalFormattingViewOpened() && typeof $scope.closeRightPane === "function") {
                    $scope.shakerState.rightPaneView = $scope.RIGHT_PANE_VIEW.NONE;
                    $scope.closeRightPane();
                }
            };

            $scope.openConditionalFormattingFromColumnHeader = function (columnName) {
                $scope.conditionalFormattingState.openedFromColumnHeader = columnName;
                $scope.openRightPaneWith($scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW);
            };

            /**
             * Switches to "display color by meaning on all columns" with the table coloring scheme MEANING_AND_STATUS.
             */
            $scope.onDisplayAllColumnsColorByMeaning = function () {
                if (!$scope.shaker || !$scope.shaker.coloring || $scope.isColoringSchemeByMeaning()) {
                    return;
                }

                WT1.event("dataset-change-display-mode", { displayMode: "meaning", appliesTo: "all-columns" });

                ConditionalFormattingEditorService.prepareMigrationToColoringGroups(
                    $scope.shaker.coloring,
                    $scope.conditionalFormattingState.tableHeaders
                );

                // No migration, only updating the table coloring scheme
                $scope.shaker.coloring.scheme = "MEANING_AND_STATUS";
                updateColoringStates();

                $scope.autoSaveForceRefresh();
            };

            /**
             * Switches to "display color by rules on all columns" with the table coloring scheme COLORING_GROUPS. 
             * Opens the Conditional formatting right pane if the user has the write access 
             * or migrates to COLORING_GROUPS scheme only.
             */
            $scope.onDisplayAllColumnsColorByRules = function () {
                if (!$scope.shaker || !$scope.shaker.coloring) return;

                if (!$scope.isColoringSchemeByRules()) {
                    WT1.event("dataset-change-display-mode", { displayMode: "rules", appliesTo: "all-columns" });
                }

                if (
                    $scope.shakerState.writeAccess &&
                    $scope.table &&
                    $scope.table.headers &&
                    $scope.table.headers.length
                ) {
                    $scope.openRightPaneWith($scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW);
                } else {
                    if (
                        ConditionalFormattingEditorService.migrateToColoringGroups(
                            $scope.shaker.coloring,
                            $scope.conditionalFormattingState.tableHeaders
                        )
                    ) {
                        $scope.setColoredByRules();
                        $scope.autoSaveForceRefresh();
                    }
                }
            };

            /**
             * Switches to "display color scale on all columns" with the table coloring scheme ALL_COLUMNS_VALUES.
             */
            $scope.onDisplayAllColumnsColorScale = function () {
                if (!$scope.shaker || !$scope.shaker.coloring || $scope.isColoringSchemeByScale()) {
                    return;
                }

                WT1.event("dataset-change-display-mode", { displayMode: "colorscale", appliesTo: "all-columns" });

                ConditionalFormattingEditorService.prepareMigrationToColoringGroups(
                    $scope.shaker.coloring,
                    $scope.conditionalFormattingState.tableHeaders
                );

                // No migration, only updating the table coloring scheme
                $scope.shaker.coloring.scheme = "ALL_COLUMNS_VALUES";
                updateColoringStates();

                $scope.autoSaveForceRefresh();
            };

            $scope.toggleScientificNotation = function(columnName) {
                $scope.shaker.columnUseScientificNotationByName[columnName] = $scope.shaker.columnUseScientificNotationByName[columnName] === true ? false : true;
                $scope.autoSaveForceRefresh();
            }
            $scope.sortDirection = function(column) {
                var sortElem = ($scope.shaker.sorting || []).filter(function(e) {return e.column == column;})[0];
                return sortElem == null ? null : sortElem.ascending;
            };
            $scope.toggleSort = function(column) {
                if ($scope.shaker.sorting == null) {
                    $scope.shaker.sorting = [];
                }
                var sorting = $scope.shaker.sorting;
                if (sorting.length == 1 && sorting[0].column == column) {
                    sorting[0].ascending = !sorting[0].ascending;
                } else {
                    $scope.shaker.sorting = [{column:column, ascending:true}];
                }
                $scope.autoSaveForceRefresh();
            }
            $scope.addSort = function(column) {
                if ($scope.shaker.sorting == null) {
                    $scope.shaker.sorting = [];
                }
                var sorting = $scope.shaker.sorting;
                var matching = sorting.filter(function(s) {return s.column == column;});
                if (matching.length > 0) {
                    matching[0].ascending = !matching[0].ascending;
                } else {
                    $scope.shaker.sorting.push({column:column, ascending:true});
                }
                $scope.autoSaveForceRefresh();
            }

            // Callback called when dropping a column while reordering (see fatDraggable directive) - for the explore view
            $scope.reorderColumnCallback = function(draggedColumn, hoveredColumn, columnName, referenceColumnName) {

                let movement = {};

                let columnOldPosition = $scope.columns.indexOf(columnName);
                let columnNewPosition = $scope.columns.indexOf(referenceColumnName);

                if (columnOldPosition < 0 || columnNewPosition < 0) {
                    return;
                }

                if (columnNewPosition === 0) {
                    movement.reorderAction = "AT_START";
                } else if (columnNewPosition === $scope.columns.length - 1) {
                    movement.reorderAction = "AT_END";
                } else if (columnOldPosition > columnNewPosition) {
                    movement.reorderAction = "BEFORE_COLUMN";
                } else {
                    movement.reorderAction = "AFTER_COLUMN";
                }

                movement.movedColumn = $scope.columns[columnOldPosition];

                if (movement.reorderAction === "BEFORE_COLUMN" || movement.reorderAction === "AFTER_COLUMN") {
                    movement.referenceColumn = $scope.columns[columnNewPosition];
                }

                if ($scope.shaker) {
                   if (!$scope.shaker.columnOrder) {
                       $scope.shaker.columnOrder = [movement];
                   } else {
                       $scope.shaker.columnOrder.push(movement);
                   }
                }

                $scope.autoSaveForceRefresh();
            };

            $scope.openColumnsSelectionModal = function(){
                CreateModalFromTemplate("/templates/shaker/select-columns-modal.html", $scope);
            }
            $scope.openSortSelectionModal = function(){
                CreateModalFromTemplate("/templates/shaker/select-sort-modal.html", $scope);
            }
            $scope.clearSort = function(column) {
                if (column && $scope.shaker.sorting) {
                    var sorting = $scope.shaker.sorting;
                    var matching = sorting.filter(function(s) {return s.column == column;});
                    if (matching.length > 0) {
                        sorting.splice(sorting.indexOf(matching[0]), 1);
                    }
                } else {
                    $scope.shaker.sorting = [];
                }
                $scope.autoSaveForceRefresh();
            }

            $scope.clearResize = function() {
                const minColumnWidth = 100;
                $scope.shaker.columnWidthsByName = computeColumnWidths($scope.table.initialChunk, $scope.table.headers, minColumnWidth, $scope.hasAnyFilterOnColumn, $scope.shaker.columnWidthsByName, $scope.shaker.columnUseScientificNotationByName, true)[1];
                $scope.autoSaveAutoRefresh();
            }

            $scope.clearColumnOrder = function() {
                $scope.shaker.columnOrder = [];
                $scope.autoSaveAutoRefresh();
            }

            this.$scope = $scope; // fugly
    
            $scope.$on('datasetSchemaChanged', (event, data) => {
                $scope.refreshTable(false);
            });
        }
    }
});

/**
 * Finds the coloring group ColoringGroup that is applied to a column.
 * 
 * Iterates over the list of coloring groups (reverse order, like a stack) to find the coloring group
 * which is enabled and targets the given column.
 * If no coloring group is found for the column, returns null.
 * 
 * If optional filters `{ filterScope, filterScheme }` are provided, returns the first coloring group targeting 
 * the column which has the provided scope and/or scheme.
 */
app.service("findAppliedColoringGroup", function () {
    return function (coloringGroups, columnName, { filterScope, filterScheme } = { filterScope: null, filterScheme: null }) {
        for (let i = coloringGroups.length - 1; i >= 0; i--) {
            const coloringGroup = coloringGroups[i];
            if (!coloringGroup.enabled) {
                continue;
            }

            // Skips coloring groups that do not respect the optional filters on scope and/or scheme
            if ((filterScope && filterScope != coloringGroup.scope) || (filterScheme && filterScheme != coloringGroup.scheme)) {
                continue;
            }

            // Targeted by an "all columns" logic
            if (coloringGroup.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN") {
                return coloringGroup;
            }

            // Specifically targeted
            if (coloringGroup.scope === "COLUMNS") {
                const isTargeted = coloringGroup.targetedColumnNames.some(targeted => columnName === targeted);
                if (isTargeted) {
                    return coloringGroup;
                }
            }
        }

        return null;
    };
});

app.directive('quickColumnsView', function(DataikuAPI, Fn, Debounce, MonoFuture, $filter, $stateParams) {
    var COLUMN_CHUNK = 50,
        dateFmt = Fn(function(d){ return new Date(d); }, d3.time.format('%Y-%m-%d %H:%M')),
        numFmt = $filter('smartNumber');
    return {
        scope: true,
        require: '^shakerExploreBase',
        templateUrl: '/templates/shaker/quick-columns-view.html',
        link: function(scope, element, attrs, exploreCtrl) {
            var monoLoad = [];
            scope.$watch('shakerState.rightPaneView', function(newVal) {
                if (newVal === scope.RIGHT_PANE_VIEW.QUICK_COLUMNS_VIEW) {
                    scope.openRightPane();
                }
            });
            scope.onClose = function() {
                scope.shakerState.rightPaneView = scope.RIGHT_PANE_VIEW.NONE;
                scope.closeRightPane();
            }
            scope.initColumnScope = function(cScope) {
                if (!cScope.col) return;
                cScope.activateColBar = scope.activateBar.bind(null, cScope);
            };
            scope.quickColumnsChanged = function() {
                scope.quickColumnsFilterChanged(scope.quickColumnsFilter);
            };
            scope.quickColumnCacheCleared = function() {
                monoLoad.forEach(function(m){ if (m.running) m.abort(); });
            };

            scope.quickColumnsFilter = '';
            scope.quickColumnsFilterChanged = function(newVal, oldVal) {
                // $watch triggers this listener a first time to initialize the watcher.
                // But it needs to be stopped to avoid a first call to quickColumnCacheCleared
                // which aborts our initial running mono futures (in particular multiColumnAnalysis).
                if (newVal === oldVal) {
                    return;
                } 

                scope.quickColumnCacheCleared();
                scope.quickColumnsFiltered = !newVal ? scope.quickColumns : scope.quickColumns.filter(
                    function(c) { return c.name.toLowerCase().indexOf(this) >= 0; }, newVal.toLowerCase());
                // append MonoFuture at will
                for (var i = monoLoad.length; i < Math.ceil(scope.quickColumnsFiltered.length / COLUMN_CHUNK); i++) {
                    monoLoad[i] = MonoFuture(scope);
                }
            };
            // Can’t use PagedAsyncTableModel because of divergent invalidation policy:
            // cache is kept when closing QCV or filtering columns,
            // but reset when editing shaker steps
            scope.tableModel = function() {
                var model = new fattable.TableModel();
                model.hasCell = Fn.cst(true); // always drawable
                model.getCell = function(i, j, cb) {
                    if (scope.shakerState.rightPaneView !== scope.RIGHT_PANE_VIEW.QUICK_COLUMNS_VIEW) return;
                    var page = Math.floor(i / COLUMN_CHUNK);
                    // Initiate block fetch...
                    loadQuickColumns(page, cb);
                    // ...but render immediately (name, type, validity)
                    cb(scope.quickColumnsFiltered[i]);
                };
                return model;
            };
            function loadQuickColumns(page, cb) {
                if (monoLoad[page].running) return;
                var uncached = scope.quickColumnsFiltered
                    .slice(page * COLUMN_CHUNK, (page + 1) * COLUMN_CHUNK)
                    .map(Fn.prop('name'))
                    .filter(Fn.not(Fn.dict(scope.quickColumnsCache)));
                if (!uncached.length) return;
                monoLoad[page].running = true;
                monoLoad[page].exec(
                    DataikuAPI.shakers.multiColumnAnalysis(
                        $stateParams.projectKey,
                        scope.inputDatasetProjectKey, scope.inputDatasetName, scope.inputStreamingEndpointId,
                        scope.shakerHooks.shakerForQuery(),
                        scope.requestedSampleId, uncached, '*', 40))
                .success(function(data){
                    monoLoad[page].running = false;
                    if (!data.hasResult) return;
                    data = data.result;
                    // Update quick column using data from scope table
                    scope.setQuickColumns();
                    for (var k in data) {
                        if (data[k].facets) {
                            scope.quickColumnsCache[k] = {
                                values: data[k].facets.counts,
                                labels: data[k].facets.values
                            };
                        } else {
                            scope.quickColumnsCache[k] = { values: data[k].histogram };
                            var col = scope.quickColumns.filter(Fn(Fn.prop("name"), Fn.eq(k)))[0],
                                fmt = col && col.selectedType && ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(col.selectedType.name) >= 0 ? dateFmt : numFmt;
                            scope.quickColumnsCache[k].labels =
                              data[k].histogramLowerBounds.map(fmt).map(
                                function(lb, i) { return lb + " - " + this[i]; },
                                data[k].histogramUpperBounds.map(fmt))
                        }
                    }
                }).error(function() {
                    monoLoad[page].running = false;
                    setErrorInScope.apply(exploreCtrl.$scope, arguments);
                });
            }
            scope.$watch('quickColumnsCache', function(newVal, oldVal) {
                scope.quickColumnCacheCleared();
            });
            scope.$watch('quickColumns', function(newVal, oldVal) {
                scope.quickColumnsChanged();
            }, true);
            scope.$watch('quickColumnsFilter',
                function(newVal,oldVal) {
                    Debounce().withDelay(150,300).withScope(scope).wrap(scope.quickColumnsFilterChanged)(newVal, oldVal);
                });
            scope.activateBar = function(colScope, value, i) {
                colScope.setLabels(value !== null ? {
                    pop: value.toFixed(0),
                    label: scope.quickColumnsCache[colScope.col.name].labels[i],
                    part: (colScope.col.selectedType ? (value * 100 / colScope.col.selectedType.totalCount).toFixed(1) + '&nbsp;%' : '')
                } : null);
            };
            scope.defaultAction = !scope.scrollToColumn ? null :
                function(column) { scope.scrollToColumn(column.name); };
        }
    };
});

app.directive('quickConditionalFormattingView', function($q, $stateParams, DataikuAPI, WT1, Logger) {
    return {
        scope: true,
        require: '^shakerExploreBase',
        templateUrl: '/templates/shaker/quick-conditional-formatting-view.html',
        link: function(scope) {
            scope.$watch("shakerState.rightPaneView", function (newVal) {
                if (newVal === scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW) {
                    const coloring = scope.shaker.coloring;
                    const nbRules = coloring ? (coloring.coloringGroups ? coloring.coloringGroups.length : 0) : 0;
                    WT1.event("conditional-formatting-pane-open", {
                        nbRules: nbRules,
                    });

                    scope.setColoredByRules();
                    scope.conditionalFormattingState.columns = scope.$parent.shaker.columnsSelection.list;
                    scope.conditionalFormattingState.tableHeaders = scope.$parent.table.headers;
                    scope.openRightPane();
                }
            });

            scope.onClose = function () {
                scope.shakerState.rightPaneView = scope.RIGHT_PANE_VIEW.NONE;
                scope.closeRightPane();
            };

            scope.saveAndRefresh = function () {
                scope.autoSaveForceRefresh();
            };

            // Returns a promise which resolves when saving the explore config is done
            scope.savePromise = function () {
                // Same checks done in autoSaveForceRefresh
                if (!scope.isRecipe && (scope.canWriteProject() || scope.isLocalSave)) {
                    return scope.shakerHooks.saveForAuto();
                }

                // Returns an empty promise
                return $q.when();
            };

            // Returns a promise
            scope.setScaleMinMaxFromAnalysis = function (columnName, colorScaleDef) {
                if (columnName && colorScaleDef) {
                    DataikuAPI.shakers
                        .detailedColumnAnalysis(
                            $stateParams.projectKey,
                            scope.inputDatasetProjectKey,
                            scope.inputDatasetName,
                            scope.shakerHooks.shakerForQuery(),
                            scope.requestedSampleId,
                            columnName,
                            0,
                            null,
                            false
                        )
                        .success(function (data) {
                            if (data.hasOwnProperty("numericalAnalysis")) {
                                colorScaleDef.max = data.numericalAnalysis.max;
                                colorScaleDef.min = data.numericalAnalysis.min;
                            }
                        })
                        .error(function (a, b, c) {
                            Logger.error("setScaleMinMaxFromAnalysis error", a, b, c);
                        });
                }
            };
        },
    };
});

/**
 * Base directive for all instances where a shaker table is made on a dataset
 * (explore, analysis script, prepare recipe).
 * (Counter examples: predicted data)
 */
app.directive("shakerOnDataset", function() {
    return {
        priority: 50,
        scope: true,
        controller  : function ($rootScope, $scope, $state, $stateParams, DataikuAPI, MonoFuture) {
            const monoFuture = MonoFuture($scope);
            const monoFuturizedRefresh = monoFuture.wrap(DataikuAPI.shakers.refreshTable);

            $scope.shakerState.onDataset = true;

            $scope.shakerHooks.isMonoFuturizedRefreshActive = monoFuture.active;

            $scope.shakerHooks.shakerForQuery = function(){
                var queryObj = angular.copy($scope.shaker);
                if ($scope.isRecipe) {
                    queryObj.recipeSchema = $scope.recipeOutputSchema;
                }
                queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
                return queryObj;
            }

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                $scope.shaker.columnWidthsByName[name] = width;
                $scope.autoSaveAutoRefresh();
            };

            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                const ret = monoFuturizedRefresh($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                    $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, filtersOnly, filterRequest, false);

                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            /**
            * Fetches a chunk of the table. Returns a promise.
            * Out of bound is NOT handled, and will throw.
            */
            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.getTableChunk(
                    $stateParams.projectKey,
                    $scope.inputDatasetProjectKey,
                    $scope.inputDatasetName,
                    $scope.shakerHooks.shakerForQuery(),
                    $scope.requestedSampleId,
                    firstRow,
                    nbRows,
                    firstCol,
                    nbCols,
                    filterRequest);
            }

            $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
                DataikuAPI.shakers.detailedColumnAnalysis($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics).success(function(data){
                            setAnalysis(data);
                }).error(function(a, b, c) {
                    if (handleError) {
                        handleError(a, b, c);
                    }
                    setErrorInScope.bind($scope)(a, b, c);
                });
            };
            $scope.shakerHooks.fetchClusters = function(setClusters, columnName, setBased, radius, timeOut, blockSize) {
                DataikuAPI.shakers.getClusters($stateParams.projectKey,
                    $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId,
                        columnName, setBased, radius, timeOut, blockSize
                    ).success(function(data) {
                        setClusters(data);
                    }).error(setErrorInScope.bind($scope));
            };
            $scope.shakerHooks.fetchTextAnalysis = function(setTextAnalysis, columnName, textSettings) {
                DataikuAPI.shakers.textAnalysis(
                        $stateParams.projectKey,
                        $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId,
                        columnName, textSettings)
                    .success(function(data){setTextAnalysis(data);})
                    .error(setErrorInScope.bind($scope));
            };

        }
    }
});

app.directive("shakerForPreview", function(Assert, DataikuAPI, DatasetErrorCta, MonoFuture) {
    return {
        scope: true,
        controller: function ($scope) {
            Assert.inScope($scope, 'shakerHooks');
            $scope.setSpinnerPosition = () => {};
            const monoFuture = MonoFuture($scope, !$scope.refreshNoSpinner);
            const monoFuturizedRefresh = monoFuture.wrap(DataikuAPI.shakers.refreshTable);
            function isDefined(variable) {
                return !(variable === null || variable === undefined);
            }
            $scope.shaker = {
                steps: [],
                explorationFilters: [],
                explorationSampling: {
                    selection: {
                        samplingMethod: "HEAD_SEQUENTIAL",
                        maxRecords: 50,
                        maxStoredBytes: 1 * 50 * 1024 * 1024 // i.e. max 1 MB RAM per record
                    }
                },
                coloring: {
                    scheme: isDefined($scope.shakerColoringScheme) ? $scope.shakerColoringScheme : "MEANING_AND_STATUS"
                },
                origin: isDefined($scope.shakerOrigin) ? $scope.shakerOrigin : "DATASET_EXPLORE",
                contextProjectKey: $scope.contextProjectKey ? $scope.contextProjectKey : $scope.projectKey,
                $headerOptions: {
                    showName: isDefined($scope.showName) ? $scope.showName : true,
                    showMeaning: isDefined($scope.showMeaning) ? $scope.showMeaning : true,
                    showStorageType: isDefined($scope.showStorageType) ? $scope.showStorageType : true,
                    showDescription: isDefined($scope.showDescription) ? $scope.showDescription : true,
                    showCustomFields: isDefined($scope.showCustomFields) ? $scope.showCustomFields : true,
                    showProgressBar: isDefined($scope.showProgressBar) ? $scope.showProgressBar : true,
                    showHeaderSeparator: isDefined($scope.showHeaderSeparator) ? $scope.showHeaderSeparator : false,
                    disableHeaderMenu: isDefined($scope.disableHeaderMenu) ? $scope.disableHeaderMenu : true,
                }
            };

            /** Error Management **/
            $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);
            $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError), true);

            /** Refresh table **/
            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.getTableChunk(
                    $scope.contextProjectKey ? $scope.contextProjectKey : null,
                    $scope.projectKey,
                    $scope.datasetName,
                    $scope.shaker,
                    $scope.requestedSampleId,
                    firstRow,
                    nbRows,
                    firstCol,
                    nbCols,
                    filterRequest);
            }

            refreshPreviewTable(true);

            $scope.$on('refresh-preview-table', () => {
                refreshPreviewTable(true);
            });

            $scope.$on('refresh-preview-table-without-cache', () => {
                refreshPreviewTable(false);
            });

            $scope.$on("shakerTableChangedGlobal", () => {
                $scope.$root.$broadcast("refresh-preview-table-done", $scope.projectKey, $scope.datasetName)
            });

            function refreshPreviewTable(useCache) {
                $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                    // Either do not send any sample ID (so the backend computes it itself from the sample settings), or send a dummy one to force cache invalidation
                    const requestedSampleId = useCache ? null : -1;
                    const contextProjectKey = $scope.contextProjectKey ? $scope.contextProjectKey : null;
                    var ret = monoFuturizedRefresh(contextProjectKey, $scope.projectKey, $scope.datasetName, $scope.shaker, requestedSampleId, filtersOnly, null, true);
                    return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
                };
                $scope.refreshTable(false);
            }
        }
    }
});

/**
 * Base directive for all instances where a shaker table is made on a streaming endpoint
 */
app.directive("shakerOnStreamingEndpoint", function() {
    return {
        scope: true,
        controller  : function ($scope, $state, $stateParams, DataikuAPI, MonoFuture, WT1) {
            const monoFuture = MonoFuture($scope);
            const monoFuturizedRefresh = monoFuture.wrap(DataikuAPI.shakers.refreshCapture);

            $scope.shakerState.onDataset = false;

            $scope.shakerHooks.isMonoFuturizedRefreshActive = monoFuturizedRefresh.active;

            $scope.shakerHooks.shakerForQuery = function(){
                var queryObj = angular.copy($scope.shaker);
                if ($scope.isRecipe) {
                    queryObj.recipeSchema = $scope.recipeOutputSchema;
                }
                queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
                return queryObj;
            }

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                $scope.shaker.columnWidthsByName[name] = width;
                $scope.autoSaveAutoRefresh();
            };

            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                WT1.event("streaming-refresh-explore")

                var ret = monoFuturizedRefresh($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputStreamingEndpointId,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, filtersOnly, filterRequest);

                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            /**
            * Fetches a chunk of the table. Returns a promise.
            * Out of bound is NOT handled, and will throw.
            */
            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.getCaptureChunk(
                    $stateParams.projectKey,
                    $scope.inputDatasetProjectKey,
                    $scope.inputStreamingEndpointId,
                    $scope.shakerHooks.shakerForQuery(),
                    $scope.requestedSampleId,
                    firstRow,
                    nbRows,
                    firstCol,
                    nbCols,
                    filterRequest);
            }

            $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
                DataikuAPI.shakers.detailedStreamingColumnAnalysis($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputStreamingEndpointId,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics).success(function(data){
                            setAnalysis(data);
                }).error(function(a, b, c) {
                    if (handleError) {
                        handleError(a, b, c);
                    }
                    setErrorInScope.bind($scope)(a, b, c);
                });
            };
            $scope.shakerHooks.fetchClusters = function(setClusters, columnName, setBased, radius, timeOut, blockSize) {
                // Do nothing
            };
            $scope.shakerHooks.fetchTextAnalysis = function(setTextAnalysis, columnName, textSettings) {
                // Do nothing
            };
        }
    }
});

app.service("DatasetChartsUtils", function(SamplingData){
    var svc = {
        makeSelectionFromScript: function(script) {
            return {
                 selection : SamplingData.makeStreamableFromMem(script.explorationSampling.selection)
            }
        }
    }
    return svc;
})

app.controller("_ChartOnDatasetSamplingEditorBase", function($scope, $stateParams, Logger, DatasetChartsUtils,
                                                                    DataikuAPI, CreateModalFromTemplate, SamplingData){
    $scope.getPartitionsList = function() {
        return DataikuAPI.datasets.listPartitions($scope.dataset)
                .error(setErrorInScope.bind($scope))
                .then(function(ret) { return ret.data });
    };

    $scope.$watch("chart.copySelectionFromScript", function(nv, ov) {
        if ($scope.canCopySelectionFromScript) {
            if ($scope.chart.copySelectionFromScript === false && !$scope.chart.refreshableSelection) {
                $scope.chart.refreshableSelection = DatasetChartsUtils.makeSelectionFromScript($scope.script);
            }
        } else {
            Logger.warn("Can't copy selection from script");
        }
    })

    $scope.showFilterModal = function() {
        var newScope = $scope.$new();
        newScope.updateFilter = (filter) => $scope.chart.refreshableSelection.selection.filter = filter;
        DataikuAPI.datasets.get($scope.dataset.projectKey, $scope.dataset.name, $stateParams.projectKey)
        .success(function(data){
            newScope.dataset = data;
            newScope.schema = data.schema;
            newScope.filter = angular.copy($scope.chart.refreshableSelection.selection.filter);
            CreateModalFromTemplate('/static/dataiku/nested-filters/input-filter-block/filter-modal.component.html', newScope, undefined, false, false, 'static');
        }).error(setErrorInScope.bind($scope));
    }

    $scope.SamplingData = SamplingData;
});

app.controller("ChartsCommonController", function ($scope, $timeout, ChartLegendsWrapper, DefaultDSSVisualizationTheme) {
    $scope.$on("listeningToForceExecuteChart", function() {
        $scope.canForceExecuteChart = true;
    });

    /**
     * Broadcast for a forceToExecute() call only if it's sure someone is already listening to such a broadcast.
     * Otherwise recheck every 100ms until some directive has told it it was listening or that 3s have passed (at which point broadcast will be made).
     */
    $scope.forceExecuteChartOrWait = function(){
        let nbTimeouts = 0;

        // Inner function does the job to isolate nbTimeouts
        function inner() {
            if ($scope.canForceExecuteChart || nbTimeouts > 30) {
                $scope.$broadcast("forceExecuteChart");
            } else {
                nbTimeouts++;
                $scope.forceExecuteChartTimeout = $timeout(inner,100);
            }
        }
        inner();
    };

    //avoid two concurrent timeouts if two calls were made to forceExecuteChartOrWait()
    $scope.$watch('forceExecuteChartTimeout', function(nv, ov) {
        if (ov!= null) {
            $timeout.cancel(ov);
        }
    });

    // We pass the scope in params to be able to hide existing props
    // in the top scope, without touching the props in child scope.
    $scope.initChartCommonScopeConfig = function(scope, chart) {
        if (!scope) {
            scope = $scope;
        }
        
        scope.legendsWrapper = new ChartLegendsWrapper();
        scope.animation = {};
        scope.tooltips = {};
        scope.displayedGeometries = [];
        scope.chartSpecific = {};
        scope.chartPicker = {};

        scope.getChartTheme = () => {
            return scope.chart && scope.chart.theme;
        };
        
        if (chart) {
            scope.chart = chart;
        }
    };
});

app.controller("ShakerChartsCommonController", function ($scope, $sce, translate, $timeout, $controller, $state, WT1, $stateParams, Logger, CreateModalFromTemplate, MonoFuture, ContextualMenu, Dialogs, ChartCustomMeasures, ChartRBNDs, ChartHierarchies, ChartUsableColumns, ColumnAvailability, ChartsContext, DataikuAPI, ChartLabels, DKUPivotCharts, ChartDimension, ChartFeatures, ChartTypeChangeHandler, ChartRequestComputer, DefaultDSSVisualizationTheme, ChartColumnTypeUtils, ChartHierarchyDimension, DropdownMenuUtilsService) {
    $controller("ChartsCommonController", {$scope:$scope});

    $scope.summary = {};
    $scope.currentChart = {index: 0};
    $scope.chartBottomOffset = 30;
    $scope.filterState = {
        chartFilter: ''
    };
    $scope.hierarchiesMissingColumns = {};
    $scope.getHierarchyMissingColumnsTooltip = ChartHierarchies.getHierarchyMissingColumnsTooltip;
    $scope.hasColumnMenu = d => !ChartColumnTypeUtils.isCount(d) && d.type !== 'GEOPOINT'
    $scope.columnTypeIcons =  ChartColumnTypeUtils.getColumnTypeIcons();

    const getHashCodedDatasetNameFromContext = () => {
        const context = $scope.analysisCoreParams || $scope.insight || $scope.dataset || { name: 'unknown' };
        return context.name.dkuHashCode();
    };

    $scope.addChart = function (params = { chart: {}, datasetName: undefined, replace: false, copyOfName: false, index: undefined, wt1Event: 'chart-create' }) {
        const { chart, datasetName, replace, index, copyOfName, wt1Event } = params;
        const newChart = {
            ...$scope.getDefaultNewChart(),
            ...angular.copy(chart || {})
        };

        if (copyOfName) {
            newChart.def.name = `Copy of ${newChart.def.name}`;
            newChart.def.userEditedName = true;
        } else {
            newChart.def.name = ChartLabels.NEW_CHART_LABEL;
        }

        const isInInsight = $state.current.name.indexOf('insight') != -1;
        const isInNoteBooks = $state.current.name.indexOf('notebooks') != -1;

        if (replace) {
            if (isInInsight || isInNoteBooks) {
                $scope.chart = newChart;
            } else {
                const chartIndex = index === undefined ? $scope.currentChart.index : index;
                $scope.charts[chartIndex] = newChart;
            }
        } else {
            // if copied, put the new chart just after the current one, otherwise put it at the end
            const targetIdx = index === undefined ? $scope.charts.length : index + 1;
            $scope.charts.splice(targetIdx, 0, newChart);
            $scope.currentChart.index = targetIdx;
        }

        if (typeof $scope.fetchColumnsSummaryForCurrentChart === 'function') {
            $scope.fetchColumnsSummaryForCurrentChart();
        } else if (typeof $scope.fetchColumnsSummary  === 'function') {
            $scope.fetchColumnsSummary().then(() => $scope.forceExecuteChartOrWait());
        }

        const hashCodedDatasetname = datasetName ? datasetName.dkuHashCode() : getHashCodedDatasetNameFromContext();

        WT1.event(wt1Event, {
            chartId: `${$stateParams.projectKey.dkuHashCode()}.${hashCodedDatasetname}.${newChart.def.name.dkuHashCode()}`,
            chartType: newChart.def.type,
            chartVariant: newChart.def.variant
        });

        if (isInInsight) {
            $scope.insight.params.def = newChart.def;
            $scope.saveChart();
        } else if (!isInNoteBooks){
            $scope.saveShaker();
        }
    };

    $scope.pageSortOptions = {
        helper: 'clone',
        axis: 'x',
        cursor: 'move',
        update: onSortUpdated,
        handle: '.thumbnail',
        items: '> a.chart',
        delay: 100,
        'ui-floating': true
    };

    function onSortUpdated(evt, ui) {
        var prevIdx = ui.item.sortable.index, newIdx = ui.item.sortable.dropindex;
        if (prevIdx == $scope.currentChart.index) {
            $scope.currentChart.index = ui.item.sortable.dropindex;
        } else if (prevIdx < $scope.currentChart.index && newIdx >= $scope.currentChart.index) {
            $scope.currentChart.index--;
        } else if (prevIdx > $scope.currentChart.index && newIdx <= $scope.currentChart.index) {
            $scope.currentChart.index++;
        }

        $timeout($scope.saveShaker);
    }

    $scope.pasteChart = function(params, index) {
        $scope.addChart({
            chart: {
                ...params.chartDef,
                def: {
                    ...params.chartDef.def,
                    id: window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16)
                }
            },
            index,
            copyOfName: true,
            replace: !params.pasteAfter,
            wt1Event: 'chart-paste'
        });
    };

    $scope.duplicateChart = function(index) {
        $scope.addChart({
            chart: {
                ...$scope.charts[index],
                def: {
                    ...$scope.charts[index].def,
                    id: window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16)
                }
            },
            index,
            copyOfName: true,
            wt1Event: 'chart-duplicate'
        });
    };

    $scope.deleteCurrentChart = function(index) {
        $scope.deleteChart(index);
    };

    $scope.getCurrentChartFromContext = function() {
        const charts = getChartsFromContext();

        if ($scope.currentChart && charts && charts.length > $scope.currentChart.index) {
            return charts[$scope.currentChart.index];
        }
        return undefined;
    }

    $scope.deleteChart = function(idx, datasetName) {
        const hashCodedDatasetname = datasetName ? datasetName.dkuHashCode() : getHashCodedDatasetNameFromContext();

        WT1.event('chart-delete', {
            chartId: `${$stateParams.projectKey.dkuHashCode()}.${hashCodedDatasetname}.${$scope.charts[idx].def.name.dkuHashCode()}`,
            chartType: $scope.charts[idx].def.type,
            chartVariant: $scope.charts[idx].def.variant
        });

        $scope.charts.splice(idx, 1);
        if ($scope.currentChart.index >= $scope.charts.length) {
            $scope.currentChart.index = $scope.charts.length - 1;
        }
        if ($scope.charts.length == 0) {
            $scope.addChart();
        }
        $scope.saveShaker();
    };

    $scope.makeUsableColumns = function(data) {
        $scope.usableColumns = ChartUsableColumns.makeUsableColumns(data, getChartsCacheableContext());
    };

    $scope.onPercentileChoice = (measure, event) => {
        measure.isCustomPercentile = event.isCustomPercentile;
        measure.percentile = event.percentile;
    };

    $scope.chartEntityMenu = new ContextualMenu({
        template: "/templates/shaker/chart-entity-edition-menu.html",
        cssClass: "chart-dataset-edition-menu",
        scope: $scope,
        contextual: false,
        enableClick: true
    });

    $scope.dimensionMenu = new ContextualMenu({
        template: "/templates/shaker/dimension-menu.html",
        cssClass: "chart-dataset-edition-menu",
        scope: $scope,
        contextual: false,
        enableClick: true
    });
    
    $scope.$on("craftBinningFormChart", function($event, params) {
        $scope.openReusableDimensionPanel( { ...params, fromChart: true });
    });

    $scope.$on("createReusableDimensionFromChart", function($event, params) {
        let existingRBNDs = getReusableDimensionsFromContext().map(c => ({ name: c.name, column: c.column }));            
        $scope.openRBNDNameModal(existingRBNDs, true, params, true);
    });

    $scope.openRBNDNameModal = (existingRBNDs, applyChangeToDimensionRef, params) => {
        const options = { btnConfirm: translate('GLOBAL.DIALOGS.CREATE', 'Create') };
        $scope.promptRBNDName(existingRBNDs, translate('CHARTS.REUSABLE_DIMENSION.NAME_PROMPT_TITLE', 'Add a name to your Reusable Dimension'), '', options, applyChangeToDimensionRef, params);
    };

    $scope.editEntity = (entity, type = 'CUSTOM_MEASURE') => {
        $scope.hideChartEntityMenu();
        switch(type) {
            case 'RBND':
                $scope.openModalRBND({ dimensionRef: entity, dimension: angular.copy(entity), isEditMode: true, fromChart: false });
                break;
            case 'HIERARCHY':
                $scope.openHierarchyEditionModal({ hierarchyRef: entity, hierarchy: angular.copy(entity), isEditMode: true });
                break;
            case 'CUSTOM_MEASURE':
                $scope.openModalCustomMeasure(entity).then(data => {
                    if (entity.isDefaultMeasure && data) {
                        ChartCustomMeasures.setImpactedCharts(getChartsFromContext(), entity);
                        ChartCustomMeasures.onItemEdit(getChartsFromContext(), data.newCustomMeasure);
                    }
                });
                break;
        }
    };

    $scope.duplicateEntity = (entity, type) => {
        $scope.hideChartEntityMenu();
        const copy = { ...angular.copy(entity), name: computeUniqueTitle(entity.name, type) };
        switch(type) {
            case 'RBND':
                modifyReusableDimensionInContext({ dimension: copy });
                break;
            case 'HIERARCHY':
                modifyHierarchyInContext({ hierarchy: copy });
                break;
            case 'CUSTOM_MEASURE':
                modifyMeasuresInContext(undefined, { ...copy, formula: copy.function });
                break;
        }
    };

    $scope.openReusableDimensionPanel = (params) => {
        if (!params.fromChart) {
            WT1.event('reusable-dimension-create', {});
        }

        $scope.hideDimensionMenu();
        $scope.openModalRBND(params);
    };

    $scope.openHierarchyPanel = (params) => {
        $scope.hideDimensionMenu();
        $scope.openHierarchyEditionModal(params);
    };
    
    $scope.getCurrentChartsContext = () => {
        if ($scope.explore) {
            return ChartsContext.EXPLORE;
        } else if($scope.insight) {
            return ChartsContext.INSIGHT;
        } else if ($scope.acp) {
            return ChartsContext.ACP;
        } else if ($scope.mlTaskDesign) {
            return ChartsContext.ML;
        } else {
            return ChartsContext.SQL_NOTEBOOK;
        }

        Logger.error('Could not find current context in scope');
    };
    
    $scope.datasetColumnsFilter = (datasetColumn) => {
        const filterText = $scope.filterState.chartFilter?.trim().toLowerCase();
        if (!filterText) {
            return true;
        }
        
        if (datasetColumn.column && datasetColumn.column.toLowerCase().includes(filterText) ||
            datasetColumn.type && datasetColumn.type.toLowerCase().includes(filterText)) {
            return true;
        }

        const reusableDimensions = $scope.getReusableDimensions(datasetColumn);
        if (reusableDimensions?.length) {
            for (let i = 0; i < reusableDimensions.length; i++) {
                if (reusableDimensions[i].name && reusableDimensions[i].name.toLowerCase().includes(filterText)) {
                    return true;
                }
            }
        }
    
        return false;
    };
    
    /**
     * Get the charts list reference from the correct context:
     * Analysis, Insight, Explorer or SqlNotebook
     */
    const getChartsFromContext = () => {
        return $scope.explore && $scope.explore.charts
            || $scope.insight && $scope.insight.params && [$scope.insight.params]
            || $scope.acp && $scope.acp.charts
            || $scope.mlTaskDesign && $scope.mlTaskDesign.predictionDisplayCharts
            || $scope.isSqlNotebook && $scope.chart && [$scope.chart];
    };

    const getChartsCacheableContext = () => {
        const context = $scope.getCurrentChartsContext();
        const dataSpec = $scope.getDataSpec();

        return {
            datasetProjectKey: dataSpec.datasetProjectKey,
            datasetName: dataSpec.datasetName,
            context
        }
    };

    const saveCorrectContext = () => {
        $scope.saveShaker && $scope.saveShaker() || $scope.saveChart && $scope.saveChart();
    };

    /**
     * Fetch current measures context & remove/add measure
     * according to function parameters. Presence of `oldMeasure` will remove it
     * from context and presence of `newMeasure` will add it to it.
     */
    const modifyMeasuresInContext = (oldMeasure, newMeasure) => {
        const measures = getCustomMeasuresFromContext();
        oldMeasure && ChartCustomMeasures.removeByNameInPlace(measures, oldMeasure.name);
        newMeasure && measures.push(newMeasure);
        const customMeasuresLike = $scope.addCustomMeasuresToScopeAndCache(measures);

        ColumnAvailability.updateAvailableColumns(undefined, undefined, customMeasuresLike);
        saveCorrectContext();
    };

    const modifyReusableDimensionInContext = (params) => {
        const hierarchies = getHierarchiesFromContext();
        const { dimensionRef: oldDimension, dimension: newDimension, isEditMode } = params;
        const applyChange = () => {
            const reusableDimensions = getReusableDimensionsFromContext();
            oldDimension && ChartRBNDs.removeByEqualityInPlace(reusableDimensions, oldDimension);
            newDimension && reusableDimensions.push(newDimension);
            $scope.addBinnedDimensionToScopeAndCache(reusableDimensions);
            // reusable dimensions can be used in hierarchies, so those can change too
            $scope.addHierarchiesToScopeAndCache(hierarchies);
            saveCorrectContext();
        }

        if (isEditMode) {
            ChartRBNDs.setImpactedCharts(getChartsFromContext(), oldDimension);
            ChartRBNDs.setImpactedHierarchies(hierarchies, oldDimension);
            if (!Object.keys(ChartRBNDs.impactedCharts).length && !Object.keys(ChartRBNDs.impactedHierarchies).length) {
                applyChange();
            } else {
                const dialogContent = Object.keys(ChartRBNDs.impactedCharts).length
                    ? translate('CHARTS.REUSABLE_DIMENSION.CONFIRM_EDIT_WARNING_MESSAGE.CHARTS', 'This reusable dimension is used in at least one chart. Editing it will affect charts using it.')
                    : translate('CHARTS.REUSABLE_DIMENSION.CONFIRM_EDIT_WARNING_MESSAGE.HIERARCHIES', 'This reusable dimension is used in at least one hierarchy. Editing it will affect hierarchies using it.')
                Dialogs.confirm($scope, translate('CHARTS.REUSABLE_DIMENSION.CONFIRM_EDIT', 'Confirm edit'), dialogContent)
                .then(() => {
                    ChartRBNDs.onItemEdit(getChartsFromContext(), newDimension, hierarchies);
                    applyChange();
                }, () => {
                    $scope.openModalRBND(params);
                });
            }
        } else {
            applyChange();
        }     
    };


    const modifyHierarchyInContext = (params) => {

        const { hierarchyRef: oldHierarchy, hierarchy: newHierarchy, isEditMode } = params;
        const applyChange = () => {
            const hierarchies = getHierarchiesFromContext();
            oldHierarchy && ChartHierarchies.removeByEqualityInPlace(hierarchies, oldHierarchy);
            newHierarchy && hierarchies.push(newHierarchy);
            $scope.addHierarchiesToScopeAndCache(hierarchies);
            saveCorrectContext();
        }

        if (isEditMode) {
            ChartHierarchies.setImpactedCharts(getChartsFromContext(), oldHierarchy);
            if (!Object.keys(ChartHierarchies.impactedCharts).length) {
                applyChange();
            } else {
                Dialogs.confirm($scope, translate('CHARTS.HIERARCHY.CONFIRM_EDIT', 'Confirm edit'), translate('CHARTS.HIERARCHY.CONFIRM_EDIT_WARNING_MESSAGE', 'This hierarchy is used in at least one chart. Editing it will affect charts using it.'))
                .then(() => {
                    ChartHierarchies.onItemEdit(getChartsFromContext(), newHierarchy);
                    applyChange();
                }, () => {
                    $scope.openHierarchyEditionModal(params);
                });
            }
        } else {
            applyChange();
        }     
    };

    const computeUniqueTitle = (name, type) => {
        let entities;
        switch (type) {
            case 'CUSTOM_MEASURE':
                entities = getCustomMeasuresFromContext();
                break;
            case 'RBND':
                entities = getReusableDimensionsFromContext();
                break;
            case 'HIERARCHY':
                entities = getHierarchiesFromContext();
                break;
        }
        const regex = /\s\(\d+\)$/;
        const baseTitle = name.replace(regex, '');
        let title = `${baseTitle} (2)`;
        let index = 2;
        while (entities.some(m => m.name === title)) {
            index++;
            title = `${baseTitle} (${index})`;
        }
    
        return title;
    };

    const getCustomMeasuresFromContext = () => {
        const cmc = getChartsCacheableContext();
        return ChartCustomMeasures.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    const getReusableDimensionsFromContext = () => {
        const cmc = getChartsCacheableContext();
        return ChartRBNDs.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    const getHierarchiesFromContext = () => {
        const cmc = getChartsCacheableContext();
        return ChartHierarchies.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    $scope.deleteEntity = (entity, type) => {
        switch(type) {
            case 'CUSTOM_MEASURE':
                ChartCustomMeasures.setImpactedCharts(getChartsFromContext(), entity);
                if (Object.keys(ChartCustomMeasures.impactedCharts).length) {
                    Dialogs.confirm($scope, "Confirm deletion", "This custom measure is used in at least one chart. Deleting it will remove it from the charts using it.").then(() => {
                        ChartCustomMeasures.onItemDelete(getChartsFromContext());
                        modifyMeasuresInContext(entity);
                    }, () => {});
                } else {
                    modifyMeasuresInContext(entity);
                }
                break;
            
            case 'RBND':
                ChartRBNDs.setImpactedCharts(getChartsFromContext(), entity);
                if (Object.keys(ChartRBNDs.impactedCharts).length) {
                    Dialogs.confirm($scope, "Confirm deletion", "This reusable dimension is used in at least one chart. Deleting it will remove it from the charts using it.").then(() => {
                        ChartRBNDs.onItemDelete(getChartsFromContext());
                        modifyReusableDimensionInContext({ dimensionRef: entity });
                    }, () => {});
                } else {
                    modifyReusableDimensionInContext({ dimensionRef: entity });
                }
                break;
            case 'HIERARCHY':
                ChartHierarchies.setImpactedCharts(getChartsFromContext(), entity);
                if (Object.keys(ChartHierarchies.impactedCharts).length) {
                    Dialogs.confirm($scope, "Confirm deletion", translate("CHARTS.HIERARCHY.CONFIRM_DELETE_WARNING_MESSAGE", "This reusable dimension is used in at least one chart. Deleting it will remove it from the charts using it.")).then(() => {
                        ChartHierarchies.onItemDelete(getChartsFromContext());
                        modifyHierarchyInContext({ hierarchyRef: entity });
                    }, () => {});
                } else {
                    modifyHierarchyInContext({ hierarchyRef: entity });
                }
                break;

        }
       
        $scope.hideChartEntityMenu();
    };

    $scope.addOrUpdateNewMeasure = (newMeasure, oldMeasure) => {
        if (oldMeasure) {
            ChartCustomMeasures.setImpactedCharts(getChartsFromContext(), oldMeasure);
            if (!Object.keys(ChartCustomMeasures.impactedCharts).length) {
                modifyMeasuresInContext(oldMeasure, newMeasure);
            } else {
                Dialogs.confirm($scope, translate("CHARTS.CUSTOM_MEASURES_EDITION.EDIT_AGGREGATION.CONFIRM", "Confirm edit"), translate("CHARTS.CUSTOM_MEASURES_EDITION.EDIT_AGGREGATION.BODY", "This custom measure is used in at least one chart. Editing it will affect charts using it."))
                .then(() => {
                    ChartCustomMeasures.onItemEdit(getChartsFromContext(), newMeasure);
                    modifyMeasuresInContext(oldMeasure, newMeasure);
                }, () => {});
            }
        } else {
            modifyMeasuresInContext(undefined, newMeasure);
        }
    };

    $scope.showChartEntityMenu = (entity, type, $event) => {
        $scope.chartEntityMenu.scope.type = type;
        $scope.chartEntityMenu.scope.entity = entity;
        $scope.chartEntityMenu.openAtXY($event.pageX, $event.pageY);
    };

    $scope.hideChartEntityMenu = () => {
        $scope.chartEntityMenu.closeAny();
    };

    $scope.showDimensionMenu = (datasetColumn, $event) => {
        $scope.datasetColumn = angular.copy(datasetColumn);
        $scope.dimensionMenu.openAtXY($event.pageX, $event.pageY);
    };

    $scope.hideDimensionMenu = () => {
        $scope.dimensionMenu.closeAny();
    };

    $scope.addCustomMeasuresToScopeAndCache = function(data) {
        const cmc = getChartsCacheableContext();
        ChartCustomMeasures.setItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context, data);
        const measures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
        // read only for html templates
        $scope.customMeasures = measures;
        return measures;
    };

    $scope.addBinnedDimensionToScopeAndCache = function(data) {
        const cmc = getChartsCacheableContext();
        ChartRBNDs.setItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context, data);
        $scope.reusableDimensions = ChartRBNDs.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    $scope.addHierarchiesToScopeAndCache = function(data) {
        const cmc = getChartsCacheableContext();
        ChartHierarchies.setItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context, data);
        // new object reference to trigger the watcher below
        $scope.hierarchies = [...ChartHierarchies.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context)];
    };

    $scope.$watchGroup(['usableColumns', 'hierarchies', 'reusableDimensions', 'reusableDimensions.length'], function([usableColumns, hierarchies, reusableDimensions]) {
        if (!usableColumns || !hierarchies || !reusableDimensions) {
            return;
        }
        $scope.hierarchiesMissingColumns = ChartHierarchies.getHierarchiesMissingColumns(hierarchies, usableColumns, reusableDimensions);
    })

    $scope.isDimensionMissing = (hierarchyName, dimension) => {
        if (!$scope.hierarchiesMissingColumns || _.isEmpty($scope.hierarchiesMissingColumns) || !$scope.hierarchiesMissingColumns[hierarchyName] || !$scope.hierarchiesMissingColumns[hierarchyName].length) {
            return false;
        }
        return ChartHierarchyDimension.isDimensionMissing($scope.hierarchiesMissingColumns[hierarchyName], dimension);
    }

    $scope.openModalCustomMeasure = (oldMeasure) => {
        const customMeasures = getCustomMeasuresFromContext();
        return $scope._openModalCustomMeasure(oldMeasure, customMeasures.map(c => c.name));
    }

    $scope.openHierarchyEditionModal = (params = {}) => {
        let existingHierarchies = getHierarchiesFromContext();
        if (params.isEditMode) {
            existingHierarchies = existingHierarchies.filter(h => !ChartHierarchies.itemEqual(params.hierarchy, h));
        }
        const usableDatasetColumns = ($scope.usableColumns || [])
            .filter(d => d.cacheable)
            .filter(d => !ChartColumnTypeUtils.isCount(d) && d.type !== 'GEOPOINT');
        params = {
            ...params,
            existingHierarchies,
            usableColumns: [...usableDatasetColumns,  ...$scope.reusableDimensions]
        };
        return $scope._openModalHierarchy(params);
    }

    $scope.openModalRBND = (params) => {
        let existingRBNDs = getReusableDimensionsFromContext().map(c => ({ name: c.name, column: c.column }));
        if (params.isEditMode) {
            existingRBNDs = existingRBNDs.filter(rbnd => !(rbnd.column === params.dimension.column && rbnd.name === params.dimension.name));
        }
        params = {
            ...params,
            existingRBNDs
        };
        return $scope._openModalRBND(params);
    }

    $scope.getReusableDimensions = (column) => {
        return $scope.reusableDimensions && $scope.reusableDimensions.filter(cb => cb.column === column.column && cb.type === column.type);
    }

    const getComplexity = formula => formula ? formula.length : 0;

    $scope._openModalCustomMeasure = (oldMeasure, allMeasureNames) => {
        $scope.sections = [null, null];

        const doclink = `<doclink page="/visualization/custom-aggregations" title="${translate('CHARTS.CUSTOM_MEASURES_EDITION.EXAMPLE_AND_DOCUMENTATION.DOCUMENTATION_TITLE','Documentation')}" show-icon></doclink>`
        
        $scope.exampleAndDocumentationBody = $sce.trustAsHtml(
            translate('CHARTS.CUSTOM_MEASURES_EDITION.EXAMPLE_AND_DOCUMENTATION.BODY', 'You can use your dataset columns to create new aggregations. You can combine existing columns, apply specific calculations, and define your own unique aggregations based on the requirements of your analysis. For more details about custom aggregation please visit the {{ documentationLink }}.', { documentationLink: doclink })
        );

        return CreateModalFromTemplate("/templates/shaker/custom-measures-edition.html", $scope, null, function(newScope) {

            const toColumn = (chartColumn) => {
                let type = chartColumn.type ? chartColumn.type.toLowerCase() : 'unspecified';
                if (chartColumn.type === 'ALPHANUM') {
                    type = 'string';
                }
                if (chartColumn.type === 'NUMERICAL') {
                    type = 'double'
                }
                return { name: chartColumn.column, type };
            }

            newScope.toggle = () => {
                newScope.sections.forEach(section => section.scope().foldableToggle());
            };
            newScope.inFullScreen = true;
            newScope.modalCanBeClosed = true;
            newScope.grelExpressionError = false;
            newScope.grelExpressionValid = false;
            newScope.grelExpressionEmpty = true;
            newScope.changeInProgress = false;
            newScope.oldMeasure = oldMeasure;
            newScope.newCustomMeasureNameIsValid = true;
            newScope.allMeasureNames = allMeasureNames;
            newScope.newCustomMeasure = oldMeasure ? { name: oldMeasure.column, formula: oldMeasure.function } : { name: '', formula: '' };
            newScope.formulaValid = !!oldMeasure;
            newScope.availableColumns = newScope.usableColumns.filter(c => c.column !== "__COUNT__").map(c => toColumn(c));
            newScope.isEditMode = oldMeasure && !oldMeasure.isDefaultMeasure;

            const validateExpressionMonoFuture = MonoFuture($scope, false, 250).wrap($scope.getExecutePromise);
            newScope.expressionValidator = (expression) => {
                // test the formula on a small dataset sample
                const dataSpec = angular.copy(newScope.getDataSpec());
                dataSpec.copySelectionFromScript = false;
                dataSpec.sampleSettings = {
                    selection: {
                        maxRecords: 10,
                        samplingMethod: 'HEAD_SEQUENTIAL',
                    }
                }

                return validateExpressionMonoFuture(
                    {
                        type: 'NO_PIVOT_AGGREGATED',
                        aggregations: [{
                            column: '',
                            function: 'CUSTOM',
                            type: 'CUSTOM',
                            customFunction: expression
                        }],
                        formulaValidation: true
                    },
                    false,
                    true,
                    newScope.requiredSampleId,
                    dataSpec
                );
            };

            newScope.onExpressionChange = () => {
                newScope.changeInProgress = true;
            };

            newScope.onCustomMeasureNameChange = () => {
                newScope.newCustomMeasureNameIsValid = !newScope.allMeasureNames.filter(n => n !== (oldMeasure && oldMeasure.name)).find(n => n === newScope.newCustomMeasure.name);
                newScope.measureForm.name.$setValidity('name', newScope.newCustomMeasureNameIsValid);
            };

            newScope.onEscape = (state) => {
                newScope.modalCanBeClosed = !state.completion;
            }

            newScope.canCloseModal = () => {
                return newScope.modalCanBeClosed;
            }

            newScope.canSaveOrEdit = () => {
                return newScope.formulaValid() && newScope.newCustomMeasure.name && newScope.newCustomMeasureNameIsValid;
            };

            newScope.formulaValid = () => {
                return newScope.newCustomMeasure.formula && !newScope.changeInProgress && newScope.grelExpressionValid;
            };

            newScope.onValidate = (result) => {
                newScope.changeInProgress = false;
                newScope.grelExpressionValid = result.valid !== false;
                newScope.grelExpressionError = result.error;
                newScope.grelExpressionEmpty = result.inputExpr.trim().length == 0;
                newScope.requiredSampleId = result.data.updatedSampleId;
                const pivotResponse = result.data.pivotResponse;
                if (pivotResponse) {
                    newScope.grelExpressionNeedFixup = pivotResponse.hasExpressionPlusWithNotNumeric;
                    newScope.newCustomMeasure.inferredType = pivotResponse.aggregations[0].type;
                }
            };

            newScope.fixupFormula = (newFormula, fixName = "plus") => {
                newScope.fixExpression(newFormula, fixName).then(data => {
                    newScope.newCustomMeasure.formula = data.data;
                });
            };

            // Can be overloaded for specific contexts (e.g. predicted charts, SQL Notebooks charts)
            // that require a dedicated backend call.
            if (!newScope.fixExpression) {
                newScope.fixExpression = (newFormula, fixName = "plus") => {
                    const spec = $scope.getDataSpec();
                    return DataikuAPI.shakers.fixExpression($stateParams.projectKey, spec.datasetProjectKey, spec.datasetName, { ...spec.script, contextProjectKey: $stateParams.projectKey, origin: 'DATASET_EXPLORE' }, $scope.requestedSampleId, newFormula, fixName, -1, -1, -1, spec.copySelectionFromScript, true);
                };
            }

            newScope.onError = (data) => {
                newScope.changeInProgress = false;
                newScope.grelExpressionNeedFixup = false;
                newScope.grelExpressionValid = false;
                newScope.grelExpressionEmpty = newScope.newCustomMeasure.formula.trim().length == 0;
                newScope.grelExpressionError = data.message || 'Unexpected error.';
                if (newScope.getDataSpec().engineType === 'SQL' && (newScope.grelExpressionError.startsWith('ERROR: operator does not exist: text +') || newScope.grelExpressionError.match(/ERROR: operator does not exist: .+ \+ text/))) {
                    newScope.grelExpressionNeedFixup = true;
                }
            };

            newScope.formulaLabelClick = () => {
                newScope.$broadcast('codemirror-focus-input');
            }

            newScope.createOrUpdateMeasure = () => {
                $scope.addOrUpdateNewMeasure(newScope.newCustomMeasure, newScope.oldMeasure);
                newScope.resolveModal({ newCustomMeasure: newScope.newCustomMeasure, oldMeasure: newScope.oldMeasure });
            }

            if (newScope.newCustomMeasure.name) {
                newScope.onCustomMeasureNameChange();
            }
        }).then((data) => {
            const wt1Event = data.oldMeasure ? 'custom-measure-edited' : 'custom-measure-saved';
            const measureComplexity = getComplexity(data.newCustomMeasure && data.newCustomMeasure.formula);
            const oldMeasureComplexity = getComplexity(data.oldMeasure && data.oldMeasure.function);
            WT1.event(wt1Event, { measureComplexity, oldMeasureComplexity });
            return data;
        }, () => {
            WT1.event('custom-measure-cancel');
        });
    };

    /**
     * Opens a modal for configuring a hierarchy.
     * 
     * @typedef {Object} HierarchyParams
     * @property {hierarchy} hierarchy
     * @property {boolean} isEditMode
     * @property {HierarchyDef[]} existingHierarchies
     * 
     * @param {HierarchyParams} params
     * @returns {void}
     */
    $scope._openModalHierarchy = (params) => {
        ChartHierarchies.openHierarchyEditionModal(params).then((newParams) => {
            modifyHierarchyInContext(newParams);
        });
    }

    $scope.alreadyUsedNameWarning = translate('CHARTS.REUSABLE_DIMENSION.ALREADY_USED_NAME_WARNING', 'This name is already taken by another reusable dimension');

    $scope.promptRBNDName = (existingRBNDs, title, name, options, applyChangeToDimensionRef, params, modalScope) => {
        const comesFromRNBDModal = !!modalScope;

        Dialogs.prompt($scope, title, translate('CHARTS.REUSABLE_DIMENSION.NAME', 'Name'), name, options).then((name) => {
            const dimension = comesFromRNBDModal ? modalScope.dimension : params.dimension;
            if (existingRBNDs.some(d => d.name === name && d.column === dimension.column)) {
                $scope.promptRBNDName(existingRBNDs, title, name, {...options, warningMessageContent: $scope.alreadyUsedNameWarning }, applyChangeToDimensionRef, params, modalScope);
                return;
            } else {
                $scope.propagateNewDimension(name, applyChangeToDimensionRef, modalScope, params);
            }
        }, () => {
            if (comesFromRNBDModal) {
                $scope._openModalRBND( {...params, dimension: modalScope.dimension });
            }
        });
    };

    $scope.propagateNewDimension = (name, applyChangeToDimensionRef = false, modalScope, params) => {
            let finalParams = params || {};
            if (modalScope) {
                modalScope.dimension.name = name;
                modalScope.dimension.isRBND = true;

                if (applyChangeToDimensionRef) {
                    Object.assign(modalScope.dimensionRef, modalScope.dimension);
                }
                finalParams = {
                    dimensionRef: modalScope.dimensionRef,
                    dimension: modalScope.dimension,
                    isEditMode: modalScope.isEditMode,
                    fromChart: modalScope.fromChart,
                }
            } else {
                params.dimension.name = name;
                params.dimension.isRBND = true;
                if (applyChangeToDimensionRef) {
                    Object.assign(params.dimensionRef, params.dimension);
                }
            }
            modifyReusableDimensionInContext(finalParams);
        };

    /**
     * Opens a modal for configuring a RBND or custom sorting
     * 
     * @typedef {Object} RBNDParams
     * @property {DimensionDef} dimension
     * @property {DimensionDef} dimensionRef
     * @property {boolean} fromChart
     * @property {boolean} isEditMode
     * @property {DimensionDef[]} existingRBNDs
     * 
     * @param {RBNDParams} params
     * @returns {void}
     */
    $scope._openModalRBND = (params) => { 
        return CreateModalFromTemplate(params.customBinningOnly ? "/templates/shaker/custom-binning.html" : "/templates/shaker/rbnd-edition.html", $scope, null, function(newScope, element) {

            const fillManualBins = (dimension) => {
                newScope.manualBinning = { values: (dimension && dimension.numParams && dimension.numParams.customBinValues || []).map((d) => ({ value: d })) };
            };

            const onChartError = (message) => {
                newScope.response = null;
                newScope.requestError = true;
                newScope.showChartPreview = true;
                newScope.lastErrorMessage = message;
            };

            const getChart = () => {
                let dimension = {
                    column: newScope.dimension.column,
                    type: newScope.dimension.type,
                    isA: 'dimension',
                    numParams: {
                      mode: 'FIXED_NB',
                      nbBins: 35,
                      emptyBinsMode: 'ZEROS',
                      niceBounds: true
                    }
                };

                if (!shouldDisplayDistribution()) {
                    dimension = newScope.dimension;
                }

                const chart = { 
                    def: {
                        $isInReusableDimensionPreview: true,
                        type: 'grouped_columns',
                        variant: 'normal',
                        genericMeasures: [{
                            displayAxis: 'axis1',
                            column: newScope.dimension.column,
                            type: newScope.dimension.type,
                            function: 'COUNT',
                            isA: 'measure'
                        }],
                        genericDimension0: [dimension]
                    },
                    theme: {
                        ...newScope.chartTheme,
                    }
                };

                return chart;
            };

            let distributionModePivotResponse;

            // set scope context
            Object.assign(newScope, params)
            newScope.DKUPivotCharts = DKUPivotCharts;
            newScope.ChartFeatures = ChartFeatures;
            newScope.ChartDimension = ChartDimension;
            newScope.newRBNDNameValid = true;
            newScope.isRBND = true;
            newScope.noClickableAxisLabels = true;
            newScope.dirty = false;
            newScope.canResetCustomBins = false;
            newScope.showChartPreview = false;
            newScope.toggleIsApplyBinningMode = false;
            newScope.requestError = false;
            newScope.chartTheme = newScope.getCurrentChartFromContext()?.theme || DefaultDSSVisualizationTheme;
            
            // fill values and configuration
            fillManualBins(newScope.dimension);
            ChartTypeChangeHandler.autocompleteGenericDimension({ type: 'dummy', variant: 'dummy' }, newScope.dimension);
            let pivotReponse = {};
            const axesDef =  { x: 0 };

            let debounce;
            newScope.$watch('dimension', function(nv, ov) {
                newScope.dirty = !angular.equals(newScope.dimensionRef, newScope.dimension) && newScope.newRBNDNameValid;
                newScope.canResetCustomBins = newScope.isEditMode && !angular.equals(newScope.dimension.numParams.customBinValues, newScope.dimensionRef.numParams.customBinValues);
                clearTimeout(debounce);
                debounce = setTimeout(() => setTimeout(() => drawChart(false, ov, nv), 200), 500);
            }, true);

            newScope.onTogglePreviewChange = (mode) => {
                if (newScope.toggleIsApplyBinningMode !== mode) {
                    newScope.toggleIsApplyBinningMode = mode;
                    drawChart(true);
                }
            };

            newScope.onRBNDNameChange = () => {
                newScope.newRBNDNameValid = !newScope.existingRBNDs.find(rbnd => rbnd.name === newScope.dimension.name && rbnd.column === newScope.dimension.column);
                newScope.rbndForm.name.$setValidity('name', newScope.newRBNDNameValid);
            };
            
            const shouldDisplayDistribution = () => {
                return !newScope.toggleIsApplyBinningMode || (!!newScope.dimension.numParams.customBinValues && newScope.dimension.numParams.customBinValues.length === 0 && newScope.dimension.numParams.mode === 'CUSTOM');
            };
            
            newScope.showMe = () => {
                drawChart(true);
            };

            const needsCustomBinningRedraw = (oldDimension, newDimension) => {
                if (!oldDimension?.numParams || !newDimension?.numParams) {
                    return false;
                }
                const oldMode = oldDimension.numParams.mode;
                const newMode = newDimension.numParams.mode;
                const oldCustomValues = oldDimension.numParams.customBinValues;
                const newCustomValues = newDimension.numParams.customBinValues;
                const modeChanged = oldMode !== newMode;
                const oneModeIsCustom = oldMode === 'CUSTOM' || newMode === 'CUSTOM';
                const customValuesChanged = newMode === 'CUSTOM' && !angular.equals(oldCustomValues, newCustomValues);
                return (modeChanged && oneModeIsCustom) || customValuesChanged;
            }

            const drawChart = (forceRedraw = false, oldDimension, newDimension) => {
                if (!forceRedraw && !newScope.showChartPreview) return;

                const displayDistribution = shouldDisplayDistribution();
                const customBinningRequiresRedraw = needsCustomBinningRedraw(oldDimension, newDimension);
                const shouldRedraw = forceRedraw || !displayDistribution || customBinningRequiresRedraw;

                if (!shouldRedraw) return;

                const isCustomBinningMode = newScope.dimension.numParams.mode === 'CUSTOM';
                const withRefLines = !newScope.toggleIsApplyBinningMode && displayDistribution && isCustomBinningMode;
                
                const chart = getChart();
                chart.def.referenceLines = [];
                withRefLines && (newScope.dimension.numParams.customBinValues || []).forEach(element => {
                    chart.def.referenceLines.push({
                        sourceType: 'Constant',
                        constantValue: element,
                        displayValue: false,
                        lineFormatting: {
                        color: 'black',
                        type: 'DASHED',
                            size: 2
                        },
                        axis: {
                            type: 'X_AXIS'
                        }
                    })
                });

                const renderChart = () => {
                    ChartTypeChangeHandler.fixupChart(chart.def, chart.theme);
                    ChartTypeChangeHandler.fixupSpec(chart);
                    const rootElement = element.find('.pivot-charts').css('display', '');
                    if (rootElement && chart.theme && chart.theme.generalFormatting && chart.theme.generalFormatting.fontFamily) {
                        element.css('--visualization-font-family', chart.theme.generalFormatting.fontFamily);
                    }
                    DKUPivotCharts.GroupedColumnsChart(rootElement, chart.def, newScope, axesDef, pivotReponse);
                };

                const onSucess = (data) => {
                    newScope.response = data;
                    pivotReponse = data.result.pivotResponse;
                    newScope.showChartPreview = true;
                    newScope.requestError = false;
                };

                const useCachedDistributionData = displayDistribution && distributionModePivotResponse;
                if (useCachedDistributionData) {
                    onSucess(distributionModePivotResponse);
                    renderChart();
                } else {
                    newScope.initChartCommonScopeConfig(newScope, chart);
                    ChartTypeChangeHandler.fixupChart(chart.def, chart.theme);
                    ChartTypeChangeHandler.fixupSpec(chart);
                    const request = ChartRequestComputer.compute(chart.def, 10, 10, newScope.chartSpecific);
                    const executePivotRequest = MonoFuture(newScope).wrap(newScope.getExecutePromise);
                    executePivotRequest(request, false, false)
                        .update((data) => {
                            newScope.response = data;
                        })
                        .success((data) => {
                            onSucess(data);
                            renderChart();
                            if (displayDistribution) {
                                distributionModePivotResponse = data;
                            }
                        }).error((error) => {
                            onChartError(error?.message ||  translate('CHARTS.REUSABLE_DIMENSION.SOMETHING_WENT_WRONG', 'Something went wrong'));
                        });
                }
            }

            newScope.createOrUpdateBinning = (applyChangeToDimensionRef = false) => {
                newScope.resolveModal({ dimension: newScope.dimension, isEditMode: newScope.isEditMode, fromChart: newScope.fromChart, changeAppliedToSourceReusableDimension: applyChangeToDimensionRef });
                if (newScope.isEditMode) {
                    $scope.propagateNewDimension(newScope.dimension.name, false, newScope);
                } else {
                    $scope.openRBNDNameModal(newScope.existingRBNDs, applyChangeToDimensionRef, newScope);
                }
            };

            newScope.applyBinningToChart = () => {
                newScope.resolveModal({ dimension: newScope.dimension, isEditMode: newScope.isEditMode, fromChart: newScope.fromChart, changeAppliedToSourceReusableDimension: false });
                DropdownMenuUtilsService.notifyDimensionUpdate(newScope.dimension);
            };

            newScope.reset = () => {
                newScope.dimension.numParams.customBinValues = angular.copy(newScope.dimensionRef.numParams.customBinValues);
                fillManualBins(newScope.dimension);
            };

            newScope.computeBins = () => {
                const customBinValues = newScope.manualBinning.values.map(input => parseFloat(input.value)).filter(value => !isNaN(value));
                if (!angular.equals(newScope.dimension.numParams.customBinValues, customBinValues)) {
                    newScope.dimension.numParams.customBinValues = customBinValues;
                }
                return ChartDimension.getCustomBins(customBinValues);
            };
        }).then((params) => {
            if (!params.customBinningOnly) {
                WT1.event("reusable-dimension-panel-save", 
                    { 
                        isReusableDimension: !!params.dimension.isRBND, 
                        mode: params.dimension?.numParams?.mode,
                        fromChart: !!params.fromChart,
                        isEditMode: !!params.isEditMode,
                        changeAppliedToSourceReusableDimension: !!params.changeAppliedToSourceReusableDimension
                    });
            }
        });
    };

    if ($stateParams.tabSelect === undefined) {
        $timeout(function() {
            $scope.$broadcast("tabSelect", "columns");
        }, 0);
    }
});

// Chart management for Analyses & Datasets
app.directive("shakerChartsCommon", function() {
    return {
        scope: true,
        priority : 100,
        controller  : 'ShakerChartsCommonController'
    }
});

})();
